
// Numerical method accuracy test

// Method: Runge-Kutta 4
// Needs linking with math library (GCC option -lm)
// Copyright: (c) Petr Peringer, FIT TU Brno, 2020

#include <stdio.h>
#include <math.h>               // exp(),sin() - for error computation
#include <stdlib.h>             // strtod()

//////////////////////////////////////////////////////////////////
// System: damped oscillator
#define SIZE 2                  // state vector size of the system
// "Dynamic section" = continuous system description
// Input:  t = time of step start,
//         y = state vector = integrator outputs
// Output: yin = vector of derivatives = integrators input values
void f(double t, double *y, double *yin)
{
    yin[0] = y[1];
    yin[1] = -0.2 * y[1] - y[0];
}

//////////////////////////////////////////////////////////////////
// Runge-Kutta 4-th order method step
// Parameters: t = start time,
//             h = step size,
//             y = state vector
// Output:  new state in y
// Warning: it does not update time
void RK4step(double t, double h, double *y)
{
    double yin[SIZE];           // integrator input = derivative
    double ystart[SIZE];        // initial state
    // 4 stages results:
    double k1[SIZE];
    double k2[SIZE];
    double k3[SIZE];
    double k4[SIZE];

    int i;
    for (i = 0; i < SIZE; i++)  // save initial value
        ystart[i] = y[i];

    f(t, y, yin);               // yin = f(t, y(t))

    for (i = 0; i < SIZE; i++) {
        k1[i] = h * yin[i];     // k1 = h * f(t, y(t))
        y[i] = ystart[i] + k1[i] / 2;
    }

    f(t + h / 2, y, yin);       // yin = f(t+h/2, y(t)+k1/2)

    for (i = 0; i < SIZE; i++) {
        k2[i] = h * yin[i];     // k2 = h * f(t+h/2, y(t)+k1/2)
        y[i] = ystart[i] + k2[i] / 2;
    }

    f(t + h / 2, y, yin);       // yin = f(t+h/2, y(t)+k2/2)

    for (i = 0; i < SIZE; i++) {
        k3[i] = h * yin[i];     // k3 = h * f(t+h/2, y(t)+k2/2)
        y[i] = ystart[i] + k3[i];
    }

    f(t + h, y, yin);           // yin = f(t+h, y(t)+k3)

    for (i = 0; i < SIZE; i++) {
        k4[i] = h * yin[i];     // k4 = h * f(t+h, y(t)+k3)
        // Result:
        // y(t+h) = y(t) + k1/6 + k2/3 + k3/3 + k4/6;
        y[i] = ystart[i] + k1[i] / 6 + k2[i] / 3 + k3[i] / 3 + k4[i] / 6;
    }
    // Return: y = new state at time t+h
}

//////////////////////////////////////////////////////////////////
// Maxima solution:
//   ode2('diff(y,x,2)+0.2*'diff(y,x)+y=0, y, x);
//   ic2(%o1,x=0,y=0,'diff(y,x)=1);
// y=(10*%e^(-x/10)*sin((3*sqrt(11)*x)/10))/(3*sqrt(11))
double analytic(double t) {
    return (10*exp(-0.1*t) * sin(0.1*3*sqrt(11)*t))/(3*sqrt(11));
}

//////////////////////////////////////////////////////////////////
// Experiment control
int main(int argc, char *argv[])
{
    double y[SIZE] = { 0, 1 };
    double stepsize = 0.01;
    if(argc>1) stepsize = strtod(argv[1],NULL);

    double t = 0;
    while (t < 50) {
        printf("%9.6f % -11g % -11g  % e\n", t, y[0], y[1], y[0] - analytic(t));
        RK4step(t, stepsize, y);        // make step
        t += stepsize;          // advance the simulation time
    }
}

