////////////////////////////////////////////////////////////////////////////
// Model centrala.cc             SIMLIB/C++
//
// Pklad simulanho programu - hovory pes centrlu
//
// Velk podnik m vnitn telefonn s se 200 klapkami, spojenou pes
// centrlu s 6 telefony s veejnou sit. Pedpokldejme, e poadavky
// spojen pichzej z veejn sit prmrn po 2 minutch (s exponen-
// cilnm rozloenm). V ppad, e klapka nen voln s pravdpodob-
// nost 90% si volajc pok ne se uvoln. Hovor trv prmrn 7minut.
// Souasn se uskuteuj vnitn hovory prmrn po 30 sekund a trvaj
// prmrn 3 minuty. Piblin po 10 minutch pijde poadavek na hovor
// zevnit ven, kter trv prmrn 10 minut. Zjistte prmrn zaten
// centrly a prmrn doby ekn na spojen vnjch
// a vnitnch hovor.
//
// Zdroj: domc koly
//

#include "simlib.h"

////////////////////////////////////////////////////////////////////////////
// asov konstanty  (za jednotku asu povaujeme 1 minutu)
const double DOBA_SIMULACE       = 8*60;// 8 hodin

// stedn doba mezi pchody:
const double PRICHOD_VNEJSICH    = 2;   // vnjch poadavk
const double PRICHOD_VNITRNICH   = 0.5; // vnitnch poadavk
const double PRICHOD_ZEVNITR_VEN = 10;  // poadavk zevnit ven

// doby hovoru v jednotlivch ppadech:
const double HOVOR_VNEJSI        = 7;
const double HOVOR_VNITRNI       = 3;
const double HOVOR_ZEVNITR_VEN   = 10;

////////////////////////////////////////////////////////////////////////////
// deklarace globlnch objekt
const int N_TEL = 200;          // poet telefon
Store Centrala("Centrla",6);   // centrla - kapacita 6 spojen najednou
Facility Telefon[N_TEL];        // telefony

Histogram Tabulka1("ekan na spojen: vnitn poadavky",0,0.1,20);
Histogram Tabulka2("ekan na spojen: vnj poadavky",0,0.1,20);

// nhodn vbr telefonu (rovnomrn rozdlen) 
int RandomTel() {
    return int(N_TEL*Random()); //  0 .. N_TEL-1
}

////////////////////////////////////////////////////////////////////////////
// tdy modelujc hovory
//
class Hovor : public Process {
 protected:
   double prichod;             // doba vstupu poadavku do vnitn st
   int odkud, kam;             // sla telefon (-1 == venku)
 public:
   Hovor() : prichod(0), odkud(-1), kam(-1) { Activate(); }
};

class HovorZvenku : public Hovor { // vnj poadavky
   void Behavior() {
      kam = RandomTel();        // komu se vol
      Enter(Centrala,1);        // obsad jeden z telefon centrly
      prichod = Time;           // poadavek vstoupil do vnitn st
      if (!Telefon[kam].Busy() || Random()<0.9) {
	 // ppad, e volan telefon je voln, anebo obsazen ale
	 // volajc je ochoten ekat (p=90%).
	 Seize(Telefon[kam]);      // obsazen nebo ekn
	 Tabulka2(Time-prichod);   // zznam doby ekn do tabulky
	 Wait(Exponential(HOVOR_VNEJSI)); // probh hovor
	 Release(Telefon[kam]);    // zaven telefonu = uvolnn linky
      }
      Leave(Centrala,1);           // uvolnn jednoho telefonu centrly
   } //Behavior
 public:
   static void Create() { new HovorZvenku; }
};

class VnitrniHovor : public Hovor { // hovory uvnit podniku
   double prichod;              // doba pchodu poadavku
   void Behavior() {
      // nhodn vybere volajc telefon - 'odkud' (mus bt voln)
      do odkud=RandomTel(); while(Telefon[odkud].Busy());
      prichod=Time;
      Seize(Telefon[odkud]);    // zvedne telefon (obsad vnitn linku)
      // nhodn vybere volan slo telefonu - kam (podmnka: kam!=odkud)
      do kam=RandomTel(); while(kam==odkud);
      // vyto slo a ek...
      Seize(Telefon[kam]);      // volan zved telefon
      Tabulka1(Time-prichod);   // zznam doby ekn
      Wait(Exponential(HOVOR_VNITRNI)); // probh hovor
      Release(Telefon[kam]);    // hovor skonil = uvolnn linek
      Release(Telefon[odkud]);
   } //Behavior
 public:
   static void Create() { new VnitrniHovor; }
};

class HovorZevnitrVen : public Hovor { // poadavky na hovory ven
   void Behavior() {
      // nhodn vybere volajc telefon - 'odkud' (mus bt voln)
      do odkud=RandomTel(); while(Telefon[odkud].Busy());
      Seize(Telefon[odkud]);    // zvedne telefon (obsad linku)
      Enter(Centrala,1);        // obsad jeden z telefon centrly
      Wait(Exponential(HOVOR_ZEVNITR_VEN));  // probh hovor
      Leave(Centrala,1);        // uvolnn telefonu centrly
      Release(Telefon[odkud]);  // zavs telefon = uvolnn linky
   } //Behavior
 public:
   static void Create() { new HovorZevnitrVen; }
};

/////////////////////////////////////////////////////////////////////////////
// genertor poadavk
//
typedef void (*CreatePtr_t)();         // typ ukazatel na statickou metodu
class Generator : public Event {       // genertor vnjch poadavk
   CreatePtr_t create;  // ukazatel na metodu Create()
   double dt;           // interval mezi vytvoenm poadavk
   void Behavior() {
      create();                        // generovn poadavku
      Activate(Time+Exponential(dt));  // dal poadavek
   }
 public:
   Generator(CreatePtr_t p, double _dt) : create(p), dt(_dt) {
     Activate();
   }
};

/////////////////////////////////////////////////////////////////////////////
// Experiment
//
int main() {
   SetOutput("centrala.out");
   Print("Model telefonovn pes centrlu\n");
   Init(0,DOBA_SIMULACE);                // inicializace experimentu
   // aktivace genertor poadavk
   new Generator(HovorZvenku::Create, PRICHOD_VNEJSICH);
   new Generator(VnitrniHovor::Create, PRICHOD_VNITRNICH);
   new Generator(HovorZevnitrVen::Create, PRICHOD_ZEVNITR_VEN);
   // simulace
   Run();
   // tisk vsledk
   Centrala.Output();
   Tabulka1.Output();
   Tabulka2.Output();
   SIMLIB_statistics.Output(); // print run statistics
}

//
