////////////////////////////////////////////////////////////////////////////
// Model pocsit                   SIMLIB/C++
//
// Pklad simulanho programu - potaov s v obchodnm domu
//

// as simulace:
//                 113 s na 386/40 bez koprocesoru
//                  31 s na 386/40 s koprocesorem
//                  10 s na P90/Win95/DosPrompt
//

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Zadn:                                                                 //
//  -------                                                                //
//    Obchodn dm  je vybaven potaovou st,  kter spojuje jednotliv //
//  oddlen a sklady a d se pomoc n distribuce zbo. Kad prodejn //
//  oddlen obchodnho  domu je vybaveno mikropotaem,  ke ktermu jsou //
//  pipojeny  pokladny  danho  oddlen  a  terminl vedoucho oddlen, //
//  kter  umouje  zskvat  pehled  o  zbo  a trb. Mikropota je //
//  spojen  s  hlavnm  potaem,  kter  v  uritch intervalech pebr //
//  informace o prodanm zbo.                                            //
//                                                                         //
//      Blokov struktura mikropotae na oddlen :                      //
//                                                                         //
// +----------------------------------------------------------------------//
// |                                                          |//
// | mikro-        |//
// | procesor                                                     |//
// |                                                    |//
// |                      |//
// |             RAM   DMA   PPI    USART   USART  ..  USART  |//
// |                      |//
// |                                                                 |//
// |                                                         |//
// |                    USART                                       |//
// |                                                         |//
// -----------------------------------------------------------------//
//                                                                    //
//       hlavn pota  <             //
//                             terminl               ...          //
//                                       //
//                                                 POKLADNY                //
//                                                                         //
//  Pomoc  modelu   provte,  zda  vyhovuje   nsledujc  konfigurace   //
// mikropotae na jednom oddlen obchodnho domu :                      //
//                                                                         //
// - poet pokladen : 3                                                    //
// - doba  mezi pchody poadavk  na zaplacen 1  ks zbo je  nhodn   //
//   veliina  s exponencilnm  rozloenm pravdpodobnosti  se stedn   //
//   hodnotou 10 s                                                         //
// - penosov rychlost mezi pokladnou a mikropotaem je 1200 b/s        //
// - velikost  penenho bloku informac psluejcho  1 ks zbo je   //
//   100 byt                                                              //
// - doba mezi otzkami zadvanmi na terminl je nhodn veliina s expo- //
//   nencilnm rozloenm pravdpodobnosti se stedn hodnotou 10 minut   //
// - penosov rychlost mezi terminlem a mikropotaem je 1 byte/100 s  //
// - velikost bloku psluejcho otzce i odpovdi je 100 a 1000 byt   //
//   (rovnomrn rozloen)                                                //
// - doba,  ne hlavn pota pevezme  otzku : 0 a  10 s (rovnomrn   //
//   rozloen)                                                            //
// - doba vypracovn odpovdi : 0.5 a 15 s (rovnomrn rozloen)        //
// - interval mezi penosy dat  do hlavnho potae je nhodn veliina   //
//   s exponencilnm rozloenm pravdpodobnosti  se stedn hodnotou 5   //
//   minut                                                                 //
// - penosov rychlost mezi hlavnm potaem a mikropotaem je 19200   //
//   bit za sekundu                                                       //
// - kapacita pamti mikropotae je 24 kB                                //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "simlib.h"

// deklarace konstant

#define T_POC  0.0         // doba simulace poten (v sek)
#define T_KON 10.0*3.6e3   //               konen
#define P_POKL 3           // poet pokladen
#define T_POKL 10.0        // stedn doba mezi pchody poadavk na pokladnu
#define V_POKLMIKR  1.2e3  // penosov rychlost pokladna - mikropota (b/s)
#define V_TERMMIKR  8.0e4  // penosov rychlost terminl - mikropota (b/s)
#define V_HLPOCMIKR 1.92e4 // penosov rychlost hl.pota - mikropota (b/s)
#define S_BLOKZBOZI  100   // velikost bloku psluejcho 1 ks zbo
#define Smin_DOTAZODPOV  100 // velikost bloku (minim.) pro otzku a odpov
#define Smax_DOTAZODPOV 1000 //     -  ||  -   (maxim.)   -  ||  -
#define T_TERM 6e2           // doba mezi zadnm nov lohy na terminl
#define Tmin_PREVDOTAZU 0.0  // doba pevzet dotazu hl. potaem (minim.)
#define Tmax_PREVDOTAZU 10.0 //             -  ||  -               (maxim.)
#define Tmin_ODPOVEDI    0.5 // doba vypracovn odpovdi (minim.)
#define Tmax_ODPOVEDI   15.0 //      -  ||  -             (maxim.)
#define T_HLPOC 1.8e2        // stedn doba mezi penosy dat do hl. potae
#define KAP_PAMETI 24576     // byt

// deklarace globlnch objekt

Store Pamet("Pam",KAP_PAMETI);
Facility Procesor("Procesor"),
	 Sbernice("Sbrnice"),
	 DMA("DMA");
Histogram TPokladna ("Doba obsluhy poadavk z pokladny",0,1,10);
Histogram TTerminal ("Doba obsluhy poadavk z terminlu",0,5,10);
unsigned ObsazKap = 0;  // kapacita pamti, uren pro penos do hl. potae


class Pokladna : public Process {  // tda loh zpracovvanch pokladnou
  double Prichod;
  void Behavior() {
    Prichod = Time;      // doba pchodu poadavku na nkterou z pokladen
    Seize(Procesor); //,1);   // vy priorita, aby pokladna dlouho neekala
    Seize(Sbernice);
    Enter(Pamet,S_BLOKZBOZI);
    ObsazKap+=S_BLOKZBOZI;
    Wait(S_BLOKZBOZI*8/V_POKLMIKR);  // penos bloku do mikropotae
    Release(Sbernice);
    Release(Procesor);
    TPokladna(Time - Prichod);  // zznam doby zpracovn do tabulky
  }
 public: Pokladna() { Activate(); }
};

class GenerPokladna : public Event {  // tda genertor loh pro pokladny
  void Behavior() {                   // popis chovn genertoru
    new Pokladna;
    Activate(Time+Exponential(T_POKL));
  }
 public: GenerPokladna() { Activate(); }
};


class Terminal : public Process { // tda loh zpracovvanch terminlem
  double Prichod;                 // as vzniku lohy zpracovvan terminlem
  int Otazka, Odpoved;            // velikost otzky a odpovdi v bytech
  void Behavior() {
    Prichod = Time;
    Seize(Procesor);
    Seize(Sbernice);
    Otazka = int(Uniform(Smin_DOTAZODPOV,Smax_DOTAZODPOV));
    Enter(Pamet,Otazka);
    Wait(Otazka*8/V_TERMMIKR);   // penos otzky do mikropotae
    Release(Sbernice);
    Release(Procesor);
    Wait(Uniform(Tmin_PREVDOTAZU,Tmax_PREVDOTAZU));
				 // pevzet otzky hl. potaem
    Seize(DMA);
    Seize(Sbernice,1);
    Wait(Otazka*8/V_HLPOCMIKR);  // penos otzky do hl. potae
    Leave(Pamet,Otazka);
    Release(Sbernice);
    Release(DMA);
    Wait(Uniform(Tmin_ODPOVEDI,Tmax_ODPOVEDI));  // vypracovn odpovdi
    Odpoved = int(Uniform(Smin_DOTAZODPOV,Smax_DOTAZODPOV));
    Seize(DMA);
    Seize(Sbernice,1);
    Enter(Pamet,Odpoved);
    Wait(Odpoved*8/V_HLPOCMIKR);  // penos odpovdi do mikropotae
    Release(Sbernice);
    Release(DMA);
    Seize(Procesor);
    Seize(Sbernice);
    Wait(Odpoved*8/V_TERMMIKR);   // penos odpovdi na terminl
    Leave(Pamet,Odpoved);
    Release(Sbernice);
    Release(Procesor);
    TTerminal(Time - Prichod);    // zznam doby zpracovn do tabulky
  }
 public: Terminal() { Activate(); }
};

class GenerTerminal : public Event {  // genertor loh pro terminl
  void Behavior() {                   // popis chovn genertoru
    new Terminal;
    Activate(Time+Exponential(T_TERM));
  }
 public: GenerTerminal() { Activate(); }
};


class HlPocitac : public Process {   // lohy zpracovvan hlavnm potaem
  void Behavior() {
    Seize(DMA);
    Seize(Sbernice); //,1);
    Wait(ObsazKap*8/V_HLPOCMIKR);   // penos dat do hlavnho potae
    Leave(Pamet,ObsazKap);
    ObsazKap = 0;
    Release(Sbernice);
    Release(DMA);
  }
 public: HlPocitac() { Activate(); }
};

class GenerHlPoc : public Event {     // genertor loh pro hlavn pota
  void Behavior() {                   // popis chovn genertoru
    new HlPocitac;
    Activate(Time+Exponential(T_HLPOC));
  }
 public: GenerHlPoc() { Activate(); }
};


int main() {
  SetOutput("pocsit.out");       // nastaven vstupnho souboru
  Init(T_POC,T_KON);             // inicializace experimentu
  for (int k=0; k<P_POKL; k++)
    new GenerPokladna;           // inicializace genertor loh pro pokladny
  new GenerTerminal;             // inicializace genertoru loh pro terminl
  new GenerHlPoc;                // inicializace genertoru loh pro hl. po.
  Print(" POCSIT --- model potaov st\n");
  Run();                         // simulace
  Procesor.Output();             // tisk vsledk
  Sbernice.Output();
  DMA.Output();
  Pamet.Output();
  TPokladna.Output();
  TTerminal.Output();
  SIMLIB_statistics.Output();     // print run statistics
}

//
