// cz - pli luouk k pl belsk dy
/////////////////////////////////////////////////////////////////////////////
// fuzzy.h
//
// SIMLIB version: 2.16.4
// Date: 2001-04-04
// Copyright (c) 1999-2001 Dr. Ing. Petr Peringer, David Martinek 
//
// This library is licensed under GNU Library GPL. See the file COPYING.
//
// Warning: this is EXPERIMENTAL code, interfaces can be changed
//
// Fuzzy subsystem for SIMLIB
// version 0.6 --- We apr 4 10:33:52 CEST 2001 
// 
// subsystem interfaces:
//  - membership functions
//  - fuzzy variable
//  - fuzzification and defuzzification methods
//  - fuzzy block 
//  - sampled fuzzy block
//  - interfaces for typing inference rules in
//


#ifndef FUZZY_H
#define FUZZY_H

#if 0
 #include <stdio.h>
 #define TRACE(x) do{ if(1) x; } while(0)
#else
 #define TRACE(x)
#endif

#include "simlib.h"
#include <vector>

namespace simlib3 {

/**
 * @mainpage FuzzySIMLIB 
 * @version 1.0
 * @author David Martinek
 * @author Dr. Ing. Petr Peringer
 * 
 * FuzzySIMLIB is part of SIMulation LIBrary including the fuzzy and xml module.
 * 
 * FuzzySIMLIB je st simulan knihovny SIMLIB a obsahuje fuzzy a xml modul.
 */
 
/**
 * @defgroup fuzzy Fuzzy modul
 * This module contains number of classes which is needed for fuzzy controllers modeling.
 *
 * Tento modul obsahuje soubor td potebnch pro modelovn fuzzy regultor.
 */

/**
 * @defgroup xml XML modul
 * General interface for loading of external data. Now only interface for reding fuzzy models
 * from XML is implemented here.
 *
 * Obecn rozhran pro natn extern uloench dat. Prozatm je zde implementovno
 * konkrtn rozhran pro natn fuzzy model z XML soubor.
 */


/**
 * @example auta.cc
 * This is complete general interface example for typing inference rules in. Time for start 
 * is a little slower but the run is quite fast. This interface is quite uncomfortable.
 * You can use rather interface which is described in FuzzyExpr and FuzzyGeneralRules classes.
 *
 * Toto je kompletn pklad pouit obecnho rozhran pro zadvn inferennch
 * pravidel. Start tohoto programu je trochu del, ale samotn bh modelu je
 * pomrn rychl. Toto rozhran je celkem nepohodln. Radji pouvejte rozhran popsan
 * ve tdch FuzzyExpr a FuzzyGeneralRules.
 */
/**
 * @example auta2.cc
 * This is comfortable general interface example for typing inference rules in. This is the
 * slowest but the  most general solution.
 *
 * Pklad komfortnho obecnho rozhran pro zpis inferennch pravidel. Toto je nejpomalej
 * ale nejobecnj een.
 */
/**
 * @example autanottree.cc
 * This is complete specialized interface example for typing infrence rules in. This is the most 
 * faster solution from all these examples.
 * 
 * Toto je kompletn pklad pouit specializovanho rozhran pro zadvn inferennch
 * pravidel. Pouitm tohoto rozhran je inicializace modelu nejrychlej ze vech 
 * tchto pklad.
 */
/**
 * @example xml.cc 
 * This is complete example of use XML for model specification. The analyzis takes a quite
 * muxh time, but it has no influence on time for run. This solution is most flexible.
 *
 * Toto je kompletn pklad pouit XML pi specifikaci modelu. Analza xml zpsobuje
 * pomrn dlouhou inicialializaci, ale na bh modelu nem dn vliv. Toto een
 * je nejvce flexibiln.
 */
/**
 * @example ftycka.cc 
 * Next complete example of use XML for model specification.  The analyzis takes a quite
 * muxh time, but it has no influence on time for run. This solution is most flexible.
 *
 * Dal kompletn pklad pouit XML pi specifikaci modelu. Analza xml zpsobuje
 * pomrn dlouhou inicialializaci, ale na bh modelu nem dn vliv. Toto een
 * je nejvce flexibiln.
 */


/////////////////////////////////////////////////////////////////////////////
// general membership function
/////////////////////////////////////////////////////////////////////////////
/**
 * Abstract baze class for membership functions.<br>
 * Abstraktn bzov tda pro funkce pslunosti.
 * @ingroup fuzzy
 */
class FuzzyMembershipFunction 
{
    char *Name;		/**< Word value.  */  // TODO: std::string
  protected:
    int defValues;      /**< Number of definition values. */
  public:
    /**
     * It assigns word value to the membership function. This does not copy poiner of parameter
     * name, but it cretes copy of memory.<br>
     * Piad slovn hodnotu funkci pslunosti. Nekopruje ukazatel parametru name, ale vytv
     * kopii pamti.
     * @param name Word value for this function.<br>Slovn hodnota pro tuto funkci
     */
    // implemented in fuzzymf.cc
    FuzzyMembershipFunction(const char *name);
    /**
     * Copy constructor. It creates copy of all member variables. For poiners creates copy of 
     * memory block.<br>
     * Kopy konstruktor. Vytv kopii vech lenskch promnnch. Pro ukazatele vytvo kopii
     * pamti.
     * @param duplicate Object that will be duplicated.<br>Objekt, kter bude duplikovn.
     */
    // implemented in fuzzymf.cc
    FuzzyMembershipFunction(const FuzzyMembershipFunction &duplicate);
    /** Destructor */
    // implemented in fuzzymf.cc
    virtual ~FuzzyMembershipFunction();
    /** Gets word value of this function.<br>Vrac slovn hodnotu tto funkce. */
    const char *wordValue() const { return Name; }
    /** It duplicates object.<br>Duplikuje objekt. */
    virtual FuzzyMembershipFunction *clone() const = 0; 
    /** It computes function value (membership).<br>Vypote funkn hodnotu (pslunost). */
    virtual double Membership(double x) const = 0; 
    /** Center of this function.<br>Sted tto funkce. */
    virtual double center() const = 0;
    /** First occurence of maximum in this function.<br>Prvn vskyt maxima v tto funkci. */
    virtual double min1() const = 0;
    /** Last occurence of maximum in this function.<br>Posledn vskyt maxima v tto funkci. */
    virtual double max1() const = 0;
    /** Print of membership table.<br>Tisk tabulky pslunosti. */
    // implemented in fuzzymf.cc
    void Print(double a, double b) const;
    /** Test if is definition of function complete.<br>Test jestli je definice funkce kompletn. */
    bool isComplete() { return defValues == getNumValues(); }
    /** This adds next definition value.<br>Pid dal definin hodnotu. */
    virtual void addDefValue(double value) = 0;
    /** It retuns number of definition values.<br>Vrt poet defininch hodnot. */
    virtual int getNumValues() = 0;
};//FuzzyMembershipFunction 

/////////////////////////////////////////////////////////////////////////////
// special membership functions:
/////////////////////////////////////////////////////////////////////////////
/**
 * Membership function of type impulse.<br>
 * Funkce pslunosti typu impuls.
 * @ingroup fuzzy
 */
class FuzzySingleton: public FuzzyMembershipFunction
{
    double x0;
  public:
    /**
     * It assigns word value to the membership function and sets definition value.<br>
     * Piad slovn hodnotu funkci pslunosti a nastav definin hodnotu.
     * @param name Word value for this function.<br>Slovn hodnota pro tuto funkci
     * @param a A definition value.<br>Definin hodnota.
     */
    // implemented in fuzzymf.cc
    FuzzySingleton(const char *name, double a);
    /**
     * It assigns word value to the membership function.<br>
     * Piad slovn hodnotu funkci pslunosti.
     * @param name Word value for this function.<br>Slovn hodnota pro tuto funkci
     */
    // implemented in fuzzymf.cc
    FuzzySingleton(const char *name);
    // implemented in fuzzymf.cc
    
    virtual ~FuzzySingleton() { TRACE(printf("~FuzzySingleton()\n")); }
    /** It duplicates object.<br>Duplikuje objekt. */
    // implemented in fuzzymf.cc
    virtual FuzzySingleton *clone() const;
    /** It computes function value (membership).<br>Vypote funkn hodnotu (pslunost). */
    // implemented in fuzzymf.cc
    virtual double Membership(double x) const;
    /** Center of this function.<br>Sted tto funkce. */
    virtual double center() const { return x0; } 
    /** First occurence of maximum in this function.<br>Prvn vskyt maxima v tto funkci. */
    virtual double min1() const { return x0; }
    /** Last occurence of maximum in this function.<br>Posledn vskyt maxima v tto funkci. */
    virtual double max1() const { return x0; }
    /** This adds next definition value.<br>Pid dal definin hodnotu. */
    // implemented in fuzzymf.cc
    virtual void addDefValue(double value);
    /** It retuns number of definition values.<br>Vrt poet defininch hodnot. */
    virtual int getNumValues() { return 1; }
};//FuzzySingleton

/**
 * Triangulal membership function.<br>Trojhelnkov funkce pslunosti.
 * @ingroup fuzzy
 */
class FuzzyTriangle: public FuzzyMembershipFunction 
{
    double x0, x1, x2;
  public:
    /**
     * It assigns word value to the membership function and sets definition values.<br>
     * Piad slovn hodnotu funkci pslunosti a nastav definin hodnotu.
     * @param name Word value for this function.<br>Slovn hodnota pro tuto funkci
     * @param a A left vertex.<br>Lev vrchol.
     * @param b A center vertex.<br>Prostedn vrchol.
     * @param c A right vertex.<br>Prav vrchol.
     */
    // implemented in fuzzymf.cc
    FuzzyTriangle(const char *name, double a, double b, double c);
    /**
     * It assigns word value to the membership function.<br>
     * Piad slovn hodnotu funkci pslunosti.
     * @param name Word value for this function.<br>Slovn hodnota pro tuto funkci
     */
    // implemented in fuzzymf.cc
    FuzzyTriangle(const char* name);
    /** Destructor. */
    virtual ~FuzzyTriangle() { TRACE(printf("~FuzzyTriangle\n")); } 
    /** It duplicates object.<br>Duplikuje objekt. */
    // implemented in fuzzymf.cc
    virtual FuzzyTriangle *clone() const;
    /** It computes function value (membership).<br>Vypote funkn hodnotu (pslunost). */
    // implemented in fuzzymf.cc
    virtual double Membership(double x) const;
    /** Center of this function.<br>Sted tto funkce. */
    virtual double center() const { return (x0+x1+x2)/3; } 
    /** First occurence of maximum in this function.<br>Prvn vskyt maxima v tto funkci. */
    virtual double min1() const { return x1; }
    /** Last occurence of maximum in this function.<br>Posledn vskyt maxima v tto funkci. */
    virtual double max1() const { return x1; }
    /** It retuns number of definition values.<br>Vrt poet defininch hodnot. */
    virtual int getNumValues() { return 3; } 
    /** This adds next definition value.<br>Pid dal definin hodnotu. */
    // implemented in fuzzymf.cc
    virtual void addDefValue(double value);
};//FuzzyTriangle

/////////////////////////////////////////////////////////////////////////////
// FuzzyTrapez ---
/**
 * Trapezoidal membership function.<br>
 * Lichobnkov funkce pslunosti.
 * @ingroup fuzzy
 */
class FuzzyTrapez: public FuzzyMembershipFunction 
{
    double x0, x1, x2, x3;
  public:
    /**
     * It assigns word value to the membership function and sets definition values.<br>
     * Piad slovn hodnotu funkci pslunosti a nastav definin hodnotu.
     * @param name Word value for this function.<br>Slovn hodnota pro tuto funkci
     * @param a A left down vertex.<br>Lev spodn vrchol.
     * @param b A left top vertex.<br>Lev horn vrchol.
     * @param c A right top vertex.<br>Prav horn vrchol.
     * @param d A right down vertex.<br>Prav spodn vrchol.
     */
    // implemented in fuzzymf.cc
    FuzzyTrapez(const char *name, double a, double b, double c, double d);
    /**
     * It assigns word value to the membership function.<br>
     * Piad slovn hodnotu funkci pslunosti.
     * @param name Word value for this function.<br>Slovn hodnota pro tuto funkci
     */
    // implemented in fuzzymf.cc
    FuzzyTrapez(const char* name);
    /** Destructor. */
    virtual ~FuzzyTrapez() { TRACE(printf("~FuzzyTrapez()\n")); }
    /** It duplicates object.<br>Duplikuje objekt. */
    // implemented in fuzzymf.cc
    virtual FuzzyTrapez *clone() const;
    /** It computes function value (membership).<br>Vypote funkn hodnotu (pslunost). */
    // implemented in fuzzymf.cc
    virtual double Membership(double x) const;
//    virtual double center() const { return (x1+x2)/2; } // bad### 
    /** Center of this function.<br>Sted tto funkce. */
    virtual double center() const { return (x0+2*x1+2*x2+x3)/6; } 
    /** First occurence of maximum in this function.<br>Prvn vskyt maxima v tto funkci. */
    virtual double min1() const { return x1; }
    /** Last occurence of maximum in this function.<br>Posledn vskyt maxima v tto funkci. */
    virtual double max1() const { return x2; }
    /** It retuns number of definition values.<br>Vrt poet defininch hodnot. */
    virtual int getNumValues() { return 4; } 
    /** This adds next definition value.<br>Pid dal definin hodnotu. */
    // implemented in fuzzymf.cc
    virtual void addDefValue(double value);
};// FuzzyTrapez

/////////////////////////////////////////////////////////////////////////////
// FuzzyGauss ---
/**
 * Gaussian membership function.<br>
 * Gaussovsk funkce pslunosti.
 * @ingroup fuzzy
 */
class FuzzyGauss: public FuzzyMembershipFunction 
{
    double sigma, c, twoSqrSigma;
  public:
    /**
     * It assigns word value to the membership function and sets definition values.<br>
     * Piad slovn hodnotu funkci pslunosti a nastav definin hodnoty.
     * @param name Word value for this function.<br>Slovn hodnota pro tuto funkci
     * @param center A center of function.(mi) <br>Sted funkce. (mi)
     * @param radius A radius of function. (3*sigma)<br>Polomr funkce. (3*sigma)
     */
    // implemented in fuzzymf.cc
    FuzzyGauss(const char *name, double center, double radius);
    /**
     * It assigns word value to the membership function.<br>
     * Piad slovn hodnotu funkci pslunosti.
     * @param name Word value for this function.<br>Slovn hodnota pro tuto funkci
     */
    // implemented in fuzzymf.cc
    FuzzyGauss(const char* name);
    /** Destructor. */
    virtual ~FuzzyGauss() { TRACE(printf("~FuzzyGauss()\n")); }
    /** It duplicates object.<br>Duplikuje objekt. */
    // implemented in fuzzymf.cc
    virtual FuzzyGauss *clone() const;
    /** It computes function value (membership).<br>Vypote funkn hodnotu (pslunost). */
    // implemented in fuzzymf.cc
    virtual double Membership(double x) const;
    /** Center of this function.<br>Sted tto funkce. */
    virtual double center() const { return c; }
    /** First occurence of maximum in this function.<br>Prvn vskyt maxima v tto funkci. */
    virtual double min1() const { return c; }
    /** Last occurence of maximum in this function.<br>Posledn vskyt maxima v tto funkci. */
    virtual double max1() const { return c; }
    /** It retuns number of definition values.<br>Vrt poet defininch hodnot. */
    int getNumValues() { return 2; } 
    /** This adds next definition value.<br>Pid dal definin hodnotu. */
    // implemented in fuzzymf.cc
    virtual void addDefValue(double value);
};// FuzzyGauss

/**
 * Nonsymetric Gaussian membership function.<br>
 * Nesymetrick Gaussovsk funkce pslunosti.
 * @ingroup fuzzy
 */
class FuzzyGauss2 : public FuzzyMembershipFunction 
{
  public:
    /**
     * It assigns word value to the membership function and sets definition values.<br>
     * Piad slovn hodnotu funkci pslunosti a nastav definin hodnoty.
     * @param name Word value for this function.<br>Slovn hodnota pro tuto funkci
     * @param leftCenter A center of the left function.(mi) <br>Sted lev funkce. (mi)
     * @param leftRadius A radius of the left function. (3*sigma)<br>Polomr lev funkce. (3*sigma)
     * @param rightCenter A center of the right function.(mi) <br>Sted prav funkce. (mi)
     * @param rightRadius A radius of the right function. (3*sigma)<br>Polomr prav funkce. (3*sigma)
     */
    // implemented in fuzzymf.cc
    FuzzyGauss2(const char* name, 
                double leftCenter, double leftRadius, 
                double rightCenter, double rightRadius);
    /**
     * It assigns word value to the membership function.<br>
     * Piad slovn hodnotu funkci pslunosti.
     * @param name Word value for this function.<br>Slovn hodnota pro tuto funkci
     */
    // implemented in fuzzymf.cc
    FuzzyGauss2(const char* name);
    /** Destructor. */
    virtual ~FuzzyGauss2() { TRACE(printf("~FuzzyGauss2()\n")); }
    /** It duplicates object.<br>Duplikuje objekt. */
    // implemented in fuzzymf.cc
    virtual FuzzyGauss2 *clone() const;
    /** It computes function value (membership).<br>Vypote funkn hodnotu (pslunost). */
    // implemented in fuzzymf.cc
    virtual double Membership(double x) const;
    /** Center of this function.<br>Sted tto funkce. */
    // implemented in fuzzymf.cc
    virtual double center() const; 
    /** First occurence of maximum in this function.<br>Prvn vskyt maxima v tto funkci. */
    // implemented in fuzzymf.cc
    virtual double min1() const;
    /** Last occurence of maximum in this function.<br>Posledn vskyt maxima v tto funkci. */
    // implemented in fuzzymf.cc
    virtual double max1() const;
    /** It retuns number of definition values.<br>Vrt poet defininch hodnot. */
    int getNumValues() { return 4; } 
    /** This adds next definition value.<br>Pid dal definin hodnotu. */
    // implemented in fuzzymf.cc
    virtual void addDefValue(double value);
  protected:
    double leftSigma;     /**< A radius of the left function. */
    double leftCenter;    /**< A center of the left function. */
    double rightSigma;    /**< A radius of the right function. */
    double rightCenter;   /**< A center of the right function. */
    /** Position of vertex when leftCenter > rightCenter.<br>Pozice vrcholu, kdy leftCenter > rightCenter. */
    // implemented in fuzzymf.cc
    double vertexPosition() const;
  private:
    double twoSqrSigmaL;  /**< 2*leftSigma*leftSigma */
    double twoSqrSigmaR;  /**< 2*rightSigma*rightSigma */
};

// TODO: add other mf

/////////////////////////////////////////////////////////////////////////////
// FuzzySet --- definition of all membership functions for fuzzy set
//
// TODO: make INTERNAL to fuzzy set, use reference counting???
//       add load/store 
//       dynamic change of parameters
//
/**
 * Implementation of a fuzzy set.<br>
 * Implementace fuzzy mnoiny.
 * @ingroup fuzzy
 */
class FuzzySet 
{
  protected:
    unsigned n;         /**< actual number of elements */
    enum { MAX=10 };    /**< implementation limit */
    const FuzzyMembershipFunction *array[MAX]; /**< is owner of objects on pointers */
    double xmin, xmax;  /**< limits */
    char * Name;        /**< name of this set */
  public:
    /**
     * Creates fuzzy set.
     * @param min Minimal value of universum.<br>Spodn mez univerza.
     * @param max Maximal value of universum.<br>Horn mez univerza.
     */
    // implemented in fuzzy.cc
    FuzzySet(const char * name, double min, double max, 
     const FuzzyMembershipFunction &m1);
    /**
     * Creates fuzzy set.
     * @param min Minimal value of universum.<br>Spodn mez univerza.
     * @param max Maximal value of universum.<br>Horn mez univerza.
     */
    // implemented in fuzzy.cc
    FuzzySet(const char * name, double min, double max, 
     const FuzzyMembershipFunction &m1,
		 const FuzzyMembershipFunction &m2);
    /**
     * Creates fuzzy set.
     * @param min Minimal value of universum.<br>Spodn mez univerza.
     * @param max Maximal value of universum.<br>Horn mez univerza.
     */
    // implemented in fuzzy.cc 
    FuzzySet(const char * name, double min, double max, 
     const FuzzyMembershipFunction &m1,
		 const FuzzyMembershipFunction &m2,
		 const FuzzyMembershipFunction &m3);
    /**
     * Creates fuzzy set.
     * @param min Minimal value of universum.<br>Spodn mez univerza.
     * @param max Maximal value of universum.<br>Horn mez univerza.
     */
    // implemented in fuzzy.cc 
    FuzzySet(const char * name, double min, double max, 
     const FuzzyMembershipFunction &m1,
		 const FuzzyMembershipFunction &m2,
		 const FuzzyMembershipFunction &m3,
		 const FuzzyMembershipFunction &m4);
    /**
     * Creates fuzzy set.
     * @param min Minimal value of universum.<br>Spodn mez univerza.
     * @param max Maximal value of universum.<br>Horn mez univerza.
     */
    // implemented in fuzzy.cc 
    FuzzySet(const char * name, double min, double max,
     const FuzzyMembershipFunction &m1,
		 const FuzzyMembershipFunction &m2,
		 const FuzzyMembershipFunction &m3,
		 const FuzzyMembershipFunction &m4,
		 const FuzzyMembershipFunction &m5);
    /**
     * Creates fuzzy set.
     * @param min Minimal value of universum.<br>Spodn mez univerza.
     * @param max Maximal value of universum.<br>Horn mez univerza.
     */
    // implemented in fuzzy.cc 
    FuzzySet(const char * name, double min, double max); 
    /** Destructor removes all membership functions.<br>Destruktor uvoln vechny funkce pslunosti. */
    // implemented in fuzzy.cc
    virtual ~FuzzySet();
    /** It duplicates object.<br>Duplikuje objekt. */
    // implemented in fuzzy.cc
    virtual FuzzySet *clone() const;
    /**
     * It adds next word value into the universum.<br>Pid dal slovn hodnotu do univerza.
     * @param x Membership function representing word value.<br>Funkce pslunosti reprezentujc slovn hodnotu.
     */
    // implemented in fuzzy.cc
    void add(const FuzzyMembershipFunction &x); 
    /** Number of membership functions.<br>Poet funkc pslunosti. */
    int count() const { return n; } 
    /** It selects i-th member function.<br>Vybere i-tou funkci pslunosti. */
    // implemented in fuzzy.cc
    virtual const FuzzyMembershipFunction *operator[] (int i) const;
    /** It computes i-th function value (membership).<br>Vypote i-tou funkn hodnotu (pslunost). */
    // implemented in fuzzy.cc
    double Membership(int i, double x) const;
    /** Minimal value of universum.<br>Spodn mez univerza. */
    double min() { return xmin; }
    /** Maximal value of universum.<br>Horn mez univerza. */
    double max() { return xmax; }
    /** Name of this fuzzy set.<br>Jmno tto fuzzy mnoiny. */
    const char * name() const { return Name; }
    /** Word value of i-th membership function.<br>Slovn hodnota i-t funkce pslunosti. */
    const char * wordValue(int i) const { return array[i]->wordValue(); }
    /** Minimum of maxims.<br>Minimum z maxim. */
    // implemented in fuzzy.cc
    double min1(); // min of 1
    /** Maximum of maxims.<br>Maximum z maxim. */
    // implemented in fuzzy.cc
    double max1(); // max of 1
    
}; // FuzzySet


/////////////////////////////////////////////////////////////////////////////
// FuzzyVariable --- fuzzy set
//
//TODO: dynamic change of variable type
//
class FuzzyBlock; // forward
/**
 * General fuzzy variable. Baze class for FuzzyInput and FuzzyOutput.<br>
 * Obecn fuzzy promnn. Bzov tda pro FuzzyInput a FuzzyOutput.
 * @ingroup fuzzy
 */
class FuzzyVariable : public aContiBlock 
{ 
    FuzzyBlock *where;  /**< location */ 
//    const FuzzySet *m;	/**< pattern: n membership functions, parameters */ 
    FuzzySet *m;        /**< pattern: n membership functions, parameters */ 
    const unsigned n;   /**< number of values */ 
    double *mval;	      /**< membership values, dynamically, size given by m */ 
  public:
    /**
     * It connects fuzzy set with variable. If it is created inside FuzzyBlock then it is not need
     * to call registerOwner. Object FuzzySet is cloned and stored inside. Therefore you must release 
     * memory allocated by t in your way.<br>
     * Spoj fuzzy mnoinu s fuzzy promnnou. Pokud je promnn vytvena
     * uvnit tdy FuzzyBlock, nen teba volat metodu registerOwner. Objekt FuzzySet je pi pedn
     * klonovn a uloen uvnit objektu. Proto muste uvolnit pam alokovanou pro t vy sami.
     * @param t Fuzzy set.<br>Fuzzy mnoina.
     */
    // implemented in fuzzy.cc
    FuzzyVariable(const FuzzySet &t);
    /** It releases memory allocated by FuzzySet.<br>Uvoln pam alokovanou pro FuzzySet. */
    virtual ~FuzzyVariable() 
    {
       TRACE(printf("~FuzzyVariable()\n")); 
       delete m;
       delete [] mval;
    }
      
    /**
     * It registers owner of this variable.<br>Zaregistruje vlastnka tto promnn.
     * @param owner Owner of this variable.<br>Vlastnk tto promnn.
     */
    // implemented in fuzzy.cc
    void registerOwner(FuzzyBlock *owner);  // registration inside owner
    /** Number of members.<br>Poet len. */
    unsigned count() const { return n; } 
    /** I-th member function.<br>I-t funkce pslunosti. */
    // implemented in fuzzy.cc
    const FuzzyMembershipFunction *mf(int i) const;
    /** It gets center of i-th member function.<br>Vrac sted i-t funkce pslunosti. */
    // implemented in fuzzy.cc
    double center(int i) const; 
    /** It gets i-th word value.<br>Vrac i-tou slovn hondotu. */
    // implemented in fuzzy.cc
    const char *wordValue(int i) const; // name of i-th membership function
    /** Get/set fuzzy value.<br>Vra/nastav fuzzy hodnotu. */
    // implemented in fuzzy.cc
    double &operator[] (int i) const;
    /** Get/set fuzzy value.<br>Vra/nastav fuzzy hodnotu. */
    double &operator[] (const char *s) const { return mval[search(s)]; }
    /** Search by member name.<br>Hled podle jmna. */
    // implemented in fuzzy.cc
    unsigned search(const char *s) const; 
    
    /** Fuzzify all membership functions.<br>Fuzzifikuje vechny funkce pslunosti.*/
    // implemented in fuzzyio.cc
    void Fuzzify(double x);
    /**
     * Initializes all values for membership functions or fuzzify all value<br>
     * Inicializuje vechny hodnoty funkc pslunosti nebo provede fuzzifikaci.
     */
    virtual void Init() { for(unsigned i=0; i!=n; i++) mval[i]=0.0F; }
    virtual void Done() {}
    /** It gets owner.<br>Vrac vlastnka. */
    FuzzyBlock *Where()           // location
    { 
//      if(where==0) SIMLIB_error("Fuzzy set should be used inside FuzzyBlock only");
      return where; 
    }
    // implemented in fuzzy.cc
    void Print(); // print contents
  private:
    // implemented in fuzzy.cc
    double SetMembership(int i, double x);
}; // FuzzyVariable

/////////////////////////////////////////////////////////////////////////////
// FuzzyInput --- continuous input fuzzy variable
//
/**
 * Input varible. It fuzzifies inputs. I.e. it gets Input value and transforms this
 * sharp value into fuzzy reprezentation.<br>
 * Vstupn promnn. Fuzzifikuje vstup. To znamen, e vezme ostrou vstupn hodnotu 
 * (Input) a transformuje ji do fuzzy reprezentace.
 * @ingroup fuzzy
 */
class FuzzyInput : public FuzzyVariable {
    Input in; /**< continuous input */ 
  public:
    /**
     * It assignes continuous input and fuzzy set into this object.<br>
     * Piad spojit vstup a fuzzy mnoinu do tohoto objektu.
     * @param i Continuous input.<br>Spojit vstup.
     * @param t Fuzzy set.<br>Fuzzy mnoina.
     */
    FuzzyInput(Input i, const FuzzySet &t) : FuzzyVariable(t), in(i) {}
    /**
     * It assignes only fuzzy set into this object. Continuous input must be 
     * added by setInput method. <br>
     * Piad tomuto objektu pouze fuzzy mnoinu. Spojit vstup mus bt
     * pidn pomoc metody setInput.
     * @param t Fuzzy set.<br>Fuzzy mnoina.
     */
    FuzzyInput(const FuzzySet &t) : FuzzyVariable(t), in(0.0) {}
    /** Destructor. */
    virtual ~FuzzyInput() { TRACE(printf("~FuzzyInput()\n")); }
    /**
     * It adds continuous input into this object.<br>
     * Pid spojit vstup do tohoto objektu.
     * @param i Continuous input.<br>Spojit vstup.
     */
    void setInput(Input i) { in = i; }
    /**
     * It returns sharp input value.<br>Vrac ostrou vstupn hodnotu.
     * @return It returns sharp input value.<br>Vrac ostrou vstupn hodnotu.
     */
    virtual double Value() { return in.Value(); }
    /**
     * It fuzzifies continuous input. This method is called every time before call 
     * to FuzzyBlock::Behavior().<br>
     * Fuzzifikuje spojit vstup. Tato metoda je volna vdy ped volnm 
     * FuzzyBlock::Behavior().
     */
    virtual void Init() { Fuzzify(in.Value()); }
//    virtual void Done() {}
};// FuzzyInput


/////////////////////////////////////////////////////////////////////////////
// FuzzyOutput --- continuous output (aggregation, defuzzification)
//
/**
 * Output variable. It defuzzifies fuzzy values, which is obtained by processing
 * inference rules.<br>
 * Vstupn promnn. Defuzzifikuje fuzzy hodnoty, kter jsou zskny zpracovnm
 * inferennch pravidel.
 * @ingroup fuzzy
 */
class FuzzyOutput: public FuzzyVariable {
    double value; /**? value after defuzzification */ 
    double (*defuzzify)(const FuzzyVariable&); /**< defuzzification function */  // remove!!!!!!!####
  public:
    /**
     * It adds fuzzy set and defuzzification function.<br>
     * Pid fuzzy mnoinu a defuzzifikan funkci.
     * @param t   Fuzzy set.<br>Fuzzy mnoina.
     * @param def Defuzzification function.<br>Defuzzifikan funkce.
     */
    // implemented in fuzzyrul.cc
    FuzzyOutput(const FuzzySet &t, double (*def)(const FuzzyVariable&)=0);
    /**
     * It assignes fuzzy value by word value.<br>Piad fuzzy hodnotu pomoc slovn hodnoty.
     */
    // implemented in fuzzyrul.cc
    const char *operator = (const char *val);
    /** 
     * It defuzzifies itself.<br> Defuzzifikuje se. 
     * @return It returns sharp value.<br>Vrt osrou hodnotu.
     */
    // implemented in fuzzyrul.cc
    double Defuzzify();
    /** It adds defuzzification function.<br>Pid defuzzifikan funkci. */
    // implemented in fuzzyrul.cc
    void SetDefuzzifyMethod(double (*f)(const FuzzyVariable&));
    /**
     * Evaluates FuzzyBlock (variable where) and after defuzzification returns sharp value.<br>
     * Vyhodnot FuzzyBlock (promnnou whera) a po defuzzifikaci vrt ostrou hodnotu.
     * @return Sharp value after defuzzification.<br>Ostr hodnota po defuzzifikaci.
     */
    // implemented in fuzzyrul.cc
    double Value();
//inherited:  virtual void Init() { for(int i=0; i<n; i++) mval[i]=0.0F; }
    /** 
     * It defuzzifies itself.<br> Defuzzifikuje se. 
     */
    virtual void Done() { Defuzzify(); }
    // implemented in fuzzyrul.cc
    virtual ~FuzzyOutput();
}; // FuzzyOutput

/////////////////////////////////////////////////////////////////////////////
// defuzzification methods
// 

// TODO: function objects   X x(params)      a=x(fset)
// indexed-centre-of-gravity
//double i_g_centre(const FuzzyVariable &fs, double lim=0.0F);
//double centroid(const FuzzyVariable &fs);
// implemented in fuzzyio.cc
/**
 * @ingroup fuzzy
 * Defuzzification method "mean-of-maximum".<br>Defuzifikan metoda "sted maxim". 
 */
double defuzMeanOfMax(const FuzzyVariable &fs);
/**
 * @ingroup fuzzy
 * Defuzzification method "min-of-maximum".<br>Defuzifikan metoda "minimln maximum". 
 */
// implemented in fuzzyio.cc
double defuzMinOfMax(const FuzzyVariable &fs);  
/** 
 * @ingroup fuzzy
 * Defuzzification method "max-of-maximum".<br>Defuzifikan metoda "maximln maximum". 
 */
// implemented in fuzzyio.cc
double defuzMaxOfMax(const FuzzyVariable &fs); 
/**
 * @ingroup fuzzy
 * Defuzzification method "discrete-center-of-gravity".<br>Defuzifikan metoda "diskrtn tit". 
 */
// implemented in fuzzyio.cc
double defuzDCOG(const FuzzyVariable &fs);

/////////////////////////////////////////////////////////////////////////////
// FuzzyBlock --- base class for inference blocks
// 
/**
 * Base class for inference blocks. Representation of fuzzy regulator. <br>
 * Bzov tda pro inferenn bloky. Reprezentace fuzzy regultoru.
 * @ingroup fuzzy
 */
class FuzzyBlock 
{
  protected:
    FuzzyBlock *where;            /**< position in hierarchical structure */
    double lastTime;              /**< time of last evaluation */ 
    virtual void Behavior() = 0;  /**< user defined fuzzy rules */ 
    dlist<FuzzyVariable*> vlist;  /**< all fuzzy variables in the block */ 
  public:
    /**
     * If inference rules are specified by FuzzyExpr way then you must call 
     * EndConstructor method on the end of constructor.<br>
     * Jestlie jsou inferenn pravidla specifikovna pomoc FuzzyExpr, pak muste
     * zavolat metodu EndConstructor na konci konstruktoru.
     */
    // implemented in fuzzyrul.cc
    FuzzyBlock();
    /**
     * It evaluates whole block. It calls method Behavior.<br>
     * Vyhodnot cel blok. Vol metodu Behavior.
     */
    // implemented in fuzzyrul.cc
    virtual void Evaluate();              // calls Behavior()
    /**
     * If inference rules are specified by FuzzyExpr way then you must call this
     * method on the end of constructor.<br>
     * Jestlie jsou inferenn pravidla specifikovna pomoc FuzzyExpr, pak muste
     * zavolat tuto metodu na konci konstruktoru.
     */
    // implemented in fuzzyrul.cc
    void EndConstructor();        // ### marks end, should be called in constructor
    /**
     * It registers fuzzy variable inside this object. If inference rules are NOT specified 
     * by FuzzyExpr way then you must call this method inside the constructor for all FuzzyInput
     * and FuzzyOutput variables.<br>
     * Registruje fuzzy promnnou v tomto objektu. Jestlie NEjsou inferenn pravidla 
     * specifikovna pomoc FuzzyExpr, pak muste zavolat tuto metodu v konstruktoru pro vechny
     * promnn typu FuzzyInput a FuzzyOutput.
     */
    // implemented in fuzzyrul.cc
    void Register(FuzzyVariable *obj); // register variable in vlist
    virtual ~FuzzyBlock() { TRACE(printf("~FuzzyBlock()\n")); }
}; // FuzzyBlock


/////////////////////////////////////////////////////////////////////////////
// Author: David Martniek
/**
 * Base class for inference blocks with sampled input.<br>
 * Bzov tda pro inferenn bloky se vzorkovanm vstupem
 * @ingroup fuzzy
 */
class FuzzySampledBlock
: public FuzzyBlock
{
    /**
     * Inner class for sampling  
     * Vnitn tda pro vzorkovn
     * @ingroup fuzzy
     */
    class FSampler 
    : public Event
    {
      
      double timeStep;
      FuzzySampledBlock *parent;
      public:
        /**
         * Creates sampler<br>Vytvo vzorkova.
         * @param parent owner of this class<br>vlastnk tto tdy
         */
        FSampler(FuzzySampledBlock *parent) 
        { 
          this->parent = parent;
          timeStep = 0;
        }
        ~FSampler() { TRACE(printf("~FSampler()\n"));  parent = 0; }
        /** It sets the time step for sampling.<br> Nastav asov krok pro vzorkovn. */
        void setTimeStep(double timeStep) { this->timeStep = timeStep; }
        /** It gets time step.<br>Vrt asov krok. */
        double getTimeStep() { return timeStep; }
        /**
         * Sampling - it evaluates the parent object and shedules next activacion.<br>
         * Vzorkovn - vyhodnot rodiovsk objekt a naplnuje dal sputn.
         */
        void Behavior()
        {
          parent->Evaluate();
          Activate(Time+timeStep);
        }
    };
  
    FSampler *sampler;

    /**
     * Will contain user defined inference rules.<br>
     * Bude obsahovat uivatelem definovan inferenn pravidla.
     */
    virtual void Behavior() = 0;  
  public:
    /**
     * It creates and initializes the sampler object.
     * If inference rules are specified by FuzzyExpr way then you must call 
     * EndConstructor method on the end of constructor.<br>
     * Vytvo a inicalizuje vzorkova.
     * Jestlie jsou inferenn pravidla specifikovna pomoc FuzzyExpr, pak muste
     * zavolat metodu EndConstructor na konci konstruktoru.
     */
    FuzzySampledBlock() { sampler = new FSampler(this); }
    virtual ~FuzzySampledBlock() { TRACE(printf("~FuzzySampledBlock")); }
    /**
     * It sets the time step for sampling.<br>
     * Nastav asov krok pro vzorkovn.
     */
    void setTimeStep(double timeStep) { sampler->setTimeStep(timeStep); }
    /** It gets time step.<br> Vrt asov krok. */
    double getTimeStep() { return sampler->getTimeStep(); }
    /** It starts sampling.<br> Nastartuje vzorkovn. */
    void Start() { sampler->Activate(); }
    /**
     * It evaluates fuzzy inference rules but only in sampled time steps.<br>
     * Vyhodnot fuzzy inferenn pravidla, ale jenom ve vzorkovanch asovch okamicch.
     */
    // implemented in fuzzyrul.cc
    virtual void Evaluate();
}; // FuzzySampledBlock

/////////////////////////////////////////////////////////////////////////////
// rules
// 
class FuzzyRule;
class FuzzyRuleFactory;

/**
 * Abstract class for representation and evaluation of inference rules.
 * It defines interface for generaly inputs, outputs and inference rules insertion.
 * It does not implement particular format form storing inference rules.<br>
 * 
 * Abstraktn tda pro reprezentaci a vyhodnocen inferennch pravidel. 
 * Definuje rozhran pro obecn vkldn vstup, vstup a inferennch pravidel.
 * Neimplementuje konkrtn formt uloen pravidel.
 *
 * @author David Martinek <a href="mailto://DavidMartinek@seznam.cz">DavidMartinek@seznam.cz</a>
 * @version 1.0
 * @ingroup fuzzy
 */
class FuzzyInferenceRules
{
  friend class FuzzyRuleFactory;
  public:
    /** 
     * Operations for use inside inference rules.<br>
     * Operace, kter je mono pout uvnit inferennch pravidel. 
     */
    enum Operations { opAND, opOR, opNOT, opNAND, opNOR };
    
    /**
     * Destructor - does NOT free memory allocated by FuzzyInput and FuzzyOutput!<br>
     *
     * Destruktor - NEuvoluje pam zabranou promnnmi FuzzyInput a FuzzyOutput! 
     * Uvolnn tto pamti se mus provst JINDE!
     */
    virtual ~FuzzyInferenceRules()
    {
      TRACE(printf("~FuzzyInferenceRules"));
      in.erase(in.begin(), in.end()); 
      out.erase(out.begin(), out.end());
    }
    
    /**
     * It adds next input variable. Before calling this method is recommended to call
     * isComplete.<br>
     * 
     * Vlo dal vstupn promnnou. Ped volnm je vhodn pout funkci isComplete.
     * @param in Ipnut fuzzy variable. It has not to be registered inside FuzzyBlock yet.
     *           Vstupn fuzzy promnn. Zde jet nemus bt registrovan uvnit FuzzyBlock.
     * 
     * @see isComplete()
     */
    virtual void addFuzzyInput(FuzzyInput *in)
    { this->in.push_back(in); }

    /**
     * It adds next output variable. Before calling this method is recommended to call
     * isComplete. <br>
     *
     * Vlo vstupn promnnou. Ped volnm je vhodn pout funkci isComplete. 
     * @param out Output fuzzy variable. It has not to be registered inside FuzzyBlock yet.
     *            Vstupn fuzzy promnn. Zde jet nemus bt registrovan uvnit FuzzyBlock.
     * @see isComplete()
     */
    virtual void addFuzzyOutput(FuzzyOutput *out)
    { this->out.push_back(out); }
    
    /**
     * It safely creates FuzzyRuleFactory.<br>
     * Bezpenm zpsobem vytvo FuzzyRuleFactory.
     */
    //implemented in rules.cc
    virtual FuzzyRuleFactory * createRuleFactory();

    /**
     * It returns true when all variables are assigned.<br>
     * Testuje, jestli u jsou piazeny vechny promnn.
     * @see addFuzzyInput(FuzzyInput *in)
     * @see addFuzzyOutput(FuzzyOutput *out)
     */
    virtual bool isComplete() = 0;
    
    /**
     * It adds next rule into list. When it is too much rules here, error is indicated.<br>
     * 
     * Pid dal pravidlo do seznamu. Pokud u je definovno pli pravidel, nastane chyba.
     * @param rule Inference rule who is represented by tree structure.
     *             Inferenn pravidlo reprezentovan stromovou strukturou.
     * @param release k, e se po vloen uvoln pam. Implicitn nastaveno true.
     *                Memory will be released after rule storing. Default it is true.
     */
    virtual void add(FuzzyRule *rule, bool release=true) = 0;
    
    /**
     * It evaluates all rules. This method is defaultly called from a method Behavior in class
     * FuzzyRSBlock.<br>
     * Vyhodnot pravidla. Tato funkce je standardn volna z funkce Behavior tdy 
     * FuzzyRSBlock.
     */
     virtual void evaluate() = 0;
  protected:
    /** Vector of input variables. <br> Vektor vstupnch promnnch. */
    std::vector<FuzzyInput *>in;
    
    /** Vector of output variables. <br> Vektor vstupnch promnnch. */
    std::vector<FuzzyOutput *>out;
  private:
}; // FuzzyInferenceRules

/**
 * This class is representing inference rules for classic fuzzy regulator with two inputs and
 * one output. It is possible to adds rules by new defined methods add(). These methods are
 * suitable for use during XML analysis.<br>
 * 
 * Tda reprezentujc inferenn pravidla pro klasick fuzzy regultor se dvma vstupy a 
 * jednm vstupem. Pravidla je mon vkldat pomoc novch funkc add, kter jsou vhodn
 * pro pouit bhem analzy dat z XML souboru. Krom toho je mono pout zddn rozhran
 * FuzzyInferenceRules
 *
 * @author David Martinek <a href="mailto://DavidMartinek@seznam.cz">DavidMartinek@seznam.cz</a>
 * @version 1.0
 * @ingroup fuzzy
 */
class FuzzyIIORules 
: public FuzzyInferenceRules
{
  public:
/////////////////////////////////////////////////////////////////
// rozhran FuzzyInferenceRules -- interface FuzzyInferenceRules
/////////////////////////////////////////////////////////////////
    /**
     * Parameterless constructor. Fuzzy variables must be added explicitly.<br>
     * Konstruktor bez parametru. Fuzzy promnn je nutno piadit pomoc dalch funkc explicitn.
     */
    FuzzyIIORules();
    
    /**
     * It initializes all input and output variables.<br>
     * Inicializuje vechny vstupn a vstupn promnn.
     */
    //implemented in rules.cc
    FuzzyIIORules(FuzzyInput *in1, FuzzyInput *in2, FuzzyOutput *out);
    
    /**
     * Destructor - does NOT free memory allocated by FuzzyInput and FuzzyOutput!<br>
     * Destruktor - NEuvoluje pam zabranou promnnmi FuzzyInput a FuzzyOutput! 
     * Uvolnn tto pamti se mus provst JINDE!
     */
    //implemented in rules.cc
    virtual ~FuzzyIIORules();
    
    /** It initializes all member variables. <br> Inicializuje vnitn promnn */
    //implemented in rules.cc
    void init();
     
    /** It adds next input variable.<br> Vlo dal vstupn promnnou. */
    //implemented in rules.cc
    virtual void addFuzzyInput(FuzzyInput *in);

    /** It adds output variable.<br> Vlo vstupn promnnou. */
    //implemented in rules.cc
    virtual void addFuzzyOutput(FuzzyOutput *out);

    /**
     * It tests if all variables are assigned.<br>
     * Testuje, jestli u jsou piazeny vechny promnn.
     * @see addFuzzyInput(FuzzyInput *in)
     * @see addFuzzyOutput(FuzzyOutput *out)
     */
     //implemented in rules.cc
    virtual bool isComplete();

    /**
     * It evaluates the rules.<br>
     * Vyhodnot pravidla.
     */
     //implemented in rules.cc
    virtual void evaluate();

    /**
     * It adds next rule into list. If there is too much rules, an error is indicated.<br>
     * Pid dal pravidlo do seznamu. Pokud u je definovno pli pravidel, nastane chyba.
     * @param rule Inference rule with tree structure.<br>Inferenn pravidlo reprezentovan stromovou strukturou.
     * @param release Memory will be released after rule storing. Default it is true.
     *                k, e se po vloen uvoln pam. Implicitn nastaveno true.
     */
     //implemented in rules.cc
    virtual void add(FuzzyRule *rule, bool release=true);
    
//////////////////////////////////////
// nov funkce -- new functions
//////////////////////////////////////
    /**
     * Adds a rule into the list.<br>
     * Pd inferenn pravidlo do seznamu.
     * @param operation     operation which is used in rule<br>operace, kter se pouije v pravidle
     * @param in1WordValue  word value of first fuzzy input variable<br>slovn hodnota prvn vstupn fuzzy promnn
     * @param in2WordValue  word value of second fuzzy input variable<br>slovn hodnota druh vstupn fuzzy promnn
     * @param outWordValue  word value of output fuzzy variable<br>slovn hodnota vstupn fuzzy promnn
     */
     //implemented in rules.cc
    void add (const Operations operation, 
              const char *in1WordValue, 
              const char *in2WordValue, 
              const char *outWordValue);
    
    /**
     * Adds a rule into the list.<br>
     * Pd inferenn pravidlo do seznamu.
     * @param operation   operation which is used in rule<br>operace, kter se pouije v pravidle
     * @param in1WVIndex  index of first fuzzy input variable<br>index slovn hodnoty prvn vstupn promnn
     * @param in2WVIndex  index of second fuzzy input variable<br>index slovn hodnoty druh vstupn promnn
     * @param outWVIndex  index of output fuzzy variable<br>index slovn hodnoty vstupn promnn
     */
     //implemented in rules.cc
    void add (const Operations operation, 
              int in1WVIndex, 
              int in2WVIndex, 
              int outWVIndex);
    /**
     * Adds a rule into the list.<br>
     * Pd inferenn pravidlo do seznamu.
     * @param operation     operation which is used in rule<br>operace, kter se pouije v pravidle
     * @param in1WVIndex    index of first fuzzy input variable<br>index slovn hodnoty prvn vstupn promnn
     * @param in2WVIndex    index of second fuzzy input variable<br>index slovn hodnoty druh vstupn promnn
     * @param outWordValue  word value of output fuzzy variable<br>slovn hodnota vstupn fuzzy promnn
     */
     //implemented in rules.cc
    void add (const Operations operation, 
              int in1WVIndex, 
              int in2WVIndex, 
              const char *outWordValue);

  protected:
    /** Array of indexes into FuzzyOutput variable.<br> Pole index do promnn FuzzyOutput. */
    int *outWV; 
    /** 
     * Array of corresponding operations between input variables.<br> Pole odpovdajcch
     * operac mezi vstupnmi promnnmi.
     */
    Operations *operation;
    /** Number of variables and rules in object.<br> Poet promnnch a pravidel v objektu */
    int inputs, outputs, rules;
    /** Maximum number of variables.<br> Maximln poet promnnch. */
    static const int MAX_INPUTS = 2;
    static const int MAX_OUTPUTS = 1;
    
    /** It tests if all arrays are allocated.<br>Testuje, jestli u jsou alokovna pole. */
    //implemented in rules.cc
    bool isAllCreated();
  private:
    /**
     * It alocates a memory space for operation and outWV arrays. A alocated array size is
     * a product of numbers of word values of all input variables.<br>
     * Alokuje prostor pro pole operation a outWV. Velikost alokovanho pole je souinem 
     * pot slovnch hodnot vech jednotlivch vstupnch promnnch.
     */
     //implemented in rules.cc
    void createVectors();
}; // FuzzyIIORules

/**
 * General form of inference rules.<br>Obecn tvar inferennch pravidel.
 *
 * General representation of inference rules, which allows to create more complex fuzzy regulators.
 * Usualy is this class used indirectly over overloaded operators. For more details see the example
 * for class FuzzyExpr and documentation for class Rules.
 *
 * Obecn reprezentace inferennch pravidel, kter dovoluje vytvet sloitj fuzzy regultory.
 * Obvykle je tato tda pouvan nepmo pes peten opertory. Dal detaily viz pklad 
 * u tdy FuzzyExpr a dokumentace pro tdu Rules.
 */
class FuzzyGeneralRules
: public FuzzyInferenceRules
{
  public:
    /**
     * It destroys vector of rules
     */
    ~FuzzyGeneralRules()
    {
      TRACE(printf("~FuzzyGeneralRules\n"));
      rules.erase(rules.begin(), rules.end());
    }
    /**
     * It returns true when all variables are assigned.<br>
     * Testuje, jestli u jsou piazeny vechny promnn.
     * @see addFuzzyInput(FuzzyInput *in)
     * @see addFuzzyOutput(FuzzyOutput *out)
     */
     //implemented in rules.cc
    virtual bool isComplete();

    /**
     * It adds next rule into list. When it is too much rules here, error is indicated.<br>
     * 
     * Pid dal pravidlo do seznamu. Pokud u je definovno pli pravidel, nastane chyba.
     * @param rule Inference rule who is represented by tree structure.<br>
     *             Inferenn pravidlo reprezentovan stromovou strukturou.
     * @param release It has no meaning here.<br>Zde nem vznam.
     */
    //implemented in rules.cc
    virtual void add(FuzzyRule *rule, bool release=true);

    /**
     * It evaluates all rules. This method is defaultly called from a method Behavior in class
     * FuzzyRSBlock.<br>
     * Vyhodnot pravidla. Tato funkce je standardn volna z funkce Behavior tdy 
     * FuzzyRSBlock.
     */
    //implemented in rules.cc
    virtual void evaluate();
  protected:
    /** Vector of rules. */
    std::vector<FuzzyRule *> rules;
  private:
}; // FuzzyGeneralRules

/**
 * FuzzyRSBlock is base class for inference blocks with sampled input and explicitly defined
 * inference rules.<br>
 * FuzzyRSBlock je bzov tda pro inferenn bloky se vzorkovanxm vstupem a  explicitn
 * definovanmi inferennmi pravidly.
 * @ingroup fuzzy
 */
class FuzzyRSBlock
: public FuzzySampledBlock
{
  public:
    /** 
     * Constructor - sets object with inference rules.<br>
     * Konstruktor - nastav objekt s inferennmi pravidly.  
     */
    FuzzyRSBlock(FuzzyInferenceRules &r) :rules(r) { }
    
    /** Destructor - destruktor. */
    virtual ~FuzzyRSBlock() 
    { TRACE(printf("~FuzzyRuledSampledBlock()\n")); }

    /**
     * It specifies behavior of this block. It evaluates all inference rules. <br>
     * Specifikuje chovn tohoto bloku. Vyhodnot vechna inferenn pravidla.
     */
    virtual void Behavior()
    { rules.evaluate(); }  

  protected:
    FuzzyInferenceRules &rules;
}; // FuzzyRSBlock

///////////////////////////////////////////////////////////////////////////////////////////////
// FuzzyRule
// Reprezentace pravidla pomoc stromu objekt. Tento zpsob realizace pravidla nen pli
// efektivn. Na druhou stranu je dostaten obecn na to, aby se s tm dalo pracovat. Tento
// pstup je zamlen spe tak, e vytven pravidel pomoc stromu je vnj rozhran, kter
// je v rmci monost uivatelsky pjemn. Pravidlo reprezentovan tmto stromem se v kadm
// z potomk tdy FuzzyInferenceRules pekonvertuje na njak efektivnj formt.  
// 
// Pokud by se ovem vyskytla poteba opravdu obecnho zpisu pravidel, je mon tento strom
// pout pmo. Veker vyhodnocovn stromu se ale mus dlat zven, protoe tdy tak, jak 
// jsou nyn navreny mohou slouit POUZE pro reprezentaci stromu. S pihldnutm k tomu, aby
// byla manipulace s objekty co nejjednodu, maj vechny tdy vechny sv datov prvky 
// veejn. 
///////////////////////////////////////////////////////////////////////////////////////////////

/**
 * Virtual knot in tree. From descendants is created a tree, who represents inference rule.<br>
 * Virtuln uzel ve stromu. Z potomk se vytv strom, kter reprezentuje inferenn pravidlo.
 * @ingroup fuzzy
 */
class FONode
{
  public:
    /**
     * It does nothing. It is here only to make this class abstract one.<br>
     * Przdn ist virtuln funkce, kter dl z tto tdy abstraktn tdu.
     */
    /**
     * Destruktor
     */
    virtual ~FONode() { TRACE(printf("~FONode\n")); };
    
    /**
     * It returns value of node - for evaluation of rule.<br>Vrac hodnotu uzlu - pro 
     * vyhodnocovn pravidel 
     */
    virtual double getValue() = 0;
}; // FONode

/**
 * Knot in a tree, who represents inference rule. This knot represents an operation.<br>
 * Uzel ve stromu, kter reprezentuje inferenn pravidlo. Tento uzel reprezentuje operaci.
 * @ingroup fuzzy
 */
class FOperation : public FONode
{
  public:
    /** Operator. */
    FuzzyInferenceRules::Operations op;
    
    /** Left operand. <br> Lev operand. */
    FONode *L;
    
    /** Right operand. <br> Prav operand. */
    FONode *R;
    
    /**
     * Constructor for a binary operator. <br> Konstruktor pro binrn opertor.
     */
    FOperation(FONode *left, FONode *right, FuzzyInferenceRules::Operations operation)
    : //FONode(FONode::ntOperation), 
      op(operation), L(left), R(right){}

    /**
     * Constructor for an unary operator. <br> Konstruktor pro unrn opertor.
     */
    // implemented in ruletree.cc
    FOperation(FONode *operand, FuzzyInferenceRules::Operations operation);
    
    /**
     * Destructor releases all subtrees. <br> Destruktor uvoln oba podstromy.
     */
    virtual ~FOperation()
    { 
      if (R != NULL) delete R; 
      if (L != NULL) delete L; 
      TRACE(printf("~FOperation\n"));
    }
    
    /**
     * It returns value after doing operation op.<br>Vrac hodnotu po proveden operace op.
     */
    // implemented in ruletree.cc
    virtual double getValue();
    
    
  protected:
  private:
}; // FOperation

/**
 * Leaf of tree. It represents comparing (in == "low") or (in != "too"). If var is 
 * FuzzyOutput, this object can represents assigning (out = "reduce").<br>
 * List stromu. Reprezentuje porovnn typu (in == "malo") pripadne (in != "hodne"). Pokud
 * se do promnn var piad FuzzyOutput, me tento objekt reprezentovat piazen typu
 * (out = "uber".)
 * @ingroup fuzzy
 */
class FPair : public FONode
{
  public:
    /** Equal or not equal. <br> Rovno nebo nerovno. */
    bool eq;
    /** Fuzzy variable. <br> Fuzzy promnn. */
    FuzzyVariable *var;
    /** Word value. <br> Slovn hodnota. */
    const char * wordValue;
    /** Index of word value in FuzzyVariable.<br> Index slovn hodnoty ve FuzzyVariable. */
    int indexWV;
    
    /**
     * Constructor.
     */
    FPair(FuzzyVariable *variable, const char * wordvalue, bool equal=true) 
    : //FONode(FONode::ntPair), 
      eq(equal), var(variable), wordValue(wordvalue) 
    {
      indexWV = var->search(wordvalue);
    }
    
     /**
      * Destructor does not release memory alocated by var and wordValue.<br>
      * Destruktor neuvoluje pam alokovanou promnnmi var a wordValue.
      */
    virtual ~FPair()
    {
      var = NULL;
      wordValue = NULL;
      TRACE(printf("~FPair\n"));
    }
    
    /**
     * If var is the FuzzyInput then this method returns fuzzified value of input.
     * In other words - value of membership function with name wordValue. Do not use if var is
     * a class FuzzyOutput.
     *
     * Jestlie parametr var je FuzzyInput, tato metoda vrac fuzzifikovanou hodnotu vstupu.
     * Jinmi slovy - hodnotu funkce pslunosti se jmnem wordValue. Nepouvejte jestlie je 
     * var tdy FuzzyOutput.
     */
    // implemented in ruletree.cc
    virtual double getValue();
  protected:
  private:
};  // FPair

/**
 * Class for representing one inference rule in tree.<br>
 * Tda reprezentujc jedno inferenn pravidlo ve form stromov struktury.
 * @ingroup fuzzy
 */
class FuzzyRule
{
  public:
    /** Constructor. */
     //implemented in ruletree.cc
    FuzzyRule();
    
    /**
     * Destructor releases memory alocated by tree. <br> Destruktor uvoln pam alokovanou stromem.
     */
     //implemented in ruletree.cc
    virtual ~FuzzyRule();
    
    /** It adds the lvalue of rule. <br> Pid levou stranu pravidla. */
    //implemented in ruletree.cc
    void addLeft(FOperation *left);
    
    /** It adds one command into list. <br> Pid dal pkaz do seznamu. */
    //implemented in ruletree.cc
    void addRight(FPair *right);
    
    /**
     * It evaluates one inference rule according to fuzzy model Mamdani.<br>
     * Vyhodnot jedno inferenn pravidlo podle fuzzy modelu Mamdani. 
     */
    //implemented in ruletree.cc
    virtual void evaluate();
    
    /** Lvalue. <br> Lev strana. */
    FOperation * left;
    
    /** Rvalue. <br> Prav strana. */
    std::vector<FPair *> right;
  protected:
  private:
}; // FuzzyRule

/**
 * Class for user friendly creating of inference rules.<br>
 * Tda pro uivatelsky pjemnj vytven inferennch pravidel.
 * @ingroup fuzzy
 */
class FuzzyRuleFactory
{
  public:
    /**
     * It safely creates fuzzy inference rule.<br>
     * Vrac bezpenm zpsobem vytvoen pravidlo.
     */
    // implemented in ruletree.cc
    FuzzyRule * createRule();

    /**
     * It adds condition of rule.<br>
     * Pid podmnkovou st do pravidla.
     */
     //implemented in ruletree.cc
    void addCondition(FOperation * operation);

    /**
     * It adds next assign command into rule.<br>
     * Pid dal piazovac pkaz do pkazov sti pravidla (konsekvent). 
     */
     //implemented in ruletree.cc
    void addConsequent(FPair * consequent);  
    
    /**
     * It creates a leaf knot of rule. <br>
     * Vytvo listov uzel pravidla.
     * @param equal true for == and not for !=
     */
     //implemented in ruletree.cc
    FPair * createNode(FuzzyVariable *var, const char * wordvalue, bool equal=true);
    
    /**
     * It creates a nonleaf knot of rule representing binary operator.<br>
     * Vytvo nelistov uzel pravidla reprezentujc binrn opertor.
     */
     //implemented in ruletree.cc
    FOperation * createNode(FONode *left, FONode *right, FuzzyInferenceRules::Operations operation);
    
    /**
     * It creates a nonleaf knot of rule representing unary operator.<br>
     * Vytvo nelistov uzel pravidla reprezentujc unrn opertor.
     */
     //implemented in ruletree.cc
    FOperation * createNode(FONode *operand, FuzzyInferenceRules::Operations operation);
    
    /** Destructor */
    virtual ~FuzzyRuleFactory()
    {
      if (rule != NULL) delete rule; 
      TRACE(printf("~FuzzyRuleFactory\n"));
    }
  protected:
    /** 
     * Constructor is protected against user. Object of this class can create only
     * object of class FuzzyInferenceRules.<br>
     * Konstruktor je chrnn ped uivatelem. Objekt tto tdy me vytvoit
     * pouze objekt tdy FuzzyInferenceRules.
     */
    friend class FuzzyInferenceRules;
    // implemented in ruletree.cc
    FuzzyRuleFactory(FuzzyInferenceRules * owner);
    FuzzyInferenceRules * owner; /**< Vlastnk tohoto objektu */
    FuzzyRule * rule;            /**< Prv vytven pravidlo */
  private:
}; // FuzzyRuleFactory



/**
 * A static class for creating general inference rules. See the class FuzzyExpr and FuzzyGeneralRules.
 *
 * Statick tda pro vytven obecnch inferennch pravidel. Podvejte se na tdu FuzzyExpr a
 * FuzzyGeneralRules.
 */
class Rules
{
  public:

    /**
     * It adds next input into the list. This is needed because of error checking.<br>
     * Pid dal vstup do seznamu. Tato metoda je poteba kvli oeten chyb.
     */
    static void addFuzzyInput(FuzzyInput * input);

    /**
     * It adds next output into the list. This is needed because of error checking.<br>
     * Pid dal vstup do seznamu. Tato metoda je poteba kvli oeten chyb.
     */
    static void addFuzzyOutput(FuzzyOutput * output);
    
    /**
     * It returns complete definition of inference rules. See at class FuzzyExpr to see how to
     * create inference rules.<br>
     * Vrac kompletn definici inferennch pravidel. Podvejte se na tdu FuzzyExpr jak se
     * vytv inferenn pravidla.
     */
    static FuzzyGeneralRules * getRules();
    
    /**
     * It produces error if there is not enough inputs and outputs or it returns FuzzyRuleFactory.<br>
     * Zpsob chybu, jestlie nen definovno dostaten mnostv vstup a vstup nebo vrt
     * FuzzyRuleFactory.
     */
    static FuzzyRuleFactory * Factory();
    
    /**
     * It adds new rule into rules.<br>Pid nov pravidlo do rules.
     */
    static void addNewRule();
    
    static bool addRule;                  /**< Create new rule in operator ==  */
  private:
    /** This class can not be instanciated. */
    Rules() {}
    /** It initializes data members. */
    static void init();
    static bool addedInput;               /**< Is here any input? */
    static bool addedOutput;              /**< Is here any output? */
    static FuzzyGeneralRules * rules;     /**< Rules to return. */
    static FuzzyRuleFactory * factory;    /**< Rule factory. */
};//class Rules

/////////////////////////////////////////////////////////////////////////////
// FuzzyExpr --- fuzzy expression value class
/**
 * "User friendly" writing of inference rules.<br>"Uivatelsky ptelsk zpis" inferennch pravidel.
 *
 * Implementation of "user friendly" writing of inference rules. This way can not be use
 * inside threads because of using global variables!
 *
 * Implementace "uivatelsky ptelskho" zpisu inferennch pravidel. Tento zpsob neme
 * bt pouit uvnit vlken kvli pouvn globlnch promnnch.
 *
 * Example/Pklad:
 * @code
 * FuzzyInferenceRules * createRules(FuzzyInput *speed, 
 *                                   FuzzyInput *distance, 
 *                                   FuzzyInput *wet, 
 *                                   FuzzyOutput *acceleration,
 *                                   FuzzyOutput *brake)
 * {
 *   Rules::addFuzzyInput(speed);
 *   Rules::addFuzzyInput(distance);
 *   Rules::addFuzzyInput(wet);
 *   Rules::addFuzzyOutput(acceleration);
 *   Rules::addFuzzyOutput(brake);
 *  
 *   if (speed=="high" && (distance=="middle" || wet != "damp"))
 *   { 
 *     acceleration = "zero";
 *     brake = "middle";
 *   } 
 *     ...
 *
 *   return Rules::getRules();
 * }
 * @endcode
 * @ingroup fuzzy
 */
class FuzzyExpr 
{
    FONode * value;
  public:
    /**
     * Objects of this class are usualy created inside if command.<br>
     * Objekty tto tdy jsou obvykle vytveny uvnit pkazu if.
     */
    // implemented in fuzzyrul.cc
    explicit FuzzyExpr(FONode * value);
    /** It returns value of expression.<br>Vrac hodnotu vrazu. */
    FONode * Value() 
    {
      FONode * node = value;
      value = NULL;
      return node; 
    }
    /**
     * It creates object tree.<br>Vytvo strom objekt.
     */
    // implemented in fuzzyrul.cc
    operator bool(); // if(bool(expr)) actions; --- store alpha
    
};// FuzzyExpr

/////////////////////////////////////////////////////////////////////////////
// Fuzzy operators --- used in FuzzyBlock::Behavior() 
//
// implemented in fuzzyrul.cc
/**
 * @ingroup fuzzy
 * if ((FuzzyExpr) && (FuzzyExpr)) 
 */
FuzzyExpr operator && (FuzzyExpr o1, FuzzyExpr o2);        // AND
/**
 * @ingroup fuzzy
 * if ((FuzzyExpr) || (FuzzyExpr)) 
 */
FuzzyExpr operator || (FuzzyExpr o1, FuzzyExpr o2);        // OR
/**
 * @ingroup fuzzy
 * if (!(FuzzyExpr)) 
 */
FuzzyExpr operator ! (FuzzyExpr o1);                       // NOT
/**
 * @ingroup fuzzy
 * if ((input == "wordvalue")...) 
 */
FuzzyExpr operator == (FuzzyInput &s, const char *value);  // is


} // namespace

#endif
