/*
								+---------------------------------+
								|                                 |
								|  ***  Standard hash funcs  ***  |
								|                                 |
								|  Copyright   -tHE SWINe- 2006  |
								|                                 |
								|            Hash.cpp             |
								|                                 |
								+---------------------------------+
*/

/**
 *	@file Hash.cpp
 *	@author -tHE SWINe-
 *	@date 2006
 *	@brief implementation of basic message digest algorithms
 *
 *	@date 2007-12-24
 *
 *	improved linux compatibility by adding posix integer types
 *
 *	@date 2008-03-04
 *
 *	now using Integer.h header
 *
 *	@date 2008-07-04
 *
 *	replaced all occurences of unsigned char with uint8_t
 *	changed CStreamHash::Process_Data(const unit8_t*, int) to
 *	CStreamHash::Process_Data(const void*, int) so there's no need
 *	to type cast data anymore (convenience improvement)
 *
 *	@date 2009-05-04
 *
 *	fixed mixed windows / linux line endings
 *
 */

#include <assert.h>
#define _ASSERTE(x) assert(x)
//#include "NewFix.h"
//#include "CallStack.h"
#include <string.h>
#include <stdio.h>
#include "Hash.h"
#include "Unused.h"

#if defined(_MSC_VER) && !defined(__MWERKS__) && !defined(for)
#define for if(0) {} else for
#endif

/*
 *								=== test ===
 */

#if 0

void HashTest()
{
	{
		CStreamHash<TTigerHash> hash;
		hash.Process_Data("", 0);
		TTigerHash t = hash.t_Result();
		printf("tiger(\"\") = 0x%s\n", t.p_s_ToString());
		printf("should be:  0x3293ac630c13f0245f92bbb1766e16167a4e58492dde73f3\n");
	}

	{
		CStreamHash<TTiger2Hash> hash;
		hash.Process_Data("", 0);
		TTiger2Hash t = hash.t_Result();
		printf("tiger2(\"\") = 0x%s\n", t.p_s_ToString());
		printf("should be:   0x4441be75f6018773c206c22745374b924aa8313fef919f41\n");
	}

	{
		CStreamHash<TTigerHash> hash;
		hash.Process_Data("The quick brown fox jumps over the lazy dog",
			strlen("The quick brown fox jumps over the lazy dog") * sizeof(char));
		TTigerHash t = hash.t_Result();
		printf("tiger(\"The quick brown fox jumps over the lazy dog\") = 0x%s\n", t.p_s_ToString());
		printf("should be:                                             "
			"0x6d12a41e72e644f017b6f0e2f7b44c6285f06dd5d2c5b075\n");
	}

	{
		CStreamHash<TTiger2Hash> hash;
		hash.Process_Data("The quick brown fox jumps over the lazy dog",
			strlen("The quick brown fox jumps over the lazy dog") * sizeof(char));
		TTiger2Hash t = hash.t_Result();
		printf("tiger2(\"The quick brown fox jumps over the lazy dog\") = 0x%s\n", t.p_s_ToString());
		printf("should be:                                              "
			"0x976abff8062a2e9dcea3a1ace966ed9c19cb85558b4976d8\n");
	}

	{
		CStreamHash<TSHA1> hash;
		hash.Process_Data("", 0);
		TSHA1 t = hash.t_Result();
		printf("sha-1(\"\") = 0x%s\n", t.p_s_ToString());
		printf("should be:  0xda39a3ee5e6b4b0d3255bfef95601890afd80709\n");
	}

	{
		CStreamHash<TMD5> hash;
		hash.Process_Data("", 0);
		TMD5 t = hash.t_Result();
		printf("md5(\"\") =  0x%s\n", t.p_s_ToString());
		printf("should be: 0xd41d8cd98f00b204e9800998ecf8427e\n");
	}
}

#endif // 0

/*
 *								=== ~test ===
 */

/*
 *								=== left rotation with constant shift ===
 */

template <int rot>
class CConstShiftLeftRot {
private:
	enum {
		shift_A = rot,
		shift_B = 32 - rot
	};

public:
	static inline uint32_t n_Rotate(uint32_t x)
	{
		return (x << shift_A) | (x >> shift_B);
	}
};

#define n_leftrot_const(x,r) CConstShiftLeftRot<r>::n_Rotate(x)

/*
 *								=== ~left rotation with constant shift ===
 */

/*
 *								=== TMD5 ===
 */

TMD5::TMD5()
	:m_n_total_length_bits(0)
{
	m_n_hash[0] = 0x67452301;
	m_n_hash[1] = 0xefcdab89;
	m_n_hash[2] = 0x98badcfe;
	m_n_hash[3] = 0x10325476;
}

TMD5 &TMD5::operator =(const TMD5 &r_t_md5)
{
	m_n_total_length_bits = r_t_md5.m_n_total_length_bits;
	memcpy(m_n_hash, r_t_md5.m_n_hash, 4 * sizeof(uint32_t));
	return *this;
}

bool TMD5::operator ==(const TMD5 &r_t_md5) const
{
	return m_n_hash[0] == r_t_md5.m_n_hash[0] && m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] == r_t_md5.m_n_hash[2] && m_n_hash[3] == r_t_md5.m_n_hash[3];
}

bool TMD5::operator <(const TMD5 &r_t_md5) const
{
	return m_n_hash[0] < r_t_md5.m_n_hash[0] || (m_n_hash[0] == r_t_md5.m_n_hash[0] &&
		   (m_n_hash[1] < r_t_md5.m_n_hash[1] || (m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   (m_n_hash[2] < r_t_md5.m_n_hash[2] || (m_n_hash[2] == r_t_md5.m_n_hash[2] &&
		   m_n_hash[3] < r_t_md5.m_n_hash[3])))));
}

#ifdef __MD5_UNROLL_LOOPS

template <int i>
class CMD5RotationConst {
public:
	enum {
		value = (i < 16 && i % 4 == 0)? 7 :
				(i < 16 && i % 4 == 1)? 12 :
				(i < 16 && i % 4 == 2)? 17 :
				(i < 16 && i % 4 == 3)? 22 :

				(i < 32 && i % 4 == 0)? 5 :
				(i < 32 && i % 4 == 1)? 9 :
				(i < 32 && i % 4 == 2)? 14 :
				(i < 32 && i % 4 == 3)? 20 :

				(i < 48 && i % 4 == 0)? 4 :
				(i < 48 && i % 4 == 1)? 11 :
				(i < 48 && i % 4 == 2)? 16 :
				(i < 48 && i % 4 == 3)? 23 :

				(i < 64 && i % 4 == 0)? 6 :
				(i < 64 && i % 4 == 1)? 10 :
				(i < 64 && i % 4 == 2)? 15 :
				(i < 64 && i % 4 == 3)? 21 : -1
	};
};

#define n_MD5RotationConst(i) CMD5RotationConst<i>::value

#define MD5_F0(_b_,_c_,_d_) ((_d_) ^ ((_b_) & ((_c_) ^ (_d_))))
#define MD5_F1(_b_,_c_,_d_) ((_c_) ^ ((_d_) & ((_b_) ^ (_c_))))
#define MD5_F2(_b_,_c_,_d_) ((_b_) ^ (_c_) ^ (_d_))
#define MD5_F3(_b_,_c_,_d_) ((_c_) ^ ((_b_) | ~(_d_)))

#define MD5_UNROLLED_BLOCK(_a_,_b_,_c_,_d_,i,_fun_,_step_,_off_,_k_) \
	do { \
		uint32_t f = _fun_(_b_,_c_,_d_); \
		uint32_t x = _p_data[(_step_ * i + _off_) & 0x0f]; \
		_a_ = _b_ + n_leftrot_const(_a_ + f + x + _k_, n_MD5RotationConst(i)); \
	} while(0)

#endif // __MD5_UNROLL_LOOPS

void TMD5::Process_Block(const void *p_data, int UNUSED(n_size_bytes))
{
	_ASSERTE(n_size_bytes == 64);
	_ASSERTE(m_n_total_length_bits <= UINT64_MAX - 512);

	const uint32_t *_p_data = (const uint32_t*)p_data;

	uint32_t a = m_n_hash[0], b = m_n_hash[1], c = m_n_hash[2], d = m_n_hash[3];

#ifdef __MD5_UNROLL_LOOPS
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x00, MD5_F0, 1, 0, 0xd76aa478u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x01, MD5_F0, 1, 0, 0xe8c7b756u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x02, MD5_F0, 1, 0, 0x242070dbu);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x03, MD5_F0, 1, 0, 0xc1bdceeeu);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x04, MD5_F0, 1, 0, 0xf57c0fafu);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x05, MD5_F0, 1, 0, 0x4787c62au);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x06, MD5_F0, 1, 0, 0xa8304613u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x07, MD5_F0, 1, 0, 0xfd469501u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x08, MD5_F0, 1, 0, 0x698098d8u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x09, MD5_F0, 1, 0, 0x8b44f7afu);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x0a, MD5_F0, 1, 0, 0xffff5bb1u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x0b, MD5_F0, 1, 0, 0x895cd7beu);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x0c, MD5_F0, 1, 0, 0x6b901122u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x0d, MD5_F0, 1, 0, 0xfd987193u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x0e, MD5_F0, 1, 0, 0xa679438eu);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x0f, MD5_F0, 1, 0, 0x49b40821u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x10, MD5_F1, 5, 1, 0xf61e2562u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x11, MD5_F1, 5, 1, 0xc040b340u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x12, MD5_F1, 5, 1, 0x265e5a51u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x13, MD5_F1, 5, 1, 0xe9b6c7aau);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x14, MD5_F1, 5, 1, 0xd62f105du);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x15, MD5_F1, 5, 1, 0x02441453u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x16, MD5_F1, 5, 1, 0xd8a1e681u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x17, MD5_F1, 5, 1, 0xe7d3fbc8u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x18, MD5_F1, 5, 1, 0x21e1cde6u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x19, MD5_F1, 5, 1, 0xc33707d6u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x1a, MD5_F1, 5, 1, 0xf4d50d87u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x1b, MD5_F1, 5, 1, 0x455a14edu);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x1c, MD5_F1, 5, 1, 0xa9e3e905u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x1d, MD5_F1, 5, 1, 0xfcefa3f8u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x1e, MD5_F1, 5, 1, 0x676f02d9u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x1f, MD5_F1, 5, 1, 0x8d2a4c8au);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x20, MD5_F2, 3, 5, 0xfffa3942u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x21, MD5_F2, 3, 5, 0x8771f681u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x22, MD5_F2, 3, 5, 0x6d9d6122u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x23, MD5_F2, 3, 5, 0xfde5380cu);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x24, MD5_F2, 3, 5, 0xa4beea44u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x25, MD5_F2, 3, 5, 0x4bdecfa9u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x26, MD5_F2, 3, 5, 0xf6bb4b60u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x27, MD5_F2, 3, 5, 0xbebfbc70u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x28, MD5_F2, 3, 5, 0x289b7ec6u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x29, MD5_F2, 3, 5, 0xeaa127fau);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x2a, MD5_F2, 3, 5, 0xd4ef3085u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x2b, MD5_F2, 3, 5, 0x04881d05u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x2c, MD5_F2, 3, 5, 0xd9d4d039u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x2d, MD5_F2, 3, 5, 0xe6db99e5u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x2e, MD5_F2, 3, 5, 0x1fa27cf8u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x2f, MD5_F2, 3, 5, 0xc4ac5665u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x30, MD5_F3, 7, 0, 0xf4292244u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x31, MD5_F3, 7, 0, 0x432aff97u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x32, MD5_F3, 7, 0, 0xab9423a7u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x33, MD5_F3, 7, 0, 0xfc93a039u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x34, MD5_F3, 7, 0, 0x655b59c3u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x35, MD5_F3, 7, 0, 0x8f0ccc92u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x36, MD5_F3, 7, 0, 0xffeff47du);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x37, MD5_F3, 7, 0, 0x85845dd1u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x38, MD5_F3, 7, 0, 0x6fa87e4fu);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x39, MD5_F3, 7, 0, 0xfe2ce6e0u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x3a, MD5_F3, 7, 0, 0xa3014314u);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x3b, MD5_F3, 7, 0, 0x4e0811a1u);
	MD5_UNROLLED_BLOCK(a, b, c, d, 0x3c, MD5_F3, 7, 0, 0xf7537e82u);
	MD5_UNROLLED_BLOCK(d, a, b, c, 0x3d, MD5_F3, 7, 0, 0xbd3af235u);
	MD5_UNROLLED_BLOCK(c, d, a, b, 0x3e, MD5_F3, 7, 0, 0x2ad7d2bbu);
	MD5_UNROLLED_BLOCK(b, c, d, a, 0x3f, MD5_F3, 7, 0, 0xeb86d391u);
#else // __MD5_UNROLL_LOOPS
	__declspec(align(16)) static const int r[64] = {
		7, 12, 17, 22, 7, 12, 17, 22, 7, 12, 17, 22, 7, 12, 17, 22,
		5,  9, 14, 20, 5,  9, 14, 20, 5,  9, 14, 20, 5,  9, 14, 20,
		4, 11, 16, 23, 4, 11, 16, 23, 4, 11, 16, 23, 4, 11, 16, 23,
		6, 10, 15, 21, 6, 10, 15, 21, 6, 10, 15, 21, 6, 10, 15, 21
	};
	__declspec(align(16)) static const uint32_t k[64] = {
		0xd76aa478, 0xe8c7b756, 0x242070db, 0xc1bdceee, 0xf57c0faf, 0x4787c62a, 0xa8304613, 0xfd469501,
		0x698098d8, 0x8b44f7af, 0xffff5bb1, 0x895cd7be, 0x6b901122, 0xfd987193, 0xa679438e, 0x49b40821,
		0xf61e2562, 0xc040b340, 0x265e5a51, 0xe9b6c7aa, 0xd62f105d, 0x02441453, 0xd8a1e681, 0xe7d3fbc8,
		0x21e1cde6, 0xc33707d6, 0xf4d50d87, 0x455a14ed, 0xa9e3e905, 0xfcefa3f8, 0x676f02d9, 0x8d2a4c8a,
		0xfffa3942, 0x8771f681, 0x6d9d6122, 0xfde5380c, 0xa4beea44, 0x4bdecfa9, 0xf6bb4b60, 0xbebfbc70,
		0x289b7ec6, 0xeaa127fa, 0xd4ef3085, 0x04881d05, 0xd9d4d039, 0xe6db99e5, 0x1fa27cf8, 0xc4ac5665,
		0xf4292244, 0x432aff97, 0xab9423a7, 0xfc93a039, 0x655b59c3, 0x8f0ccc92, 0xffeff47d, 0x85845dd1,
		0x6fa87e4f, 0xfe2ce6e0, 0xa3014314, 0x4e0811a1, 0xf7537e82, 0xbd3af235, 0x2ad7d2bb, 0xeb86d391
	};

	for(int i = 0; i < 16; i ++) {
		uint32_t f = (d ^ (b & (c ^ d)));
		uint32_t n_temp = d;
		d = c;
		c = b;
		b += n_LeftRot(a + f + k[i] + _p_data[i], r[i]);
		a = n_temp;
	}
	for(int i = 16; i < 32; i ++) {
		uint32_t f = (c ^ (d & (b ^ c)));
		uint32_t n_temp = d;
		d = c;
		c = b;
		b += n_LeftRot(a + f + k[i] + _p_data[(5 * i + 1) & 0x0f], r[i]);
		a = n_temp;
	}
	for(int i = 32; i < 48; i ++) {
		uint32_t f = b ^ c ^ d;
		uint32_t n_temp = d;
		d = c;
		c = b;
		b += n_LeftRot(a + f + k[i] + _p_data[(3 * i + 5) & 0x0f], r[i]);
		a = n_temp;
	}
	for(int i = 48; i < 64; i ++) {
		uint32_t f = c ^ (b | ~d);
		uint32_t n_temp = d;
		d = c;
		c = b;
		b += n_LeftRot(a + f + k[i] + _p_data[(7 * i) & 0x0f], r[i]);
		a = n_temp;
	}
#endif // __MD5_UNROLL_LOOPS
	// main processing loop

	m_n_hash[0] += a;
	m_n_hash[1] += b;
	m_n_hash[2] += c;
	m_n_hash[3] += d;

	m_n_total_length_bits += 512;
}

/**
 *	@brief calculate MD5 from data
 *
 *	Calculates final MD5 from the last block of data (@c less than 512 bits).
 *		After calling this function, operator []() can be used to read MD5 value.
 *
 *	@param[in] p_data contains final block of input data
 *	@param[in] n_size_bytes is size of input data in bytes, less or equal to 64
 *
 *	@note Once Process_FinalBlock() was called, Process_Block() and Process_FinalBlock()
 *		may not be called again. This is not checked in any way. To calculate MD5 of
 *		another data, it's necessary to create a new TMD5 object.
 */
void TMD5::Process_FinalBlock(const void *p_data, int n_size_bytes)
{
	_ASSERTE(n_size_bytes < 64);
	_ASSERTE(m_n_total_length_bits <= UINT64_MAX - n_size_bytes * 8);

	m_n_total_length_bits += n_size_bytes * 8;

	uint8_t p_temp_buffer[64]; // 64 x 8b = 512b
	memcpy(p_temp_buffer, p_data, n_size_bytes);
	if(n_size_bytes >= (512 - 64) / 8) {
		p_temp_buffer[n_size_bytes] = 0x80; // add a single 1 bit
		memset(p_temp_buffer + n_size_bytes + 1, 0x00, 63 - n_size_bytes); // follow by zeros
		Process_Block(p_temp_buffer, sizeof(p_temp_buffer) / sizeof(p_temp_buffer[0]));
		// data 1 0*

		m_n_total_length_bits -= 512; // do not count padding into the length

		memset(p_temp_buffer, 0x00, 64 - sizeof(uint64_t)); // begin with zeros
		memcpy(p_temp_buffer + 64 - sizeof(uint64_t),
			&m_n_total_length_bits, sizeof(uint64_t)); // append length
		Process_Block(p_temp_buffer, sizeof(p_temp_buffer) / sizeof(p_temp_buffer[0]));
		// 0* length
	} else {
		p_temp_buffer[n_size_bytes] = 0x80; // add a single 1 bit
		memset(p_temp_buffer + n_size_bytes + 1, 0x00, 63 - n_size_bytes); // follow by zeros
		memcpy(p_temp_buffer + 64 - sizeof(uint64_t),
			&m_n_total_length_bits, sizeof(uint64_t)); // append length
		Process_Block(p_temp_buffer, sizeof(p_temp_buffer) / sizeof(p_temp_buffer[0]));
		// data 1 0* length
	}
	// padd with one and zeros, append length
}

#ifndef __MD5_UNROLL_LOOPS
inline uint32_t TMD5::n_LeftRot(uint32_t n_x, int n_y)
{
	return (n_x << n_y) | (n_x >> (32 - n_y));
}
#endif // __MD5_UNROLL_LOOPS

inline uint32_t TMD5::n_ByteSwap32(uint32_t n_x)
{
	return (n_x >> 24) | ((n_x >> 8) & 0xff00) | ((n_x & 0xff00) << 8) | (n_x << 24);
}

const char *TMD5::p_s_ToString() const
{
	static char p_s_hash[33]; // 32 hexa digits ~ 128 bytes;
#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER >= 1400
	sprintf_s(p_s_hash, sizeof(p_s_hash), "%08x%08x%08x%08x",
		n_ByteSwap32(m_n_hash[0]), n_ByteSwap32(m_n_hash[1]),
		n_ByteSwap32(m_n_hash[2]), n_ByteSwap32(m_n_hash[3]));
#else //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
	sprintf(p_s_hash, "%08x%08x%08x%08x",
		n_ByteSwap32(m_n_hash[0]), n_ByteSwap32(m_n_hash[1]),
		n_ByteSwap32(m_n_hash[2]), n_ByteSwap32(m_n_hash[3])); // @t_odo - use sprintf_s
#endif //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
	// this is little endian hash, and we want to store it as big endian (as usual)
	p_s_hash[32] = 0;

	return p_s_hash;
}

/*
 *								=== ~TMD5 ===
 */

/*
 *								=== TSHA1 ===
 */

TSHA1::TSHA1()
	:m_n_total_length_bits(0)
{
	m_n_hash[0] = 0x67452301;
	m_n_hash[1] = 0xefcdab89;
	m_n_hash[2] = 0x98badcfe;
	m_n_hash[3] = 0x10325476;
	m_n_hash[4] = 0xc3d2e1f0;
}

TSHA1 &TSHA1::operator =(const TSHA1 &r_t_sha1)
{
	m_n_total_length_bits = r_t_sha1.m_n_total_length_bits;
	memcpy(m_n_hash, r_t_sha1.m_n_hash, 5 * sizeof(uint32_t));
	return *this;
}

bool TSHA1::operator ==(const TSHA1 &r_t_sha1) const
{
	return m_n_hash[0] == r_t_sha1.m_n_hash[0] && m_n_hash[1] == r_t_sha1.m_n_hash[1] &&
		   m_n_hash[2] == r_t_sha1.m_n_hash[2] && m_n_hash[3] == r_t_sha1.m_n_hash[3] &&
		   m_n_hash[4] == r_t_sha1.m_n_hash[4];
}

bool TSHA1::operator <(const TSHA1 &r_t_sha1) const
{
	return m_n_hash[0] < r_t_sha1.m_n_hash[0] || (m_n_hash[0] == r_t_sha1.m_n_hash[0] &&
		   (m_n_hash[1] < r_t_sha1.m_n_hash[1] || (m_n_hash[1] == r_t_sha1.m_n_hash[1] &&
		   (m_n_hash[2] < r_t_sha1.m_n_hash[2] || (m_n_hash[2] == r_t_sha1.m_n_hash[2] &&
		  ( m_n_hash[3] < r_t_sha1.m_n_hash[3] || (m_n_hash[3] == r_t_sha1.m_n_hash[3] &&
		   m_n_hash[4] < r_t_sha1.m_n_hash[4])))))));
}

#ifdef __SHA1_UNROLL_LOOPS

#define SHA1_F0(_b_,_c_,_d_) (((_d_) ^ ((_b_) & ((_c_) ^ (_d_)))) + 0x5A827999)
#define SHA1_F1(_b_,_c_,_d_) (((_b_) ^ (_c_) ^ (_d_)) + 0x6ED9EBA1)
#define SHA1_F2(_b_,_c_,_d_) ((((_b_) & (_c_)) | ((_d_) & ((_b_) | (_c_)))) + 0x8F1BBCDC)
#define SHA1_F3(_b_,_c_,_d_) (((_b_) ^ (_c_) ^ (_d_)) + 0xCA62C1D6)

#define SHA1_UNROLLED_BLOCK(_a_,_b_,_c_,_d_,_e_,i,_fun_) \
	do { \
		uint32_t f = _fun_(_b_,_c_,_d_); \
		uint32_t x = p_tmp_data[i]; \
		_e_ += n_leftrot_const(_a_, 5) + f + x; \
		_b_ = n_leftrot_const(_b_, 30); \
	} while(0)

#endif // __SHA1_UNROLL_LOOPS

void TSHA1::Process_Block(const void *p_data, int UNUSED(n_size_bytes))
{
	_ASSERTE(n_size_bytes == 64);
	_ASSERTE(m_n_total_length_bits <= UINT64_MAX - 512);

	const uint32_t *_p_data = (const uint32_t*)p_data;

#if defined(_WIN32) || defined(_WIN64)
	__declspec(align(16)) uint32_t p_tmp_data[80];

#if defined(_M_X64) || defined(_M_AMD64) || defined(_M_IA64) || defined(__x86_64) || defined(__amd64) || defined(__ia64)
	for(int i = 0; i < 16; i ++)
		p_tmp_data[i] = n_EndianConversion(_p_data[i]);
#else // _M_X64 || _M_AMD64 || _M_IA64 || __x86_64 || __amd64 || __ia64
	__asm {
		lea edx, p_tmp_data
		mov ecx, _p_data
		mov eax, [ecx]
		bswap eax
		mov [edx], eax
		mov eax, [ecx+4]
		bswap eax
		mov [edx+4], eax
		mov eax, [ecx+8]
		bswap eax
		mov [edx+8], eax
		mov eax, [ecx+12]
		bswap eax
		mov [edx+12], eax
		mov eax, [ecx+16]
		bswap eax
		mov [edx+16], eax
		mov eax, [ecx+20]
		bswap eax
		mov [edx+20], eax
		mov eax, [ecx+24]
		bswap eax
		mov [edx+24], eax
		mov eax, [ecx+28]
		bswap eax
		mov [edx+28], eax
		mov eax, [ecx+32]
		bswap eax
		mov [edx+32], eax
		mov eax, [ecx+36]
		bswap eax
		mov [edx+36], eax
		mov eax, [ecx+40]
		bswap eax
		mov [edx+40], eax
		mov eax, [ecx+44]
		bswap eax
		mov [edx+44], eax
		mov eax, [ecx+48]
		bswap eax
		mov [edx+48], eax
		mov eax, [ecx+52]
		bswap eax
		mov [edx+52], eax
		mov eax, [ecx+56]
		bswap eax
		mov [edx+56], eax
		mov eax, [ecx+60]
		bswap eax
		mov [edx+60], eax
	}
#endif // _M_X64 || _M_AMD64 || _M_IA64 || __x86_64 || __amd64 || __ia64
	// use bswap for endian conversion on x86
#else // _WIN32 || _WIN64
	uint32_t p_tmp_data[80];
	for(int i = 0; i < 16; i ++)
		p_tmp_data[i] = n_EndianConversion(_p_data[i]);
#endif // _WIN32 || _WIN64

	for(int i = 16; i < 80; i ++) {
		p_tmp_data[i] = n_leftrot_const(p_tmp_data[i - 3] ^ p_tmp_data[i - 8] ^
			p_tmp_data[i - 14] ^ p_tmp_data[i - 16], 1);
	}
	// expand data, convert endian

	uint32_t a = m_n_hash[0], b = m_n_hash[1],
		c = m_n_hash[2], d = m_n_hash[3], e = m_n_hash[4];

#ifdef __SHA1_UNROLL_LOOPS
	SHA1_UNROLLED_BLOCK(a, b, c, d, e,  0, SHA1_F0);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d,  1, SHA1_F0);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c,  2, SHA1_F0);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b,  3, SHA1_F0);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a,  4, SHA1_F0);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e,  5, SHA1_F0);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d,  6, SHA1_F0);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c,  7, SHA1_F0);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b,  8, SHA1_F0);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a,  9, SHA1_F0);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 10, SHA1_F0);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 11, SHA1_F0);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 12, SHA1_F0);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 13, SHA1_F0);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 14, SHA1_F0);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 15, SHA1_F0);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 16, SHA1_F0);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 17, SHA1_F0);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 18, SHA1_F0);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 19, SHA1_F0);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 20, SHA1_F1);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 21, SHA1_F1);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 22, SHA1_F1);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 23, SHA1_F1);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 24, SHA1_F1);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 25, SHA1_F1);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 26, SHA1_F1);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 27, SHA1_F1);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 28, SHA1_F1);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 29, SHA1_F1);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 30, SHA1_F1);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 31, SHA1_F1);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 32, SHA1_F1);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 33, SHA1_F1);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 34, SHA1_F1);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 35, SHA1_F1);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 36, SHA1_F1);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 37, SHA1_F1);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 38, SHA1_F1);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 39, SHA1_F1);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 40, SHA1_F2);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 41, SHA1_F2);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 42, SHA1_F2);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 43, SHA1_F2);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 44, SHA1_F2);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 45, SHA1_F2);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 46, SHA1_F2);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 47, SHA1_F2);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 48, SHA1_F2);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 49, SHA1_F2);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 50, SHA1_F2);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 51, SHA1_F2);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 52, SHA1_F2);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 53, SHA1_F2);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 54, SHA1_F2);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 55, SHA1_F2);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 56, SHA1_F2);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 57, SHA1_F2);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 58, SHA1_F2);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 59, SHA1_F2);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 60, SHA1_F3);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 61, SHA1_F3);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 62, SHA1_F3);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 63, SHA1_F3);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 64, SHA1_F3);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 65, SHA1_F3);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 66, SHA1_F3);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 67, SHA1_F3);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 68, SHA1_F3);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 69, SHA1_F3);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 70, SHA1_F3);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 71, SHA1_F3);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 72, SHA1_F3);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 73, SHA1_F3);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 74, SHA1_F3);
	SHA1_UNROLLED_BLOCK(a, b, c, d, e, 75, SHA1_F3);
	SHA1_UNROLLED_BLOCK(e, a, b, c, d, 76, SHA1_F3);
	SHA1_UNROLLED_BLOCK(d, e, a, b, c, 77, SHA1_F3);
	SHA1_UNROLLED_BLOCK(c, d, e, a, b, 78, SHA1_F3);
	SHA1_UNROLLED_BLOCK(b, c, d, e, a, 79, SHA1_F3);
#else // __SHA1_UNROLL_LOOPS
	for(int i = 0; i < 20; i ++) {
		uint32_t f = d ^ (b & (c ^ d));
		uint32_t n_temp = n_leftrot_const(a, 5) + f + e + 0x5A827999 + p_tmp_data[i];
		e = d;
		d = c;
		c = n_leftrot_const(b, 30);
		b = a;
		a = n_temp;
	}
	for(int i = 20; i < 40; i ++) {
		uint32_t f = b ^ c ^ d;
		uint32_t n_temp = n_leftrot_const(a, 5) + f + e + 0x6ED9EBA1 + p_tmp_data[i];
		e = d;
		d = c;
		c = n_leftrot_const(b, 30);
		b = a;
		a = n_temp;
	}
	for(int i = 40; i < 60; i ++) {
		uint32_t f = (b & c) | (d & (b | c));
		uint32_t n_temp = n_leftrot_const(a, 5) + f + e + 0x8F1BBCDC + p_tmp_data[i];
		e = d;
		d = c;
		c = n_leftrot_const(b, 30);
		b = a;
		a = n_temp;
	}
	for(int i = 60; i < 80; i ++) {
		uint32_t f = b ^ c ^ d;
		uint32_t n_temp = n_leftrot_const(a, 5) + f + e + 0xCA62C1D6 + p_tmp_data[i];
		e = d;
		d = c;
		c = n_leftrot_const(b, 30);
		b = a;
		a = n_temp;
	}
#endif // __SHA1_UNROLL_LOOPS
	// main processing loop

	m_n_hash[0] += a;
	m_n_hash[1] += b;
	m_n_hash[2] += c;
	m_n_hash[3] += d;
	m_n_hash[4] += e;

	m_n_total_length_bits += 512;
}

void TSHA1::Process_FinalBlock(const void *p_data, int n_size_bytes)
{
	_ASSERTE(n_size_bytes < 64);
	_ASSERTE(m_n_total_length_bits <= UINT64_MAX - n_size_bytes * 8);

	m_n_total_length_bits += n_size_bytes * 8;

	uint64_t l = m_n_total_length_bits;
	uint64_t n_big_l =
		((l >>  8) & 0xff000000) | ((l & 0xff000000) <<  8) |
		((l >> 24) & 0x00ff0000) | ((l & 0x00ff0000) << 24) |
		((l >> 40) & 0x0000ff00) | ((l & 0x0000ff00) << 40) |
		((l >> 56) & 0x000000ff) | ((l & 0x000000ff) << 56);
	// convert length to big endian

	uint8_t p_temp_buffer[64]; // 64 x 8b = 512b
	memcpy(p_temp_buffer, p_data, n_size_bytes);
	if(n_size_bytes >= (512 - 64) / 8) {
		p_temp_buffer[n_size_bytes] = 0x80; // add a single 1 bit
		memset(p_temp_buffer + n_size_bytes + 1, 0x00, 63 - n_size_bytes); // follow by zeros
		Process_Block(p_temp_buffer, sizeof(p_temp_buffer) / sizeof(p_temp_buffer[0]));
		// data 1 0*

		m_n_total_length_bits -= 512; // do not count padding into the length

		memset(p_temp_buffer, 0x00, 64 - sizeof(uint64_t)); // begin with zeros
		memcpy(p_temp_buffer + 64 - sizeof(uint64_t), &n_big_l, sizeof(uint64_t)); // append length
		Process_Block(p_temp_buffer, sizeof(p_temp_buffer) / sizeof(p_temp_buffer[0]));
		// 0* length
	} else {
		p_temp_buffer[n_size_bytes] = 0x80; // add a single 1 bit
		memset(p_temp_buffer + n_size_bytes + 1, 0x00, 63 - n_size_bytes); // follow by zeros
		memcpy(p_temp_buffer + 64 - sizeof(uint64_t), &n_big_l, sizeof(uint64_t)); // append length
		Process_Block(p_temp_buffer, sizeof(p_temp_buffer) / sizeof(p_temp_buffer[0]));
		// data 1 0* length
	}
	// padd with one and zeros, append length
}

/*inline uint32_t TSHA1::n_LeftRot(uint32_t x, int y)
{
	return (x << y) | (x >> (32 - y));
}*/

inline uint32_t TSHA1::n_EndianConversion(uint32_t x)
{
	//return htonl(x);
	return (x >> 24) | ((x >> 8) & 0xff00) | ((x & 0xff00) << 8) | (x << 24);
}

const char *TSHA1::p_s_ToString() const
{
	static char p_s_hash[41]; // 40 hexa digits ~ 192 bytes

#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER >= 1400
	sprintf_s(p_s_hash, sizeof(p_s_hash), "%08x%08x%08x%08x%08x",
		m_n_hash[0], m_n_hash[1], m_n_hash[2], m_n_hash[3], m_n_hash[4]);
#else //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
	sprintf(p_s_hash, "%08x%08x%08x%08x%08x",
		m_n_hash[0], m_n_hash[1], m_n_hash[2], m_n_hash[3], m_n_hash[4]); // @t_odo - use sprintf_s
#endif //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
	p_s_hash[40] = 0;

	return p_s_hash;
}

/*
 *								=== ~TSHA1 ===
 */

#if 0

/*
 *								=== TTinyMD5 ===
 */

/*
 *	struct TTinyMD5
 *		- class, implementing modified MD5 algorithm
 *		- the modification lies in reducing result to a single 32-bit integer
 *		  (purpose - to have better hashing function for strings, instead of
 *		  widely used character sum)
 *		- cooperates with CStreamHash<TTinyMD5, 64, false>
 */
struct TTinyMD5 {
protected:
	uint8_t m_n_hash[4];

public:
	TTinyMD5();

	TTinyMD5 operator =(const TTinyMD5 &r_t_md5);

	bool operator ==(const TTinyMD5 &r_t_md5) const;
	bool operator !=(const TTinyMD5 &r_t_md5) const;
	bool operator >(const TTinyMD5 &r_t_md5) const;
	bool operator <(const TTinyMD5 &r_t_md5) const;
	bool operator >=(const TTinyMD5 &r_t_md5) const;
	bool operator <=(const TTinyMD5 &r_t_md5) const;

	inline uint8_t &operator [](int n_index)
	{
		return m_n_hash[n_index];
	}

	inline uint8_t operator [](int n_index) const
	{
		return m_n_hash[n_index];
	}

	inline operator uint32_t() const
	{
		return m_n_hash[0] | (m_n_hash[1] << 8) | (m_n_hash[2] << 16) | (m_n_hash[3] << 24);
	}

	void Process_Block(const uint32_t *p_data);

protected:
	inline uint8_t n_LeftRot(uint8_t x, int y);
};

TTinyMD5::TTinyMD5()
{
	m_n_hash[0] = 0x67;
	m_n_hash[1] = 0xEF;
	m_n_hash[2] = 0x98;
	m_n_hash[3] = 0x10;
}

TTinyMD5 TTinyMD5::operator =(const TTinyMD5 &r_t_md5)
{
	memcpy(m_n_hash, r_t_md5.m_n_hash, 4 * sizeof(uint8_t));
	return *this;
}

bool TTinyMD5::operator ==(const TTinyMD5 &r_t_md5) const
{
	return m_n_hash[0] == r_t_md5.m_n_hash[0] && m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] == r_t_md5.m_n_hash[2] && m_n_hash[3] == r_t_md5.m_n_hash[3];
}

bool TTinyMD5::operator !=(const TTinyMD5 &r_t_md5) const
{
	return m_n_hash[0] != r_t_md5.m_n_hash[0] || m_n_hash[1] != r_t_md5.m_n_hash[1] ||
		   m_n_hash[2] != r_t_md5.m_n_hash[2] || m_n_hash[3] != r_t_md5.m_n_hash[3];
}

bool TTinyMD5::operator >(const TTinyMD5 &r_t_md5) const
{
	return m_n_hash[0] > r_t_md5.m_n_hash[0] || (m_n_hash[0] == r_t_md5.m_n_hash[0] &&
		   m_n_hash[1] > r_t_md5.m_n_hash[1] || (m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] > r_t_md5.m_n_hash[2] || (m_n_hash[2] == r_t_md5.m_n_hash[2] &&
		   m_n_hash[3] > r_t_md5.m_n_hash[3])));
}

bool TTinyMD5::operator <(const TTinyMD5 &r_t_md5) const
{
	return m_n_hash[0] < r_t_md5.m_n_hash[0] || (m_n_hash[0] == r_t_md5.m_n_hash[0] &&
		   m_n_hash[1] < r_t_md5.m_n_hash[1] || (m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] < r_t_md5.m_n_hash[2] || (m_n_hash[2] == r_t_md5.m_n_hash[2] &&
		   m_n_hash[3] < r_t_md5.m_n_hash[3])));
}

bool TTinyMD5::operator >=(const TTinyMD5 &r_t_md5) const
{
	return m_n_hash[0] >= r_t_md5.m_n_hash[0] || (m_n_hash[0] == r_t_md5.m_n_hash[0] &&
		   m_n_hash[1] >= r_t_md5.m_n_hash[1] || (m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] >= r_t_md5.m_n_hash[2] || (m_n_hash[2] == r_t_md5.m_n_hash[2] &&
		   m_n_hash[3] >= r_t_md5.m_n_hash[3])));
}

bool TTinyMD5::operator <=(const TTinyMD5 &r_t_md5) const
{
	return m_n_hash[0] <= r_t_md5.m_n_hash[0] || (m_n_hash[0] == r_t_md5.m_n_hash[0] &&
		   m_n_hash[1] <= r_t_md5.m_n_hash[1] || (m_n_hash[1] == r_t_md5.m_n_hash[1] &&
		   m_n_hash[2] <= r_t_md5.m_n_hash[2] || (m_n_hash[2] == r_t_md5.m_n_hash[2] &&
		   m_n_hash[3] <= r_t_md5.m_n_hash[3])));
}

void TTinyMD5::Process_Block(const uint32_t *p_data)
{
	const int r[16] = {2, 3, 5, 7, 1, 1, 2, 3, 3, 1, 4, 1, 1, 8, 6, 2};
	const uint8_t k[16] = {
		0xd7, 0xe8, 0x24, 0xc1,
		0xf5, 0x47, 0xa8, 0xfd,
		0x69, 0x8b, 0xff, 0x89,
		0x6b, 0xfd, 0xa6, 0x49
	};

	for(int n = 0; n < 2; n ++, p_data ++) {
		uint8_t a = m_n_hash[0], b = m_n_hash[1], c = m_n_hash[2], d = m_n_hash[3];

		for(int i = 0; i < 4; i ++) {
			uint8_t f = (d ^ (b & (c ^ d)));
			uint8_t n_temp = d;
			d = c;
			c = b;
			b += n_LeftRot(a + f + k[i] + ((const uint8_t*)p_data)[i], r[i]);
			a = n_temp;
		}
		for(int i = 4; i < 8; i ++) {
			uint8_t f = (c ^ (d & (b ^ c)));
			uint8_t n_temp = d;
			d = c;
			c = b;
			b += n_LeftRot(a + f + k[i] + ((const uint8_t*)p_data)[(5 * i + 1) & 0x03], r[i]);
			a = n_temp;
		}
		for(int i = 8; i < 12; i ++) {
			uint8_t f = b ^ c ^ d;
			uint8_t n_temp = d;
			d = c;
			c = b;
			b += n_LeftRot(a + f + k[i] + ((const uint8_t*)p_data)[(3 * i + 5) & 0x03], r[i]);
			a = n_temp;
		}
		for(int i = 12; i < 16; i ++) {
			uint8_t f = c ^ (b | ~d);
			uint8_t n_temp = d;
			d = c;
			c = b;
			b += n_LeftRot(a + f + k[i] + ((const uint8_t*)p_data)[(7 * i) & 0x03], r[i]);
			a = n_temp;
		}
		// main processing loop

		m_n_hash[0] += a;
		m_n_hash[1] += b;
		m_n_hash[2] += c;
		m_n_hash[3] += d;
	}
}

inline uint8_t TTinyMD5::n_LeftRot(uint8_t x, int y)
{
	return (x << y) | (x >> (8 - y));
}

/*
 *								=== ~TTinyMD5 ===
 */

#endif // 0

/*
 *								=== TTigerBase ===
 */

#ifndef UINT64_C
#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER < 1400
#define UINT64_C(x) x##U
#else // _MSC_VER && !__MWERKS__ && _MSC_VER < 1400
#define UINT64_C(x) x##ULL
#endif // _MSC_VER && !__MWERKS__ && _MSC_VER < 1400
#endif // !UINT64_C

static const uint64_t p_tiger_table[4 * 256] = {
	UINT64_C(0x02AAB17CF7E90C5E), UINT64_C(0xAC424B03E243A8EC), // 0, 1
	UINT64_C(0x72CD5BE30DD5FCD3), UINT64_C(0x6D019B93F6F97F3A), // 2, 3
	UINT64_C(0xCD9978FFD21F9193), UINT64_C(0x7573A1C9708029E2), // 4, 5
	UINT64_C(0xB164326B922A83C3), UINT64_C(0x46883EEE04915870), // 6, 7
	UINT64_C(0xEAACE3057103ECE6), UINT64_C(0xC54169B808A3535C), // 8, 9
	UINT64_C(0x4CE754918DDEC47C), UINT64_C(0x0AA2F4DFDC0DF40C), // 10, 11
	UINT64_C(0x10B76F18A74DBEFA), UINT64_C(0xC6CCB6235AD1AB6A), // 12, 13
	UINT64_C(0x13726121572FE2FF), UINT64_C(0x1A488C6F199D921E), // 14, 15
	UINT64_C(0x4BC9F9F4DA0007CA), UINT64_C(0x26F5E6F6E85241C7), // 16, 17
	UINT64_C(0x859079DBEA5947B6), UINT64_C(0x4F1885C5C99E8C92), // 18, 19
	UINT64_C(0xD78E761EA96F864B), UINT64_C(0x8E36428C52B5C17D), // 20, 21
	UINT64_C(0x69CF6827373063C1), UINT64_C(0xB607C93D9BB4C56E), // 22, 23
	UINT64_C(0x7D820E760E76B5EA), UINT64_C(0x645C9CC6F07FDC42), // 24, 25
	UINT64_C(0xBF38A078243342E0), UINT64_C(0x5F6B343C9D2E7D04), // 26, 27
	UINT64_C(0xF2C28AEB600B0EC6), UINT64_C(0x6C0ED85F7254BCAC), // 28, 29
	UINT64_C(0x71592281A4DB4FE5), UINT64_C(0x1967FA69CE0FED9F), // 30, 31
	UINT64_C(0xFD5293F8B96545DB), UINT64_C(0xC879E9D7F2A7600B), // 32, 33
	UINT64_C(0x860248920193194E), UINT64_C(0xA4F9533B2D9CC0B3), // 34, 35
	UINT64_C(0x9053836C15957613), UINT64_C(0xDB6DCF8AFC357BF1), // 36, 37
	UINT64_C(0x18BEEA7A7A370F57), UINT64_C(0x037117CA50B99066), // 38, 39
	UINT64_C(0x6AB30A9774424A35), UINT64_C(0xF4E92F02E325249B), // 40, 41
	UINT64_C(0x7739DB07061CCAE1), UINT64_C(0xD8F3B49CECA42A05), // 42, 43
	UINT64_C(0xBD56BE3F51382F73), UINT64_C(0x45FAED5843B0BB28), // 44, 45
	UINT64_C(0x1C813D5C11BF1F83), UINT64_C(0x8AF0E4B6D75FA169), // 46, 47
	UINT64_C(0x33EE18A487AD9999), UINT64_C(0x3C26E8EAB1C94410), // 48, 49
	UINT64_C(0xB510102BC0A822F9), UINT64_C(0x141EEF310CE6123B), // 50, 51
	UINT64_C(0xFC65B90059DDB154), UINT64_C(0xE0158640C5E0E607), // 52, 53
	UINT64_C(0x884E079826C3A3CF), UINT64_C(0x930D0D9523C535FD), // 54, 55
	UINT64_C(0x35638D754E9A2B00), UINT64_C(0x4085FCCF40469DD5), // 56, 57
	UINT64_C(0xC4B17AD28BE23A4C), UINT64_C(0xCAB2F0FC6A3E6A2E), // 58, 59
	UINT64_C(0x2860971A6B943FCD), UINT64_C(0x3DDE6EE212E30446), // 60, 61
	UINT64_C(0x6222F32AE01765AE), UINT64_C(0x5D550BB5478308FE), // 62, 63
	UINT64_C(0xA9EFA98DA0EDA22A), UINT64_C(0xC351A71686C40DA7), // 64, 65
	UINT64_C(0x1105586D9C867C84), UINT64_C(0xDCFFEE85FDA22853), // 66, 67
	UINT64_C(0xCCFBD0262C5EEF76), UINT64_C(0xBAF294CB8990D201), // 68, 69
	UINT64_C(0xE69464F52AFAD975), UINT64_C(0x94B013AFDF133E14), // 70, 71
	UINT64_C(0x06A7D1A32823C958), UINT64_C(0x6F95FE5130F61119), // 72, 73
	UINT64_C(0xD92AB34E462C06C0), UINT64_C(0xED7BDE33887C71D2), // 74, 75
	UINT64_C(0x79746D6E6518393E), UINT64_C(0x5BA419385D713329), // 76, 77
	UINT64_C(0x7C1BA6B948A97564), UINT64_C(0x31987C197BFDAC67), // 78, 79
	UINT64_C(0xDE6C23C44B053D02), UINT64_C(0x581C49FED002D64D), // 80, 81
	UINT64_C(0xDD474D6338261571), UINT64_C(0xAA4546C3E473D062), // 82, 83
	UINT64_C(0x928FCE349455F860), UINT64_C(0x48161BBACAAB94D9), // 84, 85
	UINT64_C(0x63912430770E6F68), UINT64_C(0x6EC8A5E602C6641C), // 86, 87
	UINT64_C(0x87282515337DDD2B), UINT64_C(0x2CDA6B42034B701B), // 88, 89
	UINT64_C(0xB03D37C181CB096D), UINT64_C(0xE108438266C71C6F), // 90, 91
	UINT64_C(0x2B3180C7EB51B255), UINT64_C(0xDF92B82F96C08BBC), // 92, 93
	UINT64_C(0x5C68C8C0A632F3BA), UINT64_C(0x5504CC861C3D0556), // 94, 95
	UINT64_C(0xABBFA4E55FB26B8F), UINT64_C(0x41848B0AB3BACEB4), // 96, 97
	UINT64_C(0xB334A273AA445D32), UINT64_C(0xBCA696F0A85AD881), // 98, 99
	UINT64_C(0x24F6EC65B528D56C), UINT64_C(0x0CE1512E90F4524A), // 100, 101
	UINT64_C(0x4E9DD79D5506D35A), UINT64_C(0x258905FAC6CE9779), // 102, 103
	UINT64_C(0x2019295B3E109B33), UINT64_C(0xF8A9478B73A054CC), // 104, 105
	UINT64_C(0x2924F2F934417EB0), UINT64_C(0x3993357D536D1BC4), // 106, 107
	UINT64_C(0x38A81AC21DB6FF8B), UINT64_C(0x47C4FBF17D6016BF), // 108, 109
	UINT64_C(0x1E0FAADD7667E3F5), UINT64_C(0x7ABCFF62938BEB96), // 110, 111
	UINT64_C(0xA78DAD948FC179C9), UINT64_C(0x8F1F98B72911E50D), // 112, 113
	UINT64_C(0x61E48EAE27121A91), UINT64_C(0x4D62F7AD31859808), // 114, 115
	UINT64_C(0xECEBA345EF5CEAEB), UINT64_C(0xF5CEB25EBC9684CE), // 116, 117
	UINT64_C(0xF633E20CB7F76221), UINT64_C(0xA32CDF06AB8293E4), // 118, 119
	UINT64_C(0x985A202CA5EE2CA4), UINT64_C(0xCF0B8447CC8A8FB1), // 120, 121
	UINT64_C(0x9F765244979859A3), UINT64_C(0xA8D516B1A1240017), // 122, 123
	UINT64_C(0x0BD7BA3EBB5DC726), UINT64_C(0xE54BCA55B86ADB39), // 124, 125
	UINT64_C(0x1D7A3AFD6C478063), UINT64_C(0x519EC608E7669EDD), // 126, 127
	UINT64_C(0x0E5715A2D149AA23), UINT64_C(0x177D4571848FF194), // 128, 129
	UINT64_C(0xEEB55F3241014C22), UINT64_C(0x0F5E5CA13A6E2EC2), // 130, 131
	UINT64_C(0x8029927B75F5C361), UINT64_C(0xAD139FABC3D6E436), // 132, 133
	UINT64_C(0x0D5DF1A94CCF402F), UINT64_C(0x3E8BD948BEA5DFC8), // 134, 135
	UINT64_C(0xA5A0D357BD3FF77E), UINT64_C(0xA2D12E251F74F645), // 136, 137
	UINT64_C(0x66FD9E525E81A082), UINT64_C(0x2E0C90CE7F687A49), // 138, 139
	UINT64_C(0xC2E8BCBEBA973BC5), UINT64_C(0x000001BCE509745F), // 140, 141
	UINT64_C(0x423777BBE6DAB3D6), UINT64_C(0xD1661C7EAEF06EB5), // 142, 143
	UINT64_C(0xA1781F354DAACFD8), UINT64_C(0x2D11284A2B16AFFC), // 144, 145
	UINT64_C(0xF1FC4F67FA891D1F), UINT64_C(0x73ECC25DCB920ADA), // 146, 147
	UINT64_C(0xAE610C22C2A12651), UINT64_C(0x96E0A810D356B78A), // 148, 149
	UINT64_C(0x5A9A381F2FE7870F), UINT64_C(0xD5AD62EDE94E5530), // 150, 151
	UINT64_C(0xD225E5E8368D1427), UINT64_C(0x65977B70C7AF4631), // 152, 153
	UINT64_C(0x99F889B2DE39D74F), UINT64_C(0x233F30BF54E1D143), // 154, 155
	UINT64_C(0x9A9675D3D9A63C97), UINT64_C(0x5470554FF334F9A8), // 156, 157
	UINT64_C(0x166ACB744A4F5688), UINT64_C(0x70C74CAAB2E4AEAD), // 158, 159
	UINT64_C(0xF0D091646F294D12), UINT64_C(0x57B82A89684031D1), // 160, 161
	UINT64_C(0xEFD95A5A61BE0B6B), UINT64_C(0x2FBD12E969F2F29A), // 162, 163
	UINT64_C(0x9BD37013FEFF9FE8), UINT64_C(0x3F9B0404D6085A06), // 164, 165
	UINT64_C(0x4940C1F3166CFE15), UINT64_C(0x09542C4DCDF3DEFB), // 166, 167
	UINT64_C(0xB4C5218385CD5CE3), UINT64_C(0xC935B7DC4462A641), // 168, 169
	UINT64_C(0x3417F8A68ED3B63F), UINT64_C(0xB80959295B215B40), // 170, 171
	UINT64_C(0xF99CDAEF3B8C8572), UINT64_C(0x018C0614F8FCB95D), // 172, 173
	UINT64_C(0x1B14ACCD1A3ACDF3), UINT64_C(0x84D471F200BB732D), // 174, 175
	UINT64_C(0xC1A3110E95E8DA16), UINT64_C(0x430A7220BF1A82B8), // 176, 177
	UINT64_C(0xB77E090D39DF210E), UINT64_C(0x5EF4BD9F3CD05E9D), // 178, 179
	UINT64_C(0x9D4FF6DA7E57A444), UINT64_C(0xDA1D60E183D4A5F8), // 180, 181
	UINT64_C(0xB287C38417998E47), UINT64_C(0xFE3EDC121BB31886), // 182, 183
	UINT64_C(0xC7FE3CCC980CCBEF), UINT64_C(0xE46FB590189BFD03), // 184, 185
	UINT64_C(0x3732FD469A4C57DC), UINT64_C(0x7EF700A07CF1AD65), // 186, 187
	UINT64_C(0x59C64468A31D8859), UINT64_C(0x762FB0B4D45B61F6), // 188, 189
	UINT64_C(0x155BAED099047718), UINT64_C(0x68755E4C3D50BAA6), // 190, 191
	UINT64_C(0xE9214E7F22D8B4DF), UINT64_C(0x2ADDBF532EAC95F4), // 192, 193
	UINT64_C(0x32AE3909B4BD0109), UINT64_C(0x834DF537B08E3450), // 194, 195
	UINT64_C(0xFA209DA84220728D), UINT64_C(0x9E691D9B9EFE23F7), // 196, 197
	UINT64_C(0x0446D288C4AE8D7F), UINT64_C(0x7B4CC524E169785B), // 198, 199
	UINT64_C(0x21D87F0135CA1385), UINT64_C(0xCEBB400F137B8AA5), // 200, 201
	UINT64_C(0x272E2B66580796BE), UINT64_C(0x3612264125C2B0DE), // 202, 203
	UINT64_C(0x057702BDAD1EFBB2), UINT64_C(0xD4BABB8EACF84BE9), // 204, 205
	UINT64_C(0x91583139641BC67B), UINT64_C(0x8BDC2DE08036E024), // 206, 207
	UINT64_C(0x603C8156F49F68ED), UINT64_C(0xF7D236F7DBEF5111), // 208, 209
	UINT64_C(0x9727C4598AD21E80), UINT64_C(0xA08A0896670A5FD7), // 210, 211
	UINT64_C(0xCB4A8F4309EBA9CB), UINT64_C(0x81AF564B0F7036A1), // 212, 213
	UINT64_C(0xC0B99AA778199ABD), UINT64_C(0x959F1EC83FC8E952), // 214, 215
	UINT64_C(0x8C505077794A81B9), UINT64_C(0x3ACAAF8F056338F0), // 216, 217
	UINT64_C(0x07B43F50627A6778), UINT64_C(0x4A44AB49F5ECCC77), // 218, 219
	UINT64_C(0x3BC3D6E4B679EE98), UINT64_C(0x9CC0D4D1CF14108C), // 220, 221
	UINT64_C(0x4406C00B206BC8A0), UINT64_C(0x82A18854C8D72D89), // 222, 223
	UINT64_C(0x67E366B35C3C432C), UINT64_C(0xB923DD61102B37F2), // 224, 225
	UINT64_C(0x56AB2779D884271D), UINT64_C(0xBE83E1B0FF1525AF), // 226, 227
	UINT64_C(0xFB7C65D4217E49A9), UINT64_C(0x6BDBE0E76D48E7D4), // 228, 229
	UINT64_C(0x08DF828745D9179E), UINT64_C(0x22EA6A9ADD53BD34), // 230, 231
	UINT64_C(0xE36E141C5622200A), UINT64_C(0x7F805D1B8CB750EE), // 232, 233
	UINT64_C(0xAFE5C7A59F58E837), UINT64_C(0xE27F996A4FB1C23C), // 234, 235
	UINT64_C(0xD3867DFB0775F0D0), UINT64_C(0xD0E673DE6E88891A), // 236, 237
	UINT64_C(0x123AEB9EAFB86C25), UINT64_C(0x30F1D5D5C145B895), // 238, 239
	UINT64_C(0xBB434A2DEE7269E7), UINT64_C(0x78CB67ECF931FA38), // 240, 241
	UINT64_C(0xF33B0372323BBF9C), UINT64_C(0x52D66336FB279C74), // 242, 243
	UINT64_C(0x505F33AC0AFB4EAA), UINT64_C(0xE8A5CD99A2CCE187), // 244, 245
	UINT64_C(0x534974801E2D30BB), UINT64_C(0x8D2D5711D5876D90), // 246, 247
	UINT64_C(0x1F1A412891BC038E), UINT64_C(0xD6E2E71D82E56648), // 248, 249
	UINT64_C(0x74036C3A497732B7), UINT64_C(0x89B67ED96361F5AB), // 250, 251
	UINT64_C(0xFFED95D8F1EA02A2), UINT64_C(0xE72B3BD61464D43D), // 252, 253
	UINT64_C(0xA6300F170BDC4820), UINT64_C(0xEBC18760ED78A77A), // 254, 255
	UINT64_C(0xE6A6BE5A05A12138), UINT64_C(0xB5A122A5B4F87C98), // 256, 257
	UINT64_C(0x563C6089140B6990), UINT64_C(0x4C46CB2E391F5DD5), // 258, 259
	UINT64_C(0xD932ADDBC9B79434), UINT64_C(0x08EA70E42015AFF5), // 260, 261
	UINT64_C(0xD765A6673E478CF1), UINT64_C(0xC4FB757EAB278D99), // 262, 263
	UINT64_C(0xDF11C6862D6E0692), UINT64_C(0xDDEB84F10D7F3B16), // 264, 265
	UINT64_C(0x6F2EF604A665EA04), UINT64_C(0x4A8E0F0FF0E0DFB3), // 266, 267
	UINT64_C(0xA5EDEEF83DBCBA51), UINT64_C(0xFC4F0A2A0EA4371E), // 268, 269
	UINT64_C(0xE83E1DA85CB38429), UINT64_C(0xDC8FF882BA1B1CE2), // 270, 271
	UINT64_C(0xCD45505E8353E80D), UINT64_C(0x18D19A00D4DB0717), // 272, 273
	UINT64_C(0x34A0CFEDA5F38101), UINT64_C(0x0BE77E518887CAF2), // 274, 275
	UINT64_C(0x1E341438B3C45136), UINT64_C(0xE05797F49089CCF9), // 276, 277
	UINT64_C(0xFFD23F9DF2591D14), UINT64_C(0x543DDA228595C5CD), // 278, 279
	UINT64_C(0x661F81FD99052A33), UINT64_C(0x8736E641DB0F7B76), // 280, 281
	UINT64_C(0x15227725418E5307), UINT64_C(0xE25F7F46162EB2FA), // 282, 283
	UINT64_C(0x48A8B2126C13D9FE), UINT64_C(0xAFDC541792E76EEA), // 284, 285
	UINT64_C(0x03D912BFC6D1898F), UINT64_C(0x31B1AAFA1B83F51B), // 286, 287
	UINT64_C(0xF1AC2796E42AB7D9), UINT64_C(0x40A3A7D7FCD2EBAC), // 288, 289
	UINT64_C(0x1056136D0AFBBCC5), UINT64_C(0x7889E1DD9A6D0C85), // 290, 291
	UINT64_C(0xD33525782A7974AA), UINT64_C(0xA7E25D09078AC09B), // 292, 293
	UINT64_C(0xBD4138B3EAC6EDD0), UINT64_C(0x920ABFBE71EB9E70), // 294, 295
	UINT64_C(0xA2A5D0F54FC2625C), UINT64_C(0xC054E36B0B1290A3), // 296, 297
	UINT64_C(0xF6DD59FF62FE932B), UINT64_C(0x3537354511A8AC7D), // 298, 299
	UINT64_C(0xCA845E9172FADCD4), UINT64_C(0x84F82B60329D20DC), // 300, 301
	UINT64_C(0x79C62CE1CD672F18), UINT64_C(0x8B09A2ADD124642C), // 302, 303
	UINT64_C(0xD0C1E96A19D9E726), UINT64_C(0x5A786A9B4BA9500C), // 304, 305
	UINT64_C(0x0E020336634C43F3), UINT64_C(0xC17B474AEB66D822), // 306, 307
	UINT64_C(0x6A731AE3EC9BAAC2), UINT64_C(0x8226667AE0840258), // 308, 309
	UINT64_C(0x67D4567691CAECA5), UINT64_C(0x1D94155C4875ADB5), // 310, 311
	UINT64_C(0x6D00FD985B813FDF), UINT64_C(0x51286EFCB774CD06), // 312, 313
	UINT64_C(0x5E8834471FA744AF), UINT64_C(0xF72CA0AEE761AE2E), // 314, 315
	UINT64_C(0xBE40E4CDAEE8E09A), UINT64_C(0xE9970BBB5118F665), // 316, 317
	UINT64_C(0x726E4BEB33DF1964), UINT64_C(0x703B000729199762), // 318, 319
	UINT64_C(0x4631D816F5EF30A7), UINT64_C(0xB880B5B51504A6BE), // 320, 321
	UINT64_C(0x641793C37ED84B6C), UINT64_C(0x7B21ED77F6E97D96), // 322, 323
	UINT64_C(0x776306312EF96B73), UINT64_C(0xAE528948E86FF3F4), // 324, 325
	UINT64_C(0x53DBD7F286A3F8F8), UINT64_C(0x16CADCE74CFC1063), // 326, 327
	UINT64_C(0x005C19BDFA52C6DD), UINT64_C(0x68868F5D64D46AD3), // 328, 329
	UINT64_C(0x3A9D512CCF1E186A), UINT64_C(0x367E62C2385660AE), // 330, 331
	UINT64_C(0xE359E7EA77DCB1D7), UINT64_C(0x526C0773749ABE6E), // 332, 333
	UINT64_C(0x735AE5F9D09F734B), UINT64_C(0x493FC7CC8A558BA8), // 334, 335
	UINT64_C(0xB0B9C1533041AB45), UINT64_C(0x321958BA470A59BD), // 336, 337
	UINT64_C(0x852DB00B5F46C393), UINT64_C(0x91209B2BD336B0E5), // 338, 339
	UINT64_C(0x6E604F7D659EF19F), UINT64_C(0xB99A8AE2782CCB24), // 340, 341
	UINT64_C(0xCCF52AB6C814C4C7), UINT64_C(0x4727D9AFBE11727B), // 342, 343
	UINT64_C(0x7E950D0C0121B34D), UINT64_C(0x756F435670AD471F), // 344, 345
	UINT64_C(0xF5ADD442615A6849), UINT64_C(0x4E87E09980B9957A), // 346, 347
	UINT64_C(0x2ACFA1DF50AEE355), UINT64_C(0xD898263AFD2FD556), // 348, 349
	UINT64_C(0xC8F4924DD80C8FD6), UINT64_C(0xCF99CA3D754A173A), // 350, 351
	UINT64_C(0xFE477BACAF91BF3C), UINT64_C(0xED5371F6D690C12D), // 352, 353
	UINT64_C(0x831A5C285E687094), UINT64_C(0xC5D3C90A3708A0A4), // 354, 355
	UINT64_C(0x0F7F903717D06580), UINT64_C(0x19F9BB13B8FDF27F), // 356, 357
	UINT64_C(0xB1BD6F1B4D502843), UINT64_C(0x1C761BA38FFF4012), // 358, 359
	UINT64_C(0x0D1530C4E2E21F3B), UINT64_C(0x8943CE69A7372C8A), // 360, 361
	UINT64_C(0xE5184E11FEB5CE66), UINT64_C(0x618BDB80BD736621), // 362, 363
	UINT64_C(0x7D29BAD68B574D0B), UINT64_C(0x81BB613E25E6FE5B), // 364, 365
	UINT64_C(0x071C9C10BC07913F), UINT64_C(0xC7BEEB7909AC2D97), // 366, 367
	UINT64_C(0xC3E58D353BC5D757), UINT64_C(0xEB017892F38F61E8), // 368, 369
	UINT64_C(0xD4EFFB9C9B1CC21A), UINT64_C(0x99727D26F494F7AB), // 370, 371
	UINT64_C(0xA3E063A2956B3E03), UINT64_C(0x9D4A8B9A4AA09C30), // 372, 373
	UINT64_C(0x3F6AB7D500090FB4), UINT64_C(0x9CC0F2A057268AC0), // 374, 375
	UINT64_C(0x3DEE9D2DEDBF42D1), UINT64_C(0x330F49C87960A972), // 376, 377
	UINT64_C(0xC6B2720287421B41), UINT64_C(0x0AC59EC07C00369C), // 378, 379
	UINT64_C(0xEF4EAC49CB353425), UINT64_C(0xF450244EEF0129D8), // 380, 381
	UINT64_C(0x8ACC46E5CAF4DEB6), UINT64_C(0x2FFEAB63989263F7), // 382, 383
	UINT64_C(0x8F7CB9FE5D7A4578), UINT64_C(0x5BD8F7644E634635), // 384, 385
	UINT64_C(0x427A7315BF2DC900), UINT64_C(0x17D0C4AA2125261C), // 386, 387
	UINT64_C(0x3992486C93518E50), UINT64_C(0xB4CBFEE0A2D7D4C3), // 388, 389
	UINT64_C(0x7C75D6202C5DDD8D), UINT64_C(0xDBC295D8E35B6C61), // 390, 391
	UINT64_C(0x60B369D302032B19), UINT64_C(0xCE42685FDCE44132), // 392, 393
	UINT64_C(0x06F3DDB9DDF65610), UINT64_C(0x8EA4D21DB5E148F0), // 394, 395
	UINT64_C(0x20B0FCE62FCD496F), UINT64_C(0x2C1B912358B0EE31), // 396, 397
	UINT64_C(0xB28317B818F5A308), UINT64_C(0xA89C1E189CA6D2CF), // 398, 399
	UINT64_C(0x0C6B18576AAADBC8), UINT64_C(0xB65DEAA91299FAE3), // 400, 401
	UINT64_C(0xFB2B794B7F1027E7), UINT64_C(0x04E4317F443B5BEB), // 402, 403
	UINT64_C(0x4B852D325939D0A6), UINT64_C(0xD5AE6BEEFB207FFC), // 404, 405
	UINT64_C(0x309682B281C7D374), UINT64_C(0xBAE309A194C3B475), // 406, 407
	UINT64_C(0x8CC3F97B13B49F05), UINT64_C(0x98A9422FF8293967), // 408, 409
	UINT64_C(0x244B16B01076FF7C), UINT64_C(0xF8BF571C663D67EE), // 410, 411
	UINT64_C(0x1F0D6758EEE30DA1), UINT64_C(0xC9B611D97ADEB9B7), // 412, 413
	UINT64_C(0xB7AFD5887B6C57A2), UINT64_C(0x6290AE846B984FE1), // 414, 415
	UINT64_C(0x94DF4CDEACC1A5FD), UINT64_C(0x058A5BD1C5483AFF), // 416, 417
	UINT64_C(0x63166CC142BA3C37), UINT64_C(0x8DB8526EB2F76F40), // 418, 419
	UINT64_C(0xE10880036F0D6D4E), UINT64_C(0x9E0523C9971D311D), // 420, 421
	UINT64_C(0x45EC2824CC7CD691), UINT64_C(0x575B8359E62382C9), // 422, 423
	UINT64_C(0xFA9E400DC4889995), UINT64_C(0xD1823ECB45721568), // 424, 425
	UINT64_C(0xDAFD983B8206082F), UINT64_C(0xAA7D29082386A8CB), // 426, 427
	UINT64_C(0x269FCD4403B87588), UINT64_C(0x1B91F5F728BDD1E0), // 428, 429
	UINT64_C(0xE4669F39040201F6), UINT64_C(0x7A1D7C218CF04ADE), // 430, 431
	UINT64_C(0x65623C29D79CE5CE), UINT64_C(0x2368449096C00BB1), // 432, 433
	UINT64_C(0xAB9BF1879DA503BA), UINT64_C(0xBC23ECB1A458058E), // 434, 435
	UINT64_C(0x9A58DF01BB401ECC), UINT64_C(0xA070E868A85F143D), // 436, 437
	UINT64_C(0x4FF188307DF2239E), UINT64_C(0x14D565B41A641183), // 438, 439
	UINT64_C(0xEE13337452701602), UINT64_C(0x950E3DCF3F285E09), // 440, 441
	UINT64_C(0x59930254B9C80953), UINT64_C(0x3BF299408930DA6D), // 442, 443
	UINT64_C(0xA955943F53691387), UINT64_C(0xA15EDECAA9CB8784), // 444, 445
	UINT64_C(0x29142127352BE9A0), UINT64_C(0x76F0371FFF4E7AFB), // 446, 447
	UINT64_C(0x0239F450274F2228), UINT64_C(0xBB073AF01D5E868B), // 448, 449
	UINT64_C(0xBFC80571C10E96C1), UINT64_C(0xD267088568222E23), // 450, 451
	UINT64_C(0x9671A3D48E80B5B0), UINT64_C(0x55B5D38AE193BB81), // 452, 453
	UINT64_C(0x693AE2D0A18B04B8), UINT64_C(0x5C48B4ECADD5335F), // 454, 455
	UINT64_C(0xFD743B194916A1CA), UINT64_C(0x2577018134BE98C4), // 456, 457
	UINT64_C(0xE77987E83C54A4AD), UINT64_C(0x28E11014DA33E1B9), // 458, 459
	UINT64_C(0x270CC59E226AA213), UINT64_C(0x71495F756D1A5F60), // 460, 461
	UINT64_C(0x9BE853FB60AFEF77), UINT64_C(0xADC786A7F7443DBF), // 462, 463
	UINT64_C(0x0904456173B29A82), UINT64_C(0x58BC7A66C232BD5E), // 464, 465
	UINT64_C(0xF306558C673AC8B2), UINT64_C(0x41F639C6B6C9772A), // 466, 467
	UINT64_C(0x216DEFE99FDA35DA), UINT64_C(0x11640CC71C7BE615), // 468, 469
	UINT64_C(0x93C43694565C5527), UINT64_C(0xEA038E6246777839), // 470, 471
	UINT64_C(0xF9ABF3CE5A3E2469), UINT64_C(0x741E768D0FD312D2), // 472, 473
	UINT64_C(0x0144B883CED652C6), UINT64_C(0xC20B5A5BA33F8552), // 474, 475
	UINT64_C(0x1AE69633C3435A9D), UINT64_C(0x97A28CA4088CFDEC), // 476, 477
	UINT64_C(0x8824A43C1E96F420), UINT64_C(0x37612FA66EEEA746), // 478, 479
	UINT64_C(0x6B4CB165F9CF0E5A), UINT64_C(0x43AA1C06A0ABFB4A), // 480, 481
	UINT64_C(0x7F4DC26FF162796B), UINT64_C(0x6CBACC8E54ED9B0F), // 482, 483
	UINT64_C(0xA6B7FFEFD2BB253E), UINT64_C(0x2E25BC95B0A29D4F), // 484, 485
	UINT64_C(0x86D6A58BDEF1388C), UINT64_C(0xDED74AC576B6F054), // 486, 487
	UINT64_C(0x8030BDBC2B45805D), UINT64_C(0x3C81AF70E94D9289), // 488, 489
	UINT64_C(0x3EFF6DDA9E3100DB), UINT64_C(0xB38DC39FDFCC8847), // 490, 491
	UINT64_C(0x123885528D17B87E), UINT64_C(0xF2DA0ED240B1B642), // 492, 493
	UINT64_C(0x44CEFADCD54BF9A9), UINT64_C(0x1312200E433C7EE6), // 494, 495
	UINT64_C(0x9FFCC84F3A78C748), UINT64_C(0xF0CD1F72248576BB), // 496, 497
	UINT64_C(0xEC6974053638CFE4), UINT64_C(0x2BA7B67C0CEC4E4C), // 498, 499
	UINT64_C(0xAC2F4DF3E5CE32ED), UINT64_C(0xCB33D14326EA4C11), // 500, 501
	UINT64_C(0xA4E9044CC77E58BC), UINT64_C(0x5F513293D934FCEF), // 502, 503
	UINT64_C(0x5DC9645506E55444), UINT64_C(0x50DE418F317DE40A), // 504, 505
	UINT64_C(0x388CB31A69DDE259), UINT64_C(0x2DB4A83455820A86), // 506, 507
	UINT64_C(0x9010A91E84711AE9), UINT64_C(0x4DF7F0B7B1498371), // 508, 509
	UINT64_C(0xD62A2EABC0977179), UINT64_C(0x22FAC097AA8D5C0E), // 510, 511
	UINT64_C(0xF49FCC2FF1DAF39B), UINT64_C(0x487FD5C66FF29281), // 512, 513
	UINT64_C(0xE8A30667FCDCA83F), UINT64_C(0x2C9B4BE3D2FCCE63), // 514, 515
	UINT64_C(0xDA3FF74B93FBBBC2), UINT64_C(0x2FA165D2FE70BA66), // 516, 517
	UINT64_C(0xA103E279970E93D4), UINT64_C(0xBECDEC77B0E45E71), // 518, 519
	UINT64_C(0xCFB41E723985E497), UINT64_C(0xB70AAA025EF75017), // 520, 521
	UINT64_C(0xD42309F03840B8E0), UINT64_C(0x8EFC1AD035898579), // 522, 523
	UINT64_C(0x96C6920BE2B2ABC5), UINT64_C(0x66AF4163375A9172), // 524, 525
	UINT64_C(0x2174ABDCCA7127FB), UINT64_C(0xB33CCEA64A72FF41), // 526, 527
	UINT64_C(0xF04A4933083066A5), UINT64_C(0x8D970ACDD7289AF5), // 528, 529
	UINT64_C(0x8F96E8E031C8C25E), UINT64_C(0xF3FEC02276875D47), // 530, 531
	UINT64_C(0xEC7BF310056190DD), UINT64_C(0xF5ADB0AEBB0F1491), // 532, 533
	UINT64_C(0x9B50F8850FD58892), UINT64_C(0x4975488358B74DE8), // 534, 535
	UINT64_C(0xA3354FF691531C61), UINT64_C(0x0702BBE481D2C6EE), // 536, 537
	UINT64_C(0x89FB24057DEDED98), UINT64_C(0xAC3075138596E902), // 538, 539
	UINT64_C(0x1D2D3580172772ED), UINT64_C(0xEB738FC28E6BC30D), // 540, 541
	UINT64_C(0x5854EF8F63044326), UINT64_C(0x9E5C52325ADD3BBE), // 542, 543
	UINT64_C(0x90AA53CF325C4623), UINT64_C(0xC1D24D51349DD067), // 544, 545
	UINT64_C(0x2051CFEEA69EA624), UINT64_C(0x13220F0A862E7E4F), // 546, 547
	UINT64_C(0xCE39399404E04864), UINT64_C(0xD9C42CA47086FCB7), // 548, 549
	UINT64_C(0x685AD2238A03E7CC), UINT64_C(0x066484B2AB2FF1DB), // 550, 551
	UINT64_C(0xFE9D5D70EFBF79EC), UINT64_C(0x5B13B9DD9C481854), // 552, 553
	UINT64_C(0x15F0D475ED1509AD), UINT64_C(0x0BEBCD060EC79851), // 554, 555
	UINT64_C(0xD58C6791183AB7F8), UINT64_C(0xD1187C5052F3EEE4), // 556, 557
	UINT64_C(0xC95D1192E54E82FF), UINT64_C(0x86EEA14CB9AC6CA2), // 558, 559
	UINT64_C(0x3485BEB153677D5D), UINT64_C(0xDD191D781F8C492A), // 560, 561
	UINT64_C(0xF60866BAA784EBF9), UINT64_C(0x518F643BA2D08C74), // 562, 563
	UINT64_C(0x8852E956E1087C22), UINT64_C(0xA768CB8DC410AE8D), // 564, 565
	UINT64_C(0x38047726BFEC8E1A), UINT64_C(0xA67738B4CD3B45AA), // 566, 567
	UINT64_C(0xAD16691CEC0DDE19), UINT64_C(0xC6D4319380462E07), // 568, 569
	UINT64_C(0xC5A5876D0BA61938), UINT64_C(0x16B9FA1FA58FD840), // 570, 571
	UINT64_C(0x188AB1173CA74F18), UINT64_C(0xABDA2F98C99C021F), // 572, 573
	UINT64_C(0x3E0580AB134AE816), UINT64_C(0x5F3B05B773645ABB), // 574, 575
	UINT64_C(0x2501A2BE5575F2F6), UINT64_C(0x1B2F74004E7E8BA9), // 576, 577
	UINT64_C(0x1CD7580371E8D953), UINT64_C(0x7F6ED89562764E30), // 578, 579
	UINT64_C(0xB15926FF596F003D), UINT64_C(0x9F65293DA8C5D6B9), // 580, 581
	UINT64_C(0x6ECEF04DD690F84C), UINT64_C(0x4782275FFF33AF88), // 582, 583
	UINT64_C(0xE41433083F820801), UINT64_C(0xFD0DFE409A1AF9B5), // 584, 585
	UINT64_C(0x4325A3342CDB396B), UINT64_C(0x8AE77E62B301B252), // 586, 587
	UINT64_C(0xC36F9E9F6655615A), UINT64_C(0x85455A2D92D32C09), // 588, 589
	UINT64_C(0xF2C7DEA949477485), UINT64_C(0x63CFB4C133A39EBA), // 590, 591
	UINT64_C(0x83B040CC6EBC5462), UINT64_C(0x3B9454C8FDB326B0), // 592, 593
	UINT64_C(0x56F56A9E87FFD78C), UINT64_C(0x2DC2940D99F42BC6), // 594, 595
	UINT64_C(0x98F7DF096B096E2D), UINT64_C(0x19A6E01E3AD852BF), // 596, 597
	UINT64_C(0x42A99CCBDBD4B40B), UINT64_C(0xA59998AF45E9C559), // 598, 599
	UINT64_C(0x366295E807D93186), UINT64_C(0x6B48181BFAA1F773), // 600, 601
	UINT64_C(0x1FEC57E2157A0A1D), UINT64_C(0x4667446AF6201AD5), // 602, 603
	UINT64_C(0xE615EBCACFB0F075), UINT64_C(0xB8F31F4F68290778), // 604, 605
	UINT64_C(0x22713ED6CE22D11E), UINT64_C(0x3057C1A72EC3C93B), // 606, 607
	UINT64_C(0xCB46ACC37C3F1F2F), UINT64_C(0xDBB893FD02AAF50E), // 608, 609
	UINT64_C(0x331FD92E600B9FCF), UINT64_C(0xA498F96148EA3AD6), // 610, 611
	UINT64_C(0xA8D8426E8B6A83EA), UINT64_C(0xA089B274B7735CDC), // 612, 613
	UINT64_C(0x87F6B3731E524A11), UINT64_C(0x118808E5CBC96749), // 614, 615
	UINT64_C(0x9906E4C7B19BD394), UINT64_C(0xAFED7F7E9B24A20C), // 616, 617
	UINT64_C(0x6509EADEEB3644A7), UINT64_C(0x6C1EF1D3E8EF0EDE), // 618, 619
	UINT64_C(0xB9C97D43E9798FB4), UINT64_C(0xA2F2D784740C28A3), // 620, 621
	UINT64_C(0x7B8496476197566F), UINT64_C(0x7A5BE3E6B65F069D), // 622, 623
	UINT64_C(0xF96330ED78BE6F10), UINT64_C(0xEEE60DE77A076A15), // 624, 625
	UINT64_C(0x2B4BEE4AA08B9BD0), UINT64_C(0x6A56A63EC7B8894E), // 626, 627
	UINT64_C(0x02121359BA34FEF4), UINT64_C(0x4CBF99F8283703FC), // 628, 629
	UINT64_C(0x398071350CAF30C8), UINT64_C(0xD0A77A89F017687A), // 630, 631
	UINT64_C(0xF1C1A9EB9E423569), UINT64_C(0x8C7976282DEE8199), // 632, 633
	UINT64_C(0x5D1737A5DD1F7ABD), UINT64_C(0x4F53433C09A9FA80), // 634, 635
	UINT64_C(0xFA8B0C53DF7CA1D9), UINT64_C(0x3FD9DCBC886CCB77), // 636, 637
	UINT64_C(0xC040917CA91B4720), UINT64_C(0x7DD00142F9D1DCDF), // 638, 639
	UINT64_C(0x8476FC1D4F387B58), UINT64_C(0x23F8E7C5F3316503), // 640, 641
	UINT64_C(0x032A2244E7E37339), UINT64_C(0x5C87A5D750F5A74B), // 642, 643
	UINT64_C(0x082B4CC43698992E), UINT64_C(0xDF917BECB858F63C), // 644, 645
	UINT64_C(0x3270B8FC5BF86DDA), UINT64_C(0x10AE72BB29B5DD76), // 646, 647
	UINT64_C(0x576AC94E7700362B), UINT64_C(0x1AD112DAC61EFB8F), // 648, 649
	UINT64_C(0x691BC30EC5FAA427), UINT64_C(0xFF246311CC327143), // 650, 651
	UINT64_C(0x3142368E30E53206), UINT64_C(0x71380E31E02CA396), // 652, 653
	UINT64_C(0x958D5C960AAD76F1), UINT64_C(0xF8D6F430C16DA536), // 654, 655
	UINT64_C(0xC8FFD13F1BE7E1D2), UINT64_C(0x7578AE66004DDBE1), // 656, 657
	UINT64_C(0x05833F01067BE646), UINT64_C(0xBB34B5AD3BFE586D), // 658, 659
	UINT64_C(0x095F34C9A12B97F0), UINT64_C(0x247AB64525D60CA8), // 660, 661
	UINT64_C(0xDCDBC6F3017477D1), UINT64_C(0x4A2E14D4DECAD24D), // 662, 663
	UINT64_C(0xBDB5E6D9BE0A1EEB), UINT64_C(0x2A7E70F7794301AB), // 664, 665
	UINT64_C(0xDEF42D8A270540FD), UINT64_C(0x01078EC0A34C22C1), // 666, 667
	UINT64_C(0xE5DE511AF4C16387), UINT64_C(0x7EBB3A52BD9A330A), // 668, 669
	UINT64_C(0x77697857AA7D6435), UINT64_C(0x004E831603AE4C32), // 670, 671
	UINT64_C(0xE7A21020AD78E312), UINT64_C(0x9D41A70C6AB420F2), // 672, 673
	UINT64_C(0x28E06C18EA1141E6), UINT64_C(0xD2B28CBD984F6B28), // 674, 675
	UINT64_C(0x26B75F6C446E9D83), UINT64_C(0xBA47568C4D418D7F), // 676, 677
	UINT64_C(0xD80BADBFE6183D8E), UINT64_C(0x0E206D7F5F166044), // 678, 679
	UINT64_C(0xE258A43911CBCA3E), UINT64_C(0x723A1746B21DC0BC), // 680, 681
	UINT64_C(0xC7CAA854F5D7CDD3), UINT64_C(0x7CAC32883D261D9C), // 682, 683
	UINT64_C(0x7690C26423BA942C), UINT64_C(0x17E55524478042B8), // 684, 685
	UINT64_C(0xE0BE477656A2389F), UINT64_C(0x4D289B5E67AB2DA0), // 686, 687
	UINT64_C(0x44862B9C8FBBFD31), UINT64_C(0xB47CC8049D141365), // 688, 689
	UINT64_C(0x822C1B362B91C793), UINT64_C(0x4EB14655FB13DFD8), // 690, 691
	UINT64_C(0x1ECBBA0714E2A97B), UINT64_C(0x6143459D5CDE5F14), // 692, 693
	UINT64_C(0x53A8FBF1D5F0AC89), UINT64_C(0x97EA04D81C5E5B00), // 694, 695
	UINT64_C(0x622181A8D4FDB3F3), UINT64_C(0xE9BCD341572A1208), // 696, 697
	UINT64_C(0x1411258643CCE58A), UINT64_C(0x9144C5FEA4C6E0A4), // 698, 699
	UINT64_C(0x0D33D06565CF620F), UINT64_C(0x54A48D489F219CA1), // 700, 701
	UINT64_C(0xC43E5EAC6D63C821), UINT64_C(0xA9728B3A72770DAF), // 702, 703
	UINT64_C(0xD7934E7B20DF87EF), UINT64_C(0xE35503B61A3E86E5), // 704, 705
	UINT64_C(0xCAE321FBC819D504), UINT64_C(0x129A50B3AC60BFA6), // 706, 707
	UINT64_C(0xCD5E68EA7E9FB6C3), UINT64_C(0xB01C90199483B1C7), // 708, 709
	UINT64_C(0x3DE93CD5C295376C), UINT64_C(0xAED52EDF2AB9AD13), // 710, 711
	UINT64_C(0x2E60F512C0A07884), UINT64_C(0xBC3D86A3E36210C9), // 712, 713
	UINT64_C(0x35269D9B163951CE), UINT64_C(0x0C7D6E2AD0CDB5FA), // 714, 715
	UINT64_C(0x59E86297D87F5733), UINT64_C(0x298EF221898DB0E7), // 716, 717
	UINT64_C(0x55000029D1A5AA7E), UINT64_C(0x8BC08AE1B5061B45), // 718, 719
	UINT64_C(0xC2C31C2B6C92703A), UINT64_C(0x94CC596BAF25EF42), // 720, 721
	UINT64_C(0x0A1D73DB22540456), UINT64_C(0x04B6A0F9D9C4179A), // 722, 723
	UINT64_C(0xEFFDAFA2AE3D3C60), UINT64_C(0xF7C8075BB49496C4), // 724, 725
	UINT64_C(0x9CC5C7141D1CD4E3), UINT64_C(0x78BD1638218E5534), // 726, 727
	UINT64_C(0xB2F11568F850246A), UINT64_C(0xEDFABCFA9502BC29), // 728, 729
	UINT64_C(0x796CE5F2DA23051B), UINT64_C(0xAAE128B0DC93537C), // 730, 731
	UINT64_C(0x3A493DA0EE4B29AE), UINT64_C(0xB5DF6B2C416895D7), // 732, 733
	UINT64_C(0xFCABBD25122D7F37), UINT64_C(0x70810B58105DC4B1), // 734, 735
	UINT64_C(0xE10FDD37F7882A90), UINT64_C(0x524DCAB5518A3F5C), // 736, 737
	UINT64_C(0x3C9E85878451255B), UINT64_C(0x4029828119BD34E2), // 738, 739
	UINT64_C(0x74A05B6F5D3CECCB), UINT64_C(0xB610021542E13ECA), // 740, 741
	UINT64_C(0x0FF979D12F59E2AC), UINT64_C(0x6037DA27E4F9CC50), // 742, 743
	UINT64_C(0x5E92975A0DF1847D), UINT64_C(0xD66DE190D3E623FE), // 744, 745
	UINT64_C(0x5032D6B87B568048), UINT64_C(0x9A36B7CE8235216E), // 746, 747
	UINT64_C(0x80272A7A24F64B4A), UINT64_C(0x93EFED8B8C6916F7), // 748, 749
	UINT64_C(0x37DDBFF44CCE1555), UINT64_C(0x4B95DB5D4B99BD25), // 750, 751
	UINT64_C(0x92D3FDA169812FC0), UINT64_C(0xFB1A4A9A90660BB6), // 752, 753
	UINT64_C(0x730C196946A4B9B2), UINT64_C(0x81E289AA7F49DA68), // 754, 755
	UINT64_C(0x64669A0F83B1A05F), UINT64_C(0x27B3FF7D9644F48B), // 756, 757
	UINT64_C(0xCC6B615C8DB675B3), UINT64_C(0x674F20B9BCEBBE95), // 758, 759
	UINT64_C(0x6F31238275655982), UINT64_C(0x5AE488713E45CF05), // 760, 761
	UINT64_C(0xBF619F9954C21157), UINT64_C(0xEABAC46040A8EAE9), // 762, 763
	UINT64_C(0x454C6FE9F2C0C1CD), UINT64_C(0x419CF6496412691C), // 764, 765
	UINT64_C(0xD3DC3BEF265B0F70), UINT64_C(0x6D0E60F5C3578A9E), // 766, 767
	UINT64_C(0x5B0E608526323C55), UINT64_C(0x1A46C1A9FA1B59F5), // 768, 769
	UINT64_C(0xA9E245A17C4C8FFA), UINT64_C(0x65CA5159DB2955D7), // 770, 771
	UINT64_C(0x05DB0A76CE35AFC2), UINT64_C(0x81EAC77EA9113D45), // 772, 773
	UINT64_C(0x528EF88AB6AC0A0D), UINT64_C(0xA09EA253597BE3FF), // 774, 775
	UINT64_C(0x430DDFB3AC48CD56), UINT64_C(0xC4B3A67AF45CE46F), // 776, 777
	UINT64_C(0x4ECECFD8FBE2D05E), UINT64_C(0x3EF56F10B39935F0), // 778, 779
	UINT64_C(0x0B22D6829CD619C6), UINT64_C(0x17FD460A74DF2069), // 780, 781
	UINT64_C(0x6CF8CC8E8510ED40), UINT64_C(0xD6C824BF3A6ECAA7), // 782, 783
	UINT64_C(0x61243D581A817049), UINT64_C(0x048BACB6BBC163A2), // 784, 785
	UINT64_C(0xD9A38AC27D44CC32), UINT64_C(0x7FDDFF5BAAF410AB), // 786, 787
	UINT64_C(0xAD6D495AA804824B), UINT64_C(0xE1A6A74F2D8C9F94), // 788, 789
	UINT64_C(0xD4F7851235DEE8E3), UINT64_C(0xFD4B7F886540D893), // 790, 791
	UINT64_C(0x247C20042AA4BFDA), UINT64_C(0x096EA1C517D1327C), // 792, 793
	UINT64_C(0xD56966B4361A6685), UINT64_C(0x277DA5C31221057D), // 794, 795
	UINT64_C(0x94D59893A43ACFF7), UINT64_C(0x64F0C51CCDC02281), // 796, 797
	UINT64_C(0x3D33BCC4FF6189DB), UINT64_C(0xE005CB184CE66AF1), // 798, 799
	UINT64_C(0xFF5CCD1D1DB99BEA), UINT64_C(0xB0B854A7FE42980F), // 800, 801
	UINT64_C(0x7BD46A6A718D4B9F), UINT64_C(0xD10FA8CC22A5FD8C), // 802, 803
	UINT64_C(0xD31484952BE4BD31), UINT64_C(0xC7FA975FCB243847), // 804, 805
	UINT64_C(0x4886ED1E5846C407), UINT64_C(0x28CDDB791EB70B04), // 806, 807
	UINT64_C(0xC2B00BE2F573417F), UINT64_C(0x5C9590452180F877), // 808, 809
	UINT64_C(0x7A6BDDFFF370EB00), UINT64_C(0xCE509E38D6D9D6A4), // 810, 811
	UINT64_C(0xEBEB0F00647FA702), UINT64_C(0x1DCC06CF76606F06), // 812, 813
	UINT64_C(0xE4D9F28BA286FF0A), UINT64_C(0xD85A305DC918C262), // 814, 815
	UINT64_C(0x475B1D8732225F54), UINT64_C(0x2D4FB51668CCB5FE), // 816, 817
	UINT64_C(0xA679B9D9D72BBA20), UINT64_C(0x53841C0D912D43A5), // 818, 819
	UINT64_C(0x3B7EAA48BF12A4E8), UINT64_C(0x781E0E47F22F1DDF), // 820, 821
	UINT64_C(0xEFF20CE60AB50973), UINT64_C(0x20D261D19DFFB742), // 822, 823
	UINT64_C(0x16A12B03062A2E39), UINT64_C(0x1960EB2239650495), // 824, 825
	UINT64_C(0x251C16FED50EB8B8), UINT64_C(0x9AC0C330F826016E), // 826, 827
	UINT64_C(0xED152665953E7671), UINT64_C(0x02D63194A6369570), // 828, 829
	UINT64_C(0x5074F08394B1C987), UINT64_C(0x70BA598C90B25CE1), // 830, 831
	UINT64_C(0x794A15810B9742F6), UINT64_C(0x0D5925E9FCAF8C6C), // 832, 833
	UINT64_C(0x3067716CD868744E), UINT64_C(0x910AB077E8D7731B), // 834, 835
	UINT64_C(0x6A61BBDB5AC42F61), UINT64_C(0x93513EFBF0851567), // 836, 837
	UINT64_C(0xF494724B9E83E9D5), UINT64_C(0xE887E1985C09648D), // 838, 839
	UINT64_C(0x34B1D3C675370CFD), UINT64_C(0xDC35E433BC0D255D), // 840, 841
	UINT64_C(0xD0AAB84234131BE0), UINT64_C(0x08042A50B48B7EAF), // 842, 843
	UINT64_C(0x9997C4EE44A3AB35), UINT64_C(0x829A7B49201799D0), // 844, 845
	UINT64_C(0x263B8307B7C54441), UINT64_C(0x752F95F4FD6A6CA6), // 846, 847
	UINT64_C(0x927217402C08C6E5), UINT64_C(0x2A8AB754A795D9EE), // 848, 849
	UINT64_C(0xA442F7552F72943D), UINT64_C(0x2C31334E19781208), // 850, 851
	UINT64_C(0x4FA98D7CEAEE6291), UINT64_C(0x55C3862F665DB309), // 852, 853
	UINT64_C(0xBD0610175D53B1F3), UINT64_C(0x46FE6CB840413F27), // 854, 855
	UINT64_C(0x3FE03792DF0CFA59), UINT64_C(0xCFE700372EB85E8F), // 856, 857
	UINT64_C(0xA7BE29E7ADBCE118), UINT64_C(0xE544EE5CDE8431DD), // 858, 859
	UINT64_C(0x8A781B1B41F1873E), UINT64_C(0xA5C94C78A0D2F0E7), // 860, 861
	UINT64_C(0x39412E2877B60728), UINT64_C(0xA1265EF3AFC9A62C), // 862, 863
	UINT64_C(0xBCC2770C6A2506C5), UINT64_C(0x3AB66DD5DCE1CE12), // 864, 865
	UINT64_C(0xE65499D04A675B37), UINT64_C(0x7D8F523481BFD216), // 866, 867
	UINT64_C(0x0F6F64FCEC15F389), UINT64_C(0x74EFBE618B5B13C8), // 868, 869
	UINT64_C(0xACDC82B714273E1D), UINT64_C(0xDD40BFE003199D17), // 870, 871
	UINT64_C(0x37E99257E7E061F8), UINT64_C(0xFA52626904775AAA), // 872, 873
	UINT64_C(0x8BBBF63A463D56F9), UINT64_C(0xF0013F1543A26E64), // 874, 875
	UINT64_C(0xA8307E9F879EC898), UINT64_C(0xCC4C27A4150177CC), // 876, 877
	UINT64_C(0x1B432F2CCA1D3348), UINT64_C(0xDE1D1F8F9F6FA013), // 878, 879
	UINT64_C(0x606602A047A7DDD6), UINT64_C(0xD237AB64CC1CB2C7), // 880, 881
	UINT64_C(0x9B938E7225FCD1D3), UINT64_C(0xEC4E03708E0FF476), // 882, 883
	UINT64_C(0xFEB2FBDA3D03C12D), UINT64_C(0xAE0BCED2EE43889A), // 884, 885
	UINT64_C(0x22CB8923EBFB4F43), UINT64_C(0x69360D013CF7396D), // 886, 887
	UINT64_C(0x855E3602D2D4E022), UINT64_C(0x073805BAD01F784C), // 888, 889
	UINT64_C(0x33E17A133852F546), UINT64_C(0xDF4874058AC7B638), // 890, 891
	UINT64_C(0xBA92B29C678AA14A), UINT64_C(0x0CE89FC76CFAADCD), // 892, 893
	UINT64_C(0x5F9D4E0908339E34), UINT64_C(0xF1AFE9291F5923B9), // 894, 895
	UINT64_C(0x6E3480F60F4A265F), UINT64_C(0xEEBF3A2AB29B841C), // 896, 897
	UINT64_C(0xE21938A88F91B4AD), UINT64_C(0x57DFEFF845C6D3C3), // 898, 899
	UINT64_C(0x2F006B0BF62CAAF2), UINT64_C(0x62F479EF6F75EE78), // 900, 901
	UINT64_C(0x11A55AD41C8916A9), UINT64_C(0xF229D29084FED453), // 902, 903
	UINT64_C(0x42F1C27B16B000E6), UINT64_C(0x2B1F76749823C074), // 904, 905
	UINT64_C(0x4B76ECA3C2745360), UINT64_C(0x8C98F463B91691BD), // 906, 907
	UINT64_C(0x14BCC93CF1ADE66A), UINT64_C(0x8885213E6D458397), // 908, 909
	UINT64_C(0x8E177DF0274D4711), UINT64_C(0xB49B73B5503F2951), // 910, 911
	UINT64_C(0x10168168C3F96B6B), UINT64_C(0x0E3D963B63CAB0AE), // 912, 913
	UINT64_C(0x8DFC4B5655A1DB14), UINT64_C(0xF789F1356E14DE5C), // 914, 915
	UINT64_C(0x683E68AF4E51DAC1), UINT64_C(0xC9A84F9D8D4B0FD9), // 916, 917
	UINT64_C(0x3691E03F52A0F9D1), UINT64_C(0x5ED86E46E1878E80), // 918, 919
	UINT64_C(0x3C711A0E99D07150), UINT64_C(0x5A0865B20C4E9310), // 920, 921
	UINT64_C(0x56FBFC1FE4F0682E), UINT64_C(0xEA8D5DE3105EDF9B), // 922, 923
	UINT64_C(0x71ABFDB12379187A), UINT64_C(0x2EB99DE1BEE77B9C), // 924, 925
	UINT64_C(0x21ECC0EA33CF4523), UINT64_C(0x59A4D7521805C7A1), // 926, 927
	UINT64_C(0x3896F5EB56AE7C72), UINT64_C(0xAA638F3DB18F75DC), // 928, 929
	UINT64_C(0x9F39358DABE9808E), UINT64_C(0xB7DEFA91C00B72AC), // 930, 931
	UINT64_C(0x6B5541FD62492D92), UINT64_C(0x6DC6DEE8F92E4D5B), // 932, 933
	UINT64_C(0x353F57ABC4BEEA7E), UINT64_C(0x735769D6DA5690CE), // 934, 935
	UINT64_C(0x0A234AA642391484), UINT64_C(0xF6F9508028F80D9D), // 936, 937
	UINT64_C(0xB8E319A27AB3F215), UINT64_C(0x31AD9C1151341A4D), // 938, 939
	UINT64_C(0x773C22A57BEF5805), UINT64_C(0x45C7561A07968633), // 940, 941
	UINT64_C(0xF913DA9E249DBE36), UINT64_C(0xDA652D9B78A64C68), // 942, 943
	UINT64_C(0x4C27A97F3BC334EF), UINT64_C(0x76621220E66B17F4), // 944, 945
	UINT64_C(0x967743899ACD7D0B), UINT64_C(0xF3EE5BCAE0ED6782), // 946, 947
	UINT64_C(0x409F753600C879FC), UINT64_C(0x06D09A39B5926DB6), // 948, 949
	UINT64_C(0x6F83AEB0317AC588), UINT64_C(0x01E6CA4A86381F21), // 950, 951
	UINT64_C(0x66FF3462D19F3025), UINT64_C(0x72207C24DDFD3BFB), // 952, 953
	UINT64_C(0x4AF6B6D3E2ECE2EB), UINT64_C(0x9C994DBEC7EA08DE), // 954, 955
	UINT64_C(0x49ACE597B09A8BC4), UINT64_C(0xB38C4766CF0797BA), // 956, 957
	UINT64_C(0x131B9373C57C2A75), UINT64_C(0xB1822CCE61931E58), // 958, 959
	UINT64_C(0x9D7555B909BA1C0C), UINT64_C(0x127FAFDD937D11D2), // 960, 961
	UINT64_C(0x29DA3BADC66D92E4), UINT64_C(0xA2C1D57154C2ECBC), // 962, 963
	UINT64_C(0x58C5134D82F6FE24), UINT64_C(0x1C3AE3515B62274F), // 964, 965
	UINT64_C(0xE907C82E01CB8126), UINT64_C(0xF8ED091913E37FCB), // 966, 967
	UINT64_C(0x3249D8F9C80046C9), UINT64_C(0x80CF9BEDE388FB63), // 968, 969
	UINT64_C(0x1881539A116CF19E), UINT64_C(0x5103F3F76BD52457), // 970, 971
	UINT64_C(0x15B7E6F5AE47F7A8), UINT64_C(0xDBD7C6DED47E9CCF), // 972, 973
	UINT64_C(0x44E55C410228BB1A), UINT64_C(0xB647D4255EDB4E99), // 974, 975
	UINT64_C(0x5D11882BB8AAFC30), UINT64_C(0xF5098BBB29D3212A), // 976, 977
	UINT64_C(0x8FB5EA14E90296B3), UINT64_C(0x677B942157DD025A), // 978, 979
	UINT64_C(0xFB58E7C0A390ACB5), UINT64_C(0x89D3674C83BD4A01), // 980, 981
	UINT64_C(0x9E2DA4DF4BF3B93B), UINT64_C(0xFCC41E328CAB4829), // 982, 983
	UINT64_C(0x03F38C96BA582C52), UINT64_C(0xCAD1BDBD7FD85DB2), // 984, 985
	UINT64_C(0xBBB442C16082AE83), UINT64_C(0xB95FE86BA5DA9AB0), // 986, 987
	UINT64_C(0xB22E04673771A93F), UINT64_C(0x845358C9493152D8), // 988, 989
	UINT64_C(0xBE2A488697B4541E), UINT64_C(0x95A2DC2DD38E6966), // 990, 991
	UINT64_C(0xC02C11AC923C852B), UINT64_C(0x2388B1990DF2A87B), // 992, 993
	UINT64_C(0x7C8008FA1B4F37BE), UINT64_C(0x1F70D0C84D54E503), // 994, 995
	UINT64_C(0x5490ADEC7ECE57D4), UINT64_C(0x002B3C27D9063A3A), // 996, 997
	UINT64_C(0x7EAEA3848030A2BF), UINT64_C(0xC602326DED2003C0), // 998, 999
	UINT64_C(0x83A7287D69A94086), UINT64_C(0xC57A5FCB30F57A8A), // 1000, 1001
	UINT64_C(0xB56844E479EBE779), UINT64_C(0xA373B40F05DCBCE9), // 1002, 1003
	UINT64_C(0xD71A786E88570EE2), UINT64_C(0x879CBACDBDE8F6A0), // 1004, 1005
	UINT64_C(0x976AD1BCC164A32F), UINT64_C(0xAB21E25E9666D78B), // 1006, 1007
	UINT64_C(0x901063AAE5E5C33C), UINT64_C(0x9818B34448698D90), // 1008, 1009
	UINT64_C(0xE36487AE3E1E8ABB), UINT64_C(0xAFBDF931893BDCB4), // 1010, 1011
	UINT64_C(0x6345A0DC5FBBD519), UINT64_C(0x8628FE269B9465CA), // 1012, 1013
	UINT64_C(0x1E5D01603F9C51EC), UINT64_C(0x4DE44006A15049B7), // 1014, 1015
	UINT64_C(0xBF6C70E5F776CBB1), UINT64_C(0x411218F2EF552BED), // 1016, 1017
	UINT64_C(0xCB0C0708705A36A3), UINT64_C(0xE74D14754F986044), // 1018, 1019
	UINT64_C(0xCD56D9430EA8280E), UINT64_C(0xC12591D7535F5065), // 1020, 1021
	UINT64_C(0xC83223F1720AEF96), UINT64_C(0xC3A0396F7363A51F)  // 1022, 1023
};

#define p_tiger1 (p_tiger_table)
#define p_tiger2 (p_tiger_table + 256)
#define p_tiger3 (p_tiger_table + 256 * 2)
#define p_tiger4 (p_tiger_table + 256 * 3)

#define round(a,b,c,x,mul) \
	do { \
		c ^= (x); \
		a -= p_tiger1[((c) >> (0 * 8)) & 0xff] ^ p_tiger2[((c) >> (2 * 8)) & 0xff] ^ \
			 p_tiger3[((c) >> (4 * 8)) & 0xff] ^ p_tiger4[((c) >> (6 * 8)) & 0xff]; \
		b += p_tiger4[((c) >> (1 * 8)) & 0xff] ^ p_tiger3[((c) >> (3 * 8)) & 0xff] ^ \
			 p_tiger2[((c) >> (5 * 8)) & 0xff] ^ p_tiger1[((c) >> (7 * 8)) & 0xff]; \
		b *= (mul); \
	} while(0)

#define pass(a,b,c,mul) \
	do { \
		round(a, b, c, x0, (mul)); \
		round(b, c, a, x1, (mul)); \
		round(c, a, b, x2, (mul)); \
		round(a, b, c, x3, (mul)); \
		round(b, c, a, x4, (mul)); \
		round(c, a, b, x5, (mul)); \
		round(a, b, c, x6, (mul)); \
		round(b, c, a, x7, (mul)); \
	} while(0)

#define key_schedule \
	do { \
		x0 -= x7 ^ UINT64_C(0xA5A5A5A5A5A5A5A5); \
		x1 ^= x0; \
		x2 += x1; \
		x3 -= x2 ^ ((~x1) << 19); \
		x4 ^= x3; \
		x5 += x4; \
		x6 -= x5 ^ ((~x4) >> 23); \
		x7 ^= x6; \
		x0 += x7; \
		x1 -= x0 ^ ((~x7) << 19); \
		x2 ^= x1; \
		x3 += x2; \
		x4 -= x3 ^ ((~x2) >> 23); \
		x5 ^= x4; \
		x6 += x5; \
		x7 -= x6 ^ UINT64_C(0x0123456789ABCDEF); \
	} while(0)

TTigerBase::TTigerBase()
	:m_n_total_length_bits(0)
{
	m_n_hash[0] = UINT64_C(0x0123456789ABCDEF);
	m_n_hash[1] = UINT64_C(0xFEDCBA9876543210);
	m_n_hash[2] = UINT64_C(0xF096A5B4C3B2E187);
}

TTigerBase &TTigerBase::operator =(const TTigerBase &r_t_tiger)
{
	m_n_total_length_bits = r_t_tiger.m_n_total_length_bits;
	memcpy(m_n_hash, r_t_tiger.m_n_hash, 5 * sizeof(uint32_t));
	return *this;
}

bool TTigerBase::operator ==(const TTigerBase &r_t_tiger) const
{
	return m_n_hash[0] == r_t_tiger.m_n_hash[0] &&
		   m_n_hash[1] == r_t_tiger.m_n_hash[1] &&
		   m_n_hash[2] == r_t_tiger.m_n_hash[2];
}

bool TTigerBase::operator <(const TTigerBase &r_t_tiger) const
{
	return m_n_hash[0] < r_t_tiger.m_n_hash[0] || (m_n_hash[0] == r_t_tiger.m_n_hash[0] &&
		   (m_n_hash[1] < r_t_tiger.m_n_hash[1] || (m_n_hash[1] == r_t_tiger.m_n_hash[1] &&
		   (m_n_hash[2] < r_t_tiger.m_n_hash[2]))));
}

void TTigerBase::Process_Block(const void *p_data, int n_size_bytes)
{
	_ASSERTE(n_size_bytes == 64); // 8 uint64_t's (x0 - x7)
	const uint64_t *p_data64 = (const uint64_t*)p_data;
	// cast to bytes

	register uint64_t a = m_n_hash[0];
	register uint64_t b = m_n_hash[1];
	register uint64_t c = m_n_hash[2];

	{
		uint64_t aa = a;
		uint64_t bb = b;
		uint64_t cc = c;
		// save_abc

		register uint64_t x0 = p_data64[0];
		register uint64_t x1 = p_data64[1];
		register uint64_t x2 = p_data64[2];
		register uint64_t x3 = p_data64[3];
		register uint64_t x4 = p_data64[4];
		register uint64_t x5 = p_data64[5];
		register uint64_t x6 = p_data64[6];
		register uint64_t x7 = p_data64[7];
		// read input

		pass(a, b, c, 5);
		key_schedule;
		pass(c, a, b, 7);
		key_schedule;
		pass(b, c, a, 9);
		for(int n_pass = 3; n_pass < tiger_Pass_Num; ++ n_pass) {
			key_schedule;
			pass(a, b, c, 9);
			register uint64_t n_temp = a;
			a = c;
			c = b;
			b = n_temp;
		}
		// compress

		a ^= aa;
		b -= bb;
		c += cc;
		// feedforward
	}
	// compress

	m_n_hash[0] = a;
	m_n_hash[1] = b;
	m_n_hash[2] = c;
}

const char *TTigerBase::p_s_ToString() const
{
	static char p_s_hash[49]; // 48 hexa digits ~ 192 bytes

	uint64_t a = n_ByteSwap64(m_n_hash[0]);
	uint64_t b = n_ByteSwap64(m_n_hash[1]);
	uint64_t c = n_ByteSwap64(m_n_hash[2]);
#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER >= 1400
	sprintf_s(p_s_hash, sizeof(p_s_hash), "%08x%08x%08x%08x%08x%08x",
		uint32_t(a >> 32), uint32_t(a),
		uint32_t(b >> 32), uint32_t(b),
		uint32_t(c >> 32), uint32_t(c));
#else //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
	sprintf(p_s_hash, "%08x%08x%08x%08x%08x%08x",
		uint32_t(a >> 32), uint32_t(a),
		uint32_t(b >> 32), uint32_t(b),
		uint32_t(c >> 32), uint32_t(c)); // @t_odo - use sprintf_s
#endif //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
	p_s_hash[48] = 0;

	return p_s_hash;
}

inline uint64_t TTigerBase::n_ByteSwap64(uint64_t n_x)
{
	return ((n_x >>  8) & 0xff000000U) | ((n_x & 0xff000000U) <<  8) |
		((n_x >> 24) & 0x00ff0000U) | ((n_x & 0x00ff0000U) << 24) |
		((n_x >> 40) & 0x0000ff00U) | ((n_x & 0x0000ff00U) << 40) |
		((n_x >> 56) & 0x000000ffU) | ((n_x & 0x000000ffU) << 56);
}

#undef t1
#undef t2
#undef t3
#undef t4
#undef round
#undef pass
#undef key_schedule

/*
 *								=== ~TTigerBase ===
 */

/*
 *		-end-of-file-
 */
