#include "Track.h"

namespace Motion {


Track::Track(const KeyPoint & keypoint, const Mat & descriptor, int64 timestamp) {
	update(keypoint, descriptor, timestamp);
	firstTimestamp = timestamp;
	lastTimestamp = timestamp;
	lastUpdateBefore = 0;
	matched = false;
	reassignId = -1;
	reassignCount = 0;
}



void Track::update(const KeyPoint & keypoint, const Mat & descriptor, int64 timestamp) {
	trajectory.push_back(keypoint);
	this->descriptor = descriptor;
	this->lastUpdateBefore = 0;
	
	if(timestamp != -1) {
		this->lastTimestamp = timestamp;
	}
}


void Track::update(const Track & track) {
	update(track.trajectory[track.trajectory.size()-1], track.descriptor, track.lastTimestamp);
}


unsigned int Track::getLength() const {
	return trajectory.size();
}


const KeyPoint & Track::getKeypoint(unsigned int before) const {
	return trajectory[trajectory.size()-before-1];
}


const Mat & Track::getDescriptor() const {
	return descriptor;
}


int64 Track::getFirstTimeStamp() const {
	return firstTimestamp;
}


int64 Track::getLastTimeStamp() const {
	return lastTimestamp;
}


unsigned int Track::getUpdatedBefore() const {
	return lastUpdateBefore;
}


void Track::setUpdatedBefore(unsigned int ub) {
	lastUpdateBefore = ub;
}


void Track::propagate(Point2f pos) {
	KeyPoint keypoint = getKeypoint();
	keypoint.pt = pos;
	Mat descriptor = getDescriptor();
	
    unsigned int prevLastUpdatBefore = lastUpdateBefore;	
	update(keypoint, descriptor);
	lastUpdateBefore = prevLastUpdatBefore + 1;
}


void Track::propagate() {
	lastUpdateBefore++;
}


float Track::distancePosition(const Track & second) const {
	return sqrt(pow(getKeypoint().pt.x - second.getKeypoint().pt.x, 2) + pow(getKeypoint().pt.y - second.getKeypoint().pt.y, 2));
}


int Track::distanceDescriptorHamming(const Track & second) const {
	Mat d1 = getDescriptor();
	Mat d2 = second.getDescriptor();

	// calculate Hamming distance
	int dist = 0;
	for(int i = 0; i < d1.cols; i++) {
		unsigned int val = d1.at<uchar>(i) ^ d2.at<uchar>(i);

		while(val) {
			dist++; 
			val &= val - 1;
		}
	}

	return dist;
}


float Track::distanceDescriptorEuclidean(const Track & second) const {
	Mat d1 = getDescriptor();
	Mat d2 = second.getDescriptor();

	// calculate Euclidean distance
	double sum = 0.0;
	for(int i = 0; i < d1.cols; i++) {
		sum += (d1.at<uchar>(i)-d2.at<float>(i))*(d1.at<uchar>(i)-d2.at<float>(i));
	}

	return (float)sqrt(sum);
}


float Track::ncc(const Point2f & point, const Mat & thisGray, const Mat & secondGray) const {
	Mat roi1, roi2;

	int x1 = (int)(getKeypoint().pt.x - getKeypoint().size/2);
	int y1 = (int)(getKeypoint().pt.y - getKeypoint().size/2);	
	int x2 = (int)(point.x - getKeypoint().size/2);
	int y2 = (int)(point.y - getKeypoint().size/2);
	int s = (int)getKeypoint().size;

	if(x1 < 0 || x1+s >= thisGray.cols || y1 < 0 || y1+s >= thisGray.rows ||
	   x2 < 0 || x2+s >= thisGray.cols || y2 < 0 || y2+s >= thisGray.rows) {
		return 0.0f;
	}

	roi1 = Mat(thisGray, Rect(x1, y1, s, s));
	roi2 = Mat(secondGray, Rect(x2, y2, s, s));

	float mean1 = (float)mean(roi1)[0];
	float mean2 = (float)mean(roi2)[0];
	
	// calculate normalized cross-correlation value on ROIs
	float sum1 = 0.0;
	float sumSqr1 = 0.0;
	float sumSqr2 = 0.0;
	for(int i = 0; i < roi1.rows; i++)
		for(int j = 0; j < roi1.cols; j++) {
			float val1 = (float)roi1.at<uchar>(j,i) - mean1;
			float val2 = (float)roi2.at<uchar>(j,i) - mean2;
			sum1 += val1*val2;
			sumSqr1 += val1*val1;
			sumSqr2 += val2*val2;
		}
	
	return sum1/sqrt(sumSqr1*sumSqr2);
}


float Track::ncc(const Track & second, const Mat & thisGray, const Mat & secondGray) const {
	return ncc(second.getKeypoint().pt, thisGray, secondGray);
}


bool Track::operator>(const Track & second) const {
	if(getLength() > second.getLength()) {
		return true;
	}

	if(getKeypoint().response > second.getKeypoint().response) {
		return true;
	}

	return false;
}


}
