#include "Printer.h"


namespace vmatch {


std::string Printer::int2time(int time) const {
	int hours = (int)(time / 3600);
	int minutes = (int)((time / 60) % 60);
	int seconds = (int)(time % 60);
	
	std::ostringstream str;
	str.fill('0');
	str << std::setw(2) << hours << ":" << std::setw(2) << minutes << ":" << std::setw(2) << seconds;

	return str.str();
}


std::string Printer::formatPosition(int pos, double fps) const {	
	if(fps <= 0) {
		std::ostringstream str;
		str << pos;
		return str.str();
	}

	return int2time((int)(pos/fps));
}


void Printer::print(const Segments & segments) {
	Segments segmentsSorted = segments;
	segmentsSorted.sort();	
	printFunc(segmentsSorted);
}


void SegmentsPrinter::printFunc(const Segments & segments) {
	double fpsRef = segments.getMetaDataRef().fps;
	double fpsQuery = segments.getMetaDataQuery().fps;

	int currentRef = 0;

	printBegin(segments);

	for(Segments::const_iterator it = segments.begin(); it != segments.end(); it++) {
		SegmentPtr segment = *it;
		
		if(segment->getStartRef() > currentRef) {
			printSegment(currentRef, segment->getStartRef(), 0, 0, fpsRef, fpsQuery);
		}

		printSegment(segment, fpsRef, fpsQuery);
		currentRef = segment->getEndRef();
	}

	printEnd(segments);
}


void SegmentsPrinter::printSegment(SegmentPtr segment, double fpsRef, double fpsQuery) {
	printSegment(segment->getStartRef(), segment->getEndRef(), segment->getStartQuery(), segment->getEndQuery(), fpsRef, fpsQuery);
}


StdStreamSegmentsPrinter::StdStreamSegmentsPrinter(std::ostream *out) {
	setOut(out);
}


void StdStreamSegmentsPrinter::setOut(std::ostream *out) {
	if(out != NULL) {
		this->out = out;
	}
	else {
		this->out = &std::cout;
	}
}


TextSegmentsPrinter::TextSegmentsPrinter(std::ostream *out) : StdStreamSegmentsPrinter(out) {
}


void TextSegmentsPrinter::printBegin(const Segments & segments) {
}


void TextSegmentsPrinter::printEnd(const Segments & segments) {
}


void TextSegmentsPrinter::printSegment(int startRef, int endRef, int startQuery, int endQuery, double fpsRef, double fpsQuery) {
	*out << "ref/" << formatPosition(startRef, fpsRef) << "-" << formatPosition(endRef, fpsRef);

	if(startQuery != endQuery) {
		*out << " matches query/" << formatPosition(startQuery, fpsQuery) << "-" << formatPosition(endQuery, fpsQuery);
	}
	else {
		*out << " differs";
	}

	*out << std::endl;
}


XmlSegmentsPrinter::XmlSegmentsPrinter(std::ostream *out) : StdStreamSegmentsPrinter(out) {
}


void XmlSegmentsPrinter::printBegin(const Segments & segments) {
	*out << "<segments>" << std::endl;
}


void XmlSegmentsPrinter::printEnd(const Segments & segments) {
	*out << "</segments>" << std::endl;
}


void XmlSegmentsPrinter::printSegment(int startRef, int endRef, int startQuery, int endQuery, double fpsRef, double fpsQuery) {
	*out << "  <segment start=\"" << formatPosition(startRef, fpsRef) << "\" end=\"" << formatPosition(endRef, fpsRef) << "\"";

	if(startQuery != endQuery) {
		*out << ">" << std::endl
		   << "    <match start=\"" << formatPosition(startQuery, fpsQuery) << "\" end=\"" << formatPosition(endQuery, fpsQuery) << "\" />" << std::endl
		   << "  </segment>";
	}
	else {
		*out << " />";
	}

	*out << std::endl;
}


void DissimilaritiesPrinter::printFunc(const Segments & segments) {
	printBegin(segments);

	Segments segmentsEnd = segments;
	// TODO odstranene jen docasne
	//segmentsEnd.push_back(new Segment(segments.getMetaDataRef().length, segments.getMetaDataRef().length, segments.getMetaDataQuery().length, segments.getMetaDataQuery().length));

	int refStart  = 0;
	int refEnd    = 0;
	int refLength = 0;
	int qryStart  = 0;
	int qryEnd    = 0;	
	int qryLength = 0;

	for(Segments::iterator it = segmentsEnd.begin(); it != segmentsEnd.end(); it++) {
		SegmentPtr segment = *it;
		
		refEnd = segment->getStartRef()-1;
		qryEnd = segment->getStartQuery()-1;

		refLength = refEnd - refStart;
		qryLength = qryEnd - qryStart;

		// nalezeni disimilarit
		if(refLength > 0 || qryLength > 0) {
			printDissimilarity(UNKNOWN, refStart, MAX(refLength, qryLength));
		}

		refStart = segment->getEndRef()+1;
		qryStart = segment->getEndQuery()+1;
	}

	printEnd(segments);
}


//void DissimilaritiesPrinter::printFunc(const Segments & segments) {
//	printBegin(segments);
//
//	int posRef = 0;
//	int posQuery = 0;
//
//	for(Segments::const_iterator it = segments.begin(); it != segments.end(); it++) {
//		SegmentPtr segment = *it;
//
//		if(segment->getStartRef() > posRef && segment->getStartQuery() <= posQuery) {
//			printDissimilarity(REMOVAL, posRef, segment->getStartRef()-posRef);
//		}
//		else if(segment->getStartQuery() > posQuery && segment->getStartRef() <= posRef) {
//			printDissimilarity(INJECTION, posRef, segment->getStartRef()-posRef);
//		}
//		else if(segment->getStartRef() > posRef && segment->getStartQuery() > posQuery) {
//			printDissimilarity(DISMATCH, posRef, segment->getStartRef()-posRef);
//		}
//
//		posRef = segment->getEndRef() + 1;
//		posQuery = segment->getEndQuery() + 1;
//	}
//
//	if(posRef < segments.getMetaDataRef().length) {
//		if(posQuery <  segments.getMetaDataQuery().length) {
//			printDissimilarity(DISMATCH, posRef, segments.getMetaDataRef().length-posRef-1);
//
//		}
//		else {
//			printDissimilarity(REMOVAL, posRef, segments.getMetaDataRef().length-posRef-1);
//		}
//	}
//
//	printEnd(segments);
//}


StdStreamDissimilaritiesPrinter::StdStreamDissimilaritiesPrinter(std::ostream *out) {
	setOut(out);
}


void StdStreamDissimilaritiesPrinter::setOut(std::ostream *out) {
	if(out != NULL) {
		this->out = out;
	}
	else {
		this->out = &std::cout;
	}
}


TextDissimilaritiesPrinter::TextDissimilaritiesPrinter(std::ostream *out) : StdStreamDissimilaritiesPrinter(out) {
}


std::string TextDissimilaritiesPrinter::dissimilarityTypeToString(DissimilarityType type) {
	switch(type) {
		case UNKNOWN: return "U";
		case REMOVAL: return "R";
		case INJECTION: return "I";
		case DISMATCH: return "D";
		default:
			CV_Assert(false);
			return "";
	}
}


void TextDissimilaritiesPrinter::printBegin(const Segments & segments) {
}


void TextDissimilaritiesPrinter::printEnd(const Segments & segments) {
}


void TextDissimilaritiesPrinter::printDissimilarity(DissimilarityType type, int start, int length) {
	if(length <= 0) {
		return;
	}

	*out << dissimilarityTypeToString(type) << "\t" << start << "\t" << length << std::endl;
}


}
