#ifndef _PRINTER_H
#define _PRINTER_H


#include <iostream>
#include <iomanip>
#include "SegmentsExtractor.h"


namespace vmatch {


class Printer {
protected:
	std::string int2time(int time) const;
	std::string formatPosition(int pos, double fps) const;
	virtual void printFunc(const Segments & segments) = 0;

public:
	void print(const Segments & segments);
};


typedef cv::Ptr<Printer> PrinterPtr;


class SegmentsPrinter : public Printer {
protected:
	virtual void printFunc(const Segments & segments);
	virtual void printBegin(const Segments & segments) = 0;
	virtual void printEnd(const Segments & segments) = 0;
	virtual void printSegment(int startRef, int endRef, int startQuery, int endQuery, double fpsRef, double fpsQuery) = 0;
	void printSegment(SegmentPtr segment, double fpsRef, double fpsQuery);

public:
};


class StdStreamSegmentsPrinter : public SegmentsPrinter {
protected:
	std::ostream *out;

public:
	StdStreamSegmentsPrinter(std::ostream *out = &std::cout);
	void setOut(std::ostream *out);
};


class TextSegmentsPrinter : public StdStreamSegmentsPrinter {
protected:
	virtual void printBegin(const Segments & segments);
	virtual void printEnd(const Segments & segments);
	virtual void printSegment(int startRef, int endRef, int startQuery, int endQuery, double fpsRef, double fpsQuery);
	
public:
	TextSegmentsPrinter(std::ostream *out = &std::cout);
};


class XmlSegmentsPrinter : public StdStreamSegmentsPrinter {
protected:
	std::stringstream ss;

	virtual void printBegin(const Segments & segments);
	virtual void printEnd(const Segments & segments);
	virtual void printSegment(int startRef, int endRef, int startQuery, int endQuery, double fpsRef, double fpsQuery);

public:
	XmlSegmentsPrinter(std::ostream *out = &std::cout);
};


class DissimilaritiesPrinter : public Printer {
protected:
	typedef enum {
		UNKNOWN,
		INJECTION,
		REMOVAL,
		DISMATCH
	} DissimilarityType;

	virtual void printFunc(const Segments & segments);
	virtual void printBegin(const Segments & segments) = 0;
	virtual void printEnd(const Segments & segments) = 0;
	virtual void printDissimilarity(DissimilarityType type, int start, int length) = 0;
};


class StdStreamDissimilaritiesPrinter : public DissimilaritiesPrinter {
protected:
	std::ostream *out;

public:
	StdStreamDissimilaritiesPrinter(std::ostream *out = &std::cout);
	void setOut(std::ostream *out);
};


class TextDissimilaritiesPrinter : public StdStreamDissimilaritiesPrinter {
protected:
	std::string dissimilarityTypeToString(DissimilarityType type);
	virtual void printBegin(const Segments & segments);
	virtual void printEnd(const Segments & segments);
	virtual void printDissimilarity(DissimilarityType type, int start, int length);

public:
	TextDissimilaritiesPrinter(std::ostream *out = &std::cout);
};


}


#endif
