#include "KeySegments.h"


namespace vmatch {


KeySegmentDescriptor::KeySegmentDescriptor(const KeyFramePtrVector & keyframes) : FrameDescriptor(FrameDescriptor::_GROUP) {
	CV_Assert(keyframes.size() > 0);

	this->id = keyframes[0]->getSegmentId();
	this->descriptorType = keyframes[0]->getFrameDescriptor()->getType();

	for(KeyFramePtrVector::const_iterator it = keyframes.begin(); it != keyframes.end(); it++) {
		const KeyFramePtr kf = *it;
		CV_Assert(kf->getSegmentId() == this->id);
		CV_Assert(kf->getFrameDescriptor()->getType() == this->descriptorType);
		this->keyframes.push_back(kf);
	}
}


double KeySegmentDescriptor::compare(const FrameDescriptor & second) const {
	KeySegmentDescriptor *secondKs = (KeySegmentDescriptor *)(&second);
	CV_Assert(second.getType() == this->type);
	CV_Assert(secondKs->descriptorType == this->descriptorType);
	
	double minDist = DBL_MAX;
	for(KeyFramePtrVector::const_iterator it1 = this->keyframes.begin(); it1 != this->keyframes.end(); it1++) {
		for(KeyFramePtrVector::const_iterator it2 = secondKs->keyframes.begin(); it2 != secondKs->keyframes.end(); it2++) {
			double dist = (*it1)->getFrameDescriptor()->compare(*((*it2)->getFrameDescriptor()));
			minDist = MIN(minDist, dist);
		}
	}

	return minDist;
}


void KeySegmentDescriptor::read(cv::FileNode & nd) {
	CV_Error(CV_StsNotImplemented, "Deserialization of KeySegment is not supported.");
}


void KeySegmentDescriptor::write(cv::FileStorage & fs) const {
	CV_Error(CV_StsNotImplemented, "Serialization of KeySegment is not supported.");
}


KeySegment::KeySegment(const KeyFramePtrVector & keyframes, int segmentId) : KeyFrame(new KeySegmentDescriptor(keyframes), UNKNOWN, segmentId) {
	CV_Assert(!keyframes.empty());

	start = keyframes[0]->getFrameNumber();
	end = keyframes[0]->getFrameNumber();

	for(KeyFramePtrVector::const_iterator it = keyframes.begin(); it != keyframes.end(); it++) {
		start = MIN(start, (*it)->getFrameNumber());
		end = MAX(start, (*it)->getFrameNumber());
	}
}


int KeySegment::getStart() const {
	return start;
}


int KeySegment::getEnd() const {
	return end;
}


KeySegments::KeySegments(const KeyFrames & kf) : KeyFrames(kf.getDescriptorType()) {
	CV_Assert(kf.size() > 0);
	
	int currentSegmentId = KeyFrame::UNKNOWN;
	KeyFramePtrVector currentSegmentKfs;
	for(KeyFrames::const_iterator it = kf.begin(); it != kf.end(); it++) {
		KeyFramePtr k = *it;

		if(currentSegmentId == KeyFrame::UNKNOWN) {
			currentSegmentId = k->getSegmentId();
		}
		
		if(k->getSegmentId() == currentSegmentId) {
			currentSegmentKfs.push_back(k);
		}
		else {
			KeyFramePtr newKf = new KeySegment(currentSegmentKfs, currentSegmentId);
			this->push_back(newKf);
			
			currentSegmentId = k->getSegmentId();
			currentSegmentKfs.clear();
			currentSegmentKfs.push_back(k);

		}
	}

	if(!currentSegmentKfs.empty()) {
		KeyFramePtr newKf = new KeySegment(currentSegmentKfs, currentSegmentId);
		this->push_back(newKf);
	}
}


}
