#include "TrackContainer.h"

namespace Motion {


TrackContainer::TrackContainer(void) : vector<TrackPtr>() {
}


TrackContainer::TrackContainer(const KeyPoints & keypoints, const Mat & descriptors, int64 timestamp) : vector<TrackPtr>() {
	setContent(keypoints, descriptors, timestamp);
}


TrackPtr TrackContainer::item(unsigned int idx) const {
	return (*this)[idx];
}


void TrackContainer::setContent(const KeyPoints & keypoints, const Mat & descriptors, int64 timestamp) {
	clear();
	
	// convert all keypoints+descriptors into tracks and store them
	for(unsigned int i = 0; i < keypoints.size(); i++) {
		TrackPtr track = new Track(keypoints[i], descriptors.empty()?Mat():descriptors.row(i), timestamp);
		push_back(track);
	}
}


unsigned int TrackContainer::addContent(const TrackContainer & content, double maxDist, bool matchedOnly) {
	unsigned int addedCount = 0;
	
	for(unsigned int i = 0; i < this->size(); i++) {
		item(i)->matched = true;
	}
	
	for(unsigned int i = 0; i < content.size(); i++) {
		// don't add not-matched track if matchedOnly
		if(matchedOnly && !content.item(i)->matched) {
			continue;
		}
		
		bool doAdd = true;
		
		for(unsigned int j = 0; j < this->size(); j++) {
			// does the track collide with some other one?
			if(item(j)->matched && content.item(i)->distancePosition(*item(j)) < maxDist) {
				// keep the strongest track
				if(*item(j) > *content.item(i)) {
					doAdd = false;
					break;
				}
				else {
					item(j)->matched = false;
				}
			}
		}
		
		if(doAdd) {
			TrackPtr track = content.item(i);
			track->matched = true;
			this->push_back(track);
			addedCount++;
		}
	}
	
	removeContent(TrackContainer::NotMatched());

	return addedCount;
}


TrackContainer TrackContainer::removeContent(const Test & test, bool remove) {
	TrackContainer keep;
	TrackContainer dontKeep;

	// divide current items into two groups according to result of the test
	for(unsigned int i = 0; i < this->size(); i++) {
		TrackPtr track = item(i);
		if(test.keep(*track)) {
			keep.push_back(track);
		}
		else {
			dontKeep.push_back(track);
		}
	}

	// remove dontKeep items if wanted
	if(remove) {
		this->assign(keep.begin(), keep.end());
	}

	return dontKeep;
}


bool TrackContainer::Test::keep(const Track & track) const {
	return true;
}


bool TrackContainer::NotMatched::keep(const Track & track) const {
	return track.matched != 0;
}


TrackContainer::ShorterThan::ShorterThan(unsigned int minLength) {
	this->minLength = minLength;
}


bool TrackContainer::ShorterThan::keep(const Track & track) const {
	return track.getLength() >= minLength;
}


TrackContainer::UpdatedBefore::UpdatedBefore(unsigned int maxFramesNotUpdated) {
	this->maxFramesNotUpdated = maxFramesNotUpdated;
}


bool TrackContainer::UpdatedBefore::keep(const Track & track) const {
	return track.lastUpdateBefore < maxFramesNotUpdated;
}


}
