#include "VideoSequence.h"


namespace vmatch {


const int VideoSequence::MetaData::UNKNOWN = 0;
const std::string VideoSequence::MetaData::NODE_FPS = "fps";
const std::string VideoSequence::MetaData::NODE_LENGTH = "length";


VideoSequence::VideoSequence(FrameDescriptorExtractorPtr extractor) {
	setDescriptorExtractor(extractor);
	setFilter(new FrameFilter);
	this->frameNumber = -1;
	this->metaData = new MetaData();
}


bool VideoSequence::next() {
    if(step()) {
		frameNumber++;
		return true;
	}

	metaData->length = frameNumber+1;
	return false;
}


void VideoSequence::rewind() {
	stepToStart();
	frameNumber = -1;
}


cv::Mat VideoSequence::getFrame() {
	cv::Mat fr = frame();
	filter->apply(fr);
	return fr;
}


VideoSequence::MetaData VideoSequence::getMetaData() {
	return *metaData;
}


int VideoSequence::getFrameNumber() {
	return frameNumber;
}


FrameDescriptorPtr VideoSequence::getFrameDescriptor() {
	cv::Mat frame = getFrame();
	return extractor->extract(frame);
}


FrameDescriptorExtractorPtr VideoSequence::getDescriptorExtractor() const {
	return extractor;
}


void VideoSequence::setDescriptorExtractor(FrameDescriptorExtractorPtr extractor) {
	if(extractor == NULL) {
		CV_Error(CV_StsBadArg, "Null pointer to FrameDescriptorExtractor given.");
	}
	this->extractor = extractor;
}


VideoSequence::FrameFilterPtr VideoSequence::getFilter() const {
	return filter;
}


void VideoSequence::setFilter(VideoSequence::FrameFilterPtr filter) {
	if(filter == NULL) {
		CV_Error(CV_StsBadArg, "Null pointer to FrameFilterPtr given.");
	}
	this->filter = filter;
}


void VideoSequence::FrameFilter::apply(cv::Mat & frame) {

}


VideoSequence::MetaData::MetaData() {
	fps = (double)UNKNOWN;
	length = (int)UNKNOWN;
}


void VideoSequence::MetaData::write(cv::FileStorage & fs) const {
	fs << NODE_FPS << fps;
	fs << NODE_LENGTH << length;
}


void VideoSequence::MetaData::read(cv::FileNode & nd) {
	CV_Assert(!nd.empty());

	if(!nd[NODE_FPS].empty()) {
		nd[NODE_FPS] >> fps;
	}

	if(!nd[NODE_LENGTH].empty()) {
		nd[NODE_LENGTH] >> length;
	}
}


CvVideoSequence::CvVideoSequence(const std::string & filename, FrameDescriptorExtractorPtr extractor) : VideoSequence(extractor) {
	if(filename == "") {
		CV_Error(CV_StsBadArg, "Invalid video sequence filename.");
	}

	capture = new cv::VideoCapture(filename);

	if(!capture->isOpened())
	{
		capture.release(); // TODO zkontrolovat
		CV_Error(CV_StsObjectNotFound, "Cannot open video sequence.");
	}

	metaData->fps = capture->get(CV_CAP_PROP_FPS);
	metaData->length = (int)capture->get(CV_CAP_PROP_FRAME_COUNT);
}


bool CvVideoSequence::step() {
    if(capture->get(CV_CAP_PROP_POS_FRAMES) < capture->get(CV_CAP_PROP_FRAME_COUNT)-1) {
        return capture->grab();
    }

    return false;
}


cv::Mat CvVideoSequence::frame() {
	cv::Mat frame;
	capture->retrieve(frame);
	return frame;
}


void CvVideoSequence::stepToStart() {
	capture->set(CV_CAP_PROP_POS_FRAMES, 0);
}


cv::Ptr<cv::VideoCapture> CvVideoSequence::getCapture() {
	return capture;
}


#ifdef VTAPI
VtVideoSequence::VtVideoSequence(Sequence *sequence, FrameDescriptorExtractorPtr extractor) : VideoSequence(extractor) {
	if(sequence == NULL) {
		CV_Error(CV_StsBadArg, "Null pointer to VTAPI Sequence given.");
	}
	this->sequence = sequence;
}


bool VtVideoSequence::step() {
	return sequence->next();
}


cv::Mat VtVideoSequence::frame() {
	Image* image = sequence->newImage();
	return cv::imread(image->getDataLocation());
}


void VtVideoSequence::stepToStart() {
	//TODO dodelat
}
#endif


}
