#include "VideoSequenceDataSource.h"


namespace vmatch {


const std::string VideoSequenceDataSource::EXT_PREPROCESSED_1 = ".xml";
const std::string VideoSequenceDataSource::EXT_PREPROCESSED_2 = ".yml";


VideoSequenceDataSource::VideoSequenceDataSource(FrameDescriptorExtractorPtr frameDescriptorExtractor, KeyFramesExtractorPtr keyFramesExtractor) {
	setFrameDescriptorExtractor(frameDescriptorExtractor);
	setKeyFramesExtractor(keyFramesExtractor);
}


VideoSequenceDataSource::VideoSequenceDataSource() {
	frameDescriptorExtractor = NULL;
	keyFramesExtractor = NULL;
}


bool VideoSequenceDataSource::isPreprocessedLocal(const std::string & name) const {
	std::string ext = name.substr(name.length()-4, 4);
	return (ext == EXT_PREPROCESSED_1 || ext == EXT_PREPROCESSED_2);
}


FrameDescriptorExtractorPtr VideoSequenceDataSource::getFrameDescriptorExtractor() {
	CV_Assert(frameDescriptorExtractor != NULL);
	return frameDescriptorExtractor;
}


KeyFramesExtractorPtr VideoSequenceDataSource::getKeyFramesExtractor() {
	CV_Assert(keyFramesExtractor != NULL);
	return keyFramesExtractor;
}


void VideoSequenceDataSource::setFrameDescriptorExtractor(FrameDescriptorExtractorPtr frameDescriptorExtractor) {
	CV_Assert(frameDescriptorExtractor != NULL);
	this->frameDescriptorExtractor = frameDescriptorExtractor;
}


void VideoSequenceDataSource::setKeyFramesExtractor(KeyFramesExtractorPtr keyFramesExtractor) {
	CV_Assert(keyFramesExtractor != NULL);
	this->keyFramesExtractor = keyFramesExtractor;
}


VideoSequenceDataPtr VideoSequenceDataSource::getSequenceData(std::string name) {
	if(name.length() > 3) {
		std::string prefix = name.substr(0, 3);
		std::string sequenceName = name.substr(3, name.length()-3);
	
		if(prefix == "pc:") {
			return getSequenceDataLocal(sequenceName);
		}
#ifdef VTAPI
		else if(prefix == "vt:") {
			return getSequenceDataVt(sequenceName);
		}
#endif
		else {
			return getSequenceDataLocal(name);
		}
	}

	return NULL;
}


VideoSequenceDataPtr VideoSequenceDataSource::getSequenceDataLocal(std::string name) {
	if(isPreprocessedLocal(name)) {
		return new VideoSequenceData(name);
	}
	else {
		VideoSequencePtr sequence = new CvVideoSequence(name, frameDescriptorExtractor);
		KeyFrames keyframes = keyFramesExtractor->extract(sequence);
		return new VideoSequenceData(keyframes, sequence->getMetaData());
	}
}


VideoSequenceDataSourcePtr VideoSequenceDataSource::create(Config config) {
	switch(config) {
		case CONF_COLHIST:
			return new ColHistVideoSequenceDataSource();
	}
	
	return NULL;
}


ColHistVideoSequenceDataSource::ColHistVideoSequenceDataSource() : VideoSequenceDataSource()
{
	setFrameDescriptorExtractor(new ColHistFrameDescriptorExtractor());
    setKeyFramesExtractor(new LinearKeyFramesExtractor(25));
}


#ifdef VTAPI
VideoSequenceDataPtr ColHistVideoSequenceDataSource::getSequenceDataVt(std::string name) {
	/************ CHECK IF ALREADY PREPROCESSD AND STORED IN DB *************/
	std::string filename = "";
	bool preprocessedAlreadyInDb = false;
	/************************************************************************/


	/**************** NOT IN DB ==> PREPROCESS AND STORE IT *****************/
	if(!preprocessedAlreadyInDb) {
		VideoSequenceDataPtr data = getSequenceDataLocal(filename);
		
		// for this sequence, store its data into DB:
		// *array of N keyframes:
		//    -every keyframe consists of:
        //      -int frameNumber
		//      -int segmentId
		//      -descriptor (array of 1024 uchars)
		// *some metadata (probably already present in DB, so no extra table is required)
		//    -double fps
		//    -int length (number of frames of the whole sequence)
		
		// keyframes
		KeyFrames keyframes = data->getKeyFrames();
		for(KeyFrames::iterator it = keyframes.begin(); it != keyframes.end(); it++) {
			KeyFramePtr kf = *it;			
			FrameDescriptor *frameDescriptorBase = kf->getFrameDescriptor();
			VectorFrameDescriptor *frameDescriptorVector = (VectorFrameDescriptor *)frameDescriptorBase;

			// store these items for every keyframe
			int frameNumber = kf->getFrameNumber();
			int segmentId = kf->getSegmentId();
			const cv::Mat descriptor = frameDescriptorVector->getVector(); // (1x1024, uchar)
		}
		
		// metadata
		VideoSequence::MetaData meta = data->getMetaData();
		double fps = meta.fps;
		int length = meta.length;

		// store all into DB
		// ...

		// data already available, just return them
		return data;
	}
	/************************************************************************/

	
	/************************ LOAD FROM DB AND RETURN ***********************/
	// load keyframes from DB	
	KeyFrames keyframes;

	// dummy loop:
	for(int i = 0; i < 10; i++) {
		cv::Mat data;         // from DB (1024x uchar)
		int frameNumber = -1; // from DB
		int segmentId = -1;   // from DB

		FrameDescriptorPtr descriptor = new ColHistFrameDescriptorExtractor::ColHistFrameDescriptor(data);
		KeyFramePtr kf = new KeyFrame(descriptor, frameNumber, segmentId);

		keyframes.push_back(kf);
	}

	// complete some data about the sequence
	VideoSequence::MetaData meta;
	meta.fps = -1; // necessary
	meta.length = -1; // not necessary, but useful 

	return new VideoSequenceData(keyframes, meta);
	/************************************************************************/

    //TODO zobrazit seznam videi v DB
}
#endif


}
