#include "similaritytimeline.h"
#include "ui_similaritytimeline.h"

SimilarityTimeline::SimilarityTimeline(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::SimilarityTimeline)
{
    ui->setupUi(this);

    highlightSegments = true;
    viewType = VIEW_REF;

    cursorPosition = 0;
    cursorGrabbed = false;
    cursorVisible = true;

    colorMatch = QColor::fromRgb(155, 211, 97);
    colorInjection = QColor::fromRgb(243, 190, 83);
    colorRemoval = QColor::fromRgb(224, 76, 146);
    colorRewriting = QColor::fromRgb(237, 146, 68);
}

SimilarityTimeline::~SimilarityTimeline()
{
    delete ui;
}

void SimilarityTimeline::setSegments(Segments newSegments)
{
    segments = newSegments;
    std::sort(segments.begin(), segments.end());

    if(!segments.check())
    {
        // TODO udelat poradne, byt by teoreticky nastat nemelo nikdy, jen chybou v kodu
        qDebug() << "Invalid segments list provided!";
        throw;
    }

    cursorPosition = 0;
    cursorGrabbed = false;;

    recomputeDimensions();
    repaint();
}

void SimilarityTimeline::clearSegments()
{
    segments.clear();
    repaint();
}

void SimilarityTimeline::recomputeDimensions()
{
    totalLength = 0;

    for(Segments::iterator it = segments.begin(); it != segments.end(); it++)
    {
        totalLength += it->getLengthSecMax();
    }
}

void SimilarityTimeline::setView(ViewType view)
{
    this->viewType = view;
    repaint();
}

double SimilarityTimeline::getScale() const
{
    return (segments.size() > 0)? geometry().width()/totalLength : -1;
}

bool SimilarityTimeline::isSegmentMatchingInCurrentView(const Segment & segment) const
{
    return (viewType == VIEW_REF)? segment.isMatchRef() : segment.isMatchQry();
}

void SimilarityTimeline::setHighlightSegments(bool doHighlightSegments)
{
    this->highlightSegments = doHighlightSegments;
}

void SimilarityTimeline::setColorMatch(QColor color) {
    colorMatch = color;
}

void SimilarityTimeline::setColorInjection(QColor color)
{
    colorInjection = color;
}

void SimilarityTimeline::setColorRemoval(QColor color)
{
    colorRemoval = color;
}

void SimilarityTimeline::setColorRewriting(QColor color)
{
    colorRewriting = color;
}

void SimilarityTimeline::setCursorVisible(bool cursorVisible)
{
    this->cursorVisible = cursorVisible;
}

int SimilarityTimeline::getSegmentIdxByPosition(double position, double *positionInsideSegment)
{
    double cumulativeLength = 0;
    int i = 0;

    while(i < segments.size() && cumulativeLength+segments[i].getLengthSecMax() < position*totalLength)
    {
        cumulativeLength += segments[i].getLengthSecMax();
        i++;
    }

    if(i >= segments.size())
    {
        return -1;
    }

    if(positionInsideSegment != NULL)
    {
        *positionInsideSegment = (position*totalLength - cumulativeLength)/segments[i].getLengthSecMax();
    }

    return i;
}

void SimilarityTimeline::paintEvent(QPaintEvent *e)
{
    Q_UNUSED(e);

    QPainter painter(this);

    if(segments.size() == 0)
    {
        painter.setPen(Qt::NoPen);
        painter.setBrush(Qt::darkGray);
        painter.drawRect(0, 0, geometry().width(), geometry().height());
    }
    else
    {
        paintSegments(painter);
        paintCursor(painter);
    }
}

void SimilarityTimeline::paintSegments(QPainter & painter)
{
    double scale = getScale();
    if(scale > 0)
    {
        painter.setPen(Qt::NoPen);

        double cumulativeLength = 0;

        for(size_t i = 0; i < segments.size(); i++)
        {
            Segment segment = segments[i];

            switch(segment.getDissimilarityType())
            {
                case Segment::MATCH:
                    painter.setBrush(colorMatch);
                    break;

                case Segment::DIS_INJECTION:
                    painter.setBrush((viewType == VIEW_REF)? colorInjection : colorMatch);
                    break;

                case Segment::DIS_REMOVAL:
                    painter.setBrush((viewType == VIEW_QRY)? colorRemoval : colorMatch);
                    break;

                case Segment::DIS_REWRITING:
                    painter.setBrush(colorRewriting);
                    break;
            }

            QRect rectangle(ceil(scale*cumulativeLength), 0, ceil(scale*segment.getLengthSecMax()), height());
            painter.drawRect(rectangle);

            if(highlightSegments && i == getSegmentIdxByPosition(cursorPosition))
            {
                painter.setBrush(QBrush(Qt::gray, Qt::BDiagPattern));
                painter.drawRect(rectangle);
            }

            cumulativeLength += segment.getLengthSecMax();
        }
    }
}

void SimilarityTimeline::paintCursor(QPainter & painter)
{
    if(!cursorVisible)
    {
        return;
    }

    int cursorPositionPx = getCursorPositionPx();

    painter.setPen(Qt::SolidLine);
    painter.drawLine(cursorPositionPx, 0, cursorPositionPx, height());

    double triangleSize = 0.3;
    int triangleSideLength = triangleSize*height();

    QPolygon triangleTop;
    triangleTop << QPoint(cursorPositionPx-triangleSideLength, 0) << QPoint(cursorPositionPx+triangleSideLength, 0) << QPoint(cursorPositionPx, triangleSideLength);

    QPolygon triangleBottom;
    triangleBottom << QPoint(cursorPositionPx-triangleSideLength, height()) << QPoint(cursorPositionPx+triangleSideLength, height()) << QPoint(cursorPositionPx, height()-triangleSideLength);

    painter.setPen(Qt::NoPen);
    painter.setBrush(Qt::black);
    painter.drawPolygon(triangleTop);
    painter.drawPolygon(triangleBottom);
}

int SimilarityTimeline::getCursorPositionPx() const
{
   return ceil(cursorPosition*(width()-1));
}

void SimilarityTimeline::setCursorPositionGlobal(double positionGlobal)
{
    if(positionGlobal >= 0 && positionGlobal <=1)
    {
        cursorPosition = positionGlobal;
        repaint();
    }
}

void SimilarityTimeline::setCursorPositionSegment(int segmentIdx, double positionSegment)
{
    if(segmentIdx >= segments.size())
    {
        // TODO
        throw;
    }

    double cumulativeLength = 0;
    for(size_t i = 0; i < segmentIdx; i++)
    {
        cumulativeLength += segments[i].getLengthSecMax();
    }

    setCursorPositionGlobal((cumulativeLength + positionSegment*segments[segmentIdx].getLengthSecMax())/totalLength);
}

bool SimilarityTimeline::isCursorNear(int x, int y, int neighbourhood) const
{
    int cursorPositionPx = getCursorPositionPx();
    return y >= 0 && y <= height()-1 && x >= cursorPositionPx-neighbourhood && x <= cursorPositionPx+neighbourhood;
}


void SimilarityTimeline::mouseMoveEvent(QMouseEvent *e)
{
    if(cursorGrabbed)
    {
        double newCursorPosition = std::min<double>(std::max<double>(((double)e->x())/(width()-1), 0), 1);
        setCursorPositionGlobal(newCursorPosition);

        double currentSegmentInsidePosition = 0;
        int currentSegmentIdx = getSegmentIdxByPosition(cursorPosition, &currentSegmentInsidePosition);

        if(currentSegmentIdx >= 0)
        {
            emit cursorMoved(currentSegmentIdx, currentSegmentInsidePosition, cursorPosition);
        }
    }
}

void SimilarityTimeline::mousePressEvent(QMouseEvent *e)
{
    if(cursorVisible && isCursorNear(e->x(), e->y(), 4))
    {
        cursorGrabbed = true;
        setMouseTracking(true);
    }
}

void SimilarityTimeline::mouseReleaseEvent(QMouseEvent *e)
{
    Q_UNUSED(e);

    if(cursorGrabbed)
    {
        cursorGrabbed = false;
        setMouseTracking(false);
    }
    else {
        mouseClickEvent(e);
    }
}

void SimilarityTimeline::mouseClickEvent(QMouseEvent *e)
{
    double clickPosition = ((double)e->x())/width();
    double currentSegmentInsidePosition = 0;
    int currentSegmentIdx = getSegmentIdxByPosition(clickPosition, &currentSegmentInsidePosition);

    if(currentSegmentIdx >= 0)
    {
        emit segmentClicked(currentSegmentIdx, currentSegmentInsidePosition, clickPosition);
    }
}

SimilarityTimeline::Segment::Segment(int startRef, int endRef, int startQry, int endQry, bool match, double fpsRef, double fpsQry)
{
    // TODO dodelat kontrolu pripustnych hodnot (fps > 0 apod.)

    this->startRef = startRef;
    this->endRef = endRef;

    this->startQry = startQry;
    this->endQry = endQry;

    this->lengthSecRef = (endRef-startRef+1)/fpsRef;
    this->lengthSecQry = (endQry-startQry+1)/fpsQry;

    this->dissimilarity = classifyDissimilarity(lengthSecRef, lengthSecQry, match);

    switch(dissimilarity)
    {
        case MATCH:
            this->matchRef = true;
            this->matchQry = true;
            break;

        case DIS_INJECTION:
            this->matchRef = false;
            this->matchQry = true;
            break;

        case DIS_REMOVAL:
            this->matchRef = true;
            this->matchQry = false;
            break;

        case DIS_REWRITING:
            this->matchRef = false;
            this->matchQry = false;
            break;
    }
}

SimilarityTimeline::Segment::Dissimilarity SimilarityTimeline::Segment::classifyDissimilarity(double lengthRef, double lengthQry, bool match)
{
    if(match)
    {
        return MATCH;
    }

    double ratio = (lengthRef > lengthQry)? lengthQry/lengthRef : lengthRef/lengthQry;

    if(ratio > 0.5)
    {
        return DIS_REWRITING;
    }
    else
    {
        if(lengthRef > lengthQry)
        {
            return DIS_REMOVAL;
        }
        else
        {
            return DIS_INJECTION;
        }
    }
}

int SimilarityTimeline::Segment::getStartRef() const
{
    return startRef;
}

int SimilarityTimeline::Segment::getEndRef() const
{
    return endRef;
}

int SimilarityTimeline::Segment::getStartQry() const
{
    return startQry;
}

int SimilarityTimeline::Segment::getEndQry() const
{
    return endQry;
}

double SimilarityTimeline::Segment::getLengthSecRef() const
{
    return lengthSecRef;
}

double SimilarityTimeline::Segment::getLengthSecQry() const
{
    return lengthSecQry;
}

double SimilarityTimeline::Segment::getLengthSecMax() const
{
    return std::max<double>(lengthSecRef, lengthSecQry);
}

double SimilarityTimeline::Segment::getLengthFramesRef() const
{
    return endRef-startRef+1;
}

double SimilarityTimeline::Segment::getLengthFramesQry() const
{
    return endQry-startQry+1;
}

double SimilarityTimeline::Segment::getLengthFramesMax() const
{
    return std::max<double>(getLengthFramesRef(), getLengthFramesQry());
}

SimilarityTimeline::Segment::Dissimilarity SimilarityTimeline::Segment::getDissimilarityType() const
{
    return dissimilarity;
}

bool SimilarityTimeline::Segment::isMatchRef() const
{
    return matchRef;
}

bool SimilarityTimeline::Segment::isMatchQry() const
{
    return matchQry;
}

bool SimilarityTimeline::Segment::operator<(const Segment & second) const
{
    if(this->startRef < second.startRef)
    {
        return true;
    }

    return this->startQry < second.startQry;
}

SimilarityTimeline::Segments::Segments() : SimilarityTimeline::_Segments()
{
}

SimilarityTimeline::Segments::Segments(vmatch::Segments segments) : SimilarityTimeline::_Segments()
{
    vmatch::Segments segmentsEnd = segments;
    segmentsEnd.sort();
    segmentsEnd.push_back(new vmatch::Segment(segments.getMetaDataRef().length, segments.getMetaDataRef().length, segments.getMetaDataQuery().length, segments.getMetaDataQuery().length));

    int refStart  = 0;
    int refEnd    = 0;
    int refLength = 0;
    int qryStart  = 0;
    int qryEnd    = 0;
    int qryLength = 0;

    double fpsRef = segments.getMetaDataRef().fps;
    double fpsQry = segments.getMetaDataQuery().fps;

    for(vmatch::Segments::iterator it = segmentsEnd.begin(); it != segmentsEnd.end(); it++) {
        vmatch::SegmentPtr segment = *it;

        refEnd = segment->getStartRef()-1;
        qryEnd = segment->getStartQuery()-1;

        if(refEnd < refStart) {
            refStart = refEnd = segment->getStartRef();
        }

        if(qryEnd < qryStart) {
            qryStart = qryEnd = segment->getStartQuery();
        }

        refLength = refEnd - refStart;
        qryLength = qryEnd - qryStart;

        // gap segment (between two matches)
        if(refLength > 0 || qryLength > 0) {
            Segment gapSegment(refStart, refEnd, qryStart, qryEnd, false, fpsRef, fpsQry);
            push_back(gapSegment);
        }

        // matching segment
        if(it+1 != segmentsEnd.end()) {
            Segment matchSegment(segment->getStartRef(), segment->getEndRef(), segment->getStartQuery(), segment->getEndQuery(), true, fpsRef, fpsQry);
            push_back(matchSegment);
        }

        refStart = segment->getEndRef()+1;
        qryStart = segment->getEndQuery()+1;
    }
}

bool SimilarityTimeline::Segments::check() const
{
    // TODO dodelat kontrolu
    return true;
}
