--[[
  File: arrayToTree.lua
  Desc: Convert list to tree
  Author: Jan Drazil
]]

function string:split(sep)
  -- Lua don't have built-in split function
  -- Source: http://lua-users.org/wiki/SplitJoin
  local sep, fields = sep, {}
  local pattern = string.format("([^%s]+)", sep)
  self:gsub(pattern, function(c) fields[#fields+1] = c end)
  return fields
end

function arrayToTree(data)
  --[[
    Create structure:
      root={'value':value, 'tree':[child_1, child_2, ... ,child_n]} for n>=0
      child_n = {'value':value, 'tree':[child_1, child_2, ... ,child_n]} for n>=0
  ]]
  result = createTree(data, 1)
  return result['tree']
end

function createTree(data, actualIndex)
  args=data[actualIndex]:split(';')
  value=tonumber(args[2]:match'^%s*(.*%S)')
  if(not value) then
    value=args[2]:match'^%s*(.*%S)'
  end
  local tree

  if(args[1] == 0) then
    return {['tree']={['value']=value, ['tree']={}}, ['lastIndex']=actualIndex}
  else
    tree = {['value']=value, ['tree']={}}
    for i=1,args[1] do
      newTree=createTree(data,actualIndex+1)
      table.insert(tree['tree'],newTree['tree'])
      actualIndex=newTree['lastIndex']
    end
    return {['tree']=tree, ['lastIndex']=actualIndex}
  end
end



---Generated---
function printTree(tree, spaces, pipePos)
  local size
  spaces = spaces or 0
  pipePos = pipePos or {}
  size=#tree['tree']
  if(size==0) then
    print("--" .. tostring(tree['value']))
  else
    io.write("--" .. tostring(tree['value']))
    spaces=spaces+2+string.len(tostring(tree['value']))
    table.insert(pipePos,spaces)
    for i=1,size do
      for j=0,spaces-1 do
        if(i~=1) then
          pipe=false
          for _,v in pairs(pipePos) do
            if(v == j) then
              io.write("|")
              pipe=true
              break
            end
          end
          if(not pipe) then
            io.write(" ")
          end
        end
      end
      if(i == size) then
        table.remove(pipePos)
      end
      printTree(tree['tree'][i],spaces,pipePos)
    end
  end
end



data = {"4;461562","4;820776","4;792292","4;201588","4;739956","4;786069","0;923677","0;324012","0;965885","0;175981","4;684810","0;372796","0;344434","0;854201","0;555277","4;646066","0;154078","0;662765","0;178578","0;927091","4;685204","0;883442","0;367348","0;772098","0;582771","4;500961","4;178084","0;867356","0;863287","0;299066","0;693573","4;548352","0;636543","0;15078","0;923176","0;918908","4;689684","0;794148","0;894477","0;487410","0;229957","4;873007","0;166256","0;435980","0;28049","0;366526","4;187736","4;694279","0;161735","0;522123","0;198678","0;97104","4;612718","0;710611","0;420421","0;695025","0;34196","4;773091","0;809531","0;669310","0;483702","0;979278","4;726595","0;528274","0;195035","0;956429","0;841481","4;661053","4;876799","0;308663","0;367685","0;211749","0;596093","4;513122","0;719875","0;122135","0;112646","0;245368","4;225269","0;412004","0;117171","0;529813","0;118310","4;214879","0;170067","0;810610","0;408812","0;310090","4;751224","4;961973","4;303675","0;24240","0;252379","0;452662","0;843050","4;71416","0;333774","0;669478","0;762051","0;343963","4;62732","0;160790","0;846522","0;921749","0;953341","4;457874","0;181916","0;814765","0;868382","0;957971","4;636523","4;587631","0;650665","0;247170","0;628615","0;489327","4;581172","0;266734","0;241555","0;768714","0;766948","4;248354","0;288313","0;73663","0;912988","0;356910","4;907379","0;126714","0;967159","0;483681","0;169000","4;443918","4;863166","0;426011","0;325272","0;977064","0;765701","4;635729","0;483036","0;340710","0;48181","0;941975","4;706480","0;700901","0;622669","0;93439","0;729391","4;535936","0;296828","0;537492","0;73988","0;863302","4;815774","4;595031","0;558108","0;581664","0;990309","0;332837","4;232258","0;395510","0;500051","0;944049","0;64242","4;878791","0;769075","0;453757","0;897270","0;180581","4;993808","0;665874","0;306935","0;497016","0;222502","4;481554","4;726494","4;317277","0;40571","0;632343","0;845469","0;691844","4;816177","0;780888","0;261400","0;726900","0;372844","4;891809","0;254565","0;792667","0;99987","0;579906","4;957089","0;580505","0;441981","0;886235","0;179795","4;945552","4;177947","0;559311","0;846674","0;77847","0;972327","4;497976","0;998404","0;119207","0;368633","0;347110","4;641831","0;380464","0;450430","0;824735","0;112023","4;626460","0;724998","0;248344","0;158795","0;718142","4;563339","4;946663","0;665368","0;259259","0;64511","0;879791","4;438568","0;863292","0;583010","0;901442","0;301126","4;476934","0;61402","0;591716","0;694076","0;780221","4;481412","0;14364","0;473908","0;981571","0;877727","4;327416","4;828573","0;132421","0;806742","0;700926","0;258855","4;756559","0;741544","0;245477","0;393968","0;758669","4;438862","0;833477","0;729662","0;673344","0;333675","4;319378","0;55751","0;69157","0;356168","0;359022","4;336467","4;454968","4;744470","0;682548","0;175740","0;739501","0;489513","4;936653","0;610969","0;950340","0;517084","0;743153","4;364990","0;811907","0;109670","0;865511","0;202871","4;529701","0;718506","0;678674","0;900971","0;353701","4;970905","4;472781","0;608252","0;83209","0;425097","0;466400","4;384033","0;429182","0;361859","0;918604","0;680869","4;15946","0;47530","0;391765","0;991052","0;420544","4;590092","0;672132","0;129588","0;883977","0;347628","4;39962","4;780316","0;835833","0;539565","0;995097","0;896511","4;60753","0;547332","0;352270","0;222097","0;241833","4;209738","0;549904","0;811080","0;1284","0;964531","4;709784","0;910819","0;987740","0;374176","0;215581","4;424012","4;663141","0;737245","0;537867","0;901228","0;361161","4;884924","0;465971","0;848499","0;135036","0;806452","4;151723","0;690529","0;161472","0;930158","0;89647","4;441113","0;295029","0;586161","0;384375","0;208480","4;708261","4;119794","4;743844","4;935533","0;293338","0;556368","0;782882","0;975242","4;180029","0;30669","0;439794","0;328014","0;364656","4;956456","0;898154","0;621785","0;557580","0;522698","4;682954","0;881481","0;931215","0;637405","0;556772","4;499749","4;329632","0;99479","0;901554","0;746424","0;401904","4;803478","0;578173","0;747582","0;354962","0;537648","4;287840","0;620900","0;537013","0;945254","0;989091","4;594743","0;14674","0;603468","0;894693","0;611494","4;587089","4;323800","0;720273","0;769521","0;826372","0;745669","4;718664","0;444494","0;893149","0;114907","0;635585","4;590503","0;658353","0;341483","0;857675","0;595665","4;654900","0;517077","0;86374","0;125627","0;92254","4;162042","4;244963","0;499996","0;734055","0;121501","0;690945","4;75679","0;77921","0;347437","0;911780","0;978201","4;350892","0;396189","0;787211","0;593700","0;653388","4;511953","0;461305","0;206395","0;57592","0;50210","4;415411","4;103633","4;121285","0;464735","0;910353","0;786833","0;136924","4;758556","0;909941","0;930294","0;416024","0;843966","4;297128","0;48595","0;488925","0;488080","0;121006","4;392532","0;995938","0;244438","0;637988","0;525275","4;852355","4;581806","0;731153","0;620267","0;847581","0;801271","4;1914","0;808260","0;312507","0;385969","0;162376","4;739454","0;990460","0;580292","0;289682","0;849103","4;307757","0;16265","0;710204","0;833741","0;798254","4;586656","4;200040","0;574843","0;280389","0;355790","0;227781","4;480980","0;301642","0;70663","0;34377","0;56121","4;852375","0;201176","0;320051","0;186514","0;591739","4;986807","0;310671","0;278911","0;762009","0;993388","4;343434","4;18560","0;356967","0;6375","0;943675","0;557741","4;238576","0;333612","0;532946","0;797775","0;488784","4;994349","0;170540","0;469524","0;109833","0;621111","4;583879","0;621213","0;122454","0;140595","0;541925","4;870319","4;686573","4;511839","0;569422","0;110550","0;769650","0;785356","4;12374","0;892085","0;189027","0;431766","0;59733","4;279067","0;489902","0;177877","0;863033","0;263376","4;794894","0;84200","0;780615","0;509596","0;418684","4;461388","4;9419","0;656275","0;13167","0;604001","0;945013","4;465549","0;28531","0;667293","0;521337","0;23626","4;847283","0;285241","0;270651","0;344181","0;466937","4;162646","0;413993","0;512601","0;19863","0;469180","4;920424","4;58871","0;192116","0;360054","0;922079","0;242928","4;905132","0;830475","0;908372","0;389176","0;668959","4;830126","0;239772","0;927922","0;9365","0;551587","4;353437","0;286235","0;415632","0;69593","0;746987","4;279685","4;141655","0;269703","0;781079","0;590120","0;647476","4;415187","0;447510","0;558795","0;684079","0;857534","4;360126","0;62658","0;304840","0;201546","0;927235","4;962874","0;722519","0;370732","0;438766","0;837000","4;688558","4;160810","4;276180","0;164736","0;42051","0;899981","0;809811","4;329578","0;350958","0;826343","0;333798","0;762934","4;978272","0;661527","0;957789","0;343042","0;360921","4;984276","0;301775","0;263091","0;551376","0;107842","4;294243","4;386876","0;298753","0;763593","0;430165","0;426597","4;489825","0;395412","0;863537","0;814009","0;412898","4;242468","0;545071","0;765338","0;870659","0;658275","4;726498","0;997468","0;833825","0;9887","0;164151","4;152568","4;334080","0;620599","0;343064","0;916607","0;841012","4;11758","0;888655","0;774845","0;152782","0;15986","4;66133","0;105519","0;226187","0;989635","0;62305","4;575996","0;995359","0;766472","0;754495","0;744123","4;964088","4;237339","0;289330","0;827558","0;692259","0;707025","4;778280","0;656129","0;687940","0;248133","0;695231","4;259237","0;241735","0;473586","0;918117","0;734794","4;737744","0;954518","0;372640","0;166938","0;525054","4;780086","4;983845","4;553932","4;209486","0;857087","0;375867","0;600022","0;651232","4;324412","0;640674","0;556702","0;540612","0;39081","4;844696","0;329168","0;457539","0;967549","0;227246","4;429132","0;644657","0;670535","0;105495","0;769159","4;785266","4;202232","0;299195","0;350209","0;296302","0;12569","4;711098","0;883671","0;922873","0;287655","0;899957","4;31869","0;406736","0;816916","0;807330","0;484876","4;2062","0;89583","0;631806","0;906702","0;346621","4;399365","4;216904","0;430660","0;56273","0;768591","0;778697","4;924475","0;818217","0;130893","0;30406","0;245184","4;796597","0;49097","0;414566","0;304","0;770738","4;100730","0;247328","0;994801","0;129994","0;67731","4;738834","4;579139","0;766453","0;110668","0;77818","0;639849","4;809457","0;618226","0;352749","0;987426","0;845753","4;732497","0;58235","0;734432","0;438378","0;201574","4;587112","0;704068","0;444592","0;648422","0;871458","4;87160","4;139590","4;738003","0;148346","0;621753","0;79808","0;382263","4;457594","0;511801","0;258872","0;279353","0;696228","4;627305","0;155941","0;310432","0;920524","0;705714","4;611652","0;842860","0;621389","0;254384","0;866409","4;20134","4;223820","0;267717","0;912695","0;385535","0;584456","4;652138","0;554628","0;91135","0;341719","0;854143","4;986148","0;395007","0;240990","0;281980","0;27122","4;622401","0;504603","0;701005","0;850533","0;938765","4;419253","4;916383","0;558295","0;962529","0;32891","0;229396","4;927525","0;877618","0;146083","0;710352","0;196681","4;667367","0;771496","0;358388","0;995719","0;556933","4;542632","0;19668","0;478595","0;359456","0;209059","4;480193","4;370457","0;678813","0;919992","0;820008","0;593822","4;276864","0;756899","0;850165","0;722207","0;661860","4;440045","0;846692","0;391557","0;72111","0;290203","4;562735","0;164108","0;946896","0;868978","0;58523","4;54003","4;413679","4;480882","0;321968","0;896049","0;518303","0;541146","4;330932","0;947810","0;531009","0;973476","0;269223","4;99215","0;401646","0;316870","0;834498","0;236263","4;554848","0;395501","0;873899","0;405481","0;115915","4;599377","4;918965","0;62574","0;865958","0;63177","0;990116","4;709722","0;364649","0;661764","0;568956","0;213789","4;729653","0;40606","0;535256","0;23064","0;492586","4;595107","0;254665","0;149408","0;907515","0;19926","4;172096","4;955654","0;753523","0;307362","0;772866","0;257019","4;80360","0;129325","0;230677","0;391970","0;103036","4;429641","0;642430","0;568800","0;160024","0;639258","4;319238","0;787027","0;442269","0;566940","0;28360","4;43258","4;598004","0;807590","0;135544","0;323071","0;941021","4;345202","0;405506","0;284581","0;705898","0;955121","4;144906","0;467136","0;17585","0;95085","0;307612","4;763111","0;727530","0;477329","0;274122","0;12141","4;128530","4;741616","4;550734","0;329713","0;685605","0;939064","0;534778","4;23168","0;469318","0;678536","0;356343","0;940115","4;170038","0;882240","0;302265","0;173653","0;345393","4;549525","0;387870","0;938782","0;998847","0;57740","4;641060","4;69402","0;661188","0;762004","0;57548","0;848084","4;737221","0;994125","0;231689","0;33493","0;104766","4;942200","0;508776","0;180020","0;688283","0;701177","4;675193","0;292796","0;651867","0;662325","0;369853","4;102990","4;957295","0;589545","0;885041","0;400830","0;299451","4;738805","0;84643","0;202775","0;683025","0;919492","4;320467","0;934754","0;997200","0;79808","0;325889","4;797917","0;471222","0;75402","0;245522","0;622404","4;275778","4;65225","0;724221","0;282582","0;693004","0;709883","4;317861","0;588058","0;551032","0;857361","0;127766","4;893767","0;284893","0;308312","0;524709","0;346474","4;664185","0;784513","0;445350","0;808214","0;588602","4;111142","4;346270","4;592725","4;270615","0;359318","0;554948","0;824030","0;119209","4;712359","0;62705","0;460094","0;332835","0;230212","4;216212","0;369034","0;691515","0;278647","0;751849","4;933484","0;975925","0;678826","0;212039","0;37529","4;589364","4;726548","0;405420","0;991737","0;878308","0;293950","4;330261","0;340291","0;522588","0;974718","0;557059","4;407784","0;800897","0;302554","0;148009","0;377261","4;746885","0;180782","0;221459","0;587132","0;369644","4;842720","4;350015","0;813344","0;465337","0;990369","0;530915","4;325507","0;477550","0;159303","0;871855","0;594009","4;974673","0;773309","0;470168","0;553295","0;568482","4;226605","0;968829","0;166384","0;875217","0;288499","4;323338","4;87005","0;148895","0;946192","0;496050","0;935355","4;122943","0;461433","0;26415","0;870432","0;343172","4;821993","0;261165","0;835896","0;213345","0;492127","4;9068","0;770284","0;328600","0;31475","0;472365","4;78334","4;303860","4;634018","0;916967","0;937657","0;211669","0;753315","4;163363","0;792178","0;822833","0;421750","0;527409","4;506839","0;920408","0;666019","0;903139","0;11467","4;762137","0;853140","0;381437","0;641578","0;370269","4;49683","4;16646","0;600572","0;730677","0;101213","0;529556","4;420749","0;23681","0;971450","0;484529","0;326246","4;154386","0;952218","0;212331","0;306087","0;830205","4;634667","0;121426","0;740316","0;665882","0;208049","4;436228","4;290303","0;676897","0;201973","0;839996","0;793679","4;75148","0;561366","0;344174","0;262773","0;933830","4;581382","0;750015","0;14764","0;169211","0;930212","4;58485","0;493146","0;171921","0;173062","0;450628","4;109462","4;222742","0;676555","0;671941","0;451057","0;405404","4;200434","0;990493","0;822524","0;805808","0;554908","4;363981","0;235453","0;266068","0;919543","0;107361","4;510929","0;29847","0;307645","0;607355","0;967864","4;954026","4;374776","4;805631","0;292346","0;581368","0;818877","0;929787","4;713879","0;667011","0;776551","0;213739","0;433101","4;36093","0;406170","0;678157","0;622958","0;761266","4;798201","0;180676","0;935736","0;646423","0;508461","4;496751","4;834904","0;166777","0;256845","0;175740","0;454883","4;132315","0;249863","0;874657","0;150859","0;973821","4;568279","0;58637","0;702742","0;492229","0;969082","4;722431","0;439455","0;701675","0;569574","0;733539","4;143166","4;670064","0;387295","0;77723","0;546579","0;334059","4;545611","0;361221","0;301636","0;307275","0;433507","4;164851","0;220196","0;766668","0;271222","0;968029","4;299094","0;890963","0;670852","0;793181","0;919584","4;687485","4;300842","0;654115","0;778471","0;119206","0;804990","4;110808","0;387151","0;540871","0;113967","0;15282","4;993031","0;43314","0;103983","0;704443","0;56094","4;105915","0;227348","0;690498","0;165965","0;432536","4;286516","4;895544","4;192541","0;356535","0;143690","0;607415","0;238104","4;365358","0;524745","0;71527","0;396767","0;79540","4;699950","0;19682","0;363905","0;139461","0;83177","4;604015","0;681061","0;356214","0;672323","0;261399","4;346054","4;632838","0;824743","0;997562","0;231089","0;815170","4;487750","0;324615","0;118557","0;837677","0;961113","4;414439","0;640814","0;346728","0;865690","0;381864","4;904939","0;967561","0;105375","0;170604","0;137418","4;274924","4;873218","0;664299","0;684371","0;569701","0;619581","4;9059","0;223139","0;534859","0;904849","0;53574","4;330379","0;914368","0;255151","0;65702","0;2074","4;785290","0;628569","0;275092","0;295143","0;382208","4;191690","4;437626","0;866330","0;26762","0;289958","0;596847","4;493755","0;112742","0;176608","0;650235","0;618067","4;709440","0;6485","0;641292","0;679320","0;941806","4;305014","0;561425","0;602833","0;311425","0;223963","4;351149","4;737344","4;33905","4;326855","4;770461","0;690516","0;970210","0;458148","0;436599","4;766977","0;750554","0;163152","0;418137","0;295906","4;47633","0;624062","0;728656","0;767957","0;120490","4;97427","0;228997","0;517437","0;880976","0;447816","4;397106","4;899858","0;457104","0;531725","0;342158","0;202320","4;91131","0;49927","0;909117","0;52403","0;942208","4;413949","0;118971","0;259625","0;599663","0;529717","4;854582","0;115796","0;79312","0;339127","0;441083","4;915012","4;97516","0;57232","0;886692","0;455716","0;135623","4;706382","0;655487","0;266035","0;56802","0;76077","4;640175","0;646107","0;543472","0;175336","0;818914","4;940358","0;650348","0;219846","0;871229","0;553859","4;29587","4;928751","0;606914","0;981168","0;650165","0;25915","4;671803","0;793170","0;992783","0;399065","0;808457","4;419408","0;895309","0;995619","0;555334","0;782127","4;735164","0;410929","0;879279","0;957713","0;922549","4;123101","4;203097","4;672579","0;196846","0;168467","0;540572","0;146260","4;811275","0;743656","0;383074","0;363636","0;984660","4;91787","0;439844","0;471714","0;787255","0;717399","4;480441","0;100640","0;4897","0;364151","0;187906","4;852807","4;398563","0;795453","0;535913","0;838628","0;940424","4;31912","0;837321","0;235474","0;206835","0;220699","4;654006","0;968881","0;852496","0;130620","0;618393","4;402668","0;559771","0;982090","0;46178","0;973743","4;632992","4;918239","0;371921","0;893084","0;984833","0;95587","4;382701","0;668957","0;492829","0;25778","0;415195","4;719463","0;629623","0;563854","0;653431","0;967371","4;799997","0;837130","0;764078","0;68179","0;635149","4;503314","4;286650","0;477133","0;507290","0;56106","0;978626","4;874106","0;424757","0;593808","0;703243","0;72968","4;826120","0;736711","0;702037","0;133758","0;908823","4;688995","0;953553","0;788900","0;156756","0;462235","4;779721","4;778379","4;859211","0;553321","0;436607","0;229762","0;862500","4;227845","0;705764","0;712060","0;283058","0;996242","4;497169","0;186955","0;518050","0;252997","0;236438","4;39563","0;923892","0;748335","0;976182","0;372357","4;436210","4;233403","0;155644","0;646455","0;429060","0;226325","4;306125","0;779033","0;60963","0;544316","0;122421","4;204274","0;875966","0;930168","0;188435","0;368384","4;652565","0;973493","0;880205","0;682560","0;121458","4;973688","4;716847","0;779812","0;913468","0;253346","0;327035","4;770637","0;813051","0;186750","0;667249","0;443417","4;786114","0;279500","0;101333","0;760412","0;130550","4;307544","0;357528","0;671660","0;627460","0;875405","4;165161","4;362109","0;899717","0;507462","0;375850","0;166882","4;474979","0;417740","0;823880","0;179588","0;144941","4;620553","0;800139","0;416531","0;683733","0;636011","4;202602","0;182645","0;486729","0;766641","0;298958","4;845428","4;589808","4;174375","0;812889","0;701055","0;413042","0;650726","4;540397","0;344784","0;808188","0;31775","0;819876","4;20853","0;840567","0;411528","0;529557","0;254052","4;702565","0;548232","0;515858","0;278585","0;870532","4;137489","4;776552","0;938686","0;450955","0;701205","0;464230","4;63597","0;643094","0;185267","0;278265","0;922527","4;235569","0;400490","0;450907","0;118060","0;308857","4;639201","0;799263","0;644962","0;534571","0;965884","4;246866","4;606552","0;201177","0;192550","0;706527","0;818074","4;478652","0;67955","0;266163","0;716258","0;642276","4;516492","0;264215","0;663785","0;89832","0;806048","4;224349","0;399743","0;777560","0;361834","0;392575","4;637736","4;719323","0;428165","0;698856","0;252473","0;835198","4;976374","0;274364","0;810359","0;448376","0;1845","4;587135","0;504717","0;28764","0;569077","0;573435","4;48380","0;797039","0;458014","0;640366","0;972568","4;300812","4;914811","4;388654","4;629336","0;21247","0;848854","0;577098","0;231311","4;190931","0;739231","0;19219","0;784568","0;523322","4;811480","0;402221","0;527051","0;719614","0;932458","4;848008","0;674249","0;628239","0;572266","0;726119","4;701788","4;600583","0;831946","0;292279","0;439117","0;731514","4;134184","0;363554","0;576648","0;76931","0;937107","4;348041","0;221301","0;259901","0;464510","0;991623","4;421392","0;369310","0;436209","0;118442","0;582349","4;108105","4;991922","0;648704","0;538340","0;216402","0;906926","4;662391","0;898281","0;997694","0;230261","0;826943","4;271563","0;367686","0;929683","0;491846","0;379562","4;312695","0;554549","0;128119","0;610853","0;79987","4;451290","4;874059","0;68593","0;254375","0;631659","0;785444","4;342883","0;8755","0;496278","0;924393","0;783504","4;46988","0;509453","0;855776","0;878072","0;488561","4;346998","0;113300","0;73357","0;405102","0;215677","4;48853","4;204034","4;739389","0;230385","0;761187","0;114152","0;393515","4;460911","0;886223","0;375652","0;356420","0;875922","4;291103","0;118886","0;105703","0;265757","0;110326","4;698456","0;446916","0;937846","0;195336","0;23347","4;47645","4;667120","0;419836","0;974523","0;518711","0;113779","4;739616","0;547105","0;47971","0;389909","0;189647","4;563599","0;751338","0;450427","0;239506","0;833957","4;566448","0;323773","0;519942","0;130211","0;443498","4;235356","4;576190","0;414111","0;181703","0;543966","0;696955","4;121079","0;151658","0;213789","0;225670","0;50885","4;82414","0;625030","0;225107","0;77321","0;422029","4;95505","0;953916","0;491862","0;61869","0;113560","4;579577","4;556875","0;10429","0;866967","0;81526","0;526400","4;439798","0;860820","0;208061","0;371238","0;222338","4;153533","0;431780","0;495412","0;315847","0;36344","4;331398","0;706085","0;850976","0;597258","0;317275","4;200278","4;228930","4;395167","0;115490","0;780637","0;754632","0;480786","4;503166","0;470951","0;176317","0;227402","0;556234","4;384000","0;560115","0;497502","0;130442","0;708480","4;741671","0;293718","0;858084","0;739031","0;848845","4;506116","4;840105","0;256448","0;602472","0;366105","0;843245","4;719367","0;815902","0;342105","0;693973","0;250241","4;242440","0;974744","0;551894","0;603025","0;226175","4;633097","0;806469","0;320555","0;450321","0;555096","4;498277","4;266065","0;739882","0;304352","0;378850","0;642237","4;965846","0;609743","0;760129","0;640186","0;61382","4;569074","0;435761","0;119455","0;21197","0;190071","4;720441","0;496123","0;763471","0;135538","0;875987","4;92842","4;980735","0;642570","0;856089","0;344217","0;686475","4;70836","0;522057","0;615564","0;869926","0;621736","4;19632","0;629987","0;744763","0;446796","0;328834","4;525124","0;573945","0;731682","0;254211","0;983803","4;716569","4;486386","4;5325","0;597410","0;768761","0;581984","0;392233","4;456729","0;657202","0;482718","0;519833","0;929495","4;774025","0;610401","0;706217","0;827059","0;78732","4;420041","0;942474","0;387392","0;357784","0;208496","4;434269","4;276885","0;871211","0;139220","0;660764","0;305265","4;91924","0;541186","0;33343","0;888812","0;607530","4;794945","0;892681","0;407963","0;587824","0;524973","4;976946","0;58573","0;802041","0;986536","0;906905","4;248423","4;76236","0;372210","0;495550","0;783643","0;715565","4;893164","0;668590","0;295225","0;613280","0;319350","4;239259","0;688462","0;349961","0;212270","0;333468","4;64707","0;68305","0;825499","0;290831","0;377836","4;181939","4;863345","0;441097","0;50031","0;574281","0;506246","4;92990","0;585575","0;227424","0;832819","0;709108","4;155310","0;753021","0;523973","0;695790","0;126875","4;675529","0;665438","0;874299","0;337436","0;8613","4;961403","4;399775","4;852107","4;325815","0;971900","0;999939","0;294371","0;653288","4;754084","0;462519","0;880447","0;270348","0;105742","4;785953","0;725883","0;89329","0;411458","0;713542","4;247577","0;952781","0;146767","0;266726","0;848551","4;689052","4;448730","0;648968","0;593452","0;755359","0;452712","4;946561","0;794071","0;838290","0;938025","0;826696","4;931221","0;636397","0;366731","0;200524","0;295535","4;233039","0;56228","0;123817","0;579951","0;268516","4;691274","4;964233","0;790925","0;99505","0;725161","0;30576","4;290008","0;187438","0;989044","0;208353","0;213432","4;975951","0;146227","0;38083","0;589855","0;511735","4;908573","0;310417","0;156484","0;365552","0;377131","4;482972","4;665380","0;33187","0;136935","0;811441","0;679022","4;408528","0;382353","0;233778","0;643903","0;590237","4;236671","0;235957","0;604741","0;200617","0;954865","4;551043","0;271481","0;855315","0;786338","0;161555","4;780293","4;93236","4;809939","0;16430","0;873690","0;592776","0;46680","4;340191","0;103","0;567990","0;948954","0;70770","4;685428","0;894058","0;272652","0;741794","0;317993","4;652912","0;709823","0;5532","0;25293","0;59322","4;157204","4;897853","0;347751","0;439029","0;882546","0;175600","4;908728","0;981568","0;66678","0;974999","0;871579","4;826960","0;254689","0;118094","0;619703","0;414175","4;409382","0;835958","0;137341","0;975336","0;703220","4;224196","4;179400","0;890227","0;655270","0;73593","0;820388","4;335333","0;999102","0;831988","0;340375","0;998413","4;855600","0;334439","0;23630","0;285231","0;455097","4;244296","0;685293","0;168262","0;100279","0;143802","4;311369","4;648449","0;796807","0;269273","0;346749","0;842109","4;207857","0;621201","0;48256","0;34994","0;347240","4;896283","0;17372","0;998097","0;592657","0;231782","4;811600","0;423200","0;511662","0;901693","0;858269","4;666758","4;496829","4;922740","0;938943","0;100949","0;591805","0;158116","4;486505","0;366872","0;657705","0;710794","0;924513","4;792413","0;297716","0;380205","0;968647","0;824179","4;624179","0;736727","0;661724","0;226208","0;594225","4;815682","4;506836","0;321807","0;378161","0;786579","0;611889","4;145930","0;933124","0;647484","0;447108","0;547324","4;798718","0;53787","0;654486","0;824541","0;403940","4;27152","0;749061","0;589674","0;431792","0;921206","4;208316","4;338175","0;527000","0;418594","0;347352","0;79084","4;524124","0;780898","0;274102","0;217071","0;510807","4;971256","0;571644","0;332605","0;753922","0;943759","4;787057","0;229928","0;537367","0;217449","0;349500","4;234394","4;179280","0;494509","0;235145","0;847747","0;764199","4;37246","0;995730","0;213527","0;924412","0;908125","4;600514","0;975828","0;279770","0;93480","0;450274","4;33380","0;135832","0;84858","0;200353","0;871715","4;206057","4;99573","4;731969","0;498556","0;390100","0;317770","0;529108","4;4557","0;296925","0;908739","0;497907","0;196967","4;198047","0;706339","0;795795","0;38597","0;316771","4;80857","0;420108","0;708678","0;691315","0;256601","4;34168","4;529832","0;956050","0;563772","0;178923","0;62862","4;299348","0;178129","0;634945","0;102412","0;948234","4;893300","0;292107","0;145291","0;536804","0;92436","4;515483","0;550821","0;862870","0;916095","0;18525","4;659609","4;150524","0;546776","0;268219","0;498685","0;540657","4;355474","0;580984","0;262244","0;755077","0;67365","4;552097","0;624979","0;988893","0;576059","0;882782","4;383294","0;375890","0;268285","0;918915","0;546505","4;402815","4;338810","0;617941","0;857865","0;229671","0;386398","4;126847","0;346736","0;305467","0;13030","0;411974","4;410803","0;843264","0;974768","0;855369","0;871576","4;621214","0;365993","0;939688","0;73580","0;382514","4;382990","4;667841","4;901377","4;779397","0;468196","0;388059","0;965010","0;834747","4;483911","0;273956","0;910677","0;629976","0;293654","4;647969","0;305848","0;317575","0;656434","0;829625","4;627791","0;951003","0;688085","0;558905","0;792280","4;923417","4;57655","0;354193","0;196665","0;884620","0;546269","4;526990","0;969280","0;210778","0;520518","0;313277","4;467248","0;342396","0;237483","0;702487","0;299018","4;82269","0;603708","0;340634","0;487794","0;682787","4;212523","4;309451","0;173923","0;46644","0;353186","0;240586","4;75979","0;808564","0;11968","0;683843","0;178332","4;869024","0;394357","0;734824","0;531671","0;640","4;506254","0;188005","0;588794","0;914738","0;625619","4;796001","4;848033","0;291458","0;467573","0;409331","0;460846","4;613650","0;697977","0;959820","0;396701","0;597344","4;158839","0;641210","0;746860","0;877850","0;508177","4;416827","0;395292","0;468388","0;676729","0;32264","4;888964","4;304095","4;187431","0;933669","0;799681","0;113768","0;159986","4;854469","0;717121","0;655155","0;225621","0;556874","4;630129","0;486052","0;201844","0;842893","0;281326","4;827138","0;700857","0;470209","0;755673","0;404568","4;671465","4;592307","0;434692","0;243390","0;295237","0;235879","4;63397","0;902085","0;727507","0;873713","0;615760","4;358815","0;40370","0;712758","0;521392","0;189614","4;933158","0;602228","0;42647","0;7075","0;124026","4;540031","4;896525","0;443059","0;581630","0;38412","0;76132","4;649862","0;293336","0;50173","0;991076","0;373692","4;780187","0;507658","0;466365","0;705528","0;697774","4;45485","0;468271","0;355488","0;517570","0;872037","4;245113","4;648949","0;856090","0;118769","0;164699","0;770757","4;57478","0;117789","0;395973","0;580577","0;268229","4;33326","0;190746","0;693722","0;769972","0;451011","4;126528","0;724189","0;920725","0;456767","0;553194","4;130408","4;400544","4;687867","0;49999","0;603111","0;490843","0;156779","4;878342","0;449078","0;485937","0;688353","0;490392","4;921616","0;911754","0;578563","0;135876","0;6810","4;532267","0;483231","0;915405","0;895075","0;861911","4;998414","4;735601","0;8774","0;515587","0;100126","0;485993","4;162964","0;80468","0;135692","0;504295","0;132695","4;237150","0;530016","0;861971","0;8715","0;922311","4;568560","0;409246","0;692872","0;689721","0;96449","4;955391","4;356351","0;375905","0;836213","0;324549","0;424284","4;686948","0;248756","0;783070","0;799217","0;663561","4;926084","0;40725","0;291182","0;951690","0;199530","4;632430","0;181478","0;535230","0;345101","0;913227","4;530472","4;499225","0;323360","0;838747","0;593397","0;902089","4;981242","0;658926","0;303343","0;585042","0;327033","4;125320","0;223448","0;282094","0;882451","0;563156","4;714990","0;383269","0;342784","0;963529","0;647942","4;548765","4;773318","4;490146","0;194557","0;649513","0;59098","0;586129","4;610217","0;848485","0;381051","0;633192","0;256630","4;444694","0;737560","0;327362","0;437502","0;541555","4;762470","0;196143","0;767114","0;197401","0;335582","4;493408","4;285177","0;733580","0;238537","0;307953","0;707688","4;622948","0;135830","0;854037","0;723896","0;130016","4;833684","0;419749","0;473982","0;526692","0;106503","4;309771","0;862758","0;569244","0;511081","0;226945","4;638483","4;534372","0;237879","0;698538","0;349995","0;68873","4;975758","0;722527","0;993824","0;705977","0;781661","4;657734","0;961646","0;647350","0;378053","0;496973","4;466991","0;415762","0;686224","0;585389","0;651637","4;446451","4;330167","0;620364","0;340715","0;999491","0;45452","4;742538","0;962462","0;977305","0;130161","0;312131","4;34453","0;754348","0;420561","0;835425","0;119664","4;225226","0;914045","0;318398","0;569547","0;786156","4;34023","4;202816","4;75413","4;310014","4;302197","0;482408","0;205667","0;34706","0;386779","4;491116","0;82601","0;839438","0;129327","0;168634","4;805941","0;93039","0;236448","0;4002","0;295934","4;667194","0;968792","0;55630","0;120374","0;495582","4;261854","4;7163","0;22737","0;566665","0;327606","0;643422","4;846860","0;354900","0;976736","0;80323","0;192615","4;559501","0;559807","0;21323","0;107457","0;840316","4;110346","0;377710","0;799710","0;40094","0;889496","4;903592","4;10817","0;215135","0;313780","0;841567","0;37653","4;418898","0;427060","0;668101","0;278","0;829535","4;556362","0;925403","0;241601","0;261078","0;116041","4;576415","0;826463","0;699642","0;547615","0;109100","4;756189","4;419078","0;441759","0;223908","0;459611","0;259096","4;936582","0;702656","0;107761","0;459904","0;237840","4;867464","0;485003","0;823885","0;876135","0;898181","4;797655","0;563683","0;613232","0;61201","0;680731","4;469738","4;692630","4;68024","0;102618","0;193610","0;770579","0;126109","4;690939","0;1391","0;21032","0;615368","0;391690","4;940912","0;600890","0;73341","0;926688","0;896821","4;429642","0;527736","0;857457","0;474716","0;92043","4;764962","4;449130","0;412823","0;157457","0;453075","0;402991","4;4619","0;176697","0;478489","0;938587","0;550611","4;258959","0;714173","0;826571","0;874365","0;281270","4;563661","0;498376","0;929203","0;63942","0;584942","4;961991","4;573450","0;256069","0;135636","0;522654","0;58622","4;913368","0;175183","0;230305","0;385740","0;675323","4;279372","0;661046","0;844960","0;787362","0;522876","4;410253","0;819167","0;148805","0;282163","0;261287","4;41692","4;187632","0;222444","0;255618","0;728271","0;659999","4;57665","0;645164","0;945623","0;908529","0;409762","4;96943","0;528454","0;145380","0;115651","0;588174","4;396189","0;167894","0;420693","0;880375","0;957606","4;695278","4;741442","4;843146","0;73182","0;655220","0;607205","0;117239","4;399970","0;804366","0;941014","0;248794","0;57690","4;673873","0;387098","0;334369","0;808527","0;935896","4;588878","0;189703","0;129628","0;244140","0;365439","4;951156","4;337083","0;812780","0;667144","0;573774","0;595194","4;381598","0;751703","0;678889","0;946530","0;287382","4;584490","0;858598","0;773832","0;756494","0;69303","4;148447","0;530259","0;783717","0;472442","0;916524","4;784183","4;214747","0;258104","0;999862","0;947165","0;986434","4;125838","0;663702","0;685431","0;143713","0;468803","4;229883","0;816352","0;983959","0;635142","0;137705","4;219467","0;475097","0;119556","0;889486","0;512297","4;255177","4;954912","0;730937","0;643705","0;29893","0;1267","4;967008","0;117696","0;486908","0;732","0;876660","4;85712","0;156338","0;528021","0;789523","0;1681","4;685071","0;670664","0;131497","0;563918","0;362647","4;19838","4;897950","4;845399","0;63539","0;418915","0;814119","0;395323","4;431840","0;158092","0;707078","0;567503","0;773010","4;266117","0;746595","0;757197","0;966582","0;591753","4;107469","0;295314","0;66282","0;995737","0;592132","4;518091","4;627449","0;335928","0;30802","0;789735","0;915526","4;948067","0;11620","0;454362","0;722108","0;229153","4;558286","0;623758","0;359662","0;10343","0;825346","4;118365","0;671998","0;211732","0;312646","0;969842","4;945184","4;389382","0;652303","0;607409","0;537763","0;52078","4;645109","0;19637","0;921767","0;986231","0;546873","4;457052","0;445773","0;590912","0;7497","0;221028","4;405498","0;543562","0;807779","0;896289","0;539270","4;979873","4;887924","0;231533","0;297375","0;651869","0;752661","4;818250","0;476157","0;858642","0;118269","0;61941","4;360331","0;708536","0;980972","0;485756","0;183949","4;951564","0;83885","0;798319","0;791968","0;780333","4;596582","4;655651","4;218612","4;948505","0;605219","0;764641","0;763882","0;289892","4;983737","0;281886","0;680611","0;350961","0;353194","4;572053","0;533647","0;851288","0;551942","0;570190","4;259821","0;948517","0;166825","0;260455","0;815697","4;972745","4;797294","0;595974","0;725998","0;798514","0;151412","4;448658","0;635920","0;964029","0;243052","0;949186","4;164143","0;595441","0;326920","0;535547","0;502405","4;58935","0;297592","0;122668","0;766418","0;914164","4;505957","4;817276","0;621374","0;656075","0;797715","0;24616","4;337186","0;219545","0;808380","0;608700","0;377540","4;157563","0;367705","0;804893","0;735874","0;439225","4;988326","0;401043","0;607863","0;94957","0;217341","4;280391","4;540996","0;529157","0;553898","0;992546","0;880485","4;871068","0;626039","0;470954","0;77480","0;520994","4;519072","0;816501","0;146840","0;99297","0;553009","4;693742","0;596844","0;754506","0;815655","0;629379","4;355776","4;886359","4;327293","0;786533","0;414011","0;14128","0;232273","4;52367","0;354793","0;701946","0;608228","0;260889","4;303354","0;579787","0;426061","0;276465","0;968743","4;8515","0;287169","0;581020","0;464288","0;152398","4;64889","4;595933","0;788184","0;337718","0;58111","0;90717","4;781116","0;238664","0;587565","0;429714","0;821973","4;964725","0;472506","0;688887","0;778258","0;72223","4;310541","0;842063","0;417559","0;31789","0;515948","4;97717","4;7864","0;901277","0;268739","0;42196","0;782469","4;211259","0;307597","0;411792","0;443921","0;91077","4;653174","0;384714","0;255625","0;530258","0;87602","4;721414","0;688129","0;607338","0;890483","0;245182","4;557245","4;512700","0;44691","0;125704","0;316665","0;307552","4;113004","0;747077","0;240891","0;732320","0;18985","4;587786","0;156591","0;326323","0;688314","0;372777","4;390703","0;128706","0;5729","0;668579","0;760121","4;299433","4;829176","4;101288","0;118277","0;186385","0;853549","0;880274","4;741378","0;261885","0;977311","0;716358","0;797160","4;244398","0;752056","0;967615","0;223261","0;886373","4;874473","0;486674","0;526348","0;674734","0;461299","4;471886","4;999455","0;661350","0;76490","0;497152","0;61551","4;887660","0;741984","0;749629","0;806479","0;952106","4;192520","0;280079","0;909693","0;631587","0;825850","4;212952","0;738887","0;441674","0;952291","0;116111","4;947146","4;233778","0;262718","0;621075","0;727759","0;103991","4;737444","0;393569","0;207657","0;284952","0;833762","4;262584","0;567255","0;832552","0;791424","0;954042","4;19354","0;546336","0;708115","0;745988","0;251382","4;404695","4;72903","0;309527","0;986225","0;544587","0;970955","4;554586","0;106452","0;460567","0;428657","0;540396","4;149639","0;315850","0;837990","0;614814","0;775500","4;121991","0;326243","0;98822","0;542030","0;473846","4;341518","4;423229","4;548498","0;973697","0;915578","0;820545","0;346552","4;567501","0;562315","0;889872","0;864537","0;308411","4;949652","0;598248","0;20075","0;222711","0;853493","4;830301","0;642951","0;420875","0;208376","0;596721","4;140009","4;738134","0;60531","0;844952","0;180335","0;782916","4;834211","0;258514","0;706281","0;286468","0;608731","4;379393","0;268342","0;500129","0;68188","0;23292","4;221144","0;255955","0;554966","0;572330","0;323104","4;88253","4;156262","0;11449","0;223113","0;315176","0;824447","4;70545","0;530562","0;316537","0;29224","0;556945","4;456442","0;396697","0;254825","0;414077","0;650236","4;274863","0;945444","0;818435","0;61861","0;678163","4;380648","4;7495","0;777768","0;832522","0;58279","0;178838","4;335380","0;945405","0;495586","0;278723","0;15262","4;73555","0;390213","0;550268","0;185536","0;992582","4;752820","0;485562","0;391444","0;215890","0;306480","4;153246","4;568908","4;884780","4;666423","0;448325","0;325528","0;829091","0;761499","4;330855","0;611720","0;784228","0;864574","0;398534","4;589065","0;86253","0;891385","0;182698","0;650856","4;152042","0;591230","0;77767","0;76486","0;146077","4;464692","4;938941","0;897139","0;157250","0;87879","0;227433","4;240897","0;44568","0;944731","0;722360","0;61806","4;916583","0;944808","0;562497","0;776579","0;332881","4;52551","0;846470","0;28113","0;86921","0;62767","4;203212","4;500208","0;199091","0;515331","0;639394","0;230241","4;902490","0;786598","0;728592","0;420011","0;107582","4;562727","0;273062","0;48733","0;777994","0;23533","4;595090","0;904455","0;337540","0;504880","0;811547","4;845173","4;453547","0;455042","0;733710","0;663718","0;83673","4;144386","0;413160","0;717683","0;702011","0;415333","4;772340","0;967365","0;802174","0;959480","0;537994","4;286360","0;940899","0;658065","0;127194","0;329092","4;992188","4;378131","4;475642","0;172424","0;739358","0;338691","0;270553","4;952834","0;303920","0;221922","0;848780","0;219478","4;353456","0;681126","0;961267","0;66421","0;524574","4;180220","0;745997","0;190069","0;965349","0;122550","4;44662","4;450928","0;571986","0;387113","0;787555","0;735687","4;881100","0;860723","0;145526","0;360874","0;372994","4;588252","0;629816","0;235230","0;801955","0;18635","4;406616","0;846582","0;848702","0;837104","0;22525","4;657477","4;999251","0;932281","0;534097","0;249116","0;654997","4;529828","0;677237","0;238247","0;353245","0;603538","4;23224","0;938712","0;703811","0;92053","0;507788","4;166403","0;208002","0;684114","0;640104","0;823834","4;854079","4;383012","0;27782","0;155364","0;217381","0;88370","4;934440","0;37278","0;490568","0;294100","0;151576","4;84022","0;822358","0;337082","0;945109","0;30628","4;876160","0;324289","0;98144","0;654116","0;757154","4;47970","4;660495","4;631467","0;512148","0;530254","0;616562","0;410696","4;775335","0;497075","0;8293","0;368335","0;681408","4;743939","0;915778","0;197937","0;301203","0;412327","4;636803","0;557428","0;174245","0;255223","0;466256","4;250280","4;686062","0;646857","0;614797","0;55133","0;661893","4;16499","0;302002","0;14172","0;947050","0;656329","4;502437","0;140089","0;270693","0;172129","0;241762","4;473847","0;742594","0;608793","0;569792","0;511360","4;566035","4;789751","0;386726","0;754052","0;791101","0;3232","4;515938","0;93394","0;964818","0;980045","0;643969","4;227224","0;233929","0;12051","0;500728","0;16076","4;10868","0;441437","0;430216","0;335851","0;336012","4;560996","4;823695","0;951471","0;199142","0;953631","0;578065","4;423393","0;353169","0;89751","0;583530","0;339049","4;556441","0;479199","0;111266","0;725388","0;632542","4;564376","0;597372","0;921413","0;93915","0;643795","4;162999","4;694017","4;35818","0;833195","0;19041","0;434579","0;650153","4;593616","0;593828","0;155107","0;911441","0;713790","4;837601","0;470384","0;628763","0;445989","0;898867","4;187032","0;596844","0;78743","0;284741","0;719539","4;375120","4;712574","0;642316","0;24431","0;30385","0;903182","4;332657","0;461037","0;88660","0;149431","0;933791","4;114095","0;558197","0;351307","0;736359","0;372690","4;903582","0;357686","0;434737","0;878515","0;822785","4;23143","4;621523","0;191257","0;431790","0;723169","0;462876","4;451345","0;769159","0;703834","0;380087","0;682861","4;904862","0;36937","0;189899","0;169157","0;81941","4;84108","0;283505","0;124952","0;90220","0;591968","4;534074","4;990392","0;856960","0;649222","0;567917","0;723133","4;965943","0;927666","0;588436","0;723924","0;15214","4;685188","0;767898","0;781768","0;708386","0;832347","4;262378","0;102032","0;785724","0;966146","0;932843","4;728834","4;581850","4;471098","4;348334","0;969463","0;784197","0;537647","0;413822","4;535670","0;639585","0;728784","0;190359","0;229719","4;188313","0;155834","0;209797","0;446344","0;436906","4;258199","0;809859","0;351012","0;112721","0;741056","4;480970","4;674240","0;938629","0;650160","0;376780","0;105550","4;331589","0;932554","0;541698","0;682579","0;48651","4;375491","0;747170","0;457550","0;378938","0;876170","4;224223","0;339730","0;632368","0;302253","0;527928","4;139058","4;721662","0;351935","0;273324","0;635084","0;690860","4;676499","0;679093","0;824158","0;778644","0;98865","4;227720","0;900909","0;115039","0;640730","0;171031","4;993763","0;253329","0;59564","0;495268","0;22677","4;305422","4;194662","0;271194","0;940622","0;241510","0;785224","4;949834","0;904679","0;83640","0;521701","0;976041","4;791986","0;325902","0;180644","0;123632","0;634094","4;34502","0;19823","0;383385","0;771195","0;2965","4;411710","4;773413","4;387076","0;940527","0;294460","0;945225","0;25422","4;480355","0;112680","0;385317","0;68546","0;773120","4;976934","0;410332","0;785926","0;413722","0;206616","4;139494","0;683729","0;518679","0;899692","0;25573","4;257942","4;535870","0;715794","0;827111","0;863705","0;500430","4;685323","0;294050","0;189456","0;812109","0;324647","4;714258","0;368373","0;359707","0;998061","0;96052","4;895585","0;174387","0;932834","0;106086","0;313253","4;139287","4;333092","0;188216","0;132636","0;378515","0;279435","4;626780","0;43184","0;717902","0;88041","0;759223","4;225297","0;20036","0;461266","0;169813","0;751368","4;968359","0;911455","0;720161","0;89979","0;16610","4;719313","4;682776","0;375422","0;462900","0;840210","0;414238","4;89135","0;116108","0;77777","0;891920","0;62225","4;369125","0;804074","0;351775","0;602311","0;157664","4;113052","0;303147","0;225265","0;440868","0;437594","4;567828","4;211197","4;619169","0;167454","0;56682","0;418004","0;320056","4;46797","0;751479","0;415410","0;378543","0;988908","4;107340","0;529668","0;685392","0;276217","0;609967","4;683106","0;178821","0;542415","0;514180","0;491746","4;568669","4;402131","0;238019","0;457144","0;490274","0;427645","4;18041","0;859871","0;353035","0;341121","0;459202","4;147435","0;895160","0;29487","0;764315","0;634987","4;817503","0;693325","0;448531","0;128508","0;390444","4;151191","4;925331","0;538512","0;829081","0;198300","0;18851","4;998826","0;695944","0;528911","0;156259","0;10505","4;432733","0;291539","0;700937","0;289676","0;805797","4;165313","0;147992","0;388810","0;409623","0;16245","4;741408","4;492026","0;533340","0;603","0;344876","0;13187","4;439930","0;587535","0;386255","0;759820","0;952986","4;563176","0;364635","0;487217","0;698450","0;758046","4;348556","0;44908","0;739148","0;970507","0;235060","4;192670","4;155565","4;235850","0;240263","0;62561","0;335559","0;482914","4;8884","0;434636","0;888370","0;168985","0;783493","4;832976","0;246120","0;667507","0;82350","0;732028","4;903396","0;178966","0;802949","0;614954","0;839526","4;163819","4;410385","0;223879","0;389992","0;894125","0;629173","4;678640","0;348253","0;124065","0;959025","0;333139","4;290155","0;612730","0;844582","0;470396","0;568534","4;283910","0;386661","0;995764","0;262618","0;691204","4;676298","4;872150","0;684172","0;31462","0;495702","0;12449","4;411952","0;96995","0;468974","0;646926","0;413678","4;635358","0;41027","0;675638","0;685545","0;818687","4;196133","0;628910","0;468379","0;215841","0;161294","4;425714","4;555257","0;596831","0;561771","0;396687","0;472723","4;407159","0;473110","0;29139","0;20608","0;479707","4;887983","0;800232","0;958077","0;302056","0;355620","4;303914","0;549256","0;728801","0;239571","0;475837","4;464263","4;530107","4;785126","4;549846","4;912609","0;597019","0;64943","0;742508","0;501630","4;162390","0;238887","0;665870","0;700187","0;117327","4;826571","0;673762","0;350842","0;69992","0;362204","4;264345","0;602421","0;670374","0;198181","0;515784","4;347961","4;611953","0;352821","0;551023","0;305640","0;347865","4;358248","0;494091","0;727963","0;708817","0;611611","4;199432","0;950969","0;247274","0;152904","0;697133","4;325328","0;16434","0;297328","0;863931","0;327177","4;26414","4;328382","0;908578","0;285041","0;427055","0;62091","4;297359","0;568424","0;899624","0;95060","0;935921","4;589797","0;371444","0;275833","0;462204","0;286418","4;606492","0;732354","0;521813","0;726111","0;947271","4;413726","4;958209","0;81713","0;673871","0;967323","0;777156","4;14600","0;708857","0;393222","0;696583","0;741159","4;388721","0;833441","0;325320","0;447815","0;590201","4;41340","0;96421","0;595343","0;32652","0;373586","4;644560","4;182841","4;608393","0;219170","0;32267","0;826717","0;785523","4;324052","0;821613","0;568610","0;499843","0;276187","4;371321","0;67282","0;676696","0;413380","0;57924","4;617940","0;306780","0;893071","0;158280","0;176619","4;120609","4;66417","0;275752","0;910715","0;453561","0;868085","4;13698","0;590398","0;810495","0;819875","0;762299","4;13829","0;613924","0;588869","0;56792","0;54585","4;907302","0;648822","0;882582","0;213822","0;808768","4;847655","4;742740","0;480201","0;402174","0;987229","0;924682","4;85334","0;703547","0;432417","0;439258","0;73125","4;623853","0;825955","0;463277","0;386291","0;515983","4;398393","0;744718","0;89546","0;586607","0;92755","4;663421","4;768918","0;517633","0;305061","0;885939","0;772866","4;54807","0;133461","0;295626","0;668546","0;675967","4;348539","0;26528","0;951767","0;824584","0;445630","4;842739","0;229715","0;398757","0;784312","0;516776","4;476105","4;730732","4;116653","0;631575","0;607001","0;739701","0;278472","4;5594","0;364527","0;266274","0;430524","0;108272","4;719123","0;179508","0;742594","0;303071","0;536334","4;53152","0;832147","0;337645","0;224954","0;250586","4;763126","4;249100","0;486201","0;398252","0;547142","0;493270","4;783541","0;525740","0;619399","0;751171","0;80424","4;532060","0;646562","0;940298","0;954816","0;586340","4;572144","0;223372","0;47122","0;455386","0;349463","4;10830","4;758746","0;433024","0;852542","0;30977","0;625","4;152014","0;925516","0;717896","0;611580","0;874180","4;230921","0;322440","0;484275","0;165807","0;852877","4;836516","0;465961","0;130724","0;51313","0;422408","4;860958","4;811094","0;815841","0;268938","0;803211","0;879870","4;749401","0;49854","0;313908","0;5061","0;602236","4;257930","0;849","0;521803","0;71046","0;392544","4;772045","0;553511","0;299011","0;365977","0;39534","4;556063","4;101304","4;421363","0;297950","0;43246","0;510812","0;145672","4;764486","0;681931","0;596969","0;714679","0;588168","4;582394","0;340191","0;841611","0;322007","0;188789","4;738305","0;648843","0;384111","0;399132","0;536565","4;852716","4;102146","0;664519","0;604353","0;422082","0;636442","4;95939","0;285690","0;11837","0;85787","0;877064","4;554013","0;638102","0;2875","0;664764","0;637300","4;358644","0;263686","0;506480","0;441000","0;414797","4;56164","4;838164","0;990037","0;87837","0;830568","0;15348","4;537488","0;315075","0;723203","0;686026","0;875023","4;218257","0;102611","0;571811","0;129419","0;88764","4;965534","0;287622","0;704068","0;806816","0;556735","4;152684","4;898882","0;823173","0;857959","0;457771","0;465029","4;225836","0;929037","0;975784","0;168758","0;432444","4;245820","0;906280","0;381998","0;129665","0;130180","4;962330","0;826786","0;510232","0;620","0;808941","4;951133","4;981166","4;947955","4;449531","0;812309","0;271029","0;711798","0;288810","4;775802","0;18998","0;163957","0;489697","0;467923","4;115525","0;987251","0;388075","0;232093","0;319499","4;447276","0;237361","0;788600","0;63024","0;763164","4;600200","4;530187","0;442181","0;52351","0;872883","0;626446","4;531592","0;154077","0;537383","0;449698","0;895154","4;794166","0;885414","0;948792","0;642432","0;711357","4;39594","0;73802","0;563874","0;762468","0;132354","4;576888","4;694003","0;105142","0;517739","0;299435","0;746085","4;120922","0;697803","0;764731","0;623581","0;559114","4;991695","0;942199","0;983587","0;445103","0;999480","4;345927","0;23171","0;5258","0;340128","0;186196","4;88624","4;322265","0;763875","0;305841","0;257351","0;89927","4;96834","0;550218","0;936150","0;595357","0;30527","4;182100","0;638729","0;676394","0;707036","0;631658","4;818258","0;534497","0;285126","0;729510","0;294969","4;533881","4;238578","4;945083","0;186951","0;698672","0;855733","0;571365","4;157822","0;337585","0;252190","0;550126","0;280969","4;381764","0;812000","0;594458","0;378597","0;501876","4;714823","0;382914","0;270666","0;969815","0;916476","4;699474","4;353472","0;57599","0;861545","0;540042","0;464949","4;202717","0;150137","0;405430","0;365415","0;224395","4;388515","0;282315","0;183730","0;536698","0;880518","4;968012","0;802895","0;249370","0;880085","0;146715","4;497165","4;398434","0;164539","0;628909","0;544288","0;797648","4;251519","0;701174","0;858078","0;782674","0;910906","4;92994","0;728233","0;337791","0;417747","0;714575","4;97270","0;930382","0;760415","0;801724","0;839471","4;756600","4;343824","0;738068","0;339276","0;138201","0;12003","4;55112","0;631164","0;89927","0;769038","0;555324","4;964898","0;910001","0;859181","0;135915","0;361216","4;18482","0;470694","0;336808","0;966195","0;200813","4;171667","4;660514","4;239421","0;2976","0;434963","0;979927","0;479293","4;53167","0;94371","0;543745","0;107108","0;226037","4;374798","0;586403","0;713162","0;84313","0;263136","4;574863","0;843843","0;429224","0;955777","0;658635","4;475810","4;823416","0;508486","0;399792","0;436101","0;712017","4;470390","0;817267","0;409357","0;822460","0;98244","4;802386","0;87508","0;309474","0;366695","0;189084","4;689785","0;794993","0;878902","0;773366","0;334159","4;232260","4;297721","0;217555","0;421479","0;803155","0;40846","4;637539","0;505837","0;888740","0;702667","0;883416","4;665573","0;945453","0;75181","0;94346","0;608931","4;903642","0;870594","0;937797","0;591391","0;278793","4;675208","4;192243","0;354863","0;637343","0;938432","0;591828","4;716146","0;630246","0;558618","0;900699","0;680982","4;182276","0;102279","0;695531","0;631579","0;168605","4;147606","0;956999","0;559866","0;21260","0;277429","4;10633","4;694455","4;110027","0;57111","0;802075","0;800868","0;367694","4;198451","0;200661","0;602769","0;390321","0;595730","4;473041","0;210540","0;863541","0;625753","0;338938","4;841902","0;812469","0;667417","0;507369","0;121427","4;172569","4;692559","0;220708","0;603569","0;992613","0;208843","4;874323","0;932946","0;255176","0;949942","0;176904","4;268590","0;131968","0;675742","0;362988","0;144318","4;698150","0;258196","0;887941","0;811828","0;700015","4;13403","4;458490","0;563526","0;884045","0;436852","0;291932","4;29649","0;372347","0;5221","0;878418","0;321885","4;39430","0;831525","0;266118","0;644070","0;203322","4;199597","0;806882","0;114882","0;673968","0;710288","4;729099","4;338347","0;340415","0;465069","0;216190","0;864672","4;586642","0;372031","0;194311","0;835111","0;30691","4;990363","0;315525","0;412339","0;926970","0;408430","4;762622","0;316303","0;839157","0;451395","0;879610","4;39598","4;261445","4;453832","4;886255","0;72574","0;541478","0;111354","0;439579","4;336304","0;726746","0;954933","0;509426","0;956900","4;473614","0;327767","0;34604","0;696685","0;103853","4;3536","0;905912","0;566581","0;199332","0;546052","4;662392","4;943784","0;431733","0;178779","0;155763","0;460937","4;995981","0;17900","0;324702","0;650065","0;966993","4;991060","0;590486","0;654279","0;319773","0;15531","4;197688","0;929867","0;595630","0;476719","0;537538","4;202403","4;734276","0;349874","0;940973","0;716159","0;323440","4;145641","0;274163","0;184834","0;865866","0;405891","4;641465","0;733659","0;83027","0;232406","0;252138","4;842630","0;708316","0;538918","0;536440","0;315512","4;663107","4;928047","0;903958","0;80185","0;620706","0;73809","4;173335","0;615055","0;331818","0;706843","0;130866","4;240650","0;833597","0;510020","0;416965","0;221436","4;216316","0;681361","0;831289","0;8380","0;42268","4;120168","4;995972","4;433673","0;37953","0;877353","0;283932","0;123467","4;630113","0;584698","0;717886","0;840047","0;244680","4;153332","0;202411","0;989688","0;756437","0;194027","4;843244","0;485491","0;376586","0;743974","0;610810","4;851576","4;471886","0;91644","0;56494","0;649005","0;46200","4;12857","0;103845","0;860577","0;414339","0;472322","4;309389","0;832974","0;377366","0;976998","0;667009","4;406747","0;595637","0;599201","0;464245","0;825378","4;880143","4;91406","0;805266","0;600271","0;32868","0;5108","4;868316","0;621065","0;229651","0;648715","0;753499","4;143617","0;162270","0;460678","0;315235","0;701234","4;336817","0;532870","0;25868","0;725675","0;572955","4;644173","4;810143","0;612337","0;267198","0;526191","0;921802","4;577954","0;775134","0;396108","0;189667","0;256655","4;829059","0;423956","0;127679","0;104141","0;334836","4;788293","0;305207","0;464140","0;167242","0;286479","4;110256","4;11173","4;761028","0;386519","0;936658","0;406839","0;298199","4;900864","0;28869","0;505302","0;156424","0;77773","4;74311","0;158780","0;758693","0;380492","0;575223","4;939985","0;653217","0;801991","0;791113","0;306768","4;865383","4;866265","0;84520","0;658456","0;874171","0;532731","4;701619","0;661272","0;223635","0;645419","0;855794","4;758005","0;889007","0;298707","0;316573","0;271399","4;822397","0;710158","0;235766","0;398854","0;954888","4;452234","4;803929","0;913161","0;81521","0;712470","0;128511","4;210729","0;55325","0;779120","0;895856","0;368700","4;549325","0;130710","0;935901","0;960114","0;274151","4;457529","0;815328","0;421853","0;859637","0;566384","4;751206","4;565054","0;535449","0;888598","0;833122","0;466305","4;926256","0;797394","0;468372","0;605660","0;493122","4;731805","0;413276","0;860428","0;271679","0;961585","4;668216","0;586234","0;620578","0;515608","0;326674","4;449189","4;557511","4;732001","0;35814","0;469565","0;879581","0;901564","4;117020","0;642084","0;65419","0;733669","0;698204","4;183860","0;387978","0;412204","0;660364","0;278592","4;366745","0;199840","0;831309","0;59233","0;575874","4;429404","4;658948","0;826873","0;946338","0;701122","0;12241","4;780233","0;596808","0;365281","0;600705","0;648823","4;918495","0;140778","0;516992","0;318826","0;638174","4;804040","0;869755","0;236701","0;601614","0;751841","4;781475","4;873847","0;438394","0;444757","0;868244","0;848566","4;362431","0;557347","0;816499","0;354416","0;924657","4;849574","0;408098","0;34797","0;425367","0;465749","4;743631","0;747671","0;618176","0;730484","0;526761","4;429828","4;8247","0;399383","0;41416","0;555170","0;872550","4;321578","0;163831","0;794051","0;194299","0;321037","4;18432","0;298732","0;786466","0;842100","0;933969","4;418842","0;938114","0;108624","0;636768","0;602187","4;429647","4;89293","4;334410","4;484882","0;379251","0;788368","0;886113","0;938250","4;233465","0;853280","0;796432","0;769769","0;176505","4;357753","0;724411","0;780200","0;993864","0;333925","4;971352","0;323242","0;116269","0;110576","0;916542","4;562241","4;673775","0;168613","0;672460","0;163672","0;350725","4;748458","0;132451","0;523695","0;755769","0;995123","4;789416","0;687134","0;477694","0;471280","0;869863","4;111957","0;719380","0;192797","0;397222","0;218485","4;559354","4;783107","0;610157","0;727095","0;409889","0;969978","4;526946","0;41291","0;23334","0;858395","0;914563","4;728181","0;123078","0;899274","0;681596","0;527090","4;843786","0;644","0;264406","0;335585","0;825503","4;419135","4;961413","0;212387","0;788918","0;348798","0;73256","4;295855","0;722756","0;183170","0;449202","0;38699","4;193200","0;193907","0;388333","0;981108","0;311658","4;580473","0;25450","0;653238","0;71931","0;923954","4;879117","4;483146","4;526356","0;362717","0;414298","0;253769","0;24219","4;289538","0;292961","0;382782","0;799307","0;15453","4;827217","0;761170","0;296542","0;496563","0;315762","4;703573","0;689774","0;24623","0;732625","0;984154","4;549254","4;587334","0;23164","0;511717","0;760244","0;331367","4;894647","0;987746","0;349728","0;987693","0;228835","4;896209","0;594329","0;224108","0;342443","0;885680","4;379471","0;762792","0;258651","0;147775","0;434177","4;491036","4;254682","0;16196","0;436462","0;771474","0;724797","4;675747","0;393528","0;174012","0;936388","0;3078","4;16849","0;242894","0;529945","0;964982","0;935486","4;86713","0;945013","0;363376","0;335443","0;760873","4;321992","4;16950","0;881253","0;12846","0;687791","0;481238","4;65785","0;212682","0;137844","0;33016","0;652052","4;102963","0;100894","0;601999","0;673580","0;759727","4;556653","0;75023","0;804591","0;945852","0;467049","4;506579","4;129664","4;46344","0;303752","0;753769","0;166785","0;548564","4;794551","0;68658","0;967766","0;808308","0;572533","4;793345","0;114680","0;169483","0;107505","0;269015","4;525484","0;869493","0;54494","0;253771","0;705037","4;549300","4;63881","0;871420","0;472439","0;643067","0;717291","4;630391","0;918877","0;306484","0;552476","0;181910","4;411212","0;633934","0;548254","0;845089","0;128668","4;820696","0;126649","0;387199","0;347956","0;66350","4;267513","4;693664","0;73708","0;121593","0;626323","0;277050","4;810974","0;813376","0;220049","0;214924","0;12075","4;465004","0;51147","0;101285","0;570413","0;623907","4;5854","0;982048","0;534070","0;371659","0;516903","4;4986","4;335223","0;891715","0;53942","0;127957","0;152846","4;368322","0;746196","0;822924","0;490748","0;464305","4;63264","0;518700","0;566024","0;286768","0;671985","4;936945","0;23489","0;867569","0;398353","0;195555","4;250818","4;656257","4;818221","0;646758","0;605965","0;732419","0;447695","4;513230","0;136453","0;149313","0;202471","0;473604","4;186365","0;935945","0;760199","0;473127","0;904618","4;71367","0;798907","0;746109","0;7784","0;830376","4;525761","4;299835","0;321803","0;324454","0;219026","0;526603","4;473848","0;227363","0;783885","0;501746","0;238898","4;728637","0;8755","0;129693","0;913511","0;657478","4;27579","0;174948","0;378063","0;652186","0;424169","4;458045","4;147941","0;133857","0;628738","0;243643","0;751086","4;609972","0;855071","0;312419","0;202022","0;313485","4;244285","0;502819","0;563670","0;865778","0;914492","4;282391","0;907175","0;966614","0;758735","0;92153","4;226977","4;7318","0;507953","0;78413","0;924106","0;312767","4;462389","0;556021","0;311400","0;821531","0;594664","4;525228","0;856255","0;791613","0;146135","0;147819","4;100602","0;127338","0;287559","0;524439","0;747953"}
printTree(arrayToTree(data))