/*!
 * \file axi_tsu_ctl.c
 * \brief Tool to control timestamp unit
 * \author      Vladimir Smotlacha <vs@cesnet.cz>
 * \author      Vojtech Vitek (V-Teq) <vojtech.vitek@gmail.com>
 * \author      Jan Stourac <xstour03@stud.fit.vutbr.cz>
 * \author Tomas Fukac  <xfukac00@stud.fit.vutbr.cz>
 * \date 2012
 *
 * Copyright (C) 2012 Brno University of Technology
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 */

#include "tsu.h"

#include "hwio.h"
#include "hwio_tool.h"
#include "hwio_comp_spec.h"



#define XCTL_OPTS "Df:F:p:s:t:"

/*	#nodef CONFIG_NTP_PPS_DEBUG */
#define CONFIG_NTP
#define CONFIG_NTP_PPS

#define _GNU_SOURCE

/* system time variables */

#define	ADJ_SERVED_MODE_BITS	(MOD_OFFSET|MOD_FREQUENCY|MOD_MAXERROR| \
      MOD_ESTERROR|MOD_STATUS|MOD_TIMECONST| \
      MOD_PPSMAX|MOD_TAI|MOD_MICRO|MOD_NANO| \
      MOD_CLKB|MOD_CLKA| \
      ADJ_TICKADJ|ADJ_TIMETICK|ADJ_ADJTIME)


#define PPM_MAX		30

#define FILT_BUF_SZ 	4

/* NOM_HZ_LOG aproximates binaary logarithm of the NOM_HZ,
   the value is 26 or 27 - i.e. 2^26 = 67.1M, 2^27 = 134.2M
 */
#define NOM_HZ_LOG	26

#define MAX_ADJ		2097152	/* cca 128 us/s - 16 * 2^17 */
#define MAX_DRIFT	524288 /* cca 30 ppm - 16 * 2^15 */
#define MAX_ADJ_INT	256
#define ADJ_INTERVAL	16
#define PPS_GOOD_LIM	16
#define PPS_BAD_LIM	16
#define PPS_TOUT_LIM	4	/* max # of timeouts */
#define PPS_DROP_OUT	16	/* PPS timeout in sec */
//#define MAX_GOOD_OFFS	1075269 /*  1ms  ~ 1075269 * 2^-30 */
// #define MAX_GOOD_OFFS	10735 /* 10 us ~ 10735 * 2^-30 */
// #define MAX_GOOD_OFFS	2147 /* 2 us ~ 2147 * 2^-30 */
#define MAX_OFFS	1075269 /*  1ms  ~ 1075269 * 2^-30 */
#define MAX_DIFF	107374 /*  100us  ~ 107374 * 2^-30 */
#define GOOD_OFFS	107374 /*  100us  ~ 107374 * 2^-30 */
#define FADING_RATE	4

/* note: real coefs are 19 and 9 as there is difference
 * in offset and drift unit
 */
// #define AI_L_COEF	15
// #define AP_L_COEF	5
#define AI_L_COEF	14
#define AP_L_COEF	4
#define AP_L_CORR	3

#define IIR_FIL_STIFF 5

#define PPS_OK   0
#define PPS_REL  1
#define PPS_BAD  2
#define PPS_LOST 3

#define SET_CL_DELAY	5200

int arg_debug = 0, arg_drift = 0, arg_sys = 0, arg_ntp = 0, arg_pps_source = -1;

int tsu_gen = 0; //boolean variable 0 - tsu_cv2 | 1 - tsu_gen component

char *arg_drift_file = TSU_DRIFT_FILE;
char *arg_ntp_name;
//char *def_name = CS_PATH_DEV(0);
char file[80];
char *design_xml_file = NULL;
int arg_device_file_path = 0; //boolean variable - set to 1 if '-d' argument is entered

uint32_t	data, offs;
int32_t sys_offs;
uint32_t tsu_cv2_frequency = ((uint32_t)-1);

int c, i;


int time_state = TIME_OK;	/* clock state */
/* bit STA_UNSYNC prevents a periodic update of the CMOS clock */
int time_status;		/* clock status bits */
int32_t	time_constant;		/* poll interval (shift) (s) */
int32_t time_tai;			/* TAI offset (s) */
int32_t time_monitor;		/* last time offset scaled (ns) */
int32_t time_precision = 1;	/* clock precision (ns) */
int32_t time_maxerror;		/* maximum error (ns) -- [extension by UW] */
int32_t time_esterror;		/* estimated error (ns) -- [extension by UW] */
int32_t time_reftime;		/* time at last adjustment (s) */
int32_t time_offset;		/* time offset (ns) */
int32_t time_freq;			/* frequency offset (ns/s) [scale 16] */
int32_t time_stabil;		/* frequency stability (ns/s) [scale 16] */
int32_t time_jitter;		/* offset jitter */

int32_t time_allan;		/* sum of diffrence squares */

char str[1024];
uint32_t  log_cnt;


double sys_drift = 0.;
int sys_xdrift;

uint32_t last_pps_sec;
int32_t last_pps_xsec;
int tmp_allan_x1, tmp_allan_x2, tmp_allan_y1;


/*
 * TS Card registers
 */

struct t_inc_reg
{
    uint32_t f;
    uint32_t f_low;
} inc_reg;

struct t_rt_reg
{
    uint32_t i;
    uint32_t f;
    uint32_t f_low;
} rt_reg;

struct t_pps_reg
{
    uint32_t i;
    uint32_t f;
} pps_reg;


struct tfil
{
    int stiff;
    int x[FILT_BUF_SZ];
    int y[FILT_BUF_SZ];
};

struct tclock
{
    int drift;	/* freq difference in 2^-34 */
    int drift_low;	/* fraction of drift value */
    int offset;	/* phase offset in 2^-30s */
    int adj;	/* freq adjustment in 2^-34 */
    struct t_inc_reg ref_inc_reg;
    int last_offset;
    int pps_good_cnt, pps_bad_cnt;
    int fix_drift_flag; /* do not change drift by PI reg  */
    int ini_drift_flag; /* drift init phase */
    int phase_adj_flag; /* phase set phase, keep drift */
    int no_clock_step;  /* do not step clock, smooth phase set */
};

struct treg
{
    int ap_l;	/* prop. koef - binary log */
    int ai_l	/* integ. koef - binary log */;
};


struct tclock ptm_clock;
struct tfil iir_filter;
struct treg  pi_regul;

/*****************************************************************************/

extern char *__progname;
// function return program name ()
const char *
getprogname(void)
{
    return __progname;
}

int32_t xstrtol(const char * str, int base);
uint32_t xstrtoul(const char * str, int base);
double xstrtod(const char * str);
struct hwio_version strtover(const char *str);

struct hwio_comp_spec ctl_compat[] =
{
    {"debug", "AXI_TSU", HWIO_VER2(1, 0)},
    {"ug4-150", "axi_tsu", HWIO_VER3(1, 0, HWIO_VERSION_NA)},
    HWIO_COMPAT_END
};

void hwio_tool_help(const char *toolname,
                    const struct hwio_tool_params *params)
{
    printf("Usage: %s [-D] [-f drift] [-F file]" \
           "[-s<offset>] [-t host]\n", getprogname());
    printf(" -D           Debug mode (run in foreground)\n");
    printf(" -f drift     Set initial <drift> [ppm] (frequency correction)\n");
    printf(" -F file      Read and write drift into <file>, instead of default " TSU_DRIFT_FILE "\n");
    printf(" -h           Show this text\n");
    printf(" -p source    Select PPS source (For uSondu: 0: GPS (RS232), 1: Internal)\n");
    printf(" -s offset    Use system time with optional <offset> [hours]\n");
    printf(" -t host      Use NTP server <host>\n");
}

static int tsu_custom_parse(int opt, const char *optarg,
                            struct hwio_tool_params *params)
{
    char *e;
    int32_t tmp;

    switch (opt)
    {

    case 'D':
        arg_debug = 1;
        return HWIO_TOOL_CUSTOM_OK;

    case 'f':
        arg_drift = 1;
        sys_drift = xstrtod(optarg);
        return HWIO_TOOL_CUSTOM_OK;

    case 'F':
        arg_drift_file = malloc((strlen(optarg)+1)*sizeof(char));
        memcpy(arg_drift_file, optarg, strlen(optarg)+1);
        return HWIO_TOOL_CUSTOM_OK;

    case 'p':
        tmp=strtoul(optarg, &e, 0);
        if((*e) != '\0' || tmp > 1 || tmp < 0)
            return hwio_tool_error(HWIO_TOOL_EARGS,
                                   "Invalid PPS source. Please specify number 0 or 1.");
        arg_pps_source = tmp;
        return HWIO_TOOL_CUSTOM_OK;

    case 's':
        arg_sys = 1;
        if (optarg != 0)
            sys_offs = xstrtol(optarg,10);
        return HWIO_TOOL_CUSTOM_OK;

    case 't':
        arg_ntp = 1;
        arg_ntp_name = malloc((strlen(optarg)+1)*sizeof(char));
        memcpy(arg_ntp_name, optarg, strlen(optarg)+1);
        return HWIO_TOOL_CUSTOM_OK;

    default:
        return HWIO_TOOL_CUSTOM_UNKNOWN;
    }
}


/*****************************************************************************/

int
set_RT_REG(struct hwio_comp *c, hwio_tool_comp_err_t err,
           const struct hwio_tool_params *params, struct t_rt_reg *reg)
{
    /* set MI data */
    if(hwio_comp_write(c, TSU_CV2_MI_DATA_LOW, 0 /*reg->f_low*/))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_MI_DATA_LOW");
    if(hwio_comp_write(c, TSU_CV2_MI_DATA_MIDDLE, reg->f))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_MI_DATA_MIDDLE");
    if(hwio_comp_write(c, TSU_CV2_MI_DATA_HIGH, reg->i))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_MI_DATA_HIGH");

    /* set CNTRL -> write to RT */
    if(hwio_comp_write(c, TSU_CV2_CNTRL, TSU_CV2_CNTRL_WRITE_RT))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_CNTRL");

    hwio_tool_verbose(params, 2, "set_RT: 0x%08x 0x%08x 0x%08x", reg->f_low, reg->f, reg->i);

    return EXIT_SUCCESS;
}

int
set_INC_REG(struct hwio_comp *c, hwio_tool_comp_err_t err,
            const struct hwio_tool_params *params, struct t_inc_reg *reg)
{
    /* set MI data */
    if(hwio_comp_write(c, TSU_CV2_MI_DATA_LOW, reg->f_low))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_MI_DATA_LOW");
    if(hwio_comp_write(c, TSU_CV2_MI_DATA_MIDDLE, reg->f))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_MI_DATA_MIDDLE");
    if(hwio_comp_write(c, TSU_CV2_MI_DATA_HIGH, 0)) /*reg->i*/
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_MI_DATA_HIGH");

    /* set CNTRL -> write to INC */
    if(hwio_comp_write(c, TSU_CV2_CNTRL, TSU_CV2_CNTRL_WRITE_INC))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_CNTRL");

    /* set CNTRL -> read from INC - workaround - probably should be removed after fix in hw */
    /* TODO: next line is obsolete for NIC designs with GICS (05_0a and above) */
    if(hwio_comp_write(c, TSU_CV2_CNTRL, TSU_CV2_CNTRL_READ_INC))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_CNTRL");

    hwio_tool_verbose(params, 2, "set_INC: 0x%08x 0x%08x", reg->f_low, reg->f);

    return EXIT_SUCCESS;
}

int
get_RT_REG(struct hwio_comp *c, hwio_tool_comp_err_t err,
           const struct hwio_tool_params *params, struct t_rt_reg *reg)
{
    /* set CNTRL -> read from RT */
    if(hwio_comp_write(c, TSU_CV2_CNTRL, TSU_CV2_CNTRL_READ_RT))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_CNTRL");

    /* read MI data */
    if(hwio_comp_read(c, TSU_CV2_MI_DATA_LOW, &reg->f_low))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_CV2_MI_DATA_LOW");
    if(hwio_comp_read(c, TSU_CV2_MI_DATA_MIDDLE, &reg->f))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_CV2_MI_DATA_MIDDLE");
    if(hwio_comp_read(c, TSU_CV2_MI_DATA_HIGH, &reg->i))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_CV2_MI_DATA_HIGH");

    hwio_tool_verbose(params, 2, "get_RTR: 0x%08x 0x%08x 0x%08x", reg->f_low, reg->f, reg->i);

    return EXIT_SUCCESS;
}

int
get_INC_REG(struct hwio_comp *c, hwio_tool_comp_err_t err,
            const struct hwio_tool_params *params, struct t_inc_reg *reg)
{
    /* set CNTRL -> read from INC */
    if(hwio_comp_write(c, TSU_CV2_CNTRL, TSU_CV2_CNTRL_READ_INC))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_CNTRL");

    /* read MI data */
    if(hwio_comp_read(c, TSU_CV2_MI_DATA_LOW, &reg->f_low))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_CV2_MI_DATA_LOW");
    if(hwio_comp_read(c, TSU_CV2_MI_DATA_MIDDLE, &reg->f))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_CV2_MI_DATA_MIDDLE");
    //hwio_comp_read(c, TSU_CV2_MI_DATA_HIGH, &reg->i);

    hwio_tool_verbose(params, 2, "get_INC: 0x%08x 0x%08x", reg->f_low, reg->f);

    return EXIT_SUCCESS;
}

int
get_PPS_REG(struct hwio_comp *c, hwio_tool_comp_err_t err,
            const struct hwio_tool_params *params, struct t_pps_reg *reg)
{
    /* set CNTRL -> read from PPS */
    if(hwio_comp_write(c, TSU_CV2_CNTRL, TSU_CV2_CNTRL_READ_PPS))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_CNTRL");

    /* read MI data */
    //hwio_comp_read(c, TSU_CV2_MI_DATA_LOW, &reg->f_low);
    if(hwio_comp_read(c, TSU_CV2_MI_DATA_MIDDLE, &reg->f))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_CV2_MI_DATA_MIDDLE");
    if(hwio_comp_read(c, TSU_CV2_MI_DATA_HIGH, &reg->i))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_CV2_MI_DATA_HIGH");

    hwio_tool_verbose(params, 2, "get_PPS: 0x%08x 0x%08x 0x%08x", 0, reg->f, reg->i);

    return EXIT_SUCCESS;
}

int
get_time(struct hwio_comp *c, hwio_tool_comp_err_t err,
         const struct hwio_tool_params *params, l_fp *time)
{
    if(get_RT_REG (c, err, params, &rt_reg))
        return EXIT_FAILURE;

    time->l_ui = rt_reg.i;
    time->l_uf = rt_reg.f;

    return EXIT_SUCCESS;
}

int
set_time(struct hwio_comp *c, hwio_tool_comp_err_t err,
         const struct hwio_tool_params *params, l_fp *time)
{
    rt_reg.i = time->l_ui;
    rt_reg.f = time->l_uf;
    rt_reg.f_low = 0;
    if(set_RT_REG (c, err, params, &rt_reg))
        return EXIT_FAILURE;

    return EXIT_SUCCESS;
}

int
get_pps(struct hwio_comp *c, hwio_tool_comp_err_t err,
        const struct hwio_tool_params *params, l_fp *time)
{
    if(get_PPS_REG (c, err, params, &pps_reg))
        return EXIT_FAILURE;
    time->l_ui = pps_reg.i;
    time->l_uf = pps_reg.f;

    return EXIT_SUCCESS;
}


/* Get the time with theoretical nanosecond resolution */
int
do_clock_gettime(struct hwio_comp *c, hwio_tool_comp_err_t err,
                 const struct hwio_tool_params *params, struct timexspec *tx)
{
    l_fp t;

    if(get_time(c, err, params, &t))
        return EXIT_FAILURE;
    FP2TX(t, *tx);

    return EXIT_SUCCESS;
}

/* Set the time with theoretical nanosecond resolution */
int
do_clock_settime(struct hwio_comp *c, hwio_tool_comp_err_t err,
                 const struct hwio_tool_params *params, const struct timexspec *tx)
{
    l_fp t;

    TX2FP(*tx,t);
    if(set_time(c, err, params, &t))
        return EXIT_FAILURE;

    time_status |= STA_UNSYNC;

    return EXIT_SUCCESS;
}

/* Get clock resolution - by subsequent calling 'do_clock_gettime' */
int
do_clock_getres(struct hwio_comp *c, hwio_tool_comp_err_t err,
                const struct hwio_tool_params *params, struct timexspec *txp)
{
    struct timexspec ts, ts1, ts2;
    int retry;

    /* load code into CPU cache */
    if(do_clock_gettime(c, err, params, &ts1))
        return EXIT_FAILURE;

    retry = 1000;
    do
    {
        if(do_clock_gettime(c, err, params, &ts1))
            return EXIT_FAILURE;
        if(do_clock_gettime(c, err, params, &ts2))
            return EXIT_FAILURE;
        ts.tv_xsec = ts2.tv_xsec - ts1.tv_xsec;
        ts.tv_sec = ts2.tv_sec - ts1.tv_sec;
    }
    while ( --retry > 0 && (ts.tv_sec != 0 || ts.tv_xsec == 0));
    *txp = ts;

    return EXIT_SUCCESS;
}

/*
 * timevar_init() - initialize variables and structures
 */

int
timevar_init(struct hwio_comp *c, hwio_tool_comp_err_t err,
             const struct hwio_tool_params *params)
{
    struct timexspec txs;

    time_status = STA_UNSYNC | STA_NANO;
    time_maxerror = time_esterror = NTP_PHASE_LIMIT;
    time_offset = 0;
    time_freq = 0;
    time_stabil = 0;
    time_jitter = 0;
    time_allan = 0;
    if(do_clock_getres(c, err, params, &txs))
        return EXIT_FAILURE;
    time_precision = txs.tv_xsec;
    sys_xdrift = sys_drift * XANOSEC / 1e6 * 16;
    time_freq = sys_xdrift;
    /*
    pps.shift = PPS_FAVG;
    pps.shiftmax = PPS_FAVGDEF;
    pps.tf[0].tv_sec = pps.tf[0].tv_xsec = 0;
    pps.tf[1].tv_sec = pps.tf[1].tv_xsec = 0;
    pps.tf[2].tv_sec = pps.tf[2].tv_xsec = 0;
    pps.fcount = 0;
    L_CLR(pps.freq);
    L_LINT (pps.freq, xdrift);
    pps.fmin = MAXFREQ;
    pps.fmax = -MAXFREQ;
    pps.frange = MAXFREQ;

    inc_max = (1. + PPM_MAX/1e6) / NOM_HZ;
    inc_min = (1. - PPM_MAX/1e6) / NOM_HZ;
     */

    return EXIT_SUCCESS;
}

int
init_inc_reg(struct hwio_comp *c, hwio_tool_comp_err_t err,
             const struct hwio_tool_params *params)
{
    double tmp;

    //	tmp = (double) (1. / NOM_HZ) * (1 + sys_drift / 1e6);
    //	tmp = (double) (1. / NOM_HZ);
    tmp = (double) (1. / tsu_cv2_frequency) * (1 + sys_drift / 1e6);
    inc_reg.f = tmp * FRAC;

    tmp *= FRAC;
    tmp -= (double) (inc_reg.f);
    inc_reg.f_low = tmp * FRAC;
    if(set_INC_REG(c, err, params, &inc_reg))
        return EXIT_FAILURE;

    return EXIT_SUCCESS;
}

// xanosec -> microosec conversion
double
xs2us(int32_t x)
{
    return (double) x * 1.0e6 / XANOSEC;
}

// xanosec -> nanosec conversion
int
xs2ns(int x)
{

    double d;
    int ns;

    d = (double) x * 1.0e9 / XANOSEC;
    ns = d;
    return ns;
}

// nanosec -> xanosec conversion
int
ns2xs(int n)
{
    double d;
    int xs;

    d = (double) n / 1.0e9 * XANOSEC ;
    xs = d;
    return xs;
}

void
ireg2a(uint32_t ireg, char *a)
{
    struct  tm *t;
    time_t x;

    x = (time_t) ireg;
    memset (&t,0,sizeof(t));
    t = gmtime(&x);
    ++t->tm_mon;
    t->tm_year += 1900;
    sprintf(a, "%d.%d.%d %2d:%02d:%02d", t->tm_mday, t->tm_mon,
            t->tm_year, t->tm_hour,t->tm_min,t->tm_sec);
}

void
freg2us(uint32_t freg, char *a)
{
    double d_tmp;

    M_LFPTOD(0, freg, d_tmp);
    d_tmp *= 1e6;
    sprintf(a, "%010.3f", d_tmp);
}

void
ffreg2ns(uint32_t freg, uint32_t lowfreg, char *a)
{
    double d_tmp, d1_tmp, d2_tmp;

    M_LFPTOD(0,freg,d1_tmp);
    M_LFPTOD(0,lowfreg,d2_tmp);

    d_tmp = (d1_tmp +  d2_tmp / FRAC) * 1e9;
    sprintf(a, "%.9f", d_tmp);
}

/*
 * converts l_fp to ascii
 */
void
fptoa(l_fp fp, char *a)
{
    int32_t us;
    double tmp;
    char *p;

    ireg2a(fp.l_ui, a);
    p = a + strlen(a);

    tmp = fp.l_uf / EXP2_32;
    us = tmp * 1000000;
    sprintf(p, ".%06d", us);
}

void
update_stat(int n)
{
    if (n < 100)
    {
        time_status &= ~STA_UNSYNC;
        time_status |= STA_PPSFREQ;
    }
    else if (n < 200)
    {
        time_status &= ~STA_PPSFREQ;
        time_status |= STA_FREQHOLD;
        time_status |= STA_PPSTIME;
    }
    else
    {
        //should these two statements be in opposite order? My comment....
        time_status &= ~STA_FREQHOLD;
        time_status |= STA_PPSFREQ;
    }
}

/* Get temperature from tempctl tool */
// retuns 0 on success else returns -1
int
get_temperature(struct hwio_comp *c, hwio_tool_comp_err_t err,
                const struct hwio_tool_params *params, char *temp_data)
{
    char line[30];
    FILE *fp;

    //Init chars variables as empty strings everytime this procedure is called
    line[0] = '\0';
    temp_data[0] = '\0';

    //Call tempctl with -A argument
    fp = popen("tempctl -A", "r");

    //Get first part of command output
    if (!fgets(line, sizeof line, fp) && ferror(fp))
    {
        hwio_tool_verbose(params, 0, "Some error occured when getting information about temperature from tempctl program.");
        return -1;
    }

    //Check if tempctl tool is available for use
    if (!strcmp(line,""))
    {
        //tempctl is not available on host machine
        hwio_tool_verbose(params, 0, "Temperature can't be measured because of lack of tempctl program. ");
        //   "To solve this problem please ensure that tempctl program is installed properly.");
        return -1;
    }

    //If entire line wasn't read before, then get rest
    //of first line of tempctl -A output
    int eol, i;
    eol = 0;
    for (i = 0; i < 30; i++)
    {
        if (line[i] == '\0')
        {
            break;
        }
        if (line[i] == '\n')
        {
            eol = 1;
            break;
        }
    }
    if (!eol) while (fgets(line, sizeof line, fp));
    if (ferror(fp))
    {
        hwio_tool_verbose(params, 0, "Some error occured when getting information about temperature from tempctl program.");
        return -1;
    }

    //Get temperature data on second line of tempctl -A output
    if (!fgets(line, sizeof line, fp) && ferror(fp))
    {
        hwio_tool_verbose(params, 0, "Some error occured when getting information about temperature from tempctl program.");
        return -1;
    }

    pclose(fp);

    //Copy temperature data into output char array
    memcpy(temp_data,line+13,7);
    temp_data[7] = '\0';

    return 0;
}

int
do_log(struct hwio_comp *com, hwio_tool_comp_err_t err,
       const struct hwio_tool_params *params)
{
    char a[80], b[80], c[80];
    struct t_inc_reg ir;
    struct t_rt_reg rr;

    if (arg_debug)
    {
        if(get_INC_REG(com, err, params, &ir))
            return EXIT_FAILURE;
        if(get_RT_REG(com, err, params, &rr))
            return EXIT_FAILURE;

        ireg2a(rr.i, a);
        freg2us(rr.f, b);
        ffreg2ns(ir.f, ir.f_low, c);
        printf("%4i %s %s  INC_REG: %s\n",log_cnt,a,b,c);
    }

    if (arg_debug)
    {
        printf("offs=%d, freq=%d (%.3f), stab=%d (%.3f)\n",
               time_offset, time_freq / 16,
               (double)time_freq / 16 / PPM_COEF, time_stabil / 16,
               (double) time_stabil / 16 / PPM_COEF);
        printf("\n");
    }


    if (log_cnt++ > 1000000) log_cnt = 0;

    if (600 * (log_cnt / 600) == log_cnt)
    {
        write_drift(1.0e6 * time_freq / 16 / XANOSEC, arg_drift_file);
    }

    return EXIT_SUCCESS;
}

/*
 * Detect and select appropriate CLK source for
 * core part of GENERIC TSU.
 */
int
gen_select_clk_source(struct hwio_comp *c, hwio_tool_comp_err_t err,
                      const struct hwio_tool_params *params)
{
    uint32_t clk_source;
    uint32_t sup_clk_sources;
    int i;

    /* get number of supported clk sources */
    if(hwio_comp_read(c, TSU_GEN_SUP_CLK_PPS_REG, &sup_clk_sources))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_GEN_SUP_CLK_PPS_REG");
    /* CLK is in higer half */
    sup_clk_sources = sup_clk_sources >>16;


    for(i=sup_clk_sources-1; i>=0; i--)
    {
        if(hwio_comp_write(c, TSU_GEN_CLK_SELECT, i))
            return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_GEN_CLK_SELECT");
        sleep(1);
        if(hwio_comp_read(c, TSU_GEN_STATE_REG, &clk_source))
            return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_GEN_STATE_REG");
        clk_source = clk_source & 0x02; /*higer bit is for clk source*/
        if(clk_source)
        {
            hwio_tool_verbose(params, 1, "Selected %i clk source",i);
            return EXIT_SUCCESS;
        }
    }
    hwio_tool_verbose(params, 0, "There is no active clk source available");
    hwio_tool_verbose(params, 1, "Terminating.");
    exit(1);

    return EXIT_SUCCESS;
}

/*
 * Detect and select appropriate CLK source for
 * core part of TSU_CV2.
 */
int
select_clk_source(struct hwio_comp *c, hwio_tool_comp_err_t err,
                  const struct hwio_tool_params *params)
{
    uint32_t clk_source;
    if(tsu_gen)
    {
        if(gen_select_clk_source(c, err, params))
            return EXIT_FAILURE;
    }
    else
    {
        /* detect CLK sources */
        if(hwio_comp_read(c, TSU_CV2_STATE_REG, &clk_source))
            return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_CV2_STATE_REG");

        /* PPSX_N pulse change */
        hwio_tool_verbose(params, 1,  "Detected internal COMBO CLK as %s and external CLK as %s",
               clk_source & 0x04 ? "ACTIVE" : "INACTIVE",
               clk_source & 0x08 ? "ACTIVE" : "INACTIVE");


        /* select CLK source automatically */
        if (clk_source & 0x08)   /* External precise source clk has higher priority than internal one */
        {
            hwio_tool_verbose(params, 1, "Selected external clk source");
            if(hwio_comp_write(c, TSU_CV2_CLK_SELECT, 1))
                return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_CV2_CLK_SELECT");
        }
        else if (clk_source & 0x04)     /* Internal clk source select */
        {
            hwio_tool_verbose(params, 1, "Selected internal clk source");
            if(hwio_comp_write(c, TSU_CV2_CLK_SELECT, 0))
                return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_CV2_CLK_SELECT");
        }
        else
        {
            hwio_tool_verbose(params, 0, "Neither internal, nor external clk sources are connected to TSU");
            hwio_tool_verbose(params, 1, "Terminating.");
            exit(1);
        }

    }

    return EXIT_SUCCESS;
}

/*
 * Detect and select appropriate PPS source for
 * GENERIC TSU.
 */
int
gen_select_pps_pulse(struct hwio_comp *c, hwio_tool_comp_err_t err,
                     const struct hwio_tool_params *params)
{
    uint32_t pps_source;
    uint32_t sup_pps_sources;
    int i;

    /* get number of supported PPS sources */
    if(hwio_comp_read(c, TSU_GEN_SUP_CLK_PPS_REG, &sup_pps_sources))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_GEN_SUP_CLK_PPS_REG");
    /* PPS is in lower half */
    sup_pps_sources = sup_pps_sources & 0xffff;

    if (arg_pps_source < 0)
    {
        for(i=sup_pps_sources-1; i>=0; i--)
        {
            if(hwio_comp_write(c, TSU_GEN_PPS_SELECT, i))
                return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_GEN_PPS_SELECT");
            sleep(1);
            if(hwio_comp_read(c, TSU_GEN_STATE_REG, &pps_source))
                return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_GEN_STATE_REG");
            pps_source = pps_source & 0x01; /*lower bit is for clk source*/
            if(pps_source)
            {
                hwio_tool_verbose(params, 1, "Selected %i pps source",i);
                return EXIT_SUCCESS;
            }
        }
        hwio_tool_verbose(params, 0, "There is no active pps source available");
        hwio_tool_verbose(params, 1, "Terminating.");
        exit(1);
    }
    else
    {
        if(arg_pps_source > (sup_pps_sources-1))
        {
            hwio_tool_verbose(params, 0, "Unsupported clk source %i (highest supported clk source is %i)",
                   arg_pps_source,(sup_pps_sources-1));
            hwio_tool_verbose(params, 1, "Terminating.");
            exit(1);
        }
        else
        {
            if(hwio_comp_write(c, TSU_GEN_PPS_SELECT, arg_pps_source))
                return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_GEN_PPS_SELECT");
            hwio_tool_verbose(params, 1, "Selected %i pps source", arg_pps_source);
        }
    }

    return EXIT_SUCCESS;
}

/*
 * Detect and select appropriate PPSX_N pulse
 * to get regular PPS signal from.
 *
 * Useful not only for pulse initialization, but should
 * be run every few seconds because of possibility
 * of cable (pulse) plug-in/plug-off.
 */
int
select_pps_pulse(struct hwio_comp *c, hwio_tool_comp_err_t err,
                 const struct hwio_tool_params *params)
{
    static uint32_t ppsx_pulse_last = -1;
    uint32_t ppsx_pulse;

    if(tsu_gen)
    {
        if(gen_select_pps_pulse(c, err, params))
            return EXIT_FAILURE;
    }
    else
    {
        /* detect PPSX_N pulses */
        if(hwio_comp_read(c, TSU_CV2_STATE_REG, &ppsx_pulse))
            return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_CV2_STATE_REG");

        if (ppsx_pulse_last != ppsx_pulse)
        {
            /* PPSX_N pulse change */

            if(arg_pps_source < 0)
            {
                /* select PPSX_N pulse automatically */
                if (ppsx_pulse & 0x01)   /* GPS_PPS_N - higher priority than internal PPS_N pulse aproximation */
                {
                    hwio_tool_verbose(params, 1, "Selected GPS_PPS_N pulse to get regular PPS signal from");
                    if(hwio_comp_write(c, TSU_CV2_PPSN_SELECT, 0))
                        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_PPSN_SELECT");
                }
                else if (ppsx_pulse & 0x02)     /* PPS1_N */
                {
                    hwio_tool_verbose(params, 1, "Selected internal PPS_N pulse to get regular PPS signal from");
                    if(hwio_comp_write(c, TSU_CV2_PPSN_SELECT, 1))
                        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_PPSN_SELECT");
                }
                else
                {
                    hwio_tool_verbose(params, 0, "Neither GPS_PPS_N, nor internal PPS_N are connected to TSU_CV2");
                    hwio_tool_verbose(params, 1, "Terminating.");
                    exit(1);
                }
            }
            else
            {
                /* select PPSX_N pulse manually*/
                if (arg_pps_source == 0)
                {
                    hwio_tool_verbose(params, 1, "Selected GPS_PPS_N pulse to get regular PPS signal from");
                    if(hwio_comp_write(c, TSU_CV2_PPSN_SELECT, 0))
                        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_PPSN_SELECT");
                }
                else
                {
                    hwio_tool_verbose(params, 1, "Selected internal PPS_N pulse to get regular PPS signal from");
                    if(hwio_comp_write(c, TSU_CV2_PPSN_SELECT, 1))
                        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_PPSN_SELECT");
                }
            }

            /* store selected PPSX_N pulse */
            ppsx_pulse_last = ppsx_pulse;
        }
    }

    return EXIT_SUCCESS;
}

int
tsu_init(struct hwio_comp *c, hwio_tool_comp_err_t err,
         const struct hwio_tool_params *params)
{
    l_fp fp;

    /* log init */
    log_cnt = 0;

    /* evaluate drift */
    if (! arg_drift)
        read_drift(&sys_drift, arg_drift_file);
    if (sys_drift > PPM_MAX || sys_drift < -PPM_MAX)
        sys_drift = 0;
    hwio_tool_verbose(params, 1, "Initial drift %.3f", sys_drift);

    if(select_clk_source(c, err, params))
        return EXIT_FAILURE;

    if(select_pps_pulse(c, err, params))
        return EXIT_FAILURE;

    /* get component frequency */
    if(hwio_comp_read(c, TSU_CV2_FREQ, &tsu_cv2_frequency))
        return hwio_tool_error(EXIT_FAILURE, "Unable to read register TSU_CV2_FREQ");
    if(tsu_gen)
    {
        tsu_cv2_frequency++;
    }

    hwio_tool_verbose(params, 1, "Component core frequency: %d Hz", (int32_t) tsu_cv2_frequency);

    if (tsu_cv2_frequency <= 0)
    {
        hwio_tool_verbose(params, 0, "Component frequency is not valid.");
        hwio_tool_verbose(params, 1, "Terminating.");
        exit(1);
    }

    /* set INC_reg */
    if(init_inc_reg(c, err, params))
        return EXIT_FAILURE;

    /* set RT_reg */
    get_systime(&fp);
    if (!arg_sys)
        sys_offs = 0;
    if (sys_offs > 12 || sys_offs < -12)
        sys_offs = 0;
    fp.l_ui += 3600 * sys_offs;
    if(set_time(c, err, params, &fp))
        return EXIT_FAILURE;
    fptoa(fp, str);
    if (!sys_offs)
    {
        hwio_tool_verbose(params, 1, "System time: %s (UTC)", str);
    }
    else
    {
        hwio_tool_verbose(params, 1, "System time: %s", str);
    }

    /* set INTA_reg */
    if(hwio_comp_write(c, TSU_CV2_INTA, 1))
        return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_INTA");
    hwio_tool_verbose(params, 1, "INTA register set - start generating valid timestamps");

    /* set NTP time */
    if (arg_ntp)
    {
        //if (init_ntp(c, err, params, arg_ntp_name) < 0)
        //    errx(1, "Can't init NTP (server \"%s\")", arg_ntp_name);
        if (get_ntptime(c, err, params, arg_ntp_name, &fp) < 0)
            errx(1, "Can't get time from NTP server \"%s\"", arg_ntp_name);
        set_time(c, err, params, &fp);
        fptoa(fp, str);
        hwio_tool_verbose(params, 1, "NTP server \"%s\" time: %s (UTC)", arg_ntp_name, str);
    }

    /* timevar init */
    if(timevar_init(c, err, params))
        return EXIT_FAILURE;

    /* run in background */
    if (!arg_debug)
    {
        printf("Moving to background\n\n");
        /* no chdir, std redirection to /dev/null */
        if (daemon(1, 0) < 0)
        {
            perror("daemon() failed");
            exit(1);
        }
    }

    return EXIT_SUCCESS;
}

void
calc_allan(int offs, int drift)
{
    int x,y;

    x = (tmp_allan_x2 - 2*tmp_allan_x1 + xs2ns(offs)) / ADJ_INTERVAL;
    if (x <= (1<<15) && x >= -(1<<15))
        time_allan += (x*x - time_allan) >> FADING_RATE;

    y = drift - tmp_allan_y1;
    time_stabil += (xs2ns(abs(y)) - time_stabil) >> FADING_RATE;

    tmp_allan_x2 = tmp_allan_x1;
    tmp_allan_x1 = xs2ns(offs);
    tmp_allan_y1 = xs2ns(drift);
}

void
init_allan()
{
    tmp_allan_x2 = 0;
    tmp_allan_x1 = 0;
    tmp_allan_y1 = 0;
}

void
l_div(l_fp *a, uint32_t d)
{
    double x;

    //printf("l_dif: ui %d, uf %d\n", a->l_ui, a->l_uf);
    LFPTOD(*a, x);
    //printf("l_dif: x %f\n", x);
    x /= d;
    //printf("l_dif: x %f\n", x);
    DTOLFP(x, *a);
    //printf("l_dif: ui %d, uf %d\n", a->l_ui, a->l_uf);
}

/* PI regulator
 * input: clock, pipar
 * output: chages clock.adj, clock.drift
 */
void
pi_reg(struct tclock *cl, struct treg *pi)
{
    l_fp pdr, edr;
    struct timexspec temp;
    uint32_t aoffs;
    int neg;

    /* folowing code does implements statemet
     *		 cl->drift += cl->offset / pi->ai;
     * in extended precision in order to add very small
     * values to drift accumulator
     */

    if (cl->offset > MAX_OFFS)
    {
        cl->adj = MAX_ADJ;
    }
    else if (cl->offset < -MAX_OFFS)
    {
        cl->adj = -MAX_ADJ;
    }
    else
    {
        if (cl->fix_drift_flag || cl->ini_drift_flag)
        {
            /* fixed drift_flag drift, no I-reg term  -> we use higher P-reg loop */
            cl->adj = - cl->drift - cl->offset / (1 << (pi->ap_l - AP_L_CORR));
        }
        else
        {
            /* PI regulator, adjust drift  */
            aoffs = abs(cl->offset);
            neg = (cl->offset < 0);
            pdr.l_i = aoffs;
            pdr.l_uf = 0;
            L_RSHIFT(pdr, pi->ai_l);
            temp.tv_sec = cl->drift;
            temp.tv_xsec = cl->drift_low;
            TX2FP(temp, edr);
            if (neg)
                L_SUB(edr,pdr);
            else
                L_ADD(edr, pdr);
            FP2TX(edr,temp);
            cl->drift = temp.tv_sec;
            cl->drift_low = temp.tv_xsec;

            if (cl->drift >  MAX_DRIFT)
                cl->drift =  MAX_DRIFT;
            if (cl->drift < -MAX_DRIFT)
                cl->drift = -MAX_DRIFT;
            cl->adj = - cl->drift - cl->offset / (1 << pi->ap_l);
        }

        //cl->adj = - cl->drift - cl->offset / (1 << pi->ap_l);
        if (cl->adj >  MAX_ADJ) cl->adj =  MAX_ADJ;
        if (cl->adj < -MAX_ADJ) cl->adj = -MAX_ADJ;
    }
}

/* offset filter
 * IIR filter:   s*y[n] - (1-s)*y[n-1] = x[n]/2 + x[n-1]/2
 *
 * can be replaced by another algorithm, the fil data structure
 * remembers last FILT_BUF_SZ input and output values
 */
int
filter(int offs, struct tfil *fil)
{

    int x,y;
    int i;


    for (i = FILT_BUF_SZ - 1; i > 0; i--)
    {
        fil->x[i] = fil->x[i-1];
        fil->y[i] = fil->y[i-1];
    }

    x = offs/2  + fil->x[1]/2;
    y = fil->y[1];
    y += (x - y) / fil->stiff;

    fil->x[0] = offs;
    fil->y[0] = y;
    return y;

}

void
init_filter(struct tfil *fil)
{
    int i;

    for (i = 0; i < FILT_BUF_SZ; i++)
    {
        fil->x[i] = 0;
        fil->y[i] = 0;
    }
    fil->stiff = IIR_FIL_STIFF;
}

/*
 * waits for new PPS up to PPS_DROP_OUT;
 * return the value of last PPS and diffenece from last PPS
 */
int
new_pps(struct hwio_comp *c, hwio_tool_comp_err_t err,
        const struct hwio_tool_params *params, struct timexspec *tx,
        struct timexspec *diff)
{
    l_fp t0, now, t;

    if(get_time(c, err, params, &t0))
        return EXIT_FAILURE;
    for (;;)
    {
        if(get_pps(c, err, params, &t))
            return EXIT_FAILURE;
        FP2TX(t,*tx);
        if (tx->tv_sec != last_pps_sec || tx->tv_xsec != last_pps_xsec)
        {
            printf("seconds: %d\n", tx->tv_sec);
            printf("xanosec: %d\n", tx->tv_xsec);
            // new PPS occurred
            diff->tv_sec = tx->tv_sec - last_pps_sec;
            diff->tv_xsec = tx->tv_xsec - last_pps_xsec;
            if (diff->tv_xsec < - (XANOSEC >> 1))
            {
                diff->tv_xsec += XANOSEC;
                diff->tv_sec--;
            }
            else if (diff->tv_xsec > (XANOSEC >> 1))
            {
                diff->tv_xsec -= XANOSEC;
                diff->tv_sec++;
            }
            last_pps_sec = tx->tv_sec;
            last_pps_xsec = tx->tv_xsec;
            time_jitter += (xs2ns(abs(diff->tv_xsec)) - time_jitter) >> FADING_RATE;
            return 2;
        }

        if(get_time(c, err, params, &now))
            return EXIT_FAILURE;
        if (now.l_ui > t0.l_ui + PPS_DROP_OUT)
        {
            // no new PPS and time limit was reached
            diff->tv_sec = 0;
            diff->tv_xsec = 0;
            return 0;
        }
        usleep(100000);
    }
}

/*
 * evaluate IN_PPS signal signed offset in [2^-30 s]
 * and interval since last PPS in [s]
 */
void
comparator(struct timexspec *tx, int32_t *offset, uint32_t *sec)
{
    *sec = tx->tv_sec;
    *offset = tx->tv_xsec;

    if (*offset >= (XANOSEC >> 1))
    {
        *sec = *sec + 1;
        //*sec++; - this line was replaced by the above one -------
        *offset -= XANOSEC;
    }
}

/*
 * check offset
 * offset is GOOD if
 * abs. offset is smaller than GOOD_OFFS
 * or relative offset (diffrenece to last offset) is smaller
 * than MAX_DIFF and either fix_drift_flag or ini_drift_flag is set
 */
int
assess_pps(struct tclock *cl, int offset, int df)
{
    int a_diff, a_offs;

    a_offs = abs(offset);
    a_diff = abs(df);

    return (a_offs <= GOOD_OFFS || (a_diff <= MAX_DIFF
                                    && (cl->fix_drift_flag || cl->ini_drift_flag)));
}

void
set_clock_offs(struct tclock *cl, int offs)
{
    cl->last_offset = cl->offset;
    cl->offset = offs;
}

/*
 * set_clock_phase
 * function adjust RT_REG accoring to clock.offset value (step 2^-30 s)
 *
 */

int
set_clock_phase(struct hwio_comp *c, hwio_tool_comp_err_t err,
                const struct hwio_tool_params *params, struct tclock *cl)
{
    l_fp t, td, tc;
    struct timexspec tof,tco;
    uint32_t  aoffs;
    int neg;

    double xx_off;

    xx_off = cl->offset * 1.0E6 / (1024. * 1024*1024);
    printf("ADJ_PHASE: offset %i (cca %f [us])\n", cl->offset, xx_off);


    if (cl->offset >= 0)
    {
        aoffs = cl->offset;
        neg = 0;
    }
    else
    {
        aoffs = -cl->offset;
        neg = 1;
    }

    tof.tv_sec = 0;
    tof.tv_xsec = aoffs;
    TX2FP(tof, td);

    if(get_time(c, err, params, &t))
        return EXIT_FAILURE;
    if (neg)
    {
        L_ADD(t, td);
    }
    else
    {
        L_SUB(t, td);
    }

    /* correction of register read and write delay - approximatelly cca 5us
     */
    tco.tv_sec = 0;
    tco.tv_xsec = SET_CL_DELAY;
    TX2FP(tco, tc);
    L_ADD(t, tc);

    if(set_time(c, err, params, &t))
        return EXIT_FAILURE;
    cl->offset = 0;
    last_pps_xsec = 0;

    return EXIT_SUCCESS;
}

/*
 * adj_clock
 * adjust INC_REG accoring to clock.adj value (freq step 2^-34 s / 1s)
 * divided by NOM_HZ
 *
 */

int
adj_clock(struct hwio_comp *c, hwio_tool_comp_err_t err,
          const struct hwio_tool_params *params, struct tclock *cl)
{
    struct t_inc_reg ireg;
    l_fp t_ireg, t_adj;
    uint32_t aadj;
    int neg;

    double xx_adj;

    if (cl->adj >= 0)
    {
        aadj = cl->adj;
        neg = 0;
    }
    else
    {
        aadj = -cl->adj;
        neg = 1;
    }

    xx_adj = (double)aadj * 1.0E6 / (16. * 1024*1024*1024);
    printf("ADJ_FREQ: adj %i, abs.adj  0x%x (cca %f [us/s], drift %i)\n",
           cl->adj,aadj, xx_adj, cl->drift);

    /* calculation:
     * in: adj in parts of 2^-34,  NOM_HZ
     * out: corr (INC_REG correction) in 2^-64
     *
     *   corr = (adj * 2^32 / NOM_HZ ) / 4
     */

    t_adj.l_ui = aadj;
    t_adj.l_uf = 0;
    l_div(&t_adj, tsu_cv2_frequency); /* replace by new macro without float arithmetic */
    int two = 2; //if value added directly in next macro generates warnings
    L_RSHIFT(t_adj,two);

    //printf("t_adj: %u a %u \n", t_adj.l_ui, t_adj.l_uf);

    t_ireg.l_ui = cl->ref_inc_reg.f;
    t_ireg.l_uf = cl->ref_inc_reg.f_low;

    //printf("t_ireg: %u a %u \n", t_ireg.l_ui, t_ireg.l_uf);

    uint32_t dr_low;
    dr_low = cl->drift_low;
    dr_low *= 4;
    printf("adj 0x%x:%08x drift 0x%x:%08x\n", t_adj.l_ui, t_adj.l_uf, cl->drift, dr_low);

    //printf("neg: %d\n", neg);
    if (neg)
    {
        L_SUB(t_ireg, t_adj);
    }
    else
    {
        L_ADD(t_ireg, t_adj);
    }

    //printf("t_ireg: %u a %u \n", t_ireg.l_ui, t_ireg.l_uf);

    struct t_inc_reg ir;
    if(get_INC_REG(c, err, params, &ir))
        return EXIT_FAILURE;
    printf("inc_reg: before 0x%x:%08x", ir.f, ir.f_low);

    ireg.f = t_ireg.l_ui;
    ireg.f_low = t_ireg.l_uf;
    if(set_INC_REG(c, err, params, &ireg))
        return EXIT_FAILURE;

    if(get_INC_REG(c, err, params, &ir))
        return EXIT_FAILURE;
    printf("  after  0x%08x:%08x\n", ir.f, ir.f_low);

    return EXIT_SUCCESS;
}

/*
 * reset clock - synchronize to PPS signal and
 * set accurate clock drift + phase
 */

int
reset_clock(struct hwio_comp *c, hwio_tool_comp_err_t err,
            const struct hwio_tool_params *params, struct tclock *cl)
{
    int frst, error;
    int64_t acc;
    int32_t beg, to_cnt, good_cnt;
    struct timexspec pps, diff;
    int32_t x;
    uint32_t pps_sec;
    struct t_inc_reg ir;

    if(get_INC_REG(c, err, params, &ir))
        return EXIT_FAILURE;

    printf("inc_reg: 0x%x:%08x\n", ir.f, ir.f_low);

begin:

    frst = 1;
    acc = 0;
    to_cnt = 0;
    good_cnt = 0;
    beg = 0;

    error = new_pps(c, err, params, &pps, &diff);
    while (!error)
    {
        if (++to_cnt > PPS_TOUT_LIM)   /* too many timeouts */
        {
            return 0;
        }
        error = new_pps(c, err, params, &pps, &diff);
    }
    if(error==EXIT_FAILURE)
        return EXIT_FAILURE;


    cl->ini_drift_flag = 1;
    for (;;)
    {
        error = new_pps(c, err, params, &pps, &diff);
        if (!error)
        {
            if (++to_cnt > PPS_TOUT_LIM)   /* too many timeouts */
            {
                return 0;
            }
            goto begin;
        }
        else if(error==EXIT_FAILURE) // error on read or write
            return EXIT_FAILURE;

        comparator(&pps, &x, &pps_sec);

        if (frst)
        {
            beg = pps_sec;
            frst = 0;
        }
        else
        {
            cl->offset = x;
            if (!assess_pps(cl, x, diff.tv_xsec)) goto begin; /* bad PPS */
            acc += diff.tv_xsec;
            if (acc >  XANOSEC) acc -= XANOSEC;
            if (acc < -XANOSEC) acc += XANOSEC;
            if (++good_cnt >= (PPS_GOOD_LIM << 2)) break; /* let go on */
        }
    }
    /* multiply factor 16 - drift and offset binary order differs by 4
     */

    printf("reset clock: before - drift %d, acc %ld ", cl->drift, acc);
    cl->offset = 0;
    cl->drift += acc * (16. / (pps_sec - beg));
    cl->adj = - cl->drift;
    printf("after - drift %d, interv. %d, adj %d\n", cl->drift, pps_sec-beg, cl->adj);
    if(adj_clock(c, err, params, cl))
        return EXIT_FAILURE;

    /* set/amortize offset */
    cl->offset = x;
    cl->last_offset = x;
    if (!cl->no_clock_step && (x > GOOD_OFFS || x < -GOOD_OFFS))
    {
        if(set_clock_phase(c, err, params, cl))
            return EXIT_FAILURE;
        if(new_pps(c, err, params, &pps, &diff)==EXIT_FAILURE)
            return EXIT_FAILURE;
    }
    cl->ini_drift_flag = 0;
    return 2;
}

int
engine(struct hwio_comp *c, hwio_tool_comp_err_t err,
       const struct hwio_tool_params *params, struct tclock *cl,
       struct treg *r, struct tfil *f)
{
    struct timexspec now, pps, diff;
    unsigned int last_adj_sec, pps_sec;
    int to_cnt, error;
    int32_t x;

    cl->pps_good_cnt = 0;
    cl->pps_bad_cnt = 0;

    to_cnt = 0;

    if(do_clock_gettime(c, err, params, &now))
        return EXIT_FAILURE;
    last_adj_sec = now.tv_sec;

    if(new_pps(c, err, params, &pps, &diff)==EXIT_FAILURE) /* drop the first PPS */
        return EXIT_FAILURE;

    for (;;)
    {
        /* detect and select appropriate port to get regular PPS signal from */
        if(select_pps_pulse(c, err, params))
            return EXIT_FAILURE;

        error = new_pps(c, err, params, &pps, &diff);
        if (!error)
        {
            if (++to_cnt > PPS_TOUT_LIM)   /* too many timeouts */
            {
                return EXIT_SUCCESS;
            }
        }
        else if(error == EXIT_FAILURE)
            return EXIT_FAILURE;

        to_cnt = 0;
        comparator(&pps, &x, &pps_sec);
        if(do_clock_gettime(c, err, params, &now))
            return EXIT_FAILURE;

        if (assess_pps(cl,x,diff.tv_xsec))     /* good PPS */
        {
            if (cl->pps_bad_cnt)
            {
                if (++cl->pps_good_cnt > PPS_GOOD_LIM)
                {
                    cl->pps_bad_cnt = 0;
                    cl->pps_good_cnt = 0;
                }
            }
            cl->offset = filter(x, f);
            time_offset = xs2ns(cl->offset);
            time_freq = xs2ns(cl->drift);
            pi_reg(cl, r);
            if(do_clock_gettime(c, err, params, &now))
                return EXIT_FAILURE;
            if (now.tv_sec >= last_adj_sec + ADJ_INTERVAL)
            {
                if(adj_clock(c, err, params, cl))
                    return EXIT_FAILURE;
                last_adj_sec = now.tv_sec;
                calc_allan(cl->offset, cl->drift);
            }
            if(do_log(c, err, params))
                return EXIT_FAILURE;
        }
        else
        {
            if (++cl->pps_bad_cnt > PPS_BAD_LIM) return EXIT_SUCCESS;
            /* to many bad PPS */
            cl->pps_good_cnt = 0;
        }
    }
}

int
tsu_core(struct hwio_comp *c, hwio_tool_comp_err_t err,
         const struct hwio_tool_params *params)
{
    //int cl_init;

    pi_regul.ai_l = AI_L_COEF;
    pi_regul.ap_l = AP_L_COEF;

    init_filter(&iir_filter);
    init_allan();

    ptm_clock.drift = sys_xdrift;
    ptm_clock.drift_low = 0;
    ptm_clock.offset = 0;
    ptm_clock.adj = -ptm_clock.drift;
    ptm_clock.no_clock_step = 0;
    ptm_clock.fix_drift_flag = 0;
    ptm_clock.ini_drift_flag = 0;
    ptm_clock.phase_adj_flag = 0;

    if(get_INC_REG(c, err, params, &ptm_clock.ref_inc_reg))
        return EXIT_FAILURE;
    if(adj_clock(c, err, params, &ptm_clock))
        return EXIT_FAILURE;
    if(set_clock_phase(c, err, params, &ptm_clock))
        return EXIT_FAILURE;

    //cl_init =
    if(reset_clock(c, err, params, &ptm_clock)==EXIT_FAILURE)
        return EXIT_FAILURE;

    if(set_clock_phase(c, err, params, &ptm_clock))
        return EXIT_FAILURE;
    init_filter(&iir_filter);

    for (;;)
    {
        /* detect and select appropriate port to get regular PPS signal from */
        if(select_pps_pulse(c, err, params))
            return EXIT_FAILURE;

        if(engine(c, err, params, &ptm_clock, &pi_regul, &iir_filter))
            return EXIT_FAILURE;
        printf("engine: PPS_BAD, freq adjustment\n");
        if(reset_clock(c, err, params, &ptm_clock)==EXIT_FAILURE)
            return EXIT_FAILURE;
        init_filter(&iir_filter);
    }

    return EXIT_SUCCESS;
}

int
tsu_deinit(struct hwio_comp *c, hwio_tool_comp_err_t err,
           const struct hwio_tool_params *params)
{
    //if (dev) {
    if (c)
    {
        /* Clear INTA_reg */
        if(hwio_comp_write(c, TSU_CV2_INTA, 1))
            return hwio_tool_error(EXIT_FAILURE, "Unable to write register TSU_CV2_INTA");
        hwio_tool_verbose(params, 1, "INTA register cleared - stop generating valid timestamps");
    }

    /* Free design and detach device */
    //if (cs_design_free(dev) != 0) {
    //   VERBOSE(0, "cs_design_free failed");
    //}
    //cs_detach(&dev);
    //}


    hwio_tool_verbose(params, 1, "%s: Terminated", getprogname());


    return EXIT_SUCCESS;
}

/*
 * Make sure user does not provide garbage.
 */
int32_t
xstrtol(const char * str, int base)
{
    char *end;
    int32_t val;

    val = strtol(str, &end, base);
    if (*end != '\0' || str[0] == '\0') errx(1, "%s: not a signed number of base %d", str, base);

    return val;
}

uint32_t
xstrtoul(const char * str, int base)
{
    char *end;
    uint32_t val;

    val = strtoul(str, &end, base);
    if (*end != '\0' || str[0] == '\0')
        errx(1, "%s: not a number of base %d", str, base);

    return val;
}

double
xstrtod(const char * str)
{
    char *end;
    double val;

    val = strtod(str, &end);
    if (*end != '\0' || str[0] == '\0')
        errx(1, "%s: not a float point number", str);

    return val;
}

/* Converts number entered by user to version number, used in design.xml */
struct hwio_version
strtover(const char *str)
{
    u_int32_t major, minor;
    struct hwio_version ver=HWIO_VER2(0, 0);

    if (str != NULL) sscanf(str, "%u.%u", &major, &minor);
    else return ver;

    if ((major != 2) || ((minor != 0) && (minor != 1))) /* Allowed version check */
        return ver;
    else
    {
        hwio_version_set(&ver, major, minor, 0);
        return ver;
    }

}

int main(int argc, char **argv)
{
    struct hwio_tool_params params = hwio_tool_params_new(
                                         HWIO_TOOL_CUSTOM_NULL, argc, argv);

    switch(hwio_tool_params_parse(&params, &tsu_custom_parse, XCTL_OPTS))
    {
    case HWIO_TOOL_PARAMS_FAILED:
        return HWIO_TOOL_EARGS;

    case HWIO_TOOL_PARAMS_EXIT:
        return 0;

    default:
        break;
    }

    hwio_tool_verbose(&params, 0, "Arguments parsed successfully");
    hwio_tool_verbose(&params, 1, "Command: -%c", params.cmd.code);
    hwio_tool_verbose(&params, 2, "Argument: %s", params.cmd.arg);

    /* card interface init */
    if (!arg_device_file_path) strcpy(file, ctl_compat[0].type);
    printf("Device name: %s\n",file);


    switch (params.cmd.code)
    {
    case 'I':
        return hwio_tool_exec_generic(&params, ctl_compat,
                                      &hwio_tool_comp_info,
                                      &hwio_tool_error_handler);

    case 'L':
        return hwio_tool_exec_generic(&params, ctl_compat,
                                      &hwio_tool_comp_name,
                                      &hwio_tool_error_handler);

    case '\0':
        hwio_tool_verbose(&params, 1, "No command was specified");
        break;

    default:
        return hwio_tool_error(HWIO_TOOL_EARGS,
                               "Command -%c is not implemented",
                               params.cmd.code);
    }

    if(hwio_tool_exec_generic(&params, ctl_compat,
                              &tsu_init,
                              &hwio_tool_error_handler))
        return EXIT_FAILURE;

    /* main regulation loop */
    if(hwio_tool_exec_generic(&params, ctl_compat,
                              &tsu_core ,
                              &hwio_tool_error_handler))
        return EXIT_FAILURE;

    if(hwio_tool_exec_generic(&params, ctl_compat,
                              &tsu_deinit ,
                              &hwio_tool_error_handler))
        return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
