/***********************************************************************
 *								       *
 * Copyright (c) David L. Mills 1993-1998			       *
 *								       *
 * Permission to use, copy, modify, and distribute this software and   *
 * its documentation for any purpose and without fee is hereby	       *
 * granted, provided that the above copyright notice appears in all    *
 * copies and that both the copyright notice and this permission       *
 * notice appear in supporting documentation, and that the name	       *
 * University of Delaware not be used in advertising or publicity      *
 * pertaining to distribution of the software without specific,	       *
 * written prior permission. The University of Delaware makes no       *
 * representations about the suitability this software for any	       *
 * purpose. It is provided "as is" without express or implied	       *
 * warranty.							       *
 *								       *
 **********************************************************************/


/*
 * Double precision macros for 32-bit machines
 *
 * A 64-bit fixed-point value is represented in 32-bit architectures as
 * two 32-bit words in the following format:
 *
 *    0		    1		   2		   3
 *    0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
 *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *   |s|			Integral Part			     |
 *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *   |			       Fractional Part			     |
 *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *
 */

#ifndef L_FP_H
#define L_FP_H

#define u_char unsigned char
#define s_char char

typedef int32_t s_fp;
typedef uint32_t u_fp;


typedef struct {			/* basic type in two formats */
	union {
		uint32_t Xl_ui;
		int32_t Xl_i;
	} Ul_i;
	union {
		uint32_t Xl_uf;
		int32_t Xl_f;
	} Ul_f;
} l_fp;

#define l_ui	Ul_i.Xl_ui		/* unsigned integral part */
#define l_i	Ul_i.Xl_i		/* signed integral part */
#define l_uf	Ul_f.Xl_uf		/* unsigned fractional part */
#define l_f	Ul_f.Xl_f		/* signed fractional part */

/*
 * Byte order conversions
 */
#define HTONS_FP(x)     (htonl(x))
#define HTONL_FP(h, n)  do { (n).l_ui = htonl((h).l_ui); \
                             (n).l_uf = htonl((h).l_uf); } while (0)
#define NTOHS_FP(x)     (ntohl(x))
#define NTOHL_FP(n, h)  do { (h).l_ui = ntohl((n).l_ui); \
                             (h).l_uf = ntohl((n).l_uf); } while (0)

#define NTOHL_MFP(ni, nf, hi, hf) \
        do { (hi) = ntohl(ni); (hf) = ntohl(nf); } while (0)
#define HTONL_MFP(hi, hf, ni, nf) \
        do { (ni) = ntohl(hi); (nf) = ntohl(hf); } while (0)



#define M_ADD(r_i, r_f, a_i, a_f)	/* r += a */ \
	do { \
		register uint32_t lo_tmp; \
		register uint32_t hi_tmp; \
		\
		lo_tmp = ((r_f) & 0xffff) + ((a_f) & 0xffff); \
		hi_tmp = (((r_f) >> 16) & 0xffff) + (((a_f) >> 16) & \
		    0xffff); \
		if (lo_tmp & 0x10000) \
			 hi_tmp++; \
		(r_f) = ((hi_tmp & 0xffff) << 16) | (lo_tmp & 0xffff); \
		\
		(r_i) += (a_i); \
		if (hi_tmp & 0x10000) \
			 (r_i)++; \
	} while (0)

#define M_SUB(r_i, r_f, a_i, a_f)	/* r -= a */ \
	do { \
		register uint32_t lo_tmp; \
		register uint32_t hi_tmp; \
		\
		if ((a_f) == 0) { \
			 (r_i) -= (a_i); \
		} else { \
			lo_tmp = ((r_f) & 0xffff) + ((-((int32_t)(a_f))) \
			    & 0xffff); \
			hi_tmp = (((r_f) >> 16) & 0xffff) \
			    + (((-((int32_t)(a_f))) >> 16) & 0xffff); \
			if (lo_tmp & 0x10000) \
				 hi_tmp++; \
			(r_f) = ((hi_tmp & 0xffff) << 16) | (lo_tmp & \
			    0xffff); \
			(r_i) += ~(a_i); \
			if (hi_tmp & 0x10000) \
				 (r_i)++; \
		 } \
	} while (0)

#define M_NEG(v_i, v_f)  /* v = -v */ \
	do { \
		 if ((v_f) == 0) \
			 (v_i) = -((int32_t)(v_i)); \
		 else { \
			 (v_f) = -((int32_t)(v_f)); \
			 (v_i) = ~(v_i); \
		 } \
	} while (0)

#define M_RSHIFT(v_i, v_f, n)		/* v >>= n */ \
        do { /*printf("M_RSHIFT pred: v_i = %d, v_f = %d, n = %d", v_i, v_f, n);*/ \
                if ((v_i) < 0) { \
                        M_NEG((v_i), (v_f)); \
                        if ((n) < 32) { \
                                (v_f) = ((v_f) >> (unsigned) (n)) | ((v_i) << (32 - (n))); \
                                (v_i) = (v_i) >> (unsigned) (n); \
                        } else { \
                                if ((n) < 64) { \
                                        (v_f) = (v_i) >> (unsigned) (((n) - 32)); \
                                } else { \
                                        (v_f) = 0; \
                                } \
                                (v_i) = 0; \
                        } \
                        M_NEG((v_i), (v_f)); \
                } else { \
                        if ((n) < 32) { \
                                (v_f) = ((v_f) >> (unsigned) (n)) | ((v_i) << (32 - (n))); \
                                (v_i) = (v_i) >> (unsigned) (n); \
                        } else { \
                                if ((n) < 64) { \
                                        (v_f) = (v_i) >> (unsigned) (((n) - 32)); \
                                } else { \
                                        (v_f) = 0; \
                                } \
                                (v_i) = 0; \
                        } \
                 } \
                 /*printf(" po uprave: v_i = %d, v_f = %d, n = %d\n", v_i, v_f, n);*/ \
        } while (0)


#define M_MPY(v_i, v_f, m)		/* v *= m */ \
	do { \
		register uint32_t a, b, c, d; \
		if ((v_i) < 0) { \
			M_NEG((v_i), (v_f)); \
			d = ((v_f) & 0xffff) * (m); \
			c = ((v_f) >> 16) * (m) + (d >> 16); \
			b = ((v_i) & 0xffff) * (m) + (c >> 16); \
			a = ((v_i) >> 16) * (m) + (b >> 16); \
			(v_i) = (a << 16) + (b & 0xffff); \
			(v_f) = (c << 16) + (d & 0xffff); \
			M_NEG((v_i), (v_f)); \
		} else { \
			d = ((v_f) & 0xffff) * (m); \
			c = ((v_f) >> 16) * (m) + (d >> 16); \
			b = ((v_i) & 0xffff) * (m) + (c >> 16); \
			a = ((v_i) >> 16) * (m) + (b >> 16); \
			(v_i) = (a << 16) + (b & 0xffff); \
			(v_f) = (c << 16) + (d & 0xffff); \
		} \
	} while (0)
/*
 * Operations - u,v are 64 bits; a,n are 32 bits.
 */
#define L_ADD(v, u)	M_ADD((v).l_i, (v).l_uf, (u).l_i, (u).l_uf)
#define L_SUB(v, u)	M_SUB((v).l_i, (v).l_uf, (u).l_i, (u).l_uf)
#define L_ADDHI(v, a)	M_ADD((v).l_i, (v).l_uf, (a), 0)
#define L_NEG(v)	M_NEG((v).l_i, (v).l_uf)
#define L_RSHIFT(v, n)	M_RSHIFT((v).l_i, (v).l_uf, n)
#define L_MPY(v, a)	M_MPY((v).l_i, (v).l_uf, a)
#define L_CLR(v)	((v).l_i = (v).l_uf = 0)
#define L_ISNEG(v)	((v).l_ui < 0)
#define L_LINT(v, a)			/* load integral part */ \
	do { \
		 (v).l_i = (a); \
		 (v).l_uf = 0; \
	} while (0)
#define L_GINT(v)	((v).l_i)	/* get integral part */

/*
 * l_fp/double conversions
 */
#define FRAC            4294967296.             /* 2^32 as a double */

#define M_DTOLFP(d, r_ui, r_uf)                  /* double to l_fp */ \
	do { \
             \
		(r_ui) = (uint32_t)(d); \
		(r_uf) = (uint32_t)(((d) - (double)(r_ui)) * FRAC); \
	} while (0)

#define M_LFPTOD(r_ui, r_uf, d)                  /* l_fp to double */ \
        do { \
	     \
		(d) = (double)(r_ui) + ((double)(r_uf)) / FRAC; \
        } while (0)

#define DTOLFP(d, v)    M_DTOLFP((d), (v).l_ui, (v).l_uf)
#define LFPTOD(v, d)    M_LFPTOD((v).l_ui, (v).l_uf, (d))


struct pkt {
        u_char  li_vn_mode;     /* leap indicator, version and mode */
        u_char  stratum;        /* peer stratum */
        u_char  ppoll;          /* peer poll interval */
        s_char  precision;      /* peer clock precision */
        u_fp    rootdelay;      /* distance to primary clock */
        u_fp    rootdispersion; /* clock dispersion */
        uint32_t refid;          /* reference clock ID */
        l_fp    reftime;        /* time peer clock was last updated */
        l_fp    org;            /* originate time stamp */
        l_fp    rec;            /* receive time stamp */
        l_fp    xmt;            /* transmit time stamp */
};

/*
 * Stuff for extracting things from li_vn_mode
 */
#define PKT_MODE(li_vn_mode)    ((u_char)((li_vn_mode) & 0x7))
#define PKT_VERSION(li_vn_mode) ((u_char)(((li_vn_mode) >> 3) & 0x7))
#define PKT_LEAP(li_vn_mode)    ((u_char)(((li_vn_mode) >> 6) & 0x3))

#endif //L_FP_H

