/*!
 * \file ethctl_param.c
 * \brief Tool for controlling axi-ethernet: command line parameters 
 *        processing
 * \author Vlastimil Kosar <ikosar@fit.vutbr.cz>
 * \date 2012
 *
 * Copyright (C) 2012 Brno University of Technology
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id: filterctl.c 1469 2012-05-23 15:59:18Z xkekel00 $
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <unistd.h>
#include "ethctl_param.h"
#include "hwio_tool.h"

void hwio_tool_help(const char *toolname,
		const struct hwio_tool_params *params)
{
	usage(toolname);
}

/**
 * \brief Print help message.
 */
void usage(const char *pname) {
    printf("\
Usage: %s options\n\
-a mac    Set filter MAC addres at index specified by -n to mac.\n\
-b value  Enable [1] or disable [0] receive of bad frames.\n\
-A        Use settings specified by other command line parameters on all\n\
          axi-ethernets.\n\
-d dir    Use settings specified by other command line parameters for RX\n\
          interface [rR], TX interface [tT] or both [aAbB]. Some commands may\n\
          not be supported by both RX and TX interfaces.\n\
-e value  Enable [1] or disable [0] interface.\n\
-f value  Propage [1] or strip [0] FCS on RX interface. Use [1] FCS from logic\n\
          or compute[0] FCS on TX interface.\n\
-h        This help message.\n\
-i value  Use settings specified by other command line parameters on\n\
          axi-ethernet specified by the value. Axi-ethernet defined by -i X is the same as axi-ethernet defined by ethX.\n\
-j value  Enable [1] or disable [0] jumbo frames. Max size of jumbo frames\n\
          supported by axi-ethernet is 16KB.\n\
-l        List all all compatible axi-ethernet units within the design.\n\
          List is formated as [index] [vendor:type:name] [version].\n\
          If this parameter is used together with -i or -A the index coresponds \n\
          with Linux ethernet device eth<index>. If used with -u the index has \n\
          no meaning.\n\
-m mtu    Set MTU on interface to mtu. MTU is used by axi-ethernet only if\n\
          jumbo frames are disabled.\n\
-n index  Write MAC address of filter to filter slot index. Index in range 0-3\n\
          is supported.\n\
-p value  Enable [1] or disable [0] promiscuous mode.\n\
-s speed  Set speed of axi-ethernet core to speed. Supported speeds are\n\
          10/100/1000Mbps. Core may not support some of those speeds.\n\
          NOTE: This command does not effect the phyter.\n\
-S        Complete hex dump of statistics counters 0x200 - 0x30C.\n\
-t value  Enable [0] or disable [1] type/len check.\n\
-u value  Unit index/name in the design.\n\
-v value  Enable [1] or disable [0] VLAN support.\n\
-x path   Device tree path. Defaults to /proc/device-tree.\n\
-y device Physical device. Defaults to /dev/mem.\n\n\
To print basic statistics and configuration of axi-ethernet use just -A for\n\
all axi-ethernets, -i value for specific axi-ethernet equivalent with ethernet \n\
device eth<value> or -u value for axi-ethernet with name <value>.\n\
", pname);
}

/**
 * \brief Check if value is in specified range.
 * \param val Value to check.
 * \param a   Lower range bound.
 * \param b   Upper range bound.
 * \returns EXIT_FAILURE if value is out of range, EXIT_SUCCESS otherwise.
 */
int check_value(uint32_t val, uint32_t a, uint32_t b)
{
    return (val < a || val > b) ? EXIT_FAILURE : EXIT_SUCCESS;
}

/**
 * \brief Print error message if parameter is out of range.
 * \param param   Parameter name.
 * \param val Value to check.
 * \param a   Lower range bound.
 * \param b   Upper range bound.
 */
void print_param_error(char* param, uint32_t val, uint32_t a, uint32_t b)
{
    fprintf(stderr, "ERROR: Value of parameter %s is out of range. Invalid \
value is %u and the range is <%u,%u>\n", param, val, a, b);
}

/**
 * \brief Check command line parameters - are they sane?
 * \param params Command line parameters
 * \return EXIT_FAILURE if parameters are not sane,  EXIT_SUCCESS otherwise.
 */
int check_params(params_t *params)
{
    // Check legality of command line arguments
    if (params->sel_enable && check_value(params->enable, CMD_DIS, CMD_EN) != EXIT_SUCCESS)
    {
        print_param_error("-e", params->enable, CMD_DIS, CMD_EN);
        return EXIT_FAILURE;
    }
    if (params->sel_jumbo && check_value(params->jumbo, CMD_DIS, CMD_EN) != EXIT_SUCCESS)
    {
        print_param_error("-j", params->jumbo, CMD_DIS, CMD_EN);
        return EXIT_FAILURE;
    }
    if (params->sel_vlan && check_value(params->vlan, CMD_DIS, CMD_EN) != EXIT_SUCCESS)
    {
        print_param_error("-v", params->vlan, CMD_DIS, CMD_EN);
        return EXIT_FAILURE;
    }
    if (params->sel_fcs && check_value(params->fcs, CMD_DIS, CMD_EN) != EXIT_SUCCESS)
    {
        print_param_error("-f", params->fcs, CMD_DIS, CMD_EN);
        return EXIT_FAILURE;
    }
    if (params->sel_type && check_value(params->type, CMD_DIS, CMD_EN) != EXIT_SUCCESS)
    {
        print_param_error("-t", params->type, CMD_DIS, CMD_EN);
        return EXIT_FAILURE;
    }
    if (params->sel_promiscuite && check_value(params->promiscuite, CMD_DIS, CMD_EN) != EXIT_SUCCESS)
    {
        print_param_error("-p", params->promiscuite, CMD_DIS, CMD_EN);
        return EXIT_FAILURE;
    }
    if (params->sel_bad && check_value(params->bad, CMD_DIS, CMD_EN) != EXIT_SUCCESS)
    {
        print_param_error("-b", params->bad, CMD_DIS, CMD_EN);
        return EXIT_FAILURE;
    }
    if (params->sel_index && check_value(params->index, MAC_INDEX_MIN, MAC_INDEX_MAX) != EXIT_SUCCESS)
    {
        print_param_error("-n", params->index, MAC_INDEX_MIN, MAC_INDEX_MAX);
        return EXIT_FAILURE;
    }
    if (params->sel_mtu && check_value(params->mtu, MIN_MTU, MAX_MTU) != EXIT_SUCCESS)
    {
        print_param_error("-m", params->mtu, MIN_MTU, MAX_MTU);
        return EXIT_FAILURE;
    }
    
    // Check if parameters can be used together
    if (params->sel_jumbo && params->sel_mtu)
    {
        fprintf(stderr, "\
WARNING: Both jumbo frames enabled and MTU is set. Jumbo frames option has \
precedence before MTU settings.\n");
    }
    if (!params->sel_direction && (params->sel_enable || params->sel_jumbo || 
        params->sel_vlan || params->sel_fcs || params->sel_type || 
        params->sel_mtu || params->sel_bad))
    {
        fprintf(stderr, "\
ERROR: Parameter -d is not set. Ethtool can not perform actions specified by \
command line parameters -e -j -v -f -t -m -b without the -d parameter. Those \
actions are set separately for RX and TX parts of axi-ethernet.\n");
        return EXIT_FAILURE;
    }
    if (params->sel_all && params->sel_ifc_num)
    {
        fprintf(stderr, "\
ERROR: Parameter -A used together with -i. Specify only one of them as they \
are in conflict.\n");
        return EXIT_FAILURE;
    }
    if (params->sel_all && params->sel_name)
    {
        fprintf(stderr, "\
ERROR: Parameter -A used together with -u. Specify only one of them as they \
are in conflict.\n");
        return EXIT_FAILURE;
    }
    if (params->sel_name && params->sel_ifc_num)
    {
        fprintf(stderr, "\
ERROR: Parameter -u used together with -i. Specify only one of them as they \
are in conflict.\n");
        return EXIT_FAILURE;
    }
    if (!params->sel_all && !params->sel_ifc_num && !params->sel_name && (params->sel_full_dump || 
        params->sel_enable || params->sel_jumbo || params->sel_vlan || 
        params->sel_fcs || params->sel_type || params->sel_mtu || 
        params->sel_speed || params->sel_direction || params->sel_index || 
        params->sel_mac || params->sel_bad))
    {
        fprintf(stderr, "\
ERROR: Parameter -A, -u or -i is not set. Ethtool can not perform actions \
specified by command line parameters -d -e -j -v -f -t -m -s -S -n -a \
without the -A, -u or -i parameter. Some axi-ethernet or all of them must \
be selected.\n");
        return EXIT_FAILURE;
    }
    if (params->sel_mac && params->sel_index == 0)
    {
        fprintf(stderr, "\
ERROR: Parameter -n must be used together with used together with -a.\n");
        return EXIT_FAILURE;
    }
    
    // If only -i, -u or -A is used as command line parameters, set the list_only
    // flag to true (1)
    if ((params->sel_all || params->sel_ifc_num || params->sel_name) && !(params->sel_full_dump || 
        params->sel_enable || params->sel_jumbo || params->sel_vlan || 
        params->sel_fcs || params->sel_type || params->sel_mtu || 
        params->sel_speed || params->sel_direction || params->sel_index || 
        params->sel_mac || params->sel_bad))
    {
        params->list_only = 1;
    }
    
    return EXIT_SUCCESS;
}

/**
 * \brief Get numerical representation of MAC from string.
 * \param smac String representation of mac.
 * \param mac Resulting MAC as array.
 */
int get_mac(char* smac, mac_t mac)
{
    if (sscanf(smac, "%2X:%2X:%2X:%2X:%2X:%2X", &mac[0], &mac[1], &mac[2], &mac[3], &mac[4], &mac[5]) != 6)
        return EXIT_FAILURE;
    else
        return EXIT_SUCCESS;
}

/**
 * \brief Parse command line parameters.
 * \param argc Number of command line parameters.
 * \param argv Command line parameters.
 * \param params Parsed command line parameters.
 * \return EXIT_SUCCESS if parameters ware parsed OK, EXIT_FAILURE if error 
 *         occured.
 */
int parse_cmdline(int argc, char *argv[], params_t *params)
{
    char c;         /* getopt */
    
    memset(params, 0, sizeof(params_t)); // Set all selected flags to 0
    
    // Parse the command line arguments
    while ((c = getopt(argc, argv, ARGUMENTS)) != -1) {
        switch (c) {
        case 'S':
            params->sel_full_dump = 1;
            break;
        case 'A':
            params->sel_all = 1;
            break;
        case 'h':
            params->sel_help = 1;
            break;
        case 'i':
            params->sel_ifc_num = 1;
            params->ifc_num = atoi(optarg);
            break;
        case 'l':
            params->sel_list = 1;
            break;            
        case 'd':
            params->sel_direction = 1;
            switch (optarg[0])
            {
                case 'r':
                case 'R':
                    params->direction |= ENABLE_RX;
                    break;
                case 't':
                case 'T':
                    params->direction |= ENABLE_TX;
                    break;
                case 'a':
                case 'A':
                case 'b':
                case 'B':
                    params->direction |= ENABLE_RX;
                    params->direction |= ENABLE_TX;
                    break;
                default:
                    fprintf(stderr,"ERROR: Invalid direction specification: %c!\n", optarg[0]);
                    return EXIT_FAILURE;
            }
            break;
        case 'e':
            params->sel_enable = 1;
            params->enable = atoi(optarg);
            break;    
        case 'j':
            params->sel_jumbo = 1;
            params->jumbo = atoi(optarg);
            break;     
        case 'v':
            params->sel_vlan = 1;
            params->vlan = atoi(optarg);
            break;
        case 'f':
            params->sel_fcs = 1;
            params->fcs = atoi(optarg);
            break;
        case 't':
            params->sel_type = 1;
            params->type = atoi(optarg);
            break;    
        case 's':
            params->sel_speed = 1;
            switch (atoi(optarg))
            {
                case 10:
                    params->speed = SEL_10;
                    break;
                case 100:
                    params->speed = SEL_100;
                    break;
                case 1000:
                    params->speed = SEL_1000;
                    break;
                default:
                    fprintf(stderr,"ERROR: Invalid speed specified: %d!\n", atoi(optarg));
                    return EXIT_FAILURE;
            }
            break;    
        case 'm':
            params->sel_mtu = 1;
            params->mtu = atoi(optarg);
            break;
        case 'p':
            params->sel_promiscuite = 1;
            params->promiscuite = atoi(optarg);
            break;
        case 'n':
            params->sel_index = 1;
            params->index = atoi(optarg);
            break;
        case 'a':
            params->sel_mac = 1;
            if (get_mac(optarg, params->mac) == EXIT_FAILURE)
            {
                fprintf(stderr,"ERROR: Invalid MAC address specified: %s!\n", optarg);
                return EXIT_FAILURE;
            }
            break;
        case 'x':
            params->sel_dtree_path = 1;
            params->dtree_path = optarg;
            break;
        case 'u':
            params->sel_name = 1;
            params->name = optarg;
            break;
        case 'y':
            params->sel_device = 1;
            params->device = optarg;
            break;
        case 'b':
            params->sel_bad = 1;
            params->bad = atoi(optarg);
            break;
        default:
            return EXIT_FAILURE;
            break;
        }
    }
    
    return EXIT_SUCCESS;
}
