/*
 * Sec6Net AXI MDIO switch driver
 *
 * Author: Vlastimil Kosar
 *      kosar@fit.vutbr.cz
 *
 * Copyright (c) 2012  Faculty of Information Technologies, Brno University
 *                     of Technology
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/mutex.h>
#include <linux/slab.h>
#include <linux/of_address.h>

#define AXI_MDIO_SWITCH_NAME "axi_mdio_switch"

/*
 * Register definitions
 */
#define AXI_MDIO_SWITCH_ADDR 0x0  /* MDIO line select, addressed by phy addr */

/*
 *  
 */
struct axi_mdio_switch_local {
    struct device *dev;
    void __iomem *regs;
    struct mutex  switch_mutex;   /* Lock used for synchronization */
    };
    
static void axi_mdio_switch_write32(struct axi_mdio_switch_local *lp, off_t offset,  u32 value)
{                                  
       out_be32((lp->regs + offset), value);
}

static const struct of_device_id axi_mdio_switch_of_match[] = {
        { .compatible = "xlnx,axi-mdio-switch-1.00.a", },
        {}
};
MODULE_DEVICE_TABLE(of, axi_mdio_switch_of_match);

// Do it simple and unsafe when more than 1 mdio switch is used
static struct axi_mdio_switch_local *stat_mswitch = NULL;

// Lock the MDIO switch for communication with MDIO slave device defined by 
// phy_addr. 
void axi_mdio_switch_lock(u32 phy_addr)
{
    mutex_lock(&stat_mswitch->switch_mutex);
    axi_mdio_switch_write32(stat_mswitch, AXI_MDIO_SWITCH_ADDR, phy_addr);
}
EXPORT_SYMBOL(axi_mdio_switch_lock);

// Unlock the MDIO switch so another master can comunicate with MDIO slave 
// device.
void axi_mdio_switch_unlock(void)
{
    mutex_unlock(&stat_mswitch->switch_mutex);
}
EXPORT_SYMBOL(axi_mdio_switch_unlock);

static int __devinit axi_mdio_switch_probe(struct platform_device *dev)
{
    struct axi_mdio_switch_local *mswitch;
    
    mswitch = kzalloc(sizeof(*mswitch), GFP_KERNEL);
    if (!mswitch)
        return -ENOMEM;
    
    mutex_init(&mswitch->switch_mutex);
    
    mswitch->dev = &dev->dev;
    
    /* Map device registers */
    mswitch->regs = of_iomap(dev->dev.of_node, 0);
    if (!mswitch->regs) {
            dev_err(&dev->dev, "could not map AXI MDIO Switch regs.\n");
            goto nodev;
    }
    
    dev_set_drvdata(&dev->dev, mswitch);
    
    stat_mswitch = mswitch;
    
    return 0;

nodev:
    return -ENODEV;
}

static int __devexit axi_mdio_switch_remove(struct platform_device *dev)
{
    struct axi_mdio_switch_local *mswitch = dev_get_drvdata(&dev->dev);
    
    dev_set_drvdata(&dev->dev, NULL);
    iounmap(mswitch->regs);
    stat_mswitch = NULL;
    return 0;
}

static struct platform_driver axi_mdio_switch_driver = {
        .probe = axi_mdio_switch_probe,
        .remove = __devexit_p(axi_mdio_switch_remove),
        .driver = {
                .name = AXI_MDIO_SWITCH_NAME,
                .owner = THIS_MODULE,
                .of_match_table = axi_mdio_switch_of_match,
        },
};


static int __init axi_mdio_switch_init(void)
{
    return platform_driver_register(&axi_mdio_switch_driver);
}

module_init(axi_mdio_switch_init);

static void __exit axi_mdio_switch_exit(void)
{
    platform_driver_unregister(&axi_mdio_switch_driver);
}
module_exit(axi_mdio_switch_exit);

MODULE_AUTHOR("Vlastimil Kosar, Faculty of Information Technologies, Brno University of Technology <kosar@fit.vutbr.cz>");
MODULE_DESCRIPTION("Sec6Net AXI MDIO Switch driver");
MODULE_LICENSE("GPL");
