/*
 * Xilinx AxiEthernet device driver
 *
 * Copyright (c) 2008 Nissin Systems Co., Ltd.,  Yoshio Kashiwagi
 * Copyright (c) 2005-2008 DLA Systems,  David H. Lynch Jr. <dhlii@dlasys.net>
 * Copyright (c) 2008-2009 Secret Lab Technologies Ltd.
 * Copyright (c) 2010 Xilinx, Inc. All rights reserved.
 *
 * This is a driver for the Xilinx Axi Ethernet which is used
 * in the Virtex6 and Spartan6.
 *
 * TODO:
 * - Add Axi Fifo support.
 * - Factor out Axi DMA code into separate driver.
 * - Test and fix basic multicast filtering.
 * - Add support for extended multicast filtering.
 * - Test basic VLAN support.
 * - Add support for extended VLAN support.
 */
// #define DEBUG
#include <linux/delay.h>
#include <linux/etherdevice.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/of_mdio.h>
#include <linux/of_platform.h>
#include <linux/of_address.h>
#include <linux/skbuff.h>
#include <linux/spinlock.h>
#include <linux/phy.h>
#include <linux/mii.h>
#include <linux/atomic.h>

#include "xilinx_axienet.h"

/* Descriptors defines for Tx and Rx DMA - 2^n for the best performance */
#define TX_BD_NUM	64
#define RX_BD_NUM	128

/* Must be shorter than length of ethtool_drvinfo.driver field to fit */
#define DRIVER_NAME		"xae_stub"
#define DRIVER_DESCRIPTION	"Xilinx Axi Ethernet STUB driver"
#define DRIVER_VERSION		"1.00a"

#define AXIENET_REGS_N	32

/* Match table for of_platform binding */
// static struct of_device_id axienet_stub_of_match[] __devinitdata = {
// 	{ .compatible = "xlnx,axi-ethernet-1.00.a", },
// 	{ .compatible = "xlnx,axi-ethernet-1.01.a", },
// 	{ .compatible = "xlnx,axi-ethernet-2.01.a", },
// 	{},
// };

// MODULE_DEVICE_TABLE(of, axienet_stub_of_match);

/* Option table for setting up Axi Ethernet hardware options */
static struct axienet_option axienet_stub_options[] = {
	/* Turn on jumbo packet support for both Rx and Tx */
	{
		.opt = XAE_OPTION_JUMBO,
		.reg = XAE_TC_OFFSET,
		.m_or = XAE_TC_JUM_MASK,
	},
	{
		.opt = XAE_OPTION_JUMBO,
		.reg = XAE_RCW1_OFFSET,
		.m_or = XAE_RCW1_JUM_MASK,
	},
	/* Turn on VLAN packet support for both Rx and Tx */
	{
		.opt = XAE_OPTION_VLAN,
		.reg = XAE_TC_OFFSET,
		.m_or = XAE_TC_VLAN_MASK,
	},
	{
		.opt = XAE_OPTION_VLAN,
		.reg = XAE_RCW1_OFFSET,
		.m_or = XAE_RCW1_VLAN_MASK,
	},
	/* Turn on FCS stripping on receive packets */
	{
		.opt = XAE_OPTION_FCS_STRIP,
		.reg = XAE_RCW1_OFFSET,
		.m_or = XAE_RCW1_FCS_MASK,
	},
	/* Turn on FCS insertion on transmit packets */
	{
		.opt = XAE_OPTION_FCS_INSERT,
		.reg = XAE_TC_OFFSET,
		.m_or = XAE_TC_FCS_MASK,
	},
	/* Turn off length/type field checking on receive packets */
	{
		.opt = XAE_OPTION_LENTYPE_ERR,
		.reg = XAE_RCW1_OFFSET,
		.m_or = XAE_RCW1_LT_DIS_MASK,
	},
	/* Turn on promiscuous frame filtering (all frames will be received ) */
	{
		.opt = XAE_OPTION_PROMISC,
		.reg = XAE_FMI_OFFSET,
		.m_or = XAE_FMI_PM_MASK,
	},
	/* Enable transmitter */
	{
		.opt = XAE_OPTION_TXEN,
		.reg = XAE_TC_OFFSET,
		.m_or = XAE_TC_TX_MASK,
	},
	/* Enable receiver */
	{
		.opt = XAE_OPTION_RXEN,
		.reg = XAE_RCW1_OFFSET,
		.m_or = XAE_RCW1_RX_MASK,
	},
	{}
};

/**
 * axienet_ior - Memory mapped Axi Ethernet register read
 * @lp:     Pointer to axienet local structure
 * @offset: Address offset from the base address of Axi Ethernet core
 *
 * returns: The contents of the Axi Ethernet register
 *
 * This function returns the contents of the corresponding register.
 **/
// u32 axienet_ior(struct axienet_local *lp, int offset)
// {
//     return in_be32(lp->regs + offset);
// }

/**
 * axienet_iow - Memory mapped Axi Ethernet register write
 * @lp:     Pointer to axienet local structure
 * @offset: Address offset from the base address of Axi Ethernet core
 * @value:  Value to be written into the Axi Ethernet register
 *
 * This function writes the desired value into the corresponding Axi Ethernet
 * register.
 **/
// void axienet_iow(struct axienet_local *lp, int offset, u32 value)
// {
//     out_be32((lp->regs + offset), value);
// }

/* ----------------------------------------------------------------------------
 * net_device_ops
 */

/**
 * axienet_stub_set_mac_address - Write the MAC address
 * @ndev:	Pointer to the net_device structure
 * @address:	6 byte Address to be written as MAC address
 *
 * This function is called to initialize the MAC address of the Axi Ethernet
 * core. It writes to the UAW0 and UAW1 registers of the core.
 **/
static void axienet_stub_set_mac_address(struct net_device *ndev, void *address)
{
	struct axienet_local *lp = netdev_priv(ndev);

	if (address)
		memcpy(ndev->dev_addr, address, ETH_ALEN);

	if (!is_valid_ether_addr(ndev->dev_addr))
		random_ether_addr(ndev->dev_addr);

	/*
	 * Set up unicast MAC address filter set its mac address
	 */
	axienet_iow(lp, XAE_UAW0_OFFSET,
				(ndev->dev_addr[0]) |
				(ndev->dev_addr[1] << 8) |
				(ndev->dev_addr[2] << 16) |
				(ndev->dev_addr[3] << 24));

	axienet_iow(lp, XAE_UAW1_OFFSET,
		(((axienet_ior(lp, XAE_UAW1_OFFSET)) &
		~XAE_UAW1_UNICASTADDR_MASK) |
		(ndev->dev_addr[4] |
		(ndev->dev_addr[5] << 8))));
}

/**
 * netdev_set_mac_address - Write the MAC address (from outside the driver)
 * @ndev:	Pointer to the net_device structure
 * @p:		6 byte Address to be written as MAC address
 *
 * returns: 0 for all conditions. Presently, there is no failure case.
 *
 * This function is called to initialize the MAC address of the Axi Ethernet
 * core. It calls the core specific axienet_stub_set_mac_address. This is the
 * function that goes into net_device_ops structure entry ndo_set_mac_address.
 **/
static int stub_netdev_set_mac_address(struct net_device *ndev, void *p)
{
	struct sockaddr *addr = p;

	axienet_stub_set_mac_address(ndev, addr->sa_data);
	return 0;
}

/**
 * axienet_stub_set_multicast_list - Prepare the multicast table
 * @ndev:	Pointer to the net_device structure
 *
 * This function is called to initialize the multicast table during
 * initialization. The Axi Ethernet basic multicast support has a four-entry
 * multicast table which is initialized here. Additionally this function
 * goes into the net_device_ops structure entry ndo_set_multicast_list. This
 * means whenever the multicast table entries need to be updated this
 * function gets called.
 **/
static void axienet_stub_set_multicast_list(struct net_device *ndev)
{
	struct axienet_local *lp = netdev_priv(ndev);
	int i;
	u32 reg;
	u32 af0reg;
	u32 af1reg;

	if (ndev->flags & (IFF_ALLMULTI | IFF_PROMISC) ||
	    netdev_mc_count(ndev) > XAE_MULTICAST_CAM_TABLE_NUM) {
		/* We must make the kernel realize we had to move into
		 * promiscuous mode. If it was a promiscuous mode request
		 * the flag is already set.
		 * If not we set it. */
		ndev->flags |= IFF_PROMISC;
		reg = axienet_ior(lp, XAE_FMI_OFFSET);
		reg |= XAE_FMI_PM_MASK;
		axienet_iow(lp, XAE_FMI_OFFSET, reg);
		dev_info(&ndev->dev, "Promiscuous mode enabled.\n");
	} else if (!netdev_mc_empty(ndev)) {
		struct netdev_hw_addr *ha;

		i = 0;
		netdev_for_each_mc_addr(ha, ndev) {
			if (i >= XAE_MULTICAST_CAM_TABLE_NUM)
				break;

			af0reg = (ha->addr[0]);
			af0reg |= (ha->addr[1] << 8);
			af0reg |= (ha->addr[2] << 16);
			af0reg |= (ha->addr[3] << 24);

			af1reg = (ha->addr[4]);
			af1reg |= (ha->addr[5] << 8);

			reg = axienet_ior(lp, XAE_FMI_OFFSET) & 0xFFFFFF00;
			reg |= i;
			axienet_iow(lp, XAE_FMI_OFFSET, reg);
			axienet_iow(lp, XAE_AF0_OFFSET, af0reg);
			axienet_iow(lp, XAE_AF1_OFFSET, af1reg);
			i++;
		}
	} else {
		reg = axienet_ior(lp, XAE_FMI_OFFSET);
		reg &= ~XAE_FMI_PM_MASK;
		axienet_iow(lp, XAE_FMI_OFFSET, reg);

		for (i = 0; i < XAE_MULTICAST_CAM_TABLE_NUM; i++) {
			reg = axienet_ior(lp, XAE_FMI_OFFSET) & 0xFFFFFF00;
			reg |= i;
			axienet_iow(lp, XAE_FMI_OFFSET, reg);
			axienet_iow(lp, XAE_AF0_OFFSET, 0);
			axienet_iow(lp, XAE_AF1_OFFSET, 0);
		}
		dev_info(&ndev->dev, "Promiscuous mode disabled.\n");
	}
}

/**
 * axienet_stub_setoptions - Set an Axi Ethernet option
 * @ndev:	Pointer to the net_device structure
 * @options:	Option to be enabled/disabled
 *
 * The Axi Ethernet core has multiple features which can be selectively turned
 * on or off. The typical options could be jumbo frame option, basic VLAN
 * option, promiscuous mode option etc. This function is used to set or clear
 * these options in the Axi Ethernet hardware. This is done through
 * axienet_option structure .
 **/
static void axienet_stub_setoptions(struct net_device *ndev, u32 options)
{
	struct axienet_local *lp = netdev_priv(ndev);
	struct axienet_option *tp = &axienet_stub_options[0];
	int reg;

	while (tp->opt) {
		reg = ((axienet_ior(lp, tp->reg)) & ~(tp->m_or));
		if (options & tp->opt)
			reg |= tp->m_or;
		axienet_iow(lp, tp->reg, reg);
		tp++;
	}
	lp->options |= options;
}

/**
 * axienet_stub_device_reset - Reset and initialize the Axi Ethernet hardware.
 * @ndev:	Pointer to the net_device structure
 *
 * This function is called to reset and initialize the Axi Ethernet core. This
 * is typically called during initialization. It does a reset of the Axi DMA
 * Rx/Tx channels and initializes the Axi DMA BDs. Since Axi DMA reset lines
 * areconnected to Axi Ethernet reset lines, this in turn resets the Axi
 * Ethernet core. No separate hardware reset is done for the Axi Ethernet
 * core.
 **/
static void axienet_stub_device_reset(struct net_device *ndev)
{
	struct axienet_local *lp = netdev_priv(ndev);
	u32 timeout;
	u32 axienet_status;

	lp->max_frm_size = XAE_MAX_VLAN_FRAME_SIZE;
	lp->options &= (~XAE_OPTION_JUMBO);

	if ((ndev->mtu > XAE_MTU) && (ndev->mtu <= XAE_JUMBO_MTU)
						&& (lp->jumbo_support)) {
		lp->max_frm_size = ndev->mtu + XAE_HDR_VLAN_SIZE + XAE_TRL_SIZE;
		lp->options |= XAE_OPTION_JUMBO;
	}

	axienet_status = axienet_ior(lp, XAE_RCW1_OFFSET);
	axienet_status &= ~XAE_RCW1_RX_MASK;
	axienet_iow(lp, XAE_RCW1_OFFSET, axienet_status);

	axienet_status = axienet_ior(lp, XAE_IP_OFFSET);
	if (axienet_status & XAE_INT_RXRJECT_MASK)
		axienet_iow(lp, XAE_IS_OFFSET, XAE_INT_RXRJECT_MASK);

	axienet_iow(lp, XAE_FCC_OFFSET, XAE_FCC_FCRX_MASK);


	/* Sync default options with HW but leave receiver and
	 * transmitter disabled.*/
	axienet_stub_setoptions(ndev,
			 lp->options & ~(XAE_OPTION_TXEN | XAE_OPTION_RXEN));

// 	axienet_stub_set_mac_address(ndev, NULL);

	/* Set address filter table*/
	axienet_stub_set_multicast_list(ndev);
	axienet_stub_setoptions(ndev, lp->options);

	/* Init net_device trans_start variable */
	ndev->trans_start = jiffies; /* prevent Tx timeout */

}

/**
 * axienet_stub_adjust_link - Adjust the PHY link speed/duplex.
 * @ndev:	Pointer to the net_device structure
 *
 * This function is called to change the speed and duplex setting after
 * auto negotiation is done by the PHY. This is the function that gets
 * registered with the PHY interface through the "of_phy_connect" call.
 **/
static void axienet_stub_adjust_link(struct net_device *ndev)
{
	struct axienet_local *lp = netdev_priv(ndev);
	struct phy_device *phy = lp->phy_dev;
	u32 emmc_reg;
	u32 link_state;
	u32 setspeed = 1;
	/* Hash together the state values to decide if something has changed.*/
	link_state = phy->speed | (phy->duplex << 1) | phy->link;
   
	if (lp->last_link != link_state) {        
		if ((phy->speed == SPEED_10) || (phy->speed == SPEED_100)) {
			if (lp->phy_type == XAE_PHY_TYPE_1000BASE_X)
				setspeed = 0;
		} else {
			if ((phy->speed == SPEED_1000) &&
					(lp->phy_type == XAE_PHY_TYPE_MII))
				setspeed = 0;
		}

		if (setspeed == 1) {
			emmc_reg = axienet_ior(lp, XAE_EMMC_OFFSET);
			emmc_reg &= ~XAE_EMMC_LINKSPEED_MASK;

			switch (phy->speed) {
			case SPEED_1000:
				emmc_reg |= XAE_EMMC_LINKSPD_1000;
				break;
			case SPEED_100:
				emmc_reg |= XAE_EMMC_LINKSPD_100;
				break;
			case SPEED_10:
				emmc_reg |= XAE_EMMC_LINKSPD_10;
				break;
			default:
				dev_err(&ndev->dev,
				"Speed other than 10,100 or 1Gbps is not supported\n");
				break;

			}

			/* Write new speed setting out to Axi Ethernet */
			axienet_iow(lp, XAE_EMMC_OFFSET, emmc_reg);
			lp->last_link = link_state;
			phy_print_status(phy);

		} else {
			dev_err(&ndev->dev,
				"Error setting Axi Ethernet mac speed\n");
		}
	}
}

/**
 * axienet_stub_start_xmit - Starts the transmission.
 * @skb:	sk_buff pointer that contains data to be Txed.
 * @ndev:	Pointer to net_device structure.
 *
 * returns: NETDEV_TX_OK, on success
 *	    NETDEV_TX_BUSY, if any of the descriptors are not free
 *
 * This function is invoked from upper layers to initiate transmission. The
 * function uses the next available free BDs and populates their fields to
 * start the transmission. Additionally if checksum offloading is supported,
 * it populates AXI Stream Control fields with appropriate values.
 **/
static int axienet_stub_start_xmit(struct sk_buff *skb, struct net_device *ndev)
{
	return NETDEV_TX_OK;
}


/**
 * axienet_stub_open - Driver open routine.
 * @ndev:	Pointer to net_device structure
 *
 * returns: 0, on success.
 *	    -ENODEV, if PHY cannot be connected to
 *	    non-zero error value on failure
 *
 * This is the driver open routine. It calls phy_start to start the PHY device.
 * It also allocates interrupt service routines, enables the interrupt lines
 * and ISR handling. Axi Ethernet core is reset through Axi DMA core. Buffer
 * descriptors are initialized.
 **/
static int axienet_stub_open(struct net_device *ndev)
{
	struct axienet_local *lp = netdev_priv(ndev);
	int rc;
	int mdio_mcreg;
	long end = jiffies + 2;

	dev_dbg(&ndev->dev, "axienet_stub_open()\n");

	mdio_mcreg = axienet_ior(lp, XAE_MDIO_MC_OFFSET);
	/* Wait till MDIO interface is ready.*/
	while (!(axienet_ior(lp, XAE_MDIO_MCR_OFFSET) &
						XAE_MDIO_MCR_READY_MASK)) {

		if (end - jiffies <= 0) {
			WARN_ON(1);
			return -ETIMEDOUT;
		}
		msleep(1);
	}

	/* Disable the MDIO interface till Axi Ethernet Reset is completed.
	 * When we do an Axi Ethernet reset, it resets the complete core
	 * including the MDIO. If MDIO is not disabled when the reset
	 * process is started, MDIO will be broken afterwards. */
	axienet_iow(lp, XAE_MDIO_MC_OFFSET, (mdio_mcreg &
						(~XAE_MDIO_MC_MDIOEN_MASK)));
	axienet_stub_device_reset(ndev);

	/* Enable the MDIO */
	axienet_iow(lp, XAE_MDIO_MC_OFFSET, mdio_mcreg);
	/* Wait till MDIO interface is ready.*/
	end = jiffies + 2;
	while (!(axienet_ior(lp, XAE_MDIO_MCR_OFFSET) &
						XAE_MDIO_MCR_READY_MASK)) {
		if (end - jiffies <= 0) {
			WARN_ON(1);
			return -ETIMEDOUT;
		}
		msleep(1);
	}

	if (lp->phy_node) {
		lp->phy_dev = of_phy_connect(lp->ndev,
						lp->phy_node,
						axienet_stub_adjust_link,
						0,
						PHY_INTERFACE_MODE_GMII);
		if (!lp->phy_dev) {
			dev_err(lp->dev, "of_phy_connect() failed\n");
			return -ENODEV;
		}
		phy_start(lp->phy_dev);
	}
	return 0;
}

/**
 * axienet_stub_stop - Driver stop routine.
 * @ndev:	Pointer to net_device structure
 *
 * returns: 0, on success.
 *
 * This is the driver stop routine. It calls phy_disconnect to stop the PHY
 * device. It also removes the interrupt handlers and disables the interrupts.
 * The Axi DMA Tx/Rx BDs are released.
 **/
static int axienet_stub_stop(struct net_device *ndev)
{
	struct axienet_local *lp = netdev_priv(ndev);

	dev_dbg(&ndev->dev, "axienet_stub_close()\n");

	axienet_stub_setoptions(ndev,
			 lp->options & ~(XAE_OPTION_TXEN | XAE_OPTION_RXEN));


	if (lp->phy_dev)
		phy_disconnect(lp->phy_dev);
	lp->phy_dev = NULL;

	return 0;
}

/**
 * axienet_stub_change_mtu - Driver change mtu routine.
 * @ndev:	Pointer to net_device structure
 * @new_mtu:	New mtu value to be applied
 *
 * returns: Always returns 0 (success).
 *
 * This is the change mtu driver routine. It checks if the Axi Ethernet
 * hardware supports jumbo frames before changing the mtu. This can be
 * called only when the device is not up.
 **/
static int axienet_stub_change_mtu(struct net_device *ndev, int new_mtu)
{
	struct axienet_local *lp = netdev_priv(ndev);

	if (netif_running(ndev))
		return -EBUSY;

	if (lp->jumbo_support) {
		if ((new_mtu > XAE_JUMBO_MTU) || (new_mtu < 64))
			return -EINVAL;
		ndev->mtu = new_mtu;
	} else {
		if ((new_mtu > XAE_MTU) || (new_mtu < 64))
			return -EINVAL;
		ndev->mtu = new_mtu;
	}

	return 0;
}

#ifdef CONFIG_NET_POLL_CONTROLLER
/**
 * axienet_stub_poll_controller - Axi Ethernet poll mechanism.
 * @ndev:	Pointer to net_device structure
 *
 * This implements Rx/Tx ISR poll mechanisms. The interrupts are disabled prior
 * to polling the ISRs and are enabled back after the polling is done.
 **/
static void axienet_stub_poll_controller(struct net_device *ndev)
{
}
#endif


/**
 * axienet_stub_validate_addr - Validate ethernet address.
 * @ndev:   Pointer to net_device structure
 * 
 * For stub driver this method always returns 0 as stub driver is always 
 * promiscuous.
 */
int axienet_stub_validate_addr(struct net_device *dev)
{
    return 0;
}

static const struct net_device_ops axienet_stub_netdev_ops = {
	.ndo_open = axienet_stub_open,
	.ndo_stop = axienet_stub_stop,
	.ndo_start_xmit = axienet_stub_start_xmit,
	.ndo_change_mtu	= axienet_stub_change_mtu,
	.ndo_set_mac_address = stub_netdev_set_mac_address,
	.ndo_validate_addr = axienet_stub_validate_addr, // eth_validate_addr
	.ndo_set_multicast_list = axienet_stub_set_multicast_list,
#ifdef CONFIG_NET_POLL_CONTROLLER
	.ndo_poll_controller = axienet_stub_poll_controller,
#endif
};

/**
 * axienet_stub_ethtools_get_settings - Get Axi Ethernet settings related to PHY.
 * @ndev:	Pointer to net_device structure
 * @ecmd:	Pointer to ethtool_cmd structure
 *
 * This implements ethtool command for getting PHY settings. If PHY could
 * not be found, the function returns -ENODEV. This function calls the
 * relevant PHY ethtool API to get the PHY settings.
 * Issue "ethtool ethX" under linux prompt to execute this function.
 **/
static int axienet_stub_ethtools_get_settings(struct net_device *ndev,
						struct ethtool_cmd *ecmd)
{
	struct axienet_local *lp = netdev_priv(ndev);
	struct phy_device *phydev = lp->phy_dev;

	if (!phydev)
		return -ENODEV;

	return phy_ethtool_gset(phydev, ecmd);
}

/**
 * axienet_stub_ethtools_set_settings - Set PHY settings as passed in the argument.
 * @ndev:	Pointer to net_device structure
 * @ecmd:	Pointer to ethtool_cmd structure
 *
 * This implements ethtool command for setting various PHY settings. If PHY
 * could not be found, the function returns -ENODEV. This function calls the
 * relevant PHY ethtool API to set the PHY.
 * Issue e.g. "ethtool -s ethX speed 1000" under linux prompt to execute this
 * function.
 **/
static int axienet_stub_ethtools_set_settings(struct net_device *ndev,
						struct ethtool_cmd *ecmd)
{
	struct axienet_local *lp = netdev_priv(ndev);
	struct phy_device *phydev = lp->phy_dev;

	if (!phydev)
		return -ENODEV;

	return phy_ethtool_sset(phydev, ecmd);
}

/**
 * axienet_stub_ethtools_get_drvinfo - Get various Axi Ethernet driver information.
 * @ndev:	Pointer to net_device structure
 * @ed:		Pointer to ethtool_drvinfo structure
 *
 * This implements ethtool command for getting the driver information.
 * Issue "ethtool -i ethX" under linux prompt to execute this function.
 **/
static void axienet_stub_ethtools_get_drvinfo(struct net_device *ndev,
						struct ethtool_drvinfo *ed)
{
	memset(ed, 0, sizeof(struct ethtool_drvinfo));
	strcpy(ed->driver, DRIVER_NAME);
	strcpy(ed->version, DRIVER_VERSION);
	ed->regdump_len = (sizeof(u32))*(AXIENET_REGS_N);
}

/**
 * axienet_stub_ethtools_get_regs_len - Get the total regs length present in the
 *				   AxiEthernet core.
 * @ndev:	Pointer to net_device structure
 *
 * This implements ethtool command for getting the total register length
 * information.
 **/
static int axienet_stub_ethtools_get_regs_len(struct net_device *ndev)
{
	return AXIENET_REGS_N * sizeof(u32);
}

/**
 * axienet_stub_ethtools_get_regs - Dump the contents of all registers present
 *			       in AxiEthernet core.
 * @ndev:	Pointer to net_device structure
 * @regs:	Pointer to ethtool_regs structure
 * @ret:	Void pointer used to return the contents of the registers.
 *
 * This implements ethtool command for getting the Axi Ethernet register dump.
 * Issue "ethtool -d ethX" to execute this function.
 **/
static void axienet_stub_ethtools_get_regs(struct net_device *ndev,
					struct ethtool_regs *regs, void *ret)
{
	struct axienet_local *lp = netdev_priv(ndev);
	u32 * data = (u32 *) ret;

	regs->version = 0;
	regs->len = AXIENET_REGS_N * sizeof(u32);
	memset(ret, 0, AXIENET_REGS_N * sizeof(u32));

	data[0] = axienet_ior(lp, XAE_RAF_OFFSET);
	data[1] = axienet_ior(lp, XAE_TPF_OFFSET);
	data[2] = axienet_ior(lp, XAE_IFGP_OFFSET);
	data[3] = axienet_ior(lp, XAE_IS_OFFSET);
	data[4] = axienet_ior(lp, XAE_IP_OFFSET);
	data[5] = axienet_ior(lp, XAE_IE_OFFSET);
	data[6] = axienet_ior(lp, XAE_TTAG_OFFSET);
	data[7] = axienet_ior(lp, XAE_RTAG_OFFSET);
	data[8] = axienet_ior(lp, XAE_UAWL_OFFSET);
	data[9] = axienet_ior(lp, XAE_UAWU_OFFSET);
	data[10] = axienet_ior(lp, XAE_TPID0_OFFSET);
	data[11] = axienet_ior(lp, XAE_TPID1_OFFSET);
	data[12] = axienet_ior(lp, XAE_PPST_OFFSET);
	data[13] = axienet_ior(lp, XAE_RCW0_OFFSET);
	data[14] = axienet_ior(lp, XAE_RCW1_OFFSET);
	data[15] = axienet_ior(lp, XAE_TC_OFFSET);
	data[16] = axienet_ior(lp, XAE_FCC_OFFSET);
	data[17] = axienet_ior(lp, XAE_EMMC_OFFSET);
	data[18] = axienet_ior(lp, XAE_PHYC_OFFSET);
	data[19] = axienet_ior(lp, XAE_MDIO_MC_OFFSET);
	data[20] = axienet_ior(lp, XAE_MDIO_MCR_OFFSET);
	data[21] = axienet_ior(lp, XAE_MDIO_MWD_OFFSET);
	data[22] = axienet_ior(lp, XAE_MDIO_MRD_OFFSET);
	data[23] = axienet_ior(lp, XAE_MDIO_MIS_OFFSET);
	data[24] = axienet_ior(lp, XAE_MDIO_MIP_OFFSET);
	data[25] = axienet_ior(lp, XAE_MDIO_MIE_OFFSET);
	data[26] = axienet_ior(lp, XAE_MDIO_MIC_OFFSET);
	data[27] = axienet_ior(lp, XAE_UAW0_OFFSET);
	data[28] = axienet_ior(lp, XAE_UAW1_OFFSET);
	data[29] = axienet_ior(lp, XAE_FMI_OFFSET);
	data[30] = axienet_ior(lp, XAE_AF0_OFFSET);
	data[31] = axienet_ior(lp, XAE_AF1_OFFSET);
}

/**
 * axienet_stub_ethtools_get_rx_csum - Get the checksum offload setting on Rx path.
 * @ndev:	Pointer to net_device structure
 *
 * This implements ethtool command for getting the Axi Ethernet checksum
 * offload setting on Rx path. If the core supports either partial or full
 * checksum offload, the function returns a non-zero value.
 * Issue "ethtool -k ethX" under linux prompt to execute this function.
 **/
static u32 axienet_stub_ethtools_get_rx_csum(struct net_device *ndev)
{
	struct axienet_local *lp = netdev_priv(ndev);

	if ((lp->features & XAE_FEATURE_PARTIAL_RX_CSUM) ||
			(lp->features & XAE_FEATURE_FULL_RX_CSUM))
		return XAE_FEATURE_PARTIAL_RX_CSUM;

	else
		return XAE_NO_CSUM_OFFLOAD;
}

/**
 * axienet_stub_ethtools_set_rx_csum - Enable checksum offloading on Rx path.
 * @ndev:	Pointer to net_device structure
 * @flag:	unsigned long, used to enable/disable checksum offloading.
 *
 * This implements ethtool command for enabling Axi Ethernet checksum
 * offloading. If the core supports full checksum offloading, this function
 * enables/disables full checksum offloading. Similarly it can enable/disable
 * partial checksum offloading.
 * Issue "ethtool -K ethX rx on|off" under linux prompt to execute this
 * function.
 **/
static int axienet_stub_ethtools_set_rx_csum(struct net_device *ndev, u32 flag)
{
	struct axienet_local *lp = netdev_priv(ndev);

	if (lp->csum_offload_on_rx_path & XAE_FEATURE_PARTIAL_RX_CSUM) {
		if (flag)
			lp->features |= XAE_FEATURE_PARTIAL_RX_CSUM;
		else
			lp->features &= ~XAE_FEATURE_PARTIAL_RX_CSUM;
	} else if (lp->csum_offload_on_rx_path & XAE_FEATURE_FULL_RX_CSUM) {
		if (flag)
			lp->features |= XAE_FEATURE_FULL_RX_CSUM;
		else
			lp->features &= ~XAE_FEATURE_FULL_RX_CSUM;
	}

	return 0;
}

/**
 * axienet_stub_ethtools_get_tx_csum - Get checksum offloading on Tx path.
 * @ndev:	Pointer to net_device structure
 *
 * This implements ethtool command for getting the Axi Ethernet checksum
 * offload setting on Tx path.
 * Issue "ethtool -k ethX" under linux prompt to execute this function.
 **/
static u32 axienet_stub_ethtools_get_tx_csum(struct net_device *ndev)
{
	struct axienet_local *lp = netdev_priv(ndev);

	if (lp->features & XAE_FEATURE_PARTIAL_TX_CSUM)
		return ndev->features & NETIF_F_IP_CSUM;
	else if (lp->features & XAE_FEATURE_FULL_TX_CSUM)
		return ndev->features & NETIF_F_HW_CSUM;
	return 0;
}

/**
 * axienet_stub_ethtools_set_tx_csum - Enable checksum offloading on Tx path.
 * @ndev:	Pointer to net_device structure
 * @flag:	unsigned long, used to enable/disable checksum offloading.
 *
 * This implements ethtool command for setting the Axi Ethernet checksum
 * offload on Tx path.
 * Issue "ethtool -K ethX tx on|off" under linux prompt to execute this
 * function.
 **/
static int axienet_stub_ethtools_set_tx_csum(struct net_device *ndev, u32 flag)
{
	struct axienet_local *lp = netdev_priv(ndev);

	if (lp->csum_offload_on_tx_path & XAE_FEATURE_PARTIAL_TX_CSUM) {
		if (flag)
			ndev->features |= NETIF_F_IP_CSUM;
		else
			ndev->features &= ~NETIF_F_IP_CSUM;
	} else if (lp->csum_offload_on_tx_path & XAE_FEATURE_FULL_TX_CSUM) {
		if (flag)
			ndev->features |= NETIF_F_HW_CSUM;
		else
			ndev->features &= ~NETIF_F_HW_CSUM;
	}

	return 0;
}

/**
 * axienet_stub_ethtools_get_pauseparam - Get the pause parameter setting for
 *				     Tx and Rx paths.
 * @ndev:	Pointer to net_device structure
 * @epauseparm:	Pointer to ethtool_pauseparam structure.
 *
 * This implements ethtool command for getting axi ethernet pause frame
 * setting.
 * Issue "ethtool -a ethX" to execute this function.
 **/
static void axienet_stub_ethtools_get_pauseparam(struct net_device *ndev,
		struct ethtool_pauseparam *epauseparm)
{
	struct axienet_local *lp = netdev_priv(ndev);
	u32 regval;

	epauseparm->autoneg  = 0;

	regval = axienet_ior(lp, XAE_FCC_OFFSET);
	epauseparm->tx_pause = regval & XAE_FCC_FCTX_MASK;
	epauseparm->rx_pause = regval & XAE_FCC_FCRX_MASK;
}

/**
 * axienet_stub_ethtools_set_pauseparam - Set device pause parameter(flow control)
 * 				     settings.
 * @ndev:	Pointer to net_device structure
 * @epauseparam:Pointer to ethtool_pauseparam structure
 *
 * This implements ethtool command for enabling flow control on Rx and Tx
 * paths.
 * Issue "ethtool -A ethX tx on|off" under linux prompt to execute this
 * function.
 **/
static int axienet_stub_ethtools_set_pauseparam(struct net_device *ndev,
		struct ethtool_pauseparam *epauseparm)
{
	struct axienet_local *lp = netdev_priv(ndev);
	u32 regval = 0;

	if (netif_running(ndev)) {
		printk(KERN_ERR
			"%s: Please stop netif before applying configruation\n",
			ndev->name);
		return -EFAULT;
	}

	regval = axienet_ior(lp, XAE_FCC_OFFSET);

	if (epauseparm->tx_pause)
		regval |= XAE_FCC_FCTX_MASK;
	else
		regval &= ~XAE_FCC_FCTX_MASK;

	if (epauseparm->rx_pause)
		regval |= XAE_FCC_FCRX_MASK;
	else
		regval &= ~XAE_FCC_FCRX_MASK;

	axienet_iow(lp, XAE_FCC_OFFSET, regval);
	return 0;
}

/**
 * axienet_stub_ethtools_get_coalesce - Get DMA interrupt coalescing count.
 * @ndev:	Pointer to net_device structure
 * @ecoalesce:	Pointer to ethtool_coalesce structure
 *
 * This implements ethtool command for getting the DMA interrupt coalescing
 * count on Tx and Rx paths.
 * Issue "ethtool -c ethX" under linux prompt to execute this function.
 **/
static int axienet_stub_ethtools_get_coalesce(struct net_device *ndev,
					struct ethtool_coalesce *ecoalesce)
{
	return 0;
}

/**
 * axienet_stub_ethtools_set_coalesce - Set DMA interrupt coalescing count.
 * @ndev:	Pointer to net_device structure
 * @ecoalesce:	Pointer to ethtool_coalesce structure
 *
 * This implements ethtool command for setting the DMA interrupt coalescing
 * count on Tx and Rx paths.
 * Issue "ethtool -C ethX rx-frames 5" under linux prompt to execute this
 * function.
 **/
static int axienet_stub_ethtools_set_coalesce(struct net_device *ndev,
					struct ethtool_coalesce *ecoalesce)
{
	return 0;
}

static struct ethtool_ops axienet_stub_ethtool_ops = {
	.get_settings   = axienet_stub_ethtools_get_settings,
	.set_settings   = axienet_stub_ethtools_set_settings,
	.get_drvinfo    = axienet_stub_ethtools_get_drvinfo,
	.get_regs_len   = axienet_stub_ethtools_get_regs_len,
	.get_regs       = axienet_stub_ethtools_get_regs,
	.get_link       = ethtool_op_get_link,       /* ethtool default */
	.get_rx_csum    = axienet_stub_ethtools_get_rx_csum,
	.set_rx_csum    = axienet_stub_ethtools_set_rx_csum,
	.get_tx_csum    = axienet_stub_ethtools_get_tx_csum,
	.set_tx_csum    = axienet_stub_ethtools_set_tx_csum,
	.get_sg         = ethtool_op_get_sg,         /* ethtool default */
	.get_pauseparam = axienet_stub_ethtools_get_pauseparam,
	.set_pauseparam = axienet_stub_ethtools_set_pauseparam,
	.get_coalesce   = axienet_stub_ethtools_get_coalesce,
	.set_coalesce   = axienet_stub_ethtools_set_coalesce,
};

/**
 * axienet_stub_of_probe - Axi Ethernet probe function.
 * @op:		Pointer to platform device structure.
 * @match:	Pointer to device id structure
 *
 * returns: 0, on success
 *	    Non-zero error value on failure.
 *
 * This is the probe routine for Axi Ethernet driver. This is called before
 * any other driver routines are invoked. It allocates and sets up the Ethernet
 * device. Parses through device tree and populates fields of
 * axienet_local. It registers the Ethernet device.
 **/
int axienet_stub_of_probe(struct platform_device *op)
{
	struct device_node *np;
	struct axienet_local *lp;
	struct net_device *ndev;
	const void *addr;

	__be32 *p;
	int size, rc = 0;
    
    long end = jiffies + 2;

    
	/* Init network device structure */
	ndev = alloc_etherdev(sizeof(*lp));
	if (!ndev) {
		dev_err(&op->dev, "could not allocate device.\n");
		return -ENOMEM;
	}
	ether_setup(ndev);
	dev_set_drvdata(&op->dev, ndev);
	SET_NETDEV_DEV(ndev, &op->dev);
	ndev->flags &= ~IFF_MULTICAST;  /* clear multicast */ 
	ndev->flags |= IFF_PROMISC;  /* enable promiscuous mode */  
	ndev->features = NETIF_F_SG | NETIF_F_FRAGLIST;
	ndev->netdev_ops = &axienet_stub_netdev_ops;
	ndev->ethtool_ops = &axienet_stub_ethtool_ops;

	/* Setup Axi Ethernet private info structure */
	lp = netdev_priv(ndev);
	lp->ndev = ndev;
	lp->dev = &op->dev;
	lp->options = XAE_OPTION_DEFAULTS;

	/* Map device registers */
	lp->regs = of_iomap(op->dev.of_node, 0);
	if (!lp->regs) {
		dev_err(&op->dev, "could not map Axi Ethernet regs.\n");
		goto nodev;
	}

	/* Setup checksum offload, but default to off if not specified */
	lp->features = 0;

	p = (__be32 *)of_get_property(op->dev.of_node, "xlnx,txcsum", NULL);
	if (p) {
		switch (be32_to_cpup(p)) {
		case 1:	lp->csum_offload_on_tx_path =
					XAE_FEATURE_PARTIAL_TX_CSUM;
			lp->features |= XAE_FEATURE_PARTIAL_TX_CSUM;
			/* Can checksum TCP/UDP over IPv4. */
			ndev->features |= NETIF_F_IP_CSUM;
			break;
		case 2:	lp->csum_offload_on_tx_path =
					XAE_FEATURE_FULL_TX_CSUM;
			lp->features |= XAE_FEATURE_FULL_TX_CSUM;
			/* Can checksum TCP/UDP over IPv4. */
			ndev->features |= NETIF_F_IP_CSUM;
			break;
		default:
			lp->csum_offload_on_tx_path = XAE_NO_CSUM_OFFLOAD;
		}
	}

	p = (__be32 *)of_get_property(op->dev.of_node, "xlnx,rxcsum", NULL);
	if (p) {
		switch (be32_to_cpup(p)) {
		case 1: lp->csum_offload_on_rx_path =
					XAE_FEATURE_PARTIAL_RX_CSUM;
			lp->features |= XAE_FEATURE_PARTIAL_RX_CSUM;
			break;
		case 2:	lp->csum_offload_on_rx_path =
					XAE_FEATURE_FULL_RX_CSUM;
			lp->features |= XAE_FEATURE_FULL_RX_CSUM;
			break;
		default:
			lp->csum_offload_on_rx_path = XAE_NO_CSUM_OFFLOAD;
		}
	}

	/* For supporting jumbo frames, the Axi Ethernet hardware must have
	 * a larger Rx/Tx Memory. Typically, the size must be more than or
	 * equal to 16384 bytes, so that we can enable jumbo option and start
	 * supporting jumbo frames. Here we check for memory allocated for
	 * Rx/Tx in the hardware from the device-tree and accordingly set
	 * flags. */
	p = (__be32 *)of_get_property(op->dev.of_node, "xlnx,rxmem", NULL);
	if (p) {
		if ((be32_to_cpup(p)) >= 0x4000)
			lp->jumbo_support = 1;
	}

	p = (__be32 *)of_get_property(op->dev.of_node, "xlnx,temac-type",
									NULL);
	if (p)
		lp->temac_type = be32_to_cpup(p);

	p = (__be32 *)of_get_property(op->dev.of_node, "xlnx,phy-type", NULL);
	if (p)
		lp->phy_type = be32_to_cpup(p);

	lp->phy_node = of_parse_phandle(op->dev.of_node, "phy-handle", 0);
	rc = axienet_mdio_setup(lp, op->dev.of_node);
	if (rc)
		dev_warn(&op->dev, "error registering MDIO bus\n");

	rc = register_netdev(lp->ndev);
	if (rc) {
		dev_err(lp->dev, "register_netdev() error (%i)\n", rc);
		goto err_iounmap;
	}

    // Automaticly enable MDIO and conect phyter
    /* Enable the MDIO */
    /* Wait till MDIO interface is ready.*/
    end = jiffies + 2;
    while (!(axienet_ior(lp, XAE_MDIO_MCR_OFFSET) &
                        XAE_MDIO_MCR_READY_MASK)) {
        if (end - jiffies <= 0) {
            WARN_ON(1);
            return -ETIMEDOUT;
        }
        msleep(1);
    }

    if (lp->phy_node) {
        lp->phy_dev = of_phy_connect(lp->ndev,
                        lp->phy_node,
                        axienet_stub_adjust_link,
                        0,
                        PHY_INTERFACE_MODE_GMII);
        if (!lp->phy_dev) {
            dev_err(lp->dev, "Early of_phy_connect() failed. Use 'ifconfig ifc up' to enable PHY manamegent.\n");
            goto err_early;
        }
        phy_start(lp->phy_dev);
    }
    // Set interface up and no ARP is available
    ndev->flags |= IFF_UP | IFF_NOARP; 
    set_bit(__LINK_STATE_START, &(ndev->state));
    
err_early:
	return 0;

err_iounmap:
	iounmap(lp->regs);
nodev:
	free_netdev(ndev);
	ndev = NULL;
	return rc;
}

/**
 * axienet_stub_of_remove() - The device driver remove function.
 * @op:		Pointer to the device structure.
 *
 * returns: 0 for success and error value on failure
 *
 **/
// static int __devexit axienet_stub_of_remove(struct platform_device *op)
// {
// 	struct net_device *ndev = dev_get_drvdata(&op->dev);
// 	struct axienet_local *lp = netdev_priv(ndev);
// 
// 	axienet_mdio_teardown(lp);
// 	unregister_netdev(ndev);
// 
// 	if (lp->phy_node)
// 		of_node_put(lp->phy_node);
// 	lp->phy_node = NULL;
// 	dev_set_drvdata(&op->dev, NULL);
// 	iounmap(lp->regs);
// 	free_netdev(ndev);
// 	return 0;
// }

// static struct platform_driver axienet_stub_of_driver = {
// 	.probe = axienet_stub_of_probe,
// 	.remove = __devexit_p(axienet_stub_of_remove),
// 	.driver = {
// 		 .owner = THIS_MODULE,
// 		 .name = "xilinx_axienet",
// 		 .of_match_table = axienet_stub_of_match,
// 	},
// };
// 
// /**
//  * axienet_stub_init() - The device driver initialization function.
//  *
//  * returns: 0 for success and error value on failure
//  *
//  **/
// static int __init axienet_stub_init(void)
// {
// 	return platform_driver_register(&axienet_stub_of_driver);
// }
// module_init(axienet_stub_init);
// 
// /**
//  * axienet_stub_exit() - The device driver exit function.
//  *
//  * returns: 0 for success and error value on failure
//  *
//  **/
// static void __exit axienet_stub_exit(void)
// {
// 	platform_driver_unregister(&axienet_stub_of_driver);
// }
// module_exit(axienet_stub_exit);

// MODULE_DESCRIPTION("Xilinx Axi Ethernet STUB driver");
// MODULE_AUTHOR("Xilinx/Sec6Net");
// MODULE_LICENSE("GPL");
