/*
 * Sec6net XILINX AXI DMA driver. Source file.
 */

/* Used kernel headers */
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/of_platform.h>
#include <linux/of_address.h>
#include <linux/spinlock.h>
#include <linux/sched.h>
#include <linux/dma-mapping.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <asm/uaccess.h>
#include <linux/kernel.h>
#include <linux/cdev.h>
#include <linux/ioctl.h>
#include <linux/of_irq.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,7,0)
#include <linux/hashtable.h>
#else
#include "hashtable.h"
#endif
/* Driver headers */
#include "axi_dma.h"

/* Match table for of_platform binding */
static const struct of_device_id  sec6net_dma_of_match[] = {
    { .compatible = "sec6net,axi-dma-1.00.a", },
    {},
};

static int dma_cnt = 0;

MODULE_DEVICE_TABLE(of, sec6net_dma_of_match);

static DEFINE_HASHTABLE(hash_allocation, ALLOC_HASH_BITS);

/**
 * dma_in - Memory mapped Axi DMA register read
 * @dev:        Driver info structure
 * @reg:    Address offset from the base address of the Axi DMA core
 *
 * returns: The contents of the Axi DMA register
 *
 * This function returns the contents of the corresponding Axi DMA register.
 */
 inline u32 dma_in(struct driver_info * dev, off_t reg)
{
    return ioread32(dev->dma_regs + reg);
}

/**
 * dma_out - Memory mapped Axi DMA register write.
 * @dev:        Driver info structure
 * @reg:    Address offset from the base address of the Axi DMA core
 * @value:  Value to be written into the Axi DMA register
 *
 * This function writes the desired value into the corresponding Axi DMA
 * register.
 */
 inline void dma_out(struct driver_info * dev, off_t reg, u32 value)
{
    iowrite32(value, (dev->dma_regs + reg));
}

/**
 * free_rings - Free alocated descriptor rings and buffers
 * @dev:        Driver info structure
 * 
 * This function frees alocated buffers and descriptor rings.
 */
 void free_rings(struct driver_info * dev)
{
    // Free RX descriptor ring
    if (dev->rx_ring_ptr_v) {
        dma_free_coherent(&dev->of_dev->dev, POINTER_SIZE, dev->rx_ring_ptr_v,
                          dev->rx_ring_ptr_p);
    }
    // Free TX descriptor ring
    if (dev->tx_ring_ptr_v) {
        dma_free_coherent(&dev->of_dev->dev, POINTER_SIZE, dev->tx_ring_ptr_v,
                          dev->tx_ring_ptr_p);
    }
    // Free RX buffers
    if (dev->rx_ring_v) {
//         dma_unmap_single(&dev->of_dev->dev, dev->rx_ring_p, MAX_RING, 
//                          DMA_FROM_DEVICE);

        dma_free_coherent(&dev->of_dev->dev, MAX_RING, dev->rx_ring_v,
                          dev->rx_ring_p);
    }
    // Free TX buffers
    if (dev->tx_ring_v) {
//         dma_unmap_single(&dev->of_dev->dev, dev->tx_ring_p, MAX_RING, 
//                          DMA_FROM_DEVICE);

        dma_free_coherent(&dev->of_dev->dev, MAX_RING, dev->tx_ring_v,
                          dev->tx_ring_p);
    }
}

/**
 * alocate_rings - Alocate descriptor rings and buffers
 * @dev:        Driver info structure
 * 
 * returns: Staus of operation. 0 if all is OK, -ENOMEM othewise.
 *
 * This function alocates buffers and descriptor rings.
 */
 int alocate_rings(struct driver_info * dev)
{    
    // Alocate RX descriptor ring
    dev->rx_ring_ptr_v = dma_alloc_coherent(&dev->of_dev->dev, POINTER_SIZE, &dev->rx_ring_ptr_p,
                                   GFP_KERNEL | __GFP_ZERO);
    if (!dev->rx_ring_ptr_v)
        goto fail;
    
    // Alocate TX descriptor ring
    dev->tx_ring_ptr_v = dma_alloc_coherent(&dev->of_dev->dev, POINTER_SIZE, &dev->tx_ring_ptr_p,
                                   GFP_KERNEL | __GFP_ZERO);
    if (!dev->tx_ring_ptr_v)
        goto fail;
    
    // Alocate RX buffers
    dev->rx_ring_v = dma_alloc_coherent(&dev->of_dev->dev, MAX_RING, &dev->rx_ring_p,
                                   GFP_KERNEL | __GFP_ZERO);
    if (!dev->rx_ring_v)
        goto fail;
        
    // Alocate TX buffers
    dev->tx_ring_v = dma_alloc_coherent(&dev->of_dev->dev, MAX_RING, &dev->tx_ring_p,
                                   GFP_KERNEL | __GFP_ZERO);
    if (!dev->tx_ring_v)
        goto fail;
        
    return 0;                               
fail:
    free_rings(dev);
    return -ENOMEM;
}

/**
 * init_rings - Init descriptor rings 
 * @dev:        Driver info structure
 * @dir:        Specifies which ring should be initialized. Use DIR_RX for 
 *              RX ring and DIR_TX for TX ring.
 *
 * This function initialize descriptor rings.
 */
 void init_rings(struct driver_info * dev, u32 dir)
{
    int i;
    u32 ptr_len;
    
    // Init RX ring
    if (dir & DIR_RX)
    {
        // Compute number of used descriptors based on total allocated buffer 
        // size and bufer length
        ptr_len = dev->global_settings.rx_desc_len;
        // Init each descriptor
        for (i = 0; i < ptr_len; i++) {
            // Set physical address to next descriptor
            dev->rx_ring_ptr_v[i].next = dev->rx_ring_ptr_p + 
                                         sizeof(struct dma_ring) * ((i + 1) % ptr_len);
            // Set physical address of buffer
            dev->rx_ring_ptr_v[i].buffer = dev->rx_ring_p + i * dev->global_settings.rx_buffer_len;
            // Set size of buffer
            dev->rx_ring_ptr_v[i].control = dev->global_settings.rx_buffer_len;
            // Clear status
            dev->rx_ring_ptr_v[i].status = 0;
//             dev_warn(&dev->of_dev->dev, "RX descriptor %i: next: %x buffer: %x control: %x status: %x!\n", i, dev->rx_ring_ptr_v[i].next, dev->rx_ring_ptr_v[i].buffer, dev->rx_ring_ptr_v[i].control, dev->rx_ring_ptr_v[i].status);
        }
    }
    // Init TX ring
    if (dir & DIR_TX)
    {
        // Compute number of used descriptors based on total allocated buffer 
        // size and bufer length
        ptr_len = dev->global_settings.tx_desc_len;
        // Init each descriptor
        for (i = 0; i < ptr_len; i++) {
            // Set physical address to next descriptor
            dev->tx_ring_ptr_v[i].next = dev->tx_ring_ptr_p + 
                                         sizeof(struct dma_ring) * ((i + 1) % ptr_len);
            // Set physical address of buffer
            dev->tx_ring_ptr_v[i].buffer = dev->tx_ring_p + i * dev->global_settings.tx_buffer_len;
            // Set size of buffer
            dev->tx_ring_ptr_v[i].control = DESC_SOF_EOF | dev->global_settings.tx_buffer_len;
            // Clear status
            dev->tx_ring_ptr_v[i].status = 0; 
        }
    }
}

/**
 * init_ptrs - Init start pointers and used sizes of rings
 * @dev:       Driver info structure
 * @dir:       Specifies which ring should be initialized. Use DIR_RX for 
 *             RX ring and DIR_TX for TX ring.
 *
 * This function initialize start pointers and used sizes of rings.
 */
 void init_ptrs(struct driver_info * dev, u32 dir)
{    
    // Init RX
    if (dir & DIR_RX)
    {
        // Ring start at item 0
        dev->rx_ring_start = 0;
        // Ring is empty
        dev->rx_ring_len = 0;
    }
    // Init TX
    if (dir & DIR_TX)
    {
        // Ring start at item 0
        dev->tx_ring_start = 0;
        // Ring is empty
        dev->tx_ring_len = 0;
    }
}

/**
 * start_rx - Start RX DMA channel
 * @dev:      Driver info structure
 *
 * This function initialize and start  RX DMA channel.
 */
 void start_rx(struct driver_info * dev)
{
    u32 ret;
    u32 ptr_len;
    
    // Init descriptor ring
    init_rings(dev, DIR_RX);
    // Init pointers
    init_ptrs(dev, DIR_RX);
    
    // TODO: Clean this after debug
    /* Start updating the Rx channel control register */
//     ret = dma_in(dev, S2MM_DMACR_OFFSET);
//     ret |= ~STARTSTOP_MASK;
     /* Write to the Rx channel control register */
//     dma_out(dev, S2MM_DMACR_OFFSET, ret);
//     ret = dma_in(dev, S2MM_DMACR_OFFSET);
    /* Update the interrupt coalesce count */
//     ret = ((ret & ~THRES_MASK) | ((dev->global_settings.rx_act_timeout) << 16));
    /* Update the delay timer count */
//     ret = ((ret & ~DELAY_MASK) | ((dev->global_settings.rx_inact_timeout) << 24));
     /* Write to the Rx channel control register */
//     dma_out(dev, S2MM_DMACR_OFFSET, ret);
    dev_warn(&dev->of_dev->dev, "RX status: %x!\n", dma_in(dev, S2MM_DMASR_OFFSET));
    dma_out(dev, S2MM_DMASR_OFFSET, ALL_MASK);
    
    /* Populate the tail pointer and bring the Rx Axi DMA engine out of
     * halted state. This will make the Rx side ready for reception.*/
    dma_out(dev, S2MM_CURDESC_OFFSET, dev->rx_ring_ptr_p);
    dev_warn(&dev->of_dev->dev, "Real PHYS descriptor: %x!\n", dev->rx_ring_ptr_p);
    dev_warn(&dev->of_dev->dev, "RX CURR descriptor: %x!\n", dma_in(dev, S2MM_CURDESC_OFFSET));
    ret = dma_in(dev, S2MM_DMACR_OFFSET);
    /* Update the interrupt coalesce count */
    ret = ((ret & ~THRES_MASK) | ((dev->global_settings.rx_act_timeout) << 16));
    /* Update the delay timer count */
    ret = ((ret & ~DELAY_MASK) | ((dev->global_settings.rx_inact_timeout) << 24));
    /* Write to the Rx channel control register */
    dma_out(dev, S2MM_DMACR_OFFSET, ret);
    ret = dma_in(dev, S2MM_DMACR_OFFSET) | STARTSTOP_MASK;
    dma_out(dev, S2MM_DMACR_OFFSET, ret);
    ret = dma_in(dev, S2MM_DMACR_OFFSET) | ALL_MASK;
    dma_out(dev, S2MM_DMACR_OFFSET, ret);
    dev_warn(&dev->of_dev->dev, "RX status: %x!\n", dma_in(dev, S2MM_DMASR_OFFSET));
    wmb();
//     udelay(1);
    // If direct mode is used do not start the ring
    if (dev->global_settings.rx_mode != RX_DIRECT)
    {
        ptr_len = dev->global_settings.rx_desc_len;
        dma_out(dev, S2MM_TAILDESC_OFFSET, dev->rx_ring_ptr_p + (sizeof(struct dma_ring) * (ptr_len - 1)));
        wmb();
    }
    dev_warn(&dev->of_dev->dev, "RX status: %x!\n", dma_in(dev, S2MM_DMASR_OFFSET));
}

/**
 * start_tx - Start TX DMA channel
 * @dev:      Driver info structure
 *
 * This function initialize and start TX DMA channel.
 */
 void start_tx(struct driver_info * dev)
{
    u32 ret;
    
    // Init descriptor ring
    init_rings(dev, DIR_TX);
    // Init pointers
    init_ptrs(dev, DIR_TX);
    
    // TODO: Clean it after debug
    /* Start updating the Tx channel control register */
//     ret = dma_in(dev, MM2S_DMACR_OFFSET);
    /* Update the interrupt coalesce count */
//     ret = ((ret & ~THRES_MASK) | ((dev->global_settings.tx_act_timeout) << 16));
    /* Update the delay timer count */
//     ret = ((ret & ~DELAY_MASK) | ((dev->global_settings.tx_inact_timeout) << 24));
//     ret |= ~STARTSTOP_MASK;
//     dma_out(dev, MM2S_DMACR_OFFSET, ret);
//     dev_warn(&dev->of_dev->dev, "RX status: %x!\n", dma_in(dev, S2MM_DMASR_OFFSET));
//     dev_warn(&dev->of_dev->dev, "TX status: %x!\n", dma_in(dev, MM2S_DMASR_OFFSET));
    /* Enable coalesce, delay timer and error interrupts */
    dma_out(dev, MM2S_DMASR_OFFSET, ALL_MASK);
    /* Start updating the Tx channel control register */
    ret = dma_in(dev, MM2S_DMACR_OFFSET);
    /* Update the interrupt coalesce count */
    ret = ((ret & ~THRES_MASK) | ((dev->global_settings.tx_act_timeout) << 16));
    /* Update the delay timer count */
    ret = ((ret & ~DELAY_MASK) | ((dev->global_settings.tx_inact_timeout) << 24));
    dma_out(dev, MM2S_DMACR_OFFSET, ret);
    /* Populate the start pointer and bring the Tx Axi DMA engine out of
     * halted state. This will make the Tx side ready for reception.*/
    dma_out(dev, MM2S_CURDESC_OFFSET, dev->tx_ring_ptr_p);
    ret = dma_in(dev, MM2S_DMACR_OFFSET) | STARTSTOP_MASK;
    dma_out(dev, MM2S_DMACR_OFFSET, ret);
    
    ret = dma_in(dev, MM2S_DMACR_OFFSET);
    ret |= ALL_MASK;
    /* Write to the Tx channel control register */
    dma_out(dev, MM2S_DMACR_OFFSET, ret);
    
    wmb();
}

/**
 * stop_rx  - Stop RX DMA channel
 * @dev:      Driver info structure
 *
 * This function stops RX DMA channel.
 */
 void stop_rx(struct driver_info * dev)
{
    u32 ret;
    
    // Set run/stop bit of RX DMA channel control register to STOP
    ret = dma_in(dev, S2MM_DMACR_OFFSET);
    dma_out(dev, S2MM_DMACR_OFFSET, ret & (~STARTSTOP_MASK));
}

/**
 * stop_tx  - Stop TX DMA channel
 * @dev:      Driver info structure
 *
 * This function stops TX DMA channel.
 */
 void stop_tx(struct driver_info * dev)
{
    u32 ret;
    
    // Set run/stop bit of RX DMA channel control register to STOP
    ret = dma_in(dev, MM2S_DMACR_OFFSET);
    dma_out(dev, MM2S_DMACR_OFFSET, ret & (~STARTSTOP_MASK));
}

/**
 * reset_rx  - Reset RX DMA channel
 * @dev:       Driver info structure
 *
 * This function resets RX DMA channel. Actually both RX and TX channels are 
 * reseted.
 */
 void reset_rx(struct driver_info * dev)
{
    u32 timeout;
    /* Reset Axi DMA. The reset
     * process of Axi DMA takes a while to complete as all pending
     * commands/transfers will be flushed or completed during this
     * reset process. */
    dma_out(dev, S2MM_DMACR_OFFSET, RESET_MASK);
    timeout = DELAY_OF_ONE_MILLISEC;
    while (dma_in(dev, S2MM_DMACR_OFFSET) & RESET_MASK) {
        udelay(1);
        if (--timeout == 0) {
            dev_err(&dev->of_dev->dev, "RX DMA reset timeout!\n");
            break;
        }
    }
}

/**
 * reset_tx  - Reset TX DMA channel
 * @dev:       Driver info structure
 *
 * This function resets TX DMA channel. Actually both RX and TX channels are 
 * reseted.
 */
 void reset_tx(struct driver_info * dev)
{
    u32 timeout;
    /* Reset Axi DMA. The reset
     * process of Axi DMA takes a while to complete as all pending
     * commands/transfers will be flushed or completed during this
     * reset process. */
    dma_out(dev, MM2S_DMACR_OFFSET, RESET_MASK);
    timeout = DELAY_OF_ONE_MILLISEC;
    while (dma_in(dev, MM2S_DMACR_OFFSET) & RESET_MASK) {
        udelay(1);
        if (--timeout == 0) {
            dev_err(&dev->of_dev->dev, "TX DMA reset timeout!\n");
            break;
        }
    }
}

/**
 * dma_irqhandler_tx  - Interrupt handler for TX channel.
 * @irq:       Interrupt number
 * @data:      Driver info structure
 *
 * returns:    Retirns IRQ_HANDLED.
 *
 * This function handles TX channel interrupts.
 */
static irqreturn_t dma_irqhandler_tx(int irq, void* data){
    u32 state;
//     u32 desc_addr[2];
    u32 count;
    u32 start;
    struct driver_info *dev;
    struct dma_ring * desc;
    unsigned int status;
    unsigned long flags;
    
    u32 new_start;
    u32 new_tail;
    u32 phys;

    
    // Get driver info structure from parameter data
    dev = (struct driver_info *)data;
//     spin_lock_irqsave(&dev->write_lock, flags);
    
    // Acknowledge interrupt
    state = dma_in(dev, MM2S_DMASR_OFFSET);
    dma_out(dev, MM2S_DMASR_OFFSET, state);
    if (unlikely(state & ERROR_MASK))
    {
        dev_err(&dev->of_dev->dev,"*******************************************************************");
        dev_err(&dev->of_dev->dev,"* TX Channel raised Error interrupt state: %x                     *", state);
        dev_err(&dev->of_dev->dev,"*******************************************************************");
        goto fin;
    }
//     desc_addr[0] = dma_in(dev, MM2S_CURDESC_OFFSET);
//     desc_addr[1] = dma_in(dev, MM2S_TAILDESC_OFFSET);
//     dev_warn(&dev->of_dev->dev, "TX IRQ: state: %x desc[0]: %x desc[1]: %x tx_perf: %d.\n", state, desc_addr[0], desc_addr[1], dev->tx_perf);
//     dev_warn(&dev->of_dev->dev, "TX Config: %x!\n", dma_in(dev, MM2S_DMACR_OFFSET));
    
    // Select interrupt mode
    if (unlikely(dev->tx_perf == 1))
    {
        // Performance TX operation
        // Get index of first unprocessed item in TX ring, Must be locked.
        spin_lock_irqsave(&dev->tx_lock, flags);
            start = dev->tx_ring_start;
        spin_unlock_irqrestore(&dev->tx_lock, flags);
        
        // Init counter of transfered descriptors
        count = 0;
        // Get first descriptor
        desc = &dev->tx_ring_ptr_v[(start + count) % (dev->global_settings.tx_desc_len)];
        phys = dev->tx_ring_ptr_p + start * sizeof(struct dma_ring);
        
        // Get status of descriptor
        status = desc->status;
        
        // Iterate over all completed descriptors
        while (status & DSC_COMPLETE)
        {
            // Clear the descriptor
            desc->status = 0;
          
            // Update count of processed descriptors
            count++;
            // Fetch next descriptor
            desc = &dev->tx_ring_ptr_v[(start + count) % (dev->global_settings.tx_desc_len)];
            // Get status of descriptor
            status = desc->status;
            
//             if ((dev->tx_finished - 1) % 1000 == 0)
//                 dev_warn(&dev->of_dev->dev, "TX CNT: %d!\n", dev->tx_finished - 1);
//             dev_warn(&dev->of_dev->dev, "TX CNT: %x!\n", dev->tx_finished - 1);
            if (--dev->tx_finished == 0)
            {
                // Stop DMA
                stop_tx(dev);
                // Wake TX wait queue
//                 spin_unlock_irqrestore(&dev->write_lock, flags);
                wake_up_interruptible(&dev->tx_perf_queue);
//                 spin_lock_irqsave(&dev->write_lock, flags);
            }
        }
        // Update start of first river info structure item in TX ring and 
        // number of unprocessed descriptors. Must be locked.
        new_start = (dev->tx_ring_start + count) % (dev->global_settings.tx_desc_len);
        new_tail = dev->tx_ring_ptr_p + ((start + count - 1) % (dev->global_settings.tx_desc_len)) * sizeof(struct dma_ring);
        wmb();
        spin_lock_irqsave(&dev->tx_lock, flags);
            dev->tx_ring_start = new_start;
            dma_out(dev, MM2S_TAILDESC_OFFSET, new_tail);
        spin_unlock_irqrestore(&dev->tx_lock, flags);
//          spin_unlock_irqrestore(&dev->write_lock, flags);
//         dev_warn(&dev->of_dev->dev, "New TX start: %d New TAIL: %x Current: %x\n", new_start, new_tail, dma_in(dev, MM2S_CURDESC_OFFSET));
//         dev_warn(&dev->of_dev->dev, "TX Config: %x Staus: %x!\n", dma_in(dev, MM2S_DMACR_OFFSET), dma_in(dev, MM2S_DMASR_OFFSET));
//         if (dma_in(dev, MM2S_DMASR_OFFSET) & 0x00000f00)
//         {
//             for (i = 0; i < dev->global_settings.tx_desc_len; i++)
//             {
//                 dev_warn(&dev->of_dev->dev, "TX Desc %d: Next: %x, Buff: %x, Ctrl: %x, Stat: %x\nApp:%x %x %x %x %x\n", i,dev->tx_ring_ptr_v[i].next, dev->tx_ring_ptr_v[i].buffer, dev->tx_ring_ptr_v[i].control, dev->tx_ring_ptr_v[i].status, dev->tx_ring_ptr_v[i].application[0], dev->tx_ring_ptr_v[i].application[1], dev->tx_ring_ptr_v[i].application[2], dev->tx_ring_ptr_v[i].application[3], dev->tx_ring_ptr_v[i].application[4]);
//             }
//         }
        
    }
    else
    {
        // Normal TX operation
        // Get index of first unprocessed item in TX ring, Must be locked.
        spin_lock_irqsave(&dev->tx_lock, flags);
            start = dev->tx_ring_start;
        spin_unlock_irqrestore(&dev->tx_lock, flags);
        
        // Init counter of transfered descriptors
        count = 0;
        // Get first descriptor
        desc = &dev->tx_ring_ptr_v[(start + count) % (dev->global_settings.tx_desc_len)];
        // Get status of descriptor
        status = desc->status;
        // Iterate over all completed descriptors
        while (status & DSC_COMPLETE)
        {
//             if ((desc->status & LEN_MASK) != 1024)
//             {
//                 dev_warn(&dev->of_dev->dev, "TX IRQ Desc len mismatch %d\n", desc->status & LEN_MASK);
//                 dev_warn(&dev->of_dev->dev, "TX IRQ Desc %d: Next: %x, Buff: %x, Ctrl: %x, Stat: %x\nApp:%x %x %x %x %x\n", count, desc->next, desc->buffer, desc->control, desc->status, desc->application[0], desc->application[1], desc->application[2], desc->application[3], desc->application[4]);
//             }
            // Clear the descriptor
            desc->status = 0;
            desc->application[0] = 0;
            desc->application[1] = 0;
            desc->application[2] = 0;
            desc->application[3] = 0;
            desc->application[4] = 0;
            // Clear the buffer mark for sync and synchronise the caches for CPU
            if (desc->res1[2] != 0)
            {
                // Synchronise for device
                dma_sync_single_for_cpu(&dev->of_dev->dev, desc->buffer, desc->res1[2], DMA_BIDIRECTIONAL);
                // Mark the buffer for sync uppon reception
                desc->res1[2] = 0;
            }
//             printk("TX IRQ: Direct count: %d\n", count);
            if (desc->res1[0] != 0)
            {
//                 printk("TX IRQ: aio_complete: %x %d\n", desc->res1[1], desc->res1[0]);
                aio_complete((struct kiocb*)desc->res1[1], desc->res1[0], 0);
                desc->res1[0] = 0;
                desc->res1[1] = 0;
            }
//             dev_warn(&dev->of_dev->dev, "TX IRQ Desc %d: Next: %x, Buff: %x, Ctrl: %x, Stat: %x\nApp:%x %x %x %x %x\nRes:%x%x\n", count, desc->next, desc->buffer, desc->control, desc->status, desc->application[0], desc->application[1], desc->application[2], desc->application[3], desc->application[4], desc->res1[0], desc->res1[1]);
            // Update count of processed descriptors
            count++;
            // Fetch next descriptor
            desc = &dev->tx_ring_ptr_v[(start + count) % (dev->global_settings.tx_desc_len)];
            // Get status of descriptor
            status = desc->status;
        }
        // Update start of first river info structure item in TX ring and 
        // number of unprocessed descriptors. Must be locked.
        spin_lock_irqsave(&dev->tx_lock, flags);
//             printk("TX IRQ Before: Start: %d Len: %d Count: %d TX_Desc_Len: %d\n", dev->tx_ring_start, dev->tx_ring_len, count, dev->global_settings.tx_desc_len);
            dev->tx_ring_start = (dev->tx_ring_start + count) % (dev->global_settings.tx_desc_len);
            dev->tx_ring_len -= count;
//             printk("TX IRQ After: Start: %d Len: %d\n", dev->tx_ring_start, dev->tx_ring_len);
        spin_unlock_irqrestore(&dev->tx_lock, flags);
//         spin_unlock_irqrestore(&dev->write_lock, flags);
        // Wake TX wait queue
        wake_up_interruptible(&dev->tx_perf_queue);
    }
fin:   
    // Irq is handled.
    return IRQ_HANDLED;
};

/**
 * dma_irqhandler_rx  - Interrupt handler for RX channel.
 * @irq:       Interrupt number
 * @data:      Driver info structure
 *
 * returns:    Retirns IRQ_HANDLED.
 *
 * This function handles RX channel interrupts.
 */
static irqreturn_t dma_irqhandler_rx(int irq, void* data){
    u32 state;
//     u32 desc_addr[2];
    u32 count;
    u32 start;
    u32 len;
    unsigned int status;
    unsigned long flags;
    struct driver_info *dev;
    struct dma_ring * desc;
    
    // Get driver info structure from parameter data
    dev = (struct driver_info *)data;
//     spin_lock_irqsave(&dev->read_lock, flags);
    // Acknowledge interrupt
    state = dma_in(dev, S2MM_DMASR_OFFSET);
    dma_out(dev, S2MM_DMASR_OFFSET, state & ALL_MASK);
    
    if (unlikely(state & ERROR_MASK))
    {
        dev_err(&dev->of_dev->dev,"*******************************************************************");
        dev_err(&dev->of_dev->dev,"* RX Channel raised Error interrupt state: %x                     *", state);
        dev_err(&dev->of_dev->dev,"*******************************************************************");
        goto fin;
    }
    
//     desc_addr[0] = dma_in(dev, S2MM_CURDESC_OFFSET);
//     desc_addr[1] = dma_in(dev, S2MM_TAILDESC_OFFSET);
//     dev_warn(&dev->of_dev->dev, "RX IRQ: state: %x desc[0]: %x desc[1]: %x rx_perf: %d.\n", state, desc_addr[0], desc_addr[1], dev->rx_perf);
//     dev_warn(&dev->of_dev->dev, "RX Config: %x!\n", dma_in(dev, S2MM_DMACR_OFFSET));
    
    // Select interrupt mode
    if (unlikely(dev->rx_perf == 1))
    {
        // RX performance mode
        // Get index of first item in RX ring and RX ring length,
        spin_lock_irqsave(&dev->rx_lock, flags);
            start = dev->rx_ring_start;
        spin_unlock_irqrestore(&dev->rx_lock, flags);
        
        // Init counter of transfered descriptors
        count = 0;
        // Get first descriptor
        desc = &dev->rx_ring_ptr_v[(start + count) % (dev->global_settings.rx_desc_len)];
        // Get status of descriptor
        status = desc->status;
        // Iterate over all completed descriptors
        while (status & DSC_COMPLETE)
        {
            // Update count of processed descriptors
            count++;
            // Clear status
            desc->status = 0;
            // Clear application fields
            desc->application[0] = 0;
            desc->application[1] = 0;
            desc->application[2] = 0;
            desc->application[3] = 0;
            desc->application[4] = 0;
            // Fetch next descriptor
            desc = &dev->rx_ring_ptr_v[(start + count) % (dev->global_settings.rx_desc_len)];
            // Get status of descriptor
            status = desc->status;
            
//             if ((dev->rx_finished - 1) % 1000 == 0)
//                 dev_warn(&dev->of_dev->dev, "RX CNT: %d!\n", dev->rx_finished - 1);
            
            //dev_warn(&dev->of_dev->dev, "RX CNT: %x!\n", dev->rx_finished - 1);
            if (--dev->rx_finished == 0)
            {
                // Stop DMA
                stop_rx(dev);
                // Wake RX wait queue
//                 spin_unlock_irqrestore(&dev->read_lock, flags);
                wake_up_interruptible(&dev->rx_perf_queue);
//                 spin_lock_irqsave(&dev->read_lock, flags);
            }
        }
        // Update size of RX ring. Must be locked.
        spin_lock_irqsave(&dev->rx_lock, flags);
            dev->rx_ring_start = (dev->rx_ring_start + count) % (dev->global_settings.rx_desc_len);
            dma_out(dev, S2MM_TAILDESC_OFFSET, dev->rx_ring_ptr_p + ((start + count - 1) % (dev->global_settings.rx_desc_len)) * sizeof(struct dma_ring));
        spin_unlock_irqrestore(&dev->rx_lock, flags);
//         spin_unlock_irqrestore(&dev->read_lock, flags);
//         dev_warn(&dev->of_dev->dev, "New RX start: %d New TAIL: %x Current: %x\n", dev->rx_ring_start, dev->rx_ring_ptr_p + ((start + count - 1) % (dev->global_settings.rx_desc_len)) * sizeof(struct dma_ring), dma_in(dev, S2MM_CURDESC_OFFSET));
//         dev_warn(&dev->of_dev->dev, "RX Config: %x! Status: %x\n", dma_in(dev, S2MM_DMACR_OFFSET), dma_in(dev, S2MM_DMASR_OFFSET));
//         if (dma_in(dev, S2MM_DMASR_OFFSET) & 0x00000f00)
//         {
//             for (i = 0; i < dev->global_settings.rx_desc_len; i++)
//             {
//                 dev_warn(&dev->of_dev->dev, "RX Desc %d: Next: %x, Buff: %x, Ctrl: %x, Stat: %x\nApp:%x %x %x %x %x\n", i, dev->rx_ring_ptr_v[i].next, dev->rx_ring_ptr_v[i].buffer, dev->rx_ring_ptr_v[i].control, dev->rx_ring_ptr_v[i].status, dev->rx_ring_ptr_v[i].application[0], dev->rx_ring_ptr_v[i].application[1], dev->rx_ring_ptr_v[i].application[2], dev->rx_ring_ptr_v[i].application[3], dev->rx_ring_ptr_v[i].application[4]);
//             }
//         }
    }
    else
    {
        // Normal RX operation
        // Get index of first item in RX ring and RX ring length, Must be locked.
        spin_lock_irqsave(&dev->rx_lock, flags);
            start = dev->rx_ring_start;
            len = dev->rx_ring_len;
        spin_unlock_irqrestore(&dev->rx_lock, flags);
        
        // Init counter of transfered descriptors
        count = 0;
        // If RX_DIRECT mode is used, rx_ring_len has oposite meening - number 
        // of buffers without data
        if (likely(dev->global_settings.rx_mode == RX_DIRECT))
        {
            len = 0;
        }
        // Get first descriptor
        desc = &dev->rx_ring_ptr_v[(start + len + count) % (dev->global_settings.rx_desc_len)];
        // Get status of descriptor
        status = desc->status;
        // Iterate over all completed descriptors
        while (status & DSC_COMPLETE)
        {
            // Update count of processed descriptors
            count++;
//             if ((desc->status & LEN_MASK) != 1024)
//             {
//                 dev_warn(&dev->of_dev->dev, "RX IRQ Desc len mismatch %d\n", desc->status & LEN_MASK);
//                 dev_warn(&dev->of_dev->dev, "RX IRQ Desc %d: Next: %x, Buff: %x, Ctrl: %x, Stat: %x\nApp:%x %x %x %x %x\n", count, desc->next, desc->buffer, desc->control, desc->status, desc->application[0], desc->application[1], desc->application[2], desc->application[3], desc->application[4]);
//             }
            if (likely(dev->global_settings.rx_mode == RX_DIRECT))
            {
//                 printk("RX IRQ: Direct count: %d\n", count);
                // Clear status
                desc->status = 0;
                // Clear application fields
                desc->application[0] = 0;
                desc->application[1] = 0;
                desc->application[2] = 0;
                desc->application[3] = 0;
                desc->application[4] = 0;
                // Clear the buffer mark for sync and synchronise the caches for CPU
                if (desc->res1[2] != 0)
                {
                    // Synchronise for device
                    dma_sync_single_for_cpu(&dev->of_dev->dev, desc->buffer, desc->res1[2], DMA_BIDIRECTIONAL);
                    // Mark the buffer for sync uppon reception
                    desc->res1[2] = 0;
                }
                if (desc->res1[0] != 0)
                {
//                     printk("RX IRQ: aio_complete: %x %d\n", desc->res1[1], desc->res1[0]);
                    aio_complete((struct kiocb*)desc->res1[1], desc->res1[0], 0);
                    desc->res1[0] = 0;
                    desc->res1[1] = 0;
                }
            }
            // Fetch next descriptor
            desc = &dev->rx_ring_ptr_v[(start + len + count) % (dev->global_settings.rx_desc_len)];
            // Get status of descriptor
            status = desc->status;
        }
        // Update size of RX ring. Must be locked.
        spin_lock_irqsave(&dev->rx_lock, flags);
            if (likely(dev->global_settings.rx_mode == RX_DIRECT))
            {
//                 printk("RX IRQ Before: Start: %d Len: %d Count: %d RX_Desc_Len: %d\n", dev->rx_ring_start, dev->rx_ring_len, count, dev->global_settings.rx_desc_len);
                dev->rx_ring_start = (dev->rx_ring_start + count) % (dev->global_settings.rx_desc_len);
                dev->rx_ring_len -= count;
//                 printk("RX IRQ After: Start: %d Len: %d\n", dev->rx_ring_start, dev->rx_ring_len);
            }    
            else
            {
                dev->rx_ring_len += count;
            }
        spin_unlock_irqrestore(&dev->rx_lock, flags);
//         spin_unlock_irqrestore(&dev->read_lock, flags);
        // Wake RX wait queue
        wake_up_interruptible(&dev->rx_perf_queue);
    }

fin:
    // Irq is handled.
    return IRQ_HANDLED;
};


/**
 * open_device  - Open device file.
 * @inode:        Inode structure
 * @filp:         File pointer
 *
 * returns:    Status of operation. Returns 0 on success, any other value means
 *             failure.
 *
 * This function handles device file opening. It alocates IRQ and initializes 
 * spinlocks. Device file can be open 1x for read and 1x for write.
 */
int open_device(struct inode* inode, struct file* filp)
{
    // TODO: Fix for more opened file descriptors
    int ret;
    volatile struct driver_info *dev;
    // Get driver info structure from cdev subcomponent of this structure
    dev = container_of(inode->i_cdev, struct driver_info, device);
    // Save the driver info structure into private_data for later use
    filp->private_data = dev;

    // Device file is open for read
    if (filp->f_mode & FMODE_READ)
    {   
        // Set state of RX channel to stop
        dev->rx_state = STOP;
        
    }
    // Device file is open for write
    if (filp->f_mode & FMODE_WRITE)
    {
        // Set state of TX channel to stop
        dev->tx_state = STOP;
        
    }
    return 0;

    return ret;
}

/**
 * close_device  - Close device file.
 * @inode:        Inode structure
 * @filp:         File pointer
 *
 * returns:    Status of operation. Returns 0 on success, any other value means
 *             failure.
 *
 * This function handles device file close. It dealocates IRQ and stop DMA channel.
 */
int close_device(struct inode* inode, struct file* filp)
{   
    struct driver_info *dev;
    int i;
    
    // Get driver info strucure from privite_data
    dev = (struct driver_info *)filp->private_data;
    
    // Device file is opened for read
    if (filp->f_mode & FMODE_READ)
    {
        // Stop RX DMA channel
//         stop_rx(dev);
        // Wait until idle
//         i = 1000;
//         while (((dma_in(dev, S2MM_DMASR_OFFSET) & IDLE_MASK) == 0) && (i > 0))
//         {
//             udelay(100);
//             i--;
//         }
        // Disable IRQ
//         dma_out(dev, S2MM_DMACR_OFFSET, dma_in(dev, S2MM_DMACR_OFFSET) & ~ALL_MASK);
        //reset_rx(dev);
        // Free allocated RX IRQ
//         free_irq(dev->rx_irq, dev);
        // Set RX channel state to stop
        dev->rx_state = STOP;
    }
    
    // Device file is opened for 
    if (filp->f_mode & FMODE_WRITE)
    { 
        // Stop TX DMA channel
//         stop_tx(dev);
        // Wait until idle
//         i = 1000;
//         while (((dma_in(dev, MM2S_DMASR_OFFSET) & IDLE_MASK) == 0) && (i > 0))
//         {
//             udelay(100);
//             i--;
//         }
        // Disable IRQ
//         dma_out(dev, MM2S_DMACR_OFFSET, dma_in(dev, MM2S_DMACR_OFFSET) & ~ALL_MASK);
        //reset_tx(dev);
        // Free allocated TX IRQ
//         free_irq(dev->tx_irq, dev);
        // Set TX channel state to stop
        dev->tx_state = STOP;
    }

    return 0;
}

/**
 * get_allocation_info  - Get allocation info structure from userspace address.
 * @addr:       Userspace adresses
 *
 * returns:    Pointer to the allocation info structure. Returns NULL if 
 *             address is not in hash_allocation hash table.
 *
 * This function gets allocation info structure from userspace address. The 
 * address must be start of the allocated buffer, otherwise the allocation info 
 * structure can not be determinated.
 */
struct allocation_info * get_allocation_info(char* addr)
{
    struct allocation_info * ret = NULL;
    struct allocation_info * itr = NULL;
    struct hlist_node * node = NULL;
    
    hash_for_each_possible(hash_allocation, itr, /*node,*/ list, addr){
        if (itr->user_virt == addr)
        {
            ret = itr;
            break;
        }
    }
    return ret;
}

/**
 * get_number_of_tx_buffers  - Get number of TX buffers requested by write vector operation.
 * @dev:        Driver info structure
 * @iov:        IO vector info - base adresses and lengths of data
 * @count:      Number of user buffers
 *
 * returns:    Number of TX DMA buffers necessery to handle user requested write
 *
 * This function computes the number of TX DMA buffers necessery to handle user requested write.
 */
 u32 get_number_of_tx_buffers(struct driver_info * dev, const struct iovec *iov, unsigned long count)
{
    unsigned long i;
    u32 len = 0;
    
    // Iterate over whole IO vector and compute the number of TX DMA buffers 
    // necessery to hold the user data
    for (i = 0; i < count; i++)
    {
        len += DIV_ROUND_UP(iov[i].iov_len, dev->global_settings.tx_buffer_len);
    }
//     dev_warn(&dev->of_dev->dev, "TX Req buffers: %d IOV Count %d\n", len, count);
    return len;
}

/**
 * get_number_of_tx_direct_buffers  - Get number of TX buffers requested by direct write vector operation.
 * @dev:        Driver info structure
 * @iov:        IO vector info - base adresses and lengths of data
 * @count:      Number of user buffers
 *
 * returns:    Number of TX DMA buffers necessery to handle user requested direct write
 *
 * This function computes the number of TX DMA buffers necessery to handle user requested direct write.
 */
 u32 get_number_of_tx_direct_buffers(struct driver_info * dev, const struct iovec *iov, unsigned long count)
{
    unsigned long i;
    u32 len = 0;
    struct allocation_info * allocation;

    // Iterate over whole IO vector and compute the number of TX DMA buffers 
    // necessery to hold the user data
    for (i = 0; i < count; i++)
    {
        allocation = get_allocation_info(iov[i].iov_base);
        if (unlikely(!allocation))
        {
            dev_warn(&dev->of_dev->dev, "TX: Unknown kernel buffer. Kernel allocated buffer must be used whole.\n");
            return 0;
        }
        len += DIV_ROUND_UP(iov[i].iov_len, allocation->allocation_size);
    }
//     dev_warn(&dev->of_dev->dev, "TX Req buffers: %d IOV Count %d\n", len, count);
    return len;
}

/**
 * get_number_of_rx_buffers  - Get number of RX buffers requested by read vector operation.
 * @dev:        Driver info structure
 * @iov:        IO vector info - base adresses and lengths of data
 * @count:      Number of user buffers
 *
 * returns:    Number of RX DMA buffers necessery to handle user requested read
 *
 * This function computes the number of RX DMA buffers necessery to handle user requested read.
 */
 u32 get_number_of_rx_buffers(struct driver_info * dev, const struct iovec *iov, unsigned long count)
{
    unsigned long i;
    u32 len = 0;
    
    // Iterate over whole IO vector and compute the number of TX DMA buffers 
    // necessery to hold the user data
    for (i = 0; i < count; i++)
    {
        len += DIV_ROUND_UP(iov[i].iov_len, dev->global_settings.rx_buffer_len);
    }
    return len;
}

/**
 * get_number_of_rx_direct_buffers  - Get number of RX buffers requested by direct read vector operation.
 * @dev:        Driver info structure
 * @iov:        IO vector info - base adresses and lengths of data
 * @count:      Number of user buffers
 *
 * returns:    Number of RX DMA buffers necessery to handle user requested direct read
 *
 * This function computes the number of RX DMA buffers necessery to handle user requested direct read.
 */
 u32 get_number_of_rx_direct_buffers(struct driver_info * dev, const struct iovec *iov, unsigned long count)
{
    unsigned long i;
    u32 len = 0;
    struct allocation_info * allocation;
    
    // Iterate over whole IO vector and compute the number of TX DMA buffers 
    // necessery to hold the user data
    for (i = 0; i < count; i++)
    {
//         printk("Element %d IOV_BASE: %x IOV_LEN: %x\n", i, iov[i].iov_base, iov[i].iov_len);
        allocation = get_allocation_info(iov[i].iov_base);
        if (unlikely(!allocation))
        {
            dev_warn(&dev->of_dev->dev, "RX: Unknown kernel buffer. Kernel allocated buffer must be used whole.\n");
            return 0;
        }
//         printk("Allocation info: AAddr: %x Size: %x", allocation, allocation->allocation_size);
        len += DIV_ROUND_UP(iov[i].iov_len, allocation->allocation_size);
    }
    return len;
}

/**
 * get_free_buffers  - Get number of unused TX buffers.
 * @dev:        Driver info structure
 *
 * returns:    Number of free TX DMA buffers
 *
 * This function computes the number of free TX DMA buffers. The TX ring lock is used.
 */
 inline u32 get_free_buffers(struct driver_info * dev)
{
    unsigned long flags;
    u32 ret;
    
    // Lock the TX ring lock and get the number of free TX DMA buffers
    spin_lock_irqsave(&dev->tx_lock, flags);
        ret = dev->global_settings.tx_desc_len - dev->tx_ring_len;
    spin_unlock_irqrestore(&dev->tx_lock, flags);
    
    return ret;
}

/**
 * get_used_buffers  - Get number of used RX buffers.
 * @dev:        Driver info structure
 *
 * returns:    Number of used RX DMA buffers
 *
 * This function computes the number of used RX DMA buffers. The RX ring lock is used.
 */
 inline u32 get_used_buffers(struct driver_info * dev)
{
    u32 ret;
    unsigned long flags;
    
    // Lock the RX ring lock and get the number of used RX DMA buffers
    spin_lock_irqsave(&dev->rx_lock, flags);
        ret = dev->rx_ring_len;
    spin_unlock_irqrestore(&dev->rx_lock, flags);
    
    return ret;
}

/**
 * get_unused_buffers  - Get number of unused RX buffers.
 * @dev:        Driver info structure
 *
 * returns:    Number of unused RX DMA buffers
 *
 * This function computes the number of unused RX DMA buffers. The RX ring lock is used.
 */
 inline u32 get_unused_buffers(struct driver_info * dev)
{
    u32 ret;
    unsigned long flags;
    
    // Lock the RX ring lock and get the number of used RX DMA buffers
    spin_lock_irqsave(&dev->rx_lock, flags);
        ret = dev->global_settings.rx_desc_len - dev->rx_ring_len;
    spin_unlock_irqrestore(&dev->rx_lock, flags);
    
    return ret;
}

/**
 * get_tx_buffer_adress_v  - Get virtual address of selected  TX DMA buffer.
 * @dev:        Driver info structure
 * @start:      Index of first unprocessed buffer in TX ring
 * @len:        index of selected buffer relative to start
 *
 * returns:    Virtual address of selected  TX DMA buffer
 *
 * This function computes virtual address of selected  TX DMA buffer.
 */
 inline char* get_tx_buffer_adress_v(struct driver_info * dev, u32 start, u32 len)
{
    return dev->tx_ring_v + ((start + len) % (dev->global_settings.tx_desc_len)) * dev->global_settings.tx_buffer_len;
}

/**
 * get_rx_buffer_adress_v  - Get virtual address of selected RX DMA buffer.
 * @dev:        Driver info structure
 * @start:      Index of first unprocessed buffer in RX ring
 * @len:        index of selected buffer relative to start
 *
 * returns:    Virtual address of selected  RX DMA buffer
 *
 * This function computes virtual address of selected  RX DMA buffer.
 */
 inline char* get_rx_buffer_adress_v(struct driver_info * dev, u32 start, u32 len)
{
    return dev->rx_ring_v + ((start + len) % (dev->global_settings.rx_desc_len)) * dev->global_settings.rx_buffer_len;
}

/**
 * get_desc_adress_v  - Get virtual address of selected TX DMA descriptor.
 * @dev:        Driver info structure
 * @start:      Index of first unprocessed descriptor in TX ring
 * @len:        index of selected descriptor relative to start
 *
 * returns:    Virtual address of selected  TX DMA descriptor
 *
 * This function computes virtual address of selected  TX DMA descriptor.
 */
 inline struct dma_ring* get_desc_adress_v(struct driver_info * dev, u32 start, u32 len)
{
//     printk("DESC BASE: %x START: %d LEN: %d SIZE: %d\n", dev->tx_ring_ptr_v, start, len, sizeof(struct dma_ring));
    return dev->tx_ring_ptr_v + ((start + len) % (dev->global_settings.tx_desc_len));// * sizeof(struct dma_ring);
}

/**
 * get_rx_desc_adress_v  - Get virtual address of selected RX DMA descriptor.
 * @dev:        Driver info structure
 * @start:      Index of first unprocessed descriptor in RX ring
 * @len:        index of selected descriptor relative to start
 *
 * returns:    Virtual address of selected  RX DMA descriptor
 *
 * This function computes virtual address of selected  RX DMA descriptor.
 */
 inline struct dma_ring* get_rx_desc_adress_v(struct driver_info * dev, u32 start, u32 len)
{
//     printk("DESC BASE: %x START: %d LEN: %d SIZE: %d\n", dev->tx_ring_ptr_v, start, len, sizeof(struct dma_ring));
    return dev->rx_ring_ptr_v + ((start + len) % (dev->global_settings.rx_desc_len));// * sizeof(struct dma_ring);
}

/**
 * get_tx_desc_adress_p  - Get physical address of selected TX DMA descriptor.
 * @dev:        Driver info structure
 * @start:      Index of first unprocessed descriptor in TX ring
 * @len:        index of selected descriptor relative to start
 *
 * returns:    Physical address of selected  TX DMA descriptor
 *
 * This function computes physical address of selected  TX DMA descriptor.
 */
 inline dma_addr_t get_tx_desc_adress_p(struct driver_info * dev, u32 start, u32 len)
{
    return dev->tx_ring_ptr_p + ((start + len) % (dev->global_settings.tx_desc_len)) * sizeof(struct dma_ring);
}

/**
 * get_rx_desc_adress_p  - Get physical address of selected RX DMA descriptor.
 * @dev:        Driver info structure
 * @start:      Index of first unprocessed descriptor in RX ring
 * @len:        index of selected descriptor relative to start
 *
 * returns:    Physical address of selected  RX DMA descriptor
 *
 * This function computes physical address of selected  RX DMA descriptor.
 */
 inline dma_addr_t get_rx_desc_adress_p(struct driver_info * dev, u32 start, s32 len)
{
    if ((start + len) < 0)
    {
        start = dev->global_settings.rx_desc_len + (start + len);
        len = 0;
    }
    return dev->rx_ring_ptr_p + ((start + len) % (dev->global_settings.rx_desc_len)) * sizeof(struct dma_ring);
}

/**
 * writev_device  - Write vector operation.
 * @dev:        Kernel IO control block
 * @iov:        IO vector info - base adresses and lengths of data
 * @count:      Number of user buffers
 * @pos:        File position
 *
 * returns:    Size of written data. Returns -ENOMEM if requested write is bigger than DMA ring.
 *
 * This function perform vector write operation. The function writes copy 
 * of user buffer to TX DMA ring and initiates DMA TX channel operation. 
 * This function sleeps in wait queue until neccessery amount of TX buffers 
 * is free. 
 */
 ssize_t writev_device(struct kiocb *iocb, const struct iovec *iov, unsigned long count, loff_t pos)
{
    u32 req_buffers;
    u32 free_buffers;
    u32 b_len, c_len;
    u32 start, len, index;
    ssize_t total;
    unsigned long i;
    u32 j;
    int ret;
    char * buff_addr;
    struct dma_ring* desc;
    unsigned long flags;
    struct driver_info *dev;
    struct file *filp;
    struct allocation_info *allocation;

    total = 0;
    
//     printk("W");
    // Get driver info structure
    filp = iocb->ki_filp;
    dev = (struct driver_info *)filp->private_data;
    
//     spin_lock_irqsave(&dev->write_lock, flags);
//     dev_warn(&dev->of_dev->dev, "TX Count: %d\n", count);
    
    // Get number of buffers requested by write operation
    if (likely(dev->global_settings.tx_mode == TX_DIRECT))
        req_buffers = get_number_of_tx_direct_buffers(dev, iov, count);
    else
        req_buffers = get_number_of_tx_buffers(dev, iov, count);
//     dev_warn(&dev->of_dev->dev, "TX Req buffers: %d\n", req_buffers);
    
    // Check if number of requested buffers is bigger than number of TX ring buffers.
    if (req_buffers > (dev->global_settings.tx_desc_len))
    {
        // Warn and return error code
        dev_warn(&dev->of_dev->dev, "Not enough memory in whole TX ring for write operation.\n");
        return -ENOMEM;
    }
    // Get number of free buffers
    free_buffers = dev->global_settings.tx_desc_len - dev->tx_ring_len; //get_free_buffers(dev);
//     dev_warn(&dev->of_dev->dev, "TX Free buffers pre: %d\n", free_buffers);
    // If number of requested buffers is bigger than number of free buffers 
    // sleep and wait until we have sufficient number of free buffers
    if (free_buffers<req_buffers)
    {
//         spin_unlock_irqrestore(&dev->write_lock, flags);
        while(/*get_free_buffers(dev)*/ (dev->global_settings.tx_desc_len - dev->tx_ring_len) < req_buffers)
        {
        ret = wait_event_interruptible_timeout(dev->tx_perf_queue, /*get_free_buffers(dev)*/(dev->global_settings.tx_desc_len - dev->tx_ring_len)>=req_buffers, HZ/100);
        if (ret < 0)
            return total;
        else if (ret > 0)
            break;
        }
//         spin_lock_irqsave(&dev->write_lock, flags);
    }
//     dev_warn(&dev->of_dev->dev, "TX Free buffers post: %d\n", free_buffers);

    // Lock TX ring lock and get start index of TX ring and number of 
    // unprocessed buffers in the TX ring.
    spin_lock_irqsave(&dev->tx_lock, flags);
        start = dev->tx_ring_start;
        len = dev->tx_ring_len;
    spin_unlock_irqrestore(&dev->tx_lock, flags);
//     printk("TX write start: Len: %d", dev->tx_ring_len);
//     dev_warn(&dev->of_dev->dev, "TX start: %d Len: %d", start, len);
    
    // Initiate index by number of unprocessed buffers in the TX ring.
    index = len;
    // Use Kernel allocated user space buffers
    if (likely(dev->global_settings.tx_mode == TX_DIRECT))
    {
        // Loop over all IO vector items
        for (i = 0; i < count; i++)
        {
            allocation = get_allocation_info(iov[i].iov_base);
            if (unlikely(!allocation))
            {
                dev_warn(&dev->of_dev->dev, "TX: Kernel allocated buffer must be used whole.\n");
                return -ENOMEM;
            }
            // Get number of TX descriptors neccessery for user data
            b_len = DIV_ROUND_UP(iov[i].iov_len, allocation->allocation_size);
//             printk("TX IO Len: %d, ALLOC size: %d, Count: %d, BLEN: %d\n", iov[i].iov_len, allocation->allocation_size, count, b_len);
            // update total length
            total += iov[i].iov_len;
            // Prepere all TX buffers and descriptors for user data
            for (j = 0; j < b_len; j++)
            {
                // Get TX descriptor virtual adress
                desc = get_desc_adress_v(dev, start, index);
                // Get TX buffer physical addres
                desc->buffer = allocation->parts[j].phys;
                // Compute length of data writen to the TX buffer
                c_len = (j < b_len - 1) ? allocation->allocation_size : (iov[i].iov_len - (b_len - 1) * allocation->allocation_size);
//                 printk("TX C_LEN: %x\n", c_len);
    //             printk("KERNEL BASE: %x KERNEL: %x USER BASE: %x USER SEG: %x \n LEN: %d DESC: %x\n", dev->tx_ring_v, buff_addr, iov[i].iov_base, iov[i].iov_base + j*dev->global_settings.tx_buffer_len, c_len, desc);
                // Set the length of TX buffer data to descriptor
                desc->control = c_len;
                // If it is first TX buffer for the user data, set the SOF flag in descriptor
                if (j == 0)
                    desc->control |= DESC_SOF;
                // Mark the buffer for sync and synchronise the caches for DMA
                if (allocation->cacheable_buffers != 0)
                {
                    // Synchronise for device
                    dma_sync_single_for_device(&dev->of_dev->dev, allocation->parts[j].phys, allocation->allocation_size, DMA_BIDIRECTIONAL);
                    // Mark the buffer for sync after transmition, save the allocation_size
                    desc->res1[2] = allocation->allocation_size;
                }
                else
                {
                    desc->res1[2] = 0;
                }
                // If it is last TX buffer for the user data, set the EOF flag in descriptor
                if (j == b_len - 1)
                {
                    desc->control |= DESC_EOF;
                    if (i == count - 1)
                    {
                        desc->res1[0] = total;
                        desc->res1[1] = iocb;
//                         printk("TX: Total: %d, IOCB: %x\n", desc->res1[0], desc->res1[1]);
                        total = -EIOCBQUEUED;
                    }
                }
//                 if ((desc->control & LEN_MASK) != 1024)
//                 {
//                     dev_warn(&dev->of_dev->dev, "TX Desc len mismatch %d\n", desc->control & LEN_MASK);
//                     dev_warn(&dev->of_dev->dev, "TX Desc %d: Next: %x, Buff: %x, Ctrl: %x, Stat: %x\nApp:%x %x %x %x %x\n", i, desc->next, desc->buffer, desc->control, desc->status, desc->application[0], desc->application[1], desc->application[2], desc->application[3], desc->application[4]);
//                 }
                // Clear status
                desc->status = 0;
                // Update index
                index++;
            }
            
        }
    }
    else
    {
        // Loop over all IO vector items
        for (i = 0; i < count; i++)
        {

            // Get number of TX buffers neccessery for user data
            b_len = DIV_ROUND_UP(iov[i].iov_len, dev->global_settings.tx_buffer_len);
            // update total length
            total += iov[i].iov_len;
            // Prepere all TX buffers and descriptors for user data
            for (j = 0; j < b_len; j++)
            {

                // Get TX buffer virtual addres
                buff_addr = get_tx_buffer_adress_v(dev, start, index);
                // Get TX descriptor virtual adress
                desc = get_desc_adress_v(dev, start, index);
                // Compute length of data writen to the TX buffer
                c_len = (j < b_len - 1) ? dev->global_settings.tx_buffer_len : (iov[i].iov_len - (b_len - 1) * dev->global_settings.tx_buffer_len);

    //             printk("KERNEL BASE: %x KERNEL: %x USER BASE: %x USER SEG: %x \n LEN: %d DESC: %x\n", dev->tx_ring_v, buff_addr, iov[i].iov_base, iov[i].iov_base + j*dev->global_settings.tx_buffer_len, c_len, desc);
                // Copy data from user space to the TX buffer
                copy_from_user(buff_addr, iov[i].iov_base + j*dev->global_settings.tx_buffer_len, c_len);
    //             printk("Copy finished.\n");
                // Set the length of TX buffer data to descriptor
                desc->control = c_len;
                // If it is first TX buffer for the user data, set the SOF flag in descriptor
                if (j == 0)
                    desc->control |= DESC_SOF;
                // If it is last TX buffer for the user data, set the EOF flag in descriptor
                if (j == b_len - 1)
                    desc->control |= DESC_EOF;
                // Clear status
                desc->status = 0;
//                 if ((desc->control & LEN_MASK) != 1024)
//                 {
//                     dev_warn(&dev->of_dev->dev, "TX Desc len mismatch %d\n", desc->control & LEN_MASK);
//                     dev_warn(&dev->of_dev->dev, "TX Desc %d: Next: %x, Buff: %x, Ctrl: %x, Stat: %x\nApp:%x %x %x %x %x\n", i, desc->next, desc->buffer, desc->control, desc->status, desc->application[0], desc->application[1], desc->application[2], desc->application[3], desc->application[4]);
//                 }
                // Update index
                index++;
            }
        }
    }
    wmb();
//     printk("LOCK start.\n");
    // Lock TX lock and update length of unprocessed buffers and set the tail 
    // descriptor of TX DMA channel
    spin_lock_irqsave(&dev->tx_lock, flags);
//         printk("TX write before: Len: %d REQBUFF: %d\n", dev->tx_ring_len, req_buffers);
        dev->tx_ring_len += req_buffers;
//         printk("TX write after: Len: %d\n", dev->tx_ring_len);
        wmb();
        dma_out(dev, MM2S_TAILDESC_OFFSET, get_tx_desc_adress_p(dev, start, index - 1));
    spin_unlock_irqrestore(&dev->tx_lock, flags);
//      printk("LOCK end.\n");
//     spin_unlock_irqrestore(&dev->write_lock, flags);
    // Return number of written bytes
    return total;
}

//  ssize_t write_device(struct file *filp, const char __user *buf, size_t count, loff_t *ppos)
// {
//     struct kiocb iocb;
//     struct iovec iov = { .iov_base = buf, .iov_len = count };
//     struct driver_info *dev;
//     
//     iocb.ki_filp = filp;
//     dev = (struct driver_info *)filp->private_data;
// 
//     return writev_device(&iocb, &iov, 1, *ppos);
// }

/**
 * async_readv  - Asynchronous read vector operation.
 * @iocb:        Kernel IO control block
 * @iov:        IO vector info - base adresses and lengths of data
 * @count:      Number of user buffers
 * @pos:        File position
 *
 * returns:    Size of read data. Returns -ENOMEM if some user specified buffer is smaller than received frame size.
 *
 * This function perform asynchronous vector read operation. The function sets user/kernel buffers to the RX ring and start read operation. This function sleeps in wait queue while requested amount of descriptors is not ready.
 */
inline ssize_t async_readv(struct kiocb *iocb, const struct iovec *iov, unsigned long count, loff_t pos)
{
    u32 req_buffers;
    u32 unused_buffers;
    u32 i,j;
    u32 index = 0;
    u32 start;
    u32 len;
    ssize_t total = 0;
    u32 b_len;
    u32 c_len;
    unsigned long flags;
    int ret;
    struct dma_ring* desc;
    struct driver_info *dev;
    struct allocation_info *allocation;
    
    //TODO: Fix - only one async operation at any time, should be any sane number
    
    // Get driver info structure
    dev = (struct driver_info *)iocb->ki_filp->private_data;

    // Get number of requested buffers
    req_buffers = get_number_of_rx_direct_buffers(dev, iov, count);
//     dev_warn(&dev->of_dev->dev, "RX Req buffers: %d\n", req_buffers);
    // Get number of free buffers in RX ring
    unused_buffers = get_unused_buffers(dev);
//     dev_warn(&dev->of_dev->dev, "RX Used buffers: %d\n", used_buffers);
    // If no buffers are present sleep in wait queue
    if (unused_buffers < req_buffers)
    {
        while (get_unused_buffers(dev) < req_buffers)
        {
            ret = wait_event_interruptible_timeout(dev->rx_perf_queue, get_unused_buffers(dev) >= req_buffers, HZ/100);
            if (ret < 0)
                return total;
            else if (ret > 0)
                break;
        }
    }
    
    // Lock RX ring lock and get start index of wait queue and number of used buffers
    spin_lock_irqsave(&dev->rx_lock, flags);
        start = dev->rx_ring_start;
        len = dev->rx_ring_len;
    spin_unlock_irqrestore(&dev->rx_lock, flags);
    
    // Loop over all IO vector items
    for (i = 0; i < count; i++)
    {
        allocation = get_allocation_info(iov[i].iov_base);
        if (unlikely(!allocation))
        {
            dev_warn(&dev->of_dev->dev, "RX: Kernel allocated buffer must be used whole.\n");
            return -ENOMEM;
        }
        // Get number of RX descriptors neccessery for user data
        b_len = DIV_ROUND_UP(iov[i].iov_len, allocation->allocation_size);
//         printk("RX IO Len: %d, ALLOC size: %d, BLEN: %d\n", iov[i].iov_len, allocation->allocation_size, b_len);
        // update total length
        total += iov[i].iov_len;
        // Prepere all RX buffers and descriptors for user data
        for (j = 0; j < b_len; j++)
        {
            // Get RX descriptor virtual adress
            desc = get_rx_desc_adress_v(dev, start, index);
            // Get TX buffer physical addres
            desc->buffer = allocation->parts[j].phys;
            // Compute length of data writen to the TX buffer
            c_len = (j < b_len - 1) ? allocation->allocation_size : (iov[i].iov_len - (b_len - 1) * allocation->allocation_size);

//             printk("KERNEL BASE: %x KERNEL: %x USER BASE: %x USER SEG: %x \n LEN: %d DESC: %x\n", dev->rx_ring_v, buff_addr, iov[i].iov_base, iov[i].iov_base + j*dev->global_settings.rx_buffer_len, c_len, desc);
            // Set the length of RX buffer data to descriptor
            desc->control = c_len;
            // Mark the buffer for sync and synchronise the caches for DMA
            if (allocation->cacheable_buffers != 0)
            {
                // Synchronise for device
                dma_sync_single_for_device(&dev->of_dev->dev, allocation->parts[j].phys, allocation->allocation_size, DMA_BIDIRECTIONAL);
                // Mark the buffer for sync after transmition, save the allocation_size
                desc->res1[2] = allocation->allocation_size;
            }
            else
            {
                desc->res1[2] = 0;
            }
            // If it is last RX buffer for the user data, set the -EIOCBQUEUED 
            if ((j == b_len - 1) && (i == count - 1))
            {
                desc->res1[0] = total;
                desc->res1[1] = iocb;
//                 printk("RX: Total: %d, IOCB: %x\n", desc->res1[0], desc->res1[1]);
                total = -EIOCBQUEUED;
            }
            // Clear status
            desc->status = 0;
            // Update index
            index++;
        }
    }
    
    wmb();
    // Lock the RX lock. Update start of ring, length of unprocessed RX ring 
    // elements and set new RX channel tail descriptor.
    spin_lock_irqsave(&dev->rx_lock, flags);
        dev->rx_ring_len += req_buffers;
        wmb();
        dma_out(dev, S2MM_TAILDESC_OFFSET, get_rx_desc_adress_p(dev, start, req_buffers - 1));
    spin_unlock_irqrestore(&dev->rx_lock, flags);
//     spin_unlock_irqrestore(&dev->read_lock, flags);
    // Return total number of read data
    return total;
}

/**
 * readv_device  - Read vector operation.
 * @dev:        Kernel IO control block
 * @iov:        IO vector info - base adresses and lengths of data
 * @count:      Number of user buffers
 * @pos:        File position
 *
 * returns:    Size of read data. Returns -ENOMEM if some user specified buffer is smaller than received frame size.
 *
 * This function perform vector read operation. The function copies data from 
 * RX DMA ring into user buffer. This function sleeps in wait queue while no data are in RX ring buffers. Can return less data than requested.
 */
 ssize_t readv_device(struct kiocb *iocb, const struct iovec *iov, unsigned long count, loff_t pos)
{
    u32 req_buffers;
    u32 used_buffers;
    u32 min;
    u32 i,j, k;
    u32 start;
    u32 len;
    ssize_t total = 0;
    u32 sub_len = 0;
    u32 d_len;
    unsigned long flags;
    int ret;
    struct dma_ring* desc;
    struct driver_info *dev;
        
    // Get driver info structure
    dev = (struct driver_info *)iocb->ki_filp->private_data;
    if (likely(dev->global_settings.rx_mode == RX_DIRECT))
    {
        return async_readv(iocb, iov, count, pos);
    }
//     spin_lock_irqsave(&dev->read_lock, flags);
    // Get number of requested buffers
    req_buffers = get_number_of_rx_buffers(dev, iov, count);
//     dev_warn(&dev->of_dev->dev, "RX Req buffers: %d\n", req_buffers);
    // Get number of used buffers in RX ring
    used_buffers = get_used_buffers(dev);
//     dev_warn(&dev->of_dev->dev, "RX Used buffers: %d\n", used_buffers);
    // If no buffers are present sleep in wait queue
    // TODO: Spravit aby se cekalo na minimalni pocet bufferu potrebnych pro 
    // naplneni prvniho uzivatelskeho bufferu
    if (used_buffers == 0)
    {
//         spin_unlock_irqrestore(&dev->read_lock, flags);
        while (get_used_buffers(dev) == 0)
        {
            ret = wait_event_interruptible_timeout(dev->rx_perf_queue, get_used_buffers(dev) > 0, HZ/100);
            if (ret < 0)
                return total;
            else if (ret > 0)
                break;
        }
//         spin_lock_irqsave(&dev->read_lock, flags);
    }
    
    // Lock RX ring lock and get start index of wait queue and number of used buffers
    spin_lock_irqsave(&dev->rx_lock, flags);
        start = dev->rx_ring_start;
        len = dev->rx_ring_len;
    spin_unlock_irqrestore(&dev->rx_lock, flags);
    
//      dev_warn(&dev->of_dev->dev, "RX Start: %d Len: %d", start, len);
    
    // Get number of buffers to process
    // TODO: spravit tak aby cteni nekoncilo v pulce uzivatelskeho bufferu, 
    // je-li user buffer vetsi  nez DMA buffer.
    min = (len<req_buffers) ? len : req_buffers;
//     dev_warn(&dev->of_dev->dev, "RX Number of buffers to read: %d\n", min);
    // Init RX buffer counter 
    i = 0;
    // Init counter of user buffers
    j = 0;
    // Init counter of RX buffers needed for user buffer
    k = 0;
    // Init size of RX frame
    sub_len = 0;
    // Process the RX buffers for this operation
    while (i < min)
    {
        // Get descriptor virtual address
        desc = &dev->rx_ring_ptr_v[(start+i) % (dev->global_settings.rx_desc_len)];
        // Get length of data in associated buffer
        d_len = desc->status & LEN_MASK;
//         if (d_len != 1024)
//         {
//             dev_warn(&dev->of_dev->dev, "RX Desc len mismatch %d\n", d_len);
//             dev_warn(&dev->of_dev->dev, "RX Desc %d: Next: %x, Buff: %x, Ctrl: %x, Stat: %x\nApp:%x %x %x %x %x\n", i, desc->next, desc->buffer, desc->control, desc->status, desc->application[0], desc->application[1], desc->application[2], desc->application[3], desc->application[4]);
//         }
        // Update total length
        total += d_len;
        // Update RX frame length
        sub_len += d_len;
        // If length is bigger than RX buffer, rise error
        if (unlikely(sub_len > iov[j].iov_len))
        {
            dev_warn(&dev->of_dev->dev, "Not enough memory in element %d, declared size %d x dma size %d\n", j, iov[j].iov_len, sub_len);
            return -ENOMEM;
        }
        // Copy RX buffer to user buffer
        copy_to_user(iov[j].iov_base + k*dev->global_settings.rx_buffer_len, get_rx_buffer_adress_v(dev, start, i), d_len);
        // If last RX buffer in RX frame, update the counters
        if (desc->status & DESC_EOF)
        {
            j +=1;
            k = 0;
            sub_len = 0;
        }
        // update index in RX ring
        i++;
        // Clear status
        desc->status = 0;
        // Clear application fields
        desc->application[0] = 0;
        desc->application[1] = 0;
        desc->application[2] = 0;
        desc->application[3] = 0;
        desc->application[4] = 0;
        // Set size of RX buffer
        desc->control = dev->global_settings.rx_buffer_len;
    }
    wmb();
    // Lock the RX lock. Update start of ring, length of unprocessed RX ring 
    // elements and set new RX channel tail descriptor.
    spin_lock_irqsave(&dev->rx_lock, flags);
        dev->rx_ring_start = (dev->rx_ring_start + min) % (dev->global_settings.rx_desc_len);
        dev->rx_ring_len -= min;
        wmb();
        dma_out(dev, S2MM_TAILDESC_OFFSET, get_rx_desc_adress_p(dev, start, -1));
    spin_unlock_irqrestore(&dev->rx_lock, flags);
//     spin_unlock_irqrestore(&dev->read_lock, flags);
    // Return total number of read data
    return total;
}

//  ssize_t read_device(struct file *filp, char __user *buf, size_t count, loff_t *ppos)
// {
//     struct kiocb iocb;
//     struct iovec iov = { .iov_base = buf, .iov_len = count };
//     iocb.ki_filp = filp;
//     return readv_device(&iocb, &iov, 1, *ppos);
// }

/**
 * verify_settings - Verify user settings corectness
 * @settings: Settings to be verified.
 *
 * returns: Returns 0 if settings are correct and -1 othewise.
 *
 * This function verifies corectnes of user settings.
 */
 u32 verify_settings(struct dma_settings *settings)
{
    if (settings->rx_act_timeout > MAX_DELAY)
        goto failure;
    if (settings->rx_inact_timeout > MAX_DELAY)
        goto failure;
    if (settings->tx_act_timeout > MAX_DELAY)
        goto failure;
    if (settings->tx_inact_timeout > MAX_DELAY)
        goto failure;
    if (settings->rx_ring_len > MAX_RING)
        goto failure;
    if (settings->rx_ring_len < settings->rx_buffer_len)
        goto failure;
    if (settings->tx_ring_len > MAX_RING)
        goto failure;
    if (settings->tx_ring_len < settings->tx_buffer_len)
        goto failure;
    if (settings->rx_buffer_len < MIN_BUFFER)
        goto failure;
    if (settings->tx_buffer_len < MIN_BUFFER)
        goto failure;
    if (settings->rx_ring_len % settings->rx_buffer_len != 0)
        goto failure;
    if (settings->tx_ring_len % settings->tx_buffer_len != 0)
        goto failure;
    if (settings->rx_ring_len % MIN_BUFFER != 0)
        goto failure;
    if (settings->tx_ring_len % MIN_BUFFER != 0)
        goto failure;
    if (!((settings->rx_mode >= RX_COPY) && (settings->rx_mode <= RX_MMAP)))
        goto failure;
    if (!((settings->tx_mode >= TX_COPY) && (settings->tx_mode <= TX_MMAP)))
        goto failure;
    return 0;
failure:
    return -1;
}

/**
 * update_settings - Update RX and TX DMA channel settings
 * @global: Driver settings
 * @local: User settings
 * @mode: Mode of opened file.
 *
 * This function update RX and TX DMA channel settings. Mode 
 * of file is used to update only valid channel. 
 */
void update_settings(struct dma_settings *global, struct dma_settings *local, u32 mode)
{
    if (mode & FMODE_READ)
    {
        global->rx_act_timeout = local->rx_act_timeout;  
        global->rx_inact_timeout = local->rx_inact_timeout;
        global->rx_ring_len = local->rx_ring_len;    
        global->rx_buffer_len = local->rx_buffer_len;   
        global->rx_mode = local->rx_mode;
        global->rx_desc_len = global->rx_ring_len / global->rx_buffer_len;
    }
    if (mode & FMODE_WRITE)
    {
        global->tx_act_timeout = local->tx_act_timeout;  
        global->tx_inact_timeout = local->tx_inact_timeout;
        global->tx_ring_len = local->tx_ring_len;
        global->tx_buffer_len = local->tx_buffer_len; 
        global->tx_mode = local->tx_mode;
        global->tx_desc_len = global->tx_ring_len /  global->tx_buffer_len;
    }
    global->allocation_size = local->allocation_size;
}

/**
 * ioctl_device - Perform IOCTL on this device
 * @filp:         File pointer
 * @cmd:          IOCTL number
 * @arg:          Argument of IOCTL command
 *
 * returns:       Returns result of IOCTL operation.
 *
 * This function perform IOCTL commands on this device.
 */
 long ioctl_device(struct file* filp, unsigned int cmd, unsigned long arg){

    struct dma_settings local_settings;
    int retval = 0;           //navratova hodnota
    int ret;
    u32 ptr_len;
    u32 i, j;
    u32 used_buffers;
    u32 free_buffers;
    unsigned long flags;
    struct mmaped_dma mm_req;
    struct driver_info *dev;
    struct dma_ring* desc;
    dev = (struct driver_info *)filp->private_data;
    
    switch(cmd){

        case START_RX_DMA:
        {
            if((filp->f_mode & FMODE_READ) && (dev->rx_state != RUN)){
                dev->rx_state = RUN;
                dev->rx_perf = 0;
                // Lock the RX lock. Init start of ring and length of unprocessed RX ring 
                // elements
                spin_lock_irqsave(&dev->rx_lock, flags);
                    dev->rx_ring_start = 0;
                    dev->rx_ring_len = 0;
                spin_unlock_irqrestore(&dev->rx_lock, flags);
                start_rx(dev);               
            }
            break;
        }   
        case START_TX_DMA:                 
        {
            if((filp->f_mode & FMODE_WRITE) && (dev->tx_state != RUN)){
                dev->tx_state = RUN;
                dev->tx_perf = 0;
                // Lock TX ring lock and init start index of TX ring and number of 
                // unprocessed buffers in the TX ring.
                spin_lock_irqsave(&dev->tx_lock, flags);
                    dev->tx_ring_start = 0;
                    dev->tx_ring_len = 0;
                spin_unlock_irqrestore(&dev->tx_lock, flags);
                start_tx(dev);               
            }
            break;
        }
        case STOP_RX_DMA:                  
        {
            if (filp->f_mode & FMODE_READ)
            {
                dev->rx_state = STOP;
                stop_rx(dev);
            }
            break;
        }   
        case STOP_TX_DMA:                  
        {
            if (filp->f_mode & FMODE_WRITE)
            {
                dev->tx_state = STOP;
                stop_tx(dev);
            }
            break;
        }
        case RESET_RX_DMA:                 
        {
            if (filp->f_mode & FMODE_READ)
            {
                dev->rx_state = STOP;
                stop_rx(dev);
                reset_rx(dev);
            }
            break;
        }
        case RESET_TX_DMA:                 
        {
            if (filp->f_mode & FMODE_WRITE)
            {
                dev->tx_state = STOP;
                stop_tx(dev);
                reset_tx(dev);
            }
            break;
        }   
        case GET_SETTINGS:
        {
            copy_to_user((struct dma_settings*)arg, &(dev->global_settings), sizeof(struct dma_settings));
            break;
        }   
        case SET_SETTINGS:
        {
            copy_from_user(&local_settings, (struct dma_settings*)arg, sizeof(struct dma_settings));
            
            //TODO:FIX
            if ((((filp->f_mode & FMODE_READ) && !(filp->f_mode & FMODE_WRITE) && (dev->rx_state != RUN)) || (!(filp->f_mode & FMODE_READ) && (filp->f_mode & FMODE_WRITE) && (dev->tx_state != RUN)) || ((filp->f_mode & FMODE_READ) && (filp->f_mode & FMODE_WRITE) && (dev->rx_state != RUN) && (dev->tx_state != RUN))) && !verify_settings(&local_settings))
            {
                update_settings(&dev->global_settings, &local_settings, filp->f_mode);
            }
            else
            {
                retval = -1;
            }
            break;
        }
        case RX_MMAP_GET:
        {
            // Get number of used buffers in RX ring
            used_buffers = get_used_buffers(dev);
    
            // If no buffers are present sleep in wait queue
            if (used_buffers == 0)
                wait_event_interruptible_timeout(dev->rx_perf_queue, get_used_buffers(dev) > 0, HZ/1000);
            
            // Lock RX ring lock and get start index of wait queue and number of used buffers
            spin_lock_irqsave(&dev->rx_lock, flags);
                mm_req.start = dev->rx_ring_start;
                mm_req.len = dev->rx_ring_len;
            spin_unlock_irqrestore(&dev->rx_lock, flags);
            
            // Copy data to user
            copy_to_user((struct mmaped_dma*)arg, &mm_req, sizeof(struct mmaped_dma));
            break;
        }
        case RX_MMAP_FREE:
        {
            // Copy data from user
            copy_from_user(&mm_req, (struct mmaped_dma*)arg, sizeof(struct mmaped_dma));
            
            // TODO: verify data in mm_req.start
            
            // Prepare descriptors for use
            for (i = mm_req.start; i < mm_req.start + mm_req.len; i++)
            {
                desc = &dev->rx_ring_ptr_v[i% (dev->global_settings.rx_desc_len)];
                // Clear status
                desc->status = 0;
                // Clear application fields
                desc->application[0] = 0;
                desc->application[1] = 0;
                desc->application[2] = 0;
                desc->application[3] = 0;
                desc->application[4] = 0;
                // Set size of RX buffer
                desc->control = dev->global_settings.rx_buffer_len;
            }
            // Lock the RX lock. Update start of ring, length of unprocessed RX ring 
            // elements and set new RX channel tail descriptor.
            spin_lock_irqsave(&dev->rx_lock, flags);
                dev->rx_ring_start = (dev->rx_ring_start + mm_req.len) % (dev->global_settings.rx_desc_len);
                dev->rx_ring_len -= mm_req.len;
                dma_out(dev, S2MM_TAILDESC_OFFSET, get_rx_desc_adress_p(dev, dev->rx_ring_start, dev->rx_ring_len  - 1));
            spin_unlock_irqrestore(&dev->rx_lock, flags);
            break;
        }
        case TX_MMAP_ALLOC:
        {
            // Copy data from user
            copy_from_user(&mm_req, (struct mmaped_dma*)arg, sizeof(struct mmaped_dma));
            
            // Check if number of requested buffers is bigger than number of TX ring buffers.
            if (mm_req.len > (dev->global_settings.tx_desc_len))
            {
                // Warn and return error code
                dev_warn(&dev->of_dev->dev, "Not enough memory in whole TX ring for write operation.\n");
                return -ENOMEM;
            }
            // Get number of free buffers
            free_buffers = get_free_buffers(dev);
            // If number of requested buffers is bigger than number of free buffers 
            // sleep and wait until we have sufficient number of free buffers
            if (free_buffers<mm_req.len)
                wait_event_interruptible_timeout(dev->tx_perf_queue, get_free_buffers(dev)>=mm_req.len, HZ/1000);
            // Lock TX ring lock and get start index of TX ring and number of 
            // unprocessed buffers in the TX ring.
            spin_lock_irqsave(&dev->tx_lock, flags);
                mm_req.start = dev->tx_ring_start + dev->tx_ring_len;
                mm_req.len = dev->global_settings.tx_desc_len - dev->tx_ring_len;
            spin_unlock_irqrestore(&dev->tx_lock, flags);
            // Copy data to user
            copy_to_user((struct mmaped_dma*)arg, &mm_req, sizeof(struct mmaped_dma));
            break;
        }
        case TX_MMAP_PUT:
        {
            // Copy data from user
            copy_from_user(&mm_req, (struct mmaped_dma*)arg, sizeof(struct mmaped_dma));
            
            for (i = mm_req.start; i < mm_req.len; i++)
            {
                desc = &dev->tx_ring_ptr_v[i% (dev->global_settings.tx_desc_len)];
                // Clear status
                desc->status = 0;
            }
            
            // Lock TX lock and update length of unprocessed buffers and set the tail 
            // descriptor of TX DMA channel
            spin_lock_irqsave(&dev->tx_lock, flags);
                dev->tx_ring_len += mm_req.len;
                dma_out(dev, MM2S_TAILDESC_OFFSET, get_tx_desc_adress_p(dev, mm_req.start, mm_req.len - 1));
            spin_unlock_irqrestore(&dev->tx_lock, flags);
            
            break;
        }
        case RX_PERF:
        {
            if((filp->f_mode & FMODE_READ) && (dev->rx_state == STOP)){
                dev_warn(&dev->of_dev->dev, "Start RX_PERF for %lu buffers.\n", arg);
                dev->rx_finished = (u32)arg;
                dev->rx_perf = 1;
                spin_lock_irqsave(&dev->rx_lock, flags);
                    dev->rx_ring_start = 0;
                spin_unlock_irqrestore(&dev->rx_lock, flags);
                start_rx(dev);
                if (dev->rx_finished  > 0)
                    while(dev->rx_finished  > 0)
                    {
                        ret = wait_event_interruptible_timeout(dev->rx_perf_queue, dev->rx_finished==0, HZ/100);
                        if (ret < 0)
                        {
                            retval = -1;
                            break;
                        }
                        else if (ret > 0)
                            break;
                    }
                stop_rx(dev);
                reset_rx(dev);
                dev->rx_perf = 0;
            }
            else
            {
                retval = -1;
            }
            break;
        }
        case TX_PERF:
        {
            if((filp->f_mode & FMODE_WRITE) && (dev->tx_state == STOP)){
                dev_warn(&dev->of_dev->dev, "Start TX_PERF for %lu buffers.\n", arg);
                dev->tx_finished = (u32) arg;
                dev->tx_perf = 1;

                start_tx(dev);
                ptr_len = dev->global_settings.tx_desc_len;
                for (j = 0; j < ptr_len; j++) {
                    dev->tx_ring_ptr_v[j].control |= DESC_SOF_EOF;
                    dev->tx_ring_ptr_v[j].status = 0;
                }
                
//             for (i = 0; i < dev->global_settings.tx_desc_len; i++)
//             {
//                 dev_warn(&dev->of_dev->dev, "TX Desc %d: Next: %x, Buff: %x, Ctrl: %x, Stat: %x\nApp:%x %x %x %x %x\n", i,dev->tx_ring_ptr_v[i].next, dev->tx_ring_ptr_v[i].buffer, dev->tx_ring_ptr_v[i].control, dev->tx_ring_ptr_v[i].status, dev->tx_ring_ptr_v[i].application[0], dev->tx_ring_ptr_v[i].application[1], dev->tx_ring_ptr_v[i].application[2], dev->tx_ring_ptr_v[i].application[3], dev->tx_ring_ptr_v[i].application[4]);
//             }

                spin_lock_irqsave(&dev->tx_lock, flags);
                    dev->tx_ring_start = 0;
                spin_unlock_irqrestore(&dev->tx_lock, flags);
                dma_out(dev, MM2S_TAILDESC_OFFSET, dev->tx_ring_ptr_p + (sizeof(struct dma_ring) * (ptr_len - 1)));
                dev_warn(&dev->of_dev->dev, "TX CURR: %x TAIL: %x\n", dma_in(dev, MM2S_CURDESC_OFFSET), dma_in(dev, MM2S_TAILDESC_OFFSET));
                wmb();
                if (dev->tx_finished  > 0)
                    while(dev->tx_finished  > 0)
                    {
                        ret = wait_event_interruptible_timeout(dev->tx_perf_queue, dev->tx_finished==0, HZ/100);
                        if (ret < 0)
                        {
                            retval = -1;
                            break;
                        }
                        else if (ret > 0)
                            break;
                    }
                
                stop_tx(dev);
                reset_tx(dev);
                dev->tx_perf = 0;
            }
            else
            {
                retval = -1;
            }
            break;
        }
        case GET_STATUS:
        {
            struct dma_status state;
            // Get DMA informations
            state.rx_control = dma_in(dev, S2MM_DMACR_OFFSET);
            state.rx_status = dma_in(dev, S2MM_DMASR_OFFSET);
            state.rx_current = dma_in(dev, S2MM_CURDESC_OFFSET);
            state.rx_tail = dma_in(dev, S2MM_TAILDESC_OFFSET);
            state.tx_control = dma_in(dev, MM2S_DMACR_OFFSET);
            state.tx_status = dma_in(dev, MM2S_DMASR_OFFSET);
            state.tx_current = dma_in(dev, MM2S_CURDESC_OFFSET);
            state.tx_tail = dma_in(dev, MM2S_TAILDESC_OFFSET);
            // Copy informations to user
            copy_to_user((struct dma_status*)arg, &(state), sizeof(struct dma_status));
            break;
        }  
        default:
        {
            dev_warn(&dev->of_dev->dev, "Bad IOCTL %u valid are %u to %u!\n", cmd,START_RX_DMA, TX_MMAP_PUT);
            dev_warn(&dev->of_dev->dev, "RX perf is %lu and TX perf is %lu!\n", RX_PERF, TX_PERF);
            retval = -1;
        }
    }

return retval;
}

/**
 * free_allocation - Free alocated parts of user allocated kernel buffer
 * @dev:        Driver info structure
 * @allocation: Allocation info structure
 * @count:      Number of continuous parts of memory area
 *
 * Free alocated parts of user allocated kernel buffer.
 */
 void free_allocation(struct driver_info * dev, struct allocation_info * allocation, u32 count)
{
    int i;
    for (i = 0; i < count; i++)
    {
        if (allocation->cacheable_buffers == 0)
        {
            dma_free_coherent(&dev->of_dev->dev, allocation->allocation_size, 
                              allocation->parts[i].virt, allocation->parts[i].phys);
        }
        else
        {
            if(1/*!IS_ENABLED(CONFIG_CMA)*/)
            {
                dma_unmap_single(&dev->of_dev->dev, allocation->parts[i].phys, allocation->allocation_size, DMA_BIDIRECTIONAL);
                kfree(allocation->parts[i].virt);
            }
            else
            {
                // From ARM dma_mapping.h
//                 struct page *page = pfn_to_page(dma_to_pfn(dev, allocation->parts[i].phys));
//                 if (PageHighMem(page))
//                 {
//                     __dma_free_remap(cpu_addr, size);
//                 }
//                 else
//                 {
//                     __dma_remap(page, size, pgprot_kernel);
//                 }
//                 dma_release_from_contiguous(dev, page, size >> PAGE_SHIFT);
                // From ARM dma_mapping.h
            }
        }
    }
}


/**
 * mmap_open - MMap open supplementary function.
 * @vma:  Virtual memory area structure
 *
 * MMap open supplementary function, just increment usage count of this memory 
 * area.
 */
static void mmap_open(struct vm_area_struct *vma)
{
    struct allocation_info * allocation;
    allocation = vma->vm_private_data;
    atomic_inc(&allocation->usage_count);
}


/**
 * mmap_close - MMap close supplementary function.
 * @vma:  Virtual memory area structure
 *
 * MMap open supplementary function. Decrement usage count of this memory area.
 * If usage count drops to zero, coresponding memory area and support structures 
 * are dealocated.
 */
static void mmap_close(struct vm_area_struct *vma)
{
    struct allocation_info * allocation;
    struct driver_info *dev;
    allocation = vma->vm_private_data;
    dev = allocation->dev;
//     printk("MMAP close started.\n", vma->vm_start);
//     printk("Value of usage_count before is %d\n", allocation->usage_count.counter);
    if (!atomic_dec_and_test(&allocation->usage_count))
        return;

    free_allocation(dev, allocation, allocation->count);
    kfree(allocation->parts);
    kfree(allocation);
//     printk("Memory area %x freed.\n", vma->vm_start);
}

/*
 * Structure describing implemented virtual memory operations
 */
static const struct vm_operations_struct mmap_vm_ops = {
        .open = mmap_open,
        .close = mmap_close,
};


/**
 * mmap_device - MMap some memory for user space.
 * @filp: File pointer
 * @vma:  Virtual memory area structure
 *
 * returns: Returns 0 if mmap finished OK and negative value othewise.
 *
 * MMap some memory for user space. RX and TX descriptor and buffer ring can be 
 * mmaped. This function can also be used for allocation and user mapping of 
 * continuous kernel buffers.
 */
 int mmap_device(struct file *filp, struct vm_area_struct *vma)
{
    u32 pos;
    unsigned long start;
    unsigned long size;
    u32 ret;
    struct driver_info *dev;
    u32 i;
    struct allocation_info * allocation;
    //printk("MMAP start\n");
    // Get divice driver structure
    dev = (struct driver_info *)filp->private_data;
    //printk("STAGE 0\n");
    // Get requested size
    size = (unsigned long) vma->vm_end - vma->vm_start;
    //printk("STAGE 1\n");
    start = vma->vm_start;
    //printk("STAGE 2\n");
    //dev_warn(&dev->of_dev->dev, "MMAP start: %x End: %x\n", vma->vm_start, vma->vm_end);
    //dev_warn(&dev->of_dev->dev, "MMAP PGOFF: %x\n", vma->vm_pgoff);
    //printk("STAGE 3\n");
    // MMap corect buffer
    switch(vma->vm_pgoff)
    {
        // MAP RX descriptor ring
        case MMAP_RX_DESC_KOFFSET:
            pos =  dev->rx_ring_ptr_p;
            // Check if size is corect and we are in right mode
            if (size > POINTER_SIZE || !(filp->f_mode & FMODE_READ))
                return -EINVAL;
            break;
        // MAP TX descriptor ring
        case MMAP_TX_DESC_KOFFSET:
            pos =  dev->tx_ring_ptr_p;
            // Check if size is corect and we are in right mode
            if (size > POINTER_SIZE || !(filp->f_mode & FMODE_READ))
                return -EINVAL;
            break;
        // MAP RX buffer ring
        case MMAP_RX_RING_KOFFSET:
            pos = dev->rx_ring_p;
            // Check if size is corect and we are in right mode
            if (size > MAX_RING || !(filp->f_mode & FMODE_WRITE))
                return -EINVAL;
            break;
        // MAP TX buffer ring
        case MMAP_TX_RING_KOFFSET:
            pos = dev->tx_ring_p;
            // Check if size is corect and we are in right mode
            if (size > MAX_RING || !(filp->f_mode & FMODE_WRITE))
                return -EINVAL;
            break;
        // Kernel Buffer userspace allocation
        case MMAP_ALLOC_KOFFSET:
            // Use http://lxr.free-electrons.com/source/drivers/char/mspec.c like operations
            allocation = (struct allocation_info *) kzalloc(sizeof(struct allocation_info), GFP_KERNEL);
            if (!allocation)
            {
                dev_err(&dev->of_dev->dev, "Could not allocate allocation info structure.\n");
                return -ENOMEM;
            }
            allocation->user_virt = vma->vm_start;
            if ((dev->global_settings.cacheable_buffers != 0)/* && (!IS_ENABLED(CONFIG_CMA))*/)
            {
                allocation->allocation_size = PAGE_SIZE;
            }
            else
            {
                allocation->allocation_size = dev->global_settings.allocation_size;
            }
            allocation->count = DIV_ROUND_UP(size, allocation->allocation_size);
            allocation->cacheable_buffers = dev->global_settings.cacheable_buffers;
            allocation->parts = (struct mem_info *) kzalloc(sizeof(struct mem_info)*allocation->count, GFP_KERNEL);
            if (!allocation->parts)
            {
                dev_err(&dev->of_dev->dev, "Could not allocate memory part array of size %d elements.\n", allocation->count);
                kfree(allocation);
                return -ENOMEM;
            }
            for (i = 0; i < allocation->count; i++)
            {
                if (dev->global_settings.cacheable_buffers != 0)
                {
                    if (1 /*!IS_ENABLED(CONFIG_CMA)*/)
                    {
                        allocation->parts[i].virt = (u8*) kzalloc(allocation->allocation_size, GFP_KERNEL);
                        allocation->parts[i].phys = dma_map_single(&dev->of_dev->dev, allocation->parts[i].virt, allocation->allocation_size, DMA_BIDIRECTIONAL);
                    }
                    else
                    {
                        // From ARM dma_mapping.h
//                         unsigned long order = get_order(allocation->allocation_size);
//                         size_t count = allocation->allocation_size >> PAGE_SHIFT;
//                         struct page *page;
//                         void *ptr;
//                 
//                         page = dma_alloc_from_contiguous(dev, count, order);
//                         if (!page)
//                                 return NULL;
//                 
//                         __dma_clear_buffer(page, size);
//                 
//                         if (PageHighMem(page)) {
//                                 ptr = __dma_alloc_remap(page, size, GFP_KERNEL, pgprot_kernel, __builtin_return_address(0));
//                                 if (!ptr) {
//                                         dma_release_from_contiguous(dev, page, count);
//                                         return NULL;
//                                 }
//                         } else {
//                                 __dma_remap(page, size, pgprot_kernel);
//                                 ptr = page_address(page);
//                         }
//                         allocation->parts[i].virt = ptr;
//                         allocation->parts[i].phys = pfn_to_dma(dev, page_to_pfn(page));

                        // End
                    }
                }
                else
                {
                    allocation->parts[i].virt = (u8*) dma_alloc_coherent(&dev->of_dev->dev, 
                                            allocation->allocation_size, &allocation->parts[i].phys,
                                            GFP_KERNEL | __GFP_ZERO);
                }
                if (!allocation->parts[i].virt)
                {
                    dev_err(&dev->of_dev->dev, "Could not allocate memory element of size %dB.\n", allocation->allocation_size);
                    free_allocation(dev, allocation, i);
                    kfree(allocation->parts);
                    kfree(allocation);
                    return -ENOMEM;
                }
            }
            vma->vm_flags |= VM_RESERVED;
            //vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
            if (dev->global_settings.cacheable_buffers == 0)
            {
                vma->vm_page_prot = pgprot_dmacoherent(vma->vm_page_prot);
            }
            //pgprot_val(vma->vm_page_prot) |= pgprot_val(PAGE_SHARED);
            for (i = 0; i < allocation->count; i++)
            {
                pos = allocation->parts[i].phys >>PAGE_SHIFT;
                ret = remap_pfn_range(vma, start, pos, allocation->allocation_size, vma->vm_page_prot);
                if (ret)
                {
                    dev_warn(&dev->of_dev->dev, "Mmap allocation failed: remap_pfn_range() returned %d!\n", ret);
                    free_allocation(dev, allocation, allocation->count);
                    kfree(allocation->parts);
                    kfree(allocation);
                    return -EAGAIN;
                }
                start += allocation->allocation_size;
            }
            atomic_set(&allocation->usage_count, 1);
            allocation->dev = dev;
            vma->vm_private_data = allocation;
            vma->vm_ops = &mmap_vm_ops;
            INIT_HLIST_NODE(&allocation->list);
            hash_add(hash_allocation, &allocation->list, vma->vm_start);
            return 0;
            break;
        default:
            dev_warn(&dev->of_dev->dev, "Whole buffer/descriptor ring must be mapped!\n");
            return -EINVAL;
            break;
    }
    vma->vm_pgoff = 0;
    
    //printk("STAGE 4\n");
    vma->vm_flags |= VM_RESERVED;
    vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
    pgprot_val(vma->vm_page_prot) |= pgprot_val(PAGE_SHARED);
    //printk("STAGE 5\n");
    // Remap all pages of kernel buffer/ring to user space
    // TODO: PAGE_SHARED vs vma->vm_page_prot???
    
//     printk("Phys: %x PFN: %x phys TXD: %x TXD: %x\n", pos, virt_to_pfn(pos), __pa(dev->tx_ring_ptr_v), dev->tx_ring_ptr_v);
//     test_v = kzalloc(4096*sizeof(char), GFP_KERNEL);
//     test_p = __pa(test_v);
//     printk("Kmalloc Virt: %x Phys: %x\n", test_v, test_p);
//     kfree(test_v);
    ret = remap_pfn_range(vma, vma->vm_start, pos >> PAGE_SHIFT, size, vma->vm_page_prot);
    //printk("STAGE 6\n");
    if (ret)
    {
        dev_warn(&dev->of_dev->dev, "Mmap failed: remap_pfn_range() returned %d!\n", ret);
        return -EAGAIN;
    }
    // NOTE: This may be usefull as template if noncontinuous buffers/desc will
    // be used in future
//     while (size > 0) 
//     {
//         // Get physical addres of page determineted by pos
//         page = virt_to_pfn((void *)pos);
//         // Remap page to user addres space
//         if (remap_pfn_range(vma, start, page, PAGE_SIZE, PAGE_SHARED))
//         {
//             return -EAGAIN;
//         }
//         start+=PAGE_SIZE;
//         pos+=PAGE_SIZE;
//         size-=PAGE_SIZE;
//     }
    return 0; 
}

/* Populate the file_operations structure. This structure informs kernel what 
 * operations our device supports.
 */
struct file_operations dma_operations = {
//     .read              = read_device,    // Vector read operation, NOTE: we do it synchronously.
//     .write             = write_device,   // Vector write operation, NOTE: we do it synchronously.
    .aio_read          = readv_device,    // Vector read operation, NOTE: we do it synchronously.
    .aio_write         = writev_device,   // Vector write operation, NOTE: we do it synchronously.
    .unlocked_ioctl    = ioctl_device,    // IOCTL command handling
    .open              = open_device,     // Open device
    .release           = close_device,    // Close device
    .mmap              = mmap_device,     // MMap support
};

/**
 * sec6net_dma_of_probe - Initialize the driver
 * @of_dev: Platform device structure
 *
 * returns: Returns 0 if settings are correct and negative value othewise.
 *
 * This function initializes the driver.
 */
static int sec6net_dma_of_probe(struct platform_device *of_dev) {
    struct driver_info *dev;
    int ret = 0;
    // Alocate the driver control structure
    dev = (struct driver_info *)kzalloc(sizeof(struct driver_info), GFP_KERNEL);
    if (!dev)
    {
        dev_err(&of_dev->dev, "could not allocate driver info structure.\n");
        ret = -ENOMEM;
        goto nodev;
    }
    // Save the platform device structure
    dev->of_dev = of_dev;
    
    // Set the default value of ring settings
    dev->global_settings.rx_act_timeout = MAX_DELAY;
    dev->global_settings.rx_inact_timeout = MAX_DELAY;
    dev->global_settings.tx_act_timeout = MAX_DELAY;
    dev->global_settings.tx_inact_timeout = MAX_DELAY;
    dev->global_settings.rx_ring_len = MAX_RING;
    dev->global_settings.tx_ring_len = MAX_RING;
    dev->global_settings.rx_buffer_len = MIN_BUFFER;
    dev->global_settings.tx_buffer_len = MIN_BUFFER;
    dev->global_settings.rx_mode = RX_COPY;
    dev->global_settings.tx_mode = TX_COPY;
    dev->rx_state = STOP;
    dev->tx_state = STOP;
    dev->global_settings.rx_desc_len = dev->global_settings.rx_ring_len / dev->global_settings.rx_buffer_len;
    dev->global_settings.tx_desc_len = dev->global_settings.tx_ring_len / dev->global_settings.tx_buffer_len;
    dev->global_settings.allocation_size = PAGE_SIZE;
    
    dev_warn(&of_dev->dev, "Number of RX desc: %d Number of TX desc: %d.\n", dev->global_settings.rx_desc_len, dev->global_settings.tx_desc_len);
    // Initialize the waiting queues
    init_waitqueue_head(&dev->rx_perf_queue);
    init_waitqueue_head(&dev->tx_perf_queue);
    
    // Init allocation hash table
    hash_init(hash_allocation);

    // Map the DMA registers to kernel space
    dev->dma_regs = of_iomap(of_dev->dev.of_node, 0);
    if (!dev->dma_regs) {
        dev_err(&of_dev->dev, "could not map Axi DMA registers.\n");
        ret = -ENODEV;
        goto nodevice;
    }
    // Get RX IRQ number from device tree
    dev->rx_irq = irq_of_parse_and_map(of_dev->dev.of_node, 1);
    // Get TX IRQ number from device tree
    dev->tx_irq = irq_of_parse_and_map(of_dev->dev.of_node, 0);
    if ((!dev->rx_irq) || (!dev->tx_irq)) {
            dev_err(&of_dev->dev, "could not determine irqs\n");
            ret = -ENOMEM;
            goto iounmap;
        }
    
    // Create driver name
    scnprintf(dev->drv_name, 32, "%s-%d", DRIVER_NAME, dma_cnt++);
    
    // Request and run RX IRQ
    ret = request_irq(dev->rx_irq, dma_irqhandler_rx, 0, dev->drv_name, dev);
    if (ret)
    {
        dev_err(&dev->of_dev->dev, "Cannot request RX IRQ %d\n", dev->rx_irq);
        goto rx_irq_failure;
    }
    // Init RX ring spinlock
    spin_lock_init(&dev->rx_lock);
    dev_warn(&dev->of_dev->dev, "RX IRQ %d allocated.\n", dev->rx_irq);
        
    // Request and run TX IRQ
    ret = request_irq(dev->tx_irq, dma_irqhandler_tx, 0, dev->drv_name, dev);
    if (ret)
    {
        dev_err(&dev->of_dev->dev, "Cannot request TX IRQ %d\n", dev->tx_irq);
        goto tx_irq_failure;
    }
    // Init TX ring spinlock
    spin_lock_init(&dev->tx_lock);
    dev_warn(&dev->of_dev->dev, "TX IRQ %d allocated.\n", dev->tx_irq);
    // Alocate DMA rings
    ret = alocate_rings(dev);
    if (ret)
        goto ring_fail;
    
    // Alocate char device
    ret = alloc_chrdev_region(&(dev->region), 0, 1, dev->drv_name);
    if (ret)
       goto region_fail;
    
    // Create sys class
    dev->fc = class_create(THIS_MODULE, dev->drv_name);
    if(dev->fc == NULL){
        goto cc_fail;
    }
    
    // Create sys entry for this device
    if(device_create(dev->fc, NULL, dev->region, "%s", dev->drv_name) == NULL){
        goto dc_fail;}
    // Initialize character device
    cdev_init(&dev->device, &dma_operations);
    // Set character device owner
    dev->device.owner = THIS_MODULE;
    // Pair character device with device number
    ret = cdev_add(&dev->device, dev->region, 1);
    if (ret)
        goto cdev_fail;
    // Set driver date
    ret = dev_set_drvdata(&of_dev->dev, dev);
    if (ret)
        goto dsd_fail;
    // All is OK
    dev_warn(&of_dev->dev, "Sec6Net DMA driver probe finished OK.\n");
    return 0;

// Handle all error states
dsd_fail:
    cdev_del(&dev->device);
cdev_fail:
    device_destroy(dev->fc, dev->region);
dc_fail:
    class_destroy(dev->fc);
cc_fail:
    unregister_chrdev_region(dev->region, 1);
region_fail:
    free_rings(dev);
ring_fail:
    free_irq(dev->tx_irq, dev);
tx_irq_failure:
    free_irq(dev->rx_irq, dev);
rx_irq_failure:
iounmap:
    if (dev->dma_regs)
        iounmap(dev->dma_regs);
nodevice:
    kfree(dev);
nodev:
    return ret;
}

/**
 * sec6net_dma_of_remove - Remove device driver
 * @of_dev: Platform device structure
 *
 * returns: Returns 0 if settings are correct and negative value othewise.
 *
 * This function removes the driver.
 */
static int sec6net_dma_of_remove(struct platform_device *of_dev)
{
    // Get the driver info structure
    struct driver_info *dev = dev_get_drvdata(&of_dev->dev);
    // Free allocated RX IRQ
    free_irq(dev->rx_irq, dev);
    // Free allocated TX IRQ
    free_irq(dev->tx_irq, dev);
    // Remove device driver
    cdev_del(&dev->device);
    // Remove device driver info from sys
    device_destroy(dev->fc, dev->region);
    // Destroy device driver sys class
    class_destroy(dev->fc);
    // Unragister device number
    unregister_chrdev_region(dev->region, 1);
    // Free the DMA rings
    free_rings(dev);
    // Unmap the DMA registers
    if (dev->dma_regs)
        iounmap(dev->dma_regs);
    // Clear the driver data pointer
    dev_set_drvdata(&of_dev->dev, NULL);
    // Free the memory consumated by driver info structure
    kfree(dev);
    
    return 0;
}

/*
 * This structure sets methods for probing and removing driver and sets basic 
 * driver informations.
 */
static struct platform_driver sec6net_dma_of_driver = {
    .probe = sec6net_dma_of_probe,                // Driver init method
    .remove = sec6net_dma_of_remove,              // Driver remove method
    .driver = {                                   // Basic driver informations
    .name = DRIVER_NAME,                          // Driver name
    .owner = THIS_MODULE,                         // Owner of the driver 
    .of_match_table = sec6net_dma_of_match,       // Compatibility list of our driver
    },
};

/**
 * Lowlevel driver init.
 */
static int sec6net_dma_driver_init(void)
{
    return platform_driver_register(&sec6net_dma_of_driver);
}

/**
 * Lowlevel driver exit.
 */
static void sec6net_dma_driver_exit(void)
{
    platform_driver_unregister(&sec6net_dma_of_driver);
}

// Which function run on drver init
module_init(sec6net_dma_driver_init);
// Which function run on driver exit
module_exit(sec6net_dma_driver_exit);

// Various module informations
MODULE_AUTHOR("Vlastimil Kosar <ikosar@fit.vutbr.cz>");
MODULE_DESCRIPTION("Sec6net AXI DMA driver");
MODULE_LICENSE("GPL");
