/*
 * Sec6net XILINX AXI DMA driver. Header file.
 */

#ifndef SEC6NET_XAXIDMA_H
#define SEC6NET_XAXIDMA_H

#define MM2S_DMACR_OFFSET    0x00000000 /* MM2S DMA control */
#define MM2S_DMASR_OFFSET    0x00000004 /* MM2S DMA status */
#define MM2S_CURDESC_OFFSET  0x00000008 /* MM2S Current descriptor pointer */
#define MM2S_TAILDESC_OFFSET 0x00000010 /* MM2S Tail descriptor pointer */

#define S2MM_DMACR_OFFSET    0x00000030 /* S2MM DMA control */
#define S2MM_DMASR_OFFSET    0x00000034 /* S2MM DMA status */
#define S2MM_CURDESC_OFFSET  0x00000038 /* S2MM Current descriptor pointer */
#define S2MM_TAILDESC_OFFSET 0x00000040 /* S2MM Tail descriptor pointer */

#define STARTSTOP_MASK  0x00000001 /* Start/stop DMA channel */
#define RESET_MASK      0x00000004 /* Reset DMA engine */
#define COMPLETION_MASK 0x00001000 /* Completion interrupt */
#define IDELAY_MASK      0x00002000 /* Delay interrupt */
#define ERROR_MASK      0x00004000 /* Error interrupt */
#define ALL_MASK        0x00007000 /* All interrupts */
#define IDLE_MASK       0x00000002 /* DMA Channel is idle */
#define DESC_SOF        0x08000000 /* SOF in descriptor */
#define DESC_EOF        0x04000000 /* EOF in descriptor */
#define DESC_SOF_EOF    0x0C000000 /* Both SOF and EOF in descriptor */
#define LEN_MASK        0x007FFFFF /* Length mask for status and control */
#define DSC_COMPLETE    0x80000000 /* Descriptor is processed - complete */

#define DELAY_OF_ONE_MILLISEC       1000 /* Delay for DMA reset */

/* This structure defines DMA descriptor */
struct dma_ring {
    u32 next;           /* Physical address of next buffer descriptor */
    u32 res0;           /* Reserved */
    u32 buffer;         /* Physical address of buffer */
    u32 res1[3];        /* Reserved */
    u32 control;        /* Control word */
    u32 status;         /* Status word */
    u32 application[5]; /* Application atributes */
    u32 res2[3];        /* Reserved */
} __attribute__((aligned(64))); /* Descriptor must be aligned to 64B */

#define RING_ALIGNMENT    0x40 /* Descriptor must be aligned to 64B */

/* Settings of dma channels */
struct dma_settings {
    u32 rx_act_timeout;   /* RX Active timeout */
    u32 rx_inact_timeout; /* RX Inctive timeout */
    u32 tx_act_timeout;   /* TX Active timeout */
    u32 tx_inact_timeout; /* TX Inctive timeout */
    u32 rx_ring_len;      /* RX Ring length */
    u32 tx_ring_len;      /* TX Ring length */
    u32 rx_buffer_len;    /* RX Buffer length */
    u32 tx_buffer_len;    /* TX Buffer length */
    u32 rx_mode;          /* RX Channel settings */
    u32 tx_mode;          /* TX Channel settings */
    u32 rx_desc_len;      /* RX Descriptor ring length */
    u32 tx_desc_len;      /* TX Descriptor ring length */
    u32 allocation_size;  /* Allocation size for kernel buffer allocation 
                             form userspace */
    u32 cacheable_buffers;/* Make kernel buffers allocated from user space
                             cacheable. */
};

#define TX_COPY   0x00000000
#define TX_DIRECT 0x00000001
#define TX_MMAP   0x00000002

#define RX_COPY   0x80000000
#define RX_DIRECT 0x80000001
#define RX_MMAP   0x80000002

/* Driver information structure */
struct driver_info {
    struct cdev device;             /* Character device structure */
    void __iomem *dma_regs;         /* Virtual address of DMA registers */
    struct dma_ring *rx_ring_ptr_v; /* RX Descriptor ring virtual pointer */
    dma_addr_t rx_ring_ptr_p;       /* RX Descriptor ring physical pointer */ 
    struct dma_ring *tx_ring_ptr_v; /* TX Descriptor ring virtual pointer */
    dma_addr_t tx_ring_ptr_p;       /* TX Descriptor ring physical pointer */ 
    void *rx_ring_v;                /* RX Buffer space virtual pointer */
    dma_addr_t rx_ring_p;           /* RX Buffer space physical pointer */
    u32 rx_ring_start;              /* RX decriptor ring start index */
    u32 rx_ring_len;                /* RX descriptor ring length of used data */
    void *tx_ring_v;                /* TX Buffer space virtual pointer */
    dma_addr_t tx_ring_p;           /* TX Buffer space physical pointer */
    u32 tx_ring_start;              /* TX decriptor ring start index */
    u32 tx_ring_len;                /* TX descriptor ring length of used data */
    u32 rx_state;                   /* RX channel state */
    u32 tx_state;                   /* TX channel state */
    wait_queue_head_t rx_perf_queue;/* RX wait queue */
    wait_queue_head_t tx_perf_queue;/* TX wait queue */
    u32 rx_finished;                /* RX perf mode finished */
    u32 tx_finished;                /* TX perf mode finished */
    u32 rx_perf;                    /* RX perf mode indicator */
    u32 tx_perf;                    /* TX perf mode indicator */
    struct dma_settings global_settings; /* Settings of dma channels*/ 
    struct platform_device *of_dev; /* Platform device structure */
    int rx_irq;                     /* RX IRQ number */
    int tx_irq;                     /* TX IRQ number */
    dev_t region;                   /* DEV region */
    struct class *fc;               /* File class */
    spinlock_t rx_lock;             /* RX Lock */
    spinlock_t tx_lock;             /* TX Lock */
    spinlock_t read_lock;             /* RX Lock */
    spinlock_t write_lock;             /* TX Lock */
    char drv_name[32];              /* Driver name for SYSFS */
    struct dma_ring *zall_rx_ring_ptr_v; /* RX Descriptor ring virtual pointer */
    dma_addr_t zall_rx_ring_ptr_p;       /* RX Descriptor ring physical pointer */ 
};

/* MMaped ring start and end */
struct mmaped_dma {
    u32 start;           /* Start of ring */
    u32 len;             /* Length of ring */
};

/* Allocated continuous memory virtual and physical address */
struct mem_info {
    char       *virt;   /* Virtual kernel address */
    dma_addr_t phys;    /* Physical address */
};

/* Info about user allocated kernel buffers */
struct allocation_info {
    u32 allocation_size;      /* Size of single continuous memory area */
    u32 count;                /* Number of continuous memory areas */
    char* user_virt;          /* User space virtual address */
    struct mem_info *parts;   /* Description of continuous memory areas */
    struct hlist_node list;   /* Hlist node structure */
    atomic_t usage_count;     /* Number of user processes which map this area */
    struct driver_info *dev;  /* Driver info structure */
    u32 cacheable_buffers;    /* Make kernel buffers allocated from user space
                                 cacheable. */
};

/* Register values for RX and TX channels */
struct dma_status {
    u32 rx_control;     /* State of RX channel control register */
    u32 rx_status;      /* State of RX channel status register */
    u32 rx_current;     /* State of RX channel current descriptor register */
    u32 rx_tail;        /* State of RX channel tail descriptor register */
    u32 tx_control;     /* State of TX channel control register */
    u32 tx_status;      /* State of TX channel status register */
    u32 tx_current;     /* State of TX channel current descriptor register */
    u32 tx_tail;        /* State of TX channel tail descriptor register */
};

#define MAX_DELAY       0x000000FF /* Maximal delay for to active and inactive timeouts. */
#define THRES_MASK      0x00FF0000 /* Threshold mask */
#define DELAY_MASK      0xFF000000 /* Delay mask */
#define MAX_RING        0x00100000 /* Maximal length of each ring - 16MB*/
#define MAX_BUFFER      0x00100000 /* Maximal length of each buffer - same as max size of ring */
#define MIN_BUFFER      0x00000400 /* Minimal size of each buffer - 1kB */
#define POINTER_SIZE    0x00010000 /* Size of ring pointer array - 64kB */

#define DIR_RX 1 /* RX direction */
#define DIR_TX 2 /* TX direction */

// TODO: Rework
#define IOCTL_MAGIC 0xDE                      /* IOCTL magic number*/
#define START_RX_DMA  _IO(IOCTL_MAGIC, 0xA0)   /* Start RX DMA */
#define START_TX_DMA  _IO(IOCTL_MAGIC, 0xA1)   /* Start TX DMA */ 
#define STOP_RX_DMA   _IO(IOCTL_MAGIC, 0xA2)   /* Stop RX DMA */
#define STOP_TX_DMA   _IO(IOCTL_MAGIC, 0xA3)   /* Stop TX DMA */
#define RESET_RX_DMA  _IO(IOCTL_MAGIC, 0xA4)   /* Reset RX DMA */
#define RESET_TX_DMA  _IO(IOCTL_MAGIC, 0xA5)   /* Reset TX DMA */
#define GET_SETTINGS  _IOR(IOCTL_MAGIC, 0xA6, struct dma_settings *) /* Get DMA settings */
#define SET_SETTINGS  _IOW(IOCTL_MAGIC, 0xA7, struct dma_settings *) /* Set DMA settings */
#define RX_PERF       _IOW(IOCTL_MAGIC, 0xA8, int) /* Run DMA in RX performance mode */
#define TX_PERF       _IOW(IOCTL_MAGIC, 0xA9, int) /* Run DMA in TX performance mode */
#define RX_MMAP_GET   _IOR(IOCTL_MAGIC, 0xAA, struct mmaped_dma *) /* Get start position in RX ring 
                                                                    and length */
#define RX_MMAP_FREE  _IOW(IOCTL_MAGIC, 0xAB, struct mmaped_dma *) /* Inform driver that operations 
                                                                    on N items are finished, Ring 
                                                                    items can be reused.*/ 
#define TX_MMAP_ALLOC _IOR(IOCTL_MAGIC, 0xAC, struct mmaped_dma *) /* Get start position in TX ring 
                                                                    and number of free buffers */
#define TX_MMAP_PUT   _IOW(IOCTL_MAGIC, 0xAD, struct mmaped_dma *) /* Inform driver than N buffers 
                                                                    can be transmited */
#define GET_STATUS    _IOR(IOCTL_MAGIC, 0xAE, struct dma_status *) /* Get DMA settings */

#define DRIVER_NAME "sec6net_dma_driver" /* Driver name */

#define RUN 1  /* DMA channel is running */
#define STOP 0 /* DMA channel is stopped */

/* MMAP page offsets for mapping data to userspace: kernel variant */
#define MMAP_RX_DESC_KOFFSET 0x00000000 /* RX descriptor ring */
#define MMAP_TX_DESC_KOFFSET 0x00020000 /* TX decriptor ring */
#define MMAP_RX_RING_KOFFSET 0x00040000 /* RX buffer ring */
#define MMAP_TX_RING_KOFFSET 0x00060000 /* TX buffer ring */
#define MMAP_ALLOC_KOFFSET   0x00010000 /* Kernel Buffer userspace allocation */

/* MMAP page offsets for mapping data to userspace: kernel variant */
#define MMAP_RX_DESC_UOFFSET (0x00000000 * PAGE_SIZE) /* RX descriptor ring */
#define MMAP_TX_DESC_UOFFSET (0x00020000 * PAGE_SIZE) /* TX decriptor ring */
#define MMAP_RX_RING_UOFFSET (0x00040000 * PAGE_SIZE) /* RX buffer ring */
#define MMAP_TX_RING_UOFFSET (0x00060000 * PAGE_SIZE) /* TX buffer ring */
#define MMAP_ALLOC_UOFFSET   (0x00010000 * PAGE_SIZE) /* Kernel Buffer userspace allocation */

/* VM_RESERVED is not present in 3.7.0+ */
#ifndef VM_RESERVED
    #define VM_RESERVED (VM_DONTEXPAND | VM_DONTDUMP)
#endif

#define ALLOC_HASH_BITS 10

#define WTF_DELAY 1000

#endif /* SEC6NET_XAXIDMA_H */
