#!/bin/bash

###############################################################################
# File: runTest-remote.sh
# Desc: Run test on remote http server
# Author: Jan Drazil
###############################################################################

port=80                          #remote http port
tests="$(dirname $0)/tests"      #dir with local tests
startPath=`pwd`
timeout="2m"                     #timeout of remote script
repeat=1                         #count of test repeat
kill=0                           #don't send kill

### Process params
while getopts 'p:l:t:r:kh' opt; do
  case "$opt" in
    h)
      echo -e "Usage: runTest-remote [-p port] [-l local tests] [-r repeat] [-t timeout]"
      echo -e "                      [-k] <host> <remote tests root> [test filter] [...]"
      echo -e "Argument description:"
      echo -e "\thost\t\t address of remote http server"
      echo -e "\tremote tests root\t\t path to remote tests"
      echo -e "Options description:"
      echo -e "\t-p port\t\tRemote HTTP port"
      echo -e "\t-l local tests\tLocal directory with tests"
      echo -e "\t-r repeat\tCount of repeat tests"
      echo -e "\t-t timeout\tTimeout of tests (for format see man timeout(1))"
      echo -e "\t-k\t\tIf test timeout, then automaticly send"
      echo -e "\t\t\tfilename to remote host on port 9999"
      echo -e "\ttest filter\tFilters of tests"
      exit 0
      ;;
    p) 
      port="$OPTARG"
      if ! [[ "$port" =~ ^[0-9]+$ ]]; then
        echo "Port must be a number!" >&2
        exit 1
      fi
      ;;
    l)
      tests="$OPTARG"
      ;;
    t)
      timeout="$OPTARG"
      ;;
    k)
      kill=1
      ;;
    r)
      repeat="$OPTARG"
      if ! [[ "$repeat" =~ ^[0-9]+$ ]]; then
        echo "Repeat must be a number!" >&2
        exit 1
      fi
      ;;
  esac
done

### Check for local tests
if ! [ -d "$tests" ]; then
  echo "$tests doesn't exist!" >&2
  exit 1
fi

unprocParam=$(($# - ($OPTIND-1)))
if [ $unprocParam -lt 2  ]; then
  echo "Expecting at least two parameters. Run with -h for help"
  exit 1
elif [ $unprocParam -eq 2 ]; then
  fileFilter=".*"
else
  fileFilter=""
  for (( i=0; i < $(($# - ($OPTIND-1) - 2)); i++)); do
    eval "fileFilter=\"$fileFilter|(\$$(($OPTIND + 2 + $i)))\""
  done
  fileFilter="${fileFilter#|}"
fi

eval "host=\$$OPTIND"
eval "testPath=\$$(($OPTIND + 1))"

[[ "$testPath" != */ ]] && testPath="${testPath}/"
[[ "$testPath" != /* ]] && testPath="/${testPath}"

cd "$tests"
rm "${startPath}/wget.log" 1>/dev/null 2>&1 
exec 3<&0 ### Create copy of stdin
while read -d $'\0' testfile; do
  counter=0
  while [ $counter -lt $repeat ]; do
    echo -n "Test http://${host}:${port}${testPath}${testfile#./}: " >&2
    if [[ $timeout == "0" ]]; then
      timeout "$timeout" /usr/bin/time -vo "${testfile}.${counter}.time" wget -o "${testfile}.${counter}.log" -O "${testfile}.${counter}.out" "http://${host}:${port}${testPath}${testfile#./}"
    else
      /usr/bin/time -vo "${testfile}.${counter}.time" wget -o "${testfile}.${counter}.log" -O "${testfile}.${counter}.out" "http://${host}:${port}${testPath}${testfile#./}"
    fi
    if [[ $timeout != "0" && $? -eq 124 ]]; then
      if [ $kill -eq 1 ]; then
        lang=`basename ${testfile}`
        killStat=`netcat localhost 9999 <<< "$lang"`
        echo "TIMEOUT, kill: $killStat" >&2
        if [ "$killStat" == "FAILED" ]; then
          echo -n "To continue press enter..." >&2
          ### Waiting for user
          read discard <&3
        fi
      else
        ### Alert - timeout occured
        (
          for i in {1..5}; do
            echo -ne "\a"
            sleep 2
          done 
        ) &
        echo "TIMEOUT"
        echo -n "To continue press enter..." >&2
        ### Waiting for user
        read discard <&3
      fi
      echo "TIMEOUT!!!" > "${testfile}.${counter}.time"
    else
      if grep -q "200 OK" "${testfile}.${counter}.log"; then
        echo "OK" >&2
      else
        ### Alert - Failed
        (
          for i in {1..5}; do
            echo -ne "\a"
            sleep 2
          done 
        ) &
        echo "FAILED"
        echo -n "To continue press enter..." >&2
        ### Waiting for user
        read discard <&3
      fi
    fi
    counter=$(($counter + 1))
  done
done < <(find . -mindepth 1 -type f -print0 | grep -Evz "(\.out)|(\.time)|(\.log)\$" | grep -Ez "$fileFilter" )
