﻿#!/usr/bin/env python3

"""
  File: testResult.py
  Desc: Evaluate stats of tests
  Author: Jan Drazil
"""

import sys, io, os, re, pickle, pprint
from math import sqrt
from argparse import ArgumentParser

class BadFormat(Exception):
  pass
  
class TestTimeout(Exception):
  pass  

class GetResults:
  testExt = ["php", "py3.py", "py2.py", "lua"]
  realTime=True
  median=True
  execTime=False
  _timeEvaluated=False
  
  ##############################################################################
  
  def __init__(self, path, file=False, lang="*", category="*", test="*"):
    if(file):
      self.loadResults(path)
    else:
      self.processData(path, lang, category, test)
  
  ##############################################################################
  
  def processData(self, path, lang="*", category="*", test="*"):
    """
      Prepare data for evaluating
      path - path to tests
      lang - filter for languages
      category - filter for categories
      test - filter for test names
    """
    if not os.path.isdir(path):
      raise IOError
    
    #loop over all tests
    self._results=dict()
    self._categories=set()
    
    ### Language level
    for lDir in os.listdir(path):
      langPath = os.path.join(path, lDir)
      if(os.path.isdir(langPath) and (lang == "*" or lang == lDir)):
        self._results[lDir] = dict()
        
        ### Category level
        for cDir in os.listdir(langPath):
          categoryPath = os.path.join(langPath, cDir)
          if(os.path.isdir(categoryPath) and (category == "*" or category == cDir)):
            self._results[lDir][cDir]=[]
            self._categories.add(cDir)
            
            ### File level
            for tFile in os.listdir(categoryPath):
              testPath=os.path.join(categoryPath, tFile)
              testName,testExt = os.path.splitext(testPath)
              testName = os.path.basename(testName)
              if(os.path.isfile(testPath) and (test == "*" or testName.startswith(test))):
                addFile=False
                for ext in self.testExt:
                  if(os.path.basename(testPath).endswith("." + ext)):
                    addFile=True
                    testName=os.path.basename(testPath)[0:-(len(ext)+1)]
                    break;
                if(addFile):
                  self._results[lDir][cDir].append({"path": testPath, "name": testName, "time": None, "count": 0, "times": None})
    return None
    
  ##############################################################################
  
  def evalTests(self):
    """
      Evaluate data
    """
    for category in self._categories:
      for language in self._results:
       for test in self._results[language][category]:
        time=evalTestTime(test["path"], self.median, self.realTime, self.execTime)
        test["time"]=time[0]
        test["count"]=time[1]
        if(not self.median):
          test["stdDeviation"]=time[2]
        test["times"]=time[3]
    self._timeEvaluated=True
     

  ##############################################################################
  
  def evalCategory(self):
    """
      Summarize time in category
    """
    if(not self._timeEvaluated):
      self.evalTests()
    catEvaluated={}
    catTime={}
    for category in self._categories:
      sumTime=0
      catEvaluated[category]={}
      for language in self._results:
        langTime=0
        catEvaluated[category][language]={}
        for test in self._results[language][category]:
          if(test['time'] is None):
            continue
          langTime += test["time"]
        catEvaluated[category][language]["time"]=langTime
        sumTime += langTime
      
      if(sumTime != 0):
        for language in catEvaluated[category]:
          catEvaluated[category][language]["percent"]=(100*catEvaluated[category][language]["time"])/sumTime
        catTime[category] = sumTime
      else:
        catTime[category] = None

    return (catEvaluated, catTime)  
  
  ##############################################################################
  
  def evalLanguage(self):
    """
      Summarize time in language
    """
    if(not self._timeEvaluated):
      self.evalTests()
      
    langEvaluated={}
    sumTime = 0
    for language in self._results:
      langEvaluated[language] = {}
      langTime = 0
      for category in self._results[language]:
        for test in self._results[language][category]:
          if(test['time'] is None):
            continue
          langTime += test["time"]
      sumTime += langTime
      langEvaluated[language]["time"] = langTime
    if(sumTime != 0):
      for language in langEvaluated:
        langEvaluated[language]["percent"] = (100*langEvaluated[language]["time"])/sumTime
    else:
      sumTime = None
    
    return (langEvaluated, sumTime)
          
          
  ##############################################################################
  
  def writeResults(self, outputFile):
    """
      Write results into file
    """
    if(not self._timeEvaluated):
      self.evalTests()
    with open(outputFile, 'wb') as f:
      pickle.dump((self._categories, self._results), f, pickle.HIGHEST_PROTOCOL)
    
  ##############################################################################
  
  def loadResults(self, inputFile):
    """
      Load results from file
    """
    with open(inputFile, 'rb') as f:
      self._categories, self._results = pickle.load(f)
      self._timeEvaluated = True
################################################################################
  
def evalTestTime(testPath, median=False, realTime=True, execTime=False):
  """
    Return running time of test from time file
    median - if true return median of repeated test otherwise return arithmetic mean
    realTIme - if true use elapsed time otherwise use user and sys time for evaluating
  """
  testCount = 0
  timeoutCount = 0
  stdDeviation = 0
  times = []
  if execTime:
    ext="err"
  else:
    ext="time"
  while os.path.isfile(testPath + "." + str(testCount) + "." + ext):
    try:
      if(execTime):
        times.append(processExecTimeFile(testPath + "." + str(testCount) + "." + ext))
      else:
        times.append(processTimeFile(testPath + "." + str(testCount) + "." + ext, realTime))
    except BadFormat:
      sys.stderr.write("Bad time file format: " + testPath + "." + str(testCount) + ".time/err\n")
    except TestTimeout:
      timeoutCount += 1
    testCount += 1
  data=times.copy()
  if(len(times) == 0):
    sys.stderr.write("Test " + testPath + " don't have any result\n")
    return (None, 0)
  if median:
    times.sort()
    if( len(times) % 2 ): ### Odd count
      result = times[int((len(times)+1)/2) - 1]
    else: ### Even count
      result = (times[int(len(times)/2) - 1] + times[int(len(times)/2)])/2
  else:
    result=0
    for time in times:
      result += time
    result  /= len(times)
    for time in times:
      stdDeviation += (time - result)**2
    stdDeviation = sqrt(stdDeviation / (testCount-timeoutCount-1))
  return (result, testCount-timeoutCount, stdDeviation, data)
      
 
################################################################################

def processExecTimeFile(file):
  """"
    Return execution time from file in seconds
    file - input file for processing
  """
  timePattern=re.compile(r"^(-?[0-9]+\.[0-9]+)$")
  with open(file) as timeFile:
    for line in timeFile:
      if line.startswith("TIMEOUT!!!"):
        raise TestTimeout()
      time=timePattern.match(line)   #### Check for match?
      if time is not None:
        time=time.group(1)
        return float(time)
      else:
        raise BadFormat()
  raise BadFormat()
  
################################################################################
 
def processTimeFile(file, realTime=True):
  """
    Return time from file in seconds
    file - input file for processing
    realTime - if true use elapsed time otherwise use user and sys time for evaluating
  """
  if realTime:
    timePattern=re.compile(r"^\s*Elapsed \(wall clock\) time \(h:mm:ss or m:ss\):\s*(.*)", re.A | re.I)                                                                       
    timeFormat=re.compile(r"(?:(?:([0-9]+(?:\.[0-9]+)?)h)?\s*(?:([0-9]+(?:\.[0-9]+)?)m)?\s*([0-9]+(?:\.[0-9]+)?)s)|(?:(?:([0-9]+):)?([0-9]+):)?([0-9]+(?:\.[0-9]+)?)", re.A | re.I)
    occureCount=1
  else:
    timePattern=re.compile(r"^\s*(?:(?:System time \(seconds\):)|(?:User time \(seconds\):))\s*([.0-9]+)", re.A | re.I)
    timeFormat=re.compile(r"([0-9]+(?:\.[0-9]+)?)", re.A | re.I)
    occureCount=2
  resultTime=0
  with open(file) as timeFile:
    for line in timeFile:
      if line.startswith("TIMEOUT!!!"):
        raise TestTimeout()
      time=timePattern.match(line)
      if time is not None:
        time=timeFormat.match(time.group(1))
        if time is not None:
          time=time.groups()
          if realTime:
            if(time[2] != None):
              resultTime += num(time[0])*3600 + num(time[1])*60 + num(time[2])
            else:
              resultTime += num(time[3])*3600 + num(time[4])*60 + num(time[5])
          else:
            resultTime += num(time[0])
          
          occureCount -= 1
          if occureCount == 0:
            break;
        else:
          raise BadFormat()
  return resultTime;

################################################################################
    
def num(s):
  """
    Convert string to integer or float
  """
  if(s is None):
    return 0
  try:
      return int(s)
  except ValueError:
      return float(s)

        
################################################################################
if __name__ == "__main__":
  parser = ArgumentParser()
  parser.add_argument("path", help="Path to tests")
  parser.add_argument("-u", "--usertime", action="store_true", dest="userTime", help="If present than script use user and sys time for evaluating.", default=False)
  parser.add_argument("-e", "--exectime", action="store_true", dest="execTime", help="If present than script use execution time for evaluating.", default=False)
  parser.add_argument("-m", "--median", action="store_true", dest="median", help="If present than script use median of repeated tests", default=False)
  parser.add_argument("-d", "--dumpfile", dest="dumpFile", help="File for evaluated data in pickle format")
  parser.add_argument("-l", "--language", dest="language", help="Language filter", default="*")
  parser.add_argument("-c", "--category", dest="category", help="Category filter", default="*")
  parser.add_argument("-t", "--test", dest="test", help="Test name filter", default="*")
  parser.add_argument("--stdout", action="store_true", help="If present than result will be printed on stdout", default=False)
  opt = parser.parse_args()

  tests=GetResults(opt.path, False, opt.language, opt.category, opt.test)
  tests.median = opt.median
  tests.realTime = not opt.userTime
  tests.execTime = opt.execTime
  tests.evalTests()
  
  if(opt.dumpFile is not None):
    tests.writeResults(opt.dumpFile)
  if(opt.stdout):
    pprint.PrettyPrinter().pprint(tests._results)
    pprint.PrettyPrinter().pprint(tests.evalCategory())
    pprint.PrettyPrinter().pprint(tests.evalLanguage())