-- cockoo_hash.vhd: Cockoo hash with use of generic key length
-- Copyright (C) 2012 Brno University of Technology
-- Author(s): Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: cuckoo_hash.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;
use work.bmem_func.all;
use work.filter_core_v2_config.all;
-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------
entity cuckoo_hash is
  generic(
    -- width of key into hash table
    KEY_WIDTH           : integer := 296;
    -- width of data inside hash table
    DATA_WIDTH          : integer := 16;
    -- width of initialization vector
    IV_WIDTH            : integer := 32;
    -- number of items in one hash table
    ITEMS               : integer := 1024; -- must be power of 2
    -- number of parallel tables used with cockoo hashing algorithm (each has own hash function)
    TABLES              : integer := 3;
    -- Block Ram Type, only 1, 2, 4, 9, 18, 36 bits
    BRAM_TYPE           : integer := 18;
    -- Register inside key comparation
    CMP_REG             : boolean := false;
    -- Register after all crc engines
    CRC_REG             : boolean := false;
    -- Use internal register for key delay
    KEY_REG             : boolean := false
  );
  port(
    CLK           : in  std_logic;
    RESET         : in  std_logic;
    
    -- input interface
    KEY           : in  std_logic_vector(KEY_WIDTH-1 downto 0);
    KEY_DEL       : in  std_logic_vector(KEY_WIDTH-1 downto 0); -- valid exactly one tact after KEY_VLD, unused when KEY_REG is true and CRC_REG is true
    KEY_DEL2      : in  std_logic_vector(KEY_WIDTH-1 downto 0); -- valid exactly two tacts after KEY_VLD, unused when KEY_REG is true or CRC_REG is false
    KEY_FOUNDABLE : in  std_logic; 
    KEY_VLD       : in  std_logic;
    -- output interface
    DATA          : out std_logic_vector(DATA_WIDTH-1 downto 0);
    DATA_FOUND    : out std_logic;
    DATA_VLD      : out std_logic;
    -- config interface
    INIT_VECTORS  : in  std_logic_vector(TABLES*IV_WIDTH-1 downto 0);
    ADD_KEY       : in  std_logic_vector(KEY_WIDTH-1 downto 0);
    ADD_DATA      : in  std_logic_vector(DATA_WIDTH-1 downto 0);
    ADD_VLD       : in  std_logic;
    ADD_STOP      : in  std_logic;
    ADD_STOP_VLD  : in  std_logic;
    ADD_BUSY      : out std_logic;
    REM_KEY       : in  std_logic_vector(KEY_WIDTH-1 downto 0);
    REM_VLD       : in  std_logic;
    REM_BUSY      : out std_logic;
    DIR_ADDR      : in  std_logic_vector(log2(ITEMS)+log2(TABLES+1)-1 downto 0);
    DIR_RD        : in  std_logic; -- prednost pred CLR
    DIR_CLR       : in  std_logic;
    DIR_DRD       : out std_logic_vector(DATA_WIDTH+KEY_WIDTH downto 0);
    DIR_DRDY      : out std_logic
  );
end entity;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture arch of cuckoo_hash is
  constant ADDR_WIDTH      : integer := log2(ITEMS);
  constant MEM_WIDTH       : integer := DATA_WIDTH+KEY_WIDTH+1; 
  
  type iv_array_t  is array(0 to TABLES-1) of std_logic_vector(IV_WIDTH-1 downto 0);
  type crc_array_t is array(0 to TABLES-1) of std_logic_vector(15 downto 0);
  type key_array_t is array(0 to TABLES-1) of std_logic_vector(KEY_WIDTH-1 downto 0);
  type data_array_t is array(0 to TABLES-1) of std_logic_vector(DATA_WIDTH-1 downto 0);
  
  signal iv                : iv_array_t;
  signal crc               : crc_array_t;
  signal crc_del           : crc_array_t;
  signal key_vld_reg       : std_logic;
  signal key_foundable_reg : std_logic;
  signal key_foundable_reg2: std_logic;
  signal key_foundable_out : std_logic;
  signal key_init          : key_array_t;
  signal cfgkey_init       : key_array_t;
  signal mem_vld           : std_logic_vector(TABLES-1 downto 0);
  signal mem_outs          : std_logic_vector(TABLES*MEM_WIDTH-1 downto 0);
  signal reg_key           : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_key_help      : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal mem_key           : key_array_t;
  signal mem_key_cmp       : std_logic_vector(TABLES-1 downto 0);
  signal mem_data          : data_array_t;
  signal mem_data_reg      : data_array_t;
  signal mem_data_vld      : std_logic_vector(TABLES-1 downto 0);
  signal mem_data_vld_reg  : std_logic_vector(TABLES-1 downto 0);
  signal mem_found         : std_logic_vector(TABLES-1 downto 0);
  signal ireg              : std_logic_vector(MEM_WIDTH-1 downto 0);
  signal ireg_key          : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal cfg_key           : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal cfg_key_init      : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal ireg_key_cmp      : std_logic;
  signal ireg_data         : std_logic_vector(DATA_WIDTH-1 downto 0);
  signal ireg_data_reg     : std_logic_vector(DATA_WIDTH-1 downto 0);
  signal ireg_data_vld     : std_logic;
  signal ireg_data_vld_reg : std_logic;
  signal ireg_found        : std_logic;
  signal cfg_crc           : std_logic_vector(15 downto 0);
  signal cfg_crc_reg       : std_logic_vector(15 downto 0);
  signal add_correct       : std_logic;
  signal addrem_read       : std_logic;
  signal addrem_read_reg   : std_logic;
  signal add_write         : std_logic;
  signal rem_write         : std_logic;
  signal add_write_mem     : std_logic_vector(TABLES-1 downto 0);
  signal add_write_mema    : std_logic_vector(TABLES-1 downto 0);
  signal add_write_memb    : std_logic_vector(TABLES-1 downto 0);
  signal add_write_coll    : std_logic;
  signal add_read_vld      : std_logic_vector(TABLES-1 downto 0);
  signal add_cnt           : std_logic_vector(log2(TABLES+1)-1 downto 0);
  signal add_cnt_max       : std_logic;
  signal add_cnt_slow      : std_logic_vector(log2(TABLES+1)-1 downto 0);
  signal add_cnt_slow_max  : std_logic;
  signal add_cnt_sel       : std_logic_vector(log2(TABLES+1)-1 downto 0);
  signal drds_memb         : std_logic_vector(TABLES*MEM_WIDTH-1 downto 0);
  signal drds_memb_sel     : std_logic_vector(MEM_WIDTH-1 downto 0);
  signal add_drd_data_vld  : std_logic;
  signal swap_stop         : std_logic;
  signal swap_rerun        : std_logic;
  signal addr_memb         : std_logic_vector(ADDR_WIDTH-1 downto 0);
  signal read_memb         : std_logic_vector(TABLES-1 downto 0);
  signal write_memb        : std_logic_vector(TABLES-1 downto 0);
  signal dwr_memb          : std_logic_vector(MEM_WIDTH-1 downto 0);
  signal dir_mem_sel       : std_logic_vector(TABLES-1 downto 0);
  signal dir_ireg_sel      : std_logic;
  signal sig_rem_busy      : std_logic;
  signal rem_correct       : std_logic;
  signal rem_drd_key       : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal rem_drd_key_cmp   : std_logic;
  signal rem_key_cmp       : std_logic;
  signal rem_key_found     : std_logic;
  signal rem_end           : std_logic;
  signal rem_read_vld_in   : std_logic;
  signal add_read_vld_reg  : std_logic;
begin
  iv_division_gen : for i in 0 to TABLES-1 generate
    iv(i) <= INIT_VECTORS(IV_WIDTH*(i+1)-1 downto IV_WIDTH*i);
  end generate;
  
  key_init_gen : for i in 0 to TABLES-1 generate
    key_initializer_i : entity work.init_box (sbox)
      generic map (
        KEY_WIDTH  => KEY_WIDTH,
        INIT_WIDTH => IV_WIDTH,
        IV         => FILTER_IV_MAGIC((i+1)*32-1 downto i*32)
      ) port map (
        KEY        => KEY,
        SEED       => iv(i),
        INIT_KEY   => key_init(i)
      );
  end generate;
  
  internal_key_reg_gen : if KEY_REG=true generate
    internal_key_doublereg_gen : if CRC_REG=true generate
      reghelp_key_i : process(CLK)
      begin
        if CLK'event and CLK='1' then
          if RESET='1' then
            reg_key_help <= (others => '0');
          else 
            reg_key_help <= KEY;
          end if;
        end if;
      end process;
    end generate;
    internal_key_singlereg_gen : if CRC_REG=false generate
      reg_key_help <= KEY;
    end generate;
    reg_key_i : process(CLK)
    begin
      if CLK'event and CLK='1' then
        if RESET='1' then
          reg_key <= (others => '0');
        else 
          reg_key <= reg_key_help;
        end if;
      end if;
    end process;
  end generate;
  external_key_del_gen : if KEY_REG=false and CRC_REG=false generate
    reg_key <= KEY_DEL;
  end generate;
  external_key_del2_gen : if KEY_REG=false and CRC_REG=true generate
    reg_key <= KEY_DEL2;
  end generate;
  
  crc_gen : for i in 0 to TABLES-1 generate
    crc_i : entity work.crc16 (arch_ccitt)
      generic map (
        DATA_WIDTH => KEY_WIDTH
      ) port map (
        DI         => key_init(i),
        DO         => crc(i)
      );
  end generate;
  
  mem_gen : for i in 0 to TABLES-1 generate
    mem_i : entity work.DP_BMEM (behavioral)
      generic map(
        DATA_WIDTH     => MEM_WIDTH,
        ITEMS          => ITEMS,
        BRAM_TYPE      => BRAM_TYPE,
        WRITE_MODE_A   => "WRITE_FIRST",
        OUTPUT_REG     => false
      ) port map (
        RESET          => RESET,
  
        CLKA           => CLK,
        PIPE_ENA       => '1',
        REA            => key_vld_reg,
        WEA            => add_write_mema(i),
        ADDRA          => crc_del(i)(ADDR_WIDTH-1 downto 0),
        DIA            => dwr_memb,
        DOA_DV         => mem_vld(i),
        DOA            => mem_outs((i+1)*MEM_WIDTH-1 downto i*MEM_WIDTH),
  
        CLKB           => CLK,
        PIPE_ENB       => '1',
        REB            => read_memb(i),
        WEB            => write_memb(i),
        ADDRB          => addr_memb,
        DIB            => dwr_memb,
        DOB_DV         => add_read_vld(i),
        DOB            => drds_memb((i+1)*MEM_WIDTH-1 downto i*MEM_WIDTH)
      );
  end generate;
  with (swap_stop and (DIR_RD or DIR_CLR)) select 
    addr_memb <= DIR_ADDR(ADDR_WIDTH-1 downto 0) when '1', cfg_crc_reg(ADDR_WIDTH-1 downto 0) when others;
  with ((swap_stop and (DIR_RD or DIR_CLR)) or sig_rem_busy) select
    dwr_memb  <= (others => '0') when '1', ireg when others;
  dir_mem_cfg_gen : for i in 0 to TABLES-1 generate
    read_memb(i)  <= (dir_mem_sel(i) and DIR_RD and swap_stop) or (addrem_read_reg and not(swap_stop and (DIR_RD or DIR_CLR)) and not(add_correct or swap_rerun or rem_correct));
    write_memb(i) <= (dir_mem_sel(i) and not DIR_RD and DIR_CLR and swap_stop) or (add_write_memb(i) and not(swap_stop and (DIR_RD or DIR_CLR)));
  end generate;
  dir_mem_sel_i : process(DIR_ADDR)
  begin
    dir_mem_sel  <= (others => '0');
    dir_ireg_sel <= '1'; 
    for i in 0 to TABLES-1 loop
      if(conv_std_logic_vector(i,log2(TABLES+1))=DIR_ADDR(ADDR_WIDTH+log2(TABLES+1)-1 downto ADDR_WIDTH)) then
        dir_mem_sel(i) <= '1';
        dir_ireg_sel <= '0'; 
      end if;
    end loop;
  end process; 
  
  mem_out_division_gen : for i in 0 to TABLES-1 generate
    mem_key(i)      <= mem_outs((i+1)*MEM_WIDTH-1 downto (i+1)*MEM_WIDTH-KEY_WIDTH);
    mem_data(i)     <= mem_outs((i+1)*MEM_WIDTH-KEY_WIDTH-1 downto (i+1)*MEM_WIDTH-KEY_WIDTH-DATA_WIDTH);
    mem_data_vld(i) <= mem_outs(i*MEM_WIDTH);
    mem_found(i)    <= mem_data_vld_reg(i) and mem_key_cmp(i);
  end generate;
  ireg_key         <= ireg(MEM_WIDTH-1 downto MEM_WIDTH-KEY_WIDTH);
  ireg_data        <= ireg(MEM_WIDTH-KEY_WIDTH-1 downto MEM_WIDTH-KEY_WIDTH-DATA_WIDTH);
  ireg_data_vld    <= ireg(0);
  ireg_found       <= ireg_data_vld_reg and ireg_key_cmp;
  
  res_data_i : process(mem_found,mem_data_reg,ireg_data_reg)
    variable res_val : std_logic_vector(DATA_WIDTH-1 downto 0);
  begin
    res_val := ireg_data_reg;
    for i in 0 to TABLES-1 loop
      if mem_found(i)='1' then
        res_val := mem_data_reg(i);
      end if; 
    end loop;
    DATA <= res_val;
  end process; 
  
  res_found_i : process(mem_found,ireg_found,rem_key_cmp,rem_key_found)
    variable or_val : std_logic;
  begin
    or_val := ireg_found;
    for i in 0 to TABLES-1 loop
      or_val := or_val or mem_found(i);
    end loop;
    DATA_FOUND <= or_val and not(rem_key_cmp and rem_key_found) and key_foundable_out; 
  end process;
  
  ADD_BUSY <= ireg_data_vld;

  add_correct <= ADD_VLD and not ireg_data_vld;
  ireg_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        ireg <= (others => '0');
      elsif add_correct='1' then
        ireg <= ADD_KEY & ADD_DATA & '1';
      elsif add_write='1' then
        ireg <= drds_memb_sel;
      elsif  (dir_ireg_sel='1' and DIR_RD='0' and DIR_CLR='1') or (rem_write='1' and add_cnt_max='1') then
        ireg <= (others => '0'); 
      end if;
    end if;
  end process;
  drds_memb_mx_i : process(drds_memb,add_cnt_sel)
    variable res_val : std_logic_vector(MEM_WIDTH-1 downto 0);
  begin
    res_val := (others => '0');
    for i in 0 to TABLES-1 loop
      if add_cnt_sel = conv_std_logic_vector(i,add_cnt_sel'length) then
        res_val := drds_memb((i+1)*MEM_WIDTH-1 downto i*MEM_WIDTH);
      end if; 
    end loop;
    drds_memb_sel <= res_val;
  end process; 
  
  
  add_cnt_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        add_cnt <= (others => '0');
      elsif (add_cnt_max='1' and ((add_read_vld(0)='1' and sig_rem_busy='0')or(add_read_vld_reg='1' and sig_rem_busy='1'))) or add_correct='1' or swap_rerun='1' or rem_correct='1' then
        add_cnt <= (others => '0');
      elsif ((add_read_vld(0)='1' and sig_rem_busy='0')or(add_read_vld_reg='1' and sig_rem_busy='1')) and (swap_stop='0' or sig_rem_busy='1') then
        add_cnt <= add_cnt+1;
      end if;
    end if;
  end process;

  add_cnt_slow_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        add_cnt_slow <= (others => '0');
      elsif add_cnt_max='1' and ((add_read_vld(0)='1' and sig_rem_busy='0')or(add_read_vld_reg='1' and sig_rem_busy='1')) and swap_stop='0' then
        if add_cnt_slow_max='1' then
          add_cnt_slow <= (others => '0');
        else
          add_cnt_slow <= add_cnt_slow+1;
        end if;
      end if;
    end if;
  end process;
  process (add_cnt,add_cnt_slow) begin
    if (add_cnt     =conv_std_logic_vector(TABLES  ,add_cnt'length     )) then add_cnt_max     <='1'; else add_cnt_max     <='0'; end if;
    if (add_cnt_slow=conv_std_logic_vector(TABLES-1,add_cnt_slow'length)) then add_cnt_slow_max<='1'; else add_cnt_slow_max<='0'; end if;  
  end process; 
  
  addrem_read_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        addrem_read <= '0';
      else
        addrem_read <= add_correct or swap_rerun or (((add_read_vld(0) and not sig_rem_busy)or(add_read_vld_reg and sig_rem_busy)) and not addrem_read and not addrem_read_reg) or rem_correct;
      end if;
    end if;
  end process;
  add_write <=  not addrem_read_reg and not addrem_read and (add_read_vld(0) and ireg(0) and (not add_drd_data_vld or add_cnt_max) and not swap_stop and not sig_rem_busy);
  rem_write <=  not addrem_read_reg and not addrem_read and (add_read_vld_reg and sig_rem_busy and rem_drd_key_cmp);
  
  with (add_cnt_max) select
    add_cnt_sel <= add_cnt_slow when '1', add_cnt when others;
  
  cfg_key_gen : for i in 0 to KEY_WIDTH-1 generate
    cfg_key(i) <= (ireg_key(i) and not sig_rem_busy) or (REM_KEY(i) and sig_rem_busy);
  end generate;
  
  cfg_key_init_gen : for i in 0 to TABLES-1 generate
    cfg_key_initializer_i : entity work.init_box (sbox)
      generic map (
        KEY_WIDTH  => KEY_WIDTH,
        INIT_WIDTH => IV_WIDTH,
        IV         => FILTER_IV_MAGIC((i+1)*32-1 downto i*32)
      ) port map (
        KEY        => cfg_key,
        SEED       => iv(i),
        INIT_KEY   => cfgkey_init(i)
      );
  end generate;

  cfg_key_init_mx_i : process(cfgkey_init, add_cnt_sel)
  begin
    cfg_key_init <= (others => '0');
    for i in 0 to TABLES-1 loop
      if add_cnt_sel=conv_std_logic_vector(i,log2(TABLES+1)) then
        cfg_key_init <= cfgkey_init(i);
      end if;
    end loop;
  end process;
    
  cfg_crc_i : entity work.crc16 (arch_ccitt)
    generic map (
      DATA_WIDTH => KEY_WIDTH
    ) port map (
      DI         => cfg_key_init,
      DO         => cfg_crc
    );
  
  add_write_coll_i : process(cfg_crc_reg, crc_del, add_cnt_sel)
  begin
    add_write_coll <= '0';
    for i in 0 to TABLES-1 loop
      if (add_cnt_sel=conv_std_logic_vector(i,log2(TABLES+1))) and cfg_crc_reg(ADDR_WIDTH-1 downto 0)=crc_del(i)(ADDR_WIDTH-1 downto 0) then
        add_write_coll <= '1';
      end if;
    end loop;
  end process;
  add_write_mem_i : process(add_write,rem_write,add_cnt_sel,add_cnt_max,sig_rem_busy)
  begin
    add_write_mem <= (others => '0');
    for i in 0 to TABLES-1 loop
      if(conv_std_logic_vector(i,log2(TABLES+1))=add_cnt_sel) then
        add_write_mem(i) <= add_write or (rem_write and not add_cnt_max);
      end if;
    end loop;
  end process;
  add_write_memab_gen : for i in 0 to TABLES-1 generate
    add_write_mema(i) <= add_write_mem(i) and add_write_coll;
    add_write_memb(i) <= add_write_mem(i) and not add_write_coll;
  end generate;
  
  drd_vld_cnt_mx : process(drds_memb, ireg_key, add_cnt_sel)
  begin
    add_drd_data_vld <= '0';
    for i in 0 to TABLES-1 loop
      if add_cnt_sel=conv_std_logic_vector(i,log2(TABLES+1)) then
        add_drd_data_vld <= drds_memb(i*MEM_WIDTH);
      end if;
    end loop;
  end process;
  
  drd_key_cnt_mx : process(drds_memb, ireg_key, add_cnt_sel)
  begin
    rem_drd_key <= ireg_key;
    for i in 0 to TABLES-1 loop
      if add_cnt=conv_std_logic_vector(i,log2(TABLES+1)) then
        rem_drd_key      <= drds_memb((i+1)*MEM_WIDTH-1 downto i*MEM_WIDTH+1+DATA_WIDTH);
      end if;
    end loop;
  end process;
  
  swap_stop_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        swap_stop <= '0';
      elsif ADD_STOP_VLD='1' then
        swap_stop <= ADD_STOP;
      end if;
    end if;
  end process;
  swap_rerun <= (ADD_STOP_VLD and not ADD_STOP and swap_stop) or rem_end;
  
  dir_drdy_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      DIR_DRDY <= DIR_RD and swap_stop;
    end if;
  end process;
  
  dir_drd_i : process(add_read_vld,drds_memb,ireg)
  begin
    DIR_DRD <= ireg;
    for i in 0 to TABLES-1 loop
      if add_read_vld(i)='1' then
        DIR_DRD <= drds_memb((i+1)*MEM_WIDTH-1 downto i*MEM_WIDTH);
      end if;
    end loop;
  end process;
  
  sig_rem_busy_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        sig_rem_busy <= '0';
      elsif rem_correct='1' then
        sig_rem_busy <= '1';
      elsif rem_end='1' then
        sig_rem_busy <= '0';
      end if;        
    end if;
  end process;
  rem_correct <= REM_VLD and not sig_rem_busy;
  rem_end     <= sig_rem_busy and add_read_vld_reg and not addrem_read_reg and not addrem_read and add_cnt_max;
  REM_BUSY <= sig_rem_busy;

  cmp_reg_true_gen : if CMP_REG=true generate
    reg_cmp_i : entity work.reg_cmp
      generic map (
        DATA_WIDTH => KEY_WIDTH
      ) port map (
        CLK     => CLK,
        RESET   => RESET,
        A       => REM_KEY,
        B       => rem_drd_key,
        VLD     => rem_read_vld_in,
        RES     => rem_drd_key_cmp,
        RES_VLD => add_read_vld_reg 
      );
    rem_read_vld_in <= add_read_vld(0) and sig_rem_busy and not addrem_read and not addrem_read_reg;
    mem_outs_reg_gen : for i in 0 to TABLES-1 generate
      mem_cmp_i : entity work.reg_cmp
        generic map (
          DATA_WIDTH => KEY_WIDTH
        ) port map (
          CLK     => CLK,
          RESET   => RESET,
          A       => mem_key(i),
          B       => reg_key,
          VLD     => mem_data_vld(i),
          RES     => mem_key_cmp(i),
          RES_VLD => mem_data_vld_reg(i) 
        );
      mem_data_reg_i : process(CLK)
      begin
        if CLK'event and CLK='1' then
          if RESET='1' then
            mem_data_reg(i) <= (others => '0');
          else
            mem_data_reg(i) <= mem_data(i);
          end if;
        end if;
      end process;
    end generate;
    ireg_cmp_i : entity work.reg_cmp
      generic map (
        DATA_WIDTH => KEY_WIDTH
      ) port map (
        CLK     => CLK,
        RESET   => RESET,
        A       => ireg_key,
        B       => reg_key,
        VLD     => ireg_data_vld,
        RES     => ireg_key_cmp,
        RES_VLD => ireg_data_vld_reg 
      );
    rem_cmp_i : entity work.reg_cmp
      generic map (
        DATA_WIDTH => KEY_WIDTH
      ) port map (
        CLK     => CLK,
        RESET   => RESET,
        A       => REM_KEY,
        B       => reg_key,
        VLD     => sig_rem_busy,
        RES     => rem_key_cmp,
        RES_VLD => rem_key_found 
      );
    ireg_data_reg_i : process(CLK)
    begin
      if CLK'event and CLK='1' then
        if RESET='1' then
          ireg_data_reg <= (others => '0');
        else
          ireg_data_reg <= ireg_data;
        end if;
      end if;
    end process; 
    data_vld_reg_i : process(CLK)
    begin
      if CLK'event and CLK='1' then
        if RESET='1' then
          DATA_VLD <= '0';
          key_foundable_out <= '0';
        else
          DATA_VLD <= mem_vld(0);
          key_foundable_out <= key_foundable_reg2;
        end if;
      end if;
    end process; 
  end generate; 
  cmp_reg_false_gen : if CMP_REG=false generate
    add_read_vld_reg <= add_read_vld(0) and sig_rem_busy and not addrem_read and not addrem_read_reg;
    process (REM_KEY,rem_drd_key) begin
      if (REM_KEY=rem_drd_key) then rem_drd_key_cmp<='1'; else rem_drd_key_cmp<='0'; end if;
    end process;
    mem_outs_noreg_gen : for i in 0 to TABLES-1 generate
      process (mem_key,reg_key) begin
        if (mem_key(i)=reg_key) then mem_key_cmp(i)<='1'; else mem_key_cmp(i)<='0'; end if;
      end process;
    end generate;
    process (ireg_key,reg_key) begin
      if (ireg_key=reg_key) then ireg_key_cmp<='1'; else ireg_key_cmp<='0'; end if;
    end process;
    process (REM_KEY,reg_key) begin
      if (REM_key=reg_key) then rem_key_cmp<='1'; else rem_key_cmp<='0'; end if;
    end process;
    mem_data_reg      <= mem_data;
    mem_data_vld_reg  <= mem_data_vld;
    ireg_data_reg     <= ireg_data;
    ireg_data_vld_reg <= ireg_data_vld;
    rem_key_found     <= sig_rem_busy;
    DATA_VLD          <= mem_vld(0);
    key_foundable_out <= key_foundable_reg2;
    
  end generate;
  
  crc_reg_true_gen : if CRC_REG=true generate
    cfg_crc_reg_i : process(CLK)
    begin
      if CLK'event and CLK='1' then
        if RESET='1' then
          cfg_crc_reg      <= (others => '0');
          addrem_read_reg  <= '0';
        else
          cfg_crc_reg      <= cfg_crc;
          addrem_read_reg  <= addrem_read and not(add_correct or swap_rerun or rem_correct);     
        end if;
      end if;
    end process;
    reg_crc_gen : for i in 0 to TABLES-1 generate
      reg_crc_i : process(CLK)
      begin
        if CLK'event and CLK='1' then
          crc_del(i) <= crc(i);
        end if;
      end process;
    end generate;
    reg_key_vld_i : process(CLK)
    begin
      if CLK'event and CLK='1' then
        key_vld_reg        <= KEY_VLD;
        key_foundable_reg  <= KEY_FOUNDABLE;
        key_foundable_reg2 <= key_foundable_reg;
      end if;
    end process;
  end generate;
  crc_reg_false_gen : if CRC_REG=false generate 
    cfg_crc_reg      <= cfg_crc;
    addrem_read_reg  <= addrem_read and not(add_correct or swap_rerun or rem_correct);
    key_vld_reg       <= KEY_VLD;
    key_foundable_reg <= KEY_FOUNDABLE;
    fake_reg_crc_gen : for i in 0 to TABLES-1 generate
      crc_del(i) <= crc(i);
    end generate;
    reg_key_foundable_i : process(CLK)
    begin
      if CLK'event and CLK='1' then
        key_foundable_reg2 <= key_foundable_reg;
      end if;
    end process;
  end generate;
    
end architecture;