-- tree_config.vhd: Configurator of binary search tree over prefixes 
-- Copyright (C) 2013 Brno University of Technology
-- Author(s): Lukas Kekely <ikekely@fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: tree_config.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;
use WORK.tree_func.all;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture arch of tree_config is
  constant TREE_MEMORY_ADC_REG : boolean := false;
  type tree_item_t is record
    KEY     : std_logic_vector(KEY_WIDTH-1 downto 0);
    KEYLEN  : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
    ISEND   : std_logic;
    KEYVLD  : std_logic;
    DATA    : std_logic_vector(DATA_WIDTH-1 downto 0);
    DATALEN : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
    DATAVLD : std_logic;
  end record;
  signal in_item           : tree_item_t;
  signal in_addr           : std_logic_vector(TREE_STAGES-1 downto 0);
  signal in_we_key         : std_logic;
  signal in_we_data        : std_logic;
  signal in_en             : std_logic;
  signal out_item          : tree_item_t;
  signal out_item_keylen_fix : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
  signal out_item_rdy      : std_logic;
  
  signal cfg_key_mask : std_logic_vector(KEY_WIDTH-1 downto 0);
  
  signal new_item_start   : tree_item_t;
  signal new_item_end     : tree_item_t;
  signal new_item_sel     : tree_item_t;
  signal new_item_sel_keylen_fix : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
  signal new_item_vld     : std_logic;
  signal new_item_active  : std_logic;
  signal new_item_eq      : std_logic;
  signal new_item_eq_reg  : std_logic;
  signal new_item_cmp     : std_logic;
  signal new_item_cmp_reg : std_logic;
  
  signal change_running      : std_logic;
  signal change_is_add       : std_logic;
  signal change_valid_req    : std_logic;
  signal change_finish       : std_logic;
  signal change_state_idle   : std_logic;
  signal change_state_search : std_logic;
  signal change_state_wait   : std_logic;
  signal change_state_insert : std_logic;
  
  signal dir_running         : std_logic;
  signal dir_running_base    : std_logic;
  signal dir_is_clr          : std_logic;
  signal dir_valid_req       : std_logic;
  signal dir_finish          : std_logic;
  signal dir_clr_finish      : std_logic;
  signal dir_in_addr         : std_logic_vector(TREE_STAGES-1 downto 0);
  signal dir_in_en           : std_logic;
  signal dir_in_we           : std_logic;
  signal dir_clr_addr        : std_logic_vector(TREE_STAGES-1 downto 0);
  signal dir_clr_addr0       : std_logic;
  
  signal tree_full        : std_logic;
  signal tree_status_cnt  : std_logic_vector(TREE_STAGES-1 downto 0);

  signal search_state          : std_logic_vector(TREE_STAGES downto 0);
  signal search_state_new      : std_logic_vector(TREE_STAGES downto 0);
  signal search_state_diff     : std_logic_vector(TREE_STAGES downto 0);
  signal search_state_diff_new : std_logic_vector(TREE_STAGES downto 0);
  signal search_move_left      : std_logic;
  alias search_state_addr      : std_logic_vector(TREE_STAGES-1 downto 0) is search_state(TREE_STAGES downto 1);
  alias search_state_end       : std_logic is search_state(0);
  signal search_finish         : std_logic;
  signal search_ending         : std_logic;
  signal search_start          : std_logic;
  signal search_in_en          : std_logic;
  signal search_rdy_reg        : std_logic;
  signal search_rdy_reg2       : std_logic;
  signal search_invalid_change : std_logic;
  
  signal wait_finish           : std_logic;
  
  signal insert_wr_addr        : std_logic_vector(TREE_STAGES-1 downto 0);
  signal insert_wr_addr_start  : std_logic_vector(TREE_STAGES-1 downto 0);
  signal insert_rd_addr        : std_logic_vector(TREE_STAGES-1 downto 0);
  signal insert_rd_addr_start  : std_logic_vector(TREE_STAGES-1 downto 0);
  signal insert_addr_change    : std_logic_vector(TREE_STAGES-1 downto 0);
  signal insert_use_end        : std_logic;
  signal insert_start          : std_logic;
  signal insert_finish         : std_logic;
  signal insert_in_item        : tree_item_t;
  signal insert_in_addr        : std_logic_vector(TREE_STAGES-1 downto 0);
  signal insert_in_en          : std_logic;
  signal insert_in_we_key      : std_logic;
  signal insert_in_we_data     : std_logic;
  signal insert_prefix_level   : std_logic_vector(log2(KEY_WIDTH+1)-1 downto 0);
  signal insert_prefix_level0  : std_logic;
  signal insert_fix_data       : std_logic;
  signal insert_in_range       : std_logic;
  signal insert_use_add        : std_logic;
  signal insert_use_rem        : std_logic;
  signal insert_wait_end       : std_logic;
  signal insert_wait_end_reg   : std_logic;
  signal insert_end_add        : std_logic;
  signal insert_end_rem        : std_logic;
  signal insert_rdy_reg        : std_logic;
  signal insert_rdy_reg2       : std_logic; 
begin
  -- tree memory access simplifier
  tree_memory_adc_i : entity work.tree_config_memory_adc
    generic map (
      TREE_STAGES    => TREE_STAGES,
      KEY_WIDTH      => KEY_WIDTH,
      DATA_WIDTH     => DATA_WIDTH,
      MEM_REG        => TREE_MEMORY_ADC_REG,
      USE_STAGES     => USE_STAGES,
      USE_STAGE_LEAF => USE_STAGE_LEAF,
      USE_REGS       => USE_REGS
    ) port map (
      CLK           => CLK,
      RESET         => RESET,
      NODE_DI       => NODE_DI,
      NODE_DO       => NODE_DO,
      NODE_DRDY     => NODE_DRDY,
      NODE_ADDR     => NODE_ADDR,
      NODE_WE       => NODE_WE,
      NODE_EN       => NODE_EN,
      LEAF_DI       => LEAF_DI,
      LEAF_DO       => LEAF_DO,
      LEAF_DRDY     => LEAF_DRDY,
      LEAF_ADDR     => LEAF_ADDR,
      LEAF_WE       => LEAF_WE,
      LEAF_EN       => LEAF_EN,
      CFG_DI_KEY     => in_item.key, 
      CFG_DI_KEYLEN  => in_item.keylen,
      CFG_DI_ISEND   => in_item.isend,
      CFG_DI_KEYVLD  => in_item.keyvld,
      CFG_DI_DATA    => in_item.data,
      CFG_DI_DATALEN => in_item.datalen,
      CFG_DI_DATAVLD => in_item.datavld,
      CFG_ADDR       => in_addr,
      CFG_WE_KEY     => in_we_key,
      CFG_WE_DATA    => in_we_data,
      CFG_EN         => in_en,
      CFG_DO_KEY     => out_item.key,
      CFG_DO_KEYLEN  => out_item.keylen,
      CFG_DO_ISEND   => out_item.isend,
      CFG_DO_KEYVLD  => out_item.keyvld,
      CFG_DO_DATA    => out_item.data,
      CFG_DO_DATALEN => out_item.datalen,
      CFG_DO_DATAVLD => out_item.datavld,
      CFG_DRDY       => out_item_rdy,
      CFG_WE_KEY_SYNC=> insert_wait_end     
    );
  
  -- prefix length of the key into prefix mask
  keylen2keymask : process (CFG_KEYLEN)
  begin
    cfg_key_mask <= (others => '0');
    for i in 0 to KEY_WIDTH-1 loop
      if conv_std_logic_vector(KEY_WIDTH-i-1,log2(KEY_WIDTH))<=CFG_KEYLEN then
        cfg_key_mask(i) <= '1';
      end if;
    end loop;
  end process;
  
  -- new item registers
  new_item_record_part1_reg : process (CLK) -- filled from CFG interface
  begin
    if CLK'event and CLK='1' then
      if change_valid_req='1' then
        new_item_start.key     <= CFG_KEY and cfg_key_mask;
        new_item_end.key       <= CFG_KEY or (not cfg_key_mask);
        new_item_start.keylen  <= CFG_KEYLEN;
        new_item_start.data    <= CFG_DATA; -- not used during remove
      end if;
    end if;
  end process; 
  new_item_record_part2_reg : process (CLK) -- filled from search result
  begin
    if CLK'event and CLK='1' then
      if search_finish='1' then 
        new_item_end.data    <= out_item.data;
        new_item_end.datalen <= out_item.datalen;
        new_item_end.datavld <= out_item.datavld;
      end if;
    end if;
  end process;
  new_item_start_vld_reg : process (CLK) -- start record valid right after cfg request
  begin
    if CLK'event and CLK='1' then
      if RESET='1' or change_finish='1' then
        new_item_start.keyvld <= '0';
      elsif change_valid_req='1' then
        new_item_start.keyvld <= '1';
      end if;
    end if;
  end process;    
  new_item_end_vld_reg : process (CLK) -- end record valid only after search ends
  begin
    if CLK'event and CLK='1' then
      if RESET='1' or change_finish='1' then
        new_item_end.keyvld <= '0';
      elsif search_finish='1' then
        new_item_end.keyvld <= '1';
      end if;
    end if;
  end process;
  new_item_active_reg : process (CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' or change_finish='1' then
        new_item_active <= '0';
      elsif wait_finish='1' then
        new_item_active <= '1';
      end if;
    end if;
  end process;
  new_item_end.isend     <= '1';
  new_item_start.isend   <= '0';
  new_item_start.datavld <= '1';
  new_item_end.keylen    <= new_item_start.keylen;
  new_item_start.datalen <= new_item_start.keylen;
  new_item_vld           <= new_item_end.keyvld and new_item_start.keyvld;
  
  -- fixing and config interface connection
  FIX_KEY_START <= new_item_start.key;
  FIX_KEY_END   <= new_item_end.key;
  FIX_LEN       <= new_item_start.keylen;
  FIX_DATALEN   <= new_item_start.datalen when change_is_add='1' else new_item_end.datalen;
  FIX_DATA      <= new_item_start.data    when change_is_add='1' else new_item_end.data;
  FIX_FOUND     <= new_item_start.datavld when change_is_add='1' else new_item_end.datavld;
  FIX_ACTIVE    <= new_item_active;
  FIX_IS_ADD    <= change_is_add;
  CFG_BUSY      <= change_running or dir_running_base;
  CFG_FULL      <= tree_full;
  DIR_KEY       <= out_item.key;
  DIR_KEYLEN    <= out_item.keylen;
  DIR_ISEND     <= out_item.isend;
  DIR_KEYVLD    <= out_item.keyvld;
  DIR_DATA      <= out_item.data;
  DIR_DRDY      <= out_item_rdy and dir_running;
  
  -- add and remove running registers
  change_valid_req <= ((CFG_ADD_VLD xor CFG_REM_VLD) and not DIR_RD and not CFG_CLR_VLD) and not (change_running or dir_running_base) and not(CFG_ADD_VLD and tree_full);
  dir_valid_req    <= ((DIR_RD xor CFG_CLR_VLD) and not CFG_REM_VLD and not CFG_ADD_VLD) and not (change_running or dir_running_base);
  change_finish    <= insert_finish or (search_ending and search_invalid_change);
  dir_finish       <= dir_running and (out_item_rdy or dir_clr_finish);
  change_running   <= new_item_start.keyvld;
  dir_running      <= dir_running_base or dir_valid_req;
  dir_running_reg : process (CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' or dir_finish='1' then
        dir_running_base <= '0';
      elsif dir_valid_req='1' then
        dir_running_base <= '1';
      end if;
    end if;
  end process;
  change_is_add_reg : process (CLK)
  begin
    if CLK'event and CLK='1' then
      if change_valid_req='1' then
        change_is_add <= CFG_ADD_VLD;
      end if;
    end if;
  end process;  
  dir_is_clr_reg : process (CLK)
  begin
    if CLK'event and CLK='1' then
      if dir_valid_req='1' then
        dir_is_clr <= CFG_CLR_VLD;
      end if;
    end if;
  end process;
  
  -- tree fullness status
  tree_status_cnt(0) <= '0'; -- first record is reserved and each add/rem require 2 records
  tree_status_cnt_reg : process (CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' or dir_clr_finish='1' then
        tree_status_cnt(tree_status_cnt'length-1 downto 1) <= (others => '0');
      elsif insert_finish='1' then
        if change_is_add='1' then
          tree_status_cnt(tree_status_cnt'length-1 downto 1) <= tree_status_cnt(tree_status_cnt'length-1 downto 1)+1;
        else
          tree_status_cnt(tree_status_cnt'length-1 downto 1) <= tree_status_cnt(tree_status_cnt'length-1 downto 1)-1;
        end if;
      end if;
    end if;
  end process;
  tree_full <= '1' when tree_status_cnt(tree_status_cnt'length-1 downto 1)=(tree_status_cnt'length-1 downto 1 => '1') else '0';
  
  -- keeping actual state of change process
  change_state_idle    <= not new_item_start.keyvld;
  change_state_search  <= not new_item_end.keyvld   and new_item_start.keyvld;
  change_state_wait    <= not new_item_active       and new_item_end.keyvld;
  change_state_insert  <= new_item_active;
  
  -- record config comparator
  out_item_keylen_fix     <= out_item.keylen     xor (out_item.keylen'length-1     downto 0 => out_item.isend);
  new_item_sel_keylen_fix <= new_item_sel.keylen xor (new_item_sel.keylen'length-1 downto 0 => new_item_sel.isend);
  new_item_sel <= new_item_end when change_state_insert='1' and insert_use_end='1' else new_item_start;
  new_item_cmp <= '1' when (out_item.key&out_item.isend&out_item_keylen_fix)>=(new_item_sel.key&new_item_sel.isend&new_item_sel_keylen_fix) else '0';
  new_item_eq  <= '1' when (out_item.key&out_item.isend&out_item.keylen)=(new_item_sel.key&new_item_sel.isend&new_item_sel.keylen)  else '0';
  new_item_cmp_register : process(CLK)
  begin
    if CLK'event and CLK='1' then
      new_item_cmp_reg <= new_item_cmp;
      new_item_eq_reg  <= new_item_eq;
    end if;
  end process;
  
  -- control of input interface for tree memory
  in_item.key     <= insert_in_item.key; 
  in_item.keylen  <= insert_in_item.keylen;  
  in_item.isend   <= insert_in_item.isend;   
  in_item.keyvld  <= insert_in_item.keyvld and not dir_running;  
  in_item.data    <= insert_in_item.data;    
  in_item.datalen <= insert_in_item.datalen; 
  in_item.datavld <= insert_in_item.datavld and not dir_running; 
  in_addr    <= dir_in_addr when dir_running='1' else search_state_addr when change_state_search='1' else insert_in_addr;
  in_en      <= (dir_running      and dir_in_en) or (change_state_search and search_in_en) or (change_state_insert and insert_in_en);
  in_we_key  <= (dir_running_base and dir_in_we) or (change_state_insert and insert_in_we_key);
  in_we_data <= (dir_running_base and dir_in_we) or (change_state_insert and insert_in_we_data);
  
  -- SEARCH state
  search_state_reg : process (CLK)
  begin
    if CLK'event and CLK='1' then
      if change_valid_req='1' then
        search_state      <= '1'&(TREE_STAGES-1 downto 0 => '0');
        search_state_diff <= '1'&(TREE_STAGES-1 downto 0 => '0');
      elsif search_rdy_reg='1' then
        search_state      <= search_state_new;
        search_state_diff <= search_state_diff_new;  
      end if;
    end if;
  end process;
  search_invalid_change_reg : process (CLK)
  begin
    if CLK'event and CLK='1' then
      if change_valid_req='1' then
        search_invalid_change <= CFG_REM_VLD; -- default: ADD valid; REM invalid
      elsif new_item_eq_reg='1' and out_item.keyvld='1' and search_rdy_reg='1' then
        search_invalid_change <= change_is_add; -- upon equality: ADD invalidated; REM validated 
      end if;
    end if;
  end process;
  search_rdy_registering : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        search_rdy_reg  <= '0';
        search_rdy_reg2 <= '0';
        search_start    <= '0';
      else
        search_rdy_reg  <= out_item_rdy;
        search_rdy_reg2 <= search_rdy_reg;
        search_start    <= change_valid_req;
      end if;
    end if;
  end process;
  search_state_new      <= (search_state or search_state_diff_new) xor (search_state_diff and (TREE_STAGES downto 0 => search_move_left));
  search_state_diff_new <= '0' & search_state_diff(TREE_STAGES downto 1);
  search_ending         <= search_state_end and change_state_search and out_item_rdy;
  search_finish         <= search_ending and not search_invalid_change;
  search_move_left      <= new_item_cmp_reg or not out_item.keyvld;
  search_in_en          <= search_rdy_reg2 or search_start;
  
  -- WAIT state
  wait_finish <= change_state_wait and FIX_ACTIVATE; 
  
  -- INSERT state
  insert_wr_addr_reg : process (CLK)
  begin
    if CLK'event and CLK='1' then
      if wait_finish='1' then
        insert_wr_addr <= insert_wr_addr_start; 
      elsif insert_in_we_data='1' then
        insert_wr_addr <= insert_wr_addr+insert_addr_change;
      end if;
    end if;
  end process;
  insert_rd_addr_reg : process (CLK)
  begin
    if CLK'event and CLK='1' then
      if wait_finish='1' then
        insert_rd_addr <= insert_rd_addr_start; 
      elsif insert_rdy_reg='1' and insert_use_add='0' and in_item.keyvld='1' then
        insert_rd_addr <= insert_rd_addr+insert_addr_change;
      end if;
    end if;
  end process;
  insert_use_end_reg : process (CLK)
  begin
    if CLK'event and CLK='1' then
      if wait_finish='1' then
        insert_in_range <= '0';
        insert_end_add  <= '0'; 
      elsif (insert_use_rem='1' or insert_use_add='1') and insert_rdy_reg='1' then
        insert_in_range <= not insert_in_range;
        insert_end_add  <= insert_in_range;
      end if;
    end if;
  end process;
  insert_prefix_level_cnt : process (CLK)
  begin
    if CLK'event and CLK='1' then
      if (insert_use_rem='1' or insert_use_add='1') and insert_rdy_reg='1' then
        insert_prefix_level <= (insert_prefix_level'length-1 downto 1 => '0')&change_is_add;
      elsif out_item_rdy='1' then
        if out_item.isend='1' then
          insert_prefix_level <= insert_prefix_level-1;
        else
          insert_prefix_level <= insert_prefix_level+1;
        end if;
      end if;
    end if;
  end process;
  insert_rdy_registering : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        insert_rdy_reg      <= '0';
        insert_rdy_reg2     <= '0';
        insert_start        <= '0';
        insert_wait_end_reg <= '0';
      else
        insert_rdy_reg      <= out_item_rdy;
        insert_rdy_reg2     <= search_rdy_reg;
        insert_start        <= wait_finish;
        insert_wait_end_reg <= insert_wait_end or (insert_rdy_reg and insert_use_rem);
      end if;
    end if;
  end process;
  insert_wr_addr_start <= tree_status_cnt+2 when change_is_add='1' else search_state_addr+1;
  insert_rd_addr_start <= tree_status_cnt   when change_is_add='1' else search_state_addr+1;
  insert_prefix_level0 <= '1' when insert_prefix_level=(insert_prefix_level'length-1 downto 0 => '0') else '0';
  insert_in_item.key     <= new_item_sel.key     when insert_use_add='1' else out_item.key;
  insert_in_item.keylen  <= new_item_sel.keylen  when insert_use_add='1' else out_item.keylen;
  insert_in_item.isend   <= new_item_sel.isend   when insert_use_add='1' else out_item.isend;
  insert_in_item.keyvld  <= new_item_sel.keyvld  when insert_use_add='1' else out_item.keyvld;
  insert_in_item.data    <= new_item_sel.data    when insert_use_add='1' or insert_fix_data='1' else out_item.data;
  insert_in_item.datalen <= new_item_sel.datalen when insert_use_add='1' or insert_fix_data='1' else out_item.datalen;
  insert_in_item.datavld <= new_item_sel.datavld when insert_use_add='1' or insert_fix_data='1' else out_item.datavld;
  insert_in_addr         <= insert_wr_addr when insert_in_we_key='1' or insert_in_we_data='1' else insert_rd_addr;
  insert_in_we_key       <= insert_rdy_reg and not insert_use_rem;
  insert_in_we_data      <= (change_is_add and insert_rdy_reg) or (not change_is_add and insert_wait_end);
  insert_in_en           <= insert_start or insert_wait_end_reg or insert_in_we_key or insert_in_we_data; 
  insert_addr_change     <= (insert_addr_change'length-1 downto 1 => change_is_add)&'1'; -- -1 when change_is_add='1' else +1
  insert_finish          <= insert_wait_end and insert_end_add and (change_is_add or insert_end_rem);
  insert_fix_data        <= insert_prefix_level0 and insert_in_range and out_item.isend; 
  insert_use_end         <= insert_in_range xor change_is_add;
  insert_use_add         <= change_is_add     and (not new_item_cmp_reg or not out_item.keyvld);
  insert_use_rem         <= not change_is_add and (new_item_eq_reg and out_item.keyvld);
  insert_end_rem         <= '1' when tree_status_cnt=insert_wr_addr else '0';
  
  -- DIR access to the tree memory
  dir_clr_addr_reg : process (CLK)
  begin
    if CLK'event and CLK='1' then
      if dir_valid_req='1' then
        dir_clr_addr <= tree_status_cnt;
      elsif dir_clr_addr0='0' then
        dir_clr_addr <= dir_clr_addr-1;
      end if;
    end if;
  end process;
  dir_in_addr    <= DIR_ADDR when dir_valid_req='1' else dir_clr_addr;
  dir_in_en      <= dir_in_we or (dir_valid_req and DIR_RD);
  dir_in_we      <= dir_is_clr and not dir_clr_addr0;
  dir_clr_addr0  <= '1' when dir_clr_addr=(dir_clr_addr'length-1 downto 0 => '0') else '0';
  dir_clr_finish <= dir_running and dir_is_clr and dir_clr_addr0 and not dir_valid_req;
  
end architecture;
