/*
 * cfg_interface.sv: Tested design configuration interface
 * Copyright (C) 2013 Brno University of Technology
 * Author: Lukas Kekely <ikekely@fit.vutbr.cz>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 */
 
interface ConfigInterface #(int pKeyWidth=128, pLengthWidth=7, int pDataWidth=32) (input logic CLK, RESET);  
  
  // Interface signals 
  logic [pKeyWidth-1:0] key;
  logic [pLengthWidth-1:0] keylen;
  logic [pDataWidth-1:0] data;
  logic add_vld;
  logic rem_vld;
  logic clr_vld;
  logic busy;
  logic full;
  
  // Clocking blocks  
  clocking cb @(posedge CLK);
    output key, keylen, data, add_vld, rem_vld, clr_vld;
    input busy, full;
  endclocking: cb;

  // Control Modport
  modport dut (input key, input keylen, input data, input add_vld, input rem_vld, input clr_vld, output busy, output full);
  modport tb  (clocking cb);
  
  // While RESET is active, ADD_VLD is inactive
  property RESETADDVLD;
    @(posedge CLK) (RESET)|->!(add_vld); 
  endproperty   
  assert property (RESETADDVLD)
     else $error("ADD valid during reset.");
       
  // While RESET is active, REM_VLD is inactive
  property RESETREMVLD;
    @(posedge CLK) (RESET)|->!(rem_vld); 
  endproperty   
  assert property (RESETREMVLD)
     else $error("REM valid during reset.");
       
  // While RESET is active, CLR_VLD is inactive
  property RESETCLRVLD;
    @(posedge CLK) (RESET)|->!(clr_vld); 
  endproperty   
  assert property (RESETCLRVLD)
     else $error("CLR valid during reset.");
       
  // While ADD_VLD is active, CLR_VLD, REM_VLD, BUSY and FULL are inactive
  property ADDALONE;
    @(posedge CLK) (add_vld)|->!(clr_vld || rem_vld || busy); 
  endproperty   
  assert property (ADDALONE)
     else $error("ADD valid when it cannot be.");
  
  // While REM_VLD is active, CLR_VLD, ADD_VLD and BUSY are inactive
  property REMALONE;
    @(posedge CLK) (rem_vld)|->!(clr_vld || add_vld || busy); 
  endproperty   
  assert property (REMALONE)
     else $error("REM valid when it cannot be.");
       
  // While CLR_VLD is active, REM_VLD, ADD_VLD and BUSY are inactive
  property CLRALONE;
    @(posedge CLK) (clr_vld)|->!(rem_vld || add_vld || busy); 
  endproperty   
  assert property (CLRALONE)
     else $error("CLR valid when it cannot be.");

endinterface : ConfigInterface 
 
