/*
 * filter_rule.sv: One prefix rule in filter
 * Copyright (C) 2013 Brno University of Technology
 * Author: Lukas Kekely <ikekely@fit.vutbr.cz>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 */

class FilterRuleTransaction extends Transaction;
  
  static const byte unsigned mask[] = '{'h80, 'hC0, 'hE0, 'hF0, 'hF8, 'hFC, 'hFE, 'hFF};
   
  // Randomized transaction data
  rand byte unsigned key[];
  rand int unsigned keylen;
  rand byte unsigned data[];
  // Randomization parameters
  int keyBytes = 16;
  int dataBytes = 4;
  int maxLength = 128;
  
  // Constrains
  constraint c1 {
    key.size  == keyBytes;
    data.size == dataBytes;
    keylen    <  maxLength;
  };
  
  /*
   * Displays the current value of the transaction or data described by this
   * instance in a human-readable format on the standard output. Each line of
   * the output will be prefixed with the specified prefix. This method prints
   * the value returned by the psdisplay() method.
   */
  virtual function void display(string prefix = "");
    if (prefix != "")
    begin
      $write("---------------------------------------------------------\n");
      $write("-- %s\n",prefix);
      $write("---------------------------------------------------------\n");
    end     
    $write("Rule #%0d:", data_id);
    for (integer i=0; i < key.size; i++) begin
      if(keylen/8 == i)
        $write(" %x", key[i]&mask[keylen&7]);
      else if (keylen/8 > i)
        $write(" %x", key[i]);
      else
        $write(" 00");
    end
    $write(" / %0d  => ", keylen+1);
    for (integer i=0; i < data.size; i++) begin
      $write(" %x", data[i]);
    end
    $write("\n"); 
  endfunction : display
  
  //-- Copy ----------------------------------------------------------------- 
  // Copy constructor
  virtual function Transaction copy(Transaction to = null);
    FilterRuleTransaction tr;
    if (to == null)
      tr = new();
    else 
      $cast(tr, to);
    tr.key = key;
    tr.data = data;
    tr.keylen = keylen;
    tr.data_id = data_id;
    copy = tr;
  endfunction: copy
        
  // -- Compare --------------------------------------------------------------
  /* Compares the current value of the object instance with the current value
   * of the specified object instance, according to the specified kind.
   * Returns TRUE (i.e., non-zero) if the value is identical. If the value is
   * different, FALSE is returned and a descriptive text of the first 
   * difference found is returned in the specified stringvariable. The kind 
   * argument may be used to implement different comparison functions (e.g., 
   * full compare, comparison of rand properties only, comparison of all 
   * properties physically implemented in a protocol and so on.)
   */    
  virtual function bit compare(input Transaction to, output string diff, input int kind = -1);
    bit same = 1; // Suppose that are same
    FilterRuleTransaction tr;
    $cast(tr, to);
    same=compareKey(tr,diff);
    for (integer j=0; j < data.size; j++)
      if (data[j] != tr.data[j]) begin
        same = 0;
        $swrite(diff, "data[%0d] does not match (%x != %x)", j, data[j], tr.data[j]);
      end  
    compare = same;
  endfunction: compare 
  
  function bit coverRule(input FilterRuleTransaction to, output string diff);
    bit res = 1;
    int unsigned lenBytes = keylen >> 3;
    if (key.size != to.key.size) begin
      res = 0;
      $swrite(diff,"different type of rules");
    end else if (keylen > to.keylen) begin
      res = 0;
      $swrite(diff, "covering key prefix is longer (has smaller range)");
    end else begin
      for (integer j=0; j < lenBytes; j++)
        if (key[j] != to.key[j]) begin
          res = 0;
          $swrite(diff, "key[%0d] does not match (%x != %x)", j, key[j], to.key[j]);
        end
      if ((key[lenBytes]&mask[keylen&7]) != (to.key[lenBytes]&mask[keylen&7])) begin
        res = 0;
        $swrite(diff, "key[%0d] does not match (%x != %x with mask %x)", lenBytes, key[lenBytes], to.key[lenBytes], mask[keylen&7]);
      end
    end
    coverRule = res;    
  endfunction: coverRule
  
  function bit coveredByRule(input FilterRuleTransaction to, output string diff);
    coveredByRule = to.coverRule(this,diff);
  endfunction: coveredByRule
  
  function bit compareKey(input FilterRuleTransaction to, output string diff);
    compareKey = coverRule(to,diff) && coveredByRule(to,diff);
  endfunction: compareKey
  
  function void fromInputTransaction(InputTransaction it);
    key    = it.data;
    keylen = it.dataBytes*8-1;
    data   = {};
  endfunction 

endclass: FilterRuleTransaction