/*
 * in_driver.sv: Tested design input driver
 * Copyright (C) 2013 Brno University of Technology
 * Author: Lukas Kekely <ikekely@fit.vutbr.cz>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 */
 
class InputDriver #(int pDataWidth=32);

  string    inst;                          // Driver identification
  bit       enabled;                       // Driver is enabled
  bit       busy;                          // Driver is sending transaction
  tTransMbx transMbx;                      // Transaction mailbox
  DriverCbs cbs[$];                        // Callbacks list
  virtual InputInterface.tb #(pDataWidth) ifc;
  
  // Enable/Disable delays between transactions
  rand bit enTxDelay;   
    int txDelayEnable_wt         = 1; 
    int txDelayDisable_wt        = 3;

  // Delay between transactions
  rand int txDelay; 
    int txDelayLow          = 0;
    int txDelayHigh         = 3;
    
  // Constrains
  constraint cDelays {
    enTxDelay dist { 1'b1 := txDelayEnable_wt, 1'b0 := txDelayDisable_wt };
    txDelay inside { [txDelayLow:txDelayHigh] };
  }

  // -- Constructor ---------------------------------------------------------
  // Create driver object 
  function new ( string inst, tTransMbx transMbx, virtual InputInterface.tb #(pDataWidth) ifc );
    this.enabled     = 0;            // Driver is disabled by default
    this.busy        = 0;            // Driver is not busy by default   
    this.ifc         = ifc;          // Store pointer interface 
    this.transMbx    = transMbx;     // Store pointer to mailbox
    this.inst        = inst;         // Store driver identifier

    this.ifc.cb.data      <= 0;
    this.ifc.cb.vld       <= 0;
  endfunction: new  
  
  // -- Set Callbacks -------------------------------------------------------
  // Put callback object into List  
  function void setCallbacks(DriverCbs cbs);
    this.cbs.push_back(cbs);
  endfunction : setCallbacks
  
  // -- Enable Driver -------------------------------------------------------
  // Enable driver and runs driver process
  task setEnabled();
    enabled = 1; // Driver Enabling
    fork         
      run();     // Creating driver subprocess
    join_none;   // Don't wait for ending
  endtask : setEnabled
      
  // -- Disable Driver ------------------------------------------------------
  // Disable generator
  task setDisabled();
    enabled = 0; // Disable driver, after sending last transaction it ends
  endtask : setDisabled
  
  // -- Send Transaction ----------------------------------------------------
  // Send transaction to Frame Link interface
  task sendTransaction( InputTransaction transaction );
    Transaction tr;
    $cast(tr, transaction); 
    busy = 1;
      foreach (cbs[i]) 
        cbs[i].pre_tx(tr, inst);       
      if (enTxDelay) 
        repeat (txDelay) @(ifc.cb);
      sendData(transaction);
      ifc.cb.vld <= 0;
      foreach (cbs[i]) 
        cbs[i].post_tx(tr, inst);
    busy = 0;
  endtask : sendTransaction
  
  // -- Run Driver ----------------------------------------------------------
  // Take transactions from mailbox and generate them to interface
  task run( bit verbose = 0 );
    InputTransaction transaction;
    Transaction to;        
    @(ifc.cb);                       // Wait for clock   
    while (enabled) begin            // Repeat while enabled
      assert(randomize());           // Randomize rand variables
      transMbx.get(to);              // Get transaction from mailbox 
      $cast(transaction,to);               
      sendTransaction(transaction);  // Send Transaction
      if (verbose)
        transaction.display(inst);
    end
  endtask : run

  // -- Send transaction data -----------------------------------------------
  // Send transaction data
  task sendData(InputTransaction tr);
    for (int i=0,j=tr.data.size-1; i < pDataWidth; i+=8,j--)
      ifc.cb.data[i +: 8] <= tr.data[j];
    ifc.cb.vld          <= 1;
    @(ifc.cb);         // Send data
  endtask : sendData
     
endclass : InputDriver 