/*
 * out_monitor.sv: Tested design output monitor
 * Copyright (C) 2013 Brno University of Technology
 * Author: Lukas Kekely <ikekely@fit.vutbr.cz>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 */
 
class OutputMonitor #(int pDataWidth=32, int pLengthWidth=5);
    
  string  inst;                          // Monitor identification
  bit     enabled;                       // Monitor is enabled
  bit     busy;                          // Monitor is receiving transaction
  MonitorCbs cbs[$];                     // Callbacks list
  virtual OutputInterface.tb #(pDataWidth,pLengthWidth) ifc;
  
  // -- Constructor ---------------------------------------------------------
  function new ( string inst, virtual OutputInterface.tb #(pDataWidth,pLengthWidth) ifc );
    this.enabled     = 0;           // Monitor is disabled by default   
    this.busy        = 0;           // Monitor is not busy by default   
    this.ifc         = ifc;         // Store pointer interface 
    this.inst        = inst;        // Store driver identifier 
  endfunction

  // -- Set Callbacks -------------------------------------------------------
  // Put callback object into List 
  function void setCallbacks(MonitorCbs cbs);
    this.cbs.push_back(cbs);
  endfunction : setCallbacks

  // -- Enable Monitor ------------------------------------------------------
  // Enable monitor and runs monitor process
  task setEnabled();
    enabled = 1; // Monitor Enabling
    fork         
      run();     // Creating monitor subprocess
    join_none;   // Don't wait for ending
  endtask : setEnabled
      
  // -- Disable Monitor -----------------------------------------------------
  // Disable monitor
  task setDisabled();
    enabled = 0; // Disable monitor, after receiving last transaction
  endtask : setDisabled 
  
  // -- Run Monitor ---------------------------------------------------------
  // Receive transactions and send them for processing by call back
  task run();
    OutputTransaction transaction; 
    Transaction tr;
    while (enabled) begin              // Repeat in forewer loop
      transaction = new;               // Create new transaction object
      $cast(tr, transaction);
      foreach (cbs[i]) cbs[i].pre_rx(tr, inst);       
      receiveTransaction(transaction); // Receive Transaction
      if (!enabled) break;  // Necessary for not calling callback after monitor disabling
      #(0);                 // Necessary for not calling callback before driver
      foreach (cbs[i]) cbs[i].post_rx(tr, inst);
      busy = 0;
    end
  endtask : run
  
  // -- Wait for VLD --------------------------------------------------------
  // It waits until VLD is active
  task waitForVld();
    do begin
      if (!ifc.cb.vld)
        @(ifc.cb);
    end while (!ifc.cb.vld); //Detect valid
  endtask : waitForVld
  
  // -- Receive Transaction -------------------------------------------------
  // It receives Frame Link transaction to tr object
  task receiveTransaction(OutputTransaction tr);
    waitForVld(); // Wait for valid
    busy = 1;     // Monitor receives transaction
    tr.data = new[((pDataWidth-1)/8)+1];
    tr.data = {<< byte{ifc.cb.data}};
    tr.length = ifc.cb.length;
    tr.found = ifc.cb.found;
    @(ifc.cb);
  endtask : receiveTransaction
     
endclass : OutputMonitor 