/*
 * out_transaction.sv: Tested design output transaction format
 * Copyright (C) 2013 Brno University of Technology
 * Author: Lukas Kekely <ikekely@fit.vutbr.cz>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 */

class OutputTransaction extends Transaction;
   
  // Randomized transaction data
  rand byte unsigned data[];
  rand int unsigned length;
  rand bit found;
  
  // Randomization parameters
  int dataBytes = 4;
  int maxLength = 128;
  
  // Constrains
  constraint c1 {
    data.size == dataBytes;
    length    <  maxLength;
  };
  
  /*
   * Displays the current value of the transaction or data described by this
   * instance in a human-readable format on the standard output. Each line of
   * the output will be prefixed with the specified prefix. This method prints
   * the value returned by the psdisplay() method.
   */
  virtual function void display(string prefix = "");
    if (prefix != "") begin
      $write("---------------------------------------------------------\n");
      $write("-- %s\n",prefix);
      $write("---------------------------------------------------------\n");
    end
    $write("Transaction #%0d: ", data_id);
    if (found) begin
      $write("Found data 0x");
      for (integer i=0; i < data.size; i++)
        $write("%x ",data[i]);
      $write(" for prefix length %0d.\n", length+1);
    end else
      $write("Data not found.\n");
  endfunction : display
  
  //-- Copy ----------------------------------------------------------------- 
  // Copy constructor
  virtual function Transaction copy(Transaction to = null);
    OutputTransaction tr;
    if (to == null)
      tr = new();
    else 
      $cast(tr, to);
    tr.data = data;
    tr.length = length;
    tr.found = found;
    tr.data_id = data_id;
    copy = tr;
  endfunction: copy
        
  // -- Compare --------------------------------------------------------------
  /* Compares the current value of the object instance with the current value
   * of the specified object instance, according to the specified kind.
   * Returns TRUE (i.e., non-zero) if the value is identical. If the value is
   * different, FALSE is returned and a descriptive text of the first 
   * difference found is returned in the specified stringvariable. The kind 
   * argument may be used to implement different comparison functions (e.g., 
   * full compare, comparison of rand properties only, comparison of all 
   * properties physically implemented in a protocol and so on.)
   */    
  virtual function bit compare(input Transaction to, output string diff, input int kind = -1);
    bit same = 1; // Suppose that are same
    OutputTransaction tr;
    $cast(tr, to);
    if (found != tr.found) begin
      same = 0;
      $swrite(diff, "found flags do not match (%x != %x)", found, tr.found);
    end else if (found) begin
      if (length != tr.length) begin
        same = 0;
        $swrite(diff, "lengths do not match (%0d != %0d)", length, tr.length);    
      end else
        if (data.size != tr.data.size) begin
          same = 0;
          $swrite(diff,"data sizes do not match (%0d != %0d)", data.size, tr.data.size);
        end else
          for (integer j=0; j < data.size; j++)
            if (data[j] != tr.data[j]) begin
              same = 0;
              $swrite(diff, "data[%0d] do not match (%x != %x)", j, data[j], tr.data[j]);
            end  
    end
    compare = same;
  endfunction: compare 

endclass: OutputTransaction