/*
 * scoreboard.sv: Frame Link Unaligned Scoreboard
 * Copyright (C) 2013 Brno University of Technology
 * Author: Lukas Kekely <ikekely@fit.vutbr.cz>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 */

import sv_common_pkg::*;
import dut_io_pkg::*;
  
class ScoreboardDriverCbs extends DriverCbs;

  TransactionTable #(TR_TABLE_FIRST_ONLY) sc_table;
  FilterRuleSet                           ruleSet;
  
  // Create a class 
  function new (TransactionTable #(TR_TABLE_FIRST_ONLY) sc_table);
    this.sc_table = sc_table;
    this.ruleSet  = null;
  endfunction
  
  // Function is called before is transaction sended 
  // Allow modify transaction before is sended
  virtual task pre_tx(ref Transaction transaction, string inst);
  endtask
  
  // Function is called after is transaction sended 
  virtual task post_tx(Transaction transaction, string inst);
    OutputTransaction outTr;
    InputTransaction it;
    FilterRuleTransaction rt=new;
    if(ruleSet==null) begin
      outTr = new;    
      outTr.found = 0;
    end else begin
      $cast(it,transaction);
      rt.fromInputTransaction(it);
      outTr = ruleSet.lpm(rt);
    end
    outTr.data_id = transaction.data_id;
    sc_table.add(outTr);
  endtask

endclass : ScoreboardDriverCbs


class ScoreboardMonitorCbs extends MonitorCbs;

  TransactionTable #(TR_TABLE_FIRST_ONLY) sc_table;
  
  // Create a class 
  function new (TransactionTable #(TR_TABLE_FIRST_ONLY) sc_table);
    this.sc_table = sc_table;
  endfunction
  
  // Function is called after is transaction received (scoreboard)
  virtual task post_rx(Transaction transaction, string inst);
    bit status=0;
    sc_table.remove(transaction, status);
    if (status==0)begin
      $write("Unknown transaction received from monitor %d\n", inst);
      $timeformat(-9, 3, " ns", 8);
      $write("Time: %t\n", $time);
      transaction.display(); 
      sc_table.display();
      $stop;
    end;
  endtask

endclass : ScoreboardMonitorCbs



class Scoreboard;

  TransactionTable #(TR_TABLE_FIRST_ONLY) scoreTable;
  ScoreboardMonitorCbs  monitorCbs;
  ScoreboardDriverCbs   driverCbs;

  // Create a class 
  function new ();
    this.scoreTable = new;
    this.monitorCbs = new(scoreTable);
    this.driverCbs  = new(scoreTable);
  endfunction

  // Display 
  task display();
    scoreTable.display();
  endtask
  
  task setRuleSet(FilterRuleSet rs);
    driverCbs.ruleSet = rs;
  endtask
  
endclass : Scoreboard

