/*
 * test.sv: Automatic test
 * Copyright (C) 2013 Brno University of Technology
 * Author: Lukas Kekely <ikekely@fit.vutbr.cz>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 */

import sv_common_pkg::*;
import dut_io_pkg::*;
import test_pkg::*;

program TEST (
  input  logic         CLK,
  output logic         RESET,
  InputInterface.tb    IN,
  OutputInterface.tb   OUT,
  ConfigInterface.tb   CFG,
  DirrectInterface.tb  DIR
);

  Generator                                           inGen;
  InputTransaction                                    inTr;
  FilterRuleTransaction                               rule;
  FilterRuleSet                                       ruleSet;
  InputDriver #(KEY_WIDTH)                            inDrv;
  OutputMonitor #(DATA_WIDTH, LENGTH_WIDTH)           outMon;
  ConfigModule #(KEY_WIDTH, LENGTH_WIDTH, DATA_WIDTH) cfgMod;
  DirrectModule #(TREE_STAGES, KEY_WIDTH, LENGTH_WIDTH, DATA_WIDTH) dirMod;  
  Scoreboard                                          scrb;
  
  // String array diff
  function bit string_array_diff(string d1[], string d2[]);
    bit ret=0;
    for(int i=0; i<d1.size && i<d2.size; i++)
      if (d1[i]!=d2[i]) begin
        $write("diff error line %0d:\n + %s\n - %s\n",i,d1[i],d2[i]);
        ret=1;
      end
    if (d1.size!=d2.size) begin
      $write("diff error in sizes\n");
      ret=1;
    end
    return ret;
  endfunction
  
  // Config interface not connected
  task configUnconnected();
    CFG.cb.add_vld <= 0;
    CFG.cb.rem_vld <= 0;
    CFG.cb.clr_vld <= 0;
  endtask : configUnconnected
  
  // Dirrect memory access interface not connected
  task dirrectUnconnected();
    DIR.cb.rd <= 0;
  endtask : dirrectUnconnected 

  // Create Test Environment
  task createEnvironment();
    inGen = new("InputGenerator",0);
      inTr = new;
      inTr.dataBytes = (KEY_WIDTH-1)/8+1;
      inGen.blueprint  = inTr;        
    inDrv = new("InputDriver", inGen.transMbx, IN);  
      inDrv.txDelayEnable_wt  = INPUT_DRIVER_DELAY_ENABLE_WT;
      inDrv.txDelayDisable_wt = INPUT_DRIVER_DELAY_DISABLE_WT;
      inDrv.txDelayLow        = INPUT_DRIVER_DELAY_LOW;
      inDrv.txDelayHigh       = INPUT_DRIVER_DELAY_HIGH;
    outMon = new("OutputMonitor", OUT);  
    scrb = new;
      inDrv.setCallbacks(scrb.driverCbs);
      outMon.setCallbacks(scrb.monitorCbs);
    cfgMod = new("ConfigModule", CFG); //configUnconnected();
    dirMod = new("DirrectModule", DIR); //dirrectUnconnected();
    rule = new;
      rule.dataBytes = (DATA_WIDTH-1)/8+1;
      rule.keyBytes = (KEY_WIDTH-1)/8+1;
      rule.maxLength = KEY_WIDTH;
    ruleSet = new;
  endtask : createEnvironment

  // Resets design
  task resetDesign();
    RESET=1;                       // Init Reset variable
    #RESET_TIME     RESET = 0;     // Deactivate reset after reset_time
  endtask : resetDesign

  // Enable test Environment
  task enableTestEnvironment();
    inDrv.setEnabled();
    outMon.setEnabled();
  endtask : enableTestEnvironment

  // Wait for the end of current test
  task waitDisabled();     
    int i;
    #(10*CLK_PERIOD);
    i = 0;
    // Check if monitor is not receiving transaction for 100 CLK_PERIODs
    while (i<100) begin
      if (inDrv.busy || outMon.busy)
        i = 0;
      else 
        i++;
      #(CLK_PERIOD); 
    end 
  endtask

  // Disable test Environment
  task disableTestEnvironment(); 
    inDrv.setDisabled();
    outMon.setDisabled();
  endtask : disableTestEnvironment

  // Basic test
  task test1();
    $write("\n\n############ TEST TRANSACTIONS ############\n\n");
    inGen.setEnabled(TRANSACTION_COUNT);
    wait (inGen.enabled == 0);
    waitDisabled();
    scrb.display();
    $write("\n\n###########################################\n\n");
  endtask: test1
  
  // Basic config test
  task test2(bit precise = 0);
    bit status=0;
    int x;
    stringArray sa;
    ruleArray ra;
    $write("\n\n############ TEST CONFIGURATION ############\n\n");
    for(int i=0; i<RULE_COUNT; i++) begin // add random rules
      assert(rule.randomize);
      rule.data_id=i;
      cfgMod.addRule(rule);
      if(!cfgMod.checkFull())
        ruleSet.add(rule,status);
      if(precise) begin
        cfgMod.waitWhileBusy();  
        dirMod.readRules(sa);
        if(string_array_diff(sa,ruleSet.traverse())) begin
          ruleSet.display();
          $stop();
        end else
          $write("ADD(unique) #%0d finished OK\n",i);
      end
    end
    ra=ruleSet.traverseR();
    for(int i=0; i<RULE_COUNT/2; i++) begin // add existing rules
      ra[$urandom_range(ra.size-1,1)].copy(rule);
      foreach(rule.data[i])
        rule.data[i]=255;
      rule.data_id=RULE_COUNT+i;
      cfgMod.addRule(rule);
      if(precise) begin
        cfgMod.waitWhileBusy();  
        dirMod.readRules(sa);
        if(string_array_diff(sa,ruleSet.traverse())) begin
          ruleSet.display();
          $stop();
        end else
          $write("ADD(exist) #%0d finished OK\n",i);
      end
    end
    //ruleSet.display();
    cfgMod.waitWhileBusy();  
    dirMod.readRules(sa);
    if(string_array_diff(sa,ruleSet.traverse())) begin
      ruleSet.display();
      $stop();
    end else
      $write("ADD finished OK\n");
    for(int i=0; i<RULE_COUNT/2; i++) begin // try to remove random rules
      assert(rule.randomize);
      cfgMod.removeRule(rule);
      ruleSet.remove(rule,status);
      if(precise) begin
        cfgMod.waitWhileBusy();  
        dirMod.readRules(sa);
        if(string_array_diff(sa,ruleSet.traverse())) begin
          ruleSet.display();
          $stop();
        end else
          $write("REM(random) #%0d finished OK\n",i);
      end
    end
    for(int i=0; i<RULE_COUNT/2; i++) begin // remove one half of the actual rules
      ra[$urandom_range(ra.size-1,1)].copy(rule);
      cfgMod.removeRule(rule);
      ruleSet.remove(rule,status);
      if(precise) begin
        cfgMod.waitWhileBusy();  
        dirMod.readRules(sa);
        if(string_array_diff(sa,ruleSet.traverse())) begin
          ruleSet.display();
          $stop();
        end else
          $write("REM(actual) #%0d finished OK\n",i);
      end
    end
    //ruleSet.display();
    cfgMod.waitWhileBusy();  
    dirMod.readRules(sa);
    if(string_array_diff(sa,ruleSet.traverse()))
      $stop();
    else
      $write("REM finished OK\n");
    $write("\n\n############################################\n\n");
    scrb.setRuleSet(ruleSet);
  endtask: test2
  
  // Basic config test
  task test3();
    stringArray sa;
    $write("\n\n############ TEST CLEAR ############\n\n");
    cfgMod.clearRules();
    ruleSet = new;
    cfgMod.waitWhileBusy();  
    dirMod.readRules(sa);
    if(string_array_diff(sa,ruleSet.traverse()))
      $stop();
    else
      $write("CLR finished OK\n");
    $write("\n\n####################################\n\n");
    scrb.setRuleSet(ruleSet);
  endtask: test3
  
  // --------------------------------------------------------------------------
  //                           Main test part
  // --------------------------------------------------------------------------
  initial begin
    createEnvironment();
    resetDesign();
    enableTestEnvironment();
    test1();       // Run Tests
    test2();
    test1();
    test3();
    test1();
    disableTestEnvironment();
    $stop();       // Stop testing
  end

endprogram

