-- filter_core.vhd: Filtering engine of packet filter
-- Copyright (c) 2013 Brno University of Technology 
-- Author(s): Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: filter_core.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use work.math_pack.all; 
use work.filter_core_v2_config.all; 

architecture full of afilter_core_v2 is  
  constant FILTER_OFIFO_ITEMS          : integer := 64; -- 16, 32, 64
  constant FILTER_OFIFO_TYPE           : integer := 64; -- 16, 32, 64 
  function rtc2slv32(rtc : rule_type_config_t) return std_logic_vector is
    variable slv32 : std_logic_vector(31 downto 0); 
  begin
    slv32(7 downto 0)  := conv_std_logic_vector(rtc.tables,8);
    slv32(25 downto 8) := conv_std_logic_vector(rtc.items,18);
    if (rtc.key.ipv4_only) then
      slv32(26)        := '1';
    else
      slv32(26)        := '0';
    end if;
    if (rtc.key.proto) then
      slv32(27)        := '1';
    else
      slv32(27)        := '0';
    end if;
    if (rtc.key.dstport) then
      slv32(28)        := '1';
    else
      slv32(28)        := '0';
    end if;
    if (rtc.key.srcport) then
      slv32(29)        := '1';
    else
      slv32(29)        := '0';
    end if;
    if (rtc.key.dstip>0) then
      slv32(30)        := '1';
    else
      slv32(30)        := '0';
    end if;
    if (rtc.key.srcip>0) then
      slv32(31)        := '1';
    else
      slv32(31)        := '0';
    end if;
    return slv32;
  end function;
  
  function is_zero(i : integer) return std_logic is
  begin
    if(i=0) then 
      return '1';
    end if;
    return '0';
  end function;

  constant FILTER_NO_MATCH : std_logic_vector(RULES_DATA_WIDTH-1 downto 0) := (others => '1');
  constant KEY_MAX_WIDTH   : integer := 128*2 + 16*2 + 8;
  type rules_key_t is array (0 to FILTER_RULE_TYPES-1) of std_logic_vector(KEY_MAX_WIDTH-1 downto 0);
  type ip_len_t is array (0 to FILTER_RULE_TYPES-1) of std_logic_vector(6 downto 0);
  type rules_result_t is array (0 to FILTER_RULE_TYPES) of std_logic_vector(RULES_DATA_WIDTH-1 downto 0);
  type rules_double_result_t is array (0 to FILTER_RULE_TYPES) of std_logic_vector(2*RULES_DATA_WIDTH+1 downto 0);
  type rules_record_t is array (0 to FILTER_RULE_TYPES) of std_logic_vector(RULES_DATA_WIDTH+KEY_MAX_WIDTH+16 downto 0);
  type rules_fifo_status_t is array (0 to FILTER_RULE_TYPES-1) of std_logic_vector(log2(FILTER_OFIFO_ITEMS) downto 0);
  
  signal rules_ip_len          : ip_len_t;
  signal rules_key             : rules_key_t;
  signal rules_key_del         : rules_key_t;
  signal rules_key_del2        : rules_key_t;
  signal rules_key_foundable   : std_logic_vector(FILTER_RULE_TYPES-1 downto 0);
  signal rules_cfgkey          : rules_key_t;
  signal rules_result          : rules_result_t;
  signal rules_result_len      : std_logic_vector(6 downto 0);
  signal result_low_mx         : std_logic_vector(RULES_DATA_WIDTH-1 downto 0);
  signal result_high_mx        : std_logic_vector(RULES_DATA_WIDTH-1 downto 0);
  signal result_low_len        : std_logic_vector(7 downto 0);
  signal result_high_len       : std_logic_vector(7 downto 0);
  signal result_low            : std_logic_vector(RULES_DATA_WIDTH-1 downto 0);
  signal result_high           : std_logic_vector(RULES_DATA_WIDTH-1 downto 0);
  signal result_low_id         : std_logic_vector(log2(FILTER_RULE_TYPES)-1 downto 0);
  signal result_high_id        : std_logic_vector(log2(FILTER_RULE_TYPES)-1 downto 0);
  signal rules_result_both     : rules_double_result_t;
  signal rules_result_len_both : std_logic_vector(13 downto 0);
  signal rules_result_both_vld : std_logic_vector(FILTER_RULE_TYPES downto 0);
  signal rules_result_reg      : rules_result_t;
  signal rules_result_len_reg  : std_logic_vector(6 downto 0);
  signal rules_result_reg_vld  : std_logic_vector(FILTER_RULE_TYPES downto 0);
  signal rules_found_reg       : std_logic_vector(FILTER_RULE_TYPES downto 0);
  signal rules_found           : std_logic_vector(FILTER_RULE_TYPES downto 0);
  signal rules_result_vld      : std_logic_vector(FILTER_RULE_TYPES downto 0);
  signal rules_add             : std_logic_vector(FILTER_RULE_TYPES-1 downto 0);
  signal rules_stop            : std_logic_vector(FILTER_RULE_TYPES-1 downto 0);
  signal rules_full            : std_logic_vector(FILTER_RULE_TYPES-1 downto 0);
  signal rules_busy            : std_logic_vector(FILTER_RULE_TYPES-1 downto 0);
  signal rules_rem             : std_logic_vector(FILTER_RULE_TYPES-1 downto 0);
  signal rules_rd              : std_logic_vector(FILTER_RULE_TYPES downto 0);
  signal rules_clr             : std_logic_vector(FILTER_RULE_TYPES downto 0);
  signal rules_drd             : rules_record_t := (others => (others => '0'));
  signal rules_drd_unfilter    : rules_record_t := (others => (others => '0'));
  signal rules_drd_sel         : std_logic_vector(RULES_DATA_WIDTH+KEY_MAX_WIDTH+16 downto 0);
  signal rules_drd_id          : std_logic_vector(log2(FILTER_RULE_TYPES+1)-1 downto 0);
  signal rules_drd_data        : std_logic_vector(RULES_DATA_WIDTH-1 downto 0);
  signal rules_drdy            : std_logic_vector(FILTER_RULE_TYPES downto 0);
  signal exists_drdy           : std_logic;
  signal rules_disabled        : std_logic_vector(FILTER_RULE_TYPES downto 0);
    
  signal rules_fifo_empty  : std_logic_vector(FILTER_RULE_TYPES downto 0);
  signal exists_fifo_empty : std_logic;
  signal rules_fifo_full   : std_logic_vector(FILTER_RULE_TYPES downto 0);
  signal exists_fifo_full  : std_logic;
  signal rules_fifo_read   : std_logic;
  signal rules_fifo_status : rules_fifo_status_t;
  signal rules_fifo_dout   : rules_double_result_t;
  signal rules_fifo_dout_len : std_logic_vector(13 downto 0);
  
  signal sig_srcip        : std_logic_vector(127 downto 0);
  signal sig_dstip        : std_logic_vector(127 downto 0);
  signal sig_srcport      : std_logic_vector(15 downto 0);
  signal sig_dstport      : std_logic_vector(15 downto 0);
  signal sig_protocol     : std_logic_vector(7 downto 0);
  signal sig_data_ready   : std_logic;
  
  signal reg_srcip        : std_logic_vector(127 downto 0) := (others => '0');
  signal reg_srcip_flip   : std_logic_vector(127 downto 0) := (others => '0');
  signal reg_srcip_v4     : std_logic;
  signal reg_dstip        : std_logic_vector(127 downto 0) := (others => '0');
  signal reg_dstip_v4     : std_logic;
  signal reg_srcport      : std_logic_vector(15 downto 0)  := (others => '0');
  signal reg_dstport      : std_logic_vector(15 downto 0)  := (others => '0');
  signal reg_protocol     : std_logic_vector(7 downto 0)   := (others => '0');
  signal reg_data_ready   : std_logic := '0';
  signal dir_change       : std_logic;
  signal reg_dir_change   : std_logic := '0';
  
  signal del_srcip        : std_logic_vector(127 downto 0) := (others => '0');
  signal del_dstip        : std_logic_vector(127 downto 0) := (others => '0');
  signal del_srcport      : std_logic_vector(15 downto 0)  := (others => '0');
  signal del_dstport      : std_logic_vector(15 downto 0)  := (others => '0');
  signal del_protocol     : std_logic_vector(7 downto 0)   := (others => '0');

  signal del2_srcip       : std_logic_vector(127 downto 0) := (others => '0');
  signal del2_dstip       : std_logic_vector(127 downto 0) := (others => '0');
  signal del2_srcport     : std_logic_vector(15 downto 0)  := (others => '0');
  signal del2_dstport     : std_logic_vector(15 downto 0)  := (others => '0');
  signal del2_protocol    : std_logic_vector(7 downto 0)   := (others => '0');
  
  signal init_vectors     : std_logic_vector(FILTER_IV_NUMBER*32-1 downto 0) := (others => '0');
  
  signal cfgreg_srcip     : std_logic_vector(127 downto 0) := (others => '0');
  signal cfgreg_srcip_flip: std_logic_vector(127 downto 0) := (others => '0');
  signal cfgreg_dstip     : std_logic_vector(127 downto 0) := (others => '0');
  signal cfgreg_srcport   : std_logic_vector(15 downto 0)  := (others => '0');
  signal cfgreg_dstport   : std_logic_vector(15 downto 0)  := (others => '0');
  signal cfgreg_protocol  : std_logic_vector(7 downto 0)   := (others => '0');
  signal cfgreg_srcip_len : std_logic_vector(7 downto 0)   := (others => '0');
  signal cfgreg_dstip_len : std_logic_vector(7 downto 0)   := (others => '0');
  signal cfgreg_data      : std_logic_vector(31 downto 0)  := (others => '0');
  signal cfgreg_disable   : std_logic_vector(31 downto 0)  := (others => '0');
  signal cfgreg_ignore    : std_logic_vector(31 downto 0)  := (others => '0');
  signal cfgreg_full      : std_logic_vector(31 downto 0)  := (others => '0');
  signal cfgreg_busy      : std_logic_vector(31 downto 0)  := (others => '0');
  
  signal adc_wr           : std_logic_vector(2**(FILTER_ADC_SIZE)-1 downto 0);
  signal adc_stop_wr      : std_logic;
  signal adc_ignore_wr    : std_logic; 
  signal adc_add_wr       : std_logic;     
  signal adc_rem_wr       : std_logic;    
  signal adc_dirrd_wr     : std_logic;   
  signal adc_dirclr_wr    : std_logic;   
  signal adc_data_wr      : std_logic;   
  signal adc_protocol_wr  : std_logic;
  signal adc_ports_wr     : std_logic; 
  signal adc_srcip_wr     : std_logic_vector(4-1 downto 0);
  signal adc_dstip_wr     : std_logic_vector(4-1 downto 0);
  signal adc_iv_wr        : std_logic_vector(FILTER_IV_NUMBER-1 downto 0);
  signal adc_dir_rules_sel: std_logic_vector(FILTER_RULE_TYPES downto 0);
  
  type adc_drd_array_t is array (0 to 2**(FILTER_ADC_SIZE)-1) of std_logic_vector(31 downto 0);
  signal adc_drd          : adc_drd_array_t;
  signal sig_mi_drdy      : std_logic := '0';
  signal reg_mi_addr      : std_logic_vector(FILTER_ADC_SIZE-1 downto 0) := (others => '0');
  
  signal busy             : std_logic;
  signal atomic_change    : std_logic;
  signal add_req_reg      : std_logic;
  signal ignore_req_reg   : std_logic;
  signal rem_req_reg      : std_logic;
  signal add_req_atomic   : std_logic;
  signal ignore_req_atomic: std_logic;
  signal rem_req_atomic   : std_logic;
  signal req_vector_reg   : std_logic_vector(FILTER_RULE_TYPES-1 downto 0);
  signal ignore_reg       : std_logic_vector(FILTER_RULE_TYPES-1 downto 0) := (others => '0');
   
  signal req_tree_add     : std_logic;
  signal req_tree_rem     : std_logic;
  signal req_tree_reg     : std_logic;
  signal tree_busy        : std_logic; 
  signal tree_full        : std_logic;
  signal tree_ignore_reg  : std_logic;
  signal tree_isend       : std_logic;
  signal tree_keyvld      : std_logic;
  signal tree_drd_key_flip: std_logic_vector(127 downto 0);
  
begin
  rules_ip_lens_gen : for i in 0 to FILTER_RULE_TYPES-1 generate
    real_ip_len_gen : if FILTER_CFG(i).key_width<=FILTER_CFG(i).key.srcip generate
      ipv4_ip_len_gen : if FILTER_CFG(i).key.ipv4_only generate
        rules_ip_len(i) <= conv_std_logic_vector(FILTER_CFG(i).key.srcip+95,7);
      end generate;
      ipv6_ip_len_gen : if not FILTER_CFG(i).key.ipv4_only generate
        rules_ip_len(i) <= conv_std_logic_vector(FILTER_CFG(i).key.srcip-1,7);
      end generate;
    end generate;
    fake_ip_len_gen : if FILTER_CFG(i).key_width>FILTER_CFG(i).key.srcip generate
      rules_ip_len(i) <= conv_std_logic_vector(127,7);
    end generate;
  end generate;
  
  -- Is some of match results fifos empty?
  fifo_empty_or : process(rules_fifo_empty)
    variable o : std_logic;
  begin
    o := '0';
    for i in 0 to FILTER_RULE_TYPES loop
      o := o or rules_fifo_empty(i);
    end loop;
    exists_fifo_empty <= o;
  end process;
  
  -- Is some of match results fifos full?
  fifo_full_or : process(rules_fifo_full)
    variable o : std_logic;
  begin
    o := '0';
    for i in 0 to FILTER_RULE_TYPES loop
      o := o or rules_fifo_full(i);
    end loop;
    exists_fifo_full <= o;
  end process;
  
  -- Some basic signal connections and computings
  IN_NEXT          <= not exists_fifo_full and dir_change;
  RESULT_READY     <= not exists_fifo_empty;
  sig_data_ready   <= IN_READY and not exists_fifo_full;
  atomic_change    <= (reg_data_ready and reg_dir_change) or (not reg_dir_change and not reg_data_ready); 
  
  -- Packet direction changer
  sig_srcip    <= IN_SRCIP   when dir_change='0' else IN_DSTIP;
  sig_dstip    <= IN_DSTIP   when dir_change='0' else IN_SRCIP;
  sig_srcport  <= IN_SRCPORT when dir_change='0' else IN_DSTPORT;
  sig_dstport  <= IN_DSTPORT when dir_change='0' else IN_SRCPORT;  
  sig_protocol <= IN_PROTOCOL;
  
  -- Input (5-tuple) register
  input_reg_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      reg_srcip    <= sig_srcip;
      reg_dstip    <= sig_dstip;
      reg_srcport  <= sig_srcport;
      reg_dstport  <= sig_dstport;
      reg_protocol <= sig_protocol;
      reg_data_ready <= sig_data_ready;
      reg_dir_change <= dir_change;
    end if;
  end process;
  
  -- Input (5-tuple) delayed by 1 tact
  input_delay_reg_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      del_srcip    <= reg_srcip;
      del_dstip    <= reg_dstip;
      del_srcport  <= reg_srcport;
      del_dstport  <= reg_dstport;
      del_protocol <= reg_protocol;
    end if;
  end process;
  
  -- Input (5-tuple) delayed by 2 tact
  input_delay_reg2_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      del2_srcip    <= del_srcip;
      del2_dstip    <= del_dstip;
      del2_srcport  <= del_srcport;
      del2_dstport  <= del_dstport;
      del2_protocol <= del_protocol;
    end if;
  end process;
  
  -- Match direction signal
  dir_change_reg_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        dir_change <= '0';
      elsif sig_data_ready='1' then
        dir_change <= not dir_change;
      end if;
    end if;
  end process;
  
  -- Filters of 5-tuple fields to rule keys: matching input
  key_filters_gen : for i in 0 to FILTER_RULE_TYPES-1 generate 
    key_filter_i : entity work.key_filter
      generic map (
        KEY_WIDTH           => FILTER_CFG(i).key_width, 
        SRCIP_WIDTH         => FILTER_CFG(i).key.srcip,
        DSTIP_WIDTH         => FILTER_CFG(i).key.dstip,
        USE_SRCPORT         => FILTER_CFG(i).key.srcport,
        USE_DSTPORT         => FILTER_CFG(i).key.dstport,
        USE_PROTO           => FILTER_CFG(i).key.proto,
        IPV4_ONLY           => FILTER_CFG(i).key.ipv4_only
      ) port map (
        SRCIP               => reg_srcip,
        DSTIP               => reg_dstip,
        SRCPORT             => reg_srcport,
        DSTPORT             => reg_dstport,
        PROTO               => reg_protocol,
        KEY                 => rules_key(i) 
      );
  end generate;
  
  -- Filters of 5-tuple fields to rule keys: delayed matching input
  key_del_filters_gen : for i in 0 to FILTER_RULE_TYPES-1 generate   
    key_del_filter_i : entity work.key_filter
      generic map (
        KEY_WIDTH           => FILTER_CFG(i).key_width, 
        SRCIP_WIDTH         => FILTER_CFG(i).key.srcip,
        DSTIP_WIDTH         => FILTER_CFG(i).key.dstip,
        USE_SRCPORT         => FILTER_CFG(i).key.srcport,
        USE_DSTPORT         => FILTER_CFG(i).key.dstport,
        USE_PROTO           => FILTER_CFG(i).key.proto,
        IPV4_ONLY           => FILTER_CFG(i).key.ipv4_only
      ) port map (
        SRCIP               => del_srcip,
        DSTIP               => del_dstip,
        SRCPORT             => del_srcport,
        DSTPORT             => del_dstport,
        PROTO               => del_protocol,
        KEY                 => rules_key_del(i) 
      );
  end generate; 
  
  -- Filters of 5-tuple fields to rule keys: double delayed matching input
  key_del2_filters_gen : for i in 0 to FILTER_RULE_TYPES-1 generate     
    key_del2_filter_i : entity work.key_filter
      generic map (
        KEY_WIDTH           => FILTER_CFG(i).key_width, 
        SRCIP_WIDTH         => FILTER_CFG(i).key.srcip,
        DSTIP_WIDTH         => FILTER_CFG(i).key.dstip,
        USE_SRCPORT         => FILTER_CFG(i).key.srcport,
        USE_DSTPORT         => FILTER_CFG(i).key.dstport,
        USE_PROTO           => FILTER_CFG(i).key.proto,
        IPV4_ONLY           => FILTER_CFG(i).key.ipv4_only
      ) port map (
        SRCIP               => del2_srcip,
        DSTIP               => del2_dstip,
        SRCPORT             => del2_srcport,
        DSTPORT             => del2_dstport,
        PROTO               => del2_protocol,
        KEY                 => rules_key_del2(i) 
      );
  end generate;
  
  -- Filters of 5-tuple fields to rule keys: configuration input
  cfgkey_filters_gen : for i in 0 to FILTER_RULE_TYPES-1 generate     
    cfgkey_filter_i : entity work.key_filter
      generic map (
        KEY_WIDTH           => FILTER_CFG(i).key_width, 
        SRCIP_WIDTH         => FILTER_CFG(i).key.srcip,
        DSTIP_WIDTH         => FILTER_CFG(i).key.dstip,
        USE_SRCPORT         => FILTER_CFG(i).key.srcport,
        USE_DSTPORT         => FILTER_CFG(i).key.dstport,
        USE_PROTO           => FILTER_CFG(i).key.proto,
        IPV4_ONLY           => FILTER_CFG(i).key.ipv4_only
      ) port map (
        SRCIP               => cfgreg_srcip,
        DSTIP               => cfgreg_dstip,
        SRCPORT             => cfgreg_srcport,
        DSTPORT             => cfgreg_dstport,
        PROTO               => cfgreg_protocol,
        KEY                 => rules_cfgkey(i) 
      );      
  end generate;
  
  -- IPv4 only rules prematch
  reg_srcip_v4 <= '1' when (reg_srcip(95 downto 0)=(95 downto 0 => '0')) else '0';
  reg_dstip_v4 <= '1' when (reg_dstip(95 downto 0)=(95 downto 0 => '0')) else '0';
  ipv4_prematch_gen : for i in 0 to FILTER_RULE_TYPES-1 generate 
    real_ipv4_prematch_gen : if FILTER_CFG(i).key.ipv4_only generate 
      rules_key_foundable(i) <= ((reg_srcip_v4 or is_zero(FILTER_CFG(i).key.srcip)) and (reg_dstip_v4 or is_zero(FILTER_CFG(i).key.dstip))) and not ignore_reg(i); 
    end generate;
    fake_ipv4_prematch_gen : if not FILTER_CFG(i).key.ipv4_only generate
      rules_key_foundable(i) <= not ignore_reg(i); 
    end generate;
  end generate;
  
  -- Cockoo hash matching blocks
  chash_match_gen : for i in 0 to FILTER_RULE_TYPES-1 generate
    -- use real Cockoo hash block
    match_rules_gen : if FILTER_CFG(i).key_width>0 generate
      cuckoo_hash_i : entity work.cuckoo_hash
        generic map(
          KEY_WIDTH           => FILTER_CFG(i).key_width,
          DATA_WIDTH          => RULES_DATA_WIDTH,
          IV_WIDTH            => 32,
          ITEMS               => FILTER_CFG(i).items,
          TABLES              => FILTER_CFG(i).tables,
          BRAM_TYPE           => FILTER_CFG(i).bram_type,
          CMP_REG             => FILTER_CFG(i).use_cmp_reg,
          CRC_REG             => FILTER_CFG(i).use_crc_reg
        ) port map (
          CLK           => CLK,
          RESET         => RESET,
          
          KEY           => rules_key(i)(FILTER_CFG(i).key_width-1 downto 0),
          KEY_DEL       => rules_key_del(i)(FILTER_CFG(i).key_width-1 downto 0),
          KEY_DEL2      => rules_key_del2(i)(FILTER_CFG(i).key_width-1 downto 0),
          KEY_FOUNDABLE => rules_key_foundable(i),
          KEY_VLD       => reg_data_ready,
          
          DATA          => rules_result(i),
          DATA_FOUND    => rules_found(i),
          DATA_VLD      => rules_result_vld(i),
          
          INIT_VECTORS  => init_vectors(32*FILTER_CFG(i).tables-1 downto 0),
          ADD_KEY       => rules_cfgkey(i)(FILTER_CFG(i).key_width-1 downto 0),
          ADD_DATA      => cfgreg_data(RULES_DATA_WIDTH-1 downto 0),
          ADD_VLD       => rules_add(i),
          ADD_STOP      => rules_stop(i),
          ADD_STOP_VLD  => adc_stop_wr,
          ADD_BUSY      => rules_full(i),
          REM_KEY       => rules_cfgkey(i)(FILTER_CFG(i).key_width-1 downto 0),
          REM_VLD       => rules_rem(i),
          REM_BUSY      => rules_busy(i),
          DIR_ADDR      => MI_DWR(log2(FILTER_CFG(i).items)+log2(FILTER_CFG(i).tables+1)-1 downto 0),
          DIR_RD        => rules_rd(i),
          DIR_CLR       => rules_clr(i),
          DIR_DRD       => rules_drd(i)(FILTER_CFG(i).key_width+RULES_DATA_WIDTH downto 0),
          DIR_DRDY      => rules_drdy(i)
        );
      rules_disabled(i)   <= '0';
    end generate;
    -- empty Cockoo hash block
    no_match_rules_gen : if FILTER_CFG(i).key_width=0 generate
      rules_disabled(i)   <= '1';
      rules_result(i)     <= (others => '0');
      rules_drd(i)        <= (others => '0');
      rules_result_vld(i) <= '0';
      rules_full(i)       <= '1';
      rules_busy(i)       <= '1';  
      rules_drdy(i)       <= '0';
    end generate;   
  end generate;  
 
  -- Matching result fifos
  rules_fifo_read <= RESULT_NEXT and not exists_fifo_empty;
  match_ofifos_gen : for i in 0 to FILTER_RULE_TYPES-1 generate
    -- real matching result fifo
    match_fifo_gen : if FILTER_CFG(i).key_width>0 generate
      ofifo_i : entity work.fifo_status 
        generic map(
          DATA_WIDTH => 2*RULES_DATA_WIDTH+2, 
          DISTMEM_TYPE => FILTER_OFIFO_TYPE,
          ITEMS => FILTER_OFIFO_ITEMS
        ) port map(
          RESET     => RESET,
          CLK       => CLK,
          DATA_IN   => rules_result_both(i),
          WRITE_REQ => rules_result_both_vld(i),
          
          FULL      => open,
          LSTBLK    => open,
          STATUS    => rules_fifo_status(i),
          DATA_OUT  => rules_fifo_dout(i),
          READ_REQ  => rules_fifo_read,
          EMPTY     => rules_fifo_empty(i)
        );
      rules_fifo_full(i) <= '1' when rules_fifo_status(i)(log2(FILTER_OFIFO_ITEMS) downto 3)=(log2(FILTER_OFIFO_ITEMS) downto 3 =>'0') else '0';
    end generate;
    -- fake matching result fifo (always has record ready with no match)
    no_match_fifo_gen : if FILTER_CFG(i).key_width=0 generate
      rules_fifo_full(i)  <= '0';
      rules_fifo_empty(i) <= '0';
      rules_fifo_dout(i)  <= (others => '0');
    end generate;
  end generate;
  tree_match_fifo_gen : if FILTER_TREE_STAGES>0 generate
    tree_ofifo_i : entity work.fifo_status 
      generic map(
        DATA_WIDTH => 2*RULES_DATA_WIDTH+2, 
        DISTMEM_TYPE => FILTER_OFIFO_TYPE,
        ITEMS => FILTER_OFIFO_ITEMS
      ) port map(
        RESET     => RESET,
        CLK       => CLK,
        DATA_IN   => rules_result_both(FILTER_RULE_TYPES),
        WRITE_REQ => rules_result_both_vld(FILTER_RULE_TYPES),
        
        FULL      => open,
        LSTBLK    => open,
        STATUS    => open,
        DATA_OUT  => rules_fifo_dout(FILTER_RULE_TYPES),
        READ_REQ  => rules_fifo_read,
        EMPTY     => rules_fifo_empty(FILTER_RULE_TYPES)
      );
    tree_len_ofifo_i : entity work.fifo_status 
      generic map(
        DATA_WIDTH => 2*7, 
        DISTMEM_TYPE => FILTER_OFIFO_TYPE,
        ITEMS => FILTER_OFIFO_ITEMS
      ) port map(
        RESET     => RESET,
        CLK       => CLK,
        DATA_IN   => rules_result_len_both,
        WRITE_REQ => rules_result_both_vld(FILTER_RULE_TYPES),
        
        FULL      => open,
        LSTBLK    => open,
        STATUS    => open,
        DATA_OUT  => rules_fifo_dout_len,
        READ_REQ  => rules_fifo_read,
        EMPTY     => open
      );    
  end generate;
  no_tree_match_fifo_gen : if FILTER_TREE_STAGES=0 generate
    rules_fifo_full(FILTER_RULE_TYPES)  <= '0';
    rules_fifo_empty(FILTER_RULE_TYPES) <= '0';
    rules_fifo_dout(FILTER_RULE_TYPES)  <= (others => '0');
  end generate;
  
  -- Matching result processing 
  result_proc_gen : for i in 0 to FILTER_RULE_TYPES generate
    rules_result_both(i)     <= rules_result_reg(i) & rules_result(i) & rules_found_reg(i) & rules_found(i);
    rules_result_both_vld(i) <= rules_result_reg_vld(i) and rules_result_vld(i);
  end generate;
  rules_result_len_both <= rules_result_len_reg & rules_result_len;
  
  -- Has match been found?
  found_or : process(rules_fifo_dout)
    variable o : std_logic_vector(1 downto 0);
  begin
    o := (others => '0');
    for i in 0 to FILTER_RULE_TYPES loop
      o := o or rules_fifo_dout(i)(1 downto 0);
    end loop;
    FOUND <= o;
  end process;
  
  -- Compute matching result
  result_low_id_i : process(rules_fifo_dout)
    variable a : std_logic_vector(log2(FILTER_RULE_TYPES)-1 downto 0);
  begin
     a := (others => '0');  
     for i in 0 to (FILTER_RULE_TYPES-1) loop
        if (rules_fifo_dout(i)(0) = '1') then
           a := conv_std_logic_vector(i,log2(FILTER_RULE_TYPES));
        end if;
     end loop;
     result_low_id <= a;
  end process;
  result_high_id_i : process(rules_fifo_dout)
    variable a : std_logic_vector(log2(FILTER_RULE_TYPES)-1 downto 0);
  begin
     a := (others => '0');  
     for i in 0 to (FILTER_RULE_TYPES-1) loop
        if (rules_fifo_dout(i)(1) = '1') then
           a := conv_std_logic_vector(i,log2(FILTER_RULE_TYPES));
        end if;
     end loop;
     result_high_id <= a;
  end process;
  result_low  <= rules_fifo_dout(conv_integer(result_low_id))(RULES_DATA_WIDTH+1 downto 2);
  result_high <= rules_fifo_dout(conv_integer(result_high_id))(2*RULES_DATA_WIDTH+1 downto RULES_DATA_WIDTH+2);
  result_low_len  <= rules_fifo_dout(conv_integer(result_low_id))(0)  & rules_ip_len(conv_integer(result_low_id));
  result_high_len <= rules_fifo_dout(conv_integer(result_high_id))(1) & rules_ip_len(conv_integer(result_high_id));
  result_low_mx  <= result_low  when result_low_len  >= (rules_fifo_dout(FILTER_RULE_TYPES)(0)&rules_fifo_dout_len(6  downto 0)) else rules_fifo_dout(FILTER_RULE_TYPES)(RULES_DATA_WIDTH+1 downto 2);
  result_high_mx <= result_high when result_high_len >= (rules_fifo_dout(FILTER_RULE_TYPES)(0)&rules_fifo_dout_len(13 downto 7)) else rules_fifo_dout(FILTER_RULE_TYPES)(2*RULES_DATA_WIDTH+1 downto RULES_DATA_WIDTH+2);
  RESULT <= result_high_mx & result_low_mx;
  
  -- Matching result registering
  result_reg_gen : for i in 0 to FILTER_RULE_TYPES-1 generate
    rules_result_reg_gen : if FILTER_CFG(i).key_width>0 generate
      rules_result_reg_i : process(CLK)
      begin
        if CLK'event and CLK='1' then
          if rules_result_vld(i)='1' then
            rules_result_reg(i)     <= rules_result(i);
            rules_found_reg(i)      <= rules_found(i);
          end if;
        end if;
      end process;
      rules_result_reg_vld_i : process(CLK)
      begin
        if CLK'event and CLK='1' then
          if RESET='1' then
            rules_result_reg_vld(i) <= '0';
          elsif rules_result_vld(i)='1' then
            rules_result_reg_vld(i) <= not rules_result_reg_vld(i);
          end if;
        end if;
      end process;
    end generate;
    no_rules_result_reg_gen : if FILTER_CFG(i).key_width=0 generate
      rules_result_reg_vld(i) <= '1';
      rules_result_reg(i)     <= (others => '0');
      rules_found_reg(i)      <= '0';
    end generate; 
  end generate;
  tree_rules_result_reg_gen : if FILTER_TREE_STAGES>0 generate
    rules_result_reg_i : process(CLK)
    begin
      if CLK'event and CLK='1' then
        if rules_result_vld(FILTER_RULE_TYPES)='1' then
          rules_result_reg(FILTER_RULE_TYPES)     <= rules_result(FILTER_RULE_TYPES);
          rules_result_len_reg                    <= rules_result_len;
          rules_found_reg(FILTER_RULE_TYPES)      <= rules_found(FILTER_RULE_TYPES);
        end if;
      end if;
    end process;
    rules_result_reg_vld_i : process(CLK)
    begin
      if CLK'event and CLK='1' then
        if RESET='1' then
          rules_result_reg_vld(FILTER_RULE_TYPES) <= '0';
        elsif rules_result_vld(FILTER_RULE_TYPES)='1' then
          rules_result_reg_vld(FILTER_RULE_TYPES) <= not rules_result_reg_vld(FILTER_RULE_TYPES);
        end if;
      end if;
    end process;
  end generate;
  no_tree_rules_result_reg_gen : if FILTER_TREE_STAGES=0 generate
    rules_result_reg_vld(FILTER_RULE_TYPES) <= '1';
    rules_result_reg(FILTER_RULE_TYPES)     <= (others => '0');
    rules_result_len_reg                    <= (others => '0');
    rules_found_reg(FILTER_RULE_TYPES)      <= '0';
  end generate; 
  
  -- Unfilter key values back to 5-tuples
  key_unfilters_gen : for i in 0 to FILTER_RULE_TYPES-1 generate 
    key_unfilter_i : entity work.key_unfilter
      generic map (
        KEY_WIDTH           => FILTER_CFG(i).key_width, 
        SRCIP_WIDTH         => FILTER_CFG(i).key.srcip,
        DSTIP_WIDTH         => FILTER_CFG(i).key.dstip,
        USE_SRCPORT         => FILTER_CFG(i).key.srcport,
        USE_DSTPORT         => FILTER_CFG(i).key.dstport,
        USE_PROTO           => FILTER_CFG(i).key.proto,
        IPV4_ONLY           => FILTER_CFG(i).key.ipv4_only
      ) port map (
        SRCIP               => rules_drd_unfilter(i)(295+RULES_DATA_WIDTH+1 downto 168+RULES_DATA_WIDTH+1),
        DSTIP               => rules_drd_unfilter(i)(167+RULES_DATA_WIDTH+1 downto 40+RULES_DATA_WIDTH+1),
        SRCPORT             => rules_drd_unfilter(i)(39+RULES_DATA_WIDTH+1 downto 24+RULES_DATA_WIDTH+1),
        DSTPORT             => rules_drd_unfilter(i)(23+RULES_DATA_WIDTH+1 downto 8+RULES_DATA_WIDTH+1),
        PROTO               => rules_drd_unfilter(i)(7+RULES_DATA_WIDTH+1 downto 0+RULES_DATA_WIDTH+1),
        KEY                 => rules_drd(i)(RULES_DATA_WIDTH+KEY_MAX_WIDTH downto RULES_DATA_WIDTH+1) 
      );
    rules_drd_unfilter(i)(RULES_DATA_WIDTH downto 0) <= rules_drd(i)(RULES_DATA_WIDTH downto 0);
    ipv6_len_gen : if not FILTER_CFG(i).key.ipv4_only generate
      rules_drd_unfilter(i)(296+RULES_DATA_WIDTH+8 downto 296+RULES_DATA_WIDTH+1) <= conv_std_logic_vector(FILTER_CFG(i).key.srcip-1,8);
      rules_drd_unfilter(i)(296+RULES_DATA_WIDTH+16 downto 296+RULES_DATA_WIDTH+9) <= conv_std_logic_vector(FILTER_CFG(i).key.dstip-1,8);
    end generate;
    ipv4_len_gen : if FILTER_CFG(i).key.ipv4_only generate
      rules_drd_unfilter(i)(296+RULES_DATA_WIDTH+8 downto 296+RULES_DATA_WIDTH+1) <= conv_std_logic_vector(FILTER_CFG(i).key.srcip+95,8);
      rules_drd_unfilter(i)(296+RULES_DATA_WIDTH+16 downto 296+RULES_DATA_WIDTH+9) <= conv_std_logic_vector(FILTER_CFG(i).key.dstip+95,8);
    end generate;
  end generate;
  
  -- Select read data for rule which has drdy active
  rules_drd_id_i : process(rules_drdy)
    variable a : std_logic_vector(log2(FILTER_RULE_TYPES+1)-1 downto 0);
  begin
     a := (others => '0');  
     for i in 0 to FILTER_RULE_TYPES loop
        if (rules_drdy(i) = '1') then
           a := conv_std_logic_vector(i,log2(FILTER_RULE_TYPES+1));
        end if;
     end loop;
     rules_drd_id <= a;
  end process;  
  rules_drd_sel  <= rules_drd_unfilter(conv_integer(rules_drd_id));
  rules_drd_data <= rules_drd_sel(RULES_DATA_WIDTH downto 1) when rules_drd_sel(0)='1' else FILTER_NO_MATCH; 
  
  -- Is there rule type with active drdy?
  rules_drdy_or : process(rules_drdy)
    variable o : std_logic;
  begin
    o := '0';
    for i in 0 to FILTER_RULE_TYPES loop
      o := o or rules_drdy(i);
    end loop;
    exists_drdy <= o;
  end process;
  
  -- Configuration registers: source IP
  cfgreg_srcip_gen : for i in 0 to 3 generate
    cfgreg_srcip_i : process(CLK)
    begin
      if CLK'event and CLK='1' then
        if adc_srcip_wr(i)='1' then
          cfgreg_srcip((i+1)*32-1 downto i*32) <= MI_DWR;
        elsif exists_drdy='1' then
          cfgreg_srcip((i+1)*32-1 downto i*32) <= rules_drd_sel((i+1)*32+128+2*16+8+RULES_DATA_WIDTH downto i*32+128+2*16+8+RULES_DATA_WIDTH+1); 
        end if;
      end if;
    end process;
  end generate;
  
  -- Configuration registers: destination IP
  cfgreg_dstip_gen : for i in 0 to 3 generate
    cfgreg_dstip_i : process(CLK)
    begin
      if CLK'event and CLK='1' then
        if adc_dstip_wr(i)='1' then
          cfgreg_dstip((i+1)*32-1 downto i*32) <= MI_DWR;
        elsif exists_drdy='1' then
          cfgreg_dstip((i+1)*32-1 downto i*32) <= rules_drd_sel((i+1)*32+2*16+8+RULES_DATA_WIDTH downto i*32+2*16+8+RULES_DATA_WIDTH+1); 
        end if;
      end if;
    end process;
  end generate;
  
  -- Configuration registers: source port
  cfgreg_srcport_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if adc_ports_wr='1' then
        cfgreg_srcport <= MI_DWR(15 downto 0);
      elsif exists_drdy='1' then
        cfgreg_srcport <= rules_drd_sel(RULES_DATA_WIDTH+8+2*16 downto 16+8+RULES_DATA_WIDTH+1);  
      end if;
    end if;
  end process;
  
  -- Configuration registers: destination port
  cfgreg_dstport_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if adc_ports_wr='1' then
        cfgreg_dstport <= MI_DWR(31 downto 16);
      elsif exists_drdy='1' then
        cfgreg_dstport <= rules_drd_sel(RULES_DATA_WIDTH+8+16 downto 8+RULES_DATA_WIDTH+1);
      end if;
    end if;
  end process;
  
  -- Configuration registers: protocol & ip addresses lengths
  cfgreg_protocol_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if adc_protocol_wr='1' then
        cfgreg_protocol   <= MI_DWR(7  downto 0);
        cfgreg_srcip_len  <= MI_DWR(23 downto 16);
        cfgreg_dstip_len  <= MI_DWR(31 downto 24);
      elsif exists_drdy='1' then
        cfgreg_protocol  <= rules_drd_sel(RULES_DATA_WIDTH+8 downto RULES_DATA_WIDTH+1);
        cfgreg_srcip_len <= rules_drd_sel(296+RULES_DATA_WIDTH+8  downto 296+RULES_DATA_WIDTH+1);
        cfgreg_dstip_len <= rules_drd_sel(296+RULES_DATA_WIDTH+16 downto 296+RULES_DATA_WIDTH+1+8);
      end if;
    end if;
  end process;
  
  -- Configuration registers: rule data (result)
  cfgreg_data_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if adc_data_wr='1' then
        cfgreg_data(RULES_DATA_WIDTH-1 downto 0) <= MI_DWR(RULES_DATA_WIDTH-1 downto 0);
      elsif exists_drdy='1' then
        cfgreg_data(RULES_DATA_WIDTH-1 downto 0) <= rules_drd_data;  
      end if;
    end if;
  end process;
  cfgreg_data_padding_gen : if RULES_DATA_WIDTH<32 generate
    cfgreg_data(31 downto RULES_DATA_WIDTH) <= (others => '0');
  end generate;
  
  -- Configuration registers: status registers (ReadOnly)
  cfgreg_disable <= (31 downto FILTER_RULE_TYPES+1 => '0') & rules_disabled;
  cfgreg_ignore  <= (31 downto FILTER_RULE_TYPES+1 => '0') & tree_ignore_reg & ignore_reg;
  cfgreg_full    <= (31 downto FILTER_RULE_TYPES+1 => '0') & tree_full & rules_full;
  cfgreg_busy    <= (31 downto FILTER_RULE_TYPES+2 => '0') & tree_busy & rules_busy & busy;
  busy           <= add_req_reg or rem_req_reg or ignore_req_reg;
  
  
  -- Configuration registers: hash seed (init) vectors registers
  init_vectors_gen : for i in 0 to FILTER_IV_NUMBER-1 generate
    init_vectors_i : process(CLK)
    begin
      if CLK'event and CLK='1' then
        if adc_iv_wr(i)='1' then
          init_vectors((i+1)*32-1 downto i*32) <= MI_DWR;
        end if;
      end if;
    end process;
  end generate;
  
  -- MI32 read controll
  MI_DRD  <= adc_drd(conv_integer(reg_mi_addr));
  MI_ARDY <= MI_RD or MI_WR;
  MI_DRDY <= sig_mi_drdy;
  mi_drd_reg_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      sig_mi_drdy <= MI_RD;
      reg_mi_addr <= MI_ADDR(FILTER_ADC_SIZE+1 downto 2);
    end if;
  end process;
  
  -- Main MI32 address decoder: write enable signal
  adc_wr_sel_i : process(MI_ADDR,MI_WR)
  begin
    adc_wr <= (others => '0');
    for i in 0 to 2**(FILTER_ADC_SIZE)-1 loop
      if(conv_std_logic_vector(i,FILTER_ADC_SIZE)=MI_ADDR(FILTER_ADC_SIZE+1 downto 2)) then
        adc_wr(i) <= MI_WR;
      end if;
    end loop;
  end process; 
  
  -- Main MI32 address decoder: read addresses assignment
  adc_drd(0) <= cfgreg_disable;                                                            -- STATUS_DISABLE  = 0x0000 (R)
  adc_drd(1) <= cfgreg_ignore;                                                             -- STATUS_IGNORE   = 0x0004 (R)
  adc_drd(2) <= cfgreg_full;                                                               -- STATUS_FULL     = 0x0008 (R)
  adc_drd(3) <= cfgreg_busy;                                                               -- STATUS_BUSY     = 0x000C (R)
  adc_drd(4) <= conv_std_logic_vector(FILTER_RULE_TYPES,32);                               -- RULE_TYPES      = 0x0010 (R)
  adc_drd(5) <= conv_std_logic_vector(FILTER_IV_NUMBER,32);                                -- IV_NUMBER       = 0x0014 (R)
  adc_drd(6) <= cfgreg_data;                                                               -- DATA(ID)        = 0x0018 (R)
  adc_drd(7) <= cfgreg_dstip_len & cfgreg_srcip_len & X"00" & cfgreg_protocol;             -- PROTOCOL&LENGTH = 0x001C (R)
  adc_drd(8) <= cfgreg_dstport & cfgreg_srcport;                                           -- PORTS           = 0x0020 (R)
  adc_ip_rd_gen : for i in 1 to 4 generate
    adc_drd(8+i)   <= cfgreg_srcip(i*32-1 downto (i-1)*32);                                -- SRCIPv6         = 0x0024-0x0030 (R)
    adc_drd(8+i+4) <= cfgreg_dstip(i*32-1 downto (i-1)*32);                                -- DSTIPv6         = 0x0034-0x0040 (R)
  end generate;
  adc_iv_rd_gen : for i in 1 to FILTER_IV_NUMBER generate
    adc_drd(8+i+2*4) <= init_vectors(i*32-1 downto (i-1)*32);                              -- INIT_VECTORS    = 0x0044-0x004C (R)
  end generate;
  adc_cfg_rd_gen : for i in 0 to FILTER_RULE_TYPES-1 generate
    adc_drd(9+i+2*4+FILTER_IV_NUMBER) <= rtc2slv32(FILTER_CFG(i));                         -- CONFIGURATION   = 0x0050-       (R)
  end generate;
  adc_drd(9+FILTER_RULE_TYPES+2*4+FILTER_IV_NUMBER) <= X"000000" & conv_std_logic_vector(FILTER_TREE_STAGES,8);
  
  -- Main MI32 address decoder: write addresses assignment
  adc_stop_wr     <= adc_wr(0);                                                            -- STOP_REQ        = 0x0000 (W)
  adc_ignore_wr   <= adc_wr(1);                                                            -- IGNORE_REQ      = 0x0004 (W)
  adc_add_wr      <= adc_wr(2);                                                            -- ADD_REQ         = 0x0008 (W)
  adc_rem_wr      <= adc_wr(3);                                                            -- REM_REQ         = 0x000C (W) 
  adc_dirrd_wr    <= adc_wr(4);                                                            -- DIR_RD_REQ      = 0x0010 (W)
  adc_dirclr_wr   <= adc_wr(5);                                                            -- DIR_CLR_REQ     = 0x0014 (W)
  adc_data_wr     <= adc_wr(6);                                                            -- DATA(ID)        = 0x0018 (W)
  adc_protocol_wr <= adc_wr(7);                                                            -- PROTOCOL&LENGTH = 0x001C (W)
  adc_ports_wr    <= adc_wr(8);                                                            -- PORTS           = 0x0020 (W)
  adc_ip_wr_gen : for i in 1 to 4 generate  
    adc_srcip_wr(i-1) <= adc_wr(8+i);                                                      -- SRCIPv6         = 0x0024-0x0030 (W)
    adc_dstip_wr(i-1) <= adc_wr(8+i+4);                                                    -- DSTIPv6         = 0x0034-0x0040 (W)
  end generate;
  adc_iv_wr_gen : for i in 1 to FILTER_IV_NUMBER generate
    adc_iv_wr(i-1) <= adc_wr(8+i+2*4);                                                     -- INIT_VECTORS    = 0x0044-0x004C (W)
  end generate;
  
  -- Address decoder for dirrect memory access into Cockoo hash tables
  dir_adc_i : process(MI_DWR)
  begin 
    adc_dir_rules_sel <= (others => '0');
    adc_dir_rules_sel(conv_integer(MI_DWR(31 downto 32-log2(FILTER_RULE_TYPES+1)))) <= '1';
  end process;
  rules_rd  <= adc_dir_rules_sel and (FILTER_RULE_TYPES downto 0 => adc_dirrd_wr);
  rules_clr <= adc_dir_rules_sel and (FILTER_RULE_TYPES downto 0 => adc_dirclr_wr);
  rules_stop <= MI_DWR(FILTER_RULE_TYPES-1 downto 0);
  
  -- Atomic requests synchronization: add request flag
  atomic_add_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        add_req_reg <= '0';
      elsif adc_add_wr='1' then
        add_req_reg <= '1';
      elsif atomic_change='1' then
        add_req_reg <= '0';
      end if;
    end if;
  end process;
  
  -- Atomic requests synchronization: remove request flag
  atomic_rem_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        rem_req_reg <= '0';
      elsif adc_rem_wr='1' then
        rem_req_reg <= '1';
      elsif atomic_change='1' then
        rem_req_reg <= '0';
      end if;
    end if;
  end process;
  
  -- Atomic requests synchronization: ignore request flag
  atomic_ignore_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        ignore_req_reg <= '0';
      elsif adc_ignore_wr='1' then
        ignore_req_reg <= '1';
      elsif atomic_change='1' then
        ignore_req_reg <= '0';
      end if;
    end if;
  end process;
  
  -- Atomic requests synchronization: request validity vector
  atomic_vector_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        req_vector_reg <= (others => '0');
      elsif adc_ignore_wr='1' or adc_rem_wr='1' or adc_add_wr='1' then
        req_vector_reg <= MI_DWR(FILTER_RULE_TYPES-1 downto 0);
        req_tree_reg   <= MI_DWR(FILTER_RULE_TYPES);
      end if;
    end if;
  end process;
  req_tree_add <= MI_DWR(FILTER_RULE_TYPES) and adc_add_wr;
  req_tree_rem <= MI_DWR(FILTER_RULE_TYPES) and adc_rem_wr;
  
  -- Atomic requests synchronization
  add_req_atomic    <= atomic_change and add_req_reg;
  rem_req_atomic    <= atomic_change and rem_req_reg;
  ignore_req_atomic <= atomic_change and ignore_req_reg;  
  rules_add         <= (FILTER_RULE_TYPES-1 downto 0 => add_req_atomic) and req_vector_reg;
  rules_rem         <= (FILTER_RULE_TYPES-1 downto 0 => rem_req_atomic) and req_vector_reg;
  ignore_reg_i : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if ignore_req_atomic='1' then
        ignore_reg      <= req_vector_reg;
        tree_ignore_reg <= req_tree_reg;
      end if;
    end if;
  end process;
  
  -- Prefix tree ---------------------------------------------------------------
  real_tree_gen : if FILTER_TREE_STAGES>0 generate
    prefix_tree_i : entity work.tree
      generic map (
        TREE_STAGES    => FILTER_TREE_STAGES,
        KEY_WIDTH      => 128,
        DATA_WIDTH     => RULES_DATA_WIDTH,
        USE_STAGES     => FILTER_USE_STAGES,
        USE_STAGE_LEAF => FILTER_USE_STAGE_LEAF,
        USE_REGS       => FILTER_USE_REGS
      ) port map (
        CLK            => CLK,
        RESET          => RESET,
        
        IN_KEY         => reg_srcip_flip,
        IN_FOUNDABLE   => not tree_ignore_reg,
        IN_VLD         => reg_data_ready,
        IN_CHANGE_EN   => atomic_change,
        
        OUT_DATA       => rules_result(FILTER_RULE_TYPES),
        OUT_LENGTH     => rules_result_len,
        OUT_FOUND      => rules_found(FILTER_RULE_TYPES),
        OUT_VLD        => rules_result_vld(FILTER_RULE_TYPES),
        
        CFG_KEY        => cfgreg_srcip_flip,
        CFG_KEYLEN     => cfgreg_srcip_len(log2(128)-1 downto 0),
        CFG_DATA       => cfgreg_data(RULES_DATA_WIDTH-1 downto 0),
        CFG_ADD_VLD    => req_tree_add,
        CFG_REM_VLD    => req_tree_rem,
        CFG_CLR_VLD    => rules_clr(FILTER_RULE_TYPES),
        CFG_BUSY       => tree_busy,
        CFG_FULL       => tree_full,
        DIR_ADDR       => MI_DWR(FILTER_TREE_STAGES-1 downto 0),
        DIR_RD         => rules_rd(FILTER_RULE_TYPES),
        DIR_KEY        => tree_drd_key_flip,
        DIR_KEYLEN     => rules_drd_unfilter(FILTER_RULE_TYPES)(296+RULES_DATA_WIDTH+log2(128) downto 296+RULES_DATA_WIDTH+1),
        DIR_ISEND      => tree_isend,
        DIR_KEYVLD     => tree_keyvld,
        DIR_DATA       => rules_drd_unfilter(FILTER_RULE_TYPES)(RULES_DATA_WIDTH downto 1),
        DIR_DRDY       => rules_drdy(FILTER_RULE_TYPES)
      );
    rules_drd_unfilter(FILTER_RULE_TYPES)(0) <= tree_keyvld and not tree_isend;
    rules_disabled(FILTER_RULE_TYPES)        <= '0';
    ip_addr_flip_gen : for i in 0 to 127 generate
      cfgreg_srcip_flip(i) <= cfgreg_srcip(127-i);
      reg_srcip_flip(i)    <= reg_srcip(127-i);
      rules_drd_unfilter(FILTER_RULE_TYPES)(168+RULES_DATA_WIDTH+1+i) <= tree_drd_key_flip(127-i);
    end generate;
  end generate;
  fake_tree_gen : if FILTER_TREE_STAGES=0 generate
    rules_disabled(FILTER_RULE_TYPES)   <= '1';
    rules_result(FILTER_RULE_TYPES)     <= (others => '0');
    rules_result_len                    <= (others => '0');
    rules_drd(FILTER_RULE_TYPES)        <= (others => '0');
    rules_result_vld(FILTER_RULE_TYPES) <= '0';
    tree_full                           <= '1';
    tree_busy                           <= '1';  
    rules_drdy(FILTER_RULE_TYPES)       <= '0';
  end generate;  
  
end architecture;
