# Class describing interception transferred through INI1a
#
# Copyright (C) 2011 Matěj Grégr, Michal Kajan, Libor Polčák, Vladimír Veselý
# 
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import time
from modules.shared.nid import CreateNID
import re

mac_addr = re.compile("^([0-9a-fA-F]{2}:){5}[0-9a-fA-F]{2}$")

class INI1AIntercept(object):

    def __init__(self, cid, nid, liid, cc, start, level):
        """ Constructor

        cid CID of the intercept
        nid NID of the intercept
        liid LIID of the intercept
        cc Boolean specifing if the interception contain CC data
        start UNIX time, determines actual start of the interception
        level Specifies range of devices that are part of the intercept
        """
        self.__liid = liid
        self.__cid = cid
        self.__nid = nid
        self.__cc = cc
        self.__start = time.localtime(start)
        self.__level = level

    def increaseCIN(self):
        self.__cid.increaseCIN()

    def setNID(self,nid):
        self.__nid = CreateNID(nid)
        
    def setCID(self,cid):
        self.__cid = cid

    def setInterceptionStart(self,start):
        self.__start = start

    # Getters
    def getLIID(self):
        return self.__liid

    def getCID(self):
        return self.__cid

    def getNID(self):
        return self.__nid

    def getInterceptCC(self):
        return self.__cc

    def getInterceptionStart(self):
        return self.__start
        
    def getLevel(self):
        return self.__level

    def __repr__(self):
        return "INI1AIntercept(%s, '%s', '%s', %s, %d, %d)" % (self.__cid, \
            self.__nid, self.__liid, str(self.__cc), time.mktime(self.__start), self.__level)

    def __str__(self):
        return repr(self)

    def __eq__(self, other):
        if not isinstance(other, self.__class__):
            return False
        return self.__liid == other.__liid and \
                self.__cid == other.__cid and \
                self.__nid == other.__nid and \
                self.__cc == other.__cc and \
                self.__start == other.__start and \
                self.__level == other.__level
