""" Test package of IRI-CORE """
# Copyright (C) 2013 Radek Hranický
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import unittest
import random
import sys
import time
import os
import datetime

from .helper_socket_classes import HelperUnderlyingSocket

from collections import deque

from modules.shared.ini1aintercept import INI1AIntercept
from modules.shared.cid import CID
from modules.shared.nid import NID
from modules.shared.nid import CreateNID
from modules.shared.nid import CreateNIDType
from modules.shared.nid import NIDIP
from modules.iriiif.intercept_table import *
from modules.iriiif.cin_table import *
from modules.iriiif.nid_graph import *

import copy
import modules.tools.log as log

iricore = __import__("iri-core")

def logEvent(eventText, severity = "default"):
    pass

class FakeClass():

    def __init__(self):
        self.calls = []
        self.params = []

    def addParam(self, *args, **kwargs):
        self.params.append(copy.deepcopy((args, kwargs))) # ! deepcopy

    def __getattr__(self, attr):
        self.calls.append(attr)
        return self.addParam
        
    def clear(self):
        self.calls = []
        self.params = []
        

class Tests(unittest.TestCase):
  
    def setUp(self):
        iricore.iricore_init()

    @classmethod
    def setUpClass(cls):
        random.seed(5)
    
    def testStaticNIDCCLookup(self):
        graph = NIDGraph()
        nidX = CreateNIDType("1.2.3.4", "IPv4")
        nidY = CreateNIDType("aa-bb-cc-00-00-00", "MAC")
        self.assertTrue(graph.nidLookup(nidX, 1, "") == [nidX])
        self.assertTrue(graph.nidLookup(nidY, 1, "") == [])
       
    def testNidGraphUpdates(self):
        graph = NIDGraph()
        
        # Create test NIDs
        a = CreateNIDType("1.1.1.1", "IPv4")
        b = CreateNIDType("2.2.2.2", "IPv4")
        c = CreateNIDType("3.3.3.3", "IPv4")
        d = CreateNIDType("4.4.4.4", "IPv4")
        e = CreateNIDType("5.5.5.5", "IPv4")
        
        # Check for basic behavior
        self.assertFalse(graph.nodeExists(a))
        self.assertFalse(graph.nodeExists(b))
        self.assertFalse(graph.nodeExists(c))
        self.assertFalse(graph.nodeExists(d))
        self.assertFalse(graph.nodeExists(e))
        graph.update_addConnection(a, b, "test")
        graph.update_addConnection(b, c, "test")
        self.assertTrue(graph.nodeExists(a))
        self.assertTrue(graph.nodeExists(b))
        self.assertTrue(graph.nodeExists(c))
        self.assertFalse(graph.nodeExists(d))
        self.assertFalse(graph.nodeExists(e))
        self.assertTrue(graph.connectionExists(a, b, "test"))
        self.assertTrue(graph.connectionExists(b, a, "test"))
        graph.update_delConnection(b, c, "test")
        self.assertTrue(graph.nodeExists(b))
        self.assertFalse(graph.nodeExists(c))
        
        # Test for behavior when we have two connections between the same nodes
        graph.update_addConnection(a, b, "another")
        self.assertTrue(graph.connectionExists(a, b, "test"))
        self.assertTrue(graph.connectionExists(a, b, "another"))
        self.assertTrue(graph.nodeExists(a))
        self.assertTrue(graph.nodeExists(b))
        graph.update_delConnection(a, b, "test")
        self.assertFalse(graph.connectionExists(a, b, "test"))
        self.assertTrue(graph.connectionExists(a, b, "another"))
        self.assertTrue(graph.nodeExists(a))
        self.assertTrue(graph.nodeExists(b))
        graph.update_delConnection(a, b, "another")
        self.assertFalse(graph.connectionExists(a, b, "test"))
        self.assertFalse(graph.connectionExists(a, b, "another"))
        self.assertFalse(graph.nodeExists(a))
        self.assertFalse(graph.nodeExists(b))
    
    def testNodeLookupBasic(self):
        graph = NIDGraph()
        a = CreateNIDType("1.1.1.1", "IPv4")
        b = CreateNIDType("2.2.2.2", "IPv4")
        t1 = CreateNIDType(('1.1.1.1', 1, '2.2.2.2', 2), "TCP")
        l = CreateNIDType("L", "XMPP LOGIN")
        
        graph.addNode(a)
        graph.addNode(b)
        graph.addNode(t1)
        graph.addNode(l)
        
        graph.addConnection(a, t1, "XMPP")
        graph.addConnection(b, t1, "XMPP")
        graph.addConnection(l, t1, "XMPP")
        
        nidSet = {a, t1}
        self.assertEqual(graph.nidLookup(a,1,""), nidSet)
         
    def testNodeLookup(self):
        graph = NIDGraph()
        p = CreateNIDType("p", "PPP Login")
        r = CreateNIDType("r", "Radius Login")
        x = CreateNIDType("aa-aa-aa-aa-aa-aa", "MAC")
        y = CreateNIDType("dd-dd-dd-dd-dd-dd", "MAC")
        a = CreateNIDType("1.1.1.1", "IPv4")
        b = CreateNIDType("fd02::6666", "IPv6")
        c = CreateNIDType("9.9.9.9", "IPv4")
        d = CreateNIDType("fd02::1010", "IPv6")
        e = CreateNIDType("2.2.2.2", "IPv4")
        f = CreateNIDType("fd02::8888", "IPv6")
        g = CreateNIDType("fd02::7777", "IPv6")
        h = CreateNIDType("h", "DUID")
        i = CreateNIDType("4.4.4.4", "IPv4")
        j = CreateNIDType("3.3.3.3", "IPv4")
        k = CreateNIDType("5.5.5.5", "IPv4")
        l = CreateNIDType("L", "XMPP LOGIN")
        m = CreateNIDType("M", "XMPP LOGIN")
        t1 = CreateNIDType(('1.1.1.1', 1, '2.2.2.2', 2), "TCP")
        t2 = CreateNIDType(('3.3.3.3', 3, '4.4.4.4', 4), "TCP")
        t3 = CreateNIDType(('5.5.5.5', 5, '4.4.4.4', 6), "TCP")
        s = CreateNIDType("s", "DUID")
        
        # Add nodes
        graph.addNode(p)
        graph.addNode(r)
        graph.addNode(x)
        graph.addNode(y)
        graph.addNode(a)
        graph.addNode(b)
        graph.addNode(c)
        graph.addNode(d)
        graph.addNode(e)
        graph.addNode(f)
        graph.addNode(g)
        graph.addNode(h)
        graph.addNode(i)
        graph.addNode(j)
        graph.addNode(k)
        graph.addNode(l)
        graph.addNode(m)
        graph.addNode(t1)
        graph.addNode(t2)
        graph.addNode(t3)
        graph.addNode(s)
        
        # Add connections
        graph.addConnection(p, x, "PPPoE")
        graph.addConnection(p, y, "PPPoE")
        graph.addConnection(x, a, "DHCP")
        graph.addConnection(x, b, "SLAAC")
        graph.addConnection(h, g, "DHCPV6")
        graph.addConnection(r, y, "RADIUS")
        graph.addConnection(y, c, "RADIUS")
        graph.addConnection(r, c, "RADIUS")
        graph.addConnection(y, d, "SLAAC")
        graph.addConnection(h, f, "DHCPv6")
        graph.addConnection(l, t1, "XMPP")
        graph.addConnection(e, t1, "XMPP")
        graph.addConnection(a, t1, "XMPP")
        graph.addConnection(s, b, "DHCPv6")
        graph.addConnection(s, g, "DHCPv6")
        graph.addConnection(l, t2, "XMPP")
        graph.addConnection(i, t2, "XMPP")
        graph.addConnection(j, t2, "XMPP")
        graph.addConnection(m, t3, "XMPP")
        graph.addConnection(i, t3, "XMPP")
        graph.addConnection(k, t3, "XMPP")
        
        # Make test sets
        # sets for level 1 tests
        set1A = {l, t1, t2}
        set1B = {a, t1}
        set1C = {x, a, t1, b}
        set1D = {p, x, a, t1, b, y, c, d}
        # sets for level 2 tests
        set2A = {l, t1, a, x, b, s, g, h, f, t2, j}
        set2BC = {a, t1, x, b, s, g, h, f}
        set2D = {p, x, a, t1, b, y, c, d, s, g, h, f}
        # sets for level 3 tests
        set3A = {l, t1, a, x, b, s, g, h, f, t2, j, p, y, r, c, d}
        set3BCD = {a, t1, x, b, s, g, h, f, p, y, r, c, d}
        
        # Perform the tests
        # level 1
        self.assertEqual(graph.nidLookup(l,1,""), set1A)
        self.assertEqual(graph.nidLookup(a,1,""), set1B)
        self.assertEqual(graph.nidLookup(x,1,""), set1C)
        self.assertEqual(graph.nidLookup(p,1,""), set1D)
        # level 2
        self.assertEqual(graph.nidLookup(l,2,""), set2A)
        self.assertEqual(graph.nidLookup(a,2,""), set2BC)
        self.assertEqual(graph.nidLookup(x,2,""), set2BC)
        self.assertEqual(graph.nidLookup(p,2,""), set2D)
        # level 3
        self.assertEqual(graph.nidLookup(l,3,""), set3A)
        self.assertEqual(graph.nidLookup(a,3,""), set3BCD)
        self.assertEqual(graph.nidLookup(x,3,""), set3BCD)
        self.assertEqual(graph.nidLookup(p,3,""), set3BCD)
        
        # test correct behavior when we have NonUnique NIDs
        graph.setNonUniqueNIDs([x])
        self.assertEqual(graph.nidLookup(l,3,""), {l, t1, a, t2, j})
        graph.setNonUniqueNIDs([])
        self.assertEqual(graph.nidLookup(l,3,""), set3A)
        
    def testIceptTable(self):
        itable = InterceptTable()
        icept1 = INI1AIntercept(CID('BUT', '192.168.0.2', '0', 'CZ'), '192.168.0.2', 'xxx', True, 1374098400, 1)
        icept2 = INI1AIntercept(CID('VUTBR', '8.8.8.8', '0', 'CZ'), '8.8.8.8', 'yyy', False, 1377468000, 1)
        itable.addIntercept(icept1)
        itable.addIntercept(icept2)
        self.assertEqual(itable.getNID("xxx"), "192.168.0.2")
        cin1 = itable.getCIN("xxx")
        cin1 += 2
        itable.setCIN("xxx", cin1)
        itable.increaseCIN("yyy")
        self.assertEqual(itable.getCIN("xxx"), 2)
        self.assertEqual(itable.getCIN("yyy"), 1)
  
    def testCinTable(self):
        itable = InterceptTable()
        cintable = CINTable()
        icept1 = INI1AIntercept(CID('BUT', '192.168.0.2', '0', 'CZ'), '192.168.0.2', 'xxx', True, 1374098400, 1)
        icept2 = INI1AIntercept(CID('VUTBR', '8.8.8.8', '0', 'CZ'), '8.8.8.8', 'yyy', False, 1377468000, 1)
        itable.addIntercept(icept1)
        itable.addIntercept(icept2)
        newCin1 = itable.getCIN("xxx") + 1
        cintable.update(icept1.getNID(), icept1.getLIID(), newCin1)
        itable.setCIN("xxx", newCin1)
        self.assertEqual(cintable.getCIN(icept1.getNID(), icept1.getLIID()), 1)
        
    def testNIDipINRange(self):
        range1 = CreateNID("fe80::/64")
        ip1 = CreateNID("fe80::1")
        ip1b = CreateNID("fe80:abcd::1")
        range2 = CreateNID("10.0.0.0/24")
        ip2 = CreateNID("10.0.0.1")
        ip2b = CreateNID("10.0.1.1")
        self.assertTrue(ip1 in range1)
        self.assertFalse(range1 in ip1)
        self.assertFalse(ip1b in range1)
        self.assertTrue(ip2 in range2)
        self.assertFalse(range2 in ip2)
        self.assertFalse(ip2b in range2)

    def testNIDIipINTCP(self):
        # ========== TCP with IPv4 ==========
        conn = CreateNIDType(('1.1.1.1', 1, '2.2.2.2', 2), "TCP")
        a = CreateNID("1.1.1.1")
        b = CreateNID("2.2.2.2")
        c = CreateNID("3.3.3.3")
        d = CreateNID("1.1.1.0/24")
        e = CreateNID("4.4.4.0/24")
        # -- CONNECTION in IP
        self.assertTrue(conn in a)
        self.assertTrue(conn in b)
        self.assertFalse(conn in c)
        self.assertTrue(conn in d)
        self.assertFalse(conn in e)
        # -- IP in CONNECTION
        self.assertFalse(a in conn)
        self.assertFalse(b in conn)
        self.assertFalse(c in conn)
        self.assertFalse(d in conn)
        self.assertFalse(e in conn)
        
        # ========== TCP with IPv6 ==========
        conn6 = CreateNIDType(('2001::1', 1, '2001::2', 2), "TCP")
        u = CreateNID("2001::1")
        v = CreateNID("2001::2")
        w = CreateNID("2001::3")
        x = CreateNID("2001::/64")
        y = CreateNID("fe80::/64")
        # -- CONNECTION in IP
        self.assertTrue(conn6 in u)
        self.assertTrue(conn6 in v)
        self.assertFalse(conn6 in w)
        self.assertTrue(conn6 in x)
        self.assertFalse(conn6 in y)
        # -- IP in CONNECTION
        self.assertFalse(u in conn6)
        self.assertFalse(v in conn6)
        self.assertFalse(w in conn6)
        self.assertFalse(x in conn6)
        self.assertFalse(y in conn6)
        
        # ========== mixed IPv4/IPv6 (unacceptable) ==========
        # -- TCP (IPv6) connection and IPv4 address conflicts
        self.assertFalse(conn6 in a)
        self.assertFalse(conn6 in b)
        self.assertFalse(conn6 in c)
        self.assertFalse(conn6 in d)
        self.assertFalse(conn6 in e)
        self.assertFalse(a in conn6)
        self.assertFalse(b in conn6)
        self.assertFalse(c in conn6)
        self.assertFalse(d in conn6)
        self.assertFalse(e in conn6)
        # -- TCP (IPv4) connection and IPv6 address conflicts
        self.assertFalse(conn in u)
        self.assertFalse(conn in v)
        self.assertFalse(conn in w)
        self.assertFalse(conn in x)
        self.assertFalse(conn in y)
        self.assertFalse(u in conn)
        self.assertFalse(v in conn)
        self.assertFalse(w in conn)
        self.assertFalse(x in conn)
        self.assertFalse(y in conn)
      
    def testRangeSearch(self):
        graph = NIDGraph()
      
        # Create test NIDs
        a = CreateNIDType("1.1.1.1", "IPv4")
        b = CreateNIDType("1.1.1.2", "IPv4")
        c = CreateNIDType("1.1.1.3", "IPv4")
        d = CreateNIDType("2.2.2.1", "IPv4")
        e = CreateNIDType("2.2.2.2", "IPv4")
        f = CreateNIDType("2.2.2.3", "IPv4")
        g = CreateNIDType("3.3.3.1", "IPv4")
        h = CreateNIDType("3.3.3.2", "IPv4")
        i = CreateNIDType("3.3.3.3", "IPv4")
        x = CreateNIDType("aa-aa-aa-aa-aa-aa", "MAC")
        y = CreateNIDType("bb-bb-bb-bb-bb-bb", "MAC")
        z = CreateNIDType("cc-cc-cc-cc-cc-cc", "MAC")
        test1 = CreateNIDType("1.1.1.0/24", "IPv4")
        test2 = CreateNIDType("2.2.0.0/16", "IPv4")
        test3 = CreateNIDType("0.0.0.0/0", "IPv4")
        
        # Create relationships
        graph.update_addConnection(a, x, "test")
        graph.update_addConnection(b, x, "test")
        graph.update_addConnection(c, x, "test")
        graph.update_addConnection(d, y, "test")
        graph.update_addConnection(e, y, "test")
        graph.update_addConnection(f, y, "test")
        graph.update_addConnection(g, z, "test")
        graph.update_addConnection(h, z, "test")
        graph.update_addConnection(i, z, "test")
        
        # Test for lookup
        result1 = graph.nidLookup(test1, 3, "")
        result2 = graph.nidLookup(test2, 3, "")
        result3 = graph.nidLookup(test3, 3, "")
        self.assertTrue(a in result1 and b in result1 and c in result1 and x in result1)
        self.assertFalse(a in result2 or b in result2 or c in result2 or x in result2)
        self.assertTrue(d in result2 and e in result2 and f in result2 and y in result2)
        self.assertTrue(a in result3 and h in result3 and y in result3 and z in result3)


    def testMakingRelationship(self):
        a = CreateNIDType("1.1.1.1", "IPv4")
        b = CreateNIDType("1.1.1.2", "IPv4")
        c = CreateNIDType("1.1.1.3", "IPv4")
        x = CreateNIDType("aa-aa-aa-aa-aa-aa", "MAC")
        nidSet = {a, b, c, x}
        couples = iricore.eachWithEach(nidSet)
        self.assertEqual(len(couples),6)


    @unittest.skip("Not working in SIMS")
    def testGraphIntervalsTicket259T1(self):
        """ Test sequence provided by VRS

        A sequence of different ranges of IP addresses
        """
        def checkGeneratediri_report(iri, liid_orig, cid_orig, nid_orig):
            iritype, destination, iribody = iri
            self.assertEqual(iritype, "iri_report")
            self.assertEqual(destination, "mf")
            iristate, liid, cid, nid, _, _ = iribody
            self.assertEqual(iristate, "BEGIN")
            self.assertEqual(liid, liid_orig)
            self.assertEqual(cid, cid_orig)
            self.assertEqual(nid, nid_orig)
        def processActivateIntercept(icept_orig, liid_orig, nid_orig, cid_orig):
            sm = FakeClass()
            iricore.activateIntercept(sm, icept_orig)
            self.assertEqual(sm.calls, ['send'])
            self.assertEqual(len(sm.params), 1)
            iricall = sm.params[0]
            args, kwargs = iricall
            self.assertEqual(kwargs, {})
            iface, iri = args
            self.assertEqual(iface, "ini2")
            checkGeneratediri_report(iri, liid_orig, cid_orig, nid_orig)
        def processActivateInterceptPlus(icept_orig, liid_orig, nid_orig, cid_orig):
            sm = FakeClass()
            iricore.activateIntercept(sm, icept_orig)
            self.assertEqual(sm.calls, ['send'] * 2)
            self.assertEqual(len(sm.params), 2)
            observed_nids = []
            observed_cins = []
            for iricall in sm.params:
                args, kwargs = iricall
                self.assertEqual(kwargs, {})
                iface, iri = args
                self.assertEqual(iface, "ini2")
                iritype, destination, iribody = iri
                self.assertEqual(iritype, "iri_report")
                self.assertEqual(destination, "mf")
                iristate, liid, cid, nid, _, _ = iribody
                self.assertEqual(iristate, "BEGIN")
                self.assertEqual(liid, liid_orig)
                observed_cins.append(cid.getCIN())
                self.assertEqual(cid.getnoid(), cid_orig.getnoid())
                self.assertEqual(cid.getNID(), cid_orig.getNID())
                self.assertEqual(cid.getDCC(), cid_orig.getDCC())
                observed_nids.append(nid)
            self.assertEqual(set(observed_nids), set([nid_orig, '10.10.10.216']))
            self.assertEqual(set(observed_cins), set([1,2]))
        def processDHCPiri(liids, nids):
            sm = FakeClass()
            s = FakeClass()
            msg = ('56065', ('dhcp', 1405893600.00, 'BEGIN', 'new DHCP device',
                [('IPv4', '10.10.10.216'), ('MAC', '00:11:22:33:44:55')]))
            iricore.processMessageIRIINTERNAL(msg, sm, s)
            self.assertEqual(sm.calls, ["send"] * (len(nids) + 1)) # + iri_table
            observed_liids = []
            for args, kwargs in sm.params:
                self.assertEqual(kwargs, {})
                iface, iri = args
                self.assertEqual(iface, "ini2")
                iritype, destination, iribody = iri
                self.assertEqual(destination, "mf")
                if iritype == "iri_report":
                    iristate, liid, cid, nid, _, _ = iribody
                    self.assertEqual(iristate, "BEGIN")
                    observed_liids.append(liid)
                    self.assertTrue(liid in liids)
                    #self.assertEqual(cid.getCIN(), 2)
                    self.assertEqual(nid, "10.10.10.216")
                else:
                    self.assertEqual(iritype, 'iri_table')
            self.assertEqual(set(liids), set(observed_liids))
        iricore.iricore_init()
        liids = ("testing", "testing2", "testing3", "testing4", "testing5", "testing6", "testing7",
                "testing8", "testing9", )
        nids = ('10.0.0.0/8', '10.10.10.0/24', '10.10.10.208/28', '10.0.0.0/7', '8.0.0.0/5',
                '10.10.0.0/16', '10.10.0.0/29', '10.10.10.0/29', '10.10.10.128/25', )
        run1 = 3
        calls = [processActivateIntercept] * run1 + [processActivateInterceptPlus] * (len(liids)-run1)
        calls[-3] = processActivateIntercept
        calls[-2] = processActivateIntercept
        cids = []
        icepts = []
        for liid, nid in zip(liids, nids):
            c = CID('VUTBR', nid, 0, 'CZ')
            cids.append(c)
            icepts.append(INI1AIntercept(c, nid, liid, True, 1405893600, 1))
        zipped_icepts = list(zip(calls, icepts, liids, nids, cids))
        for call, *i in zipped_icepts[0:run1]:
            call(*i)
        processDHCPiri(liids[0:run1], nids[0:run1])
        for call, *i in zipped_icepts[run1:]:
            call(*i)

    def testGraphFromTCP(self):
        iricore.iricore_init()
        # Tests if IRI-CORE correctly updates NID graph if we receive a message (from a module) about a new TCP connection
        sm = FakeClass()
        s = FakeClass()
        msg = ('obs1', ('clockskew_javascript', 1377718543.66, 'BEGIN', 'new TCP Connection', [('TCP3', '192.168.0.1_9999'),]))
        msg2 = ('obs1', ('smtp', 1377718580.99, 'BEGIN', 'new TCP Connection', [('TCP', ('1.1.1.1', 1000, '2.2.2.2', 2000))]))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        iricore.processMessageIRIINTERNAL(msg2, sm, s)
        t1 = CreateNIDType("192.168.0.1_9999", "TCP3")
        t2 = CreateNIDType(('1.1.1.1', 1000, '2.2.2.2', 2000), "TCP")
        ip1 = CreateNIDType("192.168.0.1", "IPv4")
        ip2a = CreateNIDType("1.1.1.1", "IPv4")
        ip2b = CreateNIDType("2.2.2.2", "IPv4")
        self.assertTrue(iricore.nidGraph.nodeExists(t1))
        self.assertTrue(iricore.nidGraph.nodeExists(t2))
        self.assertTrue(iricore.nidGraph.nodeExists(ip1))
        self.assertTrue(iricore.nidGraph.nodeExists(ip2a))
        self.assertTrue(iricore.nidGraph.nodeExists(ip2b))
        self.assertTrue(iricore.nidGraph.connectionExists(t1, ip1, "clockskew_javascript"))
        self.assertTrue(iricore.nidGraph.connectionExists(t2, ip2a, "smtp"))
        self.assertTrue(iricore.nidGraph.connectionExists(t2, ip2b, "smtp"))
        self.assertFalse(iricore.nidGraph.connectionExists(ip2a, ip2b, "smtp"))
        sm.calls = []
        sm.params = []

    def testGraphFromTCP2(self):
        iricore.iricore_init()
        # Tests the behavior if we have more TCP connections in a single message
        sm = FakeClass()
        s = FakeClass()
        msg = ('obs1', ('clockskew_javascript', 1378472322.463442, 'CONTINUE', \
            'Similar clock skew', [('TCP3', '10.10.10.101_49486'), ('TCP3', 'fd71:5715:5715:5715:7ed1:c3ff:fedc:1c83_49487')]))
        iricore.processMessageIRIINTERNAL(msg, sm, s)

        t1 = CreateNIDType("10.10.10.101_49486", "TCP3")
        t2 = CreateNIDType("fd71:5715:5715:5715:7ed1:c3ff:fedc:1c83_49487", "TCP3")
        ip1 = CreateNIDType("10.10.10.101", "IPv4")
        ip2 = CreateNIDType("fd71:5715:5715:5715:7ed1:c3ff:fedc:1c83", "IPv6")
        
        self.assertTrue(iricore.nidGraph.nodeExists(t1))
        self.assertTrue(iricore.nidGraph.nodeExists(t2))
        
        # IRI-CORE should add new NIDs = IP from each TCP connection
        self.assertTrue(iricore.nidGraph.nodeExists(ip1))
        self.assertTrue(iricore.nidGraph.nodeExists(ip2))

        # IRI-CORE should connect each IP with its TCP connection
        # IRI-CORE should connect both TCP connections together (received from the same message)
        self.assertTrue(iricore.nidGraph.connectionExists(t1, ip1, "clockskew_javascript"))
        self.assertTrue(iricore.nidGraph.connectionExists(t2, ip2, "clockskew_javascript"))
        self.assertTrue(iricore.nidGraph.connectionExists(t1, t2, "clockskew_javascript"))
        
        # !!! IRI-CORE hould NOT connect both newly added IPs together
        self.assertFalse(iricore.nidGraph.connectionExists(ip1, ip2, "clockskew_javascript"))
        
        sm.calls = []
        sm.params = []
    
    
    def testCorrectCINs(self): 
        # Tests the behavior if we have more TCP connections in a single message
        sm = FakeClass()
        s = FakeClass()
        iricore.iricore_init()
        
        msgNewIcept = ('new_intercept', 'iri-iif', INI1AIntercept(CID('BUT', '10.10.10.101', '0', 'CZ'), '10.10.10.101', 'xxx', True, 1374098400, 3))
        iricore.processMessageINI1A(msgNewIcept, sm, s)
        
        msg = ('obs1', ('dhcp', 1378472300.463442, 'BEGIN', \
            'client has assigned new IP address   10.10.10.101', [('DHCP client ID', '0x017cd1c3dc1c83'), ('IPv4', '10.10.10.101'), ('MAC', '7c:d1:c3:dc:1c:83')]))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        msg = ('obs1', ('dhcp', 1378472311.463442, 'CONTINUE', \
            'client renewed IP address assignment   10.10.10.101', [('DHCP client ID', '0x017cd1c3dc1c83'), ('IPv4', '10.10.10.101'), ('MAC', '7c:d1:c3:dc:1c:83')]))
        iricore.processMessageIRIINTERNAL(msg, sm, s)

        msg = ('obs1', ('clockskew_javascript', 1378472322.463442, 'BEGIN', \
            'Similar clock skew', [('TCP3', '10.10.10.101_49486'), ('IPv4', '10.10.10.101')]))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        msg = ('obs1', ('clockskew_javascript', 1378472333.463442, 'CONTINUE', \
            'Similar clock skew', [('TCP3', '10.10.10.101_49486'), ('IPv4', '10.10.10.101')]))
        iricore.processMessageIRIINTERNAL(msg, sm, s)

        # Make test NIDs
        test_ip = CreateNIDType("10.10.10.101", "IPv4")
        test_tcp = CreateNIDType("10.10.10.101_49486", "TCP3")

        # Test for correct CINs in CIN table
        self.assertEqual(iricore.cinTable.getCIN(test_ip, "xxx"), 1)
        self.assertEqual(iricore.cinTable.getCIN(test_tcp, "xxx"), 2)
        
        # Test for correct CIN in Intercept table
        self.assertEqual(iricore.interceptTable.getCIN("xxx"), 2)

        sm.calls = []
        sm.params = []
        
    def testGraphChangesTest(self):
        graph = NIDGraph()
        a = CreateNIDType("1.1.1.1", "IPv4")
        b = CreateNIDType("2.2.2.2", "IPv4")
        c = CreateNIDType("3.3.3.3", "IPv4")        
        self.assertTrue(graph.update_addConnection(a, b, "test"))
        self.assertTrue(graph.update_addConnection(b, c, "test"))
        self.assertTrue(graph.update_addConnection(a, c, "test"))
        self.assertFalse(graph.update_addConnection(b, c, "test"))
        self.assertTrue(graph.update_delConnection(a, b, "test"))
        self.assertFalse(graph.update_delConnection(a, b, "test"))
        
    def testLIIDSpecificGraphs(self):
        graph = NIDGraph()
        a = CreateNIDType("1.1.1.1", "IPv4")
        b = CreateNIDType("2.2.2.2", "IPv4")
        
        self.assertTrue(graph.update_addConnection(a, b, "test"))
        self.assertFalse(graph.update_addConnection(a, b, "test", LIID="MyInterception"))
        self.assertTrue(graph.update_delConnection(a, b, "test"))
        
        set_x = {a}
        set_y = {a, b}
        self.assertEqual(set(graph.nidLookup(a,3,"")), set_x)
        self.assertEqual(set(graph.nidLookup(a,3,LIID="MyInterception")), set_y)
        
    def testLIIDSpecificLookup(self):
        graph = NIDGraph()
        a = CreateNIDType("1.1.1.1", "IPv4")
        b = CreateNIDType("2.2.2.2", "IPv4")
        c = CreateNIDType("3.3.3.3", "IPv4")
        
        self.assertTrue(graph.update_addConnection(a, b, "test"))
        self.assertFalse(graph.update_addConnection(b, c, "test", LIID="MyInterception"))
        
        set_x = {a, b}
        set_y = {a, b, c}
        self.assertEqual(set(graph.nidLookup(a,3,"")), set_x)
        self.assertEqual(set(graph.nidLookup(a,3,LIID="MyInterception")), set_y)
        
    def testGraphFromEMAIL(self):
        iricore.iricore_init()
        # Tests if IRI-CORE correctly updates NID graph if we receive a message (from a module) about a new E-mail message
        sm = FakeClass()
        s = FakeClass()
        msg = ('Jolanda', ('smtp', 1409158003.11, 'CONTINUE', 'An e-mail of length xxx bytes successfully sent', [('TCP', ('74.53.140.153', 25, '10.10.1.4', 1470)), ('E-mail message', (('E-mail address', 'gurpartap@patriots.in'), [('E-mail address', 'raj_deol2002in@yahoo.co.in')], ('E-mail Message-ID', 'unknown')))]))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        t = CreateNIDType(('74.53.140.153', 25, '10.10.1.4', 1470), "TCP")
        ipc = CreateNIDType("74.53.140.153", "IPv4")
        ips = CreateNIDType("10.10.1.4", "IPv4")
        
        emailmsg = CreateNIDType((('E-mail address', 'gurpartap@patriots.in'), [('E-mail address', 'raj_deol2002in@yahoo.co.in')], ('E-mail Message-ID', 'unknown')), "E-mail message")
        snd_email = CreateNIDType('gurpartap@patriots.in', "E-mail address")
        rcpt1_email = CreateNIDType('raj_deol2002in@yahoo.co.in', "E-mail address")
        msgid = CreateNIDType('unknown', "E-mail Message-ID")        
        
        self.assertTrue(iricore.nidGraph.nodeExists(t))
        self.assertTrue(iricore.nidGraph.nodeExists(ipc))
        self.assertTrue(iricore.nidGraph.nodeExists(ips))
        self.assertTrue(iricore.nidGraph.nodeExists(emailmsg))
        self.assertTrue(iricore.nidGraph.nodeExists(snd_email))
        self.assertTrue(iricore.nidGraph.nodeExists(rcpt1_email))
        
        sm.calls = []
        sm.params = []
    

    def testTicket278(self):
        iricore.iricore_init()
        
        sm = FakeClass()
        s = FakeClass()
        
        # Add interceptions
        
        msgNewIcept = ('new_intercept', 'iri-iif', INI1AIntercept(CID('BUT', 'email:sec6net.demo@seznam.cz', '0', 'CZ'), 'email:sec6net.demo@seznam.cz', 't1', True, 1374098400, 1))
        iricore.processMessageINI1A(msgNewIcept, sm, s)
        
        msgNewIcept = ('new_intercept', 'iri-iif', INI1AIntercept(CID('BUT', 'email:sec6net.demo@seznam.cz', '0', 'CZ'), 'email:sec6net.demo@seznam.cz', 't3', True, 1374098400, 3))
        iricore.processMessageINI1A(msgNewIcept, sm, s)
        
        
        # Send IRI messages
        
        msg = ("Jolanda", ('smtp', 1410010652.48, 'REPORT', 'User has connected to the SMTP server', [('TCP', ('77.75.76.48', 25, '10.10.10.226', 52129))]) )
        iricore.processMessageIRIINTERNAL(msg, sm, s)        
        #iricore.cinTable.debug_printCinTable()
        
        msg = ("Jolanda", ('smtp', 1410010652.28, 'BEGIN', 'The user was successfully authenticated', [('TCP', ('77.75.76.48', 25, '10.10.10.226', 52129))]) )
        iricore.processMessageIRIINTERNAL(msg, sm, s)        
        #iricore.cinTable.debug_printCinTable()
        
        msg = ("Jolanda", ('smtp', 1410010652.14, 'CONTINUE', 'An e-mail of length xxx bytes successfully sent', [('TCP', ('77.75.76.48', 25, '10.10.10.226', 52129)), ('E-mail message', ( ('E-mail address', 'sec6net.demo@seznam.cz'), [('E-mail address', 'ipolcak@fit.vutbr.cz')], ('E-mail Message-ID', 'unknown')) )], (0, 6, "77.75.76.48", 25, "10.10.10.226", 52129)) )
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        #iricore.cinTable.debug_printCinTable()
        
        msg = ("Jolanda", ('smtp', 1410010652.11, 'END', 'User has disconnected from the SMTP server - (TCP, (77.75.76.48, 25, 10.10.10.226, 52129))', []) )
        iricore.processMessageIRIINTERNAL(msg, sm, s)        
        #iricore.cinTable.debug_printCinTable()
        
        server_ip = CreateNID("10.10.10.226")
        mail_addr = CreateNIDType("sec6net.demo@seznam.cz", "E-mail address") 
        t3_nids = iricore.nidGraph.nidLookup(mail_addr, 3, LIID="t3")
        
        self.assertFalse(server_ip in t3_nids)
        
        sm.calls = []
        sm.params = []
        
        
    def testNodePersistence(self):
        graph = NIDGraph()
        
        # Create test NIDs
        a = CreateNIDType("1.1.1.1", "IPv4")
        b = CreateNIDType("2.2.2.2", "IPv4")
        c = CreateNIDType("3.3.3.3", "IPv4")
        d = CreateNIDType("4.4.4.4", "IPv4")
        e = CreateNIDType("5.5.5.5", "IPv4")
        
        # Test node behavior
        graph.addNode(a)
        self.assertTrue(graph.nodeExists(a))
        graph.delNode(a)
        self.assertFalse(graph.nodeExists(a))
        graph.addPersistentNode(a)
        self.assertTrue(graph.nodeExists(a))
        graph.delNode(a)
        self.assertTrue(graph.nodeExists(a))
            
       
    def testSpecialLists_persistent(self):
        iricore.iricore_init()
        sm = FakeClass()
        s = FakeClass()
        
        a = CreateNIDType("1.1.1.1", "IPv4")
        b = CreateNIDType("2.2.2.2", "IPv4")
        
        msg = ('obs1', ('test_module', 1377718543.66, 'BEGIN', 'xxx', [('IP', '1.1.1.1'),('IP', '2.2.2.2')], [], [('IP', '1.1.1.1')]))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        self.assertTrue(iricore.nidGraph.nodeExists(a))
        self.assertTrue(iricore.nidGraph.nodeExists(b))
        self.assertTrue(iricore.nidGraph.connectionExists(a, b, "test_module"))
        
        msg = ('obs1', ('test_module', 1377718543.66, 'END', 'xxx', [('IP', '1.1.1.1'),('IP', '2.2.2.2')], [], []))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        self.assertTrue(iricore.nidGraph.nodeExists(a))
        self.assertFalse(iricore.nidGraph.nodeExists(b))
        self.assertFalse(iricore.nidGraph.connectionExists(a, b, "test_module"))
        
        sm.calls = []
        sm.params = []
        
        
    def testSpecialLists_tcp3_behavior(self):
        iricore.iricore_init()
        sm = FakeClass()
        s = FakeClass()
        
        tcp3 = CreateNIDType("10.10.10.101_49486", "TCP3")
        tcp3_ip = CreateNIDType("10.10.10.101", "IPv4")
        
        msg = ('obs1', ('test_module', 1377718543.66, 'BEGIN', 'xxx', [('TCP3', '10.10.10.101_49486'),('MAC', 'aa-aa-aa-aa-aa-aa')], [], []))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        self.assertTrue(iricore.nidGraph.nodeExists(tcp3))
        self.assertTrue(iricore.nidGraph.nodeExists(tcp3_ip))
        self.assertTrue(iricore.nidGraph.connectionExists(tcp3, tcp3_ip, "test_module"))
        
        msg = ('obs1', ('test_module', 1377718543.66, 'END', 'xxx', [('TCP3', '10.10.10.101_49486'),('MAC', 'aa-aa-aa-aa-aa-aa')], [], []))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        self.assertFalse(iricore.nidGraph.nodeExists(tcp3))
        self.assertFalse(iricore.nidGraph.nodeExists(tcp3_ip))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp3, tcp3_ip, "test_module"))
        
        msg = ('obs1', ('test_module', 1377718543.66, 'BEGIN', 'xxx', [('TCP3', '10.10.10.101_49486'),('MAC', 'aa-aa-aa-aa-aa-aa')], [('IP', '10.10.10.101')], []))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        self.assertTrue(iricore.nidGraph.nodeExists(tcp3))
        self.assertFalse(iricore.nidGraph.nodeExists(tcp3_ip))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp3, tcp3_ip, "test_module"))
        
        msg = ('obs1', ('test_module', 1377718543.66, 'END', 'xxx', [('TCP3', '10.10.10.101_49486'),('MAC', 'aa-aa-aa-aa-aa-aa')], [], []))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        self.assertFalse(iricore.nidGraph.nodeExists(tcp3))
        self.assertFalse(iricore.nidGraph.nodeExists(tcp3_ip))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp3, tcp3_ip, "test_module"))

        sm.calls = []
        sm.params = []
        
        
    def testSpecialLists_tcp_behavior(self):
        iricore.iricore_init()
        sm = FakeClass()
        s = FakeClass()
        
        tcp = CreateNIDType(('1.1.1.1', 1000, '2.2.2.2', 2000), "TCP")
        tcp_ip_a = CreateNIDType("1.1.1.1", "IPv4")
        tcp_ip_b = CreateNIDType("2.2.2.2", "IPv4")
        
        msg = ('obs1', ('test_module', 1377718543.66, 'BEGIN', 'xxx', [('TCP', ('1.1.1.1', 1000, '2.2.2.2', 2000)),('MAC', 'aa-aa-aa-aa-aa-aa')], [], []))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        self.assertTrue(iricore.nidGraph.nodeExists(tcp))
        self.assertTrue(iricore.nidGraph.nodeExists(tcp_ip_a))
        self.assertTrue(iricore.nidGraph.nodeExists(tcp_ip_b))
        self.assertTrue(iricore.nidGraph.connectionExists(tcp, tcp_ip_a, "test_module"))
        self.assertTrue(iricore.nidGraph.connectionExists(tcp, tcp_ip_b, "test_module"))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp_ip_a, tcp_ip_b, "test_module"))
        
        msg = ('obs1', ('test_module', 1377718543.66, 'END', 'xxx', [('TCP', ('1.1.1.1', 1000, '2.2.2.2', 2000)),('MAC', 'aa-aa-aa-aa-aa-aa')], [], []))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        self.assertFalse(iricore.nidGraph.nodeExists(tcp))
        self.assertFalse(iricore.nidGraph.nodeExists(tcp_ip_a))
        self.assertFalse(iricore.nidGraph.nodeExists(tcp_ip_b))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp, tcp_ip_a, "test_module"))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp, tcp_ip_b, "test_module"))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp_ip_a, tcp_ip_b, "test_module"))
        
        msg = ('obs1', ('test_module', 1377718543.66, 'BEGIN', 'xxx', [('TCP', ('1.1.1.1', 1000, '2.2.2.2', 2000)),('MAC', 'aa-aa-aa-aa-aa-aa')], [('IP', '1.1.1.1'),('IP', '2.2.2.2')], []))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        self.assertTrue(iricore.nidGraph.nodeExists(tcp))
        self.assertFalse(iricore.nidGraph.nodeExists(tcp_ip_a))
        self.assertFalse(iricore.nidGraph.nodeExists(tcp_ip_b))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp, tcp_ip_a, "test_module"))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp, tcp_ip_b, "test_module"))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp_ip_a, tcp_ip_b, "test_module"))
        
        msg = ('obs1', ('test_module', 1377718543.66, 'END', 'xxx', [('TCP', ('1.1.1.1', 1000, '2.2.2.2', 2000)),('MAC', 'aa-aa-aa-aa-aa-aa')], [], []))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        self.assertFalse(iricore.nidGraph.nodeExists(tcp))
        self.assertFalse(iricore.nidGraph.nodeExists(tcp_ip_a))
        self.assertFalse(iricore.nidGraph.nodeExists(tcp_ip_b))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp, tcp_ip_a, "test_module"))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp, tcp_ip_b, "test_module"))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp_ip_a, tcp_ip_b, "test_module"))
        
        msg = ('obs1', ('test_module', 1377718543.66, 'BEGIN', 'xxx', [('TCP', ('1.1.1.1', 1000, '2.2.2.2', 2000)),('MAC', 'aa-aa-aa-aa-aa-aa')], [('IP', '1.1.1.1')], []))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        self.assertTrue(iricore.nidGraph.nodeExists(tcp))
        self.assertFalse(iricore.nidGraph.nodeExists(tcp_ip_a))
        self.assertTrue(iricore.nidGraph.nodeExists(tcp_ip_b))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp, tcp_ip_a, "test_module"))
        self.assertTrue(iricore.nidGraph.connectionExists(tcp, tcp_ip_b, "test_module"))
        self.assertFalse(iricore.nidGraph.connectionExists(tcp_ip_a, tcp_ip_b, "test_module"))

        sm.calls = []
        sm.params = []


    @unittest.skip("Not working in SIMS")
    def testSpecialLists_ini2format(self):
        iricore.iricore_init()
        sm = FakeClass()
        s = FakeClass()
        
        tcp = CreateNIDType(('1.1.1.1', 1000, '2.2.2.2', 2000), "TCP")
        tcp_ip_a = CreateNIDType("1.1.1.1", "IPv4")
        tcp_ip_b = CreateNIDType("2.2.2.2", "IPv4")
        
        msgNewIcept = ('new_intercept', 'iri-iif', INI1AIntercept(CID('BUT', 'aa-aa-aa-aa-aa-aa', '0', 'CZ'), 'aa-aa-aa-aa-aa-aa', 'xxx', True, 1374098400, 1))
        iricore.processMessageINI1A(msgNewIcept, sm, s)
        
        msg = ('obs1', ('test_module', 1377718543.66, 'BEGIN', 'xxx', [('IP', '1.1.1.1'),('MAC', 'aa-aa-aa-aa-aa-aa')], [('MAC', 'bb-bb-bb-bb-bb-bb')], []))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        outer_msg = sm.params[2][0][1][2]
        self.assertEqual(len(outer_msg), 6)
        
        outer_iriType, interception, cid, msg_nidcc, inner_msg, icept_nids = outer_msg
        # Should look like this:
        # (
        # 'BEGIN', 'xxx', CID('BUT', 'aa-aa-aa-aa-aa-aa', '1', 'CZ'), '1.1.1.1',
        # ('test_module', 1377718543.66, 'BEGIN', 'xxx | {bb:bb:bb:bb:bb:bb}', [('IP', '1.1.1.1'), ('MAC', 'aa-aa-aa-aa-aa-aa')]),
        # ('aa:aa:aa:aa:aa:aa', '1.1.1.1')
        # )
        
        # Parse outer message (Whole message passed to MF)
        self.assertEqual(outer_iriType, "BEGIN")
        self.assertEqual(interception, "xxx")
        ip = CreateNID(msg_nidcc)
        self.assertEqual(ip, tcp_ip_a)
        self.assertEqual(len(inner_msg), 5)        
        
        # Parse inner message (Modified former message "from IRI module")
        moduleID, timestamp, inner_iriType, description, nidArray = inner_msg
        self.assertEqual(moduleID, "test_module")
        self.assertEqual(timestamp, 1377718543.66)
        self.assertEqual(inner_iriType, "BEGIN")
        self.assertTrue(description.find("bb-bb-bb-bb-bb-bb")) # Check if MAC address from second list was added to message description
        self.assertTrue(('IP', '1.1.1.1') in nidArray)
        self.assertTrue(('MAC', 'aa-aa-aa-aa-aa-aa') in nidArray)
        
        sm.calls = []
        sm.params = []
        
        
    def testSpecialLists_ini2format_join13(self):
        iricore.iricore_init()
        sm = FakeClass()
        s = FakeClass()
        
        mac = CreateNIDType("aa-aa-aa-aa-aa-aa", "MAC")
        ip_a = CreateNIDType("1.1.1.1", "IPv4")
        ip_b = CreateNIDType("2.2.2.2", "IPv4")
        
        msgNewIcept = ('new_intercept', 'iri-iif', INI1AIntercept(CID('BUT', 'aa-aa-aa-aa-aa-aa', '0', 'CZ'), 'aa-aa-aa-aa-aa-aa', 'xxx', True, 1374098400, 1))
        iricore.processMessageINI1A(msgNewIcept, sm, s)
        
        msg = ('obs1', ('test_module', 1377718543.66, 'BEGIN', 'xxx', [('IP', '1.1.1.1'),('MAC', 'aa-aa-aa-aa-aa-aa')], [], [('IP', '2.2.2.2')]))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        outer_msg = sm.params[2][0][1][2]
        self.assertEqual(len(outer_msg), 6)
        
        outer_iriType, interception, cid, msg_nidcc, inner_msg, icept_nids = outer_msg
        
        # Parse outer message (Whole message passed to MF)
        self.assertEqual(outer_iriType, "BEGIN")
        self.assertEqual(interception, "xxx")
        self.assertEqual(len(inner_msg), 5)        
        
        # Parse inner message (Modified former message "from IRI module")
        moduleID, timestamp, inner_iriType, description, nidArray = inner_msg
        self.assertEqual(moduleID, "test_module")
        self.assertEqual(timestamp, 1377718543.66)
        self.assertEqual(inner_iriType, "BEGIN")
        
        # Check is IRI joined 1st and 3rd list:
        self.assertTrue(('IP', '1.1.1.1') in nidArray)
        self.assertTrue(('IP', '2.2.2.2') in nidArray)
        self.assertTrue(('MAC', 'aa-aa-aa-aa-aa-aa') in nidArray)

        sm.calls = []
        sm.params = []
        
        
    def testSpecialLists_tcp3_clockskew(self):
        iricore.iricore_init()
        sm = FakeClass()
        s = FakeClass()
        
        ip_a = CreateNIDType("147.229.13.162", "IPv4")
        ip_b = CreateNIDType("147.229.14.185", "IPv4")
        
        msg = ('obs1', ('test_module', 1377718543.66, 'BEGIN', 'xxx', [('TCP3', '147.229.13.162_43939'), ('TCP3', '147.229.14.185_80')], [('IP', '147.229.13.162'),('IP', '147.229.14.185')], []))
        iricore.processMessageIRIINTERNAL(msg, sm, s)
        
        self.assertFalse(iricore.nidGraph.nodeExists(ip_a))
        self.assertFalse(iricore.nidGraph.nodeExists(ip_b))

        sm.calls = []
        sm.params = []
        
    def testConnSearch(self):
        graph = NIDGraph()
        a = CreateNIDType("1.1.1.1", "IPv4")
        b = CreateNIDType("2.2.2.2", "IPv4")
        c = CreateNIDType("3.3.3.3", "IPv4")
        d = CreateNIDType("4.4.4.4", "IPv4")
        e = CreateNIDType("5.5.5.5", "IPv4")
        
        graph.addNode(a)
        graph.addNode(b)
        graph.addNode(c)
        graph.addNode(d)
        graph.addNode(e, "LIIDa")
        
        graph.addConnection(a,b,"u")
        graph.addConnection(b,d,"v")
        graph.addConnection(a,c,"w")
        graph.addConnection(c,d,"x")
        graph.addConnection(d,c,"y")
        graph.addConnection(d,e,"z", "LIIDa")
        
        nids = [a,b,c,d,e]
        found_cons = graph.findConnections_str(nids)
        self.assertTrue(('1.1.1.1', '2.2.2.2') in found_cons or ('2.2.2.2', '1.1.1.1') in found_cons)
        self.assertTrue(('1.1.1.1', '3.3.3.3') in found_cons or ('3.3.3.3', '1.1.1.1') in found_cons)
        self.assertTrue(('4.4.4.4', '2.2.2.2') in found_cons or ('2.2.2.2', '4.4.4.4') in found_cons)
        self.assertTrue(('3.3.3.3', '4.4.4.4') in found_cons or ('4.4.4.4', '3.3.3.3') in found_cons)
        self.assertTrue(('4.4.4.4', '5.5.5.5') in found_cons or ('5.5.5.5', '4.4.4.4') in found_cons)

    

def suite():
    test = unittest.makeSuite(Tests, "test")
    return unittest.TestSuite(tuple(test))

def test():
    runner = unittest.TextTestRunner()
    runner.run(suite())
