#ifndef CODEBLOCK_H
#define CODEBLOCK_H

#include "imageptr.h"
#include "vec2.h"
struct transform_t;

/// fill additional code-block parameters
#define CONFIG_CODEBLOCK_PARAMETERS

/**
 * @brief The code-block.
 */
struct codeblock_t {
	/** Separated LL subband. */
	struct imageptr_t *llband;
	/** Separated HL subband. */
	struct imageptr_t *hlband;
	/** Separated LH subband. */
	struct imageptr_t *lhband;
	/** Separated HH subband. */
	struct imageptr_t *hhband;

	/** The size of each code-block (LL, HL, LH, HH). */
	struct vec2_t cb_exp;

	/** Decomposition level. */
	int n;

	/** The tile-component's coordinates at a particular resolution level. */
	struct vec2_t tr;

	/** Beginning (inclusive) of valid data. */
	struct vec2_t local_c0;
	/** End (exclusive) of valid data. */
	struct vec2_t local_c1;
	/** Size of valid data. */
	struct vec2_t local_size;

#ifdef CONFIG_CODEBLOCK_PARAMETERS
	/** r = N - n (add +1 to map into OpenJPEG's resolution) */
	int r;

	/** horizontal and vertical sequence number (p, q) */
	struct vec2_t index;
	struct vec2_t llband_index;
	struct vec2_t hlband_index;
	struct vec2_t lhband_index;
	struct vec2_t hhband_index;

	/** (p, q) mapped into linear sequence (OpenJPEG's cblk_no) */
	int seq;
	int llband_seq;
	int hlband_seq;
	int lhband_seq;
	int hhband_seq;
#endif
};

typedef void (*codeblock_callback_t)(void *params, const struct codeblock_t *codeblock);

/**
 * @brief Allocate the code-block structure.
 */
struct codeblock_t *codeblock_create(
	const struct transform_t *transform
);

/**
 * @brief Allocate the pool of code-blocks.
 */
struct codeblock_t *codeblock_create_pool(
	const struct transform_t *transform
);

/**
 * @brief Free the code-block structure.
 */
void codeblock_destroy(
	const struct transform_t *transform,
	struct codeblock_t *codeblock
);

/**
 * @brief Free the pool of code-blocks.
 */
void codeblock_destroy_pool(
	const struct transform_t *transform,
	struct codeblock_t *codeblock
);

/**
 * @brief Acquire an empty code-block.
 *
 * @param[in] tr the coordinates of the code-block
 */
struct codeblock_t *codeblock_acquire(
	const struct transform_t *transform,
	const struct vec2_t tr
);

/**
 * @brief Release the code-block structure acquired by @ref codeblock_acquire.
 */
void codeblock_release(
	const struct transform_t *transform,
	struct codeblock_t *codeblock
);

/**
 * @brief Process (i.e. encode) the code-block.
 */
void codeblock_process(
	const struct transform_t *transform,
	const struct codeblock_t *codeblock
);

/**
 * @brief Process (i.e. encode) the pool of code-blocks.
 *
 * @param transform transform from which the code-block comes
 * @param codeblock most recently generated code-block (from the pool)
 */
void codeblock_process_pool(
	const struct transform_t *transform,
	const struct codeblock_t *codeblock
);

#endif
