#ifndef VEC_H
#define VEC_H

#include "system.h"
#include "intmath.h"

/**
 * @brief A pair of integers.
 *
 * The pair represents coordinates and sizes.
 */
struct vec2_t {
	int x, y;
};

// NOTE: compound literals have highest priority, so no brackets are needed
/** Get zero vector. */
#define vec2_zero (struct vec2_t) { 0, 0 }

/** Create the vector from given cordinates. */
#define vec2_create(x,y) (struct vec2_t) { (x), (y) }

/**
 * @brief Element-wise minimum of two integer pairs.
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_min(
	const struct vec2_t a,
	const struct vec2_t b
)
{
	return vec2_create(
		min(a.x, b.x),
		min(a.y, b.y)
	);
}

/**
 * @brief Element-wise maximum.
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_max(
	const struct vec2_t a,
	const struct vec2_t b
)
{
	return vec2_create(
		max(a.x, b.x),
		max(a.y, b.y)
	);
}

/**
 * @brief Sum of two pairs.
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_add(
	const struct vec2_t l,
	const struct vec2_t r
)
{
	return vec2_create(
		l.x + r.x,
		l.y + r.y
	);
}

/**
 * @brief Point-wise multiply two vectors.
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_mul(
	const struct vec2_t l,
	const struct vec2_t r
)
{
	return vec2_create(
		l.x * r.x,
		l.y * r.y
	);
}

/**
 * @brief Point-wise multiply by two.
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_mul2(
	const struct vec2_t v
)
{
	return vec2_create(
		2 * v.x,
		2 * v.y
	);
}

/**
 * @brief Difference (subtraction) of two pairs.
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_sub(
	const struct vec2_t l,
	const struct vec2_t r
)
{
	return vec2_create(
		l.x - r.x,
		l.y - r.y
	);
}

/**
 * @brief Negate the vector.
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_neg(
	const struct vec2_t l
)
{
	return vec2_create(
		-l.x,
		-l.y
	);
}

/**
 * @brief Element-wise smallest integral value not less than \f$ x / 2^n \f$.
 *
 * @param[in] v the input vector
 * @param[in] n the exponent of the power of two
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_ceil_div_pow2(
	const struct vec2_t v,
	int n
)
{
	return vec2_create(
		ceil_div_pow2(v.x, n),
		ceil_div_pow2(v.y, n)
	);
}

UNUSED_FUNCTION
static
struct vec2_t vec2_ceil_div_pow2v(
	const struct vec2_t v,
	const struct vec2_t n
)
{
	return vec2_create(
		ceil_div_pow2(v.x, n.x),
		ceil_div_pow2(v.y, n.y)
	);
}

/**
 * @brief Element-wise smallest integral value not less than \f$ x / 2 \f$.
 *
 * @param[in] v the input vector
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_ceil_div_pow2_1(
	const struct vec2_t v
)
{
	return vec2_create(
		ceil_div_pow2_1(v.x),
		ceil_div_pow2_1(v.y)
	);
}

/**
 * @brief Element-wise largest integral value not greater than \f$ x / 2^n \f$.
 *
 * @param[in] v the input vector
 * @param[in] n the exponent of the power of two
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_floor_div_pow2(
	const struct vec2_t v,
	int n
)
{
	return vec2_create(
		floor_div_pow2(v.x, n),
		floor_div_pow2(v.y, n)
	);
}

/**
 * @brief Element-wise largest integral value not greater than \f$ x / 2 \f$.
 *
 * @param[in] v the input vector
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_floor_div_pow2_1(
	const struct vec2_t v
)
{
	return vec2_create(
		floor_div_pow2_1(v.x),
		floor_div_pow2_1(v.y)
	);
}

/**
 * @brief Vector a vector (x+0, y+0).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_0_0(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+0, v.y+0 };
}

/**
 * @brief Vector a vector (x+1, y+0).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_1_0(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+1, v.y+0 };
}

/**
 * @brief Vector a vector (x+2, y+0).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_2_0(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+2, v.y+0 };
}

/**
 * @brief Vector a vector (x+3, y+0).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_3_0(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+3, v.y+0 };
}

/**
 * @brief Vector a vector (x+4, y+0).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_4_0(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+4, v.y+0 };
}

/**
 * @brief Vector a vector (x+5, y+0).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_5_0(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+5, v.y+0 };
}

/**
 * @brief Vector a vector (x+6, y+0).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_6_0(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+6, v.y+0 };
}

/**
 * @brief Vector a vector (x+7, y+0).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_7_0(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+7, v.y+0 };
}

/**
 * @brief Vector a vector (x+0, y+1).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_0_1(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+0, v.y+1 };
}

/**
 * @brief Vector a vector (x+1, y+1).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_1_1(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+1, v.y+1 };
}

/**
 * @brief Vector a vector (x+2, y+1).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_2_1(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+2, v.y+1 };
}

/**
 * @brief Vector a vector (x+3, y+1).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_3_1(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+3, v.y+1 };
}

/**
 * @brief Vector a vector (x+4, y+1).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_4_1(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+4, v.y+1 };
}

/**
 * @brief Vector a vector (x+5, y+1).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_5_1(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+5, v.y+1 };
}

/**
 * @brief Vector a vector (x+6, y+1).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_6_1(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+6, v.y+1 };
}

/**
 * @brief Vector a vector (x+7, y+1).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_7_1(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+7, v.y+1 };
}

/**
 * @brief Vector a vector (x+0, y+2).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_0_2(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+0, v.y+2 };
}

/**
 * @brief Vector a vector (x+1, y+2).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_1_2(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+1, v.y+2 };
}

/**
 * @brief Vector a vector (x+2, y+2).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_2_2(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+2, v.y+2 };
}

/**
 * @brief Vector a vector (x+3, y+2).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_3_2(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+3, v.y+2 };
}

/**
 * @brief Vector a vector (x+4, y+2).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_4_2(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+4, v.y+2 };
}

/**
 * @brief Vector a vector (x+5, y+2).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_5_2(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+5, v.y+2 };
}

/**
 * @brief Vector a vector (x+6, y+2).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_6_2(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+6, v.y+2 };
}

/**
 * @brief Vector a vector (x+7, y+2).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_7_2(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+7, v.y+2 };
}

/**
 * @brief Vector a vector (x+0, y+3).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_0_3(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+0, v.y+3 };
}

/**
 * @brief Vector a vector (x+1, y+3).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_1_3(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+1, v.y+3 };
}

/**
 * @brief Vector a vector (x+2, y+3).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_2_3(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+2, v.y+3 };
}

/**
 * @brief Vector a vector (x+3, y+3).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_3_3(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+3, v.y+3 };
}

/**
 * @brief Vector a vector (x+4, y+3).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_4_3(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+4, v.y+3 };
}

/**
 * @brief Vector a vector (x+5, y+3).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_5_3(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+5, v.y+3 };
}

/**
 * @brief Vector a vector (x+6, y+3).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_6_3(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+6, v.y+3 };
}

/**
 * @brief Vector a vector (x+7, y+3).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_7_3(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+7, v.y+3 };
}

/**
 * @brief Vector a vector (x+0, y+4).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_0_4(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+0, v.y+4 };
}

/**
 * @brief Vector a vector (x+1, y+4).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_1_4(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+1, v.y+4 };
}

/**
 * @brief Vector a vector (x+2, y+4).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_2_4(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+2, v.y+4 };
}

/**
 * @brief Vector a vector (x+3, y+4).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_3_4(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+3, v.y+4 };
}

/**
 * @brief Vector a vector (x+4, y+4).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_4_4(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+4, v.y+4 };
}

/**
 * @brief Vector a vector (x+5, y+4).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_5_4(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+5, v.y+4 };
}

/**
 * @brief Vector a vector (x+6, y+4).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_6_4(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+6, v.y+4 };
}

/**
 * @brief Vector a vector (x+7, y+4).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_7_4(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+7, v.y+4 };
}

/**
 * @brief Vector a vector (x+0, y+5).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_0_5(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+0, v.y+5 };
}

/**
 * @brief Vector a vector (x+1, y+5).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_1_5(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+1, v.y+5 };
}

/**
 * @brief Vector a vector (x+2, y+5).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_2_5(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+2, v.y+5 };
}

/**
 * @brief Vector a vector (x+3, y+5).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_3_5(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+3, v.y+5 };
}

/**
 * @brief Vector a vector (x+4, y+5).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_4_5(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+4, v.y+5 };
}

/**
 * @brief Vector a vector (x+5, y+5).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_5_5(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+5, v.y+5 };
}

/**
 * @brief Vector a vector (x+6, y+5).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_6_5(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+6, v.y+5 };
}

/**
 * @brief Vector a vector (x+7, y+5).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_7_5(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+7, v.y+5 };
}

/**
 * @brief Vector a vector (x+0, y+6).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_0_6(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+0, v.y+6 };
}

/**
 * @brief Vector a vector (x+1, y+6).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_1_6(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+1, v.y+6 };
}

/**
 * @brief Vector a vector (x+2, y+6).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_2_6(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+2, v.y+6 };
}

/**
 * @brief Vector a vector (x+3, y+6).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_3_6(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+3, v.y+6 };
}

/**
 * @brief Vector a vector (x+4, y+6).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_4_6(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+4, v.y+6 };
}

/**
 * @brief Vector a vector (x+5, y+6).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_5_6(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+5, v.y+6 };
}

/**
 * @brief Vector a vector (x+6, y+6).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_6_6(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+6, v.y+6 };
}

/**
 * @brief Vector a vector (x+7, y+6).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_7_6(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+7, v.y+6 };
}

/**
 * @brief Vector a vector (x+0, y+7).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_0_7(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+0, v.y+7 };
}

/**
 * @brief Vector a vector (x+1, y+7).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_1_7(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+1, v.y+7 };
}

/**
 * @brief Vector a vector (x+2, y+7).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_2_7(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+2, v.y+7 };
}

/**
 * @brief Vector a vector (x+3, y+7).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_3_7(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+3, v.y+7 };
}

/**
 * @brief Vector a vector (x+4, y+7).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_4_7(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+4, v.y+7 };
}

/**
 * @brief Vector a vector (x+5, y+7).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_5_7(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+5, v.y+7 };
}

/**
 * @brief Vector a vector (x+6, y+7).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_6_7(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+6, v.y+7 };
}

/**
 * @brief Vector a vector (x+7, y+7).
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_offset_7_7(
	const struct vec2_t v
)
{
	return (struct vec2_t){ v.x+7, v.y+7 };
}

/**
 * @brief Is the coordinate @c c on the code-block position?
 *
 * @param c the coordinates to check
 * @param cb_exp code-block width and height exponent offset values
 */
UNUSED_FUNCTION
static
int vec2_is_codeblock(
	const struct vec2_t c,
	const struct vec2_t cb_exp
)
{
	return is_codeblock(c.x, cb_exp.x) && is_codeblock(c.y, cb_exp.y);
}

/**
 * @brief Test validity of 8&times;8 core coordinates.
 */
UNUSED_FUNCTION
static
int vec2_is_core8x8(
	const struct vec2_t c
)
{
	return is_core8(c.x) && is_core8(c.y);
}

/**
 * @brief Create a vector from the scalar @c s.
 *
 * @param s the element for use in the vector (s, s)
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_scalar(
	int s
)
{
	return vec2_create( s, s );
}

/**
 * @brief Element-wise floor the coordinates @c c to the code-block boundary.
 *
 * @param c target coordinates in the original domain
 * @param cb_exp base 2 logarithm of the code-block sizes in the reduced/transform domain
 *
 * @returns target coordinates in the original domain
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_floor_codeblock(
	const struct vec2_t c,
	const struct vec2_t cb_exp
)
{
	return vec2_create(
		floor_codeblock(c.x, cb_exp.x),
		floor_codeblock(c.y, cb_exp.y)
	);
}

/**
 * @brief Element-wise ceil the coordinates @c c to the code-block boundary.
 *
 * @param c target coordinates in the original domain
 * @param cb_exp base 2 logarithm of the code-block sizes in the reduced/transform domain
 *
 * @returns target coordinates in the original domain
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_ceil_codeblock(
	const struct vec2_t c,
	const struct vec2_t cb_exp
)
{
	return vec2_create(
		ceil_codeblock(c.x, cb_exp.x),
		ceil_codeblock(c.y, cb_exp.y)
	);
}

/**
 * @brief Element-wise ceil the coordinates @c c to the code-block boundary (in the reduced domain).
 *
 * @param c target coordinates in the reduced domain
 * @param cb_exp base 2 logarithm of the code-block sizes in the reduced/transform domain
 *
 * @returns target coordinates in the reduced domain
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_ceil_codeblock1(
	const struct vec2_t c,
	const struct vec2_t cb_exp
)
{
	return vec2_create(
		ceil_codeblock1(c.x, cb_exp.x),
		ceil_codeblock1(c.y, cb_exp.y)
	);
}

/**
 * @brief Are the vectors equal?
 *
 * @return non-zero if true, zero if false
 */
UNUSED_FUNCTION
static
int vec2_equals(
	const struct vec2_t l,
	const struct vec2_t r
)
{
	return l.x == r.x && l.y == r.y;
}

/**
 * @brief Element-wise closest even integers not larger than @c c.
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_floor2(
	const struct vec2_t c
)
{
	return vec2_create(
		floor2(c.x),
		floor2(c.y)
	);
}

UNUSED_FUNCTION
static
struct vec2_t vec2_floor8(
	const struct vec2_t c
)
{
	return vec2_create(
		floor8(c.x),
		floor8(c.y)
	);
}

/**
 * @brief  Element-wise closest even integers not less than @c c.
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_ceil2(
	const struct vec2_t c
)
{
	return vec2_create(
		ceil2(c.x),
		ceil2(c.y)
	);
}

UNUSED_FUNCTION
static
struct vec2_t vec2_ceil8(
	const struct vec2_t c
)
{
	return vec2_create(
		ceil8(c.x),
		ceil8(c.y)
	);
}

/**
 * @brief Symmetric periodic extension of coordinates. (local image coordinates)
 *
 * @param[in] c the coordinates to extend
 * @param[in] size image size
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_coord_extr(
	const struct vec2_t c,
	const struct vec2_t size
)
{
	return vec2_create(
		coord_extr(c.x, size.x),
		coord_extr(c.y, size.y)
	);
}

UNUSED_FUNCTION
static
struct vec2_t vec2_coord_extr_limited(
	const struct vec2_t c,
	const struct vec2_t size
)
{
	return vec2_create(
		coord_extr_limited(c.x, size.x),
		coord_extr_limited(c.y, size.y)
	);
}

/**
 * @brief Does the coordinate fall into the valid image area? (relative to the 0)
 *
 * @param[in] c the coordinates to test
 * @param[in] size image size
 */
UNUSED_FUNCTION
static
int vec2_coord_valid(
	const struct vec2_t c,
	const struct vec2_t size
)
{
	return coord_valid(c.x, size.x) && coord_valid(c.y, size.y);
}

/**
 * @brief Convert the tile-component coordinates into the coordinates at a particular resolution level.
 *
 * @param[in] tc coordinates at decomposition level 0
 * @param[in] n desired decomposition level
 */
UNUSED_FUNCTION
static
struct vec2_t convert_tc_to_tr(
	const struct vec2_t tc,
	int n
)
{
	return vec2_ceil_div_pow2(tc, n);
}

/**
 * @brief Convert the tile-component coordinates into the coordinates at the next resolution level.
 *
 * @param[in] tc coordinates at the current decomposition level
 */
UNUSED_FUNCTION
static
struct vec2_t convert_tc_to_next_tr(
	const struct vec2_t tc
)
{
	return vec2_ceil_div_pow2_1(tc);
}

/**
 * @brief Convert the tile-component coordinates into the coordinates at the next LL sub-band
 *
 * @param[in] tc coordinates at the current decomposition level
 *
 * @note (B-15) of ITU-T Rec. T.800 (08/2002 E)
 */
UNUSED_FUNCTION
static
struct vec2_t convert_tc_to_next_tb_ll(
	const struct vec2_t tc
)
{
	return vec2_create(
		( tc.x - 0 + 1 ) >> 1,
		( tc.y - 0 + 1 ) >> 1
	);
}

/**
 * @brief Convert the tile-component coordinates into the coordinates at the next HL sub-band
 *
 * @param[in] tc coordinates at the current decomposition level
 */
UNUSED_FUNCTION
static
struct vec2_t convert_tc_to_next_tb_hl(
	const struct vec2_t tc
)
{
	return vec2_create(
		( tc.x - 1 + 1 ) >> 1,
		( tc.y - 0 + 1 ) >> 1
	);
}

/**
 * @brief Convert the tile-component coordinates into the coordinates at the next LH sub-band
 *
 * @param[in] tc coordinates at the current decomposition level
 */
UNUSED_FUNCTION
static
struct vec2_t convert_tc_to_next_tb_lh(
	const struct vec2_t tc
)
{
	return vec2_create(
		( tc.x - 0 + 1 ) >> 1,
		( tc.y - 1 + 1 ) >> 1
	);
}

/**
 * @brief Convert the tile-component coordinates into the coordinates at the next HH sub-band
 *
 * @param[in] tc coordinates at the current decomposition level
 */
UNUSED_FUNCTION
static
struct vec2_t convert_tc_to_next_tb_hh(
	const struct vec2_t tc
)
{
	return vec2_create(
		( tc.x - 1 + 1 ) >> 1,
		( tc.y - 1 + 1 ) >> 1
	);
}

/**
 * @brief Convert the tile-component coordinates into the coordinates at a particular resolution level.
 *
 * @param[in] tc coordinates at decomposition level 0
 * @param[in] n desired decomposition level
 *
 * @warning The @p tc have to be multiple of @f$ 2^n @f$.
 */
UNUSED_FUNCTION
static
struct vec2_t convert_tc_to_tr_fast(
	const struct vec2_t tc,
	int n
)
{
	assert( 0 == (tc.x % (1<<n)) && 0 == (tc.y % (1<<n)) );

	return vec2_floor_div_pow2(tc, n);
}

/**
 * @brief Convert the tile-component coordinates into the coordinates at next resolution level.
 *
 * @param[in] tc the coordinates of current resolution level
 *
 * @warning The @p tc have to be multiple of 2.
 */
UNUSED_FUNCTION
static
struct vec2_t convert_tc_to_next_tr_fast(
	const struct vec2_t tc
)
{
	assert( 0 == (tc.x & 1) && 0 == (tc.y & 1) );

	return vec2_floor_div_pow2_1(tc);
}

/**
 * @brief Right shift.
 */
UNUSED_FUNCTION
static
struct vec2_t vec2_rshift(
	const struct vec2_t v,
	const struct vec2_t e
)
{
	return vec2_create(
		(v.x >> e.x),
		(v.y >> e.y)
	);
}

/**
 * @brief Compare two vectors.
 */
UNUSED_FUNCTION
static
int vec2_cmp(
	const struct vec2_t l,
	const struct vec2_t r
)
{
	return ( (l.x == r.x) && (l.y == r.y) );
}

UNUSED_FUNCTION
static
int vec2_size_valid(
	const struct vec2_t size
)
{
	return size.x > 0 && size.y > 0;
}

#endif
