#ifndef _FRAMEDESCRIPTOREXTRACTOR_H
#define _FRAMEDESCRIPTOREXTRACTOR_H


#include <map>
#include <opencv2/opencv.hpp>
#include <opencv2/nonfree/features2d.hpp>
#include <opencv2/legacy/legacy.hpp>
#include "FrameDescriptor.h"


namespace vmatch {


class FrameDescriptorExtractor {
protected:
	void rgb2IOO(const cv::Mat & src, cv::Mat & dst);
	void rgb2rg(const cv::Mat & src, cv::Mat & dst);
	void rgb2gradients(const cv::Mat & src, cv::Mat & dst);
	
	void tileHistCompute3(const cv::Mat & img, cv::Mat & tileHist, CvRect tile, cv::Mat & kernel, int histBinsCH1, int histBinsCH2, int histBinsCH3);
	void tileHistCompute3I2(const cv::Mat & img1, const cv::Mat & img2, cv::Mat & tileHist, CvRect tile, cv::Mat & kernel, int histBinsCH1, int histBinsCH2, int histBinsCH3);
	void tileHistCompute1(const cv::Mat & img, cv::Mat & tileHist, CvRect tile, cv::Mat & kernel, int histBinsCH1);
	void get2DGaussian(cv::Mat & normKernel, int width, int height);

public:
	FrameDescriptorExtractor();
	virtual FrameDescriptorPtr extract(const cv::Mat & frame) = 0;
	virtual void defaults();
	virtual int getType() = 0;
	virtual FrameDescriptorPtr createEmptyDescriptor() = 0;
};


typedef cv::Ptr<FrameDescriptorExtractor> FrameDescriptorExtractorPtr;


class FrameDescriptorExtractorFactory {
private:
	typedef std::map<int, FrameDescriptorExtractorPtr> LookupTable;

	static LookupTable lookup;
	static bool initialized;

public:
	static void add(int type, FrameDescriptorExtractorPtr extractor);
	static void remove(int type, FrameDescriptorExtractorPtr extractor);
	static void addDefaults();
	static bool isAssigned(int type);
	static FrameDescriptorExtractorPtr create(int type);
};


class ColHistFrameDescriptorExtractor : public FrameDescriptorExtractor {
protected:
	enum Mode {
		MODE_IOO = 1,
		MODE_RGB = 2,
		MODE_RG = 3
	};

	Mode mode;

	int xDivision; // 4;
	int yDivision; // 4

	float tileOverlap; // 0.5

	int histBinsCH1; // 16
	int histBinsCH2; // 8
	int histBinsCH3; // 8

	void computeDescriptor(const cv::Mat & frame, cv::Mat & descriptor);
	long getDescriptorSize();

public:
	class ColHistFrameDescriptor;

	ColHistFrameDescriptorExtractor();
	virtual FrameDescriptorPtr extract(const cv::Mat & frame);
	virtual int getType();
	virtual void defaults();
	virtual FrameDescriptorPtr createEmptyDescriptor();
};


class ColHistFrameDescriptorExtractor::ColHistFrameDescriptor : public VectorFrameDescriptor {
public:
	ColHistFrameDescriptor(cv::Mat data);
};


class GradHistFrameDescriptorExtractor : public FrameDescriptorExtractor {
protected:
	static const int DEFAULT_X_DIVISON;
	static const int DEFAULT_Y_DIVISON;
	static const float DEFAULT_TILE_OVERLAP;
	static const int DEFAULT_HIST_BINS;

	int xDivision;
	int yDivision;
	float tileOverlap;
	int histBinsCH1;

	void computeDescriptor(const cv::Mat & frame, cv::Mat & descriptor);
	long getDescriptorSize();

public:
	class GradHistFrameDescriptor;

	GradHistFrameDescriptorExtractor(int xDivision = DEFAULT_X_DIVISON, int yDivision = DEFAULT_Y_DIVISON, float tileOverlap = DEFAULT_TILE_OVERLAP, int bins = DEFAULT_HIST_BINS);
	virtual FrameDescriptorPtr extract(const cv::Mat & frame);
	virtual int getType();
	virtual void defaults();
	virtual FrameDescriptorPtr createEmptyDescriptor();
	void setTileOverlap(float overlap);
	void setXDivision(int xDivision);
	void setYDivision(int yDivision);
	void setHistogramBinCount(int bins);
};


class GradHistFrameDescriptorExtractor::GradHistFrameDescriptor : public VectorFrameDescriptor {
public:
	GradHistFrameDescriptor(cv::Mat data);
};


class GradRGHistFrameDescriptorExtractor : public FrameDescriptorExtractor {
protected:
	int xDivision; // 4;
	int yDivision; // 4

	float tileOverlap; // 0.5

	int histBinsCH1; // 16
	int histBinsCH2; // 8
	int histBinsCH3; // 8

	void computeDescriptor(const cv::Mat & frameRG, const cv::Mat & frameGradients, cv::Mat & descriptor);
	long getDescriptorSize();

public:
	class GradRGHistFrameDescriptor;

	GradRGHistFrameDescriptorExtractor();
	virtual FrameDescriptorPtr extract(const cv::Mat & frame);
	virtual int getType();
	virtual void defaults();
	virtual FrameDescriptorPtr createEmptyDescriptor();
};


class GradRGHistFrameDescriptorExtractor::GradRGHistFrameDescriptor : public VectorFrameDescriptor {
public:
	GradRGHistFrameDescriptor(cv::Mat data);
};


class QuickHistogramFrameDescriptorExtractor : public FrameDescriptorExtractor {
protected:
	static const int DEFAULT_X_DIVISON;
	static const int DEFAULT_Y_DIVISON;
	static const float DEFAULT_TILE_OVERLAP;
	static const int DEFAULT_HIST_BINS;
	static const int DEFAULT_DOWNSAMPLING;

	int xDivision;
	int yDivision;
	float tileOverlap;
	int histBins;
	int downsamplingFactor;

	void computeHistogram(const cv::Mat & img, const cv::Rect & tile, cv::Mat & hist, int bins);
	void computeDescriptor(const cv::Mat & img, cv::Mat & desc);
	int getDescriptorSize();

public:
	class QuickHistogramFrameDescriptor;

	QuickHistogramFrameDescriptorExtractor(int xDivision = DEFAULT_X_DIVISON, int yDivision = DEFAULT_Y_DIVISON, float tileOverlap = DEFAULT_TILE_OVERLAP, int bins = DEFAULT_HIST_BINS, int downsamplingFactor = DEFAULT_DOWNSAMPLING);
	virtual FrameDescriptorPtr extract(const cv::Mat & frame);
	virtual int getType();
	virtual void defaults();
	virtual FrameDescriptorPtr createEmptyDescriptor();
	void setTileOverlap(float overlap);
	void setXDivision(int xDivision);
	void setYDivision(int yDivision);
	void setHistogramBinCount(int bins);
	void setDownsampling(int downsamplingFactor);
};


class QuickHistogramFrameDescriptorExtractor::QuickHistogramFrameDescriptor : public FrameDescriptor {
protected:
	cv::Mat data;

public:
	QuickHistogramFrameDescriptor(const cv::Mat & data);
	virtual double compare(const FrameDescriptor & second) const;
	virtual void read(cv::FileNode & nd);
	virtual void write(cv::FileStorage & fs) const;
};


class NullFrameDescriptorExtractor : public FrameDescriptorExtractor {
public:
	class NullFrameDescriptor;

	NullFrameDescriptorExtractor();
	virtual FrameDescriptorPtr extract(const cv::Mat & frame);
	virtual int getType();
	virtual FrameDescriptorPtr createEmptyDescriptor();
};


class NullFrameDescriptorExtractor::NullFrameDescriptor : public FrameDescriptor {
public:
	NullFrameDescriptor();
	virtual double compare(const FrameDescriptor & second) const;
	virtual void read(cv::FileNode & nd);
	virtual void write(cv::FileStorage & fs) const;
};


class SurfFrameDescriptorExtractor : public FrameDescriptorExtractor {
protected:

public:
	class SurfFrameDescriptor;

	SurfFrameDescriptorExtractor();
	virtual FrameDescriptorPtr extract(const cv::Mat & frame);
	virtual int getType();
	virtual FrameDescriptorPtr createEmptyDescriptor();
};


class SurfFrameDescriptorExtractor::SurfFrameDescriptor : public FrameDescriptor {
protected:
	cv::Mat descriptors;

public:
	SurfFrameDescriptor();
	SurfFrameDescriptor(const cv::Mat & descriptors);
	virtual double compare(const FrameDescriptor & second) const;
	virtual void read(cv::FileNode & nd);
	virtual void write(cv::FileStorage & fs) const;
};

}


#endif
