#include "VideoSequence.h"

extern const char *p_s_R_file, *p_s_Q_file;

namespace vmatch {


const int VideoSequence::MetaData::UNKNOWN = 0;
const std::string VideoSequence::MetaData::NODE_FPS = "fps";
const std::string VideoSequence::MetaData::NODE_LENGTH = "length";


VideoSequence::VideoSequence(FrameDescriptorExtractorPtr extractor) {
	setDescriptorExtractor(extractor);
	setFilter(new FrameFilter);
	this->frameNumber = -1;
	this->metaData = new MetaData();
}


bool VideoSequence::next() {
    if(step()) {
		frameNumber++;
		return true;
	}

	metaData->length = frameNumber+1;
	return false;
}


void VideoSequence::rewind() {
	stepToStart();
	frameNumber = -1;
}


cv::Mat VideoSequence::getFrame() {
	cv::Mat fr = frame();
	filter->apply(fr);
	return fr;
}


VideoSequence::MetaData VideoSequence::getMetaData() {
	return *metaData;
}


int VideoSequence::getFrameNumber() {
	return frameNumber;
}


FrameDescriptorPtr VideoSequence::getFrameDescriptor() {
	cv::Mat frame = getFrame();
	return extractor->extract(frame);
}


FrameDescriptorExtractorPtr VideoSequence::getDescriptorExtractor() const {
	return extractor;
}


void VideoSequence::setDescriptorExtractor(FrameDescriptorExtractorPtr extractor) {
	if(extractor == NULL) {
		CV_Error(CV_StsBadArg, "Null pointer to FrameDescriptorExtractor given.");
	}
	this->extractor = extractor;
}


VideoSequence::FrameFilterPtr VideoSequence::getFilter() const {
	return filter;
}


void VideoSequence::setFilter(VideoSequence::FrameFilterPtr filter) {
	if(filter == NULL) {
		CV_Error(CV_StsBadArg, "Null pointer to FrameFilterPtr given.");
	}
	this->filter = filter;
}


void VideoSequence::FrameFilter::apply(cv::Mat & frame) {

}


VideoSequence::MetaData::MetaData() {
	fps = (double)UNKNOWN;
	length = (int)UNKNOWN;
}


void VideoSequence::MetaData::write(cv::FileStorage & fs) const {
	fs << NODE_FPS << fps;
	fs << NODE_LENGTH << length;
}


void VideoSequence::MetaData::read(cv::FileNode & nd) {
	CV_Assert(!nd.empty());

	if(!nd[NODE_FPS].empty()) {
		nd[NODE_FPS] >> fps;
	}

	if(!nd[NODE_LENGTH].empty()) {
		nd[NODE_LENGTH] >> length;
	}
}

CvVideoSequence::CvVideoSequence(const std::string & filename, FrameDescriptorExtractorPtr extractor) : VideoSequence(extractor) {
	if(filename == "") {
		CV_Error(CV_StsBadArg, "Invalid video sequence filename.");
	}

	if(filename.find('*') != std::string::npos) {
		capture1 = new CCueSheet_FFmpegVideo(filename.c_str());
		//if(filename == p_s_Q_file)
		//	capture1->NoDump(); // only dump the ref file, we will dump query synced to it // take it back, want them all for comparison how it looks unsynced
		n_capture_type = 1;

		if(!capture1->b_Opened()) {
			capture1.release(); // TODO zkontrolovat
			CV_Error(CV_StsObjectNotFound, "Cannot open cued video sequence.");
		}
		metaData->fps = capture1->f_FPS();
		metaData->length = (int)(capture1->f_Duration() / capture1->f_FrameTime());
	} else {
		capture0 = new CFFmpegVideo(filename.c_str());
		n_capture_type = 0;
		if(!capture0->b_Opened()) {
			capture0.release(); // TODO zkontrolovat
			CV_Error(CV_StsObjectNotFound, "Cannot open video sequence.");
		}
		metaData->fps = capture0->f_FPS();
		metaData->length = (int)(capture0->f_Duration() / capture0->f_FrameTime());
	}

	/*capture = new cv::VideoCapture(filename);

	if(!capture->isOpened())
	{
		capture.release(); // TODO zkontrolovat
		CV_Error(CV_StsObjectNotFound, "Cannot open video sequence.");
	}

	metaData->fps = capture->get(CV_CAP_PROP_FPS);
	metaData->length = (int)capture->get(CV_CAP_PROP_FRAME_COUNT);*/

}

bool CvVideoSequence::step() {
	TBmp t_img = (n_capture_type == 0)? capture0->t_Get_NextFrame() : capture1->t_Get_NextFrame();
	if(t_img.n_width > 0) {
		m_t_img = t_img; // otherwise keep the last frame in memory
		return true;
	}
    /*if(capture->get(CV_CAP_PROP_POS_FRAMES) < capture->get(CV_CAP_PROP_FRAME_COUNT)-1) {
        return capture->grab();
    }*/

    return false;
}

cv::Mat CvVideoSequence::frame() {
	IplImage image = {0};
	image.nSize = sizeof(IplImage);
	image.nChannels = 4;
	image.depth = IPL_DEPTH_8U;
	image.width = m_t_img.n_width;
	image.height = m_t_img.n_height;
	image.imageId = "           ";
	//strcpy(image.tileInfo, "           ");
	image.imageSize = m_t_img.n_width * m_t_img.n_height * 4;
	image.imageData = (char*)m_t_img.p_buffer;
	image.widthStep = m_t_img.n_width * 4;
	image.align = 4;

	cv::Mat sframe = cv::Mat(&image, false);
	cv::Mat frame;
	cv::cvtColor(sframe, frame, cv::COLOR_BGRA2BGR, 3); // need to move the data, klicnar is ignoring align / widthstep

	//cv::Mat frame = cv::Mat(&image, false);
	//capture->retrieve(frame);

	if(0) {
		cv::imshow("bluh", frame); // seems to work now
		cv::waitKey(0);
	}

	return frame;
}


void CvVideoSequence::stepToStart() {
	if(n_capture_type == 0)
		capture0->Seek(0);
	else
		capture1->Seek(0);
	//capture->set(CV_CAP_PROP_POS_FRAMES, 0);
}


/*cv::Ptr<cv::VideoCapture> CvVideoSequence::getCapture() {
	return capture;
}*/


#ifdef VTAPI
VtVideoSequence::VtVideoSequence(Sequence *sequence, FrameDescriptorExtractorPtr extractor) : VideoSequence(extractor) {
	if(sequence == NULL) {
		CV_Error(CV_StsBadArg, "Null pointer to VTAPI Sequence given.");
	}
	this->sequence = sequence;
}


bool VtVideoSequence::step() {
	return sequence->next();
}


cv::Mat VtVideoSequence::frame() {
	Image* image = sequence->newImage();
	return cv::imread(image->getDataLocation());
}


void VtVideoSequence::stepToStart() {
	//TODO dodelat
}
#endif


}
