#ifndef _VIDEOSEQUENCE_H
#define _VIDEOSEQUENCE_H


#include <opencv2/opencv.hpp>
#include <algorithm>
#include "FrameDescriptorExtractor.h"

#ifdef VTAPI
#include "vtapi.h"
#endif

class CFFmpegVideo;
class CCueSheet_FFmpegVideo;
#include "UberLame_src/Bitmap.h"

#include "UberLame_src/iface/FFmpeg.h"
#include "UberLame_src/iface/PNGLoad.h"
#include "UberLame_src/StlUtils.h"
#include "UberLame_src/Dir.h"

class CCueSheet_FFmpegVideo {
protected:
	CFFmpegVideo m_raw_video;
	double m_f_start, m_f_end;
	size_t m_n_frame_counter;
	std::string m_s_seq_name;
	bool m_b_dump_frames;

public:
	inline const std::string &s_Seq_Name() const
	{
		return m_s_seq_name;
	}

	CCueSheet_FFmpegVideo()
		:m_f_start(0), m_f_end(0)
	{}

	CCueSheet_FFmpegVideo(const char *p_s_filename,
		bool b_dump_format = true, size_t n_video_stream_index = 0) // throw(std::bad_alloc)
	{
		Open(p_s_filename, b_dump_format, n_video_stream_index);
	}

	void NoDump()
	{
		m_b_dump_frames = false;
	}

	bool Open(const char *p_s_filename, bool b_dump_format = true,
		size_t n_video_stream_index = 0) // throw(std::bad_alloc)
	{
		m_b_dump_frames = true;
		m_n_frame_counter = 0;
		std::string s_filename = p_s_filename;
		std::vector<std::string> field_list;
		if(!stl_ut::Split(field_list, s_filename, "*"))
			throw std::bad_alloc(); // rethrow
		if(field_list.size() != 4) {
			fprintf(stderr, "error: cue file name must be cuesheet*video*tag*seq-nr");
			return false;
		}
		const std::string s_cue_sheet = field_list[0];
		const std::string s_video = field_list[1];
		const std::string s_tag = field_list[2];
		const std::string s_seq_nr = field_list[3];
		const int n_seq_nr = atol(s_seq_nr.c_str());
		fprintf(stderr, "opening cue-sheet \'%s\', video \'%s\', tag \'%s\' nr. %d (zero-based)\n",
			s_cue_sheet.c_str(), s_video.c_str(), s_tag.c_str(), n_seq_nr);
		// split the file name to cue name and file name

		FILE *p_fr;
		if(!(p_fr = fopen(s_cue_sheet.c_str(), "r"))) {
			fprintf(stderr, "error: failed to open the cue-sheet for reading\n");
			return false;
		}
		std::string s_header, s_line;
		stl_ut::ReadLine(s_header, p_fr);
		size_t n_line_no = 1, n_seq_counter = 0;
		while(!feof(p_fr)) {
			++ n_line_no;
			if(!stl_ut::ReadLine(s_line, p_fr)) {
				fclose(p_fr);
				fprintf(stderr, "error: i/o error while reading the cue-sheet\n");
				return false;
			}
			if(s_line.empty())
				continue;
			int hs, ms, he, me;
			double ss, se;
			if(sscanf(s_line.c_str(), "%d:%d:%lf,%d:%d:%lf", &hs, &ms, &ss, &he, &me, &se) != 6) {
				fprintf(stderr, "warning: \'%s\' on line " PRIsize ": truncated\n",
					s_filename.c_str(), n_line_no);
			}
			// parse times

			s_line.erase(0, s_line.rfind(',') + 1);
			// get only the tag

			if(s_line == s_tag && (++ n_seq_counter) - 1 == n_seq_nr) {
				fprintf(stderr, "found cue tag \'%s\', %02d:%02d:%06.3f --> %02d:%02d:%06.3f\n",
					s_tag.c_str(), hs, ms, ss, he, me, se);
				ms += 60 * hs;
				ss += 60 * ms;
				me += 60 * he;
				se += 60 * me;
				m_f_start = ss;
				m_f_end = se;
				fclose(p_fr);
				p_fr = 0;
				break;
			}
			// match the tag and sequence number
		}
		if(p_fr) {
			fclose(p_fr);
			fprintf(stderr, "error: the cue-sheet does not contain the tag / seq. id\n");
			return false;
		}
		// read the begin / end time

		stl_ut::Format(m_s_seq_name, "_%s-" PRIsize "_", s_tag.c_str(), n_seq_nr);
		// remember for debugging

		if(!m_raw_video.Open(s_video.c_str(), b_dump_format, n_video_stream_index))
			return false;
		return My_Seek(m_f_start);
		// open the video and seek to start
	}

	void Close()
	{
		m_raw_video.Close();
	}

	inline bool b_Opened() const
	{
		return m_raw_video.b_Opened();
	}

	/*inline const AVFormatContext *p_FormatContext() const
	{
		return m_p_format_context;
	}

	inline const AVCodec *p_Codec() const
	{
		return m_p_codec;
	}

	inline const AVCodecContext *p_CodecContext() const
	{
		return m_p_codec_context;
	}*/

	inline int n_Width() const
	{
		return m_raw_video.n_Width();
	}

	inline int n_Height() const
	{
		return m_raw_video.n_Height();
	}

	inline double f_FrameTime() const
	{
		return m_raw_video.f_FrameTime();
	}

	inline double f_FPS() const
	{
		return m_raw_video.f_FPS();
	}

	inline double f_Duration() const
	{
		return m_f_end - m_f_start;
	}

	inline double f_NextFrame_Time() const
	{
		return m_raw_video.f_NextFrame_Time() - m_f_start;
	}

	bool Seek(double f_seek_time, bool b_relative = false)
	{
		if(b_relative)
			f_seek_time += m_f_start;
		// convert to absolute

		if(f_seek_time < m_f_start)
			f_seek_time = m_f_start;
		if(f_seek_time > m_f_end)
			f_seek_time = m_f_end;
		// clip

		return My_Seek(f_seek_time);
		// this might be a low-precision seek though, might want to simply skip/decode the frames
	}

	const TBmp &t_Get_NextFrame()
	{
		static const TBmp noimg = {0};
		if(m_raw_video.f_NextFrame_Time() > m_f_end)
			return noimg;
		const TBmp &r_t_frame = m_raw_video.t_Get_NextFrame();

		if(m_b_dump_frames) {
			char p_s_filename[256];
			stl_ut::Format(p_s_filename, sizeof(p_s_filename),
				"framedump/sequence_%s_frame_%05" _PRIsize ".png", m_s_seq_name.c_str(), m_n_frame_counter);
			TFileInfo f(p_s_filename);
			if(!f.b_exists || !f.n_Size64())
				CPngCodec::Save_PNG(p_s_filename, r_t_frame); // only save if it is not already there
			++ m_n_frame_counter;
		}
		// debug - verify that the seeking indeed works (and not in 10 second increments)

		return r_t_frame;
	}

protected:
	bool My_Seek(double f_seek_time)
	{
#if 0
		return m_raw_video.Seek(f_seek_time); // in a perfect world ...
#else
		double f_frame_time = m_raw_video.f_FrameTime();
		double f_cur_time = m_raw_video.f_NextFrame_Time();

		if(fabs(f_seek_time - f_cur_time) < f_frame_time)
			return true; // another job well done
		// we're ok

		if(f_seek_time < f_cur_time) {
			if(!m_raw_video.Seek(0, false))
				return false;
			f_cur_time = m_raw_video.f_NextFrame_Time();
		}
		// in case we are seeking backwards, jump to the beginning

		_ASSERTE(f_cur_time <= f_seek_time);
		while(f_cur_time + f_frame_time * .5 < f_seek_time) {
			printf("seeking to %.3f ... %.3f\r", f_seek_time, f_cur_time);
			if(!m_raw_video.t_Get_NextFrame().p_buffer) // duh ...
				return false; // eof before seek succeeded
			f_cur_time = m_raw_video.f_NextFrame_Time();
		}
		printf("seeking finished. now at %.3f (%02d:%02d:%06.3f)\n", f_cur_time,
			int(floor(f_cur_time / 60)) / 60, int(floor(f_cur_time / 60)) % 60, fmod(f_cur_time, 60));
		// read frames until er arrive to the required time

		return true;
#endif
	}
};

namespace vmatch {


class VideoSequence {
public:
	class FrameFilter;
	class MetaData;
	typedef cv::Ptr<FrameFilter> FrameFilterPtr;
	typedef cv::Ptr<MetaData> MetaDataPtr;

private:
	FrameDescriptorExtractorPtr extractor;
	FrameFilterPtr filter;
	int frameNumber;

protected:
	MetaDataPtr metaData;
	virtual bool step() = 0;
	virtual void stepToStart() = 0;
	virtual cv::Mat frame() = 0;

public:
	VideoSequence(FrameDescriptorExtractorPtr extractor);	
	bool next();
	void rewind();
	MetaData getMetaData();
	cv::Mat getFrame();
	int getFrameNumber();
	FrameDescriptorPtr getFrameDescriptor();
	FrameDescriptorExtractorPtr getDescriptorExtractor() const;
	void setDescriptorExtractor(FrameDescriptorExtractorPtr extractor);
	FrameFilterPtr getFilter() const;
	void setFilter(FrameFilterPtr filter);
};


typedef cv::Ptr<VideoSequence> VideoSequencePtr;


class VideoSequence::FrameFilter {
public:
	virtual void apply(cv::Mat & frame);
};


class VideoSequence::MetaData {
protected:
	static const std::string NODE_FPS;
	static const std::string NODE_LENGTH;

public:
	static const int UNKNOWN;

	double fps;
	int length;

	MetaData();
	void write(cv::FileStorage & fs) const;
	void read(cv::FileNode & nd);
};

class CvVideoSequence : public VideoSequence {
protected:
	cv::Ptr<CFFmpegVideo> capture0;
	cv::Ptr<CCueSheet_FFmpegVideo> capture1;
	int n_capture_type;
	TBmp m_t_img;
	//cv::Ptr<cv::VideoCapture> capture;

	virtual bool step();
	void stepToStart();
	virtual cv::Mat frame();

public:
	CvVideoSequence(const std::string & filename, FrameDescriptorExtractorPtr extractor);
    //cv::Ptr<cv::VideoCapture> getCapture();
};


#ifdef VTAPI
class VtVideoSequence : public VideoDescriptorSequence {
protected:
	virtual bool step();
	void stepToStart();
	virtual cv::Mat frame();

	Sequence* sequence;

public:
	VtVideoSequence(Sequence* sequence, FrameDescriptorExtractorPtr extractor);
};
#endif


}


#endif
