#!/usr/bin/env python3
# -*- coding: utf-8 -*-

""" Test package for consensus parser """
# Copyright (C) 2018 Libor Polčák
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import unittest
import filecmp
from io import StringIO
import os

from time_parser import TimeWrapper
import parametrizable_tc as ptc

import consensus_parser as cp

PCVESELY_JSON = '''[{
    "allow_ports": "reject 25,119,135-139,445,563,1214,4661-4666,6346-6429,6699,6881-6999",
    "bandwidth": {
        "Bandwidth": "20",
        "Unmeasured": "1"
    },
    "digest": "pkYCSI2r79XMzeFFVLbF+73Sr3w",
    "dirport": "9030",
    "flags": [
        "Exit",
        "Fast",
        "Running",
        "V2Dir",
        "Valid"
    ],
    "identity": "M5dvWj/g37C7nYQ2QHw9bA1QNT8",
    "inconsensus_fresh_until": [
        "2013-04-02",
        "22:00:00"
    ],
    "inconsensus_val_after": [
        "2013-04-02",
        "21:00:00"
    ],
    "inconsensus_val_until": [
        "2013-04-03",
        "00:00:00"
    ],
    "ip": "147.229.13.223",
    "nickname": "default",
    "orport": "443",
    "publication": [
        "2013-04-02",
        "20:08:40"
    ],
    "version": "Tor 0.2.3.25"
}
,{
    "allow_ports": "reject 25,119,135-139,445,563,1214,4661-4666,6346-6429,6699,6881-6999",
    "bandwidth": {
        "Bandwidth": "20",
        "Unmeasured": "1"
    },
    "digest": "+LzaWDVh0IrUPzGyrGHCXijdAE0",
    "dirport": "9030",
    "flags": [
        "Exit",
        "Fast",
        "Running",
        "V2Dir",
        "Valid"
    ],
    "identity": "M5dvWj/g37C7nYQ2QHw9bA1QNT8",
    "inconsensus_fresh_until": [
        "2013-04-02",
        "23:00:00"
    ],
    "inconsensus_val_after": [
        "2013-04-02",
        "22:00:00"
    ],
    "inconsensus_val_until": [
        "2013-04-03",
        "01:00:00"
    ],
    "ip": "147.229.13.223",
    "nickname": "default",
    "orport": "443",
    "publication": [
        "2013-04-02",
        "21:39:09"
    ],
    "version": "Tor 0.2.3.25"
}
,]'''

PREPROCESSED_DIR_CREATE = "testfiles/preprocessedtestauto"
CONSENSUS_PATH = "testfiles/descriptorstubs"

class fake_cp_args():
    def __init__(self):
        self.consensus_path = None
        self.update = False
        self.update_keeprunning = False
        self.fixup = None
        self.write_preprocessed = False
        self.additional_info = False
        self.ipaddress = None
        self.preprocessed_input = None
        self.geolite_dir = None
        self.time = None
        self.date_prefix = None

class test_consensus_parser(unittest.TestCase, metaclass=ptc.parametrizable_tc_meta):

    @ptc.parametrizable_test([
        ("no_restrictions", [None, None, PCVESELY_JSON]),
        ("time_active", [TimeWrapper("22:00:00 2013-04-02"), None, PCVESELY_JSON]),
        ("time_inactive", [TimeWrapper("2013-04-02 00:00"), None, "IP address 147.229.13.223 was not active in Tor at 2013-04-02 00:00:00\n"]),
        ("date_active", [None, "2013-04-02", PCVESELY_JSON]),
        ("date_inactive", [None, "2013-04-01", "IP address 147.229.13.223 was not active in Tor during date prefix 2013-04-01\n"]),
        ])
    def test_main_findip_address(self, t, date, expected):
        args = fake_cp_args()
        args.ipaddress = "147.229.13.223"
        args.preprocessed_input = "testfiles/preprocessed"
        args.time = t
        args.date_prefix = date
        fake_file = StringIO()
        cp.main(args, fake_file)
        fake_file.seek(0)
        self.assertEqual(fake_file.read(), expected)

    def test_get_last_preprocessed_consensus(self):
        t = 123
        os.system("rm -rf %s" % PREPROCESSED_DIR_CREATE)
        os.makedirs(PREPROCESSED_DIR_CREATE, exist_ok = True)
        with open("%s/last" % PREPROCESSED_DIR_CREATE, "w") as f:
            f.write(str(t))
            f.write("\n")
        self.assertEqual(cp.get_last_preprocessed_consensus(os.path.abspath(PREPROCESSED_DIR_CREATE)), t)

    def check_preprocessed_consensus_descriptorstubs(self):
        ors = cp.find_preprocessed_ip_address("162.247.74.201", PREPROCESSED_DIR_CREATE)
        self.assertEqual(len(ors), 1)
        theor = ors[0]
        self.assertEqual(theor.get_nickname(), "CalyxInstitute14")
        self.assertEqual(theor.get_identity(), "ABG9JIWtRdmE7EFZyI/AZuXjMA4")
        self.assertEqual(theor.get_inconsensus_val_after(), ("2018-08-04", "19:00:00"))
        self.assertEqual(theor.get_inconsensus_fresh_until(), ("2018-08-04", "20:00:00"))
        self.assertEqual(theor.get_inconsensus_val_until(), ("2018-08-04", "22:00:00"))

    def test_main_update(self):
        t = int(TimeWrapper("2018-08-04 18-00-00").get())
        oldtime = cp.time.time
        try:
            cp.time.time = lambda: t+3600+1800
            args = fake_cp_args()
            os.system("rm -rf %s" % PREPROCESSED_DIR_CREATE)
            os.makedirs(PREPROCESSED_DIR_CREATE, exist_ok = True)
            with open("%s/last" % PREPROCESSED_DIR_CREATE, "w") as f:
                f.write(str(t))
                f.write("\n")
            args.update = True
            args.consensus_path = CONSENSUS_PATH
            args.write_preprocessed = PREPROCESSED_DIR_CREATE
            cp.main(args, None)
            self.check_preprocessed_consensus_descriptorstubs()
        finally:
            cp.time.time = oldtime

    def test_main_create(self):
        args = fake_cp_args()
        os.system("rm -rf %s" % PREPROCESSED_DIR_CREATE)
        os.makedirs(PREPROCESSED_DIR_CREATE, exist_ok = True)
        args.consensus_path = CONSENSUS_PATH
        args.write_preprocessed = PREPROCESSED_DIR_CREATE
        cp.main(args, None)
        self.check_preprocessed_consensus_descriptorstubs()

    def check_pcvesely_or1(self, orouter):
        self.assertEqual(orouter.get_ip(), "147.229.13.223")
        self.assertEqual(orouter.get_nickname(), "default")
        self.assertEqual(orouter.get_inconsensus_val_after(), ("2013-04-02", "21:00:00"))

    def check_pcvesely_or2(self, orouter):
        self.assertEqual(orouter.get_ip(), "147.229.13.223")
        self.assertEqual(orouter.get_nickname(), "default")
        self.assertEqual(orouter.get_inconsensus_val_after(), ("2013-04-02", "22:00:00"))

    def check_pcvesely_ors(self, ors):
        self.assertEqual(len(ors), 2)
        self.check_pcvesely_or1(ors[0])
        self.check_pcvesely_or2(ors[1])

    def test_parse_ors_from_file_2or(self):
        ors = cp.parse_ors_from_file("testfiles/preprocessed/147/229/147.229.13.223")
        self.check_pcvesely_ors(ors)

    def test_find_preprocessed_ip_address(self):
        ors = cp.find_preprocessed_ip_address("147.229.13.223", "testfiles/preprocessed")
        self.check_pcvesely_ors(ors)

    def test_find_preprocessed_ip_address_time_filter_None(self):
        ors = cp.find_preprocessed_ip_address_time_filter("147.229.13.223",
                "testfiles/preprocessed/", None, None, None)
        self.check_pcvesely_ors(ors)

    def test_find_preprocessed_ip_address_time_filter_time(self):
        ors = cp.find_preprocessed_ip_address_time_filter("147.229.13.223",
                "testfiles/preprocessed/", TimeWrapper("2013-04-02 21:05:00"), None, None)
        self.check_pcvesely_or1(ors[0])

    @ptc.parametrizable_test([
        ("full_date", ["2013-04-02", check_pcvesely_ors]),
        ("full_date_valid_after_midnight", ["2013-04-03", check_pcvesely_ors]),
        ("plus1", ["2013-04-04", lambda self, ors: self.assertEqual(ors, [])]),
        ("minus1", ["2013-04-01", lambda self, ors: self.assertEqual(ors, [])]),
        ("bad_year", ["2012", lambda self, ors: self.assertEqual(ors, [])]),
        ("dekada", ["2013-04-0", check_pcvesely_ors]),
        ("month", ["2013-04", check_pcvesely_ors]),
        ("year", ["2013", check_pcvesely_ors]),
        ])
    def test_find_preprocessed_ip_address_time_filter_dateprefix(self, date_prefix, checkfunc):
        ors = cp.find_preprocessed_ip_address_time_filter("147.229.13.223",
                "testfiles/preprocessed/", None, date_prefix, None)
        checkfunc(self, ors)

    def fix_pcvesely_ors_for_merge(self, ors):
        ors[1]._onion_router__digest = "pkYCSI2r79XMzeFFVLbF+73Sr3w"
        ors[1]._onion_router__publication = ('2013-04-02', '20:08:40')

    def check_pcvesely_merge(self, ors):
        self.fix_pcvesely_ors_for_merge(ors)
        ors = cp.merge_subsequent_ors(ors)
        self.assertEqual(len(ors), 1)
        orouter = ors[0]
        self.assertEqual(orouter.get_ip(), "147.229.13.223")
        self.assertEqual(orouter.get_nickname(), "default")
        self.assertEqual(orouter.get_inconsensus_val_after(), ("2013-04-02", "21:00:00"))
        self.assertEqual(orouter.get_inconsensus_fresh_until(), ("2013-04-02", "23:00:00"))
        self.assertEqual(orouter.get_inconsensus_val_until(), ("2013-04-03", "01:00:00"))
        return ors

    def test_merge_subsequent_ors(self):
        ors = cp.parse_ors_from_file("testfiles/preprocessed/147/229/147.229.13.223")
        self.check_pcvesely_merge(ors)

    def test_merge_subsequent_ors_failed(self):
        ors = cp.parse_ors_from_file("testfiles/preprocessed/147/229/147.229.13.223")
        self.check_pcvesely_ors(ors)
        ors = cp.merge_subsequent_ors(ors)
        self.assertEqual(len(ors), 2)
        self.check_pcvesely_ors(ors)

    def test_merge_subsequent_ors_dns1(self):
        ors = cp.parse_ors_from_file("testfiles/preprocessed/147/229/147.229.13.223")
        dnsdict = {
                "dns": "pcvesely.fit.vutbr.cz",
                "queried": [("2013-04-02", "21:00:00")]
            }
        ors[0]._onion_router__reverse_name = dnsdict
        ors = self.check_pcvesely_merge(ors)
        self.assertEqual(ors[0]._onion_router__reverse_name, dnsdict)

    def test_merge_subsequent_ors_dns2(self):
        ors = cp.parse_ors_from_file("testfiles/preprocessed/147/229/147.229.13.223")
        dnsdict = {
                "dns": "pcvesely.fit.vutbr.cz",
                "queried": [("2013-04-02", "22:00:00")]
            }
        ors[1]._onion_router__reverse_name = dnsdict
        ors = self.check_pcvesely_merge(ors)
        self.assertEqual(ors[0]._onion_router__reverse_name, dnsdict)

    def test_merge_subsequent_ors_dns_both(self):
        ors = cp.parse_ors_from_file("testfiles/preprocessed/147/229/147.229.13.223")
        dnsdict = [{
                "dns": "pcvesely.fit.vutbr.cz",
                "queried": [("2013-04-02", "21:00:00")]
            },
            {
                "dns": "pcvesely.fit.vutbr.cz",
                "queried": [("2013-04-02", "22:00:00")]
            }]
        ors[0]._onion_router__reverse_name = dnsdict[0]
        ors[1]._onion_router__reverse_name = dnsdict[1]
        ors = self.check_pcvesely_merge(ors)
        expected = {
                "dns": "pcvesely.fit.vutbr.cz",
                "queried": [("2013-04-02", "21:00:00"), ("2013-04-02", "22:00:00")]
            }
        self.assertEqual(ors[0]._onion_router__reverse_name, expected)

    def test_get_ip_address_activity(self):
        self.assertEqual(cp.get_ip_address_activity("147.229.13.223", "testfiles/preprocessed/"),
                ["2013-04-02", "2013-04-03"])

    def test_fixup(self):
        PREPROCESSED_TEMPLATE = "testfiles/preprocessed"
        os.system("rm -rf %s" % PREPROCESSED_DIR_CREATE)
        IP_ADDR = "67.161.31.147"
        DYNAMIC_FILE = cp.create_ip_filename(PREPROCESSED_DIR_CREATE, IP_ADDR, create_path = True)
        TEMPLATE_FILE = cp.create_ip_filename(PREPROCESSED_TEMPLATE, IP_ADDR)
        os.system("cp %s %s" % (TEMPLATE_FILE, DYNAMIC_FILE))
        cp.fixup_missing_consensus(os.path.abspath(CONSENSUS_PATH),
                os.path.abspath(PREPROCESSED_DIR_CREATE), TimeWrapper("2018-08-04 19-00").get())
        for addr in ["67.161.31.147",
                "174.127.217.73",
                "162.247.74.201",
                "77.123.42.148",
                "2001:470:71:9b9:f66d:4ff:fee7:954c"]:
            self.assertTrue(filecmp.cmp(cp.create_ip_filename(PREPROCESSED_DIR_CREATE, addr),
                    "testfiles/check/test_cp_test_fixup/%s" % addr, shallow=False))

    @ptc.parametrizable_test([
            ("tmp_0", ("/tmp", 0, "/tmp/consensuses-1970-01/01", "1970-01-01-00-00-00-consensus")),
            ("conpath_2018", ("/mnt/data/descriptors", TimeWrapper("2018-08-08 09-00-00").get(),
                "/mnt/data/descriptors/consensuses-2018-08/08", "2018-08-08-09-00-00-consensus")),
        ])
    def test_get_consensus_fullpath(self, abspath, t, directory, fname):
        fullpath = "%s/%s" % (directory, fname)
        self.assertEqual(cp.get_consensus_fullpath(abspath, t), (directory, fname, fullpath))

    def test_download_consensus_if_missing(self):
        cpath = os.path.abspath("testfiles/test_download_consensus_if_missing")
        try:
            os.makedirs(cpath, exist_ok = True)
            now = cp.time.gmtime()
            t = TimeWrapper(cp.time.strftime("%Y-%m-%d %H:%M", now)).get() - 7200 # aim to get a past consensus
            cp.download_consensus_if_missing(cpath, t)
            _, _, fullpath = cp.get_consensus_fullpath(cpath, t)
            self.assertTrue(len(cp.create_snapshot_from_consensus_file(fullpath).get_ipaddrs()) > 0)
        finally:
            os.system("rm -rf %s" % cpath)

def suite():
        test = unittest.makeSuite(test_consensus_parser, "test")
        return unittest.TestSuite(tuple(test))

def test(verbosity=2, failfast=False):
        runner = unittest.TextTestRunner(verbosity=verbosity,failfast=failfast)
        runner.run(suite())

if __name__ == '__main__':
    test(verbosity=2)
