#!/usr/bin/env python3
# -*- coding: utf-8 -*-

""" Test package for time wrapper """
# Copyright (C) 2018 Libor Polčák
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import unittest

from time_parser import TimeWrapper, TorTimeWrapper, FormatTimeWrapper

import parametrizable_tc as ptc

class test_time_wrapper(unittest.TestCase, metaclass=ptc.parametrizable_tc_meta):

    @ptc.parametrizable_test([
            ("tor_format", ["2011-05-02 11:00:00", "2011-05-02 11:00:00"]),
            ("add_seconds", ["2011-05-02 11:00", "2011-05-02 11:00:00"]),
            ("human_us", ["Jan 2 2016", "2016-01-02 00:00:00"]),
            ("human_en", ["5th Jan 2018", "2018-01-05 00:00:00"]),
            ("dots_en/us", ["1.5.2018", "2018-01-05 00:00:00"]),
            ("dots_en/us_spaces", ["1. 5. 2018", "2018-01-05 00:00:00"]),
            ("time.time", [1529062032.8638437, "2018-06-15 11:27:12"]),
            ("onion_router_validity", [('2018-06-18', '22:00:00'), "2018-06-18 22:00:00"]),
        ])
    def test_TimeWrapper(self, valuein, expected):
        self.assertEqual(str(TimeWrapper(valuein)), expected)

    def test_TimeWrapper_get(self):
        self.assertEqual(TimeWrapper("1970-01-01 00:00:00").get(), 0)

    @ptc.parametrizable_test([
            ("tor_format", ["2011-05-02", "11:00:00", "2011-05-02 11:00:00"]),
            ("epoch_start", ["1970-01-01", "00:00:00", "1970-01-01 00:00:00"]),
            ("after_epoch", ["2040-01-01", "00:00:00", "2040-01-01 00:00:00"]),
        ])
    def test_TorTimeWrapper(self, datein, timein, expected):
        self.assertEqual(str(TorTimeWrapper(datein, timein)), expected)

    def test_TorTimeWrapper_get(self):
        self.assertEqual(TorTimeWrapper("1970-01-01", "00:00:00").get(), 0)

    @ptc.parametrizable_test([
            ("tor_format", ["2011-05-02 11:00:00", "%Y-%m-%d %H:%M:%S", "2011-05-02 11:00:00"]),
            ("alacz_format", ["02.05.2011 11:00:00", "%d.%m.%Y %H:%M:%S", "2011-05-02 11:00:00"]),
            ("dateonly_format-geolitemodule", ["20110502", "%Y%m%d", "2011-05-02 00:00:00"]),
        ])
    def test_FormatTimeWrapper(self, timein, formatin, expected):
        self.assertEqual(str(FormatTimeWrapper(timein, formatin)), expected)

    def test_FormatTimeWrapper_get(self):
        self.assertEqual(FormatTimeWrapper("1970-01-01", "%Y-%m-%d").get(), 0)

    @ptc.parametrizable_test([
            ("TimeWrapper", [TimeWrapper, ("1970-01-01",)]),
            ("TimeWrapper_dots", [TimeWrapper, ("1.1.1970",)]),
            ("TorTimeWrapper", [TorTimeWrapper, ("1970-01-01", "00:00:00")]),
            ("FormatTimeWrapper", [FormatTimeWrapper, ("1970-01-01", "%Y-%m-%d")]),
        ])
    def test_zero(self, cls, args):
        t = cls(*args)
        self.assertEqual(t.get(), 0)
        self.assertEqual(t.format("%Y-%m-%d"), "1970-01-01")
        self.assertEqual(str(t), "1970-01-01 00:00:00")

    @ptc.parametrizable_test([
            ("date_only", ["2018-07-17 12:00:00", "%Y-%m-%d", "2018-07-17"]),
        ])
    def test_format(self, timein, reqformat, expected):
        self.assertEqual(TimeWrapper(timein).format(reqformat), expected)

    @ptc.parametrizable_test([
            ("test_written", ["2018-07-17", "2018-07-18"]),
            ("last_in_month", ["2018-07-31", "2018-08-01"]),
            ("last_in_year", ["2018-12-31", "2019-01-01"]),
            ("feb29", ["2016-02-28", "2016-02-29"]),
        ])
    def test_next_day(self, timein, expected_date):
        self.assertEqual(TimeWrapper(timein).next_day().format("%Y-%m-%d"), expected_date)

    @ptc.parametrizable_test([
            ("test_written", ["2018-07-17", "2018-07-16"]),
            ("first_in_month", ["2018-07-01", "2018-06-30"]),
            ("first_in_year", ["2018-01-01", "2017-12-31"]),
            ("feb29", ["2016-03-01", "2016-02-29"]),
        ])
    def test_prev_day(self, timein, expected_date):
        self.assertEqual(TimeWrapper(timein).prev_day().format("%Y-%m-%d"), expected_date)

    @ptc.parametrizable_test([
            ("last_in_month_31", ["2018-10-31", "2018-11-30 00:00:00"]),
            ("first_in_month_31", ["2018-10-01", "2018-11-01 00:00:00"]),
            ("last_in_month_31_backtoback", ["2018-07-31", "2018-08-31 00:00:00"]),
            ("first_in_month_31_backtoback", ["2018-07-01", "2018-08-01 00:00:00"]),
            ("last_in_month_30", ["2018-09-30", "2018-10-30 00:00:00"]),
            ("first_in_month_30", ["2018-09-01", "2018-10-01 00:00:00"]),
            ("last_in_month_29", ["2016-02-29", "2016-03-29 00:00:00"]),
            ("first_in_month_29", ["2016-02-01", "2016-03-01 00:00:00"]),
            ("last_before_month_29", ["2016-01-31", "2016-02-29 00:00:00"]),
            ("first_before_month_29", ["2016-01-01", "2016-02-01 00:00:00"]),
            ("last_in_month_28", ["2018-02-28", "2018-03-28 00:00:00"]),
            ("first_in_month_28", ["2018-02-01", "2018-03-01 00:00:00"]),
            ("last_before_month_28", ["2018-01-31", "2018-02-28 00:00:00"]),
            ("first_before_month_28", ["2018-01-01", "2018-02-01 00:00:00"]),
            ("last_in_year", ["2018-12-31", "2019-01-31 00:00:00"]),
            ("first_in_year", ["2018-01-01", "2018-02-01 00:00:00"]),
        ])
    def test_next_month(self, timein, expected_date):
        self.assertEqual(str(TimeWrapper(timein).next_month()), expected_date)

    @ptc.parametrizable_test([
            ("last_in_month_31", ["2018-10-31", "2018-09-30 00:00:00"]),
            ("first_in_month_31", ["2018-10-01", "2018-09-01 00:00:00"]),
            ("last_in_month_31_backtoback", ["2018-08-31", "2018-07-31 00:00:00"]),
            ("first_in_month_31_backtoback", ["2018-08-01", "2018-07-01 00:00:00"]),
            ("last_in_month_30", ["2018-09-30", "2018-08-30 00:00:00"]),
            ("first_in_month_30", ["2018-09-01", "2018-08-01 00:00:00"]),
            ("last_in_month_29", ["2016-02-29", "2016-01-29 00:00:00"]),
            ("first_in_month_29", ["2016-02-01", "2016-01-01 00:00:00"]),
            ("last_after_month_29", ["2016-03-31", "2016-02-29 00:00:00"]),
            ("first_after_month_29", ["2016-03-01", "2016-02-01 00:00:00"]),
            ("last_in_month_28", ["2018-02-28", "2018-01-28 00:00:00"]),
            ("first_in_month_28", ["2018-02-01", "2018-01-01 00:00:00"]),
            ("last_after_month_28", ["2018-03-31", "2018-02-28 00:00:00"]),
            ("first_after_month_28", ["2018-03-01", "2018-02-01 00:00:00"]),
            ("last_in_first_month", ["2018-01-31", "2017-12-31 00:00:00"]),
            ("first_in_year", ["2018-01-01", "2017-12-01 00:00:00"]),
        ])
    def test_prev_month(self, timein, expected_date):
        self.assertEqual(str(TimeWrapper(timein).prev_month()), expected_date)

def suite():
        test = unittest.makeSuite(test_time_wrapper, "test")
        return unittest.TestSuite(tuple(test))

def test(verbosity=2, failfast=False):
        runner = unittest.TextTestRunner(verbosity=verbosity,failfast=failfast)
        runner.run(suite())

if __name__ == '__main__':
    test(verbosity=2)
