// CEPNobj.cpp - implementation for drawing objects
//

#include "stdafx.h"
#include "CEPNobj.h"
#include <math.h>

#include "MainFrm.h"
	
////////////////////////////////////////////////////////////////////////////
// CPNElement
IMPLEMENT_SERIAL(CPNElement,CObject,1|VERSIONABLE_SCHEMA)

CPen CPNElement::m_sel_pen(PS_SOLID,1,SEL_CLR_PEN);
CBrush CPNElement::m_sel_brush(SEL_CLR_BRUSH);

CPNElement::CPNElement()
{
	m_selected=false;
	m_position.x=m_position.y=0;
	m_old_pen=NULL;
	m_old_brush=NULL;
}

CPNElement::CPNElement(CPoint point)
{
	// init
	m_selected=false;
	m_position.x=m_position.y=0;
	m_old_pen=NULL;
	m_old_brush=NULL;

	m_position=point;
}

CPNElement::~CPNElement()
{
}

void CPNElement::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);

	CObject::Serialize(ar);
	if(ar.IsStoring())
	{
		ar << m_position;
	}
	else
	{
		ar >> m_position;
	}
}

void CPNElement::Draw(CDC*)
{
	ASSERT_VALID(this);
}

CRect CPNElement::MoveTo(const CPoint& point)
{
	ASSERT_VALID(this);

	m_position=point;
	return GetRect();
}

CPoint CPNElement::CalculateIntersection(const CPoint&)
{
	ASSERT_VALID(this);

	return m_position;
}

bool CPNElement::HitTest(const CPoint&)
{
	ASSERT_VALID(this);

	return false;
}

CRect CPNElement::GetRect()
{
	ASSERT_VALID(this);

	return CRect(0,0,0,0);
}

void CPNElement::DrawSelMark(CDC* pDC,int x,int y)
{
	ASSERT_VALID(this);
	ASSERT(pDC);
	m_old_pen=(CPen*)pDC->SelectObject(&m_sel_pen);
	m_old_brush=(CBrush*)pDC->SelectObject(&m_sel_brush);
	pDC->Rectangle(x,y,x+SEL_RECT,y+SEL_RECT);
	pDC->SelectObject(m_old_brush);
	pDC->SelectObject(m_old_pen);
}

void CPNElement::DrawSelRect(CDC* pDC,int center_x,int center_y,int half_edge_x,int half_edge_y)
{
	ASSERT_VALID(this);
	ASSERT(pDC);
	DrawSelMark(pDC,center_x-half_edge_x-SEL_RECT,center_y-half_edge_y-SEL_RECT);
	DrawSelMark(pDC,center_x+half_edge_x,center_y-half_edge_y-SEL_RECT);
	DrawSelMark(pDC,center_x-half_edge_x-SEL_RECT,center_y+half_edge_y);
	DrawSelMark(pDC,center_x+half_edge_x,center_y+half_edge_y);
}

void CPNElement::UpdatePropertiesDlg()
{
	ASSERT_VALID(this);
}

////////////////////////////////////////////////////////////////////////////
// CPNLabel
IMPLEMENT_SERIAL(CPNLabel,CPNElement,1|VERSIONABLE_SCHEMA)

CPNLabel::CPNLabel()
{
	m_text_color=TEXT_COLOR;
	::ZeroMemory(&m_logfont,sizeof(m_logfont));
	m_logfont.lfCharSet=1;
	m_logfont.lfHeight=-12;
	strcpy(m_logfont.lfFaceName,"Arial");
	m_font.CreateFontIndirect(&m_logfont);
	m_visible=true;
	m_hit_rect.SetRectEmpty();
	m_owner=NULL;
	m_old_font=NULL;
}

CPNLabel::CPNLabel(CPoint point) : CPNElement(point)
{
	m_text_color=TEXT_COLOR;
	::ZeroMemory(&m_logfont,sizeof(m_logfont));
	m_logfont.lfCharSet=1;
	m_logfont.lfHeight=-12;
	strcpy(m_logfont.lfFaceName,"Arial");
	m_font.CreateFontIndirect(&m_logfont);
	m_visible=true;
	m_hit_rect.SetRectEmpty();
	m_owner=NULL;
	m_old_font=NULL;
}

CPNLabel::~CPNLabel()
{
}

void CPNLabel::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);

	CPNElement::Serialize(ar);
	if(ar.IsStoring())
	{
		ar << m_text;
		ar << m_text_color;
		ar << m_logfont.lfCharSet;
		ar << m_logfont.lfClipPrecision;
		ar << m_logfont.lfEscapement;
		CString facename=m_logfont.lfFaceName;
		ar << facename;
		ar << m_logfont.lfHeight;
		ar << m_logfont.lfItalic;
		ar << m_logfont.lfOrientation;
		ar << m_logfont.lfOutPrecision;
		ar << m_logfont.lfPitchAndFamily;
		ar << m_logfont.lfQuality;
		ar << m_logfont.lfStrikeOut;
		ar << m_logfont.lfUnderline;
		ar << m_logfont.lfWeight;
		ar << m_logfont.lfWidth;
		ar << m_visible;
		ar << m_hit_rect;
		ar << m_owner;
	}
	else
	{
		ar >> m_text;
		ar >> m_text_color;
		ar >> m_logfont.lfCharSet;
		ar >> m_logfont.lfClipPrecision;
		ar >> m_logfont.lfEscapement;
		CString facename;
		ar >> facename;
		strcpy(m_logfont.lfFaceName,facename.operator LPCTSTR());
		ar >> m_logfont.lfHeight;
		ar >> m_logfont.lfItalic;
		ar >> m_logfont.lfOrientation;
		ar >> m_logfont.lfOutPrecision;
		ar >> m_logfont.lfPitchAndFamily;
		ar >> m_logfont.lfQuality;
		ar >> m_logfont.lfStrikeOut;
		ar >> m_logfont.lfUnderline;
		ar >> m_logfont.lfWeight;
		ar >> m_logfont.lfWidth;
		ar >> m_visible;
		ar >> m_hit_rect;
		ar >> m_owner;

		m_font.DeleteObject();
		m_font.CreateFontIndirect(&m_logfont);

/*		ar >> m_hit_rect;
		ar >> m_visible;
		ar >> m_text;
		ar >> m_owner;*/

	}
}

void CPNLabel::Draw(CDC* pDC)
{	// draw this object on pDC
	ASSERT_VALID(this);

	if(!m_visible)return;

	if(m_selected)
	{	// four rectangles around object
		DrawSelRect(pDC,m_hit_rect.left+m_hit_rect.Width()/2,m_hit_rect.top+m_hit_rect.Height()/2,m_hit_rect.Width()/2,m_hit_rect.Height()/2);
	}
	int old_mode=pDC->SetBkMode(TRANSPARENT);
	m_old_font=pDC->SelectObject(&m_font);
	COLORREF old_color=pDC->SetTextColor(m_text_color);
	pDC->DrawText(m_text,&m_hit_rect,DT_EXTERNALLEADING|DT_NOPREFIX|DT_NOCLIP);
	pDC->SetTextColor(old_color);
	pDC->SelectObject(m_old_font);
	pDC->SetBkMode(old_mode);
}

CRect CPNLabel::MoveTo(const CPoint& point)
{	// move the center of this object to point
	ASSERT_VALID(this);

	CRect old_rect=GetRect();
	m_hit_rect.OffsetRect(point-m_position);
	m_position=point;
	CRect invalid_rect;
	invalid_rect.UnionRect(old_rect,GetRect());
	return invalid_rect;
}

bool CPNLabel::HitTest(const CPoint& point)
{	// return whether point is in this object
	ASSERT_VALID(this);

	return m_visible && m_hit_rect.PtInRect(point)!=0;
}

CRect CPNLabel::GetRect()
{	// return rect for redraw
	ASSERT_VALID(this);

	CRect rect=m_hit_rect;
	rect.InflateRect(SEL_RECT+1,SEL_RECT+1); // for four rectangles when element is selected
	return rect;
}

void CPNLabel::SetText(const CString& text,CDC* pDC)
{	// set the visible text
	ASSERT_VALID(this);

	m_text=text;
	UpdateText(pDC);
}

void CPNLabel::SetText(const CString& text)
{
	m_text=text;
}

void CPNLabel::SetLogFont(const LOGFONT& font)
{
	m_logfont=font;
	m_font.DeleteObject();
	m_font.CreateFontIndirect(&m_logfont);
}

void CPNLabel::UpdateText(CDC* pDC)
{
	ASSERT_VALID(this);

	m_hit_rect.SetRectEmpty();
	m_old_font=pDC->SelectObject(&m_font);
	pDC->DrawText(m_text,&m_hit_rect,DT_CALCRECT|DT_EXTERNALLEADING|DT_NOPREFIX);
	pDC->SelectObject(m_old_font);
	m_hit_rect.OffsetRect(-m_hit_rect.Width()/2,-m_hit_rect.Height()/2);
	m_hit_rect.OffsetRect(m_position);
}

void CPNLabel::UpdatePropertiesDlg()
{
	ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
	CMainFrame* frame=(CMainFrame*)AfxGetApp()->m_pMainWnd;

	frame->ShowPropertiesDlg(CMainFrame::PROPDLG_LABEL,this);
}

////////////////////////////////////////////////////////////////////////////
// CPNCondition
IMPLEMENT_SERIAL(CPNCondition,CPNElement,1|VERSIONABLE_SCHEMA)

CPen CPNCondition::m_tkn_pen(PS_SOLID,1,TKN_CLR);
CBrush CPNCondition::m_tkn_brush(TKN_CLR);

CPNCondition::CPNCondition()
{
	m_token=false;
	m_half_full_token=false;
	m_pLabel=NULL;
}

CPNCondition::CPNCondition(CPoint point,CPNLabel* pLabel) : CPNElement(point)
{
	// init
	m_token=false;
	m_half_full_token=false;
	m_pLabel=NULL;

	if(pLabel!=NULL)
	{
		m_pLabel=pLabel;
		point.Offset(0,-CND_RDS-LBL_DST);
		m_pLabel->MoveTo(point);
	}
}

CPNCondition::~CPNCondition()
{
}

void CPNCondition::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);

	CPNElement::Serialize(ar);
	if(ar.IsStoring())
	{
		ar << m_token;
		ar << m_pLabel;
	}
	else
	{
		ar >> m_token;
		ar >> m_pLabel;
	}
}

void CPNCondition::Draw(CDC* pDC)
{	// draw this object on pDC
	ASSERT_VALID(this);

	if(m_selected)
	{	// four rectangles around object
		DrawSelRect(pDC,m_position.x,m_position.y,CND_SEL_RDS,CND_SEL_RDS);
	}
	pDC->Ellipse(m_position.x-CND_RDS,m_position.y-CND_RDS,m_position.x+CND_RDS,m_position.y+CND_RDS);
	if(m_token || m_half_full_token)
	{
		m_old_pen=pDC->SelectObject(&m_tkn_pen);
		if(!m_half_full_token)m_old_brush=pDC->SelectObject(&m_tkn_brush);
		pDC->Ellipse(m_position.x-TKN_RDS,m_position.y-TKN_RDS,m_position.x+TKN_RDS,m_position.y+TKN_RDS);
		if(!m_half_full_token)pDC->SelectObject(m_old_brush);
		pDC->SelectObject(m_old_pen);
	}
}

CRect CPNCondition::MoveTo(const CPoint& point)
{	// move the center of this object to point
	ASSERT_VALID(this);

	CSize move=point-m_position;

	CRect cond_rect;
	CRect old_rect=GetRect();
	m_position=point;
	CRect new_rect=GetRect();
	cond_rect.UnionRect(old_rect,new_rect);

	CRect label_rect;
	label_rect.SetRectEmpty();
	if(m_pLabel)
	{	// move associated label
		old_rect=m_pLabel->GetRect();
		CPoint position=m_pLabel->GetPosition();
		position.Offset(move);
		m_pLabel->MoveTo(position);
		new_rect=m_pLabel->GetRect();
		label_rect.UnionRect(old_rect,new_rect);
	}

	CRect invalid_rect;
	invalid_rect.UnionRect(label_rect,cond_rect);
	return invalid_rect;
}

CPoint CPNCondition::CalculateIntersection(const CPoint& pt_to)
{	// calculate intersection with this element and line from center of this element to point pt_to
	ASSERT_VALID(this);

	double u=pt_to.x-m_position.x;
	double v=pt_to.y-m_position.y;
	// line is in parametric form
	double t=sqrt((EVN_RDS*EVN_RDS)/(u*u+v*v));
	// two possible results
	CPoint p_1(static_cast<int>(m_position.x+t*u),static_cast<int>(m_position.y+t*v));
	CPoint p_2(static_cast<int>(m_position.x-t*u),static_cast<int>(m_position.y-t*v));
	// compute the hit rectangle of line
	CRect hit_rect;
	hit_rect.SetRect(m_position.x,m_position.y,pt_to.x,pt_to.y);
	hit_rect.NormalizeRect();
	hit_rect.InflateRect(1,1);
	// on which line the point belong
	CPoint point=p_2; // default is p_2
	if(hit_rect.PtInRect(p_1)!=0)point=p_1;

	return point;
}

bool CPNCondition::HitTest(const CPoint& point)
{	// return whether point is in this object
	ASSERT_VALID(this);

	// circle: (x-m)^2+(y-n)^2=R^2
	int x=point.x-m_position.x;
	int y=point.y-m_position.y;
	// f(x,y)=x^2+y^2-R^2, <0 : inside, =0 : on, >0 : outside
	return (x*x+y*y-CND_RDS*CND_RDS)<=0;
}

CRect CPNCondition::GetRect()
{	// return rect for redraw
	ASSERT_VALID(this);

	CRect rect(m_position.x-CND_RDS,m_position.y-CND_RDS,m_position.x+CND_RDS,m_position.y+CND_RDS);
	rect.InflateRect(SEL_RECT+1,SEL_RECT+1); // for four rectangles when element is selected
	return rect;
}

void CPNCondition::UpdatePropertiesDlg()
{
	ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
	CMainFrame* frame=(CMainFrame*)AfxGetApp()->m_pMainWnd;

	frame->ShowPropertiesDlg(CMainFrame::PROPDLG_COND,this);
}

////////////////////////////////////////////////////////////////////////////
// CPNEvent
IMPLEMENT_SERIAL(CPNEvent,CPNElement,1|VERSIONABLE_SCHEMA)

CPen CPNEvent::m_line_pen(PS_SOLID,1,LINE_CLR);
CPen CPNEvent::m_rlz_pen(PS_SOLID,1,RLZ_CLR);
CPen CPNEvent::m_rlzing_pen(PS_SOLID,1,RLZING_CLR);

CPNEvent::CPNEvent()
{
	m_pLabel=NULL;
	m_hit_rect.SetRectEmpty();
	m_isrealizable=false;
	m_realizing=false;
}

CPNEvent::CPNEvent(CPoint point,CPNLabel* pLabel) : CPNElement(point)
{
	// init
	m_pLabel=NULL;
	m_hit_rect.SetRectEmpty();
	m_isrealizable=false;
	m_realizing=false;

	if(pLabel!=NULL)
	{
		m_pLabel=pLabel;
		point.Offset(0,-EVN_RDS-LBL_DST);
		m_pLabel->MoveTo(point);
	}
	ComputeHitRect();
}

CPNEvent::~CPNEvent()
{
}

void CPNEvent::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);

	CPNElement::Serialize(ar);
	if(ar.IsStoring())
	{
		ar << m_pLabel;
	}
	else
	{
		ar >> m_pLabel;

		ComputeHitRect(); // compute coeficients m_rect_coef[]

/*		ar >> m_pLabel;
		ar >> m_hit_rect;
		for(int i=0;i<4;i++)
		{
			ar >> m_rect_coef[i].x;
			ar >> m_rect_coef[i].y;
			ar >> m_rect_coef[i].u;
			ar >> m_rect_coef[i].v;
		}
*/
	}
}

void CPNEvent::Draw(CDC* pDC)
{	// draw this object on pDC
	ASSERT_VALID(this);

	if(m_selected)
	{	// four rectangles around object
		DrawSelRect(pDC,m_position.x,m_position.y,EVN_RDS,EVN_RDS);
	}
	if(m_isrealizable)
	{
		m_old_pen=pDC->SelectObject(&m_rlz_pen);
	}
	else
	{
		if(m_realizing)
			m_old_pen=pDC->SelectObject(&m_rlzing_pen);
		else
			m_old_pen=pDC->SelectObject(&m_line_pen);
	}
	pDC->Rectangle(m_position.x-EVN_RDS,m_position.y-EVN_RDS,m_position.x+EVN_RDS,m_position.y+EVN_RDS);
	pDC->SelectObject(m_old_pen);
}

CRect CPNEvent::MoveTo(const CPoint& point)
{	// move the center of this object to point
	ASSERT_VALID(this);

	CSize move=point-m_position;

	CRect event_rect;
	CRect old_rect=GetRect();
	m_position=point;
	ComputeHitRect();
	CRect new_rect=GetRect();
	event_rect.UnionRect(old_rect,new_rect);

	CRect label_rect;
	label_rect.SetRectEmpty();
	if(m_pLabel)
	{	// move associated label	
		old_rect=m_pLabel->GetRect();
		CPoint position=m_pLabel->GetPosition();
		position.Offset(move);
		m_pLabel->MoveTo(position);
		new_rect=m_pLabel->GetRect();
		label_rect.UnionRect(old_rect,new_rect);
	}

	CRect invalid_rect;
	invalid_rect.UnionRect(label_rect,event_rect);
	return invalid_rect;
}

CPoint CPNEvent::CalculateIntersection(const CPoint& to)
{	// calculate intersect with this element and line from center of this element and the center of second element
	ASSERT_VALID(this);

	int u=to.x-m_position.x;
	int v=to.y-m_position.y;

	CRect hit_rect;
	// compute the hint rectangle of line
	hit_rect.SetRect(m_position.x,m_position.y,to.x,to.y);
	hit_rect.NormalizeRect();
	hit_rect.InflateRect(1,1);

	CPoint point=m_position;
	for(int i=0;i<4;i++)
	{	
		double cit=u*m_rect_coef[i].y-m_position.y*u-v*m_rect_coef[i].x+v*m_position.x;
		int jm=v*m_rect_coef[i].u-u*m_rect_coef[i].v;
		if(!jm)continue; // lines are collinear => division by zero
		double t=cit/jm;
		if((t<0.0) || (t>1.0))continue; // out of line segment
		// point of intersection
		point.x=static_cast<int>(m_rect_coef[i].x+t*m_rect_coef[i].u);
		point.y=static_cast<int>(m_rect_coef[i].y+t*m_rect_coef[i].v);
		if(hit_rect.PtInRect(point)!=0)break; // point is point of intersection
	}
	return point;
}

bool CPNEvent::HitTest(const CPoint& point)
{	// return whether point is in this object
	ASSERT_VALID(this);

	return m_hit_rect.PtInRect(point)!=0;
}

CRect CPNEvent::GetRect()
{	// return rect for redraw
	ASSERT_VALID(this);

	CRect rect=m_hit_rect;
	rect.InflateRect(SEL_RECT+1,SEL_RECT+1); // for four rectangles when element is selected
	return rect;
}

void CPNEvent::ComputeHitRect()
{	// compute hit rect and lines for intersection
	m_hit_rect.SetRect(m_position.x-EVN_RDS,m_position.y-EVN_RDS,m_position.x+EVN_RDS,m_position.y+EVN_RDS);
	m_hit_rect.InflateRect(1,1);
	// coefficients for lines
	// top line
	m_rect_coef[0].x=m_position.x-EVN_RDS-1;
	m_rect_coef[0].y=m_position.y-EVN_RDS;
	m_rect_coef[0].u=2*EVN_RDS+1;
	m_rect_coef[0].v=0;
	// right line
	m_rect_coef[1].x=m_position.x+EVN_RDS;
	m_rect_coef[1].y=m_position.y-EVN_RDS;
	m_rect_coef[1].u=0;
	m_rect_coef[1].v=2*EVN_RDS;
	// bottom line
	m_rect_coef[2].x=m_position.x+EVN_RDS;
	m_rect_coef[2].y=m_position.y+EVN_RDS;
	m_rect_coef[2].u=(-2)*EVN_RDS-1;
	m_rect_coef[2].v=0;
	// left line
	m_rect_coef[3].x=m_position.x-EVN_RDS-1;
	m_rect_coef[3].y=m_position.y+EVN_RDS;
	m_rect_coef[3].u=0;
	m_rect_coef[3].v=(-2)*EVN_RDS;
}

bool CPNEvent::UpdateRealizable(CSetConditions& case_set,bool forward)
{	// find out whether is this event realizable in case_set
	ASSERT_VALID(this);

	m_isrealizable=case_set.IsRealizable(m_preset,m_postset,forward);
	return m_isrealizable;
}

void CPNEvent::UpdatePropertiesDlg()
{
	ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
	CMainFrame* frame=(CMainFrame*)AfxGetApp()->m_pMainWnd;

	frame->ShowPropertiesDlg(CMainFrame::PROPDLG_EVNT,this);
}

////////////////////////////////////////////////////////////////////////////
// CEdgePoint - support class
IMPLEMENT_SERIAL(CEdgePoint,CObject,1|VERSIONABLE_SCHEMA)

CEdgePoint::CEdgePoint()
{
}

CEdgePoint::CEdgePoint(CEdgePoint& edgepoint)
{
	point=edgepoint.point;
	hit_rect=edgepoint.hit_rect;
	bend=edgepoint.bend;
	side=edgepoint.side;
}

void CEdgePoint::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);
	
	BOOL b;
	CObject::Serialize(ar);
	if(ar.IsStoring())
	{
		ar << point;
		ar << hit_rect;
		b=bend;
		ar << b;
		b=side;
		ar << b;
	}
	else
	{
		ar >> point;
		ar >> hit_rect;
		ar >> b;
		bend=(b==TRUE);
		ar >> b;
		side=(b==TRUE);
	}
}

CEdgePoint& CEdgePoint::operator = (CEdgePoint& edgepoint)
{
	point=edgepoint.point;
	hit_rect=edgepoint.hit_rect;
	bend=edgepoint.bend;
	side=edgepoint.side;
	return *this;
}

////////////////////////////////////////////////////////////////////////////
// CPNEdge
IMPLEMENT_SERIAL(CPNEdge,CPNElement,1|VERSIONABLE_SCHEMA)

CPNEdge::CPNEdge()
{
	m_element_from=m_element_to=NULL;
	m_correct_edge=false;
	m_hit_rect.SetRectEmpty();
	m_drag_point_pos_right=m_drag_point_pos_left=m_drag_point_pos=NULL;
	m_multi_point=false;
	m_multi_pos_left=m_multi_pos_right=NULL;
	m_multi_correct_left=m_multi_correct_right=false;
}

CPNEdge::CPNEdge(CPNElement* from,CPNElement* to,int onlyprepare/*=false*/)
{
	ASSERT(from);
	ASSERT(to);

	// init
	m_element_from=m_element_to=NULL;
	m_correct_edge=false;
	m_hit_rect.SetRectEmpty();
	m_drag_point_pos_right=m_drag_point_pos_left=m_drag_point_pos=NULL;
	m_multi_point=false;
	m_multi_pos_left=m_multi_pos_right=NULL;
	m_multi_correct_left=m_multi_correct_right=false;

	m_element_from=from;
	m_element_to=to;
	if(!onlyprepare)
	{
		// construct edge
		m_multi_correct_left=m_multi_correct_right=true;
		m_correct_edge=true;
		CEdgePoint point_fst,point_mid,point_lst;
		point_fst.point=m_element_from->GetPosition();
		point_fst.bend=true;
		point_fst.side=true;
		point_lst.point=m_element_to->GetPosition();
		point_lst.bend=true;
		point_lst.side=true;
		// calculate intersection points
		point_fst.point=m_element_from->CalculateIntersection(point_lst.point);
		point_lst.point=m_element_to->CalculateIntersection(point_fst.point);
		point_mid.point.x=(point_fst.point.x+point_lst.point.x)/2;
		point_mid.point.y=(point_fst.point.y+point_lst.point.y)/2;
		point_mid.bend=false;
		point_mid.side=false;
		m_points.AddTail(point_fst);
		m_points.AddTail(point_mid);
		m_points.AddTail(point_lst);

		m_correct_edge=IsEdgeCorrect();
		ComputeHitRect();
	}
}

CPNEdge::~CPNEdge()
{
	m_points.RemoveAll();
}

void CPNEdge::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);

	BOOL b;
	CPNElement::Serialize(ar);
	if(ar.IsStoring())
	{
		ar << m_element_from;
		ar << m_element_to;
		b=m_correct_edge;
		ar << b;
		ar << m_hit_rect;
		b=m_multi_point;
		ar << b;
		b=m_multi_correct_left;
		ar << b;
		b=m_multi_correct_right;
		ar << b;

		m_points.Serialize(ar);
	}
	else
	{
		ar >> m_element_from;
		ar >> m_element_to;
		ar >> b;
		m_correct_edge=(b==TRUE);
		ar >> m_hit_rect;
		ar >> b;
		m_multi_point=(b==TRUE);
		ar >> b;
		m_multi_correct_left=(b==TRUE);
		ar >> b;
		m_multi_correct_right=(b==TRUE);

		m_points.Serialize(ar);

		// update pointers to border points
		m_multi_pos_right=m_multi_pos_left=NULL;
		POSITION pos=m_points.GetHeadPosition();
		if(pos!=NULL)
		{
			m_points.GetNext(pos);
			m_points.GetNext(pos);
			m_multi_pos_left=pos;
			pos=m_points.GetTailPosition();
			m_points.GetPrev(pos);
			m_points.GetPrev(pos);
			m_multi_pos_right=pos;
		}

/*		ar >> m_element_from;
		ar >> m_element_to;
		CPoint point;
		ar >> point;
		ar >> point;
		ar >> point;
		CRect rect;
		ar >> rect;
		ar >> rect;
		BOOL m_dragged;
		ar >> m_dragged;

	// construct edge
	m_multi_correct_left=m_multi_correct_right=true;
	m_correct_edge=true;
	CEdgePoint point_fst,point_mid,point_lst;
	point_fst.point=m_element_from->GetPosition();
	point_fst.bend=true;
	point_fst.side=true;
	point_lst.point=m_element_to->GetPosition();
	point_lst.bend=true;
	point_lst.side=true;
	// calculate intersection points
	point_fst.point=m_element_from->CalculateIntersection(point_lst.point);
	point_lst.point=m_element_to->CalculateIntersection(point_fst.point);
	point_mid.point.x=(point_fst.point.x+point_lst.point.x)/2;
	point_mid.point.y=(point_fst.point.y+point_lst.point.y)/2;
	point_mid.bend=false;
	point_mid.side=false;
	m_points.AddTail(point_fst);
	m_points.AddTail(point_mid);
	m_points.AddTail(point_lst);

	m_correct_edge=IsEdgeCorrect();
	ComputeHitRect();*/

	}
}

void CPNEdge::Draw(CDC* pDC)	
{	// draw this object on pDC
	ASSERT_VALID(this);

	if((!m_multi_point) && (!m_correct_edge))return;

	bool head=true;
	bool not_draw_left=((m_multi_point) && (!m_multi_correct_left));
	bool not_draw_right=((m_multi_point) && (!m_multi_correct_right));
	CEdgePoint edgepoint,arrowfrom,arrowto;
	POSITION pos=m_points.GetHeadPosition();
	while(pos!=NULL)
	{
		if(!head && edgepoint.bend)arrowfrom=edgepoint;
		edgepoint=m_points.GetNext(pos);
		if(head)
		{
			pDC->MoveTo(edgepoint.point);
			head=false;
		}
		else
		{	// draw lines, ignore non bend points
			if(edgepoint.bend)
			{
				bool draw=true;
				if(not_draw_left)
				{	// skip first line
					pDC->MoveTo(edgepoint.point);
					draw=false;
					not_draw_left=false;
				}
				if(not_draw_right)
				{
					draw=draw && (pos!=NULL);
				}
				if(draw)pDC->LineTo(edgepoint.point);
			}
		}
	}
	arrowto=edgepoint;
	if(!not_draw_right)
	{	// windows don't draw the last pixel in line, force it to draw
		edgepoint.point.Offset(1,1);
		pDC->LineTo(edgepoint.point);
	}

	if(!not_draw_right)
	{	// draw arrow
		CPoint points[3];
		points[0].x=arrowto.point.x;
		points[0].y=arrowto.point.y;
		// line vector
		double vec_line[2];
		vec_line[0]=points[0].x-arrowfrom.point.x;
		vec_line[1]=points[0].y-arrowfrom.point.y;
		// vector normal to line vector
		double vec_base[2];
		vec_base[0]=-vec_line[1];
		vec_base[1]=vec_line[0];
		// parameter for arrow height
		double t_line;
		t_line=ARROW_HEIGHT/sqrt(vec_line[0]*vec_line[0]+vec_line[1]*vec_line[1]);
		CPoint pt_base;
		pt_base.x=static_cast<int>(points[0].x-t_line*vec_line[0]+0.5);
		pt_base.y=static_cast<int>(points[0].y-t_line*vec_line[1]+0.5);
		// parameter for arrow width
		double t_base;
		t_base=ARROW_WIDTH/(2*sqrt(vec_base[0]*vec_base[0]+vec_base[1]*vec_base[1]));
		points[1].x=static_cast<int>(pt_base.x+t_base*vec_base[0]+0.5);
		points[1].y=static_cast<int>(pt_base.y+t_base*vec_base[1]+0.5);
		points[2].x=static_cast<int>(pt_base.x-t_base*vec_base[0]+0.5);
		points[2].y=static_cast<int>(pt_base.y-t_base*vec_base[1]+0.5);
		m_old_brush=(CBrush*)pDC->SelectStockObject(BLACK_BRUSH);
		pDC->Polygon(points,3); // draw it
		pDC->SelectObject(m_old_brush);
	}
	if(m_selected)
	{	// draw drag points marks
		not_draw_left=((m_multi_point) && (!m_multi_correct_left));
		pos=m_points.GetHeadPosition();
		m_points.GetNext(pos); // don't draw first point
		while((pos!=NULL) && (pos!=m_points.GetTailPosition())) // don't draw last point
		{
			edgepoint=m_points.GetNext(pos);
			bool draw=true;
			if(not_draw_left)
			{	// skip first mark
				draw=false;
				not_draw_left=false;
			}
			if(not_draw_right)
			{
				draw=draw && (pos!=m_points.GetTailPosition());
			}
			if(draw)DrawPoint(pDC,edgepoint);
		}
	}
}

CRect CPNEdge::MoveTo(const CPoint& point)
{	// move the center of this object to point
	ASSERT_VALID(this);

	CRect old_rect=GetRect();
	if(m_drag_point_pos)
	{
		CEdgePoint edgepoint;
		edgepoint=m_points.GetAt(m_drag_point_pos);
		m_points.GetAt(m_drag_point_pos).point=point;

		if(!edgepoint.bend)
		{	// insert new points
			CEdgePoint newedgepoint;
			newedgepoint.bend=false;
			newedgepoint.side=false;
			newedgepoint.point.x=(m_drag_point_left.x+point.x)/2;
			newedgepoint.point.y=(m_drag_point_left.y+point.y)/2;
			m_drag_point_pos_left=m_points.InsertBefore(m_drag_point_pos,newedgepoint);
			newedgepoint.point.x=(m_drag_point_right.x+point.x)/2;
			newedgepoint.point.y=(m_drag_point_right.y+point.y)/2;
			m_drag_point_pos_right=m_points.InsertAfter(m_drag_point_pos,newedgepoint);
			m_points.GetAt(m_drag_point_pos).bend=true;
			if(m_multi_point)
			{	// rozsireni ze 3 a vice bodu na jeste vice bodu, update border points
				POSITION pos=m_points.GetHeadPosition();
				m_points.GetNext(pos);
				m_points.GetNext(pos);
				m_multi_pos_left=pos;
				pos=m_points.GetTailPosition();
				m_points.GetPrev(pos);
				m_points.GetPrev(pos);
				m_multi_pos_right=pos;
			}
			else
			{	// rozsireni ze 2 hlavnich bodu na 3
				m_multi_pos_left=m_multi_pos_right=m_drag_point_pos;
				m_multi_point=true;
			}
		}
		else
		{	// zmena pozic bodu
			CPoint newpoint;
			newpoint.x=(m_drag_point_left.x+point.x)/2;
			newpoint.y=(m_drag_point_left.y+point.y)/2;
			m_points.GetAt(m_drag_point_pos_left).point=newpoint;
			newpoint.x=(m_drag_point_right.x+point.x)/2;
			newpoint.y=(m_drag_point_right.y+point.y)/2;
			m_points.GetAt(m_drag_point_pos_right).point=newpoint;
		}
		Recalculate(m_element_to);
		ComputeHitRect();
	}
	CRect invalid_rect;
	invalid_rect.UnionRect(old_rect,GetRect());
	return invalid_rect;
}

bool CPNEdge::HitTest(const CPoint& point)
{	// return whether point is in this object
	ASSERT_VALID(this);

	if((!m_multi_point) && (!m_correct_edge))return false;
	if(m_hit_rect.PtInRect(point)==0)return false;

	bool head=true;
	bool not_test_left=((m_multi_point) && (!m_multi_correct_left));
	bool not_test_right=((m_multi_point) && (!m_multi_correct_right));
	CEdgePoint edgepoint,edgefrom;
	POSITION pos=m_points.GetHeadPosition();
	while(pos!=NULL)
	{
		edgepoint=m_points.GetNext(pos);
		if(head)
		{
			edgefrom=edgepoint;
			head=false;
		}
		else
		{
			if(edgepoint.bend)
			{
				bool test=true;
				if(not_test_left)
				{	// skip first line
					test=false;
					not_test_left=false;
				}
				if(not_test_right)
				{
					test=(pos!=NULL);
				}
				if(test && (edgepoint.hit_rect.PtInRect(point)!=0))
				{	// point is in region of this line
					int dx=edgepoint.point.x-edgefrom.point.x;
					int dy=edgefrom.point.y-edgepoint.point.y;
					double norm=dy*point.x+dx*point.y+edgefrom.point.x*edgepoint.point.y-edgepoint.point.x*edgefrom.point.y;
					double line=sqrt(dx*dx+dy*dy);
					if(abs(static_cast<int>(norm/line))<=SEL_DST)return true;
				}
				edgefrom=edgepoint;
			}
		}
	}
	return false;
}

CRect CPNEdge::GetRect()
{	// return rect for redraw
	ASSERT_VALID(this);

	CRect rect;
	if(m_multi_point || ((!m_multi_point) && (m_correct_edge)))
	{
		CPoint arrow_point=m_points.GetTail().point;
		CRect arrow_rect; // include rect for arrow
		arrow_rect.SetRect(arrow_point.x-15,arrow_point.y-15,arrow_point.x+15,arrow_point.y+15);
		rect.UnionRect(m_hit_rect,arrow_rect);
	}
	else
		rect.SetRectEmpty();
	return rect;
}

void CPNEdge::ComputeHitRect()
{	// compute hit rect
	if((!m_multi_point) && (!m_correct_edge))return;
	CRect rect;
	rect.SetRectEmpty();
	bool head=true;
	bool not_compute_left=((m_multi_point) && (!m_multi_correct_left));
	bool not_compute_right=((m_multi_point) && (!m_multi_correct_right));
	CEdgePoint edgepoint,edgefrom;
	POSITION oldpos;
	POSITION pos=m_points.GetHeadPosition();
	while(pos!=NULL)
	{
		oldpos=pos;
		edgepoint=m_points.GetNext(pos);
		if(head)
		{
			edgefrom=edgepoint;
			head=false;
		}
		else
		{
			if(edgepoint.bend)
			{
				bool compute=true;
				if(not_compute_left)
				{	// skip first line
					compute=false;
					not_compute_left=false;
				}
				if(not_compute_right)
				{
					compute=compute && (pos!=NULL);
				}
				if(compute)
				{
					edgepoint.hit_rect.SetRect(edgefrom.point.x,edgefrom.point.y,edgepoint.point.x,edgepoint.point.y);
					edgepoint.hit_rect.NormalizeRect();
					edgepoint.hit_rect.InflateRect(SEL_DST,SEL_DST);
					m_hit_rect.UnionRect(rect,edgepoint.hit_rect);
					rect=m_hit_rect;
					m_points.SetAt(oldpos,edgepoint);
				}
				edgefrom=edgepoint;
			}
		}
	}
	m_hit_rect.InflateRect(EDGE_SEL_RECT_BIG/2,EDGE_SEL_RECT_BIG/2);
}

void CPNEdge::DrawPoint(CDC* pDC,CEdgePoint& edgepoint)
{
	ASSERT_VALID(this);
	ASSERT(pDC);
	m_old_pen=(CPen*)pDC->SelectObject(&m_sel_pen);
	m_old_brush=(CBrush*)pDC->SelectObject(&m_sel_brush);
	if(edgepoint.bend)
		pDC->Rectangle(edgepoint.point.x-EDGE_SEL_RECT_BIG/2,edgepoint.point.y-EDGE_SEL_RECT_BIG/2,edgepoint.point.x+EDGE_SEL_RECT_BIG/2,edgepoint.point.y+EDGE_SEL_RECT_BIG/2);
	else
		pDC->Rectangle(edgepoint.point.x-EDGE_SEL_RECT_SML/2,edgepoint.point.y-EDGE_SEL_RECT_SML/2,edgepoint.point.x+EDGE_SEL_RECT_SML/2,edgepoint.point.y+EDGE_SEL_RECT_SML/2);
	pDC->SelectObject(m_old_brush);
	pDC->SelectObject(m_old_pen);
}

bool CPNEdge::IsDragging(const CPoint& point)
{	// return if point is on some drag point
	ASSERT_VALID(this);

	CRect rect;
	CEdgePoint edgepoint;
	POSITION oldpos;
	POSITION pos=m_points.GetHeadPosition();
	m_points.GetNext(pos); // can't drag first point
	while((pos!=NULL) && (pos!=m_points.GetTailPosition())) // can't drag last point
	{
		oldpos=pos;
		edgepoint=m_points.GetNext(pos);
		rect.SetRect(edgepoint.point.x-EDGE_SEL_RECT_BIG/2,edgepoint.point.y-EDGE_SEL_RECT_BIG/2,edgepoint.point.x+EDGE_SEL_RECT_BIG/2,edgepoint.point.y+EDGE_SEL_RECT_BIG/2);
		if(rect.PtInRect(point)!=0)
		{	// taha se za tento ctverecek
			m_position=edgepoint.point;
			m_drag_point_pos=oldpos;
			POSITION pom;
			if(edgepoint.bend)
			{	// bod je jiz velky
				m_drag_point_pos_right=pos;
				m_drag_point_pos_left=oldpos;
				m_points.GetPrev(m_drag_point_pos_left);
				pom=m_drag_point_pos_left;
				m_points.GetPrev(pom);
				m_drag_point_left=m_points.GetAt(pom).point;
				pom=m_drag_point_pos_right;
				m_points.GetNext(pom);
				m_drag_point_right=m_points.GetAt(pom).point;
			}
			else
			{	// bod je maly, bude se rozdelovat
				pom=m_drag_point_pos;
				m_points.GetPrev(pom);
				m_drag_point_left=m_points.GetAt(pom).point;
				pom=m_drag_point_pos;
				m_points.GetNext(pom);
				m_drag_point_right=m_points.GetAt(pom).point;
				m_drag_point_pos_right=m_drag_point_pos_left=NULL;
			}
			return true;
		}
	}
	m_position=CPoint(0,0);
	m_drag_point_pos=NULL;
	return false;
}

bool CPNEdge::CanDeletePoint()
{	// 
	if(m_drag_point_pos==NULL)return false;
	return m_points.GetAt(m_drag_point_pos).bend;
}

bool CPNEdge::DeletePoint()
{	// 
	ASSERT_VALID(this);

	if(m_drag_point_pos==NULL)return false;
	CEdgePoint edgepoint;
	edgepoint=m_points.GetAt(m_drag_point_pos);
	if(!edgepoint.bend)return false;
	if(m_points.GetCount()==5)
	{	// 
		DeleteAllPoints();
		return true;
	}
	// delete surround points
	POSITION pos;
	pos=m_drag_point_pos;
	m_points.GetNext(pos);
	m_points.RemoveAt(pos);
	pos=m_drag_point_pos;
	m_points.GetPrev(pos);
	m_points.RemoveAt(pos);
	// update old point from big to small
	edgepoint.bend=false;
	POSITION left_pos,right_pos;
	left_pos=m_drag_point_pos;
	m_points.GetPrev(left_pos);
	CEdgePoint left_point;
	left_point=m_points.GetAt(left_pos);
	right_pos=m_drag_point_pos;
	m_points.GetNext(right_pos);
	CEdgePoint right_point;
	right_point=m_points.GetAt(right_pos);
	if(left_point.side)
	{	// calculate intersection point for side point
		left_point.point=m_element_from->CalculateIntersection(right_point.point);
		m_points.SetAt(left_pos,left_point);
	}
	if(right_point.side)
	{	// calculate intersection point for side point
		right_point.point=m_element_to->CalculateIntersection(left_point.point);
		m_points.SetAt(right_pos,right_point);
	}
	edgepoint.point.x=(left_point.point.x+right_point.point.x)/2;
	edgepoint.point.y=(left_point.point.y+right_point.point.y)/2;
	m_points.SetAt(m_drag_point_pos,edgepoint);
	m_drag_point_pos=NULL;

	// update pointers to border points
	pos=m_points.GetHeadPosition();
	m_points.GetNext(pos);
	m_points.GetNext(pos);
	m_multi_pos_left=pos;
	pos=m_points.GetTailPosition();
	m_points.GetPrev(pos);
	m_points.GetPrev(pos);
	m_multi_pos_right=pos;

	// 
	m_correct_edge=IsEdgeCorrect();
	Recalculate(m_element_to);
	ComputeHitRect();
	return true;
}

void CPNEdge::DeleteAllPoints()
{	// straighten the line
	ASSERT_VALID(this);

	m_points.RemoveAll();
	m_multi_point=false;
	m_multi_correct_left=m_multi_correct_right=true;
	m_multi_pos_left=m_multi_pos_right=NULL;
	CEdgePoint point_fst,point_mid,point_lst;
	point_fst.point=m_element_from->GetPosition();
	point_fst.bend=true;
	point_fst.side=true;
	point_lst.point=m_element_to->GetPosition();
	point_lst.bend=true;
	point_lst.side=true;
	// calculate intersection points
	point_fst.point=m_element_from->CalculateIntersection(point_lst.point);
	point_lst.point=m_element_to->CalculateIntersection(point_fst.point);
	point_mid.point.x=(point_fst.point.x+point_lst.point.x)/2;
	point_mid.point.y=(point_fst.point.y+point_lst.point.y)/2;
	point_mid.bend=false;
	point_mid.side=false;
	m_points.AddTail(point_fst);
	m_points.AddTail(point_mid);
	m_points.AddTail(point_lst);

	m_correct_edge=IsEdgeCorrect();
	Recalculate(m_element_to);
	ComputeHitRect();
}

bool CPNEdge::Compare(CPNEdge* pedge)
{	// compare two edges
	ASSERT_VALID(this);

	if(pedge==NULL)return false;
	return (pedge->m_element_from==m_element_from)&&(pedge->m_element_to==m_element_to);
}

bool CPNEdge::IsCorresponding(CPNElement* pElement)
{	// return whether this edge corresponds to pelement
	ASSERT_VALID(this);

	return ((m_element_from==pElement) || (m_element_to==pElement));
}

bool CPNEdge::IsCorresponding(CPNElement* pFrom,CPNElement* pTo)
{	// return whether this edge is between pFrom and pTo
	ASSERT_VALID(this);

	return ((m_element_from==pFrom) && (m_element_to==pTo));
}

bool CPNEdge::IsEdgeCorrect()
{	// test if the edge is correct
	if(m_multi_point)
	{	// mame minimalne tri hlavni body
		m_multi_correct_left=!m_element_from->HitTest(m_points.GetAt(m_multi_pos_left).point);
		m_multi_correct_right=!m_element_to->HitTest(m_points.GetAt(m_multi_pos_right).point);
		return (m_multi_correct_left) && (m_multi_correct_right);
	}
	else
	{	// pouze dva hlavni body
		CPoint pt_to,pt_from;
		pt_from=m_element_from->GetPosition();
		pt_to=m_element_to->GetPosition();
		CRgn region;
		region.CreateEllipticRgn(pt_from.x-CND_RDS-1,pt_from.y-CND_RDS-1,pt_from.x+CND_RDS+1,pt_from.y+CND_RDS+1);
		CRect rect;
		rect.SetRect(pt_to.x-EVN_RDS-2,pt_to.y-EVN_RDS-2,pt_to.x+EVN_RDS+2,pt_to.y+EVN_RDS+2);
		// test zda jsou objekty prilis blizko sebe, jestli nejaka jejich cast zasahuje do druheho objektu
//		BOOL correct=region.RectInRegion(&rect)==0;
//		region.DeleteObject();
//		return correct;
		return region.RectInRegion(&rect)==0;
	}
}

void CPNEdge::FirstRecalculate()
{	// recalculate for only prepared edges
	ASSERT_VALID(this);

	// construct edge
	m_points.RemoveAll();
	m_multi_correct_left=m_multi_correct_right=true;
	m_correct_edge=true;
	CEdgePoint point_fst,point_mid,point_lst;
	point_fst.point=m_element_from->GetPosition();
	point_fst.bend=true;
	point_fst.side=true;
	point_lst.point=m_element_to->GetPosition();
	point_lst.bend=true;
	point_lst.side=true;
	// calculate intersection points
	point_fst.point=m_element_from->CalculateIntersection(point_lst.point);
	point_lst.point=m_element_to->CalculateIntersection(point_fst.point);
	point_mid.point.x=(point_fst.point.x+point_lst.point.x)/2;
	point_mid.point.y=(point_fst.point.y+point_lst.point.y)/2;
	point_mid.bend=false;
	point_mid.side=false;
	m_points.AddTail(point_fst);
	m_points.AddTail(point_mid);
	m_points.AddTail(point_lst);

	m_correct_edge=IsEdgeCorrect();
	ComputeHitRect();
}

bool CPNEdge::Recalculate(const CPNElement* pElement/*=NULL*/)
{	// recalculate intersections
	ASSERT_VALID(this);

	// test if it is our corresponding element
	if((pElement!=NULL) && (pElement!=m_element_to) && (pElement!=m_element_from))return false;
	POSITION side_pos,mid_pos,end_pos;
	side_pos=m_points.GetHeadPosition();
	mid_pos=side_pos;
	m_points.GetNext(mid_pos);
	end_pos=mid_pos;
	m_points.GetNext(end_pos);
	CPoint tmp_point;
	if(m_multi_point)
		tmp_point=m_points.GetAt(end_pos).point;
	else
		tmp_point=m_element_to->GetPosition();
	m_points.GetAt(side_pos).point=m_element_from->CalculateIntersection(tmp_point);
	CPoint fst_point;
	fst_point=m_points.GetAt(side_pos).point;

//	TRACE("P1x:%d P1y:%d\n",m_points.GetAt(side_pos).point.x,m_points.GetAt(side_pos).point.y);

	if(m_multi_point)
	{
		m_points.GetAt(mid_pos).point.x=(m_points.GetAt(side_pos).point.x+tmp_point.x)/2;
		m_points.GetAt(mid_pos).point.y=(m_points.GetAt(side_pos).point.y+tmp_point.y)/2;
	}

//	TRACE("PM1x:%d PM1y:%d\n",m_points.GetAt(mid_pos).point.x,m_points.GetAt(mid_pos).point.y);

	side_pos=m_points.GetTailPosition();
	mid_pos=side_pos;
	m_points.GetPrev(mid_pos);
	end_pos=mid_pos;
	m_points.GetPrev(end_pos);
	if(m_multi_point)
		tmp_point=m_points.GetAt(end_pos).point;
	else
		tmp_point=m_element_from->GetPosition();
	m_points.GetAt(side_pos).point=m_element_to->CalculateIntersection(tmp_point);

//	TRACE("P2x:%d P2y:%d\n",m_points.GetAt(side_pos).point.x,m_points.GetAt(side_pos).point.y);

	if(fst_point==m_points.GetAt(side_pos).point)
	{
		m_points.GetAt(mid_pos).point=fst_point;
		m_correct_edge=false;
	}
	else
	{
		m_points.GetAt(mid_pos).point.x=(m_points.GetAt(side_pos).point.x+tmp_point.x)/2;
		m_points.GetAt(mid_pos).point.y=(m_points.GetAt(side_pos).point.y+tmp_point.y)/2;
		m_correct_edge=IsEdgeCorrect();
	}

//	TRACE("PM2x:%d PM2y:%d\n",m_points.GetAt(mid_pos).point.x,m_points.GetAt(mid_pos).point.y);

	ComputeHitRect();
	return true;
}

CRect CPNEdge::MoveByVector(CSize move)
{
	CRect invalid_rect,old_rect;
	old_rect=GetRect();
	POSITION pos=m_points.GetHeadPosition();
	while(pos!=NULL)
	{
		m_points.GetAt(pos).point=m_points.GetAt(pos).point+move;
		m_points.GetNext(pos);
	}
	ComputeHitRect();
	invalid_rect.UnionRect(old_rect,GetRect());
	return invalid_rect;
}

void CPNEdge::UpdateCorrespondingSets()
{	// compute preset and postset for corresponding elements
	ASSERT_VALID(this);
	CPNEvent* pEvent;
	CPNCondition* pCondition;
	pEvent=dynamic_cast<CPNEvent*>(m_element_from);
	pCondition=dynamic_cast<CPNCondition*>(m_element_to);
	if((pEvent!=NULL) && (pCondition!=NULL))
	{	// edge from event to condition
		pEvent->GetPostSet().Add(pCondition);
		pCondition->GetPreSet().Add(pEvent);
	}
	pEvent=dynamic_cast<CPNEvent*>(m_element_to);
	pCondition=dynamic_cast<CPNCondition*>(m_element_from);
	if((pEvent!=NULL) && (pCondition!=NULL))
	{	// edge from condition to event
		pCondition->GetPostSet().Add(pEvent);
		pEvent->GetPreSet().Add(pCondition);
	}
}

void CPNEdge::ComputeTokenPath()
{
	POSITION pos;
	int path_len=0;
	bool head=true;
	CEdgePoint edgepoint,edgefrom;
	pos=m_points.GetHeadPosition();
	while(pos!=NULL)
	{	// compute path length
		edgepoint=m_points.GetNext(pos);

/*		CString str;
		str.Format("Souradnice bodu: %d %d",edgepoint.point.x,edgepoint.point.y);
		MessageBox(NULL,str,"",MB_OK);*/

		if(head)
		{
			edgefrom=edgepoint;
			head=false;
		}
		else
		{
			if(edgepoint.bend)
			{
				int len=(edgepoint.point.x-edgefrom.point.x)*(edgepoint.point.x-edgefrom.point.x)+(edgepoint.point.y-edgefrom.point.y)*(edgepoint.point.y-edgefrom.point.y);
				if(len!=0)len=int(sqrt(len));
				path_len+=len;
				edgefrom=edgepoint;
			}
		}
	}
	int path_part=path_len/(TOKEN_PATH_POINTS+1);
	int line_coef_x;
	int line_coef_y;
	int line_coef_u;
	int line_coef_v;
	double line_coef_t;
	int distance=0;
	head=true;
	int index=0;
	pos=m_points.GetHeadPosition();
	while(pos!=NULL)
	{	// compute path points
		edgepoint=m_points.GetNext(pos);
		if(head)
		{
			edgefrom=edgepoint;
			head=false;
		}
		else
		{
			if(edgepoint.bend)
			{
				int len=(edgepoint.point.x-edgefrom.point.x)*(edgepoint.point.x-edgefrom.point.x)+(edgepoint.point.y-edgefrom.point.y)*(edgepoint.point.y-edgefrom.point.y);
				if(len!=0)
				{
					len=int(sqrt(len));
					int aux_path_len=(index+1)*path_part;
					int aux_distance=distance+len;
					line_coef_x=edgefrom.point.x;
					line_coef_y=edgefrom.point.y;
					line_coef_u=edgepoint.point.x-edgefrom.point.x;
					line_coef_v=edgepoint.point.y-edgefrom.point.y;
					while((aux_distance>aux_path_len) && ((path_len-aux_path_len)>=path_part) && (index<TOKEN_PATH_POINTS))
					{
						int lenght_rest=aux_path_len-distance;
						line_coef_t=(double)lenght_rest/(double)len;
						m_token_path[index].x=long(line_coef_x+line_coef_t*line_coef_u);
						m_token_path[index].y=long(line_coef_y+line_coef_t*line_coef_v);
						index++;
						aux_path_len=(index+1)*path_part;
					}

/*		CString str;
		str.Format("Posledni index: %d",index);
		MessageBox(NULL,str,"",MB_OK);*/

					distance=aux_distance;
				}
				edgefrom=edgepoint;
			}
		}
	}

/*	for(int i=0;i<TOKEN_PATH_POINTS;i++)
	{
		CPoint point=m_token_path[i];
		CString str;
		str.Format("Nulovy bod v indexu: %d",i);
		if((point.x==0)&&(point.y==0))MessageBox(NULL,str,"",MB_OK);
	}*/
}

CPoint CPNEdge::GetTokenPathPoint(int index)
{
	if((index>=0) && (index<TOKEN_PATH_POINTS))
	{
		return m_token_path[index];
	}
	else
	{	// error out of bounds
		ASSERT(FALSE);
		return CPoint(0,0);
	}
}

void CPNEdge::UpdatePropertiesDlg()
{
	ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
	CMainFrame* frame=(CMainFrame*)AfxGetApp()->m_pMainWnd;

	frame->ShowPropertiesDlg(CMainFrame::PROPDLG_HIDE,this);
//	frame->ShowPropertiesDlg(CMainFrame::PROPDLG_DESKTOP,this);
}

////////////////////////////////////////////////////////////////////////////
// CPNPolygon
IMPLEMENT_SERIAL(CPNPolygon,CPNElement,1|VERSIONABLE_SCHEMA)

CPNPolygon::CPNPolygon()
{
	m_line_color=LINE_COLOR;
	m_line_style=LINE_STYLE_SOLID;
	m_line_width=1;
	m_fill_color=FILL_COLOR;
	m_fill_mode=FILL_MODE_EMPTY;
	m_hit_rect.SetRectEmpty();
	m_poly_points=NULL;
	m_poly_count=0;
	m_drag_point_pos=NULL;
}

CPNPolygon::CPNPolygon(CPoint point) : CPNElement(point)
{
	// init
	m_line_color=LINE_COLOR;
	m_line_style=LINE_STYLE_SOLID;
	m_line_width=1;
	m_fill_color=FILL_COLOR;
	m_fill_mode=FILL_MODE_EMPTY;
	m_hit_rect.SetRectEmpty();
	m_poly_points=NULL;
	m_poly_count=0;
	m_drag_point_pos=NULL;

	// construct polygon
	CEdgePoint edgepoint;
	edgepoint.side=true;
	edgepoint.bend=true;
	edgepoint.point=point; // top left
	m_points.AddTail(edgepoint);
	edgepoint.side=false;
	edgepoint.bend=false;
	edgepoint.point=CPoint(point.x,point.y+POLYGON_HEIGHT/2);
	m_points.AddTail(edgepoint);
	edgepoint.bend=true;
	edgepoint.point=CPoint(point.x,point.y+POLYGON_HEIGHT);
	m_points.AddTail(edgepoint);
	edgepoint.bend=false;
	edgepoint.point=CPoint(point.x+POLYGON_WIDTH/2,point.y+POLYGON_HEIGHT);
	m_points.AddTail(edgepoint);
	edgepoint.bend=true;
	edgepoint.point=CPoint(point.x+POLYGON_WIDTH,point.y+POLYGON_HEIGHT);
	m_points.AddTail(edgepoint);
	edgepoint.bend=false;
	edgepoint.point=CPoint(point.x+POLYGON_WIDTH,point.y+POLYGON_HEIGHT/2);
	m_points.AddTail(edgepoint);
	edgepoint.bend=true;
	edgepoint.point=CPoint(point.x+POLYGON_WIDTH,point.y);
	m_points.AddTail(edgepoint);
	edgepoint.bend=false;
	edgepoint.point=CPoint(point.x+POLYGON_WIDTH/2,point.y);
	m_points.AddTail(edgepoint);
	edgepoint.side=true;
	edgepoint.bend=true;
	edgepoint.point=point; // uzavreni
	m_points.AddTail(edgepoint);

	m_poly_count=5;
	m_poly_points=new POINT[m_poly_count];
	m_drag_point_pos=NULL;

	ComputeHitRect();
}

CPNPolygon::~CPNPolygon()
{
	if(m_poly_points!=NULL)delete m_poly_points;
}

void CPNPolygon::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);

	CPNElement::Serialize(ar);
	if(ar.IsStoring())
	{
		ar << m_line_color;
		ar << m_line_style;
		ar << m_line_width;
		ar << m_fill_color;
		ar << m_fill_mode;
		ar << m_hit_rect;

		m_points.Serialize(ar);
	}
	else
	{
		ar >> m_line_color;
		ar >> m_line_style;
		ar >> m_line_width;
		ar >> m_fill_color;
		ar >> m_fill_mode;
		ar >> m_hit_rect;

		m_points.Serialize(ar);

		// update polygon points array
		int count=m_points.GetCount();
		count=count/2+1;
		m_poly_count=count;
		if(m_poly_points!=NULL)delete[] m_poly_points;
		m_poly_points=new POINT[m_poly_count];
	}
}

void CPNPolygon::Draw(CDC* pDC)
{	// draw this object on pDC
	ASSERT_VALID(this);

	int count=0;
	bool head=true;
	CEdgePoint edgepoint;
	POSITION pos=m_points.GetHeadPosition();
	while(pos!=NULL)
	{
		edgepoint=m_points.GetNext(pos);
		if(head)
		{
			m_poly_points[count]=edgepoint.point;
			count++;
			head=false;
		}
		else
		{	// draw lines, ignore non bend points
			if(edgepoint.bend)
			{
				m_poly_points[count]=edgepoint.point;
				count++;
			}
		}
	}
	CPen pen(m_line_style,m_line_width,m_line_color);
	m_old_pen=pDC->SelectObject(&pen);
	CBrush brush(m_fill_color);
	if(m_fill_mode==FILL_MODE_EMPTY)
		m_old_brush=(CBrush*)pDC->SelectStockObject(NULL_BRUSH);
	else
		m_old_brush=pDC->SelectObject(&brush);
	pDC->Polygon(m_poly_points,count);
	pDC->SelectObject(m_old_brush);
	pDC->SelectObject(m_old_pen);
	if(m_selected)
	{	// draw drag points marks
		pos=m_points.GetHeadPosition();
		m_points.GetNext(pos); // don't draw first point
		while(pos!=NULL)
		{
			edgepoint=m_points.GetNext(pos);
			DrawPoint(pDC,edgepoint);
		}
	}
}

CRect CPNPolygon::MoveTo(const CPoint& point)
{	// move the center of this object to point
	ASSERT_VALID(this);

	CRect old_rect=GetRect();
	if(m_drag_point_pos)
	{	// move one point
		CEdgePoint edgepoint;
		edgepoint=m_points.GetAt(m_drag_point_pos);
		m_points.GetAt(m_drag_point_pos).point=point;
		m_position=point;

		if(!edgepoint.bend)
		{	// insert new points
			CEdgePoint newedgepoint;
			newedgepoint.bend=false;
			newedgepoint.side=false;
			newedgepoint.point.x=(m_drag_point_left.x+point.x)/2;
			newedgepoint.point.y=(m_drag_point_left.y+point.y)/2;
			m_drag_point_pos_left=m_points.InsertBefore(m_drag_point_pos,newedgepoint);
			newedgepoint.point.x=(m_drag_point_right.x+point.x)/2;
			newedgepoint.point.y=(m_drag_point_right.y+point.y)/2;
			m_drag_point_pos_right=m_points.InsertAfter(m_drag_point_pos,newedgepoint);
			m_points.GetAt(m_drag_point_pos).bend=true;
			POSITION pos=m_points.GetHeadPosition();
			m_points.GetNext(pos);
			m_points.GetNext(pos);
			pos=m_points.GetTailPosition();
			m_points.GetPrev(pos);
			m_points.GetPrev(pos);
			// update polygon points array for redraw
			int count=m_points.GetCount();
			count=count/2+1;
			if(m_poly_count<count)
			{
				m_poly_count=count+2;
				if(m_poly_points)delete[] m_poly_points;
				m_poly_points=new POINT[m_poly_count];
			}
		}
		else
		{	// zmena pozic bodu
			CPoint newpoint;
			newpoint.x=(m_drag_point_left.x+point.x)/2;
			newpoint.y=(m_drag_point_left.y+point.y)/2;
			m_points.GetAt(m_drag_point_pos_left).point=newpoint;
			newpoint.x=(m_drag_point_right.x+point.x)/2;
			newpoint.y=(m_drag_point_right.y+point.y)/2;
			m_points.GetAt(m_drag_point_pos_right).point=newpoint;
			if(edgepoint.side)m_points.GetHead().point=point; // prvni bod se prekryva s poslednim
		}
	}
	else
	{	// move entire object
		CSize move_vector;
		move_vector.cx=point.x-m_position.x;
		move_vector.cy=point.y-m_position.y;
		POSITION pos=m_points.GetHeadPosition();
		while(pos!=NULL)
		{
			m_points.GetAt(pos).point+=move_vector;
			m_points.GetNext(pos);
		}
		m_position=m_points.GetAt(m_points.GetHeadPosition()).point;
	}
	ComputeHitRect();
	CRect invalid_rect;
	invalid_rect.UnionRect(old_rect,GetRect());
	return invalid_rect;
}

bool CPNPolygon::HitTest(const CPoint& point)
{	// return whether point is in this object
	ASSERT_VALID(this);

	if(m_hit_rect.PtInRect(point)==0)return false;

	bool head=true;
	CEdgePoint edgepoint,edgefrom;
	POSITION pos=m_points.GetHeadPosition();
	while(pos!=NULL)
	{
		edgepoint=m_points.GetNext(pos);
		if(head)
		{
			edgefrom=edgepoint;
			head=false;
		}
		else
		{
			if(edgepoint.bend)
			{
				if(edgepoint.hit_rect.PtInRect(point)!=0)
				{	// point is in region of this line
					int dx=edgepoint.point.x-edgefrom.point.x;
					int dy=edgefrom.point.y-edgepoint.point.y;
					double norm=dy*point.x+dx*point.y+edgefrom.point.x*edgepoint.point.y-edgepoint.point.x*edgefrom.point.y;
					double line=sqrt(dx*dx+dy*dy);
					if(abs(static_cast<int>(norm/line))<=SEL_DST)return true;
				}
				edgefrom=edgepoint;
			}
		}
	}
	return false;
}

CRect CPNPolygon::GetRect()
{	// return rect for redraw
	ASSERT_VALID(this);

	m_hit_rect.InflateRect(m_line_width/2+1,m_line_width/2+1);
	return m_hit_rect;
}

bool CPNPolygon::IsDragging(const CPoint& point)
{	// point is on some drag point
	ASSERT_VALID(this);

	CRect rect;
	CEdgePoint edgepoint;
	POSITION oldpos;
	POSITION pos=m_points.GetHeadPosition();
	m_position=m_points.GetAt(pos).point;
	m_points.GetNext(pos); // can't drag first point
	while(pos!=NULL)
	{
		oldpos=pos;
		edgepoint=m_points.GetNext(pos);
		rect.SetRect(edgepoint.point.x-EDGE_SEL_RECT_SML,edgepoint.point.y-EDGE_SEL_RECT_SML,edgepoint.point.x+EDGE_SEL_RECT_SML,edgepoint.point.y+EDGE_SEL_RECT_SML);
		if(rect.PtInRect(point)!=0)
		{	// taha se za tento ctverecek
			m_position=edgepoint.point;
			m_drag_point_pos=oldpos;
			POSITION pom;
			if(edgepoint.bend)
			{	// bod je jiz velky, ziskani poloh okolnich bodu
				if(edgepoint.side)
				{	// koncovy bod, protoze za prvni se neda tahat
					m_drag_point_pos_left=oldpos;
					m_points.GetPrev(m_drag_point_pos_left);
					pom=m_drag_point_pos_left;
					m_points.GetPrev(pom);
					m_drag_point_left=m_points.GetAt(pom).point;
					m_drag_point_pos_right=m_points.GetHeadPosition();
					m_points.GetNext(m_drag_point_pos_right);
					pom=m_drag_point_pos_right;
					m_points.GetNext(pom);
					m_drag_point_right=m_points.GetAt(pom).point;
				}
				else
				{
					m_drag_point_pos_left=oldpos;
					m_points.GetPrev(m_drag_point_pos_left);
					pom=m_drag_point_pos_left;
					m_points.GetPrev(pom);
					m_drag_point_left=m_points.GetAt(pom).point;
					m_drag_point_pos_right=pos;
					pom=m_drag_point_pos_right;
					m_points.GetNext(pom);
					m_drag_point_right=m_points.GetAt(pom).point;
				}
			}
			else
			{	// bod je maly, bude se rozdelovat
				pom=m_drag_point_pos;
				m_points.GetPrev(pom);
				m_drag_point_left=m_points.GetAt(pom).point;
				pom=m_drag_point_pos;
				m_points.GetNext(pom);
				m_drag_point_right=m_points.GetAt(pom).point;
				m_drag_point_pos_right=m_drag_point_pos_left=NULL;
			}
			return true;
		}
	}
	m_drag_point_pos=NULL;
	return false;
}

bool CPNPolygon::CanDeletePoint()
{	// 
	if(m_drag_point_pos==NULL)return false;
	if(m_points.GetCount()<=7)return false; // minimum je trojuhelnik
	return m_points.GetAt(m_drag_point_pos).bend;
}

bool CPNPolygon::DeletePoint()
{	// 
	ASSERT_VALID(this);

	if(m_drag_point_pos==NULL)return false;
	CEdgePoint edgepoint;
	edgepoint=m_points.GetAt(m_drag_point_pos);
	if(!edgepoint.bend)return false;
	if(edgepoint.side)
	{	
		// odstrani se zacatek seznamu
		m_points.RemoveHead();
		m_points.RemoveHead();
		m_points.GetHead().side=true;
		m_points.GetTail().point=m_points.GetHead().point;
		CPoint right_point=m_points.GetAt(m_drag_point_pos).point;
		POSITION pos,mid_pos;
		mid_pos=m_drag_point_pos;
		m_points.GetPrev(mid_pos);
		pos=mid_pos;
		m_points.GetPrev(pos);
		CPoint left_point=m_points.GetAt(pos).point;
		CPoint mid_point;
		mid_point.x=(left_point.x+right_point.x)/2;
		mid_point.y=(left_point.y+right_point.y)/2;
		m_points.GetAt(mid_pos).point=mid_point;
	}
	else
	{	// delete surround points
		POSITION pos;
		pos=m_drag_point_pos;
		m_points.GetPrev(pos);
		m_points.RemoveAt(pos);
		pos=m_drag_point_pos;
		m_points.GetNext(pos);
		m_points.RemoveAt(pos);
		// update old point from big to small
		edgepoint.bend=false;
		pos=m_drag_point_pos;
		m_points.GetPrev(pos);
		CPoint left_point=m_points.GetAt(pos).point;
		pos=m_drag_point_pos;
		m_points.GetNext(pos);
		CPoint right_point=m_points.GetAt(pos).point;
		edgepoint.point.x=(left_point.x+right_point.x)/2;
		edgepoint.point.y=(left_point.y+right_point.y)/2;
		m_points.SetAt(m_drag_point_pos,edgepoint);
	}
	ComputeHitRect();
	return true;
}

void CPNPolygon::DrawPoint(CDC* pDC,CEdgePoint& edgepoint)
{
	ASSERT_VALID(this);
	ASSERT(pDC);
	m_old_pen=(CPen*)pDC->SelectObject(&m_sel_pen);
	m_old_brush=(CBrush*)pDC->SelectObject(&m_sel_brush);
	if(edgepoint.bend)
		pDC->Rectangle(edgepoint.point.x-EDGE_SEL_RECT_BIG/2,edgepoint.point.y-EDGE_SEL_RECT_BIG/2,edgepoint.point.x+EDGE_SEL_RECT_BIG/2,edgepoint.point.y+EDGE_SEL_RECT_BIG/2);
	else
		pDC->Rectangle(edgepoint.point.x-EDGE_SEL_RECT_SML/2,edgepoint.point.y-EDGE_SEL_RECT_SML/2,edgepoint.point.x+EDGE_SEL_RECT_SML/2,edgepoint.point.y+EDGE_SEL_RECT_SML/2);
	pDC->SelectObject(m_old_brush);
	pDC->SelectObject(m_old_pen);
}

void CPNPolygon::ComputeHitRect()
{	// compute hit rect
	CRect rect;
	rect.SetRectEmpty();
	bool head=true;
	CEdgePoint edgepoint,edgefrom;
	POSITION oldpos;
	POSITION pos=m_points.GetHeadPosition();
	while(pos!=NULL)
	{
		oldpos=pos;
		edgepoint=m_points.GetNext(pos);
		if(head)
		{
			edgefrom=edgepoint;
			head=false;
		}
		else
		{
			if(edgepoint.bend)
			{
				edgepoint.hit_rect.SetRect(edgefrom.point.x,edgefrom.point.y,edgepoint.point.x,edgepoint.point.y);
				edgepoint.hit_rect.NormalizeRect();
				edgepoint.hit_rect.InflateRect(SEL_DST,SEL_DST);
				m_hit_rect.UnionRect(rect,edgepoint.hit_rect);
				rect=m_hit_rect;
				m_points.SetAt(oldpos,edgepoint);
				edgefrom=edgepoint;
			}
		}
	}
	m_hit_rect.InflateRect(EDGE_SEL_RECT_BIG/2,EDGE_SEL_RECT_BIG/2);
}

CRect CPNPolygon::MoveByVector(CSize move)
{
	CRect invalid_rect,old_rect;
	old_rect=GetRect();
	POSITION pos=m_points.GetHeadPosition();
	while(pos!=NULL)
	{
		m_points.GetAt(pos).point=m_points.GetAt(pos).point+move;
		m_points.GetNext(pos);
	}
	ComputeHitRect();
	invalid_rect.UnionRect(old_rect,GetRect());
	return invalid_rect;
}

void CPNPolygon::UpdatePropertiesDlg()
{
	ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
	CMainFrame* frame=(CMainFrame*)AfxGetApp()->m_pMainWnd;

	frame->ShowPropertiesDlg(CMainFrame::PROPDLG_POLYGON,this);
}

////////////////////////////////////////////////////////////////////////////
// CPNCase
IMPLEMENT_SERIAL(CPNCase,CPNCondition,1|VERSIONABLE_SCHEMA)

CBrush CPNCase::m_high_brush(HIGHLIGHT_BRUSH);
CBrush CPNCase::m_reach_brush(REACH_BRUSH);
CBrush CPNCase::m_reach_init_brush(REACH_INIT_BRUSH);

CPNCase::CPNCase()
{
	m_show_reachable=false;
	m_reachable=FALSE;
	m_reachable_init=FALSE;
	m_highlight=false;
}

CPNCase::CPNCase(const CSetConditions& caseset) : CPNCondition(CPoint(0,0),NULL)
{
	m_show_reachable=false;
	m_reachable=FALSE;
	m_reachable_init=FALSE;
	m_highlight=false;
	m_case=caseset;
	m_case_str=m_case.Format();
}

CPNCase::~CPNCase()
{
}

void CPNCase::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);

	CPNCondition::Serialize(ar);
	m_case.Serialize(ar);
	m_label.Serialize(ar);
	if(ar.IsStoring())
	{
		ar << m_reachable;
		ar << m_reachable_init;
	}
	else
	{
		ar >> m_reachable;
		ar >> m_reachable_init;
		m_case_str=m_case.Format();
	}
}

void CPNCase::Draw(CDC* pDC)
{	// draw this object on pDC
	ASSERT_VALID(this);

	CPNCondition::Draw(pDC);
	bool bDraw=false;
	if(m_highlight)
	{	// highlight has priority
		bDraw=true;
		m_old_brush=pDC->SelectObject(&m_high_brush);
	}
	else
	{
		if(m_show_reachable)
		{	// zobrazovat dopredne dosazitelne zvyrazneni
			if(m_reachable_init)
			{	// vychozi pripad
				bDraw=true;
				m_old_brush=pDC->SelectObject(&m_reach_init_brush);
			}
			else
			{
				if(m_reachable)
				{
					bDraw=true;
					m_old_brush=pDC->SelectObject(&m_reach_brush);
				}
			}
		}
	}
	if(bDraw)
	{
		pDC->Ellipse(m_position.x-CND_RDS,m_position.y-CND_RDS,m_position.x+CND_RDS,m_position.y+CND_RDS);
		pDC->SelectObject(m_old_brush);
	}
	m_label.Draw(pDC);
}

CRect CPNCase::MoveTo(const CPoint& point)
{
	CRect cond_rct=CPNCondition::MoveTo(point);
	m_label.MoveTo(m_position);
	return cond_rct;
}

void CPNCase::UpdatePropertiesDlg()
{
	ASSERT((CMainFrame*)AfxGetApp()->m_pMainWnd);
	CMainFrame* frame=(CMainFrame*)AfxGetApp()->m_pMainWnd;

	frame->ShowPropertiesDlg(CMainFrame::PROPDLG_HIDE,NULL);
//	frame->ShowPropertiesDlg(CMainFrame::PROPDLG_DESKTOP,NULL);
}

////////////////////////////////////////////////////////////////////////////
// CPNStep
IMPLEMENT_SERIAL(CPNStep,CPNEdge,1|VERSIONABLE_SCHEMA)

CPNStep::CPNStep()
{
	m_pLabel=NULL;
}

CPNStep::CPNStep(CSetEvents& stepset,CPNCase* from,CPNCase* to) : m_step(stepset), CPNEdge(from,to,true)
{
	m_pLabel=NULL;
}

CPNStep::~CPNStep()
{
}

void CPNStep::SetLabel(CPNLabel* pLabel)
{
	m_pLabel=pLabel;
	// move se provede az ve FirstRecalculate()
	m_step_str=m_step.Format();
}

void CPNStep::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);

	CPNEdge::Serialize(ar);
	m_step.Serialize(ar);
	if(ar.IsStoring())
	{
		ar << m_pLabel;
	}
	else
	{
		ar >> m_pLabel;
		m_step_str=m_step.Format();
	}
}

void CPNStep::Draw(CDC* pDC)
{	// draw this object on pDC
	ASSERT_VALID(this);

	CPNEdge::Draw(pDC);
}

bool CPNStep::IsEdgeCorrect()
{	// test if the edge is correct
	if(m_multi_point)
	{	// mame minimalne tri hlavni body
		CPoint side_point,mid_point;
		POSITION pos;
		pos=m_points.GetHeadPosition();
		side_point=m_points.GetNext(pos).point;
		mid_point=m_points.GetAt(pos).point;
		m_multi_correct_left=(side_point!=mid_point)&&(!m_element_from->HitTest(m_points.GetAt(m_multi_pos_left).point));
		pos=m_points.GetTailPosition();
		side_point=m_points.GetPrev(pos).point;
		mid_point=m_points.GetAt(pos).point;
		m_multi_correct_right=(side_point!=mid_point)&&(!m_element_to->HitTest(m_points.GetAt(m_multi_pos_right).point));
		return (m_multi_correct_left) && (m_multi_correct_right);
	}
	else
	{	// pouze dva hlavni body
		CPoint pt_to,pt_from;
		pt_from=m_element_from->GetPosition();
		pt_to=m_element_to->GetPosition();
		if(((pt_to.x-pt_from.x)*(pt_to.x-pt_from.x)+(pt_to.y-pt_from.y)*(pt_to.y-pt_from.y))<=(4*CND_RDS*CND_RDS+10*10))return false;
//		int d=((pt_to.x-pt_from.x)*(pt_to.x-pt_from.x)+(pt_to.y-pt_from.y)*(pt_to.y-pt_from.y));
//		TRACE("d:%d Fx:%d Fy:%d Tx:%d Ty:%d\n",d,pt_from.x,pt_from.y,pt_to.x,pt_to.y);
//		if(d<=(4*CND_RDS*CND_RDS))return false;
		return true;
	}
}

CRect CPNStep::MoveTo(const CPoint& point)
{	// move the center of this object to point
	ASSERT_VALID(this);

/*	CRect edge_rect=CPNEdge::MoveTo(point);
	POSITION mid_pos;
	if(m_multi_point)
	{
		mid_pos=m_multi_pos_left;
	}
	else
	{
		mid_pos=m_points.GetHeadPosition();
		m_points.GetNext(mid_pos);
	}
	CPoint point;
	point=m_points.GetAt(mid_pos).point;
	point.Offset(0,-LBL_DST);
	m_pLabel->MoveTo(point);*/
	return CPNEdge::MoveTo(point);
}

bool CPNStep::Recalculate(const CPNElement* pElement/*=NULL*/)
{	// recalculate intersections
	ASSERT_VALID(this);

	bool recalc=CPNEdge::Recalculate(pElement);
	if(!m_multi_point)
	{
		POSITION mid_pos;
		mid_pos=m_points.GetHeadPosition();
		m_points.GetNext(mid_pos);
		CPoint point;
		point=m_points.GetAt(mid_pos).point;
		point.Offset(0,-LBL_DST);
		m_pLabel->MoveTo(point);
	}
	return recalc;
}

void CPNStep::FirstRecalculate()
{	// recalculate fot only prepared edges
	ASSERT_VALID(this);

	CPNEdge::FirstRecalculate();

	POSITION pos=m_points.GetHeadPosition();
	m_points.GetNext(pos);

	CPoint point;
	point=m_points.GetAt(pos).point;
	point.Offset(0,-LBL_DST);
	m_pLabel->MoveTo(point);
}

////////////////////////////////////////////////////////////////////////////
// CPNPlace
IMPLEMENT_SERIAL(CPNPlace,CPNCondition,1|VERSIONABLE_SCHEMA)

CPNPlace::CPNPlace()
{
	m_pCondition=NULL;
	m_occurrenced=FALSE;
}

CPNPlace::CPNPlace(CPNCondition* pCondition) : m_pCondition(pCondition), CPNCondition(CPoint(0,0),NULL)
{
	m_occurrenced=FALSE;
	m_condition_name=pCondition->m_pLabel->GetText();
}

CPNPlace::~CPNPlace()
{
}

void CPNPlace::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);

	CPNCondition::Serialize(ar);
	if(ar.IsStoring())
	{
		ar << m_pCondition;
		ar << m_occurrenced;
		ar << m_condition_name; // NENI NUTNE UKLADAT
	}
	else
	{
		ar >> m_pCondition;
		ar >> m_occurrenced;
		ar >> m_condition_name;
	}
	m_label.Serialize(ar);
}

void CPNPlace::Draw(CDC* pDC)
{	// draw this object on pDC
	ASSERT_VALID(this);

	CPNCondition::Draw(pDC);
	m_label.Draw(pDC);
}

CRect CPNPlace::MoveTo(const CPoint& point)
{
	CRect cond_rct=CPNCondition::MoveTo(point);
	m_label.MoveTo(m_position);
	return cond_rct;
}

////////////////////////////////////////////////////////////////////////////
// CPNPlace
IMPLEMENT_SERIAL(CPNOccurrence,CPNEvent,1|VERSIONABLE_SCHEMA)

CPNOccurrence::CPNOccurrence()
{
	m_pEvent=NULL;
	m_nConnections_to=0;
	m_nConnections_from=0;
}

CPNOccurrence::CPNOccurrence(CPNEvent* pEvent) : m_pEvent(pEvent), CPNEvent(CPoint(0,0),NULL)
{
	m_nConnections_to=0;
	m_nConnections_from=0;
	m_event_name=pEvent->m_pLabel->GetText();
}

CPNOccurrence::~CPNOccurrence()
{
}

void CPNOccurrence::Serialize(CArchive& ar)
{	// save and load
	ASSERT_VALID(this);

	CPNEvent::Serialize(ar);
	if(ar.IsStoring())
	{
		ar << m_pEvent;
		ar << m_event_name;
		ar << m_nConnections_from;
		ar << m_nConnections_to;
	}
	else
	{
		ar >> m_pEvent;
		ar >> m_event_name;
		ar >> m_nConnections_from;
		ar >> m_nConnections_to;
	}
	m_label.Serialize(ar);
}

void CPNOccurrence::Draw(CDC* pDC)
{	// draw this object on pDC
	ASSERT_VALID(this);

	CPNEvent::Draw(pDC);
	m_label.Draw(pDC);
}

CRect CPNOccurrence::MoveTo(const CPoint& point)
{
	CRect event_rct=CPNEvent::MoveTo(point);
	m_label.MoveTo(m_position);
	return event_rct;
}
