// CEPNobj.h - interface for CPNElement and derivatives
//

#ifndef __CEPNOBJ_H__
#define __CEPNOBJ_H__

#include "ceset.h"

/////////////////////////////////////////////////////////////////////////////
// CPNElement - base class for all Petri Net elements
class CPNElement : public CObject
{
protected:
	DECLARE_SERIAL(CPNElement);

	enum { CND_RDS=15, EVN_RDS=15 }; // radius for circle and rectangle
	enum { CND_SEL_RDS=12 }; // radius for circle and rectangle
	enum { LBL_DST=15 }; // distance from element to label
	enum { SEL_DST=4 }; // selection distance from edge
	enum { SEL_RECT=5 }; // size of selection mark
	enum { SEL_CLR_PEN=RGB(255,0,0) }; // pen color for drawing select mark
	enum { SEL_CLR_BRUSH=RGB(255,255,255) }; // brush color for drawing select mark

	bool	m_selected;
	CPoint	m_position; // center of this element

	static	CPen	m_sel_pen; // pen for drawing selection marks
	static	CBrush	m_sel_brush; // brush for drawing selection marks
	CPen*			m_old_pen;
	CBrush*			m_old_brush;

protected:
	void DrawSelMark(CDC* pDC,int x,int y);
	void DrawSelRect(CDC* pDC,int center_x,int center_y,int half_edge_x,int half_edge_y);

public:
	CPNElement();
	CPNElement(CPoint point);
	virtual ~CPNElement();

	virtual void   Serialize(CArchive& ar);
	virtual void   Draw(CDC*);
	virtual CRect  MoveTo(const CPoint& point);
	virtual CPoint CalculateIntersection(const CPoint&);
	virtual bool   HitTest(const CPoint&);
	virtual CRect  GetRect();
	virtual void   UpdatePropertiesDlg();

	void   Select(bool select=true) { m_selected=select; }
	CPoint GetPosition()			{ ASSERT_VALID(this); return m_position; }

};
typedef CTypedPtrList<CObList,CPNElement*> CPNElementList;

/////////////////////////////////////////////////////////////////////////////
class CPNLabel : public CPNElement
{
private:
	DECLARE_SERIAL(CPNLabel);

	enum { TEXT_COLOR=RGB(0,0,0) };

	CString		m_text;
	COLORREF	m_text_color;
	LOGFONT		m_logfont;
	BOOL		m_visible;
	CRect		m_hit_rect;
	CPNElement* m_owner;
	CFont		m_font;
	COLORREF	m_old_clr;
	CFont*		m_old_font;

public:
	CPNLabel();
	CPNLabel(CPoint point);
	virtual ~CPNLabel();

	virtual void  Serialize(CArchive& ar);
	virtual void  Draw(CDC* pDC);
	virtual CRect MoveTo(const CPoint& point);
	virtual bool  HitTest(const CPoint& point);
	virtual CRect GetRect();
	virtual void  UpdatePropertiesDlg();

	CPNElement*		GetOwner()		{ return m_owner; }
	LOGFONT&		GetLogFont()	{ return m_logfont; }
	bool			IsVisible()		{ return m_visible!=0; }
	COLORREF		GetColor()		{ return m_text_color; }
	const CString&	GetText()		{ return m_text; }

	void SetOwner(CPNElement* owner)	{ m_owner=owner; }
	void SetColor(COLORREF color)		{ m_text_color=color; }
	void SetVisible(bool visible=true)	{ m_visible=visible; }
	void SetLogFont(const LOGFONT& font);
	void SetText(const CString& text,CDC* pDC);
	void SetText(const CString& text);

	void UpdateText(CDC* pDC);
};

////////////////////////////////////////////////////////////////////////
class CPNCondition : public CPNElement
{
private:
	DECLARE_SERIAL(CPNCondition);

	enum { TKN_RDS=4 }; // radius for token circle
	enum { TKN_CLR=RGB(0,0,255) }; // color for drawing tokens
	enum { LINE_CLR=RGB(0,0,0) }; // color for drawing condition circle

	BOOL	  m_token;
	CPNLabel* m_pLabel;
	bool	  m_half_full_token;
	static	CPen	m_tkn_pen; // pen for drawing token
	static	CBrush	m_tkn_brush; // brush for drawing token

	CSetEvents m_preset;
	CSetEvents m_postset;

public:
	CPNCondition();
	CPNCondition(CPoint point,CPNLabel* pLabel);
	virtual ~CPNCondition();

	virtual void   Serialize(CArchive& ar);
	virtual void   Draw(CDC* pDC);
	virtual CRect  MoveTo(const CPoint& point);
	virtual CPoint CalculateIntersection(const CPoint& to);
	virtual bool   HitTest(const CPoint& point);
	virtual CRect  GetRect();
	virtual void   UpdatePropertiesDlg();

	bool		GetToken()		{ return m_token!=0; }
	CPNLabel*	GetLabel()		{ return m_pLabel; }
	CSetEvents& GetPreSet()		{ return m_preset; }
	CSetEvents& GetPostSet()	{ return m_postset; }

	void SetToken(bool token=true)		{ m_token=token; }
	void SetHalfFullToken(bool half_full=true)	{ m_half_full_token=half_full; }
};

/////////////////////////////////////////////////////////////////////////////
class CPNEvent : public CPNElement
{
private:
	DECLARE_SERIAL(CPNEvent);

	enum { RLZ_CLR=RGB(0,255,0) }; // color for drawing realized events
	enum { RLZING_CLR=RGB(0,0,255) }; // color for drawing realizing events
	enum { LINE_CLR=RGB(0,0,0) }; // color for drawing events

	typedef struct
	{	// coefficients of line in parametric form: _x=x+t*u, _y=y+t*v
		int x,y,u,v;
	} COEF;

	CPNLabel* m_pLabel;
	CRect	  m_hit_rect;
	COEF	  m_rect_coef[4]; // coefficients of lines of rect

	static CPen	m_line_pen; // pen for events
	static CPen	m_rlz_pen; // pen for drawing realized event
	static CPen	m_rlzing_pen; // pen for drawing realizing event

	CSetConditions	m_preset;
	CSetConditions	m_postset;
	bool			m_isrealizable;
	bool			m_realizing;

private:
	void ComputeHitRect();

public:
	CPNEvent();
	CPNEvent(CPoint point,CPNLabel* pLabel);
	virtual ~CPNEvent();

	virtual void   Serialize(CArchive& ar);
	virtual void   Draw(CDC* pDC);
	virtual CRect  MoveTo(const CPoint& point);
	virtual CPoint CalculateIntersection(const CPoint& to);
	virtual bool   HitTest(const CPoint& point);
	virtual CRect  GetRect();
	virtual void   UpdatePropertiesDlg();

	bool IsRealizable()		{ return m_isrealizable; }
	void ClearRealizable()	{ m_isrealizable=false; }
	bool UpdateRealizable(CSetConditions& case_set,bool forward=true);

	void SetRealizing(bool realizing=true) { m_realizing=realizing; }

	CPNLabel* GetLabel()			{ return m_pLabel; }
	CSetConditions& GetPreSet()		{ return m_preset; }
	CSetConditions& GetPostSet()	{ return m_postset; }
};

/////////////////////////////////////////////////////////////////////////////
// support class
class CEdgePoint : public CObject
{
	DECLARE_SERIAL(CEdgePoint);

public:
	CPoint	point;
	CRect	hit_rect;
	bool	bend; // velky bod, za ktery se da tahat, tvori zlom, vsechny bend jsou zaroven side
	bool	side;

public:
	CEdgePoint();
	CEdgePoint(CEdgePoint& edgepoint);
	CEdgePoint& operator = (CEdgePoint& edgepoint);

public:
	virtual void  Serialize(CArchive& ar);
};

/////////////////////////////////////////////////////////////////////////////
class CPNEdge : public CPNElement
{
public:
	enum { TOKEN_PATH_POINTS=10 }; // 

protected:
	DECLARE_SERIAL(CPNEdge);

	enum { EDGE_SEL_RECT_BIG=7, EDGE_SEL_RECT_SML=5 }; // size of selection point

	CPNElement* m_element_from;
	CPNElement* m_element_to;
	bool		m_correct_edge; // edge is between correct points and is visible
	CRect		m_hit_rect;
	POSITION	m_drag_point_pos; // ukazatel na hlavni point kterym se prave hybe
	POSITION	m_drag_point_pos_left; // ukazatel na bocni neohnute pointy, ktere se musi posunovat spolu s hlavnim
	POSITION	m_drag_point_pos_right;
	CPoint		m_drag_point_left; // pozice hlavnich bocnich pointu pro dopocet bocnich
	CPoint		m_drag_point_right;
	bool		m_multi_point; // line is bend, minimalne 5 bodu
	POSITION	m_multi_pos_left; // hlavni levy bod hned od head(napravo)
	POSITION	m_multi_pos_right; // hlavni pravy bod hned od tail(nalevo)
	bool		m_multi_correct_left;
	bool		m_multi_correct_right;
	CList<CEdgePoint,CEdgePoint&> m_points;

	CPoint	m_token_path[TOKEN_PATH_POINTS];

	enum { ARROW_WIDTH=8, ARROW_HEIGHT=10 };

protected:
	void DrawPoint(CDC* pDC,CEdgePoint& edgepoint);
	void ComputeHitRect();

protected:
	virtual bool IsEdgeCorrect();

public:
	CPNEdge();
	CPNEdge(CPNElement* from,CPNElement* to,int onlyprepare=false);
	virtual ~CPNEdge();

	virtual void  Serialize(CArchive& ar);
	virtual void  Draw(CDC* pDC);
	virtual CRect MoveTo(const CPoint& point);
	virtual bool  HitTest(const CPoint& point);
	virtual CRect GetRect();
	virtual void  UpdatePropertiesDlg();
	virtual bool  Recalculate(const CPNElement* pElement=NULL);
	virtual void  FirstRecalculate();

	bool Compare(CPNEdge* pedge);
	bool IsCorresponding(CPNElement* pElement);
	bool IsCorresponding(CPNElement* pFrom,CPNElement* pTo);
	bool IsDragging(const CPoint& point);
	bool IsMultipoint()		{ return m_multi_point; }
	CPNElement* GetFrom()	{ return m_element_from; }
	CPNElement* GetTo()		{ return m_element_to; }
	bool CanDeletePoint();
	bool DeletePoint();
	void DeleteAllPoints();
	CRect MoveByVector(CSize move);
	void UpdateCorrespondingSets();
	void ComputeTokenPath();
	CPoint GetTokenPathPoint(int index);
};

/////////////////////////////////////////////////////////////////////////////
class CPNPolygon : public CPNElement
{
private:
	DECLARE_SERIAL(CPNPolygon);

	enum { EDGE_SEL_RECT_BIG=7, EDGE_SEL_RECT_SML=5 }; // size of selection point
	enum { POLYGON_HEIGHT=40, POLYGON_WIDTH=160 };
	enum { LINE_STYLE_SOLID=0, LINE_STYLE_DASH, LINE_STYLE_DOT };
	enum { FILL_MODE_EMPTY=0, FILL_MODE_FILLED };
	enum { LINE_COLOR=RGB(0,0,0), FILL_COLOR=RGB(200,200,255) };

	COLORREF m_line_color;
	int		 m_line_style;
	int		 m_line_width;
	COLORREF m_fill_color;
	int		 m_fill_mode;
	POINT*	 m_poly_points;
	int		 m_poly_count;
	CRect	 m_hit_rect;
	POSITION m_drag_point_pos; // ukazatel na hlavni point kterym se prave hybe
	POSITION m_drag_point_pos_left; // ukazatel na bocni neohnute pointy
	POSITION m_drag_point_pos_right;
	CPoint	 m_drag_point_left; // pozice hlavnich bocnich pointu pro dopocet bocnich
	CPoint	 m_drag_point_right;
	CList<CEdgePoint,CEdgePoint&> m_points;

private:
	void DrawPoint(CDC* pDC,CEdgePoint& edgepoint);
	void ComputeHitRect();

public:
	CPNPolygon();
	CPNPolygon(CPoint point);
	virtual ~CPNPolygon();

	virtual void  Serialize(CArchive& ar);
	virtual void  Draw(CDC* pDC);
	virtual CRect MoveTo(const CPoint& point);
	virtual bool  HitTest(const CPoint& point);
	virtual CRect GetRect();
	virtual void  UpdatePropertiesDlg();

	bool IsDragging(const CPoint& point);
	bool CanDeletePoint();
	bool DeletePoint();
	CRect MoveByVector(CSize move);
	
	COLORREF GetLineColor()	{ return m_line_color; }
	int		 GetLineStyle()	{ return m_line_style; }
	int		 GetLineWidth()	{ return m_line_width; }
	COLORREF GetFillColor()	{ return m_fill_color; }
	int		 GetFillMode()	{ return m_fill_mode; }

	void SetLineColor(COLORREF color)	{ m_line_color=color; }
	void SetLineStyle(int style)		{ m_line_style=style; }
	void SetLineWidth(int width)		{ m_line_width=width; }
	void SetFillColor(COLORREF color)	{ m_fill_color=color; }
	void SetFillMode(int mode)			{ m_fill_mode=mode; }
};

////////////////////////////////////////////////////////////////////////
class CPNCase : public CPNCondition
{
private:
	DECLARE_SERIAL(CPNCase);

	enum { REACH_BRUSH=RGB(64,255,0), REACH_INIT_BRUSH=RGB(100,100,255), HIGHLIGHT_BRUSH=RGB(255,0,0) };

	CPNLabel		m_label;
	CSetConditions	m_case;
	CString			m_case_str;
	bool			m_highlight; // highlight actual case
	BOOL			m_reachable; // case is reachable
	BOOL			m_reachable_init; // case is initial for reachable cases
	bool			m_show_reachable;
	static	CBrush	m_reach_brush; // highlight reachable cases brush
	static	CBrush	m_reach_init_brush; // vychozi highlight reachable cases brush
	static	CBrush	m_high_brush; // highlight actual case brush

public:
	CPNCase();
	CPNCase(const CSetConditions& caseset);
	virtual ~CPNCase();

	virtual void  Serialize(CArchive& ar);
	virtual void  Draw(CDC* pDC);
	virtual CRect MoveTo(const CPoint& point);
	virtual void  UpdatePropertiesDlg();

	void ShowReachAble(bool show=true)			{ m_show_reachable=show; }
	void SetReachAble(BOOL set=TRUE)			{ m_reachable=set; }
	void SetReachAbleInit(BOOL set=TRUE)		{ m_reachable_init=set; }
	void SetText(const CString& text,CDC* pDC)	{ m_label.SetText(text,pDC); }
	void HighLight(bool highlight)				{ m_highlight=highlight; }

	bool			IsReachAble()	{ return m_reachable!=0; }
	bool			IsReachAbleInit()	{ return m_reachable_init!=0; }
	CSetConditions& GetCase()		{ return m_case; }
	const CString&	GetCaseStr()	{ return m_case_str; }
	const CString&	GetText()		{ return m_label.GetText(); }
};

////////////////////////////////////////////////////////////////////////
class CPNStep : public CPNEdge
{
private:
	DECLARE_SERIAL(CPNStep);

	CSetEvents	m_step;
	CString		m_step_str;
	CPNLabel*	m_pLabel;

public:
	CPNStep();
	CPNStep(CSetEvents& stepset,CPNCase* from,CPNCase* to);
	virtual ~CPNStep();

	virtual void  Serialize(CArchive& ar);
	virtual void  Draw(CDC* pDC);
	virtual CRect MoveTo(const CPoint& point);
	virtual bool  Recalculate(const CPNElement* pElement=NULL);
	virtual void  FirstRecalculate();

	void SetLabel(CPNLabel* pLabel);

	CPNLabel*	GetLabel()		{ return m_pLabel; }
	CSetEvents&	GetStep()		{ return m_step; }
	CString&	GetStepStr()	{ return m_step_str; }
	CPNElement*	GetElementFrom(){ return m_element_from; }
	CPNElement*	GetElementTo()	{ return m_element_to; }
	
protected:
	virtual bool IsEdgeCorrect();
};

////////////////////////////////////////////////////////////////////////
class CPNPlace : public CPNCondition
{
private:
	DECLARE_SERIAL(CPNPlace);

	CPNCondition*	m_pCondition; // original condition for which this is reflection
	BOOL			m_occurrenced; // has edge

	// NAPROSTO SPATNE NAVRZENO, POUZIT LABEL OD CONDITION
	CPNLabel		m_label;


	CString			m_condition_name;

public:
	CPNPlace();
	CPNPlace(CPNCondition* pCondition);
	virtual ~CPNPlace();

	virtual void  Serialize(CArchive& ar);
	virtual void  Draw(CDC* pDC);
	virtual CRect MoveTo(const CPoint& point);

	void SetText(const CString& text,CDC* pDC)	{ m_label.SetText(text,pDC); }
	void SetOccurrenced()						{ m_occurrenced=TRUE; }

	CPNCondition* GetReflectionCondition()		{ return m_pCondition; }
	const CString& GetConditionName()			{ return m_condition_name; }
	const CString&	GetText()					{ return m_label.GetText(); }

	bool IsReflection(CPNCondition* pCondition)	{ return ((!m_occurrenced) && (m_pCondition==pCondition)); }
};

////////////////////////////////////////////////////////////////////////
class CPNOccurrence : public CPNEvent
{
private:
	DECLARE_SERIAL(CPNOccurrence);

	CPNEvent*	m_pEvent; // original condition for which this is reflection
	CPNLabel	m_label;
	CString		m_event_name;
	int			m_nConnections_from;
	int			m_nConnections_to; // pocet hran do udalosti

public:
	CPNOccurrence();
	CPNOccurrence(CPNEvent* pEvent);
	virtual ~CPNOccurrence();

	virtual void  Serialize(CArchive& ar);
	virtual void  Draw(CDC* pDC);
	virtual CRect MoveTo(const CPoint& point);

	void AddConnectionFrom()					{ m_nConnections_from++; }
	void AddConnectionTo()						{ m_nConnections_to++; }
	void SetText(const CString& text,CDC* pDC)	{ m_label.SetText(text,pDC); }


	CPNEvent* GetReflectionEvent()				{ return m_pEvent; }
	const CString& GetEventName()				{ return m_event_name; }
	const CString&	GetText()					{ return m_label.GetText(); }
	bool IsReflection(CPNEvent* pEvent)			{ return m_pEvent==pEvent; }
	bool IsMultiConnectionFrom()				{ return m_nConnections_from>1; }
	bool IsMultiConnectionTo()					{ return m_nConnections_to>1; }
};

#endif

