/*
  Ruzicka's formal model core - header file
  Copyright (C) 2007 Jaroslav Skarvada <skarvada@fit.vutbr.cz>  
  
  This program is free software: you can redistribute it and/or modify  
  it under the terms of the GNU General Public License as published by  
  the Free Software Foundation, either version 3 of the License, or  
  (at your option) any later version.  
  
  This program is distributed in the hope that it will be useful,  
  but WITHOUT ANY WARRANTY; without even the implied warranty of  
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
  GNU General Public License for more details.  
 
  You should have received a copy of the GNU General Public License  
  along with this program.  If not, see <http://www.gnu.org/licenses/>.  
*/

#ifndef imap
#include <map>
#define imap
#endif

#ifndef iset
#include <set>
#define iset
#endif

#ifndef ivector
#include <vector>
#define ivector
#endif

#ifndef istring
#include <string>
#define istring
#endif

#ifndef ierror
#include "error.h"
#define ierror
#endif

#define c_comment '#'
#define c_string '"'
#define c_eof 26
#define seps "$,=-;:."

using namespace std;

typedef int tid;		// elements/gates IDs
typedef set < tid > telset;	// set of element

// template class for pair of elements
template < class t1, class t2 > class tmpair
{
public:
  t1 f;				// first element
  t2 s;				// second element
  tmpair (t1 fi = (t1) 0, t2 si = (t2) 0)
  {
    f = fi;
    s = si;
  }
  bool operator< (const tmpair p) const	// less operator
  {
    return ((f < p.f) || (f == p.f) && (s < p.s));
  }
};


typedef tmpair < tid, tid > tpair;	// pair of ID
typedef tmpair < tid, string > tidstrpair;	// pair of ID and string
typedef map < tpair, int >tdelta;	// pair to int mapping
typedef set < tpair > tpairset;	// set of pairs
typedef vector < tid > tsequence;	// sequence of elements
typedef multimap < tpair, tsequence > tro;	// type RO
typedef vector < string > telv;	// type element names
typedef map < tid, string > tgnames;	// type gate names
typedef map < tpair, string > tpairstrmap;	// pair to string mapping
typedef map < tid, tid > tidmap;	// type ID to ID mapping
typedef map < tid, int >tidintmap;	// type ID to int mapping
typedef map < tid, telset > tidsetmap;	// set of registers for ID
typedef set < tidstrpair > tmiel;	// mi component
typedef multimap < tpair, tmiel > tmi;	// type mi 
typedef map < string, telset > tstrelsetmap;	// string to elset mapping
typedef map < string, tid > tstridmap;	// string to id mapping

// class unit under analysis
class tuua
{
private:
  // extracts one char <char> from stream <f>
  char get_char (istream & f);

  // tries to skip char <chr> in stream <f>, returns true on success
  bool skip_char (istream & f, char chr);

  // gets token <token> from stream <f>
  bool get_token (istream & f, string & token);

  // gets string <str> from stream <f> and removes quoting 
  bool get_string (istream & f, string & str);

  // gets elset <elset> from stream <f> 
  bool get_elset (istream & f, telset & elset);

  // gets sequence <sequence> from stream <f> 
  bool get_sequence (istream & f, tsequence & sequence);

  // gets gnames <gnames> from istream <f>
  bool get_gnames (istream & f, tgnames & gnames);

  // gets idsetmap <idsetmap> from istream <f>
  bool get_idsetmap (istream & f, tidsetmap & idsetmap);

  // gets pairset <pairset> from istream <f>
  bool get_pairset (istream & f, tpairset & pairset);

  // gets pairstrmap <pm> from istream <f>
  bool get_pairstrmap (istream & f, tpairstrmap & pm);

  // gets ro <ro> from istream <f>
  bool get_ro (istream & f, tro & ro);

  // gets delta <delta> from istream <f>
  bool get_delta (istream & f, tdelta & delta);

  // gets miel <miel> from istream <f>
  bool get_miel (istream & f, tmiel & miel);

  // gets mi <mi> from istream <f>
  bool get_mi (istream & f, tmi & mi);

  // gets delta <delta> from istream <f>
  bool get_width (istream & f, tidintmap & width);

  // skips unknown set in istream <f>
  void skip_set (fstream & f);

  // formats string <str> for output as string
  string format_str (string str);

  // prints elset <elset> to ostream <f>
  void print_elset (ostream & f, telset & elset);

  // prints gnames <gnames> to ostream <f>
  void print_gnames (ostream & f, tgnames & gnames);

  // prints idsetmap <idsetmap> to ostream <f>
  void print_idsetmap (ostream & f, tidsetmap & idsetmap);

  // prints pairset <pairset> to ostream <f>
  void print_pairset (ostream & f, tpairset & pairset);

  // prints pairstrmap <pm> to ostream <f>
  void print_pairstrmap (ostream & f, tpairstrmap & pm);

  // prints ro <ro> to ostream <f>
  void print_ro (ostream & f, tro & ro);

  // prints delta <delta> to ostream <f>
  void print_delta (ostream & f, tdelta & delta);

  // prints mi <mi> to ostream <f>
  void print_mi (ostream & f, tmi & mi);

  // prints width <width> to ostream <f>
  void print_width (ostream & f, tidintmap & width);

  // prints UUA formal model to ostream <f>
  void print_uua (ostream & f);
public:
  string name;		// UUA label
  telset E, INP, OUTP, CI, PI, PO, P, R;	// sets
  telset INV;			// sets of invertors
  telset SCAN;			// scan chain
  tpairset C, M, I, II;		// pairs
  tro RO;			// ro
  tmi MI;			// mi
  tdelta DELTA;			// delta
  tidsetmap V;			// nodes
  tidsetmap TDR, TRV, TIR, TOR;	// special registers
  tidsetmap PSI;		// psi
  tgnames ELN;			// ID to element name mapping
  tgnames ELT;			// ID to element type mapping
  tgnames GN;			// gates IDs to names mapping
  tgnames EG;			// ID to element group map
  tgnames PG;			// ID to port group map
  tpairstrmap CG;		// connections to group mapping
  tidintmap WIDTH;		// gates IDs to WIDTH mapping
  // clears all sets
  void clear (void);

  // ipt predicate, returns true if there exists I-path from gate <from> to
  // gate <to>
  bool ipt (tid from, tid to);

  // makes some simple validity tests 
  // so far only non empty sets test and test for positive IDXs
  terr check_validity (void);

  // check for unbound inputs from INPUT set, can be self.CI or self.INP
  tid check_ipunbound (telset & INPUT);

  // checks for unbound outputs
  tid check_opunbound (void);

  // write formal model to a file
  terr WriteFormalUUA (string ruzname);

  // UUA file parser
  terr LoadFormalUUA (string name);
};
