/*
  Ruzicka's formal model support routines
  Copyright (C) 2007 Jaroslav Skarvada <skarvada@fit.vutbr.cz>  
  
  This program is free software: you can redistribute it and/or modify  
  it under the terms of the GNU General Public License as published by  
  the Free Software Foundation, either version 3 of the License, or  
  (at your option) any later version.  
  
  This program is distributed in the hope that it will be useful,  
  but WITHOUT ANY WARRANTY; without even the implied warranty of  
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
  GNU General Public License for more details.  
 
  You should have received a copy of the GNU General Public License  
  along with this program.  If not, see <http://www.gnu.org/licenses/>.  
*/

#ifndef icstdio
#include <cstdio>
#define icstdio
#endif

#ifndef ifstream
#include <fstream>
#define ifstream
#endif

#ifndef iiostream
#include <iostream>
#define iiostream
#endif

#ifndef istring
#include <string>
#define istring
#endif

#ifndef iclimits
#include <climits>
#define iclimits
#endif

#ifndef isetup
#include "setup.h"
#define isetup
#endif

#ifndef ierror
#include "error.h"
#define ierror
#endif

#ifndef iruzic
#include "ruzic.h"
#define iruzic
#endif

#ifndef iutils
#include "utils.h"
#define iutils
#endif

#ifndef izboril
#include "zboril.h"
#define izboril
#endif

using namespace std;

// reads I-path condition from "*.vhc"
// returns true if OK, false otherwise
bool
get_ms (fstream & f, tms & ms)
{
  char ctl[pnsize];
  char val[pnsize];

  if (!skip_token (f, "[") || !get_str (f, ctl, pnsize)
      || !skip_token (f, ",") || !get_data (f, val, pnsize, ']'))
    return false;
  ms.mset.insert (tmset::value_type (string (ctl), string (val)));
  return true;
}

// reads I-path condition from "*.vhc"
// returns true if OK, false otherwise
bool
get_msv (fstream & f, tmsv & msv)
{
  char c;
  tms ms;
  char from[pnsize];
  char to[pnsize];

  ms.mset.clear ();
  if (!skip_token (f, "[") || !skip_token (f, "[")
      || !get_str (f, from, pnsize) || !skip_token (f, ",")
      || !get_str (f, to, pnsize) || !skip_token (f, "]")
      || !skip_token (f, ",") || !get_ms (f, ms) || !get_tok (f, c))
    return false;
  while (c == ',')		// gets all ms
    if (!get_ms (f, ms) || !get_tok (f, c))
      return false;
  if (c != ']')
    return false;

  ms.from = string (from);
  ms.to = string (to);
  msv.push_back (ms);
  return true;
}

void tpenalty::clear(void)
{
  penalty.clear();
}

int tpenalty::get(tid id)
{
  tscore::iterator it;
  
  if ((it=penalty.find(id))!=penalty.end())
    return (*it).second;
  else
    return 0;    
}

void tpenalty::set(tid id, int pen)
{
  if (penalty.find(id)!=penalty.end())
    penalty.erase(id);
  penalty.insert(tscore::value_type(id,pen));
}

void
truua::clear (void)
{
  tuua::clear ();
  ATDR.clear ();
  ATRV.clear ();
  ATIR.clear ();
  ATOR.clear ();
  CTR.clear ();
  OBS.clear ();
  NOBS.clear ();
  NCTR.clear ();
  CTDR.clear ();
  OTRV.clear ();
  IPSI.clear ();
  SCORE.clear ();
  exc.clear ();
  vhc.clear ();
  fname.erase ();
  foname.erase ();
}

// parser of "*.vhc" file, parses file <fname> to <vhc> and gets number
// of parsed elements in <fels>, return value is error code
terr
truua::parsevhc (string fname, int &fels)
{
  bool eread;
  fstream f;
  char c, c2;
  char elname[pnsize];
  tels els;

  fels = 0;
  vhc.clear ();
  eread = false;
  f.open (fname.c_str (), ios::in);
  if (!f)
    return e_fopen;
  while (!eread && f)
    {
      els.msv.clear ();
      if (get_tok (f, c))
	f.putback (c);
      if (f)
	{
	  if (!skip_token (f, "element") || !skip_token (f, "(") ||
	      !get_str (f, elname, pnsize) || !skip_token (f, ",")
	      || !get_qchar (f, c2) || !skip_token (f, ",")
	      || !skip_token (f, "[") || !get_msv (f, els.msv)
	      || !get_tok (f, c))
	    eread = true;
	  while (!eread && (c == ','))	// gets all msv
	    if (!get_msv (f, els.msv) || !get_tok (f, c))
	      eread = true;
	  if (!eread)
	    if (c == ']')
	      f.ignore (INT_MAX, '\n');
	    else
	      eread = true;
	  if (!eread)
	    {
	      els.isreg = (c2 == 'T');
	      els.isinv = (c2 == 'I');
	      vhc.insert (tvhc::value_type (string (elname), els));
	    }
	  fels++;
	}
    }
  f.close ();
  if (eread)
    return e_fformat;
  else
    return e_ok;
}

// parser of "*.exc" files, parses file <fname> to <exc> and returns error code
terr
truua::parseexc (string fname)
{
  fstream f;
  char str[pnsize];

  exc.clear ();
  f.open (fname.c_str (), ios::in);
  if (!f)
    return e_fopen;
  while (f && get_str (f, str, pnsize))
    exc.insert (string (str));
  if (f)
    return e_fformat;
  f.close ();
  return e_ok;
}

// Unit Under Analysis (UUA) files consistency checker
terr
truua::check_uuaconsist (fstream & fprv, fstream & fbrn, fstream & fspj,
			 fstream & fspi)
{
  tprvek prvek;
  tbrana brana;
  tspoj spoj;
  longint spi;
  char str[32];

  read_prvek (fprv, prvek);
  read_brana (fbrn, brana);
  read_spoj (fspj, spoj);
  fspi.read ((char *) &spi, sizeof (spi));
  if (!fprv || !fbrn || !fspj || !fspi)
    return e_fformat;
  if ((prvek.cislo != brana.cislo) || (prvek.cislo != spoj.cislo) ||
      (prvek.cislo != spi) || (strcmp (prvek.jmeno, brana.jmeno)) ||
      (strcmp (prvek.jmeno, spoj.jmeno)))
    return e_fsinconsist;
  sprintf (str, "%d", spi);
  fid = string (str);
  fdate = passtr2stl (prvek.jmeno);

  return e_ok;
}

// exclude file "*.exc" consistency checker
bool
truua::check_exc (string name)
{
  for (texc::iterator it = exc.begin (); it != exc.end (); it++)
    if ((*it) == name)
      return true;

  return false;
}

// reads connection
bool
truua::read_con (fstream & fspj, fstream & fspi, int from)
{
  int x;
  tpair pair;
  tspoj spoj;
  tspi spi;
  telset gates;

  gates.clear ();
  read_spoj (fspj, spoj);
  if (!fspj)
    return false;
  fspi.seekg (spoj.iprv * spi_size);
  x = spoj.podkud + spoj.podkam + spoj.pkam;
  pair.f = from;
  gates.insert (from);
  while (x && fspi)
    {
      read_spi (fspi, spi);
      pair.s = spi;
      C.insert (pair);
      gates.insert (spi);
      x--;
    }
  if (aux.find (from) == aux.end ())
    V.insert (tidsetmap::value_type (from, gates));
  aux.insert (gates.begin (), gates.end ());
  return fspi;
}

// reads circuit
terr
truua::read_circ (fstream & fprv, fstream & fbrn, fstream & fspj,
		  fstream & fspi, int &nprv, int &nbrn)
{
  int x;
  tprvek prvek;
  tbrana brana;

  read_prvek (fprv, prvek);
  if (!fprv || (passtr2stl (prvek.typ) != "Circuit"))
    return e_nocirc;
  nprv++;
  name = passtr2stl (prvek.jmeno);
  fbrn.seekg (prvek.iprv * brana_size);
// for all input gates
  for (x = 0; (x < prvek.pvst) && fbrn && fspj && fspi; x++, nbrn++)
    {
      read_brana (fbrn, brana);
      PI.insert (brana.cislo);
      GN.
	insert (tgnames::
		value_type (brana.cislo, passtr2stl (brana.jmeno)));
      WIDTH.insert (tidintmap::value_type (brana.cislo, brana.sspoje));
      fspj.seekg (brana.ispoje * spoj_size);
      read_con (fspj, fspi, brana.cislo);
    }
// for all input/output gates
  for (x = 0; (x < prvek.pvsvy) && fbrn && fspj && fspi; x++, nbrn++)
    {
      read_brana (fbrn, brana);
      PI.insert (brana.cislo);
      PO.insert (brana.cislo);
      GN.
	insert (tgnames::
		value_type (brana.cislo, passtr2stl (brana.jmeno)));
      WIDTH.insert (tidintmap::value_type (brana.cislo, brana.sspoje));
      fspj.seekg (brana.ispoje * spoj_size);
      read_con (fspj, fspi, brana.cislo);
    }
// for all output gates
  for (x = 0; (x < prvek.pvyst) && fbrn && fspj && fspi; x++, nbrn++)
    {
      read_brana (fbrn, brana);
      PO.insert (brana.cislo);
      GN.
	insert (tgnames::
		value_type (brana.cislo, passtr2stl (brana.jmeno)));
      WIDTH.insert (tidintmap::value_type (brana.cislo, brana.sspoje));
      fspj.seekg (brana.ispoje * spoj_size);
      read_con (fspj, fspi, brana.cislo);
    }
  if (!fbrn || !fspj || !fspi)
    return e_fformat;
  return e_ok;
}

// gets first end of M-path
bool
truua::getMend (tvhc::iterator & it_vhc, string ins, string & outs)
{
  tmsv::iterator it_msv;

  for (it_msv = (*it_vhc).second.msv.begin ();
       it_msv != (*it_vhc).second.msv.end (); it_msv++)
    if ((*it_msv).from == ins)
      {
	outs = (*it_msv).to;
	return true;
      }

  return false;
}

// finds all M-path
terr
truua::findM (fstream & fbrn, tvhc::iterator & it, int offs, int inn,
	      int inoutn, int outn)
{
  bool found;
  int x, y, ing;
  long fpos;
  string outgs;
  tpair pair;
  tbrana brana;

  fbrn.seekg (offs);
  for (x = 0; (x < (inn + inoutn)) && fbrn; x++)
    {
      read_brana (fbrn, brana);
      if (!fbrn)
	return e_fformat;
      ing = brana.cislo;
      if (getMend (it, passtr2stl (brana.jmeno), outgs))
	{
	  fpos = fbrn.tellg ();
	  fbrn.seekg (offs + inn * brana_size);
	  y = inoutn + outn;
	  found = false;
	  while (y && !found && fbrn)
	    {
	      read_brana (fbrn, brana);
	      found = (passtr2stl (brana.jmeno) == outgs);
	      y--;
	    }
	  if (!found)
	    return e_fsinconsist;
	  pair.f = ing;
	  pair.s = brana.cislo;
	  M.insert (pair);
	  fbrn.seekg (fpos);
	}
    }
  if (!fbrn)
    return e_fformat;

  return e_ok;
}

// reads element
terr
truua::read_element (fstream & fprv, fstream & fbrn, fstream & fspj,
		     fstream & fspi, int &nprv, int &nbrn)
{
  int x;
  tprvek prvek;
  tbrana brana;
  string s;
  telset gates;
  tvhc::iterator it;

  read_prvek (fprv, prvek);
  if (!fprv)
    return e_ok;
  nprv++;

  gates.clear ();
  ELN.
    insert (tgnames::value_type (prvek.cislo, passtr2stl (prvek.jmeno)));
  ELT.
    insert (tgnames::
	    value_type (prvek.cislo, s = passtr2stl (prvek.typ)));
  E.insert (prvek.cislo);
  it = vhc.find (s);
  if (it != vhc.end ())
    {
      if ((*it).second.isreg)	// if it is register
	R.insert (prvek.cislo);	// insert register
      if ((*it).second.isinv)	// if it is invertor
	INV.insert (prvek.cislo);	// insert invertor
      if (findM (fbrn, it, prvek.iprv * brana_size, prvek.pvst, prvek.pvsvy,
		 prvek.pvyst))
	return e_fsinconsist;
    }
  fbrn.seekg (prvek.iprv * brana_size);
// for all input gates
  for (x = 0; (x < prvek.pvst) && fbrn && fspj && fspi; x++, nbrn++)
    {
      read_brana (fbrn, brana);
      if (!check_exc (passtr2stl (brana.jmeno)))
	INP.insert (brana.cislo);	// input gate
      else
	CI.insert (brana.cislo);	// control gate
      IPSI.insert (tidmap::value_type (brana.cislo, prvek.cislo));
      gates.insert (brana.cislo);	// for PSI
      GN.
	insert (tgnames::
		value_type (brana.cislo, passtr2stl (brana.jmeno)));
      WIDTH.insert (tidintmap::value_type (brana.cislo, brana.sspoje));
      fspj.seekg (brana.ispoje * spoj_size);
      read_con (fspj, fspi, brana.cislo);
    }
// for all input/output gates
  for (x = 0; (x < prvek.pvsvy) && fbrn && fspj && fspi; x++, nbrn++)
    {
      read_brana (fbrn, brana);
      if (!check_exc (passtr2stl (brana.jmeno)))
	{
	  INP.insert (brana.cislo);	// input gate
	  OUTP.insert (brana.cislo);	// output gate
	}
      else
	CI.insert (brana.cislo);	// control gate
      IPSI.insert (tidmap::value_type (brana.cislo, prvek.cislo));
      gates.insert (brana.cislo);	// for PSI
      GN.
	insert (tgnames::
		value_type (brana.cislo, passtr2stl (brana.jmeno)));
      WIDTH.insert (tidintmap::value_type (brana.cislo, brana.sspoje));
      fspj.seekg (brana.ispoje * spoj_size);
      read_con (fspj, fspi, brana.cislo);
    }
// for all output gates
  for (x = 0; (x < prvek.pvyst) && fbrn && fspj && fspi; x++, nbrn++)
    {
      read_brana (fbrn, brana);
      if (!check_exc (passtr2stl (brana.jmeno)))
	OUTP.insert (brana.cislo);	// output gate
      else
	CI.insert (brana.cislo);	// control gate
      IPSI.insert (tidmap::value_type (brana.cislo, prvek.cislo));
      gates.insert (brana.cislo);	// for PSI
      GN.
	insert (tgnames::
		value_type (brana.cislo, passtr2stl (brana.jmeno)));
      WIDTH.insert (tidintmap::value_type (brana.cislo, brana.sspoje));
      fspj.seekg (brana.ispoje * spoj_size);
      read_con (fspj, fspi, brana.cislo);
    }
  PSI.insert (tidsetmap::value_type (prvek.cislo, gates));
  if (!fbrn || !fspj || !fspi)
    return e_fformat;

  return e_ok;
}

// loads whole Unit Under Analysis (UUA) from Verilog
terr
truua::LoadUUAv (string fveriname, int &nprv, int &nbrn)
{
/*
  bool eread;
  terr err;
  tcharset delimset1,delimset2;
  tstrelsetmap consmap;  // Mapping of connections for closure finding
 // tstridmap 
  string fbasename;
  char elname[pnsize]={0};
  fstream fveri;
  
  fbasename=fveriname.substr(0,fveriname.find('.'));
  eread=false;
  err=e_ok;
  E.clear(); INP.clear(); OUTP.clear(); CI.clear(); PI.clear(); PO.clear(); 
  R.clear(); P.clear(); C.clear(); M.clear(); ELN.clear(); ELT.clear(); 
  GN.clear(); WIDTH.clear(); IPSI.clear(); PSI.clear(); V.clear(); INV.clear();
  consmap.clear();
  nprv=nbrn=0;
  nprv=nbrn=0;
  delimset1.clear(); delimset2.clear();
  delimset1.insert(' ');
  delimset1.insert('\t');
  delimset2=delimset1;
  delimset1.insert('(');
  fveri.open(fveriname.c_str(),ios::in);
  if (!fveri)
  {
    err=e_fopen;
  }
  else
  {
    do
    {
      while (!eread && skip_comment(fveri,veri_comment))
        ;
      if (!eread && (!skip_token(fveri,"module") || !get_data2(fveri,elname,
                                                             pnsize,delimset1)))
        eread=true;
      if (fbasename!=string(elname))
        eread=!find_line(fveri,"endmodule");
    } while (!eread && (fbasename!=string(elname)));
    do
    {
      if (!eread && fveri && !get_data2(fveri,elname,pnsize,delimset2))
        eread=true;
      else
      {
        if (string(elname)==string("input"))
        {
          eread=!read_inputs(fveri);
        }
        else if (string(elname)==string("output"))
        {
          eread=!read_outputs(fveri);
        }
        else if (string(elname)==string("wire"))
        {
          eread=!read_wires(fveri);
        }  
        else if (string(elname)!=string("endmodule")
        {
          eread=!get_els(fveri);// ?
        }
      } 
    } while (!eread && f && (string(elname)!=string("endmodule")));  
  }
    
      
      if (get_tok(fveri,c))
        fveri.putback(c);
      if (fveri)
      {
        
        if (!skip_token(f,"element") || !skip_token(f,"(") ||
            !get_str(f,elname,pnsize) || !skip_token(f,",") || !get_qchar(f,c2) ||
            !skip_token(f,",") || !skip_token(f,"[") || !get_msv(f,els.msv) ||
            !get_tok(f,c))
        eread=true;
      while (!eread && (c==','))                 // gets all msv
        if (!get_msv(f,els.msv) || !get_tok(f,c))
          eread=true;
      if (!eread)
        if (c==']')
          f.ignore(INT_MAX,'\n');
        else eread=true;
      if (!eread)
      {
        els.isreg=(c2=='T');
        vhc.insert(tvhc::value_type(string(elname),els));
      }
      fels++;
    }
  }
  f.close();

  }
  if (!err)
    if (!fprv || !fbrn || !fspj || !fspi)
      err=e_fformat;
  if (!err)                                      // reads circuit
    err=read_circ(fprv,fbrn,fspj,fspi,nprv,nbrn);
  while (!err && fprv)                           // reads all elements
    err=read_element(fprv,fbrn,fspj,fspi,nprv,nbrn);

  if (!err)
  {
    P.insert(INP.begin(),INP.end());
    P.insert(OUTP.begin(),OUTP.end());
    P.insert(CI.begin(),CI.end());
  }
  fveri.close();

  if ((err==e_ok) && eread)
    err=e_fformat;

  return err;
*/
  return 0;
}

// loads whole Unit Under Analysis (UUA) from zboril
terr
truua::LoadUUA (string fprvname, string fbrnname, string fspjname,
		string fspiname, int &nprv, int &nbrn)
{
  terr err;
  ios::fmtflags flags;
  fstream fprv, fbrn, fspj, fspi;

  err = e_ok;
  E.clear ();
  INP.clear ();
  OUTP.clear ();
  CI.clear ();
  PI.clear ();
  PO.clear ();
  R.clear ();
  P.clear ();
  C.clear ();
  M.clear ();
  ELN.clear ();
  ELT.clear ();
  GN.clear ();
  WIDTH.clear ();
  IPSI.clear ();
  PSI.clear ();
  V.clear ();
  INV.clear ();
  aux.clear ();
  nprv = nbrn = 0;
  nprv = nbrn = 0;
  fprv.open (fprvname.c_str (), ios::in | iosparm);
  fbrn.open (fbrnname.c_str (), ios::in | iosparm);
  fspj.open (fspjname.c_str (), ios::in | iosparm);
  fspi.open (fspiname.c_str (), ios::in | iosparm);
  if (!fprv || !fbrn || !fspj || !fspi)
    err = e_fopen;
  else
    {				// proceed with all chars
      flags = fprv.flags ();
      fprv.flags (flags & ~ios::skipws);
      flags = fbrn.flags ();
      fbrn.flags (flags & ~ios::skipws);
      flags = fspj.flags ();
      fspj.flags (flags & ~ios::skipws);
      flags = fspi.flags ();
      fspi.flags (flags & ~ios::skipws);
      err = check_uuaconsist (fprv, fbrn, fspj, fspi);	// check consist
    }
  if (!err)
    if (!fprv || !fbrn || !fspj || !fspi)
      err = e_fformat;
  if (!err)			// reads circuit
    err = read_circ (fprv, fbrn, fspj, fspi, nprv, nbrn);
  while (!err && fprv)		// reads all elements
    err = read_element (fprv, fbrn, fspj, fspi, nprv, nbrn);

  if (!err)
    {
      P.insert (INP.begin (), INP.end ());
      P.insert (OUTP.begin (), OUTP.end ());
      P.insert (CI.begin (), CI.end ());
    }

  fprv.close ();
  fbrn.close ();
  fspj.close ();
  fspi.close ();

  return err;
}


// find smallest delta
void
truua::get_smallest_delta (tid from, telset & gates, tid & to, int &dist)
{
  bool first = true;
  int pen = 0;

  for (tdelta::iterator it = DELTA.begin (); it != DELTA.end (); it++)
    if (((*it).first.f == from)
	&& (gates.find ((*it).first.s) != gates.end ()))
      {
	if (first)
	  {
	    first = false;
	    to = (*it).first.s;
	    dist = (*it).second;
	    pen = penalty.get((*it).first.s);
	  }
	else if (((*it).second < dist) || ((*it).second == dist) && (penalty.get((*it).first.s) < pen))
	  {
	    to = (*it).first.s;
	    dist = (*it).second;
	    pen = penalty.get((*it).first.s);
	  }
      }
}

// returns true if PORT belongs to invertor
bool
truua::is_invertor (tid port)
{
  tidmap::iterator it;

  if (((it = IPSI.find (port)) != IPSI.end ())
      && (INV.find ((*it).second) != INV.end ()))
    return true;

  return false;
}

// returns true if in RO exists pair PAIR with sequence SEQ
bool truua::exists_ro(tpair pair, tsequence seq)
{
  tro::iterator it_ro;
  
  if (RO.find(pair) != RO.end())
    for (it_ro = RO.lower_bound(pair); it_ro!=RO.upper_bound(pair); it_ro++)
     if ((*it_ro).second == seq)
       return true;
    
  return false;
}

// finds all I-paths
void
truua::findI (ostream & f)
{
  bool found;
  int d, iter, prog_cnt;
  tid u;
  tpair pair1, pair2, pair3;
  tsequence seq;
  telset A, S;
  telset::iterator s, x;
  tro::iterator it_ro;
  tdelta::iterator it_d;
  tsequence::iterator it_seq;
  tpairset::iterator it_m;
  pair<tro::iterator, tro::iterator> rng;

  I.clear ();
  II.clear ();
  DELTA.clear ();
  RO.clear ();
  A.insert (PI.begin (), PI.end ());
  A.insert (PO.begin (), PO.end ());
  A.insert (P.begin (), P.end ());
  prog_cnt = progress_step;
// Dijkstra init
  for (s = A.begin (); s != A.end (); s++)
    {
      for (x = A.begin (); x != A.end (); x++)
	{
	  pair1.f = *s;
	  pair1.s = *x;
	  if (C.find (pair1) != C.end ())
	    {
	      DELTA.insert (tdelta::value_type (pair1, 0));
	      seq.clear ();
	      seq.push_back (*s);
	      seq.push_back (*x);
	      RO.insert (tro::value_type (pair1, seq));
	      I.insert (pair1);
	    }
	  else if ((it_m = M.find (pair1)) != M.end ())
	    {
	      DELTA.insert (tdelta::value_type (pair1, 1));
	      seq.clear ();
	      seq.push_back (*s);
	      seq.push_back (*x);
	      RO.insert (tro::value_type (pair1, seq));
	      if (is_invertor ((*it_m).f))	// it is invertor
		{
		  II.insert (pair1);
		}
	      else		// it is not invertor
		{
		  I.insert (pair1);
		}
	    }
	  else
	    {
	      DELTA.insert (tdelta::value_type (pair1, INT_MAX));
	    }
	}
    }	
  for (s = A.begin (), iter = 1; s != A.end (); s++, iter++)
    {
      S = A;
      penalty.clear();
      while (!S.empty ())
	{
          S.erase (*s);
// finds gate u with smallest distance to gate s
	  get_smallest_delta (*s, S, u, d);
	  for (x = A.begin (); x != A.end (); x++)
	    {
	      pair1.f = *s;
	      pair1.s = u;
              if (RO.find(pair1) != RO.end())
	      for (it_ro = RO.lower_bound(pair1); it_ro!=RO.upper_bound(pair1); it_ro++)
		{		// for all ROs
		  pair2.f = u;	// pair2 = (u,p)
		  pair2.s = *x;
		  pair3.f = (*it_ro).second[(*it_ro).second.size () - 2];	// pair3 = (pn,u)
		  pair3.s = u;
		  found = false;
		  for (it_seq = (*it_ro).second.begin ();
		       (it_seq != (*it_ro).second.end ()) && !found; it_seq++)
		    if (*it_seq == *x)
		      found = true;
		  if ((u != *x) && (!found) &&
		      (((C.find (pair2) != C.end ())
			&& (M.find (pair3) != M.end ()))
		       || ((M.find (pair2) != M.end ())
			   && (C.find (pair3) != C.end ()))))
		    {
		      if (((II.find (pair1) != II.end ()) &&
			   (((it_m = M.find (pair2)) == M.end ())
			    || (!is_invertor ((*it_m).f)))) ||
			  // II, M
			  ((II.find (pair1) == II.end ()) &&
			   (((it_m = M.find (pair2)) != M.end ())
			    && (is_invertor ((*it_m).f)))))
			// I, invertor
			{
			  pair3.f = *s;	// pair3 = (s,p)
			  pair3.s = *x;
			  II.insert (pair3);
			}
		      else	// I, I | II, invertor
			{
			  pair3.f = *s;	// pair3 = (s,p)
			  pair3.s = *x;
			  I.insert (pair3);
			}
		      it_d = DELTA.find (pair1);
		      d = 1;
		      if (it_d != DELTA.end ())
			{
			  d = (*it_d).second;
			  it_d = DELTA.find (pair2);
			  if (it_d != DELTA.end ())
			    d += (*it_d).second;
			  else
			    d++;
			}
		      it_d = DELTA.find (pair3);
		      if ((it_d != DELTA.end ()) && (d < (*it_d).second))	// DELTA update
			DELTA.erase (pair3);
                      DELTA.insert (tdelta::value_type (pair3, d));
		      seq = (*it_ro).second;
		      seq.push_back (*x);
                      if (!exists_ro(pair3,seq))
		        RO.insert (tro::value_type (pair3, seq));
		    }
		}
	    }
          if (penalty.get(u) == (int)RO.count(pair1))              
  	    S.erase (u);
	  else
            penalty.set(u,RO.count(pair1));
	}

// show progress
      while ((f != NULL) && (((float) iter / A.size () * 100) >= prog_cnt))
	{
	  f << prog_cnt << "% " << flush;
	  prog_cnt += progress_step;
	}
    }
  prog_cnt -= progress_step;
  if ((f != NULL) && (prog_cnt != 100))
    f << "100%" << endl;
}

// gets PSI for element <reg>, filter it with <filter> and return
// results in <gates>
void
truua::get_psi (int reg, telset & filter, telset & gates)
{
  tidmap::iterator it;

  gates.clear ();
  for (it = IPSI.begin (); it != IPSI.end (); it++)
    if (((*it).second == reg) && (filter.find ((*it).first) != filter.end ()))
      gates.insert ((*it).first);
}

// check M-path conditions
bool
truua::check_Mcond (int first, int second, string cond)
{
  string f, s, elt;
  tgnames::iterator it_gn, it_elt;
  tidmap::iterator it_ix1, it_ix2;
  tvhc::iterator it_vhc;
  tmsv::iterator it_msv;
  tmset::iterator it_ms;

  if ((it_gn = GN.find (first)) == GN.end ())
    return false;
  f = (*it_gn).second;
  if ((it_gn = GN.find (second)) == GN.end ())
    return false;
  s = (*it_gn).second;
  if ((it_ix1 = IPSI.find (first)) == IPSI.end ())
    return false;
  if ((it_ix2 = IPSI.find (second)) == IPSI.end ())
    return false;
  if ((*it_ix1).second != (*it_ix2).second)
    return false;
  if ((it_elt = ELT.find ((*it_ix1).second)) == ELT.end ())
    return false;
  elt = (*it_elt).second;
  if ((it_vhc = vhc.find (elt)) == vhc.end ())
    return false;
  for (it_msv = (*it_vhc).second.msv.begin ();
       it_msv != (*it_vhc).second.msv.end (); it_msv++)
    if (((*it_msv).from == f) && ((*it_msv).to == s))
      for (it_ms = (*it_msv).mset.begin (); it_ms != (*it_msv).mset.end ();
	   it_ms++)
	if ((*it_ms).second == cond)
	  return true;
  return false;
}

// checks if M-path is edge free (i.e. no register on path)
bool
truua::check_Medgefree_path (tpair & pair)
{
  bool found;
  int cnt, cnt2, x;
  tpair pair2;
  tro::iterator it_ro;

  if (I.find (pair) == I.end ())
    return false;

  for (cnt = 0; cnt < (int) RO.count (pair); cnt++)	// for all ROs
    {
      it_ro = RO.find (pair);
      cnt2 = cnt;
      while (cnt2--)
	it_ro++;
      found = true;
      for (x = 0; (x < (int) ((*it_ro).second.size () - 1)) && found; x++)
	{
	  pair2.f = (*it_ro).second[x];
	  pair2.s = (*it_ro).second[x + 1];
	  if (M.find (pair2) != M.end ())
	    if (check_Mcond (pair2.f, pair2.s, c_edgeup) ||
		check_Mcond (pair2.f, pair2.s, c_edgedn))
	      found = false;
	}
      if (found)
	return true;
    }

  return false;
}

// finds TDR registers for all input gates
void
truua::findTDR (void)
{
  tid el = 0;
  tpair pair;
  telset Q, tdr;
  telset::iterator it_ip, it_r, it_q;
  tidmap::iterator it_psi;

  TDR.clear ();
  for (it_ip = INP.begin (); it_ip != INP.end (); it_ip++)	// for all ip gates
    {
      if ((it_psi = IPSI.find (*it_ip)) != IPSI.end ())
	el = (*it_psi).second;
      for (it_r = R.begin (); it_r != R.end (); it_r++)	// check all regs
	if (*it_r != el)	// not for same reg as ip
	  {
	    get_psi (*it_r, OUTP, Q);
	    for (it_q = Q.begin (); it_q != Q.end (); it_q++)	// for all q
	      {
		pair.f = *it_q;
		pair.s = *it_ip;
		if (check_Medgefree_path (pair))
		  {
		    if (TDR.find (*it_ip) == TDR.end ())
		      TDR.insert (tidsetmap::value_type (*it_ip, tdr));
		    (*(TDR.find (*it_ip))).second.insert (*it_r);
		  }
	      }
	  }
    }
  make_union (TDR, ATDR);	// ATDR creation
}

// checks if in pairset <p> exists pair with given criteria
bool
truua::exists_pair (tid f, telset & s, tpairset & p)
{
  tpair pair;

  for (telset::iterator it_s = s.begin (); it_s != s.end (); it_s++)
    {
      pair.f = f;
      pair.s = *it_s;
      if (p.find (pair) != p.end ())
	return true;
    }

  return false;
}

// find all TRVs for all output gates
void
truua::findTRV (void)
{
  tid el = 0;
  tpair pair;
  telset Q, trv;
  telset::iterator it_op, it_r, it_q;
  tidmap::iterator it_psi;

  TRV.clear ();
  for (it_op = OUTP.begin (); it_op != OUTP.end (); it_op++)	// for all op gates
    {
      if ((it_psi = IPSI.find (*it_op)) != IPSI.end ())
	el = (*it_psi).second;
      for (it_r = R.begin (); it_r != R.end (); it_r++)	// check all regs
	if (*it_r != el)	// not for same reg as ip
	  {
	    get_psi (*it_r, INP, Q);
	    for (it_q = Q.begin (); it_q != Q.end (); it_q++)	// for all q
	      {
		pair.f = *it_op;
		pair.s = *it_q;
		if (check_Medgefree_path (pair)
		    && !exists_pair (*it_op, PO, C))
		  {
		    if (TRV.find (*it_op) == TRV.end ())
		      TRV.insert (tidsetmap::value_type (*it_op, trv));
		    (*(TRV.find (*it_op))).second.insert (*it_r);
		  }
	      }
	  }
    }
  make_union (TRV, ATRV);	// ATRV creation
}

// creates union <UNION> from parts <PARTS> (f.e All Test DRivers set)
void
truua::make_union (tidsetmap & PARTS, telset & UNION)
{
  UNION.clear ();
  for (tidsetmap::iterator it_tdr = PARTS.begin (); it_tdr != PARTS.end ();
       it_tdr++)
    for (telset::iterator it_r = (*it_tdr).second.begin ();
	 it_r != (*it_tdr).second.end (); it_r++)
      UNION.insert (*it_r);
}

// tests if given gate <gate> exists as target (second) gate in M
bool
truua::gate_issecond_inM (tid gate)
{
  for (tpairset::iterator it = M.begin (); it != M.end (); it++)
    if (gate == (*it).s)
      return true;

  return false;
}

// finds all TIRs for all TDRs
void
truua::findTIR (void)
{
  bool found;
  tpair pair;
  telset Q1, Q2, Q3, tir;
  telset::iterator it_tdr, it_r, it_q1, it_q2, it_d;
  tpairset::iterator it_i;

  TIR.clear ();
  for (it_tdr = ATDR.begin (); it_tdr != ATDR.end (); it_tdr++)	// for all tdr
    {
      for (it_r = R.begin (); it_r != R.end (); it_r++)	// check all regs
	{
	  get_psi (*it_tdr, INP, Q1);
	  get_psi (*it_r, OUTP, Q2);
	  get_psi (*it_r, INP, Q3);
	  for (it_q1 = Q1.begin (); it_q1 != Q1.end (); it_q1++)	// for all q1
	    for (it_q2 = Q2.begin (); it_q2 != Q2.end (); it_q2++)	// for all q2
	      for (it_d = Q3.begin (); it_d != Q3.end (); it_d++)	// for all d
		{
		  pair.f = *it_q2;
		  pair.s = *it_q1;
		  if ((I.find (pair) != I.end ()) || (*it_r == *it_tdr))
		    {
		      found = false;
		      for (it_i = I.begin (); (it_i != I.end ()) && !found;
			   it_i++)
			if (((*it_i).s == *it_d) && ((*it_i).f != *it_d)
			    && ((PI.find ((*it_i).f) != PI.end ())
				|| ((OUTP.find ((*it_i).f) != OUTP.end ())
				    && !gate_issecond_inM ((*it_i).f))))
			  {
			    pair.f = (*it_i).f;
			    pair.s = (*it_i).s;
			    if (check_Medgefree_path (pair))
			      found = true;
			  }
		      if (found)
			{
			  if (TIR.find (*it_tdr) == TIR.end ())
			    {
			      tir.clear ();
			      TIR.
				insert (tidsetmap::value_type (*it_tdr, tir));
			    }
			  (*(TIR.find (*it_tdr))).second.insert (*it_r);
			}
		    }
		}
	}
      if (TIR.find (*it_tdr) == TIR.end ())
	{
	  tir.clear ();
	  tir.insert (*it_tdr);
	  TIR.insert (tidsetmap::value_type (*it_tdr, tir));
	}
    }
  make_union (TIR, ATIR);	// ATIR creation
}

// tests if given gate <gate> exist as source (first) gate in M
bool
truua::gate_isfirst_inM (tid gate)
{
  for (tpairset::iterator it = M.begin (); it != M.end (); it++)
    if (gate == (*it).f)
      return true;

  return false;
}

// find all TORs for all TRVs
void
truua::findTOR (void)
{
  bool found;
  tpair pair;
  telset Q1, Q2, Q3, tor;
  telset::iterator it_trv, it_r, it_q1, it_q2, it_d;
  tpairset::iterator it_i;

  TOR.clear ();
  for (it_trv = ATRV.begin (); it_trv != ATRV.end (); it_trv++)	// for all trv
    {
      for (it_r = R.begin (); it_r != R.end (); it_r++)	// check all regs
	{
	  get_psi (*it_trv, OUTP, Q1);
	  get_psi (*it_r, INP, Q2);
	  get_psi (*it_r, OUTP, Q3);
	  for (it_q1 = Q1.begin (); it_q1 != Q1.end (); it_q1++)	// for all q1
	    for (it_q2 = Q2.begin (); it_q2 != Q2.end (); it_q2++)	// for all q2
	      for (it_d = Q3.begin (); it_d != Q3.end (); it_d++)	// for all d
		{
		  pair.f = *it_q1;
		  pair.s = *it_q2;
		  if ((I.find (pair) != I.end ()) || (*it_r == *it_trv))
		    {
		      found = false;
		      for (it_i = I.begin (); (it_i != I.end ()) && !found;
			   it_i++)
			if (((*it_i).f == *it_d) && ((*it_i).s != *it_d)
			    && ((PO.find ((*it_i).s) != PO.end ())
				|| ((INP.find ((*it_i).s) != INP.end ())
				    && !gate_isfirst_inM ((*it_i).s))))
			  {
			    pair.f = (*it_i).f;
			    pair.s = (*it_i).s;
			    if (check_Medgefree_path (pair))
			      found = true;
			  }
		      if (found)
			{
			  if (TOR.find (*it_trv) == TOR.end ())
			    {
			      tor.clear ();
			      TOR.
				insert (tidsetmap::value_type (*it_trv, tor));
			    }
			  (*(TOR.find (*it_trv))).second.insert (*it_r);
			}
		    }
		}
	}
      if (TOR.find (*it_trv) == TOR.end ())
	{
	  tor.clear ();
	  tor.insert (*it_trv);
	  TOR.insert (tidsetmap::value_type (*it_trv, tor));
	}
    }
  make_union (TOR, ATOR);	// ATOR creation
}

// creates set of all controllable TIRs
void
truua::makeCTR (void)
{
  bool found;
  tpair pair;
  telset Q;
  telset::iterator it_tir, it_q, it_pi;

  CTR.clear ();
  for (it_tir = ATIR.begin (); it_tir != ATIR.end (); it_tir++)
    {
      get_psi (*it_tir, INP, Q);
      found = false;
      for (it_q = Q.begin (); it_q != Q.end (); it_q++)
	for (it_pi = PI.begin (); (it_pi != PI.end ()) && !found; it_pi++)	// for all pi
	  {
	    pair.f = *it_pi;
	    pair.s = *it_q;
	    if (I.find (pair) != I.end ())
	      {
		found = true;
		CTR.insert (*it_tir);
	      }
	  }
    }
  NCTR = get_diffset (ATIR, CTR);	// NCTR creation
}

// creates set of all observable TOR
void
truua::makeOBS (void)
{
  bool found;
  tpair pair;
  telset Q;
  telset::iterator it_tor, it_q, it_po;

  OBS.clear ();
  for (it_tor = ATOR.begin (); it_tor != ATOR.end (); it_tor++)
    {
      get_psi (*it_tor, OUTP, Q);
      found = false;
      for (it_q = Q.begin (); it_q != Q.end (); it_q++)
	for (it_po = PO.begin (); (it_po != PO.end ()) && !found; it_po++)	// for all po
	  {
	    pair.f = *it_q;
	    pair.s = *it_po;
	    if (I.find (pair) != I.end ())
	      {
		found = true;
		OBS.insert (*it_tor);
	      }
	  }
    }
  NOBS = get_diffset (ATOR, OBS);
}

// computes difference of two sets
telset
truua::get_diffset (telset & S1, telset & S2)
{
  telset S;

  for (telset::iterator it = S1.begin (); it != S1.end (); it++)
    if (S2.find (*it) == S2.end ())
      S.insert (*it);

  return S;
}

// creates set of all controllable TDRs
void
truua::makeCTDR (telset & LATDR, tidsetmap & LTIR, telset & LCTR,
		 telset & LCTDR)
{
  bool found;
  telset::iterator it_tdr, it_r;
  tidsetmap::iterator it_tir;

  LCTDR.clear ();
  for (it_tdr = LATDR.begin (); it_tdr != LATDR.end (); it_tdr++)
    if ((it_tir = LTIR.find (*it_tdr)) != LTIR.end ())
      {
	found = false;
	for (it_r = (*it_tir).second.begin ();
	     (it_r != (*it_tir).second.end ()) && !found; it_r++)
	  if (LCTR.find (*it_r) != LCTR.end ())
	    {
	      found = true;
	      LCTDR.insert (*it_tdr);
	    }
      }
}

// converts gate name <gn> of element <el> to corresponding id <tid>
tid
truua::gn2id (tid el, string gn)
{
  telset::iterator it_gate;
  tgnames::iterator it_map;
  tidsetmap::iterator it_psi;

  if ((it_psi = PSI.find (el)) == PSI.end ())
    return (tid) - 1;
  for (it_map = GN.begin (); it_map != GN.end (); it_map++)
    if (((*it_map).second == gn) &&
	((it_gate = (*it_psi).second.find ((*it_map).first)) !=
	 (*it_psi).second.end ()))
      return *it_gate;
  return (tid) - 1;
}

// tries to substitute gate name included as part of <str> to corresponding 
// gate id of element <el> and returns result in <string>
string
truua::subst_mset (tid el, string str)
{
  int y;
  size_t x;
  string str2;
  char s[pnsize];

  if ((str == c_edgeup) || (str == c_edgedn))
    return str;
  if (((x = str.find_last_of ('=')) == string::npos) &&
      ((x = str.find_last_of ('<')) == string::npos) &&
      ((x = str.find_last_of ('>')) == string::npos))
    return str;
  if ((y = gn2id (el, str.substr ((int) x + 1))) != -1)
    {
      sprintf (s, "%d", y);
      str2 = str.substr (0, x + 1) + "[" + string (s) + "]";
      return str2;
    }
  return str;
}

// computes MI set
void
truua::makeMI (void)
{
  tmset::iterator it_mset;
  tmsv::iterator it_msv;
  tvhc::iterator it_vhc;
  telset::iterator it_el;
  tgnames::iterator it_elt;
  tmiel miel;

  MI.clear ();
  for (it_el = E.begin (); it_el != E.end (); it_el++)	// for all els
    {
      if ((it_elt = ELT.find ((tid) * it_el)) == ELT.end ())	// gets type
	return;
      if ((it_vhc = vhc.find ((*it_elt).second)) == vhc.end ())	// vhc for type
	return;
      for (it_msv = (*it_vhc).second.msv.begin ();
	   it_msv != (*it_vhc).second.msv.end (); it_msv++)
	{
	  miel.clear ();
	  for (it_mset = (*it_msv).mset.begin ();
	       it_mset != (*it_msv).mset.end (); it_mset++)
	    miel.
	      insert (tidstrpair
		      (gn2id (*it_el, (*it_mset).first),
		       subst_mset (*it_el, (*it_mset).second)));
	  MI.
	    insert (tmi::
		    value_type (tpair
				(gn2id (*it_el, (*it_msv).from),
				 gn2id (*it_el, (*it_msv).to)), miel));
	}
    }
}

// computes score for decision which registers add to scan
void
truua::makeSCORE (void)
{
  int cnt;
  telset::iterator it_nctr;
  tidsetmap::iterator it_tir;

  SCORE.clear ();
  for (it_nctr = NCTR.begin (); it_nctr != NCTR.end (); it_nctr++)
    {
      cnt = 0;
      for (it_tir = TIR.begin (); it_tir != TIR.end (); it_tir++)
	if (((*it_tir).second.find (*it_nctr) != (*it_tir).second.end ()) &&
	    (CTDR.find ((*it_tir).first) == CTDR.end ()))
	  cnt++;
      SCORE.insert (tscore::value_type (*it_nctr, cnt));
    }
}

// gets maximal score
int
truua::get_maxscore (void)
{
  int max;
  tscore::iterator it_sc, it_max;

  if (SCORE.size () == 0)
    return 0;
  if (SCORE.size () == 1)
    return (*(SCORE.begin ())).first;
  it_max = SCORE.begin ();
  max = (*it_max).second;
  for (it_sc = SCORE.begin (); it_sc != SCORE.end (); it_sc++)
    if ((*it_sc).second > max)
      {
	it_max = it_sc;
	max = (*it_sc).second;
      }
  return (*it_max).first;
}

// makes given TDRs controllable
void
truua::addTDRs (tid el, tidsetmap & LTIR, telset & LCTDR)
{
  tidsetmap::iterator it_tir;

  for (it_tir = LTIR.begin (); it_tir != LTIR.end (); it_tir++)
    if ((*it_tir).second.find (el) != (*it_tir).second.end ())
      LCTDR.insert ((*it_tir).first);
}

// computes scan set
void
truua::makeSCAN ()
{
  tid el;
  tidsetmap::iterator it_tir, it_tor;

  SCAN.clear ();
// controllability
  for (it_tir = TIR.begin (); it_tir != TIR.end (); it_tir++)
    if (((*it_tir).second.size () == 1) &&
	(NCTR.find (*((*it_tir).second.begin ())) != NCTR.end ()))
      {
	el = *((*it_tir).second.begin ());
	SCAN.insert (el);
	addTDRs (el, TIR, CTDR);
	CTR.insert (el);
	NCTR.erase (el);
      }
  while (CTDR.size () < ATDR.size ())
    {
      makeSCORE ();
      el = get_maxscore ();
      SCAN.insert (el);
      CTR.insert (el);
      NCTR.erase (el);
      addTDRs (el, TIR, CTDR);
    }
// observability
  for (it_tor = TOR.begin (); it_tor != TOR.end (); it_tor++)
    if (((*it_tor).second.size () == 1) &&
	(NOBS.find (*((*it_tor).second.begin ())) != NOBS.end ()))
      {
	el = *((*it_tor).second.begin ());
	SCAN.insert (el);
	addTORs (el, TOR, OTRV);
	OBS.insert (el);
	NOBS.erase (el);
      }
  while (OTRV.size () < ATRV.size ())
    {
      makeSCORE ();
      el = get_maxscore ();
      SCAN.insert (el);
      OBS.insert (el);
      NOBS.erase (el);
      addTORs (el, TOR, OTRV);
    }
}

// simple testability check
tid
truua::check_testable (void)
{
  bool found;
  tpair pair;
  tidmap::iterator it_ipsi;
  telset::iterator it_i, it_p;

// test if controllable
  for (it_i = INP.begin (); it_i != INP.end (); it_i++)
    {
      it_ipsi = IPSI.find (*it_i);
      if (it_ipsi != IPSI.end ())
	found = (R.find ((*it_ipsi).second) != R.end ());
      else
	found = false;
      if (!found && (TDR.find (*it_i) == TDR.end ()))
	{
	  found = false;
	  for (it_p = PI.begin (); (it_p != PI.end ()) && !found; it_p++)
	    {
	      pair.f = *it_p;
	      pair.s = *it_i;
	      if (I.find (pair) != I.end ())
		found = true;
	    }
	  if (!found)
	    return *it_i;
	}
    }

// test if observable
  for (it_i = OUTP.begin (); it_i != OUTP.end (); it_i++)
    {
      it_ipsi = IPSI.find (*it_i);
      if (it_ipsi != IPSI.end ())
	found = (R.find ((*it_ipsi).second) != R.end ());
      else
	found = false;
      if (!found && (TRV.find (*it_i) == TRV.end ()))
	{
	  found = false;
	  for (it_p = PO.begin (); (it_p != PO.end ()) && !found; it_p++)
	    {
	      pair.f = *it_i;
	      pair.s = *it_p;
	      if (I.find (pair) != I.end ())
		found = true;
	    }
	  if (!found)
	    return *it_i;
	}
    }
  return 0;
}
