/*
  Miscellaneous utilities
  Copyright (C) 2007 Jaroslav Skarvada <skarvada@fit.vutbr.cz>  
  
  This program is free software: you can redistribute it and/or modify  
  it under the terms of the GNU General Public License as published by  
  the Free Software Foundation, either version 3 of the License, or  
  (at your option) any later version.  
  
  This program is distributed in the hope that it will be useful,  
  but WITHOUT ANY WARRANTY; without even the implied warranty of  
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
  GNU General Public License for more details.  
 
  You should have received a copy of the GNU General Public License  
  along with this program.  If not, see <http://www.gnu.org/licenses/>.  
*/

#ifndef istring
#include <string>
#define istring
#endif

#ifndef iiostream
#include <iostream>
#define iiostream
#endif

#ifndef ifstream
#include <fstream>
#define ifstream
#endif

#ifndef ictime
#include <ctime>
#define ictime
#endif

#ifndef itimeb
#include <sys/timeb.h>
#define itimeb
#endif

#ifndef isetup
#include "setup.h"
#define isetup
#endif

#ifndef iutils
#include "utils.h"
#define iutils
#endif

using namespace std;

// tests filename <fname> for existence
bool
exists_file (string fname)
{
  bool file_exist;
  fstream f;

  f.open (fname.c_str (), ios::in);
  file_exist = (bool) (f);
  if (file_exist)
    f.close ();

  return file_exist;
}

// returns upstring of string <str>
string
upstring (string str)
{
  uint x;
  string str2;

  str2.erase ();
  for (x = 0; x < str.length (); x++)
    str2 += (char) toupper (str[x]);
  return str2;
}

// try to skip token <tok> in stream <f>
// returns true if skipped, false otherwise
bool
skip_token (fstream & f, char *tok)
{
  char c;
  ios::fmtflags flags;

  flags = f.flags ();
  f.flags (flags | ios::skipws);	// skip whitespaces
  f >> c;
  f.flags (flags & ~ios::skipws);	// proceed all chars
  while (f && (c == *tok) && (*tok++ != 0))	// skip
    f >> c;
  if (f)			// putback last
    f.putback (c);
  else
    return false;
  return (*tok == 0);
}

// try to skip comment starting with <comment_str> and ending
// with EOLN in stream <f> returns true if skipped, false otherwise
bool
skip_comment (fstream & f, char *comment_str)
{
  char c;
  ios::fmtflags flags;

  flags = f.flags ();
  f.flags (flags | ios::skipws);	// skip whitespaces
  f >> c;
  f.flags (flags & ~ios::skipws);	// proceed all chars
  if (f && (c != *comment_str))
    {
      f.putback (c);
      return false;
    }
  while (f && (c == *comment_str) && (*comment_str++ != 0))	// skip
    f >> c;
  while (f && (c != '\n'))	// skip line
    f >> c;
  return (*comment_str == 0);
}

// try to skip line in stream <f> returns true if skipped, false otherwise
bool
skip_line (fstream & f)
{
  char c;

  c = 0;
  while (f && (c != '\n'))	// skip line
    f >> c;
  return f;
}

// scan for line starting with string <line> in stream <f>
// returns true if found, false otherwise
bool
find_line (fstream & f, char *line)
{
  char c;
  char *tok;
  ios::fmtflags flags;

  do
    {
      tok = line;
      flags = f.flags ();
      f.flags (flags | ios::skipws);	// skip whitespaces
      f >> c;
      f.flags (flags & ~ios::skipws);	// proceed all chars
      while (f && (c == *tok) && (*tok++ != 0))	// skip
	f >> c;
      if (f)			// putback last
	f.putback (c);
      else
	return false;
      if (*tok != 0)
	while (f && (c != '\n'))	// skip line
	  f >> c;
    }
  while (f && (*tok != 0));
  return f;
}

// reads string in double quotes of max length <len> to <str> from stream <f>
// returns true if OK, false otherwise
bool
get_str (fstream & f, char *str, int len)
{
  char c;
  ios::fmtflags flags;

  len--;
  flags = f.flags ();
  f.flags (flags | ios::skipws);
  f >> c;
  f.flags (flags & ~ios::skipws);
  if (!f || (c != '\"'))	// starting quote
    return false;
  f >> c;
  while (f && len-- && (c != '\"'))	// string
    {
      *str++ = c;
      f >> c;
    }
  *str = 0;
  if (!f || (c != '\"'))	// ending quote
    return false;

  return true;
}

// gets single quoted char from stream <f> to <chr>
// returns true if OK, false otherwise
bool
get_qchar (fstream & f, char &chr)
{
  char c;
  ios::fmtflags flags;

  flags = f.flags ();
  f.flags (flags | ios::skipws);	// skip whitespaces
  f >> c;
  f.flags (flags & ~ios::skipws);	// proceed all chars
  if (!f || (c != '\''))
    return false;
  f >> c;
  if (!f)
    return false;
  if (c == '\'')
    return true;
  chr = c;
  f >> c;
  if (!f || (c != '\''))
    return false;

  return true;
}

// gets one char from stream <f> to <tok>
// returns true if OK, false otherwise
bool
get_tok (fstream & f, char &tok)
{
  ios::fmtflags flags;

  flags = f.flags ();
  f.flags (flags | ios::skipws);
  f >> tok;
  return f;
}

// gets string of max length <len> from stream <f> to <str> delimited
// with <delim>
// returns true if OK, false otherwise
bool
get_data (fstream & f, char *str, int len, char delim)
{
  char c;
  ios::fmtflags flags;

  len--;
  flags = f.flags ();
  f.flags (flags | ios::skipws);	// skip whitespaces
  f >> c;
  f.flags (flags & ~ios::skipws);	// process all chars
  while (f && len-- && (c != delim))	// get string
    {
      *str++ = c;
      f >> c;
    }
  *str = 0;
  if (!f || (c != delim))
    return false;

  return true;
}

// gets string of max length <len> from stream <f> to <str> delimited
// with any character from <delim>
// returns true if OK, false otherwise
bool
get_data2 (fstream & f, char *str, int len, tcharset delim)
{
  char c;
  ios::fmtflags flags;

  len--;
  flags = f.flags ();
  f.flags (flags | ios::skipws);	// skip whitespaces
  f >> c;
  f.flags (flags & ~ios::skipws);	// process all chars
  while (f && len-- && (delim.find (c) == delim.end ()))	// get string
    {
      *str++ = c;
      f >> c;
    }
  *str = 0;
  if (!f || (delim.find (c) == delim.end ()))
    return false;

  return true;
}

// gets time delta, current time from <t1>, returns seconds in <tds> and
// milliseconds in <tdms>
void
get_tdelta (timeb t1, int &tds, int &tdms)
{
  timeb t2;

  ftime (&t2);
  tds = (int) t2.time - t1.time;
  tdms = (int) (t2.millitm - t1.millitm);
  if (tdms < 0)
    {
      tds--;
      tdms += 1000;
    }
}

// displays time delta seconds <tds> and milliseconds <tdms> to stream <f>
void
disp_tdelta (ostream & f, int tds, int tdms)
{
  int tdm;

  if (tds >= 60)
    {
      tdm = tds / 60;
      tds %= 60;
      f << tdm << " minute";
      if (tdm > 1)
	f << "s";
      f << ", ";
    }
  if (tds > 0)
    {
      f << tds << " second";
      if (tds > 1)
	f << "s";
      f << ", ";
    }
  f << tdms << " millisecond";
  if (tdms > 1)
    f << "s";
}
