/*
   Zb2Ruz:   RTL digital circuit models converter from Zboril's format to
             Ruzicka's formal model and partial scan computation utility

   Version:  1.0a

   Coded by: Yarda  <skarvada@fit.vutbr.cz>
   UPSY FIT VUT 2004

   License: Public domain, but please don't use for terrorists activities
   Main program
*/

#ifndef icstdlib
#include <cstdlib>
#define icstdlib
#endif

#ifndef icstdio
#include <cstdio>
#define icstdio
#endif

#ifndef iclimits
#include <climits>
#define iclimits
#endif

#ifndef imap
#include <map>
#define imap
#endif

#ifndef iset
#include <set>
#define iset
#endif

#ifndef ivector
#include <vector>
#define ivector
#endif

#ifndef istring
#include <string>
#define istring
#endif

#ifndef iiostream
#include <iostream>
#define iiostream
#endif

#ifndef ifstream
#include <fstream>
#define ifstream
#endif

#ifndef itimeb
#include <sys/timeb.h>
#define itimeb
#endif

#ifndef isetup
#include "setup.h"
#define isetup
#endif 

#ifndef ierror
#include "error.h"
#define ierror
#endif

#ifndef iutils
#include "utils.h"
#define iutils
#endif

#ifndef iruzic
#include "ruzic.h"
#define iruzic
#endif 

// some aliases
#define makeOTRV makeCTDR

using namespace std;

bool helpf;                                      // help flag
int tds,tdms,nvhc,nprv,nbrn,nexc;                // counters
tid cnt;
terr err;                                        // global error states holder
timeb tme,tmea;                                  // timers
truua uua;                                       // Unit Under Analysis

// diplays help
void help(void)
{
  cout<<
  "zb2ruz - RTL digital circuit models converter from Zboril's format to"
  <<endl<<
  "         Ruzicka's formal model and partial scan computation utility v1.0a"
  <<endl<<
  "         by JAR"<<endl<<endl<<
  "usage: zb2ruz [OPTIONS] FNAME"<<endl<<endl<<
  "           FNAME  base filename for analysis without extension"<<endl<<
  "                  these input files must exists: {<FNAME"<<prv_suffix<<
  ">, <FNAME"<<brn_suffix<<">,"<<endl<<
  "                  <FNAME"<<spj_suffix<<">, <FNAME"<<spi_suffix<<
  ">} = RTL circuit description files"<<endl<<
  "                  <FNAME"<<vhc_suffix<<"> = M modes description file"<<endl<<
  "                  and if you want to compute SCAN chain, there must be also:"
  <<endl<<
  "                  <FNAME"<<exc_suffix<<"> = control gates description file"
  <<endl<<endl<<
  "                  output file <FNAME"<<ruz_suffix<<
  "> will be created, this file includes"<<endl<<
  "                  complete formal model due to Ruzicka's specification"
  <<endl<<
  "                  for more details please see supplied documentation"<<endl<<
  "                  output filename can be overrided by -o option, see below"
  <<endl<<endl<<
  "options: -o NAME  overrides output filename to NAME"<<endl<<
  "         -help    shows this help"<<endl;
}

// gets parameters from command line
int get_params(int argc,char *argv[],string &fname,string &foname,bool &helpf)
{
  bool read,readname;
  int x;
  char *p=NULL;
  string s;

  readname=false;
  helpf=false;
  if (argc==1)                                   // no parameters
  {
    helpf=true;
    return e_ok;
  }
  for (x=1;x<argc;x++)
  {
    read=false;
    if (argv[x][0]=='/')                         // parameter can start with /
      p=argv[x]+sizeof(char);
    else if (argv[x][0]=='-')                    // or can start with -
    {  
      p=argv[x]+sizeof(char);
      if ((strlen(p)>0)&&(p[0]=='-'))            // or can start with double -
        p+=sizeof(char);
    }
    else
    {
      p=argv[x];
      fname=p;                                  // else it is filename
      foname=fname+ruz_suffix;
      read=true;
      readname=true;
    }
// parameters parser
    if (!read)
    {
      s=upstring(string(p));                   // case insensitive
      if (s=="HELP")                           // help
      {
        helpf=true;
        return e_ok;
      }
      else if (s=="O")                         // output override
      {
        x++;
        if (x<argc)
        {
          foname=argv[x];
          if (foname.find('.')==string::npos)  // add default extension    
            foname+=ruz_suffix;
        }
        else return e_invparam;  
      }
      else return e_invparam;
    }
  }
  if (!readname)
    return e_invparam;

  return e_ok;
}

// main program
int main(int argc, char *argv[])
{
// get parameters from command line
  uua.clear();
  if ((bool)(err=get_params(argc,argv,uua.fname,uua.foname,helpf)))
  {
    cerr<<GetErrStr(err)<<endl;
    return err;
  }
  if (helpf)
  {
    help();
    return e_ok;
  }
  ftime(&tmea);                                  // start time
  cout<<"Starting "<<uua.fname<<" analysis:"<<endl<<endl;
  nexc=0;
// parse vhc
  cout<<"VHC file = ("<<uua.fname<<vhc_suffix<<")"<<endl;
  cout<<"Parsing VHC file...                                         "<<flush;
  if ((bool)(err=uua.parsevhc(uua.fname+vhc_suffix,nvhc)))
  {
    cout<<endl;
    cerr<<GetErrStr(err)<<endl;
    return err;
  }
  cout<<"<Done>"<<endl;
  cout<<"Parsed lines: "<<nvhc<<endl<<endl;

// parse exc
  cout<<"Looking for EXC file = ("<<uua.fname<<exc_suffix<<")...  "<<flush;
  if (exists_file(uua.fname+exc_suffix))
  {
    cout<<"found"<<endl;
    cout<<"Parsing EXC file...                                         "<<flush;
    if ((bool)(err=uua.parseexc(uua.fname+exc_suffix)))
    {
      cout<<endl;
      cerr<<GetErrStr(err)<<endl;
      return err;
    }
    cout<<"<Done>"<<endl;
    cout<<"EXC elements read: "<<uua.exc.size()<<endl<<endl;
  }
  else
    cout<<"NOT found"<<endl<<
    "!WARNING! CONTROL GATES SPECIFICATION FILE (EXC) MISSING, YOU WILL"<<endl<<
    "PROBABLY GET IMPROPER RESULTS"<<endl<<endl;

// load uua
  cout<<"UUA file = ("<<uua.fname<<prv_suffix<<", "<<uua.fname<<brn_suffix<<
  ", "<<uua.fname<<spj_suffix<<", "<<uua.fname<<spi_suffix<<")"<<endl;
  cout<<"Loading UUA files...                                        "<<flush;
  if ((bool)(err=uua.LoadUUA(uua.fname+prv_suffix,uua.fname+brn_suffix,
                             uua.fname+spj_suffix,uua.fname+spi_suffix,nprv,
                             nbrn)))
  {
    cout<<endl;
    cerr<<GetErrStr(err)<<endl;
    return err;
  }
  cout<<"<Done>"<<endl;
  cout<<"UUA ID                : "<<uua.fid<<endl;
  cout<<"UUA Date              : "<<uua.fdate<<endl;

// check validity
  if ((bool)(err=uua.check_validity()))
  {
    cout<<endl;
    cerr<<GetErrStr(err)<<endl;
    return err;
  }
  cout<<"Components read       : "<<nprv<<endl;
  cout<<"Number of connections : "<<uua.C.size()<<endl;
  cout<<"Number of registers   : "<<uua.R.size()<<endl;
  cout<<"Number of PI gates    : "<<uua.PI.size()<<endl;
  cout<<"Number of PO gates    : "<<uua.PO.size()<<endl;
  cout<<"Number of IN gates    : "<<uua.INP.size()<<endl;
  cout<<"Number of OUT gates   : "<<uua.OUTP.size()<<endl;
  cout<<"Number of CI gates    : "<<uua.CI.size()<<endl;
  cout<<"Number of all gates   : "<<uua.P.size()+uua.PI.size()+uua.PO.size()<<
  endl<<endl;
// check gates boundess
  cout<<"Checking gates boundness..."<<endl;
  if ((bool)(cnt=(tid)uua.check_ipunbound(uua.INP)))
    cout<<"!WARNING! THERE IS MINIMALY ONE UNBOUND INPUT GATE # "<<cnt<<
    endl<<endl;
  if ((bool)(cnt=(tid)uua.check_opunbound()))
    cout<<"!WARNING! THERE IS MINIMALY ONE UNBOUND OUTPUT GATE # "<<cnt<<
    endl<<endl;
  if ((bool)(cnt=(tid)uua.check_ipunbound(uua.CI)))
    cout<<"!WARNING! THERE IS MINIMALY ONE UNBOUND CONTROL GATE # "<<cnt<<
    endl<<endl;

// find I-paths
  cout<<"Finding I paths...  so far done:  "<<flush;
  ftime(&tme);                                   // local timer for statistic
  uua.findI(cout);
  get_tdelta(tme,tds,tdms);
  cout<<endl;
  cout<<"Found I paths : "<<uua.I.size()<<endl;
  cout<<"Found ROs     : "<<uua.RO.size()<<endl;
  cout<<"Found finished @ ";
  disp_tdelta(cout,tds,tdms);
  cout<<endl<<endl;

// find TDRs
  cout<<"Finding TDRs...                                             "<<flush;
  uua.findTDR();
  cout<<"<Done>"<<endl;
  cout<<"Found: "<<uua.ATDR.size()<<endl;
// find TRVs
  cout<<"Finding TRVs...                                             "<<flush;
  uua.findTRV();
  cout<<"<Done>"<<endl;
  cout<<"Found: "<<uua.ATRV.size()<<endl;
// find TIRs
  cout<<"Finding TIRs...                                             "<<flush;
  uua.findTIR();
  cout<<"<Done>"<<endl;
  cout<<"Found: "<<uua.ATIR.size()<<endl;
// find TORs
  cout<<"Finding TORs...                                             "<<flush;
  uua.findTOR();
  cout<<"<Done>"<<endl;
  cout<<"Found: "<<uua.ATOR.size()<<endl;

// make simple testability check
  if ((bool)(cnt=(tid)uua.check_testable()))
    cout<<"!WARNING! THERE IS MINIMALY ONE UNTESTABLE GATE # "<<cnt<<
    " - CIRCUIT MUST BE MODIFIED"<<endl<<
    "BY USER, SO NOW DOING UNCOMPLETE CALCULATIONS..."<<endl<<endl;

  cout<<"Computing auxiliary sets...                                 "<<flush;
  uua.makeMI();
  uua.makeCTR();
  uua.makeOBS();
  uua.makeCTDR(uua.ATDR,uua.TIR,uua.CTR,uua.CTDR);
  uua.makeOTRV(uua.ATRV,uua.TOR,uua.OBS,uua.OTRV);              // method aliased
  cout<<"<Done>"<<endl;

// compute scan
  cout<<"Computing SCAN...                                           "<<flush;
  uua.makeSCAN();
  cout<<"<Done>"<<endl;
  cout<<"Registers added to SCAN: "<<uua.SCAN.size()<<endl;
  cout<<endl;

// write results

  cout<<"Writing results...                                          "<<flush;
  if ((bool)(err=uua.WriteFormalUUA(uua.fname+ruz_suffix)))
  {
    cout<<endl;
    cerr<<GetErrStr(err)<<endl;
    return err;
  }
  cout<<"<Done>"<<endl<<endl;
  get_tdelta(tmea,tds,tdms);
  cout<<"Analysis finished @ ";
  disp_tdelta(cout,tds,tdms);                    // display whole time
  cout<<endl;
  return e_ok;
}

