/*
*	This file contains class that represents dispatcher between input (camera), frame queue
*	and working units.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/
#include <cassert>

#include "pipeline/Dispatcher.h"

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TDispatcher

/** Konstruktor.
*
*	\param	queue	[in] ukazatel na platnou frontu kam bude dispatcher predavat snimky
*					Pokud je tento parametr NULL pak se senimky z dispatcheru dale neposilaji.
*/
TDispatcher::TDispatcher( TFrameQueue * queue )
{
	m_width		= 0;
	m_height	= 0;

	m_frame = NULL;

	m_queue = queue;
}
//OK 2007-08-25 14:23:38 B04-315B\Tom

/** Destruktor.
*
*	Uvolni snimek z TDispatcher.
*/
TDispatcher::~TDispatcher(void)
{
	m_critical.Enter();
		//musime odstranit predchozi snimek - pokud existuje
		if( m_frame )
		{
			//existuje - mame nastaven zamek od tohoto objektu
			m_frame->ReleaseLock();
			//uvolnime
			m_frame->Release();

			m_frame = NULL;
		}
	m_critical.Leave();
}
//OK 2007-08-25 14:23:40 B04-315B\Tom

/** Pomoci teto metody se do dispatcheru nastavi informace o rozmerech
*	predavanych snimku.
*
*	Tyto informace jsou pro vsechny snimky stejne a mohou byt vyuzity napr. pri
*	pri inicializaci jednotek.
*
*	Tato metoda by mela byt volana ihned po zjisteni rozmeru snimku (z kamery).
*
*	\param	width	[in] sirka snimku v pixelech
*	\param	height	[in] vyska snimku v pixelech
*/
void TDispatcher::SetFramesInfo( DWORD width, DWORD height )
{
	m_width		= width;
	m_height	= height;
}
//OK 2007-08-25 14:23:42 B04-315B\Tom

/** Vlozeni noveho snimku.
*
*	Predchozi snimek je uvolnen (pokud existoval) a zaroven preposlan do 
*	fronty snimku (pokud byla nastavena).
*	Novy snimek je uzamcen a pocet referenci je zvysen.
*
*	\param	frame	[in] platny snimek - nesmi byt NULL
*/
void TDispatcher::SetFrame( TFrameReal * frame)
{
	m_critical.Enter();
		//musime odstranit predchozi snimek - pokud existuje
		if( m_frame )
		{
			//existuje - mame nastaven zamek od tohoto objektu
			m_frame->ReleaseLock();
			//uvolnime
			m_frame->Release();
		}

		//nastavime novy snimek
		m_frame = frame;
		frame->AddLock();
		frame->AddRefs();

		//pridame snimek do fronty - pokud existuje
		if ( m_queue )
		{
			m_queue->AddFrame(frame);
		}
	m_critical.Leave();
}
//OK 2007-08-25 14:23:45 B04-315B\Tom

/** Tato metoda uvolni aktualni snimek z dispatcheru.
*
*	Metoda by mela byt pouzita na konci aplikace pred odstranenim objektu.
*/
void TDispatcher::Free(void)
{
	m_critical.Enter();
		if( m_frame )
		{
			//existuje - mame nastaven zamek od tohoto objektu
			m_frame->ReleaseLock();
			//uvolnime
			m_frame->Release();

			m_frame = NULL;
		}
	m_critical.Leave();
}
//OK 2007-08-25 14:23:48 B04-315B\Tom

/** Virtualni metoda vracejici sirku poskytovanych snimku v pixelech.
*/
DWORD TDispatcher::GetWidth(void)
{
	return m_width;
}
//OK 2007-08-25 14:23:54 B04-315B\Tom

/** Virtualni metoda vracejici vysku poskytovanych snimku v pixelech.
*/
DWORD TDispatcher::GetHeight(void)
{
	return m_height;
}
//OK 2007-08-25 14:23:56 B04-315B\Tom

/** Metoda vrati aktualne ulozeny snimek.
*
*	Metoda muze vratit NULL pokud zadny snimek neni prave ulozen (napr. pri startu programu)!!!
*
*	Metoda automaticky zvysi pocet referenci na snimek pri kazdem zavolani, volajici objekt
*	je zodpovedny za spravnou praci s obdrzenou referenci!!!
*
*/
TFrame * TDispatcher::GetFrame(void)
{
	TFrameReal * tmp = NULL;

	m_critical.Enter();
		if( m_frame )
		{
			//snimek existuje
			m_frame->AddRefs();
			tmp = m_frame;
		}
		else
		{
			//zadny snimek neni v objektu ulozen - vraci se NULL
		}
	m_critical.Leave();

	return tmp;
}
//OK 2007-08-25 14:23:59 B04-315B\Tom

/** Metoda vrati aktualne ulozeny snimek.
*
*	Metoda muze vratit NULL pokud zadny snimek neni prave ulozen (napr. pri startu programu)!!!
*
*	Metoda automaticky zvysi pocet referenci na snimek pri kazdem zavolani, volajici objekt
*	je zodpovedny za spravnou praci s obdrzenou referenci!!!
*
*	Zaroven metoda zvysi pocet zamku na snimku, coz muze byt vyhodne v pripade jednotek, ktere
*	zamek na snimku vyzaduji.
*
*/
TFrame * TDispatcher::GetLockedFrame(void)
{
	TFrameReal * tmp = NULL;

	m_critical.Enter();
		if( m_frame )
		{
			m_frame->AddRefs();
			m_frame->AddLock();
			tmp = m_frame;
		}
		else
		{
			//zadny snimek v objektu
		}
	m_critical.Leave();

	return tmp;
}
//OK 2007-08-25 14:24:05 B04-315B\Tom

// TDispatcher
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
