/*
*	 This file contains classes that represents image returned from camera.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/

#include <cassert>

#include "pipeline/Image.h"
#include "pipeline/ImagePixelFormats.h"

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImageARGB

/** Vrati velikost jednoho pixelu v bajtech
*/
DWORD TImageARGB::GetPixelSize(void) const
{
	return 4;
}
//OK 2007-08-25 15:07:14 B04-315B\Tom

/** Vrati sirku v pixelech.
*/
DWORD TImageARGB::GetWidth(void) const
{
	return m_width;
}
//OK 2007-08-25 15:07:17 B04-315B\Tom

/** Vrati vysku v pixelech.
*/
DWORD TImageARGB::GetHeight(void) const
{
	return m_height;
}
//OK 2007-08-25 15:07:19 B04-315B\Tom

/** Vrati pocet bajtu mezi pixely na sousednich radcich.
*/
DWORD TImageARGB::GetPitch(void) const
{
	return m_width * 4;
}
//OK 2007-08-25 15:07:22 B04-315B\Tom

/** Vrati velikost obrazovych dat v bajtech.
*/
DWORD TImageARGB::GetDataSize(void) const
{
	return m_width * m_height * 4;
}
//OK 2007-08-25 15:07:25 B04-315B\Tom

/** Vrati obrazova data.
*/
const void*	TImageARGB::GetData(void) const
{
	return m_data;
}
//OK 2007-08-25 15:07:28 B04-315B\Tom

/** Virtualni soukromy destruktor.
*/
TImageARGB::~TImageARGB(void)
{
	delete [] m_data;
	m_data = NULL;		// can be removed
}
//OK 2007-08-25 15:07:30 B04-315B\Tom

/** Soukromy konstruktor.
*
*	\param	width	[in] sirka obrazku v pixelech
*	\param	height	[in] vyska obrazku v pixelech
*/
TImageARGB::TImageARGB(DWORD width, DWORD height)
{
	m_width		= width;
	m_height	= height;

	m_data = new unsigned char[ width * height * 4];
}
//OK 2007-08-25 15:07:33 B04-315B\Tom

/** Metoda vytvori obraz z 8-bitoveho formatu ve stupnich sedi.
*
*	\warning	Oba obrazy musi mit stejne rozmery!
*	Pokud ne pak metoda ponecha obraz nezmeneny.
*
*	\param	image	[in] obraz ve stupnich
*/
void TImageARGB::FromGray( const TImageGray * image )
{
	if ( 
		this->GetWidth()	!= image->GetWidth() ||
		this->GetHeight()	!= image->GetHeight()
	)
	{
		//rozdilna velikost obrazu
		return;
	}

	const unsigned char * data	= (const unsigned char*)image->GetData();
	const DWORD pixelCount		= this->GetWidth() * this->GetHeight();

	for ( DWORD i = 0 ; i < pixelCount ; i++ ) //pro vsechny pixely
	{
		unsigned char * pixel	= (unsigned char*)(&((DWORD*)m_data)[i]);
		
		pixel[0]	= data[i];
		pixel[1]	= data[i];
		pixel[2]	= data[i];
		pixel[3]	= 255;		
	}
}
//OK 2007-08-25 15:09:19 B04-315B\Tom

/** Metoda vytvori obraz z barevneho 24-bitoveho obrazu.
*
*	\note	Alpha kanal je nastaven na 255.
*
*	\warning	Oba obrazy musi mit stejne rozmery!
*	Pokud ne pak metoda ponecha obraz nezmeneny.
*
*	\param	image	[in] obraz ve stupnich
*/
void TImageARGB::FromRGB( const TImageRGB * image )
{
	if ( 
		this->GetWidth()	!= image->GetWidth() ||
		this->GetHeight()	!= image->GetHeight()
	)
	{
		//rozdilna velikost obrazu
		return;
	}

	const unsigned char * src	= (const unsigned char*)image->GetData();	
	const DWORD byteCount		= this->GetDataSize();

	DWORD offsetSrc = 0;
	DWORD offsetDst = 0;

	while ( offsetDst < byteCount ) //pro vsechny pixely
	{
		m_data[ offsetDst++ ]	= src[ offsetSrc++ ];	// B
		m_data[ offsetDst++ ]	= src[ offsetSrc++ ];	// G
		m_data[ offsetDst++ ]	= src[ offsetSrc++ ];	// R
		m_data[ offsetDst++ ]	= 255;					// A
	}
}
//OK 2007-09-04 13:41:21 B04-315B\Tom

/** Metoda zkopiruje zadany vyrez z poskytnuteho obrazku typu ARGB a zkopiruje jej
*	do leveho horniho rohu tohoto obrazu.
*
*	\warning	Velikost vyrezu kopirovaneho obrazu nesmi byt vetsi nez velikost tohoto
*	obrazu. Pokud ano pak metoda neprovede nic. Zaroven musi byt cely vyrez obsazen v poskytnutem
*	obraze.
*
*	\param	src		[in] zdrojovy obraz
*	\param	x		[in] x-souradnice leveho horniho rohu vyrezu
*	\param	y		[in] y-souradnice leveho horniho rohu vyrezu
*	\param	width	[in] sirka vyrezu v pixelech
*	\param	height	[in] vyska vyrezu v pixelech
*/
void TImageARGB::CopyFrom( const TImageARGB * src, DWORD x, DWORD y, DWORD width, DWORD height)
{	
	if (
		( width > 0 ) && ( height > 0 ) &&			// vyrez musi mit platnou velikost
		src->IsRectInside( x, y, width, height ) &&	//zdrojovy vyrez musi byt uvnitr zdrojoveho obrazku
		width <= m_width &&		// cely vyrez lze zkopirovat do tohoto obrazku
		height <= m_height
	)
	{
		//zkopirujeme data radek po radku

		const DWORD * tmpDataSrc = (DWORD*) src->m_data;
		DWORD * tmpDataDst = (DWORD*) this->m_data;

		for ( DWORD dy = 0 ; dy < height ; dy++ )
		{
			const DWORD * startSrc	= &tmpDataSrc[ (y + dy) * src->m_width + x ];
			DWORD * startDst		= &tmpDataDst[ dy * m_width ];

			memcpy( startDst, startSrc, sizeof( DWORD ) * width );
		}
	}
	else
	{
		//chybne rozmery vyrezu - nic se nekopiruje
	}
}
//OK 2007-08-26 17:09:54 B04-315B\Tom

// TImageARGB
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImageRGB

/** Vrati velikost jednoho pixelu v bajtech
*/
DWORD TImageRGB::GetPixelSize(void) const
{
	return 3;
}
//OK 2007-09-04 12:11:44 B04-315B\Tom

/** Vrati sirku v pixelech.
*/
DWORD TImageRGB::GetWidth(void) const
{
	return m_width;
}
//OK 2007-09-04 12:11:54 B04-315B\Tom

/** Vrati vysku v pixelech.
*/
DWORD TImageRGB::GetHeight(void) const
{
	return m_height;
}
//OK 2007-09-04 12:12:20 B04-315B\Tom

/** Vrati pocet bajtu mezi pixely na sousednich radcich.
*/
DWORD TImageRGB::GetPitch(void) const
{
	return m_width * 3;
}
//OK 2007-09-04 12:12:38 B04-315B\Tom

/** Vrati velikost obrazovych dat v bajtech.
*/
DWORD TImageRGB::GetDataSize(void) const
{
	return m_width * m_height * 3;
}
//OK 2007-09-04 12:12:51 B04-315B\Tom

/** Vrati obrazova data.
*/
const void*	TImageRGB::GetData(void) const
{
	return m_data;
}
//OK 2007-09-04 12:13:04 B04-315B\Tom

/** Virtualni soukromy destruktor.
*/
TImageRGB::~TImageRGB(void)
{
	delete [] m_data;
	m_data = NULL;		// can be removed
}
//OK 2007-09-04 12:13:16 B04-315B\Tom

/** Soukromy konstruktor.
*
*	\param	width	[in] sirka obrazku v pixelech
*	\param	height	[in] vyska obrazku v pixelech
*/
TImageRGB::TImageRGB(DWORD width, DWORD height)
{
	m_width		= width;
	m_height	= height;

	m_data = new unsigned char[ width * height * 3];
}
//OK 2007-09-04 12:13:33 B04-315B\Tom

/** Metoda vytvori obraz z 8-bitoveho formatu ve stupnich sedi.
*
*	\warning	Oba obrazy musi mit stejne rozmery!
*	Pokud ne pak metoda ponecha obraz nezmeneny.
*
*	\param	image	[in] obraz ve stupnich sedi
*/
void TImageRGB::FromGray( const TImageGray * image )
{
	if ( 
		this->GetWidth()	!= image->GetWidth() ||
		this->GetHeight()	!= image->GetHeight()
	)
	{
		//rozdilna velikost obrazu
		return;
	}

	const unsigned char * data	= (const unsigned char*)image->GetData();
	const DWORD pixelCount		= this->GetWidth() * this->GetHeight();	

	DWORD offsetDst = 0;

	for ( DWORD i = 0 ; i < pixelCount ; i++ ) //pro vsechny pixely
	{			
		m_data[ offsetDst++ ]	= data[i];
		m_data[ offsetDst++ ]	= data[i];
		m_data[ offsetDst++ ]	= data[i];				
	}
}
//OK 2007-09-04 13:15:46 B04-315B\Tom

/** Metoda vytvori obraz z 32-bitoveho ARGB formatu.
*
*	\warning	Oba obrazy musi mit stejne rozmery!
*	Pokud ne pak metoda ponecha obraz nezmeneny.
*
*	\param	image	[in] barevny obraz ve formatu ARGB
*/
void TImageRGB::FromARGB( const TImageARGB * image )
{
	if ( 
		this->GetWidth()	!= image->GetWidth() ||
		this->GetHeight()	!= image->GetHeight()
	)
	{
		//rozdilna velikost obrazu
		return;
	}

	const unsigned char * src	= (const unsigned char*)image->GetData();	
	const DWORD byteCount		= this->GetDataSize();

	DWORD offsetSrc = 0;
	DWORD offsetDst = 0;

	while ( offsetDst < byteCount ) //pro vsechny pixely
	{
		m_data[ offsetDst++ ]	= src[ offsetSrc++ ];	// B
		m_data[ offsetDst++ ]	= src[ offsetSrc++ ];	// G
		m_data[ offsetDst++ ]	= src[ offsetSrc++ ];	// R
		
		offsetSrc++;
	}
}
//OK 2007-09-04 13:49:38 B04-315B\Tom

/** Metoda zkopiruje zadany vyrez z poskytnuteho obrazku typu ARGB a zkopiruje jej
*	do leveho horniho rohu tohoto obrazu.
*
*	\warning	Velikost vyrezu kopirovaneho obrazu nesmi byt vetsi nez velikost tohoto
*	obrazu. Pokud ano pak metoda neprovede nic. Zaroven musi byt cely vyrez obsazen v poskytnutem
*	obraze.
*
*	\param	src		[in] zdrojovy obraz
*	\param	x		[in] x-souradnice leveho horniho rohu vyrezu
*	\param	y		[in] y-souradnice leveho horniho rohu vyrezu
*	\param	width	[in] sirka vyrezu v pixelech
*	\param	height	[in] vyska vyrezu v pixelech
*/
void TImageRGB::CopyFrom( const TImageRGB * src, DWORD x, DWORD y, DWORD width, DWORD height)
{	
	if (
		( width > 0 ) && ( height > 0 ) &&			// vyrez musi mit platnou velikost
		src->IsRectInside( x, y, width, height ) &&	//zdrojovy vyrez musi byt uvnitr zdrojoveho obrazku
		width <= m_width &&		// cely vyrez lze zkopirovat do tohoto obrazku
		height <= m_height
	)
	{
		//zkopirujeme data radek po radku

		const unsigned char * tmpDataSrc	= (unsigned char*) src->m_data;
		unsigned char * tmpDataDst			= (unsigned char*) this->m_data;

		const DWORD pitchSrc = src->GetPitch();
		const DWORD pitchDst = this->GetPitch();		

		DWORD offsetSrc = ( y * src->m_width + x ) * 3;
		DWORD offsetDst = 0;

		for ( DWORD dy = 0 ; dy < height ; dy++ )	// pro kazdy radek
		{
			const unsigned char * startSrc	= &tmpDataSrc[ offsetSrc ];
			unsigned char * startDst		= &tmpDataDst[ offsetDst ];

			offsetSrc += pitchSrc;
			offsetDst += pitchDst;

			memcpy( startDst, startSrc, sizeof( DWORD ) * width );
		}
	}
	else
	{
		//chybne rozmery vyrezu - nic se nekopiruje
	}
}
//OK 2007-09-04 13:35:33 B04-315B\Tom

// TImageRGB
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImageGray

/** Vrati velikost jednoho pixelu v bajtech
*/
DWORD TImageGray::GetPixelSize(void) const
{
	return 1;
}
//OK 2007-08-25 15:09:23 B04-315B\Tom

/** Vrati sirku v pixelech.
*/
DWORD TImageGray::GetWidth(void) const
{
	return m_width;
}
//OK 2007-08-25 15:09:24 B04-315B\Tom

/** Vrati vysku v pixelech.
*/
DWORD TImageGray::GetHeight(void) const
{
	return m_height;
}
//OK 2007-08-25 15:09:28 B04-315B\Tom

/** Vrati pocet bajtu mezi pixely na sousednich radcich.
*/
DWORD TImageGray::GetPitch(void) const
{
	return m_width * 1;
}
//OK 2007-08-25 15:09:31 B04-315B\Tom

/** Vrati velikost obrazovych dat v bajtech.
*/
DWORD TImageGray::GetDataSize(void) const
{
	return m_width * m_height * 1;
}
//OK 2007-08-25 15:09:38 B04-315B\Tom

/** Vrati obrazova data.
*/
const void* TImageGray::GetData(void) const
{
	return m_data;
}
//OK 2007-08-25 15:09:47 B04-315B\Tom

/** Virtualni soukromy destruktor.
*/
TImageGray::~TImageGray(void)
{
	delete [] m_data;
	m_data = NULL;		// can be removed
}
//OK 2007-08-25 15:09:52 B04-315B\Tom

/** Soukromy konstruktor.
*
*	\param	width	[in] sirka obrazku v pixelech
*	\param	height	[in] vyska obrazku v pixelech
*/
TImageGray::TImageGray(DWORD width, DWORD height)
{
	m_width		= width;
	m_height	= height;

	m_data = new unsigned char[ width * height ];
}
//OK 2007-08-25 15:10:01 B04-315B\Tom

/** Metoda vytvori obraz z 32-bitoveho formatu ARGB na monochromaticky format.
*
*	\note	Alpha kanal je ignororvan
*
*	\warning	Oba obrazy musi mit stejne rozmery!
*
*	\param	image	[in] obraz ve formatu ARGB
*/
void TImageGray::FromARGB( const TImageARGB * image )
{
	if (
		this->GetWidth()	!= image->GetWidth() ||
		this->GetHeight()	!= image->GetHeight()
	)
	{
		//rozdilna velikost obrazu
		return;
	}

	//prekonvertujeme na monochromaticky obraz
	//RGB = (0.2989 , 0.5866, 0.1144)

	const DWORD * data = (const DWORD*)image->GetData();
	const DWORD pixelCount = this->GetWidth() * this->GetHeight();

	for ( DWORD i = 0 ; i < pixelCount ; i++ ) //pro vsechny pixely
	{
		float blue	= (float) ((unsigned char*)&data[i])[0];
		float green	= (float) ((unsigned char*)&data[i])[1];
		float red	= (float) ((unsigned char*)&data[i])[2];

		float tmp	= 
			red * 0.2989f +
			green * 0.5866f +
			blue * 0.1144f;

		//TODO OPTIMIZE
		unsigned char res;
#if 1
		//predpoklada se ze vysledek nebude nikdy mensi nez 0
		if ( tmp <= 255.f )
		{
			//vice pravdepodobna varianta
			res = (unsigned char) tmp;
		}
		else
		{
			res = 255;
		}
#else
		//obecne reseni - TMP nikdy 0 byt nemuze - viz vyse - UNSIGNED CHAR rozsh je 0..255
		if ( tmp >= 0.f )
		{
			if ( tmp <= 255.f )
			{
				res = (unsigned char) tmp;
			}
			else
			{
				res = 255;
			}
		}
		else
		{
			res = 0;
		}
#endif
		m_data[i] = res;
	}
}
//OK 2007-08-25 15:14:49 B04-315B\Tom

/** Metoda prevede obraz z 24-bitoveho formatu RGB na monochromaticky format.
*
*	\note	Alpha kanal je ignororvan
*
*	\warning	Oba obrazy musi mit stejne rozmery!
*
*	\param	image	[in] obraz ve formatu ARGB
*/
void TImageGray::FromRGB( const TImageRGB * image )
{
	if (
		this->GetWidth()	!= image->GetWidth() ||
		this->GetHeight()	!= image->GetHeight()
	)
	{
		//rozdilna velikost obrazu
		return;
	}

	//prekonvertujeme na monochromaticky obraz
	//RGB = (0.2989 , 0.5866, 0.1144)

	const unsigned char * src = (unsigned char*)image->GetData();
	const DWORD byteCount = image->GetDataSize();

	DWORD offsetSrc = 0;
	DWORD offsetDst = 0;

	while ( offsetSrc < byteCount )	
	{
		float blue	= (float) src[ offsetSrc++ ];
		float green	= (float) src[ offsetSrc++ ];
		float red	= (float) src[ offsetSrc++ ];

		float tmp	= 
			red * 0.2989f +
			green * 0.5866f +
			blue * 0.1144f;

		//TODO OPTIMIZE
		unsigned char res;

		//predpoklada se ze vysledek nebude nikdy mensi nez 0
		if ( tmp <= 255.f )
		{
			//vice pravdepodobna varianta
			res = (unsigned char) tmp;
		}
		else
		{
			res = 255;
		}

		m_data[ offsetDst++ ] = res;
	}
}
//OK 2007-09-04 14:04:46 B04-315B\Tom

/** Metoda zkopiruje zadany vyrez z poskytnuteho obrazku typu GRAY a zkopiruje jej
*	do leveho horniho rohu tohoto obrazu.
*
*	\warning	Velikost vyrezu kopirovaneho obrazu nesmi byt vetsi nez velikost tohoto
*	obrazu. Pokud ano pak metoda neprovede nic. Zaroven musi byt cely vyrez obsazen v poskytnutem
*	obraze.
*
*	\param	src		[in] zdrojovy obraz
*	\param	x		[in] x-souradnice leveho horniho rohu vyrezu
*	\param	y		[in] y-souradnice leveho horniho rohu vyrezu
*	\param	width	[in] sirka vyrezu v pixelech
*	\param	height	[in] vyska vyrezu v pixelech
*/
void TImageGray::CopyFrom( const TImageGray * src, DWORD x, DWORD y, DWORD width, DWORD height)
{	
	if (
		( width > 0 ) && ( height > 0 ) &&			// vyrez musi mit platnou velikost
		src->IsRectInside( x, y, width, height ) &&	//zdrojovy vyrez musi byt uvnitr zdrojoveho obrazku
		width <= m_width &&		// cely vyrez lze zkopirovat do tohoto obrazku
		height <= m_height
	)
	{
		//zkopirujeme data radek po radku

		const unsigned char * tmpDataSrc = (unsigned char*) src->m_data;
		unsigned char * tmpDataDst = (unsigned char*) this->m_data;

		for ( DWORD dy = 0 ; dy < height ; dy++ )
		{
			const unsigned char * startSrc	= &tmpDataSrc[ (y + dy) * src->m_width + x ];
			unsigned char * startDst		= &tmpDataDst[ dy * m_width ];

			memcpy( startDst, startSrc, sizeof( unsigned char ) * width );
		}
	}
	else
	{
		//chybne rozmery vyrezu - nic se nekopiruje
	}
}
//OK 2007-08-26 17:18:44 B04-315B\Tom

// TImageGray
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImageSetReal

/** Vrati obraz typu ARGB.
*/
const TImage* TImageSetReal::GetARGB(void) const
{
	return m_image_ARGB;
}
//OK 2007-08-25 15:14:55 B04-315B\Tom

/** Vrati obraz typu GRAY.
*/
const TImage* TImageSetReal::GetGray(void) const
{
	return m_image_gray;
}
//OK 2007-08-25 15:14:58 B04-315B\Tom

/** Vrati obraz typu RGB.
*/
const TImage* TImageSetReal::GetRGB(void) const
{
	return m_image_RGB;
}
//OK 2007-09-04 14:14:34 B04-315B\Tom

/** Konstruktor.
*
*	\param	manager		[in] manager kde byl tento objekt vytvoren
*	\param	id			[in] jedinecna identifikace tohoto objektu v manageru
*	\param	width		[in] pozadovana zakladni sirka obrazku
*	\param	height		[in] pozadovana zakladni vyska obrazku
*/
TImageSetReal::TImageSetReal(TImageSetManager * manager, DWORD id, DWORD width, DWORD height)
{
	m_manager		= manager;
	m_managerID		= id;

	m_image_ARGB	= new TImageARGB( width, height ); 
	m_image_gray	= new TImageGray( width, height );
	m_image_RGB		= new TImageRGB( width, height );
}
//OK 2007-08-25 15:15:47 B04-315B\Tom
//OK 2007-09-04 14:15:30 B04-315B\Tom

/**	Soukromy virtualni destruktor.
*/
TImageSetReal::~TImageSetReal(void)
{
	delete m_image_ARGB;
	m_image_ARGB = NULL;

	delete m_image_gray;
	m_image_gray = NULL;

	delete m_image_RGB;
	m_image_RGB = NULL;
}
//OK 2007-08-25 15:15:48 B04-315B\Tom
//OK 2007-09-04 14:15:32 B04-315B\Tom

// TImageSetReal
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImageSetManager

/** Konstruktor.
*
*	Inicializuje manazer.
*
*	\param	width		[in] zakladni sirka obrazku v pixelech - musi byt nenulova
*	\param	height		[in] zakladni vyska obrazku v pixelech - musi byt nenulova
*	\param	initSize	[in] pocatecni mnozstvi predalokovanych obrazku
*/
TImageSetManager::TImageSetManager( DWORD width, DWORD height, DWORD initSize )
{
	assert(width);
	assert(height);

	m_width		= width;
	m_height	= height;	

	//naalokujeme obrazky
	for( DWORD i = 0 ; i < initSize ; i++ )
	{
		//vytvorime obrazek
		TImageSetReal * img = new TImageSetReal( this, i, m_width, m_height );			
		m_imageSets.push_back(img);
		//oznacime jako volny
		m_imgFlags.push_back(FALSE);	
	}
}
//OK 2007-08-25 15:00:34 B04-315B\Tom

/** Destruktor.
*
*	\warning	Mel by byt volan az v okamziku kdy nejsou zadne zdroje z manageru pouzity v aplikaci.
*/
TImageSetManager::~TImageSetManager(void)
{
	m_critical.Enter();
		const DWORD size = (DWORD)m_imageSets.size();	
		for( DWORD i = 0 ; i < size ; i++ )
		{		
			delete m_imageSets[i];
			m_imageSets[i] = NULL;
		}
	m_critical.Leave();
}
//OK 2007-08-25 15:00:36 B04-315B\Tom

/** Ziska novy nepouzity obrazek.
*
*	Metoda je synchronizovana s metodou ReleaseImageSet().
*
*	\warning	Muze vratit NULL pokud uz neni zadny obrazek k dispozici.
*/ 
TImageSetReal* TImageSetManager::GetImageSet(void)
{
	m_critical.Enter();	
		//najdeme prvni nepouzivany obrazek
		DWORD i = 0;
		const DWORD size = (DWORD)m_imageSets.size();
		BOOL found = FALSE;
		while( i < size )
		{
			if ( FALSE == m_imgFlags[i] )
			{
				//tento obrazek je volny - pouzijeme ho
				m_imgFlags[i] = TRUE;
				found = TRUE;
				break;
			}
			i++;
		}

		TImageSetReal * ret = NULL;	//ziskany obrazek 
		if(found)
		{
			//nalezli jsme volny obrazek
			ret = m_imageSets[i];
		}
		else
		{
#if 1
			//nemame zadny obrazek - vracime NULL
#else
			//zadny obrazek neni volny - musi alokovat novy a pridat
			//velikost puvodniho pole obrazku je rovna ID noveho obrazku!!!
			ret = new TImageSetReal( this, size, m_width, m_height );
			m_imageSets.push_back(ret);

			m_imgFlags.push_back(TRUE);	// obrazek uz je pouzit
#endif
		}
	m_critical.Leave();

	return ret;
}
//OK 2007-08-25 15:00:50 B04-315B\Tom

/** Vraceni obrazku zpet do manageru. Timto se tento obrazek stane nepouzivanym
*	a muze byt znovu vracen metodou GetImageSet().
*
*	Metoda je synchronizovana s metodou GetImageSet().
*	
*	\param	img		[in]	objekt s obrazky jednoho snimku
*/
void TImageSetManager::ReleaseImageSet(const TImageSetReal * img)
{
	m_critical.Enter();

		const DWORD size	= (DWORD)m_imageSets.size();
		const DWORD id		= img->GetID();
		if(id < size)
		{
			// ID je validni - nastavime prislusny flag
			m_imgFlags[id] = FALSE;	// obrazek muze byt znovu pouzit
		}
		else
		{
			// ID nepatri do seznamu obrazku
			assert(false);
		}

	m_critical.Leave();
}
//OK 2007-08-25 15:02:30 B04-315B\Tom

// TImageSetManager
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TFrameReal

/** Vrati skupinu obrazu ulozenych v tomto snimku.
*/
const TImageSet* TFrameReal::GetImageSet(void) const
{
	return m_image;
}
//OK 2007-08-25 15:04:02 B04-315B\Tom

/** Vrati casovou znacku tohoto objektu.
*/
const TTimeStamp & TFrameReal::GetTimestamp(void) const
{
	return m_time;
}
//OK 2007-08-25 15:04:05 B04-315B\Tom

/** Zvyseni poctu referenci na snimek.
*/
void TFrameReal::AddRefs(void)
{
	m_critSection.Enter();
		m_refs++;		
	m_critSection.Leave();
}
//OK 2007-08-25 15:04:16 B04-315B\Tom

/** Snizeni poctu referenci nad danym snimkem.
*	Po dosazeni poctu referenci na 0 je snimek odstranen.
*/
void TFrameReal::Release(void)
{
	bool remove = false;

	m_critSection.Enter();
		if( m_refs > 0 )
		{
			m_refs--;
			if( m_refs == 0 )
			{
				//zrusime objekt - posledni reference byla odstranena
				remove = true;
			}
			else
			{
				// objekt ma reference - nic dalsiho se proto nedeje
			}
		}
		else
		{
			//objekt uz nema reference - k tomu by nemelo dojit
			remove = true;
		}
	m_critSection.Leave();

	if( remove )
	{
		//pokud toto nastane pak uz v aplikaci neni zadna reference a je tedy mozne
		//vynechat synchronizaci
		delete this;
	}
}
//OK 2007-08-25 15:04:34 B04-315B\Tom

/** Vrati pocet referenci na snimek.
*/
DWORD TFrameReal::GetRefs(void) const
{
	return m_refs;
}
//OK 2007-08-25 15:04:44 B04-315B\Tom

/** Pridani zamku na tento snimek.
*
*	Zamek muze pridat napriklad vypocetni jednotka pokud je pozadovano, aby
*	dochazelo k synchronizaci zobrazovani s timto snimkem.
*
*	Metoda vraci pocet zamku na snimek po zvyseni poctu zamku.
*/
DWORD TFrameReal::AddLock(void)
{
	m_critSection2.Enter();
		m_locks++;
		DWORD ret = m_locks;
	m_critSection2.Leave();

	return ret;
}
//OK 2007-08-25 15:04:59 B04-315B\Tom

/** Uvolneni zamku ze snimku.
*
*	Pokud je pocet zamku 0 pak muze snimek opustit frontu snimku a byt zobrazen.
*
*	Metoda vraci pocet zamku na snimek po snizeni poctu zamku.
*/
DWORD TFrameReal::ReleaseLock(void)
{
	m_critSection2.Enter();
		if( m_locks > 0 )
		{
			m_locks--;
		}
		else
		{
			//CHYBA
		}
		DWORD ret = m_locks;
	m_critSection2.Leave();

	return ret;
}
//OK 2007-08-25 15:05:27 B04-315B\Tom

/** Vrati pocet zamku tohoto snimku.
*/
DWORD TFrameReal::GetLockCount(void) const
{
	return m_locks;
}
//OK 2007-08-25 15:05:32 B04-315B\Tom

/** Soukromy virtualni destruktor.
*
*	Je volany z metody TFrameReal::Release() v okamziku kdy
*	je pocet referenci roven 0. Nehrozi tedy kolize s ostatnimi vlakny.
*
*	Automaticky vraci alokovany snimek zpatky do manazeru snimku.
*/
TFrameReal::~TFrameReal(void)
{
	//musime vratit pouzity snimek
	m_image->GetManager()->ReleaseImageSet( m_image );
}
//OK 2007-08-25 15:05:59 B04-315B\Tom

/** Konstruktor.
*
*	Vytvori objekt s poctem referenci 1, zadnym zamkem a casem vytvoreni rovnym casu volani metody.
*
*	\param	images	[in] snimek ziskany z manazeru - je ulozen v tomto objektu a vracen
*					destruktorem
*/
TFrameReal::TFrameReal( TImageSetReal * images ):
	m_time()
{
	m_image = images;

	m_refs = 1;
	m_locks = 0;
}
//OK 2007-08-25 15:06:37 B04-315B\Tom

// TFrameReal
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
